/*! DataTables 1.10.24
 * ©2008-2021 SpryMedia Ltd - datatables.net/license
 */

/**
 * @summary     DataTables
 * @description Paginate, search and order HTML tables
 * @version     1.10.24
 * @file        jquery.dataTables.js
 * @author      SpryMedia Ltd
 * @contact     www.datatables.net
 * @copyright   Copyright 2008-2021 SpryMedia Ltd.
 *
 * This source file is free software, available under the following license:
 *   MIT license - http://datatables.net/license
 *
 * This source file is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE. See the license files for details.
 *
 * For details please refer to: http://www.datatables.net
 */

/*jslint evil: true, undef: true, browser: true */
/*globals $,require,jQuery,define,_selector_run,_selector_opts,_selector_first,_selector_row_indexes,_ext,_Api,_api_register,_api_registerPlural,_re_new_lines,_re_html,_re_formatted_numeric,_re_escape_regex,_empty,_intVal,_numToDecimal,_isNumber,_isHtml,_htmlNumeric,_pluck,_pluck_order,_range,_stripHtml,_unique,_fnBuildAjax,_fnAjaxUpdate,_fnAjaxParameters,_fnAjaxUpdateDraw,_fnAjaxDataSrc,_fnAddColumn,_fnColumnOptions,_fnAdjustColumnSizing,_fnVisibleToColumnIndex,_fnColumnIndexToVisible,_fnVisbleColumns,_fnGetColumns,_fnColumnTypes,_fnApplyColumnDefs,_fnHungarianMap,_fnCamelToHungarian,_fnLanguageCompat,_fnBrowserDetect,_fnAddData,_fnAddTr,_fnNodeToDataIndex,_fnNodeToColumnIndex,_fnGetCellData,_fnSetCellData,_fnSplitObjNotation,_fnGetObjectDataFn,_fnSetObjectDataFn,_fnGetDataMaster,_fnClearTable,_fnDeleteIndex,_fnInvalidate,_fnGetRowElements,_fnCreateTr,_fnBuildHead,_fnDrawHead,_fnDraw,_fnReDraw,_fnAddOptionsHtml,_fnDetectHeader,_fnGetUniqueThs,_fnFeatureHtmlFilter,_fnFilterComplete,_fnFilterCustom,_fnFilterColumn,_fnFilter,_fnFilterCreateSearch,_fnEscapeRegex,_fnFilterData,_fnFeatureHtmlInfo,_fnUpdateInfo,_fnInfoMacros,_fnInitialise,_fnInitComplete,_fnLengthChange,_fnFeatureHtmlLength,_fnFeatureHtmlPaginate,_fnPageChange,_fnFeatureHtmlProcessing,_fnProcessingDisplay,_fnFeatureHtmlTable,_fnScrollDraw,_fnApplyToChildren,_fnCalculateColumnWidths,_fnThrottle,_fnConvertToWidth,_fnGetWidestNode,_fnGetMaxLenString,_fnStringToCss,_fnSortFlatten,_fnSort,_fnSortAria,_fnSortListener,_fnSortAttachListener,_fnSortingClasses,_fnSortData,_fnSaveState,_fnLoadState,_fnSettingsFromNode,_fnLog,_fnMap,_fnBindAction,_fnCallbackReg,_fnCallbackFire,_fnLengthOverflow,_fnRenderer,_fnDataSource,_fnRowAttributes*/

(function( factory ) {
	"use strict";

	if ( typeof define === 'function' && define.amd ) {
		// AMD
		define( ['jquery'], function ( $ ) {
			return factory( $, window, document );
		} );
	}
	else if ( typeof exports === 'object' ) {
		// CommonJS
		module.exports = function (root, $) {
			if ( ! root ) {
				// CommonJS environments without a window global must pass a
				// root. This will give an error otherwise
				root = window;
			}

			if ( ! $ ) {
				$ = typeof window !== 'undefined' ? // jQuery's factory checks for a global window
					require('jquery') :
					require('jquery')( root );
			}

			return factory( $, root, root.document );
		};
	}
	else {
		// Browser
		factory( jQuery, window, document );
	}
}
(function( $, window, document, undefined ) {
	"use strict";

	/**
	 * DataTables is a plug-in for the jQuery Javascript library. It is a highly
	 * flexible tool, based upon the foundations of progressive enhancement,
	 * which will add advanced interaction controls to any HTML table. For a
	 * full list of features please refer to
	 * [DataTables.net](href="http://datatables.net).
	 *
	 * Note that the `DataTable` object is not a global variable but is aliased
	 * to `jQuery.fn.DataTable` and `jQuery.fn.dataTable` through which it may
	 * be  accessed.
	 *
	 *  @class
	 *  @param {object} [init={}] Configuration object for DataTables. Options
	 *    are defined by {@link DataTable.defaults}
	 *  @requires jQuery 1.7+
	 *
	 *  @example
	 *    // Basic initialisation
	 *    $(document).ready( function {
	 *      $('#example').dataTable();
	 *    } );
	 *
	 *  @example
	 *    // Initialisation with configuration options - in this case, disable
	 *    // pagination and sorting.
	 *    $(document).ready( function {
	 *      $('#example').dataTable( {
	 *        "paginate": false,
	 *        "sort": false
	 *      } );
	 *    } );
	 */
	var DataTable = function ( options )
	{
		/**
		 * Perform a jQuery selector action on the table's TR elements (from the tbody) and
		 * return the resulting jQuery object.
		 *  @param {string|node|jQuery} sSelector jQuery selector or node collection to act on
		 *  @param {object} [oOpts] Optional parameters for modifying the rows to be included
		 *  @param {string} [oOpts.filter=none] Select TR elements that meet the current filter
		 *    criterion ("applied") or all TR elements (i.e. no filter).
		 *  @param {string} [oOpts.order=current] Order of the TR elements in the processed array.
		 *    Can be either 'current', whereby the current sorting of the table is used, or
		 *    'original' whereby the original order the data was read into the table is used.
		 *  @param {string} [oOpts.page=all] Limit the selection to the currently displayed page
		 *    ("current") or not ("all"). If 'current' is given, then order is assumed to be
		 *    'current' and filter is 'applied', regardless of what they might be given as.
		 *  @returns {object} jQuery object, filtered by the given selector.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Highlight every second row
		 *      oTable.$('tr:odd').css('backgroundColor', 'blue');
		 *    } );
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Filter to rows with 'Webkit' in them, add a background colour and then
		 *      // remove the filter, thus highlighting the 'Webkit' rows only.
		 *      oTable.fnFilter('Webkit');
		 *      oTable.$('tr', {"search": "applied"}).css('backgroundColor', 'blue');
		 *      oTable.fnFilter('');
		 *    } );
		 */
		this.$ = function ( sSelector, oOpts )
		{
			return this.api(true).$( sSelector, oOpts );
		};
		
		
		/**
		 * Almost identical to $ in operation, but in this case returns the data for the matched
		 * rows - as such, the jQuery selector used should match TR row nodes or TD/TH cell nodes
		 * rather than any descendants, so the data can be obtained for the row/cell. If matching
		 * rows are found, the data returned is the original data array/object that was used to
		 * create the row (or a generated array if from a DOM source).
		 *
		 * This method is often useful in-combination with $ where both functions are given the
		 * same parameters and the array indexes will match identically.
		 *  @param {string|node|jQuery} sSelector jQuery selector or node collection to act on
		 *  @param {object} [oOpts] Optional parameters for modifying the rows to be included
		 *  @param {string} [oOpts.filter=none] Select elements that meet the current filter
		 *    criterion ("applied") or all elements (i.e. no filter).
		 *  @param {string} [oOpts.order=current] Order of the data in the processed array.
		 *    Can be either 'current', whereby the current sorting of the table is used, or
		 *    'original' whereby the original order the data was read into the table is used.
		 *  @param {string} [oOpts.page=all] Limit the selection to the currently displayed page
		 *    ("current") or not ("all"). If 'current' is given, then order is assumed to be
		 *    'current' and filter is 'applied', regardless of what they might be given as.
		 *  @returns {array} Data for the matched elements. If any elements, as a result of the
		 *    selector, were not TR, TD or TH elements in the DataTable, they will have a null
		 *    entry in the array.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Get the data from the first row in the table
		 *      var data = oTable._('tr:first');
		 *
		 *      // Do something useful with the data
		 *      alert( "First cell is: "+data[0] );
		 *    } );
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Filter to 'Webkit' and get all data for
		 *      oTable.fnFilter('Webkit');
		 *      var data = oTable._('tr', {"search": "applied"});
		 *
		 *      // Do something with the data
		 *      alert( data.length+" rows matched the search" );
		 *    } );
		 */
		this._ = function ( sSelector, oOpts )
		{
			return this.api(true).rows( sSelector, oOpts ).data();
		};
		
		
		/**
		 * Create a DataTables Api instance, with the currently selected tables for
		 * the Api's context.
		 * @param {boolean} [traditional=false] Set the API instance's context to be
		 *   only the table referred to by the `DataTable.ext.iApiIndex` option, as was
		 *   used in the API presented by DataTables 1.9- (i.e. the traditional mode),
		 *   or if all tables captured in the jQuery object should be used.
		 * @return {DataTables.Api}
		 */
		this.api = function ( traditional )
		{
			return traditional ?
				new _Api(
					_fnSettingsFromNode( this[ _ext.iApiIndex ] )
				) :
				new _Api( this );
		};
		
		
		/**
		 * Add a single new row or multiple rows of data to the table. Please note
		 * that this is suitable for client-side processing only - if you are using
		 * server-side processing (i.e. "bServerSide": true), then to add data, you
		 * must add it to the data source, i.e. the server-side, through an Ajax call.
		 *  @param {array|object} data The data to be added to the table. This can be:
		 *    <ul>
		 *      <li>1D array of data - add a single row with the data provided</li>
		 *      <li>2D array of arrays - add multiple rows in a single call</li>
		 *      <li>object - data object when using <i>mData</i></li>
		 *      <li>array of objects - multiple data objects when using <i>mData</i></li>
		 *    </ul>
		 *  @param {bool} [redraw=true] redraw the table or not
		 *  @returns {array} An array of integers, representing the list of indexes in
		 *    <i>aoData</i> ({@link DataTable.models.oSettings}) that have been added to
		 *    the table.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    // Global var for counter
		 *    var giCount = 2;
		 *
		 *    $(document).ready(function() {
		 *      $('#example').dataTable();
		 *    } );
		 *
		 *    function fnClickAddRow() {
		 *      $('#example').dataTable().fnAddData( [
		 *        giCount+".1",
		 *        giCount+".2",
		 *        giCount+".3",
		 *        giCount+".4" ]
		 *      );
		 *
		 *      giCount++;
		 *    }
		 */
		this.fnAddData = function( data, redraw )
		{
			var api = this.api( true );
		
			/* Check if we want to add multiple rows or not */
			var rows = Array.isArray(data) && ( Array.isArray(data[0]) || $.isPlainObject(data[0]) ) ?
				api.rows.add( data ) :
				api.row.add( data );
		
			if ( redraw === undefined || redraw ) {
				api.draw();
			}
		
			return rows.flatten().toArray();
		};
		
		
		/**
		 * This function will make DataTables recalculate the column sizes, based on the data
		 * contained in the table and the sizes applied to the columns (in the DOM, CSS or
		 * through the sWidth parameter). This can be useful when the width of the table's
		 * parent element changes (for example a window resize).
		 *  @param {boolean} [bRedraw=true] Redraw the table or not, you will typically want to
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable( {
		 *        "sScrollY": "200px",
		 *        "bPaginate": false
		 *      } );
		 *
		 *      $(window).on('resize', function () {
		 *        oTable.fnAdjustColumnSizing();
		 *      } );
		 *    } );
		 */
		this.fnAdjustColumnSizing = function ( bRedraw )
		{
			var api = this.api( true ).columns.adjust();
			var settings = api.settings()[0];
			var scroll = settings.oScroll;
		
			if ( bRedraw === undefined || bRedraw ) {
				api.draw( false );
			}
			else if ( scroll.sX !== "" || scroll.sY !== "" ) {
				/* If not redrawing, but scrolling, we want to apply the new column sizes anyway */
				_fnScrollDraw( settings );
			}
		};
		
		
		/**
		 * Quickly and simply clear a table
		 *  @param {bool} [bRedraw=true] redraw the table or not
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Immediately 'nuke' the current rows (perhaps waiting for an Ajax callback...)
		 *      oTable.fnClearTable();
		 *    } );
		 */
		this.fnClearTable = function( bRedraw )
		{
			var api = this.api( true ).clear();
		
			if ( bRedraw === undefined || bRedraw ) {
				api.draw();
			}
		};
		
		
		/**
		 * The exact opposite of 'opening' a row, this function will close any rows which
		 * are currently 'open'.
		 *  @param {node} nTr the table row to 'close'
		 *  @returns {int} 0 on success, or 1 if failed (can't find the row)
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable;
		 *
		 *      // 'open' an information row when a row is clicked on
		 *      $('#example tbody tr').click( function () {
		 *        if ( oTable.fnIsOpen(this) ) {
		 *          oTable.fnClose( this );
		 *        } else {
		 *          oTable.fnOpen( this, "Temporary row opened", "info_row" );
		 *        }
		 *      } );
		 *
		 *      oTable = $('#example').dataTable();
		 *    } );
		 */
		this.fnClose = function( nTr )
		{
			this.api( true ).row( nTr ).child.hide();
		};
		
		
		/**
		 * Remove a row for the table
		 *  @param {mixed} target The index of the row from aoData to be deleted, or
		 *    the TR element you want to delete
		 *  @param {function|null} [callBack] Callback function
		 *  @param {bool} [redraw=true] Redraw the table or not
		 *  @returns {array} The row that was deleted
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Immediately remove the first row
		 *      oTable.fnDeleteRow( 0 );
		 *    } );
		 */
		this.fnDeleteRow = function( target, callback, redraw )
		{
			var api = this.api( true );
			var rows = api.rows( target );
			var settings = rows.settings()[0];
			var data = settings.aoData[ rows[0][0] ];
		
			rows.remove();
		
			if ( callback ) {
				callback.call( this, settings, data );
			}
		
			if ( redraw === undefined || redraw ) {
				api.draw();
			}
		
			return data;
		};
		
		
		/**
		 * Restore the table to it's original state in the DOM by removing all of DataTables
		 * enhancements, alterations to the DOM structure of the table and event listeners.
		 *  @param {boolean} [remove=false] Completely remove the table from the DOM
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      // This example is fairly pointless in reality, but shows how fnDestroy can be used
		 *      var oTable = $('#example').dataTable();
		 *      oTable.fnDestroy();
		 *    } );
		 */
		this.fnDestroy = function ( remove )
		{
			this.api( true ).destroy( remove );
		};
		
		
		/**
		 * Redraw the table
		 *  @param {bool} [complete=true] Re-filter and resort (if enabled) the table before the draw.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Re-draw the table - you wouldn't want to do it here, but it's an example :-)
		 *      oTable.fnDraw();
		 *    } );
		 */
		this.fnDraw = function( complete )
		{
			// Note that this isn't an exact match to the old call to _fnDraw - it takes
			// into account the new data, but can hold position.
			this.api( true ).draw( complete );
		};
		
		
		/**
		 * Filter the input based on data
		 *  @param {string} sInput String to filter the table on
		 *  @param {int|null} [iColumn] Column to limit filtering to
		 *  @param {bool} [bRegex=false] Treat as regular expression or not
		 *  @param {bool} [bSmart=true] Perform smart filtering or not
		 *  @param {bool} [bShowGlobal=true] Show the input global filter in it's input box(es)
		 *  @param {bool} [bCaseInsensitive=true] Do case-insensitive matching (true) or not (false)
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Sometime later - filter...
		 *      oTable.fnFilter( 'test string' );
		 *    } );
		 */
		this.fnFilter = function( sInput, iColumn, bRegex, bSmart, bShowGlobal, bCaseInsensitive )
		{
			var api = this.api( true );
		
			if ( iColumn === null || iColumn === undefined ) {
				api.search( sInput, bRegex, bSmart, bCaseInsensitive );
			}
			else {
				api.column( iColumn ).search( sInput, bRegex, bSmart, bCaseInsensitive );
			}
		
			api.draw();
		};
		
		
		/**
		 * Get the data for the whole table, an individual row or an individual cell based on the
		 * provided parameters.
		 *  @param {int|node} [src] A TR row node, TD/TH cell node or an integer. If given as
		 *    a TR node then the data source for the whole row will be returned. If given as a
		 *    TD/TH cell node then iCol will be automatically calculated and the data for the
		 *    cell returned. If given as an integer, then this is treated as the aoData internal
		 *    data index for the row (see fnGetPosition) and the data for that row used.
		 *  @param {int} [col] Optional column index that you want the data of.
		 *  @returns {array|object|string} If mRow is undefined, then the data for all rows is
		 *    returned. If mRow is defined, just data for that row, and is iCol is
		 *    defined, only data for the designated cell is returned.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    // Row data
		 *    $(document).ready(function() {
		 *      oTable = $('#example').dataTable();
		 *
		 *      oTable.$('tr').click( function () {
		 *        var data = oTable.fnGetData( this );
		 *        // ... do something with the array / object of data for the row
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Individual cell data
		 *    $(document).ready(function() {
		 *      oTable = $('#example').dataTable();
		 *
		 *      oTable.$('td').click( function () {
		 *        var sData = oTable.fnGetData( this );
		 *        alert( 'The cell clicked on had the value of '+sData );
		 *      } );
		 *    } );
		 */
		this.fnGetData = function( src, col )
		{
			var api = this.api( true );
		
			if ( src !== undefined ) {
				var type = src.nodeName ? src.nodeName.toLowerCase() : '';
		
				return col !== undefined || type == 'td' || type == 'th' ?
					api.cell( src, col ).data() :
					api.row( src ).data() || null;
			}
		
			return api.data().toArray();
		};
		
		
		/**
		 * Get an array of the TR nodes that are used in the table's body. Note that you will
		 * typically want to use the '$' API method in preference to this as it is more
		 * flexible.
		 *  @param {int} [iRow] Optional row index for the TR element you want
		 *  @returns {array|node} If iRow is undefined, returns an array of all TR elements
		 *    in the table's body, or iRow is defined, just the TR element requested.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Get the nodes from the table
		 *      var nNodes = oTable.fnGetNodes( );
		 *    } );
		 */
		this.fnGetNodes = function( iRow )
		{
			var api = this.api( true );
		
			return iRow !== undefined ?
				api.row( iRow ).node() :
				api.rows().nodes().flatten().toArray();
		};
		
		
		/**
		 * Get the array indexes of a particular cell from it's DOM element
		 * and column index including hidden columns
		 *  @param {node} node this can either be a TR, TD or TH in the table's body
		 *  @returns {int} If nNode is given as a TR, then a single index is returned, or
		 *    if given as a cell, an array of [row index, column index (visible),
		 *    column index (all)] is given.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      $('#example tbody td').click( function () {
		 *        // Get the position of the current data from the node
		 *        var aPos = oTable.fnGetPosition( this );
		 *
		 *        // Get the data array for this row
		 *        var aData = oTable.fnGetData( aPos[0] );
		 *
		 *        // Update the data array and return the value
		 *        aData[ aPos[1] ] = 'clicked';
		 *        this.innerHTML = 'clicked';
		 *      } );
		 *
		 *      // Init DataTables
		 *      oTable = $('#example').dataTable();
		 *    } );
		 */
		this.fnGetPosition = function( node )
		{
			var api = this.api( true );
			var nodeName = node.nodeName.toUpperCase();
		
			if ( nodeName == 'TR' ) {
				return api.row( node ).index();
			}
			else if ( nodeName == 'TD' || nodeName == 'TH' ) {
				var cell = api.cell( node ).index();
		
				return [
					cell.row,
					cell.columnVisible,
					cell.column
				];
			}
			return null;
		};
		
		
		/**
		 * Check to see if a row is 'open' or not.
		 *  @param {node} nTr the table row to check
		 *  @returns {boolean} true if the row is currently open, false otherwise
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable;
		 *
		 *      // 'open' an information row when a row is clicked on
		 *      $('#example tbody tr').click( function () {
		 *        if ( oTable.fnIsOpen(this) ) {
		 *          oTable.fnClose( this );
		 *        } else {
		 *          oTable.fnOpen( this, "Temporary row opened", "info_row" );
		 *        }
		 *      } );
		 *
		 *      oTable = $('#example').dataTable();
		 *    } );
		 */
		this.fnIsOpen = function( nTr )
		{
			return this.api( true ).row( nTr ).child.isShown();
		};
		
		
		/**
		 * This function will place a new row directly after a row which is currently
		 * on display on the page, with the HTML contents that is passed into the
		 * function. This can be used, for example, to ask for confirmation that a
		 * particular record should be deleted.
		 *  @param {node} nTr The table row to 'open'
		 *  @param {string|node|jQuery} mHtml The HTML to put into the row
		 *  @param {string} sClass Class to give the new TD cell
		 *  @returns {node} The row opened. Note that if the table row passed in as the
		 *    first parameter, is not found in the table, this method will silently
		 *    return.
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable;
		 *
		 *      // 'open' an information row when a row is clicked on
		 *      $('#example tbody tr').click( function () {
		 *        if ( oTable.fnIsOpen(this) ) {
		 *          oTable.fnClose( this );
		 *        } else {
		 *          oTable.fnOpen( this, "Temporary row opened", "info_row" );
		 *        }
		 *      } );
		 *
		 *      oTable = $('#example').dataTable();
		 *    } );
		 */
		this.fnOpen = function( nTr, mHtml, sClass )
		{
			return this.api( true )
				.row( nTr )
				.child( mHtml, sClass )
				.show()
				.child()[0];
		};
		
		
		/**
		 * Change the pagination - provides the internal logic for pagination in a simple API
		 * function. With this function you can have a DataTables table go to the next,
		 * previous, first or last pages.
		 *  @param {string|int} mAction Paging action to take: "first", "previous", "next" or "last"
		 *    or page number to jump to (integer), note that page 0 is the first page.
		 *  @param {bool} [bRedraw=true] Redraw the table or not
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *      oTable.fnPageChange( 'next' );
		 *    } );
		 */
		this.fnPageChange = function ( mAction, bRedraw )
		{
			var api = this.api( true ).page( mAction );
		
			if ( bRedraw === undefined || bRedraw ) {
				api.draw(false);
			}
		};
		
		
		/**
		 * Show a particular column
		 *  @param {int} iCol The column whose display should be changed
		 *  @param {bool} bShow Show (true) or hide (false) the column
		 *  @param {bool} [bRedraw=true] Redraw the table or not
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Hide the second column after initialisation
		 *      oTable.fnSetColumnVis( 1, false );
		 *    } );
		 */
		this.fnSetColumnVis = function ( iCol, bShow, bRedraw )
		{
			var api = this.api( true ).column( iCol ).visible( bShow );
		
			if ( bRedraw === undefined || bRedraw ) {
				api.columns.adjust().draw();
			}
		};
		
		
		/**
		 * Get the settings for a particular table for external manipulation
		 *  @returns {object} DataTables settings object. See
		 *    {@link DataTable.models.oSettings}
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *      var oSettings = oTable.fnSettings();
		 *
		 *      // Show an example parameter from the settings
		 *      alert( oSettings._iDisplayStart );
		 *    } );
		 */
		this.fnSettings = function()
		{
			return _fnSettingsFromNode( this[_ext.iApiIndex] );
		};
		
		
		/**
		 * Sort the table by a particular column
		 *  @param {int} iCol the data index to sort on. Note that this will not match the
		 *    'display index' if you have hidden data entries
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Sort immediately with columns 0 and 1
		 *      oTable.fnSort( [ [0,'asc'], [1,'asc'] ] );
		 *    } );
		 */
		this.fnSort = function( aaSort )
		{
			this.api( true ).order( aaSort ).draw();
		};
		
		
		/**
		 * Attach a sort listener to an element for a given column
		 *  @param {node} nNode the element to attach the sort listener to
		 *  @param {int} iColumn the column that a click on this node will sort on
		 *  @param {function} [fnCallback] callback function when sort is run
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *
		 *      // Sort on column 1, when 'sorter' is clicked on
		 *      oTable.fnSortListener( document.getElementById('sorter'), 1 );
		 *    } );
		 */
		this.fnSortListener = function( nNode, iColumn, fnCallback )
		{
			this.api( true ).order.listener( nNode, iColumn, fnCallback );
		};
		
		
		/**
		 * Update a table cell or row - this method will accept either a single value to
		 * update the cell with, an array of values with one element for each column or
		 * an object in the same format as the original data source. The function is
		 * self-referencing in order to make the multi column updates easier.
		 *  @param {object|array|string} mData Data to update the cell/row with
		 *  @param {node|int} mRow TR element you want to update or the aoData index
		 *  @param {int} [iColumn] The column to update, give as null or undefined to
		 *    update a whole row.
		 *  @param {bool} [bRedraw=true] Redraw the table or not
		 *  @param {bool} [bAction=true] Perform pre-draw actions or not
		 *  @returns {int} 0 on success, 1 on error
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *      oTable.fnUpdate( 'Example update', 0, 0 ); // Single cell
		 *      oTable.fnUpdate( ['a', 'b', 'c', 'd', 'e'], $('tbody tr')[0] ); // Row
		 *    } );
		 */
		this.fnUpdate = function( mData, mRow, iColumn, bRedraw, bAction )
		{
			var api = this.api( true );
		
			if ( iColumn === undefined || iColumn === null ) {
				api.row( mRow ).data( mData );
			}
			else {
				api.cell( mRow, iColumn ).data( mData );
			}
		
			if ( bAction === undefined || bAction ) {
				api.columns.adjust();
			}
		
			if ( bRedraw === undefined || bRedraw ) {
				api.draw();
			}
			return 0;
		};
		
		
		/**
		 * Provide a common method for plug-ins to check the version of DataTables being used, in order
		 * to ensure compatibility.
		 *  @param {string} sVersion Version string to check for, in the format "X.Y.Z". Note that the
		 *    formats "X" and "X.Y" are also acceptable.
		 *  @returns {boolean} true if this version of DataTables is greater or equal to the required
		 *    version, or false if this version of DataTales is not suitable
		 *  @method
		 *  @dtopt API
		 *  @deprecated Since v1.10
		 *
		 *  @example
		 *    $(document).ready(function() {
		 *      var oTable = $('#example').dataTable();
		 *      alert( oTable.fnVersionCheck( '1.9.0' ) );
		 *    } );
		 */
		this.fnVersionCheck = _ext.fnVersionCheck;
		

		var _that = this;
		var emptyInit = options === undefined;
		var len = this.length;

		if ( emptyInit ) {
			options = {};
		}

		this.oApi = this.internal = _ext.internal;

		// Extend with old style plug-in API methods
		for ( var fn in DataTable.ext.internal ) {
			if ( fn ) {
				this[fn] = _fnExternApiFunc(fn);
			}
		}

		this.each(function() {
			// For each initialisation we want to give it a clean initialisation
			// object that can be bashed around
			var o = {};
			var oInit = len > 1 ? // optimisation for single table case
				_fnExtend( o, options, true ) :
				options;

			/*global oInit,_that,emptyInit*/
			var i=0, iLen, j, jLen, k, kLen;
			var sId = this.getAttribute( 'id' );
			var bInitHandedOff = false;
			var defaults = DataTable.defaults;
			var $this = $(this);
			
			
			/* Sanity check */
			if ( this.nodeName.toLowerCase() != 'table' )
			{
				_fnLog( null, 0, 'Non-table node initialisation ('+this.nodeName+')', 2 );
				return;
			}
			
			/* Backwards compatibility for the defaults */
			_fnCompatOpts( defaults );
			_fnCompatCols( defaults.column );
			
			/* Convert the camel-case defaults to Hungarian */
			_fnCamelToHungarian( defaults, defaults, true );
			_fnCamelToHungarian( defaults.column, defaults.column, true );
			
			/* Setting up the initialisation object */
			_fnCamelToHungarian( defaults, $.extend( oInit, $this.data() ), true );
			
			
			
			/* Check to see if we are re-initialising a table */
			var allSettings = DataTable.settings;
			for ( i=0, iLen=allSettings.length ; i<iLen ; i++ )
			{
				var s = allSettings[i];
			
				/* Base check on table node */
				if (
					s.nTable == this ||
					(s.nTHead && s.nTHead.parentNode == this) ||
					(s.nTFoot && s.nTFoot.parentNode == this)
				) {
					var bRetrieve = oInit.bRetrieve !== undefined ? oInit.bRetrieve : defaults.bRetrieve;
					var bDestroy = oInit.bDestroy !== undefined ? oInit.bDestroy : defaults.bDestroy;
			
					if ( emptyInit || bRetrieve )
					{
						return s.oInstance;
					}
					else if ( bDestroy )
					{
						s.oInstance.fnDestroy();
						break;
					}
					else
					{
						_fnLog( s, 0, 'Cannot reinitialise DataTable', 3 );
						return;
					}
				}
			
				/* If the element we are initialising has the same ID as a table which was previously
				 * initialised, but the table nodes don't match (from before) then we destroy the old
				 * instance by simply deleting it. This is under the assumption that the table has been
				 * destroyed by other methods. Anyone using non-id selectors will need to do this manually
				 */
				if ( s.sTableId == this.id )
				{
					allSettings.splice( i, 1 );
					break;
				}
			}
			
			/* Ensure the table has an ID - required for accessibility */
			if ( sId === null || sId === "" )
			{
				sId = "DataTables_Table_"+(DataTable.ext._unique++);
				this.id = sId;
			}
			
			/* Create the settings object for this table and set some of the default parameters */
			var oSettings = $.extend( true, {}, DataTable.models.oSettings, {
				"sDestroyWidth": $this[0].style.width,
				"sInstance":     sId,
				"sTableId":      sId
			} );
			oSettings.nTable = this;
			oSettings.oApi   = _that.internal;
			oSettings.oInit  = oInit;
			
			allSettings.push( oSettings );
			
			// Need to add the instance after the instance after the settings object has been added
			// to the settings array, so we can self reference the table instance if more than one
			oSettings.oInstance = (_that.length===1) ? _that : $this.dataTable();
			
			// Backwards compatibility, before we apply all the defaults
			_fnCompatOpts( oInit );
			_fnLanguageCompat( oInit.oLanguage );
			
			// If the length menu is given, but the init display length is not, use the length menu
			if ( oInit.aLengthMenu && ! oInit.iDisplayLength )
			{
				oInit.iDisplayLength = Array.isArray( oInit.aLengthMenu[0] ) ?
					oInit.aLengthMenu[0][0] : oInit.aLengthMenu[0];
			}
			
			// Apply the defaults and init options to make a single init object will all
			// options defined from defaults and instance options.
			oInit = _fnExtend( $.extend( true, {}, defaults ), oInit );
			
			
			// Map the initialisation options onto the settings object
			_fnMap( oSettings.oFeatures, oInit, [
				"bPaginate",
				"bLengthChange",
				"bFilter",
				"bSort",
				"bSortMulti",
				"bInfo",
				"bProcessing",
				"bAutoWidth",
				"bSortClasses",
				"bServerSide",
				"bDeferRender"
			] );
			_fnMap( oSettings, oInit, [
				"asStripeClasses",
				"ajax",
				"fnServerData",
				"fnFormatNumber",
				"sServerMethod",
				"aaSorting",
				"aaSortingFixed",
				"aLengthMenu",
				"sPaginationType",
				"sAjaxSource",
				"sAjaxDataProp",
				"iStateDuration",
				"sDom",
				"bSortCellsTop",
				"iTabIndex",
				"fnStateLoadCallback",
				"fnStateSaveCallback",
				"renderer",
				"searchDelay",
				"rowId",
				[ "iCookieDuration", "iStateDuration" ], // backwards compat
				[ "oSearch", "oPreviousSearch" ],
				[ "aoSearchCols", "aoPreSearchCols" ],
				[ "iDisplayLength", "_iDisplayLength" ]
			] );
			_fnMap( oSettings.oScroll, oInit, [
				[ "sScrollX", "sX" ],
				[ "sScrollXInner", "sXInner" ],
				[ "sScrollY", "sY" ],
				[ "bScrollCollapse", "bCollapse" ]
			] );
			_fnMap( oSettings.oLanguage, oInit, "fnInfoCallback" );
			
			/* Callback functions which are array driven */
			_fnCallbackReg( oSettings, 'aoDrawCallback',       oInit.fnDrawCallback,      'user' );
			_fnCallbackReg( oSettings, 'aoServerParams',       oInit.fnServerParams,      'user' );
			_fnCallbackReg( oSettings, 'aoStateSaveParams',    oInit.fnStateSaveParams,   'user' );
			_fnCallbackReg( oSettings, 'aoStateLoadParams',    oInit.fnStateLoadParams,   'user' );
			_fnCallbackReg( oSettings, 'aoStateLoaded',        oInit.fnStateLoaded,       'user' );
			_fnCallbackReg( oSettings, 'aoRowCallback',        oInit.fnRowCallback,       'user' );
			_fnCallbackReg( oSettings, 'aoRowCreatedCallback', oInit.fnCreatedRow,        'user' );
			_fnCallbackReg( oSettings, 'aoHeaderCallback',     oInit.fnHeaderCallback,    'user' );
			_fnCallbackReg( oSettings, 'aoFooterCallback',     oInit.fnFooterCallback,    'user' );
			_fnCallbackReg( oSettings, 'aoInitComplete',       oInit.fnInitComplete,      'user' );
			_fnCallbackReg( oSettings, 'aoPreDrawCallback',    oInit.fnPreDrawCallback,   'user' );
			
			oSettings.rowIdFn = _fnGetObjectDataFn( oInit.rowId );
			
			/* Browser support detection */
			_fnBrowserDetect( oSettings );
			
			var oClasses = oSettings.oClasses;
			
			$.extend( oClasses, DataTable.ext.classes, oInit.oClasses );
			$this.addClass( oClasses.sTable );
			
			
			if ( oSettings.iInitDisplayStart === undefined )
			{
				/* Display start point, taking into account the save saving */
				oSettings.iInitDisplayStart = oInit.iDisplayStart;
				oSettings._iDisplayStart = oInit.iDisplayStart;
			}
			
			if ( oInit.iDeferLoading !== null )
			{
				oSettings.bDeferLoading = true;
				var tmp = Array.isArray( oInit.iDeferLoading );
				oSettings._iRecordsDisplay = tmp ? oInit.iDeferLoading[0] : oInit.iDeferLoading;
				oSettings._iRecordsTotal = tmp ? oInit.iDeferLoading[1] : oInit.iDeferLoading;
			}
			
			/* Language definitions */
			var oLanguage = oSettings.oLanguage;
			$.extend( true, oLanguage, oInit.oLanguage );
			
			if ( oLanguage.sUrl )
			{
				/* Get the language definitions from a file - because this Ajax call makes the language
				 * get async to the remainder of this function we use bInitHandedOff to indicate that
				 * _fnInitialise will be fired by the returned Ajax handler, rather than the constructor
				 */
				$.ajax( {
					dataType: 'json',
					url: oLanguage.sUrl,
					success: function ( json ) {
						_fnLanguageCompat( json );
						_fnCamelToHungarian( defaults.oLanguage, json );
						$.extend( true, oLanguage, json );
			
						_fnCallbackFire( oSettings, null, 'i18n', [oSettings]);
						_fnInitialise( oSettings );
					},
					error: function () {
						// Error occurred loading language file, continue on as best we can
						_fnInitialise( oSettings );
					}
				} );
				bInitHandedOff = true;
			}
			else {
				_fnCallbackFire( oSettings, null, 'i18n', [oSettings]);
			}
			
			/*
			 * Stripes
			 */
			if ( oInit.asStripeClasses === null )
			{
				oSettings.asStripeClasses =[
					oClasses.sStripeOdd,
					oClasses.sStripeEven
				];
			}
			
			/* Remove row stripe classes if they are already on the table row */
			var stripeClasses = oSettings.asStripeClasses;
			var rowOne = $this.children('tbody').find('tr').eq(0);
			if ( $.inArray( true, $.map( stripeClasses, function(el, i) {
				return rowOne.hasClass(el);
			} ) ) !== -1 ) {
				$('tbody tr', this).removeClass( stripeClasses.join(' ') );
				oSettings.asDestroyStripes = stripeClasses.slice();
			}
			
			/*
			 * Columns
			 * See if we should load columns automatically or use defined ones
			 */
			var anThs = [];
			var aoColumnsInit;
			var nThead = this.getElementsByTagName('thead');
			if ( nThead.length !== 0 )
			{
				_fnDetectHeader( oSettings.aoHeader, nThead[0] );
				anThs = _fnGetUniqueThs( oSettings );
			}
			
			/* If not given a column array, generate one with nulls */
			if ( oInit.aoColumns === null )
			{
				aoColumnsInit = [];
				for ( i=0, iLen=anThs.length ; i<iLen ; i++ )
				{
					aoColumnsInit.push( null );
				}
			}
			else
			{
				aoColumnsInit = oInit.aoColumns;
			}
			
			/* Add the columns */
			for ( i=0, iLen=aoColumnsInit.length ; i<iLen ; i++ )
			{
				_fnAddColumn( oSettings, anThs ? anThs[i] : null );
			}
			
			/* Apply the column definitions */
			_fnApplyColumnDefs( oSettings, oInit.aoColumnDefs, aoColumnsInit, function (iCol, oDef) {
				_fnColumnOptions( oSettings, iCol, oDef );
			} );
			
			/* HTML5 attribute detection - build an mData object automatically if the
			 * attributes are found
			 */
			if ( rowOne.length ) {
				var a = function ( cell, name ) {
					return cell.getAttribute( 'data-'+name ) !== null ? name : null;
				};
			
				$( rowOne[0] ).children('th, td').each( function (i, cell) {
					var col = oSettings.aoColumns[i];
			
					if ( col.mData === i ) {
						var sort = a( cell, 'sort' ) || a( cell, 'order' );
						var filter = a( cell, 'filter' ) || a( cell, 'search' );
			
						if ( sort !== null || filter !== null ) {
							col.mData = {
								_:      i+'.display',
								sort:   sort !== null   ? i+'.@data-'+sort   : undefined,
								type:   sort !== null   ? i+'.@data-'+sort   : undefined,
								filter: filter !== null ? i+'.@data-'+filter : undefined
							};
			
							_fnColumnOptions( oSettings, i );
						}
					}
				} );
			}
			
			var features = oSettings.oFeatures;
			var loadedInit = function () {
				/*
				 * Sorting
				 * @todo For modularisation (1.11) this needs to do into a sort start up handler
				 */
			
				// If aaSorting is not defined, then we use the first indicator in asSorting
				// in case that has been altered, so the default sort reflects that option
				if ( oInit.aaSorting === undefined ) {
					var sorting = oSettings.aaSorting;
					for ( i=0, iLen=sorting.length ; i<iLen ; i++ ) {
						sorting[i][1] = oSettings.aoColumns[ i ].asSorting[0];
					}
				}
			
				/* Do a first pass on the sorting classes (allows any size changes to be taken into
				 * account, and also will apply sorting disabled classes if disabled
				 */
				_fnSortingClasses( oSettings );
			
				if ( features.bSort ) {
					_fnCallbackReg( oSettings, 'aoDrawCallback', function () {
						if ( oSettings.bSorted ) {
							var aSort = _fnSortFlatten( oSettings );
							var sortedColumns = {};
			
							$.each( aSort, function (i, val) {
								sortedColumns[ val.src ] = val.dir;
							} );
			
							_fnCallbackFire( oSettings, null, 'order', [oSettings, aSort, sortedColumns] );
							_fnSortAria( oSettings );
						}
					} );
				}
			
				_fnCallbackReg( oSettings, 'aoDrawCallback', function () {
					if ( oSettings.bSorted || _fnDataSource( oSettings ) === 'ssp' || features.bDeferRender ) {
						_fnSortingClasses( oSettings );
					}
				}, 'sc' );
			
			
				/*
				 * Final init
				 * Cache the header, body and footer as required, creating them if needed
				 */
			
				// Work around for Webkit bug 83867 - store the caption-side before removing from doc
				var captions = $this.children('caption').each( function () {
					this._captionSide = $(this).css('caption-side');
				} );
			
				var thead = $this.children('thead');
				if ( thead.length === 0 ) {
					thead = $('<thead/>').appendTo($this);
				}
				oSettings.nTHead = thead[0];
			
				var tbody = $this.children('tbody');
				if ( tbody.length === 0 ) {
					tbody = $('<tbody/>').appendTo($this);
				}
				oSettings.nTBody = tbody[0];
			
				var tfoot = $this.children('tfoot');
				if ( tfoot.length === 0 && captions.length > 0 && (oSettings.oScroll.sX !== "" || oSettings.oScroll.sY !== "") ) {
					// If we are a scrolling table, and no footer has been given, then we need to create
					// a tfoot element for the caption element to be appended to
					tfoot = $('<tfoot/>').appendTo($this);
				}
			
				if ( tfoot.length === 0 || tfoot.children().length === 0 ) {
					$this.addClass( oClasses.sNoFooter );
				}
				else if ( tfoot.length > 0 ) {
					oSettings.nTFoot = tfoot[0];
					_fnDetectHeader( oSettings.aoFooter, oSettings.nTFoot );
				}
			
				/* Check if there is data passing into the constructor */
				if ( oInit.aaData ) {
					for ( i=0 ; i<oInit.aaData.length ; i++ ) {
						_fnAddData( oSettings, oInit.aaData[ i ] );
					}
				}
				else if ( oSettings.bDeferLoading || _fnDataSource( oSettings ) == 'dom' ) {
					/* Grab the data from the page - only do this when deferred loading or no Ajax
					 * source since there is no point in reading the DOM data if we are then going
					 * to replace it with Ajax data
					 */
					_fnAddTr( oSettings, $(oSettings.nTBody).children('tr') );
				}
			
				/* Copy the data index array */
				oSettings.aiDisplay = oSettings.aiDisplayMaster.slice();
			
				/* Initialisation complete - table can be drawn */
				oSettings.bInitialised = true;
			
				/* Check if we need to initialise the table (it might not have been handed off to the
				 * language processor)
				 */
				if ( bInitHandedOff === false ) {
					_fnInitialise( oSettings );
				}
			};
			
			/* Must be done after everything which can be overridden by the state saving! */
			if ( oInit.bStateSave )
			{
				features.bStateSave = true;
				_fnCallbackReg( oSettings, 'aoDrawCallback', _fnSaveState, 'state_save' );
				_fnLoadState( oSettings, oInit, loadedInit );
			}
			else {
				loadedInit();
			}
			
		} );
		_that = null;
		return this;
	};

	
	/*
	 * It is useful to have variables which are scoped locally so only the
	 * DataTables functions can access them and they don't leak into global space.
	 * At the same time these functions are often useful over multiple files in the
	 * core and API, so we list, or at least document, all variables which are used
	 * by DataTables as private variables here. This also ensures that there is no
	 * clashing of variable names and that they can easily referenced for reuse.
	 */
	
	
	// Defined else where
	//  _selector_run
	//  _selector_opts
	//  _selector_first
	//  _selector_row_indexes
	
	var _ext; // DataTable.ext
	var _Api; // DataTable.Api
	var _api_register; // DataTable.Api.register
	var _api_registerPlural; // DataTable.Api.registerPlural
	
	var _re_dic = {};
	var _re_new_lines = /[\r\n\u2028]/g;
	var _re_html = /<.*?>/g;
	
	// This is not strict ISO8601 - Date.parse() is quite lax, although
	// implementations differ between browsers.
	var _re_date = /^\d{2,4}[\.\/\-]\d{1,2}[\.\/\-]\d{1,2}([T ]{1}\d{1,2}[:\.]\d{2}([\.:]\d{2})?)?$/;
	
	// Escape regular expression special characters
	var _re_escape_regex = new RegExp( '(\\' + [ '/', '.', '*', '+', '?', '|', '(', ')', '[', ']', '{', '}', '\\', '$', '^', '-' ].join('|\\') + ')', 'g' );
	
	// http://en.wikipedia.org/wiki/Foreign_exchange_market
	// - \u20BD - Russian ruble.
	// - \u20a9 - South Korean Won
	// - \u20BA - Turkish Lira
	// - \u20B9 - Indian Rupee
	// - R - Brazil (R$) and South Africa
	// - fr - Swiss Franc
	// - kr - Swedish krona, Norwegian krone and Danish krone
	// - \u2009 is thin space and \u202F is narrow no-break space, both used in many
	// - Ƀ - Bitcoin
	// - Ξ - Ethereum
	//   standards as thousands separators.
	var _re_formatted_numeric = /['\u00A0,$£€¥%\u2009\u202F\u20BD\u20a9\u20BArfkɃΞ]/gi;
	
	
	var _empty = function ( d ) {
		return !d || d === true || d === '-' ? true : false;
	};
	
	
	var _intVal = function ( s ) {
		var integer = parseInt( s, 10 );
		return !isNaN(integer) && isFinite(s) ? integer : null;
	};
	
	// Convert from a formatted number with characters other than `.` as the
	// decimal place, to a Javascript number
	var _numToDecimal = function ( num, decimalPoint ) {
		// Cache created regular expressions for speed as this function is called often
		if ( ! _re_dic[ decimalPoint ] ) {
			_re_dic[ decimalPoint ] = new RegExp( _fnEscapeRegex( decimalPoint ), 'g' );
		}
		return typeof num === 'string' && decimalPoint !== '.' ?
			num.replace( /\./g, '' ).replace( _re_dic[ decimalPoint ], '.' ) :
			num;
	};
	
	
	var _isNumber = function ( d, decimalPoint, formatted ) {
		var strType = typeof d === 'string';
	
		// If empty return immediately so there must be a number if it is a
		// formatted string (this stops the string "k", or "kr", etc being detected
		// as a formatted number for currency
		if ( _empty( d ) ) {
			return true;
		}
	
		if ( decimalPoint && strType ) {
			d = _numToDecimal( d, decimalPoint );
		}
	
		if ( formatted && strType ) {
			d = d.replace( _re_formatted_numeric, '' );
		}
	
		return !isNaN( parseFloat(d) ) && isFinite( d );
	};
	
	
	// A string without HTML in it can be considered to be HTML still
	var _isHtml = function ( d ) {
		return _empty( d ) || typeof d === 'string';
	};
	
	
	var _htmlNumeric = function ( d, decimalPoint, formatted ) {
		if ( _empty( d ) ) {
			return true;
		}
	
		var html = _isHtml( d );
		return ! html ?
			null :
			_isNumber( _stripHtml( d ), decimalPoint, formatted ) ?
				true :
				null;
	};
	
	
	var _pluck = function ( a, prop, prop2 ) {
		var out = [];
		var i=0, ien=a.length;
	
		// Could have the test in the loop for slightly smaller code, but speed
		// is essential here
		if ( prop2 !== undefined ) {
			for ( ; i<ien ; i++ ) {
				if ( a[i] && a[i][ prop ] ) {
					out.push( a[i][ prop ][ prop2 ] );
				}
			}
		}
		else {
			for ( ; i<ien ; i++ ) {
				if ( a[i] ) {
					out.push( a[i][ prop ] );
				}
			}
		}
	
		return out;
	};
	
	
	// Basically the same as _pluck, but rather than looping over `a` we use `order`
	// as the indexes to pick from `a`
	var _pluck_order = function ( a, order, prop, prop2 )
	{
		var out = [];
		var i=0, ien=order.length;
	
		// Could have the test in the loop for slightly smaller code, but speed
		// is essential here
		if ( prop2 !== undefined ) {
			for ( ; i<ien ; i++ ) {
				if ( a[ order[i] ][ prop ] ) {
					out.push( a[ order[i] ][ prop ][ prop2 ] );
				}
			}
		}
		else {
			for ( ; i<ien ; i++ ) {
				out.push( a[ order[i] ][ prop ] );
			}
		}
	
		return out;
	};
	
	
	var _range = function ( len, start )
	{
		var out = [];
		var end;
	
		if ( start === undefined ) {
			start = 0;
			end = len;
		}
		else {
			end = start;
			start = len;
		}
	
		for ( var i=start ; i<end ; i++ ) {
			out.push( i );
		}
	
		return out;
	};
	
	
	var _removeEmpty = function ( a )
	{
		var out = [];
	
		for ( var i=0, ien=a.length ; i<ien ; i++ ) {
			if ( a[i] ) { // careful - will remove all falsy values!
				out.push( a[i] );
			}
		}
	
		return out;
	};
	
	
	var _stripHtml = function ( d ) {
		return d.replace( _re_html, '' );
	};
	
	
	/**
	 * Determine if all values in the array are unique. This means we can short
	 * cut the _unique method at the cost of a single loop. A sorted array is used
	 * to easily check the values.
	 *
	 * @param  {array} src Source array
	 * @return {boolean} true if all unique, false otherwise
	 * @ignore
	 */
	var _areAllUnique = function ( src ) {
		if ( src.length < 2 ) {
			return true;
		}
	
		var sorted = src.slice().sort();
		var last = sorted[0];
	
		for ( var i=1, ien=sorted.length ; i<ien ; i++ ) {
			if ( sorted[i] === last ) {
				return false;
			}
	
			last = sorted[i];
		}
	
		return true;
	};
	
	
	/**
	 * Find the unique elements in a source array.
	 *
	 * @param  {array} src Source array
	 * @return {array} Array of unique items
	 * @ignore
	 */
	var _unique = function ( src )
	{
		if ( _areAllUnique( src ) ) {
			return src.slice();
		}
	
		// A faster unique method is to use object keys to identify used values,
		// but this doesn't work with arrays or objects, which we must also
		// consider. See jsperf.com/compare-array-unique-versions/4 for more
		// information.
		var
			out = [],
			val,
			i, ien=src.length,
			j, k=0;
	
		again: for ( i=0 ; i<ien ; i++ ) {
			val = src[i];
	
			for ( j=0 ; j<k ; j++ ) {
				if ( out[j] === val ) {
					continue again;
				}
			}
	
			out.push( val );
			k++;
		}
	
		return out;
	};
	
	// Surprisingly this is faster than [].concat.apply
	// https://jsperf.com/flatten-an-array-loop-vs-reduce/2
	var _flatten = function (out, val) {
		if (Array.isArray(val)) {
			for (var i=0 ; i<val.length ; i++) {
				_flatten(out, val[i]);
			}
		}
		else {
			out.push(val);
		}
	  
		return out;
	}
	
	// Array.isArray polyfill.
	// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/isArray
	if (! Array.isArray) {
	    Array.isArray = function(arg) {
	        return Object.prototype.toString.call(arg) === '[object Array]';
	    };
	}
	
	// .trim() polyfill
	// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/String/trim
	if (!String.prototype.trim) {
	  String.prototype.trim = function () {
	    return this.replace(/^[\s\uFEFF\xA0]+|[\s\uFEFF\xA0]+$/g, '');
	  };
	}
	
	/**
	 * DataTables utility methods
	 * 
	 * This namespace provides helper methods that DataTables uses internally to
	 * create a DataTable, but which are not exclusively used only for DataTables.
	 * These methods can be used by extension authors to save the duplication of
	 * code.
	 *
	 *  @namespace
	 */
	DataTable.util = {
		/**
		 * Throttle the calls to a function. Arguments and context are maintained
		 * for the throttled function.
		 *
		 * @param {function} fn Function to be called
		 * @param {integer} freq Call frequency in mS
		 * @return {function} Wrapped function
		 */
		throttle: function ( fn, freq ) {
			var
				frequency = freq !== undefined ? freq : 200,
				last,
				timer;
	
			return function () {
				var
					that = this,
					now  = +new Date(),
					args = arguments;
	
				if ( last && now < last + frequency ) {
					clearTimeout( timer );
	
					timer = setTimeout( function () {
						last = undefined;
						fn.apply( that, args );
					}, frequency );
				}
				else {
					last = now;
					fn.apply( that, args );
				}
			};
		},
	
	
		/**
		 * Escape a string such that it can be used in a regular expression
		 *
		 *  @param {string} val string to escape
		 *  @returns {string} escaped string
		 */
		escapeRegex: function ( val ) {
			return val.replace( _re_escape_regex, '\\$1' );
		}
	};
	
	
	
	/**
	 * Create a mapping object that allows camel case parameters to be looked up
	 * for their Hungarian counterparts. The mapping is stored in a private
	 * parameter called `_hungarianMap` which can be accessed on the source object.
	 *  @param {object} o
	 *  @memberof DataTable#oApi
	 */
	function _fnHungarianMap ( o )
	{
		var
			hungarian = 'a aa ai ao as b fn i m o s ',
			match,
			newKey,
			map = {};
	
		$.each( o, function (key, val) {
			match = key.match(/^([^A-Z]+?)([A-Z])/);
	
			if ( match && hungarian.indexOf(match[1]+' ') !== -1 )
			{
				newKey = key.replace( match[0], match[2].toLowerCase() );
				map[ newKey ] = key;
	
				if ( match[1] === 'o' )
				{
					_fnHungarianMap( o[key] );
				}
			}
		} );
	
		o._hungarianMap = map;
	}
	
	
	/**
	 * Convert from camel case parameters to Hungarian, based on a Hungarian map
	 * created by _fnHungarianMap.
	 *  @param {object} src The model object which holds all parameters that can be
	 *    mapped.
	 *  @param {object} user The object to convert from camel case to Hungarian.
	 *  @param {boolean} force When set to `true`, properties which already have a
	 *    Hungarian value in the `user` object will be overwritten. Otherwise they
	 *    won't be.
	 *  @memberof DataTable#oApi
	 */
	function _fnCamelToHungarian ( src, user, force )
	{
		if ( ! src._hungarianMap ) {
			_fnHungarianMap( src );
		}
	
		var hungarianKey;
	
		$.each( user, function (key, val) {
			hungarianKey = src._hungarianMap[ key ];
	
			if ( hungarianKey !== undefined && (force || user[hungarianKey] === undefined) )
			{
				// For objects, we need to buzz down into the object to copy parameters
				if ( hungarianKey.charAt(0) === 'o' )
				{
					// Copy the camelCase options over to the hungarian
					if ( ! user[ hungarianKey ] ) {
						user[ hungarianKey ] = {};
					}
					$.extend( true, user[hungarianKey], user[key] );
	
					_fnCamelToHungarian( src[hungarianKey], user[hungarianKey], force );
				}
				else {
					user[hungarianKey] = user[ key ];
				}
			}
		} );
	}
	
	
	/**
	 * Language compatibility - when certain options are given, and others aren't, we
	 * need to duplicate the values over, in order to provide backwards compatibility
	 * with older language files.
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnLanguageCompat( lang )
	{
		// Note the use of the Hungarian notation for the parameters in this method as
		// this is called after the mapping of camelCase to Hungarian
		var defaults = DataTable.defaults.oLanguage;
	
		// Default mapping
		var defaultDecimal = defaults.sDecimal;
		if ( defaultDecimal ) {
			_addNumericSort( defaultDecimal );
		}
	
		if ( lang ) {
			var zeroRecords = lang.sZeroRecords;
	
			// Backwards compatibility - if there is no sEmptyTable given, then use the same as
			// sZeroRecords - assuming that is given.
			if ( ! lang.sEmptyTable && zeroRecords &&
				defaults.sEmptyTable === "No data available in table" )
			{
				_fnMap( lang, lang, 'sZeroRecords', 'sEmptyTable' );
			}
	
			// Likewise with loading records
			if ( ! lang.sLoadingRecords && zeroRecords &&
				defaults.sLoadingRecords === "Loading..." )
			{
				_fnMap( lang, lang, 'sZeroRecords', 'sLoadingRecords' );
			}
	
			// Old parameter name of the thousands separator mapped onto the new
			if ( lang.sInfoThousands ) {
				lang.sThousands = lang.sInfoThousands;
			}
	
			var decimal = lang.sDecimal;
			if ( decimal && defaultDecimal !== decimal ) {
				_addNumericSort( decimal );
			}
		}
	}
	
	
	/**
	 * Map one parameter onto another
	 *  @param {object} o Object to map
	 *  @param {*} knew The new parameter name
	 *  @param {*} old The old parameter name
	 */
	var _fnCompatMap = function ( o, knew, old ) {
		if ( o[ knew ] !== undefined ) {
			o[ old ] = o[ knew ];
		}
	};
	
	
	/**
	 * Provide backwards compatibility for the main DT options. Note that the new
	 * options are mapped onto the old parameters, so this is an external interface
	 * change only.
	 *  @param {object} init Object to map
	 */
	function _fnCompatOpts ( init )
	{
		_fnCompatMap( init, 'ordering',      'bSort' );
		_fnCompatMap( init, 'orderMulti',    'bSortMulti' );
		_fnCompatMap( init, 'orderClasses',  'bSortClasses' );
		_fnCompatMap( init, 'orderCellsTop', 'bSortCellsTop' );
		_fnCompatMap( init, 'order',         'aaSorting' );
		_fnCompatMap( init, 'orderFixed',    'aaSortingFixed' );
		_fnCompatMap( init, 'paging',        'bPaginate' );
		_fnCompatMap( init, 'pagingType',    'sPaginationType' );
		_fnCompatMap( init, 'pageLength',    'iDisplayLength' );
		_fnCompatMap( init, 'searching',     'bFilter' );
	
		// Boolean initialisation of x-scrolling
		if ( typeof init.sScrollX === 'boolean' ) {
			init.sScrollX = init.sScrollX ? '100%' : '';
		}
		if ( typeof init.scrollX === 'boolean' ) {
			init.scrollX = init.scrollX ? '100%' : '';
		}
	
		// Column search objects are in an array, so it needs to be converted
		// element by element
		var searchCols = init.aoSearchCols;
	
		if ( searchCols ) {
			for ( var i=0, ien=searchCols.length ; i<ien ; i++ ) {
				if ( searchCols[i] ) {
					_fnCamelToHungarian( DataTable.models.oSearch, searchCols[i] );
				}
			}
		}
	}
	
	
	/**
	 * Provide backwards compatibility for column options. Note that the new options
	 * are mapped onto the old parameters, so this is an external interface change
	 * only.
	 *  @param {object} init Object to map
	 */
	function _fnCompatCols ( init )
	{
		_fnCompatMap( init, 'orderable',     'bSortable' );
		_fnCompatMap( init, 'orderData',     'aDataSort' );
		_fnCompatMap( init, 'orderSequence', 'asSorting' );
		_fnCompatMap( init, 'orderDataType', 'sortDataType' );
	
		// orderData can be given as an integer
		var dataSort = init.aDataSort;
		if ( typeof dataSort === 'number' && ! Array.isArray( dataSort ) ) {
			init.aDataSort = [ dataSort ];
		}
	}
	
	
	/**
	 * Browser feature detection for capabilities, quirks
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnBrowserDetect( settings )
	{
		// We don't need to do this every time DataTables is constructed, the values
		// calculated are specific to the browser and OS configuration which we
		// don't expect to change between initialisations
		if ( ! DataTable.__browser ) {
			var browser = {};
			DataTable.__browser = browser;
	
			// Scrolling feature / quirks detection
			var n = $('<div/>')
				.css( {
					position: 'fixed',
					top: 0,
					left: $(window).scrollLeft()*-1, // allow for scrolling
					height: 1,
					width: 1,
					overflow: 'hidden'
				} )
				.append(
					$('<div/>')
						.css( {
							position: 'absolute',
							top: 1,
							left: 1,
							width: 100,
							overflow: 'scroll'
						} )
						.append(
							$('<div/>')
								.css( {
									width: '100%',
									height: 10
								} )
						)
				)
				.appendTo( 'body' );
	
			var outer = n.children();
			var inner = outer.children();
	
			// Numbers below, in order, are:
			// inner.offsetWidth, inner.clientWidth, outer.offsetWidth, outer.clientWidth
			//
			// IE6 XP:                           100 100 100  83
			// IE7 Vista:                        100 100 100  83
			// IE 8+ Windows:                     83  83 100  83
			// Evergreen Windows:                 83  83 100  83
			// Evergreen Mac with scrollbars:     85  85 100  85
			// Evergreen Mac without scrollbars: 100 100 100 100
	
			// Get scrollbar width
			browser.barWidth = outer[0].offsetWidth - outer[0].clientWidth;
	
			// IE6/7 will oversize a width 100% element inside a scrolling element, to
			// include the width of the scrollbar, while other browsers ensure the inner
			// element is contained without forcing scrolling
			browser.bScrollOversize = inner[0].offsetWidth === 100 && outer[0].clientWidth !== 100;
	
			// In rtl text layout, some browsers (most, but not all) will place the
			// scrollbar on the left, rather than the right.
			browser.bScrollbarLeft = Math.round( inner.offset().left ) !== 1;
	
			// IE8- don't provide height and width for getBoundingClientRect
			browser.bBounding = n[0].getBoundingClientRect().width ? true : false;
	
			n.remove();
		}
	
		$.extend( settings.oBrowser, DataTable.__browser );
		settings.oScroll.iBarWidth = DataTable.__browser.barWidth;
	}
	
	
	/**
	 * Array.prototype reduce[Right] method, used for browsers which don't support
	 * JS 1.6. Done this way to reduce code size, since we iterate either way
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnReduce ( that, fn, init, start, end, inc )
	{
		var
			i = start,
			value,
			isSet = false;
	
		if ( init !== undefined ) {
			value = init;
			isSet = true;
		}
	
		while ( i !== end ) {
			if ( ! that.hasOwnProperty(i) ) {
				continue;
			}
	
			value = isSet ?
				fn( value, that[i], i, that ) :
				that[i];
	
			isSet = true;
			i += inc;
		}
	
		return value;
	}
	
	/**
	 * Add a column to the list used for the table with default values
	 *  @param {object} oSettings dataTables settings object
	 *  @param {node} nTh The th element for this column
	 *  @memberof DataTable#oApi
	 */
	function _fnAddColumn( oSettings, nTh )
	{
		// Add column to aoColumns array
		var oDefaults = DataTable.defaults.column;
		var iCol = oSettings.aoColumns.length;
		var oCol = $.extend( {}, DataTable.models.oColumn, oDefaults, {
			"nTh": nTh ? nTh : document.createElement('th'),
			"sTitle":    oDefaults.sTitle    ? oDefaults.sTitle    : nTh ? nTh.innerHTML : '',
			"aDataSort": oDefaults.aDataSort ? oDefaults.aDataSort : [iCol],
			"mData": oDefaults.mData ? oDefaults.mData : iCol,
			idx: iCol
		} );
		oSettings.aoColumns.push( oCol );
	
		// Add search object for column specific search. Note that the `searchCols[ iCol ]`
		// passed into extend can be undefined. This allows the user to give a default
		// with only some of the parameters defined, and also not give a default
		var searchCols = oSettings.aoPreSearchCols;
		searchCols[ iCol ] = $.extend( {}, DataTable.models.oSearch, searchCols[ iCol ] );
	
		// Use the default column options function to initialise classes etc
		_fnColumnOptions( oSettings, iCol, $(nTh).data() );
	}
	
	
	/**
	 * Apply options for a column
	 *  @param {object} oSettings dataTables settings object
	 *  @param {int} iCol column index to consider
	 *  @param {object} oOptions object with sType, bVisible and bSearchable etc
	 *  @memberof DataTable#oApi
	 */
	function _fnColumnOptions( oSettings, iCol, oOptions )
	{
		var oCol = oSettings.aoColumns[ iCol ];
		var oClasses = oSettings.oClasses;
		var th = $(oCol.nTh);
	
		// Try to get width information from the DOM. We can't get it from CSS
		// as we'd need to parse the CSS stylesheet. `width` option can override
		if ( ! oCol.sWidthOrig ) {
			// Width attribute
			oCol.sWidthOrig = th.attr('width') || null;
	
			// Style attribute
			var t = (th.attr('style') || '').match(/width:\s*(\d+[pxem%]+)/);
			if ( t ) {
				oCol.sWidthOrig = t[1];
			}
		}
	
		/* User specified column options */
		if ( oOptions !== undefined && oOptions !== null )
		{
			// Backwards compatibility
			_fnCompatCols( oOptions );
	
			// Map camel case parameters to their Hungarian counterparts
			_fnCamelToHungarian( DataTable.defaults.column, oOptions, true );
	
			/* Backwards compatibility for mDataProp */
			if ( oOptions.mDataProp !== undefined && !oOptions.mData )
			{
				oOptions.mData = oOptions.mDataProp;
			}
	
			if ( oOptions.sType )
			{
				oCol._sManualType = oOptions.sType;
			}
	
			// `class` is a reserved word in Javascript, so we need to provide
			// the ability to use a valid name for the camel case input
			if ( oOptions.className && ! oOptions.sClass )
			{
				oOptions.sClass = oOptions.className;
			}
			if ( oOptions.sClass ) {
				th.addClass( oOptions.sClass );
			}
	
			$.extend( oCol, oOptions );
			_fnMap( oCol, oOptions, "sWidth", "sWidthOrig" );
	
			/* iDataSort to be applied (backwards compatibility), but aDataSort will take
			 * priority if defined
			 */
			if ( oOptions.iDataSort !== undefined )
			{
				oCol.aDataSort = [ oOptions.iDataSort ];
			}
			_fnMap( oCol, oOptions, "aDataSort" );
		}
	
		/* Cache the data get and set functions for speed */
		var mDataSrc = oCol.mData;
		var mData = _fnGetObjectDataFn( mDataSrc );
		var mRender = oCol.mRender ? _fnGetObjectDataFn( oCol.mRender ) : null;
	
		var attrTest = function( src ) {
			return typeof src === 'string' && src.indexOf('@') !== -1;
		};
		oCol._bAttrSrc = $.isPlainObject( mDataSrc ) && (
			attrTest(mDataSrc.sort) || attrTest(mDataSrc.type) || attrTest(mDataSrc.filter)
		);
		oCol._setter = null;
	
		oCol.fnGetData = function (rowData, type, meta) {
			var innerData = mData( rowData, type, undefined, meta );
	
			return mRender && type ?
				mRender( innerData, type, rowData, meta ) :
				innerData;
		};
		oCol.fnSetData = function ( rowData, val, meta ) {
			return _fnSetObjectDataFn( mDataSrc )( rowData, val, meta );
		};
	
		// Indicate if DataTables should read DOM data as an object or array
		// Used in _fnGetRowElements
		if ( typeof mDataSrc !== 'number' ) {
			oSettings._rowReadObject = true;
		}
	
		/* Feature sorting overrides column specific when off */
		if ( !oSettings.oFeatures.bSort )
		{
			oCol.bSortable = false;
			th.addClass( oClasses.sSortableNone ); // Have to add class here as order event isn't called
		}
	
		/* Check that the class assignment is correct for sorting */
		var bAsc = $.inArray('asc', oCol.asSorting) !== -1;
		var bDesc = $.inArray('desc', oCol.asSorting) !== -1;
		if ( !oCol.bSortable || (!bAsc && !bDesc) )
		{
			oCol.sSortingClass = oClasses.sSortableNone;
			oCol.sSortingClassJUI = "";
		}
		else if ( bAsc && !bDesc )
		{
			oCol.sSortingClass = oClasses.sSortableAsc;
			oCol.sSortingClassJUI = oClasses.sSortJUIAscAllowed;
		}
		else if ( !bAsc && bDesc )
		{
			oCol.sSortingClass = oClasses.sSortableDesc;
			oCol.sSortingClassJUI = oClasses.sSortJUIDescAllowed;
		}
		else
		{
			oCol.sSortingClass = oClasses.sSortable;
			oCol.sSortingClassJUI = oClasses.sSortJUI;
		}
	}
	
	
	/**
	 * Adjust the table column widths for new data. Note: you would probably want to
	 * do a redraw after calling this function!
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnAdjustColumnSizing ( settings )
	{
		/* Not interested in doing column width calculation if auto-width is disabled */
		if ( settings.oFeatures.bAutoWidth !== false )
		{
			var columns = settings.aoColumns;
	
			_fnCalculateColumnWidths( settings );
			for ( var i=0 , iLen=columns.length ; i<iLen ; i++ )
			{
				columns[i].nTh.style.width = columns[i].sWidth;
			}
		}
	
		var scroll = settings.oScroll;
		if ( scroll.sY !== '' || scroll.sX !== '')
		{
			_fnScrollDraw( settings );
		}
	
		_fnCallbackFire( settings, null, 'column-sizing', [settings] );
	}
	
	
	/**
	 * Covert the index of a visible column to the index in the data array (take account
	 * of hidden columns)
	 *  @param {object} oSettings dataTables settings object
	 *  @param {int} iMatch Visible column index to lookup
	 *  @returns {int} i the data index
	 *  @memberof DataTable#oApi
	 */
	function _fnVisibleToColumnIndex( oSettings, iMatch )
	{
		var aiVis = _fnGetColumns( oSettings, 'bVisible' );
	
		return typeof aiVis[iMatch] === 'number' ?
			aiVis[iMatch] :
			null;
	}
	
	
	/**
	 * Covert the index of an index in the data array and convert it to the visible
	 *   column index (take account of hidden columns)
	 *  @param {int} iMatch Column index to lookup
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {int} i the data index
	 *  @memberof DataTable#oApi
	 */
	function _fnColumnIndexToVisible( oSettings, iMatch )
	{
		var aiVis = _fnGetColumns( oSettings, 'bVisible' );
		var iPos = $.inArray( iMatch, aiVis );
	
		return iPos !== -1 ? iPos : null;
	}
	
	
	/**
	 * Get the number of visible columns
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {int} i the number of visible columns
	 *  @memberof DataTable#oApi
	 */
	function _fnVisbleColumns( oSettings )
	{
		var vis = 0;
	
		// No reduce in IE8, use a loop for now
		$.each( oSettings.aoColumns, function ( i, col ) {
			if ( col.bVisible && $(col.nTh).css('display') !== 'none' ) {
				vis++;
			}
		} );
	
		return vis;
	}
	
	
	/**
	 * Get an array of column indexes that match a given property
	 *  @param {object} oSettings dataTables settings object
	 *  @param {string} sParam Parameter in aoColumns to look for - typically
	 *    bVisible or bSearchable
	 *  @returns {array} Array of indexes with matched properties
	 *  @memberof DataTable#oApi
	 */
	function _fnGetColumns( oSettings, sParam )
	{
		var a = [];
	
		$.map( oSettings.aoColumns, function(val, i) {
			if ( val[sParam] ) {
				a.push( i );
			}
		} );
	
		return a;
	}
	
	
	/**
	 * Calculate the 'type' of a column
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnColumnTypes ( settings )
	{
		var columns = settings.aoColumns;
		var data = settings.aoData;
		var types = DataTable.ext.type.detect;
		var i, ien, j, jen, k, ken;
		var col, cell, detectedType, cache;
	
		// For each column, spin over the 
		for ( i=0, ien=columns.length ; i<ien ; i++ ) {
			col = columns[i];
			cache = [];
	
			if ( ! col.sType && col._sManualType ) {
				col.sType = col._sManualType;
			}
			else if ( ! col.sType ) {
				for ( j=0, jen=types.length ; j<jen ; j++ ) {
					for ( k=0, ken=data.length ; k<ken ; k++ ) {
						// Use a cache array so we only need to get the type data
						// from the formatter once (when using multiple detectors)
						if ( cache[k] === undefined ) {
							cache[k] = _fnGetCellData( settings, k, i, 'type' );
						}
	
						detectedType = types[j]( cache[k], settings );
	
						// If null, then this type can't apply to this column, so
						// rather than testing all cells, break out. There is an
						// exception for the last type which is `html`. We need to
						// scan all rows since it is possible to mix string and HTML
						// types
						if ( ! detectedType && j !== types.length-1 ) {
							break;
						}
	
						// Only a single match is needed for html type since it is
						// bottom of the pile and very similar to string
						if ( detectedType === 'html' ) {
							break;
						}
					}
	
					// Type is valid for all data points in the column - use this
					// type
					if ( detectedType ) {
						col.sType = detectedType;
						break;
					}
				}
	
				// Fall back - if no type was detected, always use string
				if ( ! col.sType ) {
					col.sType = 'string';
				}
			}
		}
	}
	
	
	/**
	 * Take the column definitions and static columns arrays and calculate how
	 * they relate to column indexes. The callback function will then apply the
	 * definition found for a column to a suitable configuration object.
	 *  @param {object} oSettings dataTables settings object
	 *  @param {array} aoColDefs The aoColumnDefs array that is to be applied
	 *  @param {array} aoCols The aoColumns array that defines columns individually
	 *  @param {function} fn Callback function - takes two parameters, the calculated
	 *    column index and the definition for that column.
	 *  @memberof DataTable#oApi
	 */
	function _fnApplyColumnDefs( oSettings, aoColDefs, aoCols, fn )
	{
		var i, iLen, j, jLen, k, kLen, def;
		var columns = oSettings.aoColumns;
	
		// Column definitions with aTargets
		if ( aoColDefs )
		{
			/* Loop over the definitions array - loop in reverse so first instance has priority */
			for ( i=aoColDefs.length-1 ; i>=0 ; i-- )
			{
				def = aoColDefs[i];
	
				/* Each definition can target multiple columns, as it is an array */
				var aTargets = def.targets !== undefined ?
					def.targets :
					def.aTargets;
	
				if ( ! Array.isArray( aTargets ) )
				{
					aTargets = [ aTargets ];
				}
	
				for ( j=0, jLen=aTargets.length ; j<jLen ; j++ )
				{
					if ( typeof aTargets[j] === 'number' && aTargets[j] >= 0 )
					{
						/* Add columns that we don't yet know about */
						while( columns.length <= aTargets[j] )
						{
							_fnAddColumn( oSettings );
						}
	
						/* Integer, basic index */
						fn( aTargets[j], def );
					}
					else if ( typeof aTargets[j] === 'number' && aTargets[j] < 0 )
					{
						/* Negative integer, right to left column counting */
						fn( columns.length+aTargets[j], def );
					}
					else if ( typeof aTargets[j] === 'string' )
					{
						/* Class name matching on TH element */
						for ( k=0, kLen=columns.length ; k<kLen ; k++ )
						{
							if ( aTargets[j] == "_all" ||
							     $(columns[k].nTh).hasClass( aTargets[j] ) )
							{
								fn( k, def );
							}
						}
					}
				}
			}
		}
	
		// Statically defined columns array
		if ( aoCols )
		{
			for ( i=0, iLen=aoCols.length ; i<iLen ; i++ )
			{
				fn( i, aoCols[i] );
			}
		}
	}
	
	/**
	 * Add a data array to the table, creating DOM node etc. This is the parallel to
	 * _fnGatherData, but for adding rows from a Javascript source, rather than a
	 * DOM source.
	 *  @param {object} oSettings dataTables settings object
	 *  @param {array} aData data array to be added
	 *  @param {node} [nTr] TR element to add to the table - optional. If not given,
	 *    DataTables will create a row automatically
	 *  @param {array} [anTds] Array of TD|TH elements for the row - must be given
	 *    if nTr is.
	 *  @returns {int} >=0 if successful (index of new aoData entry), -1 if failed
	 *  @memberof DataTable#oApi
	 */
	function _fnAddData ( oSettings, aDataIn, nTr, anTds )
	{
		/* Create the object for storing information about this new row */
		var iRow = oSettings.aoData.length;
		var oData = $.extend( true, {}, DataTable.models.oRow, {
			src: nTr ? 'dom' : 'data',
			idx: iRow
		} );
	
		oData._aData = aDataIn;
		oSettings.aoData.push( oData );
	
		/* Create the cells */
		var nTd, sThisType;
		var columns = oSettings.aoColumns;
	
		// Invalidate the column types as the new data needs to be revalidated
		for ( var i=0, iLen=columns.length ; i<iLen ; i++ )
		{
			columns[i].sType = null;
		}
	
		/* Add to the display array */
		oSettings.aiDisplayMaster.push( iRow );
	
		var id = oSettings.rowIdFn( aDataIn );
		if ( id !== undefined ) {
			oSettings.aIds[ id ] = oData;
		}
	
		/* Create the DOM information, or register it if already present */
		if ( nTr || ! oSettings.oFeatures.bDeferRender )
		{
			_fnCreateTr( oSettings, iRow, nTr, anTds );
		}
	
		return iRow;
	}
	
	
	/**
	 * Add one or more TR elements to the table. Generally we'd expect to
	 * use this for reading data from a DOM sourced table, but it could be
	 * used for an TR element. Note that if a TR is given, it is used (i.e.
	 * it is not cloned).
	 *  @param {object} settings dataTables settings object
	 *  @param {array|node|jQuery} trs The TR element(s) to add to the table
	 *  @returns {array} Array of indexes for the added rows
	 *  @memberof DataTable#oApi
	 */
	function _fnAddTr( settings, trs )
	{
		var row;
	
		// Allow an individual node to be passed in
		if ( ! (trs instanceof $) ) {
			trs = $(trs);
		}
	
		return trs.map( function (i, el) {
			row = _fnGetRowElements( settings, el );
			return _fnAddData( settings, row.data, el, row.cells );
		} );
	}
	
	
	/**
	 * Take a TR element and convert it to an index in aoData
	 *  @param {object} oSettings dataTables settings object
	 *  @param {node} n the TR element to find
	 *  @returns {int} index if the node is found, null if not
	 *  @memberof DataTable#oApi
	 */
	function _fnNodeToDataIndex( oSettings, n )
	{
		return (n._DT_RowIndex!==undefined) ? n._DT_RowIndex : null;
	}
	
	
	/**
	 * Take a TD element and convert it into a column data index (not the visible index)
	 *  @param {object} oSettings dataTables settings object
	 *  @param {int} iRow The row number the TD/TH can be found in
	 *  @param {node} n The TD/TH element to find
	 *  @returns {int} index if the node is found, -1 if not
	 *  @memberof DataTable#oApi
	 */
	function _fnNodeToColumnIndex( oSettings, iRow, n )
	{
		return $.inArray( n, oSettings.aoData[ iRow ].anCells );
	}
	
	
	/**
	 * Get the data for a given cell from the internal cache, taking into account data mapping
	 *  @param {object} settings dataTables settings object
	 *  @param {int} rowIdx aoData row id
	 *  @param {int} colIdx Column index
	 *  @param {string} type data get type ('display', 'type' 'filter' 'sort')
	 *  @returns {*} Cell data
	 *  @memberof DataTable#oApi
	 */
	function _fnGetCellData( settings, rowIdx, colIdx, type )
	{
		var draw           = settings.iDraw;
		var col            = settings.aoColumns[colIdx];
		var rowData        = settings.aoData[rowIdx]._aData;
		var defaultContent = col.sDefaultContent;
		var cellData       = col.fnGetData( rowData, type, {
			settings: settings,
			row:      rowIdx,
			col:      colIdx
		} );
	
		if ( cellData === undefined ) {
			if ( settings.iDrawError != draw && defaultContent === null ) {
				_fnLog( settings, 0, "Requested unknown parameter "+
					(typeof col.mData=='function' ? '{function}' : "'"+col.mData+"'")+
					" for row "+rowIdx+", column "+colIdx, 4 );
				settings.iDrawError = draw;
			}
			return defaultContent;
		}
	
		// When the data source is null and a specific data type is requested (i.e.
		// not the original data), we can use default column data
		if ( (cellData === rowData || cellData === null) && defaultContent !== null && type !== undefined ) {
			cellData = defaultContent;
		}
		else if ( typeof cellData === 'function' ) {
			// If the data source is a function, then we run it and use the return,
			// executing in the scope of the data object (for instances)
			return cellData.call( rowData );
		}
	
		if ( cellData === null && type == 'display' ) {
			return '';
		}
		return cellData;
	}
	
	
	/**
	 * Set the value for a specific cell, into the internal data cache
	 *  @param {object} settings dataTables settings object
	 *  @param {int} rowIdx aoData row id
	 *  @param {int} colIdx Column index
	 *  @param {*} val Value to set
	 *  @memberof DataTable#oApi
	 */
	function _fnSetCellData( settings, rowIdx, colIdx, val )
	{
		var col     = settings.aoColumns[colIdx];
		var rowData = settings.aoData[rowIdx]._aData;
	
		col.fnSetData( rowData, val, {
			settings: settings,
			row:      rowIdx,
			col:      colIdx
		}  );
	}
	
	
	// Private variable that is used to match action syntax in the data property object
	var __reArray = /\[.*?\]$/;
	var __reFn = /\(\)$/;
	
	/**
	 * Split string on periods, taking into account escaped periods
	 * @param  {string} str String to split
	 * @return {array} Split string
	 */
	function _fnSplitObjNotation( str )
	{
		return $.map( str.match(/(\\.|[^\.])+/g) || [''], function ( s ) {
			return s.replace(/\\\./g, '.');
		} );
	}
	
	
	/**
	 * Return a function that can be used to get data from a source object, taking
	 * into account the ability to use nested objects as a source
	 *  @param {string|int|function} mSource The data source for the object
	 *  @returns {function} Data get function
	 *  @memberof DataTable#oApi
	 */
	function _fnGetObjectDataFn( mSource )
	{
		if ( $.isPlainObject( mSource ) )
		{
			/* Build an object of get functions, and wrap them in a single call */
			var o = {};
			$.each( mSource, function (key, val) {
				if ( val ) {
					o[key] = _fnGetObjectDataFn( val );
				}
			} );
	
			return function (data, type, row, meta) {
				var t = o[type] || o._;
				return t !== undefined ?
					t(data, type, row, meta) :
					data;
			};
		}
		else if ( mSource === null )
		{
			/* Give an empty string for rendering / sorting etc */
			return function (data) { // type, row and meta also passed, but not used
				return data;
			};
		}
		else if ( typeof mSource === 'function' )
		{
			return function (data, type, row, meta) {
				return mSource( data, type, row, meta );
			};
		}
		else if ( typeof mSource === 'string' && (mSource.indexOf('.') !== -1 ||
			      mSource.indexOf('[') !== -1 || mSource.indexOf('(') !== -1) )
		{
			/* If there is a . in the source string then the data source is in a
			 * nested object so we loop over the data for each level to get the next
			 * level down. On each loop we test for undefined, and if found immediately
			 * return. This allows entire objects to be missing and sDefaultContent to
			 * be used if defined, rather than throwing an error
			 */
			var fetchData = function (data, type, src) {
				var arrayNotation, funcNotation, out, innerSrc;
	
				if ( src !== "" )
				{
					var a = _fnSplitObjNotation( src );
	
					for ( var i=0, iLen=a.length ; i<iLen ; i++ )
					{
						// Check if we are dealing with special notation
						arrayNotation = a[i].match(__reArray);
						funcNotation = a[i].match(__reFn);
	
						if ( arrayNotation )
						{
							// Array notation
							a[i] = a[i].replace(__reArray, '');
	
							// Condition allows simply [] to be passed in
							if ( a[i] !== "" ) {
								data = data[ a[i] ];
							}
							out = [];
	
							// Get the remainder of the nested object to get
							a.splice( 0, i+1 );
							innerSrc = a.join('.');
	
							// Traverse each entry in the array getting the properties requested
							if ( Array.isArray( data ) ) {
								for ( var j=0, jLen=data.length ; j<jLen ; j++ ) {
									out.push( fetchData( data[j], type, innerSrc ) );
								}
							}
	
							// If a string is given in between the array notation indicators, that
							// is used to join the strings together, otherwise an array is returned
							var join = arrayNotation[0].substring(1, arrayNotation[0].length-1);
							data = (join==="") ? out : out.join(join);
	
							// The inner call to fetchData has already traversed through the remainder
							// of the source requested, so we exit from the loop
							break;
						}
						else if ( funcNotation )
						{
							// Function call
							a[i] = a[i].replace(__reFn, '');
							data = data[ a[i] ]();
							continue;
						}
	
						if ( data === null || data[ a[i] ] === undefined )
						{
							return undefined;
						}
						data = data[ a[i] ];
					}
				}
	
				return data;
			};
	
			return function (data, type) { // row and meta also passed, but not used
				return fetchData( data, type, mSource );
			};
		}
		else
		{
			/* Array or flat object mapping */
			return function (data, type) { // row and meta also passed, but not used
				return data[mSource];
			};
		}
	}
	
	
	/**
	 * Return a function that can be used to set data from a source object, taking
	 * into account the ability to use nested objects as a source
	 *  @param {string|int|function} mSource The data source for the object
	 *  @returns {function} Data set function
	 *  @memberof DataTable#oApi
	 */
	function _fnSetObjectDataFn( mSource )
	{
		if ( $.isPlainObject( mSource ) )
		{
			/* Unlike get, only the underscore (global) option is used for for
			 * setting data since we don't know the type here. This is why an object
			 * option is not documented for `mData` (which is read/write), but it is
			 * for `mRender` which is read only.
			 */
			return _fnSetObjectDataFn( mSource._ );
		}
		else if ( mSource === null )
		{
			/* Nothing to do when the data source is null */
			return function () {};
		}
		else if ( typeof mSource === 'function' )
		{
			return function (data, val, meta) {
				mSource( data, 'set', val, meta );
			};
		}
		else if ( typeof mSource === 'string' && (mSource.indexOf('.') !== -1 ||
			      mSource.indexOf('[') !== -1 || mSource.indexOf('(') !== -1) )
		{
			/* Like the get, we need to get data from a nested object */
			var setData = function (data, val, src) {
				var a = _fnSplitObjNotation( src ), b;
				var aLast = a[a.length-1];
				var arrayNotation, funcNotation, o, innerSrc;
	
				for ( var i=0, iLen=a.length-1 ; i<iLen ; i++ )
				{
					// Protect against prototype pollution
					if (a[i] === '__proto__' || a[i] === 'constructor') {
						throw new Error('Cannot set prototype values');
					}
	
					// Check if we are dealing with an array notation request
					arrayNotation = a[i].match(__reArray);
					funcNotation = a[i].match(__reFn);
	
					if ( arrayNotation )
					{
						a[i] = a[i].replace(__reArray, '');
						data[ a[i] ] = [];
	
						// Get the remainder of the nested object to set so we can recurse
						b = a.slice();
						b.splice( 0, i+1 );
						innerSrc = b.join('.');
	
						// Traverse each entry in the array setting the properties requested
						if ( Array.isArray( val ) )
						{
							for ( var j=0, jLen=val.length ; j<jLen ; j++ )
							{
								o = {};
								setData( o, val[j], innerSrc );
								data[ a[i] ].push( o );
							}
						}
						else
						{
							// We've been asked to save data to an array, but it
							// isn't array data to be saved. Best that can be done
							// is to just save the value.
							data[ a[i] ] = val;
						}
	
						// The inner call to setData has already traversed through the remainder
						// of the source and has set the data, thus we can exit here
						return;
					}
					else if ( funcNotation )
					{
						// Function call
						a[i] = a[i].replace(__reFn, '');
						data = data[ a[i] ]( val );
					}
	
					// If the nested object doesn't currently exist - since we are
					// trying to set the value - create it
					if ( data[ a[i] ] === null || data[ a[i] ] === undefined )
					{
						data[ a[i] ] = {};
					}
					data = data[ a[i] ];
				}
	
				// Last item in the input - i.e, the actual set
				if ( aLast.match(__reFn ) )
				{
					// Function call
					data = data[ aLast.replace(__reFn, '') ]( val );
				}
				else
				{
					// If array notation is used, we just want to strip it and use the property name
					// and assign the value. If it isn't used, then we get the result we want anyway
					data[ aLast.replace(__reArray, '') ] = val;
				}
			};
	
			return function (data, val) { // meta is also passed in, but not used
				return setData( data, val, mSource );
			};
		}
		else
		{
			/* Array or flat object mapping */
			return function (data, val) { // meta is also passed in, but not used
				data[mSource] = val;
			};
		}
	}
	
	
	/**
	 * Return an array with the full table data
	 *  @param {object} oSettings dataTables settings object
	 *  @returns array {array} aData Master data array
	 *  @memberof DataTable#oApi
	 */
	function _fnGetDataMaster ( settings )
	{
		return _pluck( settings.aoData, '_aData' );
	}
	
	
	/**
	 * Nuke the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnClearTable( settings )
	{
		settings.aoData.length = 0;
		settings.aiDisplayMaster.length = 0;
		settings.aiDisplay.length = 0;
		settings.aIds = {};
	}
	
	
	 /**
	 * Take an array of integers (index array) and remove a target integer (value - not
	 * the key!)
	 *  @param {array} a Index array to target
	 *  @param {int} iTarget value to find
	 *  @memberof DataTable#oApi
	 */
	function _fnDeleteIndex( a, iTarget, splice )
	{
		var iTargetIndex = -1;
	
		for ( var i=0, iLen=a.length ; i<iLen ; i++ )
		{
			if ( a[i] == iTarget )
			{
				iTargetIndex = i;
			}
			else if ( a[i] > iTarget )
			{
				a[i]--;
			}
		}
	
		if ( iTargetIndex != -1 && splice === undefined )
		{
			a.splice( iTargetIndex, 1 );
		}
	}
	
	
	/**
	 * Mark cached data as invalid such that a re-read of the data will occur when
	 * the cached data is next requested. Also update from the data source object.
	 *
	 * @param {object} settings DataTables settings object
	 * @param {int}    rowIdx   Row index to invalidate
	 * @param {string} [src]    Source to invalidate from: undefined, 'auto', 'dom'
	 *     or 'data'
	 * @param {int}    [colIdx] Column index to invalidate. If undefined the whole
	 *     row will be invalidated
	 * @memberof DataTable#oApi
	 *
	 * @todo For the modularisation of v1.11 this will need to become a callback, so
	 *   the sort and filter methods can subscribe to it. That will required
	 *   initialisation options for sorting, which is why it is not already baked in
	 */
	function _fnInvalidate( settings, rowIdx, src, colIdx )
	{
		var row = settings.aoData[ rowIdx ];
		var i, ien;
		var cellWrite = function ( cell, col ) {
			// This is very frustrating, but in IE if you just write directly
			// to innerHTML, and elements that are overwritten are GC'ed,
			// even if there is a reference to them elsewhere
			while ( cell.childNodes.length ) {
				cell.removeChild( cell.firstChild );
			}
	
			cell.innerHTML = _fnGetCellData( settings, rowIdx, col, 'display' );
		};
	
		// Are we reading last data from DOM or the data object?
		if ( src === 'dom' || ((! src || src === 'auto') && row.src === 'dom') ) {
			// Read the data from the DOM
			row._aData = _fnGetRowElements(
					settings, row, colIdx, colIdx === undefined ? undefined : row._aData
				)
				.data;
		}
		else {
			// Reading from data object, update the DOM
			var cells = row.anCells;
	
			if ( cells ) {
				if ( colIdx !== undefined ) {
					cellWrite( cells[colIdx], colIdx );
				}
				else {
					for ( i=0, ien=cells.length ; i<ien ; i++ ) {
						cellWrite( cells[i], i );
					}
				}
			}
		}
	
		// For both row and cell invalidation, the cached data for sorting and
		// filtering is nulled out
		row._aSortData = null;
		row._aFilterData = null;
	
		// Invalidate the type for a specific column (if given) or all columns since
		// the data might have changed
		var cols = settings.aoColumns;
		if ( colIdx !== undefined ) {
			cols[ colIdx ].sType = null;
		}
		else {
			for ( i=0, ien=cols.length ; i<ien ; i++ ) {
				cols[i].sType = null;
			}
	
			// Update DataTables special `DT_*` attributes for the row
			_fnRowAttributes( settings, row );
		}
	}
	
	
	/**
	 * Build a data source object from an HTML row, reading the contents of the
	 * cells that are in the row.
	 *
	 * @param {object} settings DataTables settings object
	 * @param {node|object} TR element from which to read data or existing row
	 *   object from which to re-read the data from the cells
	 * @param {int} [colIdx] Optional column index
	 * @param {array|object} [d] Data source object. If `colIdx` is given then this
	 *   parameter should also be given and will be used to write the data into.
	 *   Only the column in question will be written
	 * @returns {object} Object with two parameters: `data` the data read, in
	 *   document order, and `cells` and array of nodes (they can be useful to the
	 *   caller, so rather than needing a second traversal to get them, just return
	 *   them from here).
	 * @memberof DataTable#oApi
	 */
	function _fnGetRowElements( settings, row, colIdx, d )
	{
		var
			tds = [],
			td = row.firstChild,
			name, col, o, i=0, contents,
			columns = settings.aoColumns,
			objectRead = settings._rowReadObject;
	
		// Allow the data object to be passed in, or construct
		d = d !== undefined ?
			d :
			objectRead ?
				{} :
				[];
	
		var attr = function ( str, td  ) {
			if ( typeof str === 'string' ) {
				var idx = str.indexOf('@');
	
				if ( idx !== -1 ) {
					var attr = str.substring( idx+1 );
					var setter = _fnSetObjectDataFn( str );
					setter( d, td.getAttribute( attr ) );
				}
			}
		};
	
		// Read data from a cell and store into the data object
		var cellProcess = function ( cell ) {
			if ( colIdx === undefined || colIdx === i ) {
				col = columns[i];
				contents = (cell.innerHTML).trim();
	
				if ( col && col._bAttrSrc ) {
					var setter = _fnSetObjectDataFn( col.mData._ );
					setter( d, contents );
	
					attr( col.mData.sort, cell );
					attr( col.mData.type, cell );
					attr( col.mData.filter, cell );
				}
				else {
					// Depending on the `data` option for the columns the data can
					// be read to either an object or an array.
					if ( objectRead ) {
						if ( ! col._setter ) {
							// Cache the setter function
							col._setter = _fnSetObjectDataFn( col.mData );
						}
						col._setter( d, contents );
					}
					else {
						d[i] = contents;
					}
				}
			}
	
			i++;
		};
	
		if ( td ) {
			// `tr` element was passed in
			while ( td ) {
				name = td.nodeName.toUpperCase();
	
				if ( name == "TD" || name == "TH" ) {
					cellProcess( td );
					tds.push( td );
				}
	
				td = td.nextSibling;
			}
		}
		else {
			// Existing row object passed in
			tds = row.anCells;
	
			for ( var j=0, jen=tds.length ; j<jen ; j++ ) {
				cellProcess( tds[j] );
			}
		}
	
		// Read the ID from the DOM if present
		var rowNode = row.firstChild ? row : row.nTr;
	
		if ( rowNode ) {
			var id = rowNode.getAttribute( 'id' );
	
			if ( id ) {
				_fnSetObjectDataFn( settings.rowId )( d, id );
			}
		}
	
		return {
			data: d,
			cells: tds
		};
	}
	/**
	 * Create a new TR element (and it's TD children) for a row
	 *  @param {object} oSettings dataTables settings object
	 *  @param {int} iRow Row to consider
	 *  @param {node} [nTrIn] TR element to add to the table - optional. If not given,
	 *    DataTables will create a row automatically
	 *  @param {array} [anTds] Array of TD|TH elements for the row - must be given
	 *    if nTr is.
	 *  @memberof DataTable#oApi
	 */
	function _fnCreateTr ( oSettings, iRow, nTrIn, anTds )
	{
		var
			row = oSettings.aoData[iRow],
			rowData = row._aData,
			cells = [],
			nTr, nTd, oCol,
			i, iLen, create;
	
		if ( row.nTr === null )
		{
			nTr = nTrIn || document.createElement('tr');
	
			row.nTr = nTr;
			row.anCells = cells;
	
			/* Use a private property on the node to allow reserve mapping from the node
			 * to the aoData array for fast look up
			 */
			nTr._DT_RowIndex = iRow;
	
			/* Special parameters can be given by the data source to be used on the row */
			_fnRowAttributes( oSettings, row );
	
			/* Process each column */
			for ( i=0, iLen=oSettings.aoColumns.length ; i<iLen ; i++ )
			{
				oCol = oSettings.aoColumns[i];
				create = nTrIn ? false : true;
	
				nTd = create ? document.createElement( oCol.sCellType ) : anTds[i];
				nTd._DT_CellIndex = {
					row: iRow,
					column: i
				};
				
				cells.push( nTd );
	
				// Need to create the HTML if new, or if a rendering function is defined
				if ( create || ((oCol.mRender || oCol.mData !== i) &&
					 (!$.isPlainObject(oCol.mData) || oCol.mData._ !== i+'.display')
				)) {
					nTd.innerHTML = _fnGetCellData( oSettings, iRow, i, 'display' );
				}
	
				/* Add user defined class */
				if ( oCol.sClass )
				{
					nTd.className += ' '+oCol.sClass;
				}
	
				// Visibility - add or remove as required
				if ( oCol.bVisible && ! nTrIn )
				{
					nTr.appendChild( nTd );
				}
				else if ( ! oCol.bVisible && nTrIn )
				{
					nTd.parentNode.removeChild( nTd );
				}
	
				if ( oCol.fnCreatedCell )
				{
					oCol.fnCreatedCell.call( oSettings.oInstance,
						nTd, _fnGetCellData( oSettings, iRow, i ), rowData, iRow, i
					);
				}
			}
	
			_fnCallbackFire( oSettings, 'aoRowCreatedCallback', null, [nTr, rowData, iRow, cells] );
		}
	}
	
	
	/**
	 * Add attributes to a row based on the special `DT_*` parameters in a data
	 * source object.
	 *  @param {object} settings DataTables settings object
	 *  @param {object} DataTables row object for the row to be modified
	 *  @memberof DataTable#oApi
	 */
	function _fnRowAttributes( settings, row )
	{
		var tr = row.nTr;
		var data = row._aData;
	
		if ( tr ) {
			var id = settings.rowIdFn( data );
	
			if ( id ) {
				tr.id = id;
			}
	
			if ( data.DT_RowClass ) {
				// Remove any classes added by DT_RowClass before
				var a = data.DT_RowClass.split(' ');
				row.__rowc = row.__rowc ?
					_unique( row.__rowc.concat( a ) ) :
					a;
	
				$(tr)
					.removeClass( row.__rowc.join(' ') )
					.addClass( data.DT_RowClass );
			}
	
			if ( data.DT_RowAttr ) {
				$(tr).attr( data.DT_RowAttr );
			}
	
			if ( data.DT_RowData ) {
				$(tr).data( data.DT_RowData );
			}
		}
	}
	
	
	/**
	 * Create the HTML header for the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnBuildHead( oSettings )
	{
		var i, ien, cell, row, column;
		var thead = oSettings.nTHead;
		var tfoot = oSettings.nTFoot;
		var createHeader = $('th, td', thead).length === 0;
		var classes = oSettings.oClasses;
		var columns = oSettings.aoColumns;
	
		if ( createHeader ) {
			row = $('<tr/>').appendTo( thead );
		}
	
		for ( i=0, ien=columns.length ; i<ien ; i++ ) {
			column = columns[i];
			cell = $( column.nTh ).addClass( column.sClass );
	
			if ( createHeader ) {
				cell.appendTo( row );
			}
	
			// 1.11 move into sorting
			if ( oSettings.oFeatures.bSort ) {
				cell.addClass( column.sSortingClass );
	
				if ( column.bSortable !== false ) {
					cell
						.attr( 'tabindex', oSettings.iTabIndex )
						.attr( 'aria-controls', oSettings.sTableId );
	
					_fnSortAttachListener( oSettings, column.nTh, i );
				}
			}
	
			if ( column.sTitle != cell[0].innerHTML ) {
				cell.html( column.sTitle );
			}
	
			_fnRenderer( oSettings, 'header' )(
				oSettings, cell, column, classes
			);
		}
	
		if ( createHeader ) {
			_fnDetectHeader( oSettings.aoHeader, thead );
		}
		
		/* ARIA role for the rows */
		$(thead).children('tr').attr('role', 'row');
	
		/* Deal with the footer - add classes if required */
		$(thead).children('tr').children('th, td').addClass( classes.sHeaderTH );
		$(tfoot).children('tr').children('th, td').addClass( classes.sFooterTH );
	
		// Cache the footer cells. Note that we only take the cells from the first
		// row in the footer. If there is more than one row the user wants to
		// interact with, they need to use the table().foot() method. Note also this
		// allows cells to be used for multiple columns using colspan
		if ( tfoot !== null ) {
			var cells = oSettings.aoFooter[0];
	
			for ( i=0, ien=cells.length ; i<ien ; i++ ) {
				column = columns[i];
				column.nTf = cells[i].cell;
	
				if ( column.sClass ) {
					$(column.nTf).addClass( column.sClass );
				}
			}
		}
	}
	
	
	/**
	 * Draw the header (or footer) element based on the column visibility states. The
	 * methodology here is to use the layout array from _fnDetectHeader, modified for
	 * the instantaneous column visibility, to construct the new layout. The grid is
	 * traversed over cell at a time in a rows x columns grid fashion, although each
	 * cell insert can cover multiple elements in the grid - which is tracks using the
	 * aApplied array. Cell inserts in the grid will only occur where there isn't
	 * already a cell in that position.
	 *  @param {object} oSettings dataTables settings object
	 *  @param array {objects} aoSource Layout array from _fnDetectHeader
	 *  @param {boolean} [bIncludeHidden=false] If true then include the hidden columns in the calc,
	 *  @memberof DataTable#oApi
	 */
	function _fnDrawHead( oSettings, aoSource, bIncludeHidden )
	{
		var i, iLen, j, jLen, k, kLen, n, nLocalTr;
		var aoLocal = [];
		var aApplied = [];
		var iColumns = oSettings.aoColumns.length;
		var iRowspan, iColspan;
	
		if ( ! aoSource )
		{
			return;
		}
	
		if (  bIncludeHidden === undefined )
		{
			bIncludeHidden = false;
		}
	
		/* Make a copy of the master layout array, but without the visible columns in it */
		for ( i=0, iLen=aoSource.length ; i<iLen ; i++ )
		{
			aoLocal[i] = aoSource[i].slice();
			aoLocal[i].nTr = aoSource[i].nTr;
	
			/* Remove any columns which are currently hidden */
			for ( j=iColumns-1 ; j>=0 ; j-- )
			{
				if ( !oSettings.aoColumns[j].bVisible && !bIncludeHidden )
				{
					aoLocal[i].splice( j, 1 );
				}
			}
	
			/* Prep the applied array - it needs an element for each row */
			aApplied.push( [] );
		}
	
		for ( i=0, iLen=aoLocal.length ; i<iLen ; i++ )
		{
			nLocalTr = aoLocal[i].nTr;
	
			/* All cells are going to be replaced, so empty out the row */
			if ( nLocalTr )
			{
				while( (n = nLocalTr.firstChild) )
				{
					nLocalTr.removeChild( n );
				}
			}
	
			for ( j=0, jLen=aoLocal[i].length ; j<jLen ; j++ )
			{
				iRowspan = 1;
				iColspan = 1;
	
				/* Check to see if there is already a cell (row/colspan) covering our target
				 * insert point. If there is, then there is nothing to do.
				 */
				if ( aApplied[i][j] === undefined )
				{
					nLocalTr.appendChild( aoLocal[i][j].cell );
					aApplied[i][j] = 1;
	
					/* Expand the cell to cover as many rows as needed */
					while ( aoLocal[i+iRowspan] !== undefined &&
					        aoLocal[i][j].cell == aoLocal[i+iRowspan][j].cell )
					{
						aApplied[i+iRowspan][j] = 1;
						iRowspan++;
					}
	
					/* Expand the cell to cover as many columns as needed */
					while ( aoLocal[i][j+iColspan] !== undefined &&
					        aoLocal[i][j].cell == aoLocal[i][j+iColspan].cell )
					{
						/* Must update the applied array over the rows for the columns */
						for ( k=0 ; k<iRowspan ; k++ )
						{
							aApplied[i+k][j+iColspan] = 1;
						}
						iColspan++;
					}
	
					/* Do the actual expansion in the DOM */
					$(aoLocal[i][j].cell)
						.attr('rowspan', iRowspan)
						.attr('colspan', iColspan);
				}
			}
		}
	}
	
	
	/**
	 * Insert the required TR nodes into the table for display
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnDraw( oSettings )
	{
		/* Provide a pre-callback function which can be used to cancel the draw is false is returned */
		var aPreDraw = _fnCallbackFire( oSettings, 'aoPreDrawCallback', 'preDraw', [oSettings] );
		if ( $.inArray( false, aPreDraw ) !== -1 )
		{
			_fnProcessingDisplay( oSettings, false );
			return;
		}
	
		var i, iLen, n;
		var anRows = [];
		var iRowCount = 0;
		var asStripeClasses = oSettings.asStripeClasses;
		var iStripes = asStripeClasses.length;
		var iOpenRows = oSettings.aoOpenRows.length;
		var oLang = oSettings.oLanguage;
		var iInitDisplayStart = oSettings.iInitDisplayStart;
		var bServerSide = _fnDataSource( oSettings ) == 'ssp';
		var aiDisplay = oSettings.aiDisplay;
	
		oSettings.bDrawing = true;
	
		/* Check and see if we have an initial draw position from state saving */
		if ( iInitDisplayStart !== undefined && iInitDisplayStart !== -1 )
		{
			oSettings._iDisplayStart = bServerSide ?
				iInitDisplayStart :
				iInitDisplayStart >= oSettings.fnRecordsDisplay() ?
					0 :
					iInitDisplayStart;
	
			oSettings.iInitDisplayStart = -1;
		}
	
		var iDisplayStart = oSettings._iDisplayStart;
		var iDisplayEnd = oSettings.fnDisplayEnd();
	
		/* Server-side processing draw intercept */
		if ( oSettings.bDeferLoading )
		{
			oSettings.bDeferLoading = false;
			oSettings.iDraw++;
			_fnProcessingDisplay( oSettings, false );
		}
		else if ( !bServerSide )
		{
			oSettings.iDraw++;
		}
		else if ( !oSettings.bDestroying && !_fnAjaxUpdate( oSettings ) )
		{
			return;
		}
	
		if ( aiDisplay.length !== 0 )
		{
			var iStart = bServerSide ? 0 : iDisplayStart;
			var iEnd = bServerSide ? oSettings.aoData.length : iDisplayEnd;
	
			for ( var j=iStart ; j<iEnd ; j++ )
			{
				var iDataIndex = aiDisplay[j];
				var aoData = oSettings.aoData[ iDataIndex ];
				if ( aoData.nTr === null )
				{
					_fnCreateTr( oSettings, iDataIndex );
				}
	
				var nRow = aoData.nTr;
	
				/* Remove the old striping classes and then add the new one */
				if ( iStripes !== 0 )
				{
					var sStripe = asStripeClasses[ iRowCount % iStripes ];
					if ( aoData._sRowStripe != sStripe )
					{
						$(nRow).removeClass( aoData._sRowStripe ).addClass( sStripe );
						aoData._sRowStripe = sStripe;
					}
				}
	
				// Row callback functions - might want to manipulate the row
				// iRowCount and j are not currently documented. Are they at all
				// useful?
				_fnCallbackFire( oSettings, 'aoRowCallback', null,
					[nRow, aoData._aData, iRowCount, j, iDataIndex] );
	
				anRows.push( nRow );
				iRowCount++;
			}
		}
		else
		{
			/* Table is empty - create a row with an empty message in it */
			var sZero = oLang.sZeroRecords;
			if ( oSettings.iDraw == 1 &&  _fnDataSource( oSettings ) == 'ajax' )
			{
				sZero = oLang.sLoadingRecords;
			}
			else if ( oLang.sEmptyTable && oSettings.fnRecordsTotal() === 0 )
			{
				sZero = oLang.sEmptyTable;
			}
	
			anRows[ 0 ] = $( '<tr/>', { 'class': iStripes ? asStripeClasses[0] : '' } )
				.append( $('<td />', {
					'valign':  'top',
					'colSpan': _fnVisbleColumns( oSettings ),
					'class':   oSettings.oClasses.sRowEmpty
				} ).html( sZero ) )[0];
		}
	
		/* Header and footer callbacks */
		_fnCallbackFire( oSettings, 'aoHeaderCallback', 'header', [ $(oSettings.nTHead).children('tr')[0],
			_fnGetDataMaster( oSettings ), iDisplayStart, iDisplayEnd, aiDisplay ] );
	
		_fnCallbackFire( oSettings, 'aoFooterCallback', 'footer', [ $(oSettings.nTFoot).children('tr')[0],
			_fnGetDataMaster( oSettings ), iDisplayStart, iDisplayEnd, aiDisplay ] );
	
		var body = $(oSettings.nTBody);
	
		body.children().detach();
		body.append( $(anRows) );
	
		/* Call all required callback functions for the end of a draw */
		_fnCallbackFire( oSettings, 'aoDrawCallback', 'draw', [oSettings] );
	
		/* Draw is complete, sorting and filtering must be as well */
		oSettings.bSorted = false;
		oSettings.bFiltered = false;
		oSettings.bDrawing = false;
	}
	
	
	/**
	 * Redraw the table - taking account of the various features which are enabled
	 *  @param {object} oSettings dataTables settings object
	 *  @param {boolean} [holdPosition] Keep the current paging position. By default
	 *    the paging is reset to the first page
	 *  @memberof DataTable#oApi
	 */
	function _fnReDraw( settings, holdPosition )
	{
		var
			features = settings.oFeatures,
			sort     = features.bSort,
			filter   = features.bFilter;
	
		if ( sort ) {
			_fnSort( settings );
		}
	
		if ( filter ) {
			_fnFilterComplete( settings, settings.oPreviousSearch );
		}
		else {
			// No filtering, so we want to just use the display master
			settings.aiDisplay = settings.aiDisplayMaster.slice();
		}
	
		if ( holdPosition !== true ) {
			settings._iDisplayStart = 0;
		}
	
		// Let any modules know about the draw hold position state (used by
		// scrolling internally)
		settings._drawHold = holdPosition;
	
		_fnDraw( settings );
	
		settings._drawHold = false;
	}
	
	
	/**
	 * Add the options to the page HTML for the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnAddOptionsHtml ( oSettings )
	{
		var classes = oSettings.oClasses;
		var table = $(oSettings.nTable);
		var holding = $('<div/>').insertBefore( table ); // Holding element for speed
		var features = oSettings.oFeatures;
	
		// All DataTables are wrapped in a div
		var insert = $('<div/>', {
			id:      oSettings.sTableId+'_wrapper',
			'class': classes.sWrapper + (oSettings.nTFoot ? '' : ' '+classes.sNoFooter)
		} );
	
		oSettings.nHolding = holding[0];
		oSettings.nTableWrapper = insert[0];
		oSettings.nTableReinsertBefore = oSettings.nTable.nextSibling;
	
		/* Loop over the user set positioning and place the elements as needed */
		var aDom = oSettings.sDom.split('');
		var featureNode, cOption, nNewNode, cNext, sAttr, j;
		for ( var i=0 ; i<aDom.length ; i++ )
		{
			featureNode = null;
			cOption = aDom[i];
	
			if ( cOption == '<' )
			{
				/* New container div */
				nNewNode = $('<div/>')[0];
	
				/* Check to see if we should append an id and/or a class name to the container */
				cNext = aDom[i+1];
				if ( cNext == "'" || cNext == '"' )
				{
					sAttr = "";
					j = 2;
					while ( aDom[i+j] != cNext )
					{
						sAttr += aDom[i+j];
						j++;
					}
	
					/* Replace jQuery UI constants @todo depreciated */
					if ( sAttr == "H" )
					{
						sAttr = classes.sJUIHeader;
					}
					else if ( sAttr == "F" )
					{
						sAttr = classes.sJUIFooter;
					}
	
					/* The attribute can be in the format of "#id.class", "#id" or "class" This logic
					 * breaks the string into parts and applies them as needed
					 */
					if ( sAttr.indexOf('.') != -1 )
					{
						var aSplit = sAttr.split('.');
						nNewNode.id = aSplit[0].substr(1, aSplit[0].length-1);
						nNewNode.className = aSplit[1];
					}
					else if ( sAttr.charAt(0) == "#" )
					{
						nNewNode.id = sAttr.substr(1, sAttr.length-1);
					}
					else
					{
						nNewNode.className = sAttr;
					}
	
					i += j; /* Move along the position array */
				}
	
				insert.append( nNewNode );
				insert = $(nNewNode);
			}
			else if ( cOption == '>' )
			{
				/* End container div */
				insert = insert.parent();
			}
			// @todo Move options into their own plugins?
			else if ( cOption == 'l' && features.bPaginate && features.bLengthChange )
			{
				/* Length */
				featureNode = _fnFeatureHtmlLength( oSettings );
			}
			else if ( cOption == 'f' && features.bFilter )
			{
				/* Filter */
				featureNode = _fnFeatureHtmlFilter( oSettings );
			}
			else if ( cOption == 'r' && features.bProcessing )
			{
				/* pRocessing */
				featureNode = _fnFeatureHtmlProcessing( oSettings );
			}
			else if ( cOption == 't' )
			{
				/* Table */
				featureNode = _fnFeatureHtmlTable( oSettings );
			}
			else if ( cOption ==  'i' && features.bInfo )
			{
				/* Info */
				featureNode = _fnFeatureHtmlInfo( oSettings );
			}
			else if ( cOption == 'p' && features.bPaginate )
			{
				/* Pagination */
				featureNode = _fnFeatureHtmlPaginate( oSettings );
			}
			else if ( DataTable.ext.feature.length !== 0 )
			{
				/* Plug-in features */
				var aoFeatures = DataTable.ext.feature;
				for ( var k=0, kLen=aoFeatures.length ; k<kLen ; k++ )
				{
					if ( cOption == aoFeatures[k].cFeature )
					{
						featureNode = aoFeatures[k].fnInit( oSettings );
						break;
					}
				}
			}
	
			/* Add to the 2D features array */
			if ( featureNode )
			{
				var aanFeatures = oSettings.aanFeatures;
	
				if ( ! aanFeatures[cOption] )
				{
					aanFeatures[cOption] = [];
				}
	
				aanFeatures[cOption].push( featureNode );
				insert.append( featureNode );
			}
		}
	
		/* Built our DOM structure - replace the holding div with what we want */
		holding.replaceWith( insert );
		oSettings.nHolding = null;
	}
	
	
	/**
	 * Use the DOM source to create up an array of header cells. The idea here is to
	 * create a layout grid (array) of rows x columns, which contains a reference
	 * to the cell that that point in the grid (regardless of col/rowspan), such that
	 * any column / row could be removed and the new grid constructed
	 *  @param array {object} aLayout Array to store the calculated layout in
	 *  @param {node} nThead The header/footer element for the table
	 *  @memberof DataTable#oApi
	 */
	function _fnDetectHeader ( aLayout, nThead )
	{
		var nTrs = $(nThead).children('tr');
		var nTr, nCell;
		var i, k, l, iLen, jLen, iColShifted, iColumn, iColspan, iRowspan;
		var bUnique;
		var fnShiftCol = function ( a, i, j ) {
			var k = a[i];
	                while ( k[j] ) {
				j++;
			}
			return j;
		};
	
		aLayout.splice( 0, aLayout.length );
	
		/* We know how many rows there are in the layout - so prep it */
		for ( i=0, iLen=nTrs.length ; i<iLen ; i++ )
		{
			aLayout.push( [] );
		}
	
		/* Calculate a layout array */
		for ( i=0, iLen=nTrs.length ; i<iLen ; i++ )
		{
			nTr = nTrs[i];
			iColumn = 0;
	
			/* For every cell in the row... */
			nCell = nTr.firstChild;
			while ( nCell ) {
				if ( nCell.nodeName.toUpperCase() == "TD" ||
				     nCell.nodeName.toUpperCase() == "TH" )
				{
					/* Get the col and rowspan attributes from the DOM and sanitise them */
					iColspan = nCell.getAttribute('colspan') * 1;
					iRowspan = nCell.getAttribute('rowspan') * 1;
					iColspan = (!iColspan || iColspan===0 || iColspan===1) ? 1 : iColspan;
					iRowspan = (!iRowspan || iRowspan===0 || iRowspan===1) ? 1 : iRowspan;
	
					/* There might be colspan cells already in this row, so shift our target
					 * accordingly
					 */
					iColShifted = fnShiftCol( aLayout, i, iColumn );
	
					/* Cache calculation for unique columns */
					bUnique = iColspan === 1 ? true : false;
	
					/* If there is col / rowspan, copy the information into the layout grid */
					for ( l=0 ; l<iColspan ; l++ )
					{
						for ( k=0 ; k<iRowspan ; k++ )
						{
							aLayout[i+k][iColShifted+l] = {
								"cell": nCell,
								"unique": bUnique
							};
							aLayout[i+k].nTr = nTr;
						}
					}
				}
				nCell = nCell.nextSibling;
			}
		}
	}
	
	
	/**
	 * Get an array of unique th elements, one for each column
	 *  @param {object} oSettings dataTables settings object
	 *  @param {node} nHeader automatically detect the layout from this node - optional
	 *  @param {array} aLayout thead/tfoot layout from _fnDetectHeader - optional
	 *  @returns array {node} aReturn list of unique th's
	 *  @memberof DataTable#oApi
	 */
	function _fnGetUniqueThs ( oSettings, nHeader, aLayout )
	{
		var aReturn = [];
		if ( !aLayout )
		{
			aLayout = oSettings.aoHeader;
			if ( nHeader )
			{
				aLayout = [];
				_fnDetectHeader( aLayout, nHeader );
			}
		}
	
		for ( var i=0, iLen=aLayout.length ; i<iLen ; i++ )
		{
			for ( var j=0, jLen=aLayout[i].length ; j<jLen ; j++ )
			{
				if ( aLayout[i][j].unique &&
					 (!aReturn[j] || !oSettings.bSortCellsTop) )
				{
					aReturn[j] = aLayout[i][j].cell;
				}
			}
		}
	
		return aReturn;
	}
	
	/**
	 * Create an Ajax call based on the table's settings, taking into account that
	 * parameters can have multiple forms, and backwards compatibility.
	 *
	 * @param {object} oSettings dataTables settings object
	 * @param {array} data Data to send to the server, required by
	 *     DataTables - may be augmented by developer callbacks
	 * @param {function} fn Callback function to run when data is obtained
	 */
	function _fnBuildAjax( oSettings, data, fn )
	{
		// Compatibility with 1.9-, allow fnServerData and event to manipulate
		_fnCallbackFire( oSettings, 'aoServerParams', 'serverParams', [data] );
	
		// Convert to object based for 1.10+ if using the old array scheme which can
		// come from server-side processing or serverParams
		if ( data && Array.isArray(data) ) {
			var tmp = {};
			var rbracket = /(.*?)\[\]$/;
	
			$.each( data, function (key, val) {
				var match = val.name.match(rbracket);
	
				if ( match ) {
					// Support for arrays
					var name = match[0];
	
					if ( ! tmp[ name ] ) {
						tmp[ name ] = [];
					}
					tmp[ name ].push( val.value );
				}
				else {
					tmp[val.name] = val.value;
				}
			} );
			data = tmp;
		}
	
		var ajaxData;
		var ajax = oSettings.ajax;
		var instance = oSettings.oInstance;
		var callback = function ( json ) {
			_fnCallbackFire( oSettings, null, 'xhr', [oSettings, json, oSettings.jqXHR] );
			fn( json );
		};
	
		if ( $.isPlainObject( ajax ) && ajax.data )
		{
			ajaxData = ajax.data;
	
			var newData = typeof ajaxData === 'function' ?
				ajaxData( data, oSettings ) :  // fn can manipulate data or return
				ajaxData;                      // an object object or array to merge
	
			// If the function returned something, use that alone
			data = typeof ajaxData === 'function' && newData ?
				newData :
				$.extend( true, data, newData );
	
			// Remove the data property as we've resolved it already and don't want
			// jQuery to do it again (it is restored at the end of the function)
			delete ajax.data;
		}
	
		var baseAjax = {
			"data": data,
			"success": function (json) {
				var error = json.error || json.sError;
				if ( error ) {
					_fnLog( oSettings, 0, error );
				}
	
				oSettings.json = json;
				callback( json );
			},
			"dataType": "json",
			"cache": false,
			"type": oSettings.sServerMethod,
			"error": function (xhr, error, thrown) {
				var ret = _fnCallbackFire( oSettings, null, 'xhr', [oSettings, null, oSettings.jqXHR] );
	
				if ( $.inArray( true, ret ) === -1 ) {
					if ( error == "parsererror" ) {
						_fnLog( oSettings, 0, 'Invalid JSON response', 1 );
					}
					else if ( xhr.readyState === 4 ) {
						_fnLog( oSettings, 0, 'Ajax error', 7 );
					}
				}
	
				_fnProcessingDisplay( oSettings, false );
			}
		};
	
		// Store the data submitted for the API
		oSettings.oAjaxData = data;
	
		// Allow plug-ins and external processes to modify the data
		_fnCallbackFire( oSettings, null, 'preXhr', [oSettings, data] );
	
		if ( oSettings.fnServerData )
		{
			// DataTables 1.9- compatibility
			oSettings.fnServerData.call( instance,
				oSettings.sAjaxSource,
				$.map( data, function (val, key) { // Need to convert back to 1.9 trad format
					return { name: key, value: val };
				} ),
				callback,
				oSettings
			);
		}
		else if ( oSettings.sAjaxSource || typeof ajax === 'string' )
		{
			// DataTables 1.9- compatibility
			oSettings.jqXHR = $.ajax( $.extend( baseAjax, {
				url: ajax || oSettings.sAjaxSource
			} ) );
		}
		else if ( typeof ajax === 'function' )
		{
			// Is a function - let the caller define what needs to be done
			oSettings.jqXHR = ajax.call( instance, data, callback, oSettings );
		}
		else
		{
			// Object to extend the base settings
			oSettings.jqXHR = $.ajax( $.extend( baseAjax, ajax ) );
	
			// Restore for next time around
			ajax.data = ajaxData;
		}
	}
	
	
	/**
	 * Update the table using an Ajax call
	 *  @param {object} settings dataTables settings object
	 *  @returns {boolean} Block the table drawing or not
	 *  @memberof DataTable#oApi
	 */
	function _fnAjaxUpdate( settings )
	{
		if ( settings.bAjaxDataGet ) {
			settings.iDraw++;
			_fnProcessingDisplay( settings, true );
	
			_fnBuildAjax(
				settings,
				_fnAjaxParameters( settings ),
				function(json) {
					_fnAjaxUpdateDraw( settings, json );
				}
			);
	
			return false;
		}
		return true;
	}
	
	
	/**
	 * Build up the parameters in an object needed for a server-side processing
	 * request. Note that this is basically done twice, is different ways - a modern
	 * method which is used by default in DataTables 1.10 which uses objects and
	 * arrays, or the 1.9- method with is name / value pairs. 1.9 method is used if
	 * the sAjaxSource option is used in the initialisation, or the legacyAjax
	 * option is set.
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {bool} block the table drawing or not
	 *  @memberof DataTable#oApi
	 */
	function _fnAjaxParameters( settings )
	{
		var
			columns = settings.aoColumns,
			columnCount = columns.length,
			features = settings.oFeatures,
			preSearch = settings.oPreviousSearch,
			preColSearch = settings.aoPreSearchCols,
			i, data = [], dataProp, column, columnSearch,
			sort = _fnSortFlatten( settings ),
			displayStart = settings._iDisplayStart,
			displayLength = features.bPaginate !== false ?
				settings._iDisplayLength :
				-1;
	
		var param = function ( name, value ) {
			data.push( { 'name': name, 'value': value } );
		};
	
		// DataTables 1.9- compatible method
		param( 'sEcho',          settings.iDraw );
		param( 'iColumns',       columnCount );
		param( 'sColumns',       _pluck( columns, 'sName' ).join(',') );
		param( 'iDisplayStart',  displayStart );
		param( 'iDisplayLength', displayLength );
	
		// DataTables 1.10+ method
		var d = {
			draw:    settings.iDraw,
			columns: [],
			order:   [],
			start:   displayStart,
			length:  displayLength,
			search:  {
				value: preSearch.sSearch,
				regex: preSearch.bRegex
			}
		};
	
		for ( i=0 ; i<columnCount ; i++ ) {
			column = columns[i];
			columnSearch = preColSearch[i];
			dataProp = typeof column.mData=="function" ? 'function' : column.mData ;
	
			d.columns.push( {
				data:       dataProp,
				name:       column.sName,
				searchable: column.bSearchable,
				orderable:  column.bSortable,
				search:     {
					value: columnSearch.sSearch,
					regex: columnSearch.bRegex
				}
			} );
	
			param( "mDataProp_"+i, dataProp );
	
			if ( features.bFilter ) {
				param( 'sSearch_'+i,     columnSearch.sSearch );
				param( 'bRegex_'+i,      columnSearch.bRegex );
				param( 'bSearchable_'+i, column.bSearchable );
			}
	
			if ( features.bSort ) {
				param( 'bSortable_'+i, column.bSortable );
			}
		}
	
		if ( features.bFilter ) {
			param( 'sSearch', preSearch.sSearch );
			param( 'bRegex', preSearch.bRegex );
		}
	
		if ( features.bSort ) {
			$.each( sort, function ( i, val ) {
				d.order.push( { column: val.col, dir: val.dir } );
	
				param( 'iSortCol_'+i, val.col );
				param( 'sSortDir_'+i, val.dir );
			} );
	
			param( 'iSortingCols', sort.length );
		}
	
		// If the legacy.ajax parameter is null, then we automatically decide which
		// form to use, based on sAjaxSource
		var legacy = DataTable.ext.legacy.ajax;
		if ( legacy === null ) {
			return settings.sAjaxSource ? data : d;
		}
	
		// Otherwise, if legacy has been specified then we use that to decide on the
		// form
		return legacy ? data : d;
	}
	
	
	/**
	 * Data the data from the server (nuking the old) and redraw the table
	 *  @param {object} oSettings dataTables settings object
	 *  @param {object} json json data return from the server.
	 *  @param {string} json.sEcho Tracking flag for DataTables to match requests
	 *  @param {int} json.iTotalRecords Number of records in the data set, not accounting for filtering
	 *  @param {int} json.iTotalDisplayRecords Number of records in the data set, accounting for filtering
	 *  @param {array} json.aaData The data to display on this page
	 *  @param {string} [json.sColumns] Column ordering (sName, comma separated)
	 *  @memberof DataTable#oApi
	 */
	function _fnAjaxUpdateDraw ( settings, json )
	{
		// v1.10 uses camelCase variables, while 1.9 uses Hungarian notation.
		// Support both
		var compat = function ( old, modern ) {
			return json[old] !== undefined ? json[old] : json[modern];
		};
	
		var data = _fnAjaxDataSrc( settings, json );
		var draw            = compat( 'sEcho',                'draw' );
		var recordsTotal    = compat( 'iTotalRecords',        'recordsTotal' );
		var recordsFiltered = compat( 'iTotalDisplayRecords', 'recordsFiltered' );
	
		if ( draw !== undefined ) {
			// Protect against out of sequence returns
			if ( draw*1 < settings.iDraw ) {
				return;
			}
			settings.iDraw = draw * 1;
		}
	
		_fnClearTable( settings );
		settings._iRecordsTotal   = parseInt(recordsTotal, 10);
		settings._iRecordsDisplay = parseInt(recordsFiltered, 10);
	
		for ( var i=0, ien=data.length ; i<ien ; i++ ) {
			_fnAddData( settings, data[i] );
		}
		settings.aiDisplay = settings.aiDisplayMaster.slice();
	
		settings.bAjaxDataGet = false;
		_fnDraw( settings );
	
		if ( ! settings._bInitComplete ) {
			_fnInitComplete( settings, json );
		}
	
		settings.bAjaxDataGet = true;
		_fnProcessingDisplay( settings, false );
	}
	
	
	/**
	 * Get the data from the JSON data source to use for drawing a table. Using
	 * `_fnGetObjectDataFn` allows the data to be sourced from a property of the
	 * source object, or from a processing function.
	 *  @param {object} oSettings dataTables settings object
	 *  @param  {object} json Data source object / array from the server
	 *  @return {array} Array of data to use
	 */
	function _fnAjaxDataSrc ( oSettings, json )
	{
		var dataSrc = $.isPlainObject( oSettings.ajax ) && oSettings.ajax.dataSrc !== undefined ?
			oSettings.ajax.dataSrc :
			oSettings.sAjaxDataProp; // Compatibility with 1.9-.
	
		// Compatibility with 1.9-. In order to read from aaData, check if the
		// default has been changed, if not, check for aaData
		if ( dataSrc === 'data' ) {
			return json.aaData || json[dataSrc];
		}
	
		return dataSrc !== "" ?
			_fnGetObjectDataFn( dataSrc )( json ) :
			json;
	}
	
	/**
	 * Generate the node required for filtering text
	 *  @returns {node} Filter control element
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlFilter ( settings )
	{
		var classes = settings.oClasses;
		var tableId = settings.sTableId;
		var language = settings.oLanguage;
		var previousSearch = settings.oPreviousSearch;
		var features = settings.aanFeatures;
		var input = '<input type="search" class="'+classes.sFilterInput+'"/>';
	
		var str = language.sSearch;
		str = str.match(/_INPUT_/) ?
			str.replace('_INPUT_', input) :
			str+input;
	
		var filter = $('<div/>', {
				'id': ! features.f ? tableId+'_filter' : null,
				'class': classes.sFilter
			} )
			.append( $('<label/>' ).append( str ) );
	
		var searchFn = function() {
			/* Update all other filter input elements for the new display */
			var n = features.f;
			var val = !this.value ? "" : this.value; // mental IE8 fix :-(
	
			/* Now do the filter */
			if ( val != previousSearch.sSearch ) {
				_fnFilterComplete( settings, {
					"sSearch": val,
					"bRegex": previousSearch.bRegex,
					"bSmart": previousSearch.bSmart ,
					"bCaseInsensitive": previousSearch.bCaseInsensitive
				} );
	
				// Need to redraw, without resorting
				settings._iDisplayStart = 0;
				_fnDraw( settings );
			}
		};
	
		var searchDelay = settings.searchDelay !== null ?
			settings.searchDelay :
			_fnDataSource( settings ) === 'ssp' ?
				400 :
				0;
	
		var jqFilter = $('input', filter)
			.val( previousSearch.sSearch )
			.attr( 'placeholder', language.sSearchPlaceholder )
			.on(
				'keyup.DT search.DT input.DT paste.DT cut.DT',
				searchDelay ?
					_fnThrottle( searchFn, searchDelay ) :
					searchFn
			)
			.on( 'mouseup', function(e) {
				// Edge fix! Edge 17 does not trigger anything other than mouse events when clicking
				// on the clear icon (Edge bug 17584515). This is safe in other browsers as `searchFn`
				// checks the value to see if it has changed. In other browsers it won't have.
				setTimeout( function () {
					searchFn.call(jqFilter[0]);
				}, 10);
			} )
			.on( 'keypress.DT', function(e) {
				/* Prevent form submission */
				if ( e.keyCode == 13 ) {
					return false;
				}
			} )
			.attr('aria-controls', tableId);
	
		// Update the input elements whenever the table is filtered
		$(settings.nTable).on( 'search.dt.DT', function ( ev, s ) {
			if ( settings === s ) {
				// IE9 throws an 'unknown error' if document.activeElement is used
				// inside an iframe or frame...
				try {
					if ( jqFilter[0] !== document.activeElement ) {
						jqFilter.val( previousSearch.sSearch );
					}
				}
				catch ( e ) {}
			}
		} );
	
		return filter[0];
	}
	
	
	/**
	 * Filter the table using both the global filter and column based filtering
	 *  @param {object} oSettings dataTables settings object
	 *  @param {object} oSearch search information
	 *  @param {int} [iForce] force a research of the master array (1) or not (undefined or 0)
	 *  @memberof DataTable#oApi
	 */
	function _fnFilterComplete ( oSettings, oInput, iForce )
	{
		var oPrevSearch = oSettings.oPreviousSearch;
		var aoPrevSearch = oSettings.aoPreSearchCols;
		var fnSaveFilter = function ( oFilter ) {
			/* Save the filtering values */
			oPrevSearch.sSearch = oFilter.sSearch;
			oPrevSearch.bRegex = oFilter.bRegex;
			oPrevSearch.bSmart = oFilter.bSmart;
			oPrevSearch.bCaseInsensitive = oFilter.bCaseInsensitive;
		};
		var fnRegex = function ( o ) {
			// Backwards compatibility with the bEscapeRegex option
			return o.bEscapeRegex !== undefined ? !o.bEscapeRegex : o.bRegex;
		};
	
		// Resolve any column types that are unknown due to addition or invalidation
		// @todo As per sort - can this be moved into an event handler?
		_fnColumnTypes( oSettings );
	
		/* In server-side processing all filtering is done by the server, so no point hanging around here */
		if ( _fnDataSource( oSettings ) != 'ssp' )
		{
			/* Global filter */
			_fnFilter( oSettings, oInput.sSearch, iForce, fnRegex(oInput), oInput.bSmart, oInput.bCaseInsensitive );
			fnSaveFilter( oInput );
	
			/* Now do the individual column filter */
			for ( var i=0 ; i<aoPrevSearch.length ; i++ )
			{
				_fnFilterColumn( oSettings, aoPrevSearch[i].sSearch, i, fnRegex(aoPrevSearch[i]),
					aoPrevSearch[i].bSmart, aoPrevSearch[i].bCaseInsensitive );
			}
	
			/* Custom filtering */
			_fnFilterCustom( oSettings );
		}
		else
		{
			fnSaveFilter( oInput );
		}
	
		/* Tell the draw function we have been filtering */
		oSettings.bFiltered = true;
		_fnCallbackFire( oSettings, null, 'search', [oSettings] );
	}
	
	
	/**
	 * Apply custom filtering functions
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnFilterCustom( settings )
	{
		var filters = DataTable.ext.search;
		var displayRows = settings.aiDisplay;
		var row, rowIdx;
	
		for ( var i=0, ien=filters.length ; i<ien ; i++ ) {
			var rows = [];
	
			// Loop over each row and see if it should be included
			for ( var j=0, jen=displayRows.length ; j<jen ; j++ ) {
				rowIdx = displayRows[ j ];
				row = settings.aoData[ rowIdx ];
	
				if ( filters[i]( settings, row._aFilterData, rowIdx, row._aData, j ) ) {
					rows.push( rowIdx );
				}
			}
	
			// So the array reference doesn't break set the results into the
			// existing array
			displayRows.length = 0;
			$.merge( displayRows, rows );
		}
	}
	
	
	/**
	 * Filter the table on a per-column basis
	 *  @param {object} oSettings dataTables settings object
	 *  @param {string} sInput string to filter on
	 *  @param {int} iColumn column to filter
	 *  @param {bool} bRegex treat search string as a regular expression or not
	 *  @param {bool} bSmart use smart filtering or not
	 *  @param {bool} bCaseInsensitive Do case insenstive matching or not
	 *  @memberof DataTable#oApi
	 */
	function _fnFilterColumn ( settings, searchStr, colIdx, regex, smart, caseInsensitive )
	{
		if ( searchStr === '' ) {
			return;
		}
	
		var data;
		var out = [];
		var display = settings.aiDisplay;
		var rpSearch = _fnFilterCreateSearch( searchStr, regex, smart, caseInsensitive );
	
		for ( var i=0 ; i<display.length ; i++ ) {
			data = settings.aoData[ display[i] ]._aFilterData[ colIdx ];
	
			if ( rpSearch.test( data ) ) {
				out.push( display[i] );
			}
		}
	
		settings.aiDisplay = out;
	}
	
	
	/**
	 * Filter the data table based on user input and draw the table
	 *  @param {object} settings dataTables settings object
	 *  @param {string} input string to filter on
	 *  @param {int} force optional - force a research of the master array (1) or not (undefined or 0)
	 *  @param {bool} regex treat as a regular expression or not
	 *  @param {bool} smart perform smart filtering or not
	 *  @param {bool} caseInsensitive Do case insenstive matching or not
	 *  @memberof DataTable#oApi
	 */
	function _fnFilter( settings, input, force, regex, smart, caseInsensitive )
	{
		var rpSearch = _fnFilterCreateSearch( input, regex, smart, caseInsensitive );
		var prevSearch = settings.oPreviousSearch.sSearch;
		var displayMaster = settings.aiDisplayMaster;
		var display, invalidated, i;
		var filtered = [];
	
		// Need to take account of custom filtering functions - always filter
		if ( DataTable.ext.search.length !== 0 ) {
			force = true;
		}
	
		// Check if any of the rows were invalidated
		invalidated = _fnFilterData( settings );
	
		// If the input is blank - we just want the full data set
		if ( input.length <= 0 ) {
			settings.aiDisplay = displayMaster.slice();
		}
		else {
			// New search - start from the master array
			if ( invalidated ||
				 force ||
				 regex ||
				 prevSearch.length > input.length ||
				 input.indexOf(prevSearch) !== 0 ||
				 settings.bSorted // On resort, the display master needs to be
				                  // re-filtered since indexes will have changed
			) {
				settings.aiDisplay = displayMaster.slice();
			}
	
			// Search the display array
			display = settings.aiDisplay;
	
			for ( i=0 ; i<display.length ; i++ ) {
				if ( rpSearch.test( settings.aoData[ display[i] ]._sFilterRow ) ) {
					filtered.push( display[i] );
				}
			}
	
			settings.aiDisplay = filtered;
		}
	}
	
	
	/**
	 * Build a regular expression object suitable for searching a table
	 *  @param {string} sSearch string to search for
	 *  @param {bool} bRegex treat as a regular expression or not
	 *  @param {bool} bSmart perform smart filtering or not
	 *  @param {bool} bCaseInsensitive Do case insensitive matching or not
	 *  @returns {RegExp} constructed object
	 *  @memberof DataTable#oApi
	 */
	function _fnFilterCreateSearch( search, regex, smart, caseInsensitive )
	{
		search = regex ?
			search :
			_fnEscapeRegex( search );
		
		if ( smart ) {
			/* For smart filtering we want to allow the search to work regardless of
			 * word order. We also want double quoted text to be preserved, so word
			 * order is important - a la google. So this is what we want to
			 * generate:
			 * 
			 * ^(?=.*?\bone\b)(?=.*?\btwo three\b)(?=.*?\bfour\b).*$
			 */
			var a = $.map( search.match( /"[^"]+"|[^ ]+/g ) || [''], function ( word ) {
				if ( word.charAt(0) === '"' ) {
					var m = word.match( /^"(.*)"$/ );
					word = m ? m[1] : word;
				}
	
				return word.replace('"', '');
			} );
	
			search = '^(?=.*?'+a.join( ')(?=.*?' )+').*$';
		}
	
		return new RegExp( search, caseInsensitive ? 'i' : '' );
	}
	
	
	/**
	 * Escape a string such that it can be used in a regular expression
	 *  @param {string} sVal string to escape
	 *  @returns {string} escaped string
	 *  @memberof DataTable#oApi
	 */
	var _fnEscapeRegex = DataTable.util.escapeRegex;
	
	var __filter_div = $('<div>')[0];
	var __filter_div_textContent = __filter_div.textContent !== undefined;
	
	// Update the filtering data for each row if needed (by invalidation or first run)
	function _fnFilterData ( settings )
	{
		var columns = settings.aoColumns;
		var column;
		var i, j, ien, jen, filterData, cellData, row;
		var fomatters = DataTable.ext.type.search;
		var wasInvalidated = false;
	
		for ( i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
			row = settings.aoData[i];
	
			if ( ! row._aFilterData ) {
				filterData = [];
	
				for ( j=0, jen=columns.length ; j<jen ; j++ ) {
					column = columns[j];
	
					if ( column.bSearchable ) {
						cellData = _fnGetCellData( settings, i, j, 'filter' );
	
						if ( fomatters[ column.sType ] ) {
							cellData = fomatters[ column.sType ]( cellData );
						}
	
						// Search in DataTables 1.10 is string based. In 1.11 this
						// should be altered to also allow strict type checking.
						if ( cellData === null ) {
							cellData = '';
						}
	
						if ( typeof cellData !== 'string' && cellData.toString ) {
							cellData = cellData.toString();
						}
					}
					else {
						cellData = '';
					}
	
					// If it looks like there is an HTML entity in the string,
					// attempt to decode it so sorting works as expected. Note that
					// we could use a single line of jQuery to do this, but the DOM
					// method used here is much faster http://jsperf.com/html-decode
					if ( cellData.indexOf && cellData.indexOf('&') !== -1 ) {
						__filter_div.innerHTML = cellData;
						cellData = __filter_div_textContent ?
							__filter_div.textContent :
							__filter_div.innerText;
					}
	
					if ( cellData.replace ) {
						cellData = cellData.replace(/[\r\n\u2028]/g, '');
					}
	
					filterData.push( cellData );
				}
	
				row._aFilterData = filterData;
				row._sFilterRow = filterData.join('  ');
				wasInvalidated = true;
			}
		}
	
		return wasInvalidated;
	}
	
	
	/**
	 * Convert from the internal Hungarian notation to camelCase for external
	 * interaction
	 *  @param {object} obj Object to convert
	 *  @returns {object} Inverted object
	 *  @memberof DataTable#oApi
	 */
	function _fnSearchToCamel ( obj )
	{
		return {
			search:          obj.sSearch,
			smart:           obj.bSmart,
			regex:           obj.bRegex,
			caseInsensitive: obj.bCaseInsensitive
		};
	}
	
	
	
	/**
	 * Convert from camelCase notation to the internal Hungarian. We could use the
	 * Hungarian convert function here, but this is cleaner
	 *  @param {object} obj Object to convert
	 *  @returns {object} Inverted object
	 *  @memberof DataTable#oApi
	 */
	function _fnSearchToHung ( obj )
	{
		return {
			sSearch:          obj.search,
			bSmart:           obj.smart,
			bRegex:           obj.regex,
			bCaseInsensitive: obj.caseInsensitive
		};
	}
	
	/**
	 * Generate the node required for the info display
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {node} Information element
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlInfo ( settings )
	{
		var
			tid = settings.sTableId,
			nodes = settings.aanFeatures.i,
			n = $('<div/>', {
				'class': settings.oClasses.sInfo,
				'id': ! nodes ? tid+'_info' : null
			} );
	
		if ( ! nodes ) {
			// Update display on each draw
			settings.aoDrawCallback.push( {
				"fn": _fnUpdateInfo,
				"sName": "information"
			} );
	
			n
				.attr( 'role', 'status' )
				.attr( 'aria-live', 'polite' );
	
			// Table is described by our info div
			$(settings.nTable).attr( 'aria-describedby', tid+'_info' );
		}
	
		return n[0];
	}
	
	
	/**
	 * Update the information elements in the display
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnUpdateInfo ( settings )
	{
		/* Show information about the table */
		var nodes = settings.aanFeatures.i;
		if ( nodes.length === 0 ) {
			return;
		}
	
		var
			lang  = settings.oLanguage,
			start = settings._iDisplayStart+1,
			end   = settings.fnDisplayEnd(),
			max   = settings.fnRecordsTotal(),
			total = settings.fnRecordsDisplay(),
			out   = total ?
				lang.sInfo :
				lang.sInfoEmpty;
	
		if ( total !== max ) {
			/* Record set after filtering */
			out += ' ' + lang.sInfoFiltered;
		}
	
		// Convert the macros
		out += lang.sInfoPostFix;
		out = _fnInfoMacros( settings, out );
	
		var callback = lang.fnInfoCallback;
		if ( callback !== null ) {
			out = callback.call( settings.oInstance,
				settings, start, end, max, total, out
			);
		}
	
		$(nodes).html( out );
	}
	
	
	function _fnInfoMacros ( settings, str )
	{
		// When infinite scrolling, we are always starting at 1. _iDisplayStart is used only
		// internally
		var
			formatter  = settings.fnFormatNumber,
			start      = settings._iDisplayStart+1,
			len        = settings._iDisplayLength,
			vis        = settings.fnRecordsDisplay(),
			all        = len === -1;
	
		return str.
			replace(/_START_/g, formatter.call( settings, start ) ).
			replace(/_END_/g,   formatter.call( settings, settings.fnDisplayEnd() ) ).
			replace(/_MAX_/g,   formatter.call( settings, settings.fnRecordsTotal() ) ).
			replace(/_TOTAL_/g, formatter.call( settings, vis ) ).
			replace(/_PAGE_/g,  formatter.call( settings, all ? 1 : Math.ceil( start / len ) ) ).
			replace(/_PAGES_/g, formatter.call( settings, all ? 1 : Math.ceil( vis / len ) ) );
	}
	
	
	
	/**
	 * Draw the table for the first time, adding all required features
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnInitialise ( settings )
	{
		var i, iLen, iAjaxStart=settings.iInitDisplayStart;
		var columns = settings.aoColumns, column;
		var features = settings.oFeatures;
		var deferLoading = settings.bDeferLoading; // value modified by the draw
	
		/* Ensure that the table data is fully initialised */
		if ( ! settings.bInitialised ) {
			setTimeout( function(){ _fnInitialise( settings ); }, 200 );
			return;
		}
	
		/* Show the display HTML options */
		_fnAddOptionsHtml( settings );
	
		/* Build and draw the header / footer for the table */
		_fnBuildHead( settings );
		_fnDrawHead( settings, settings.aoHeader );
		_fnDrawHead( settings, settings.aoFooter );
	
		/* Okay to show that something is going on now */
		_fnProcessingDisplay( settings, true );
	
		/* Calculate sizes for columns */
		if ( features.bAutoWidth ) {
			_fnCalculateColumnWidths( settings );
		}
	
		for ( i=0, iLen=columns.length ; i<iLen ; i++ ) {
			column = columns[i];
	
			if ( column.sWidth ) {
				column.nTh.style.width = _fnStringToCss( column.sWidth );
			}
		}
	
		_fnCallbackFire( settings, null, 'preInit', [settings] );
	
		// If there is default sorting required - let's do it. The sort function
		// will do the drawing for us. Otherwise we draw the table regardless of the
		// Ajax source - this allows the table to look initialised for Ajax sourcing
		// data (show 'loading' message possibly)
		_fnReDraw( settings );
	
		// Server-side processing init complete is done by _fnAjaxUpdateDraw
		var dataSrc = _fnDataSource( settings );
		if ( dataSrc != 'ssp' || deferLoading ) {
			// if there is an ajax source load the data
			if ( dataSrc == 'ajax' ) {
				_fnBuildAjax( settings, [], function(json) {
					var aData = _fnAjaxDataSrc( settings, json );
	
					// Got the data - add it to the table
					for ( i=0 ; i<aData.length ; i++ ) {
						_fnAddData( settings, aData[i] );
					}
	
					// Reset the init display for cookie saving. We've already done
					// a filter, and therefore cleared it before. So we need to make
					// it appear 'fresh'
					settings.iInitDisplayStart = iAjaxStart;
	
					_fnReDraw( settings );
	
					_fnProcessingDisplay( settings, false );
					_fnInitComplete( settings, json );
				}, settings );
			}
			else {
				_fnProcessingDisplay( settings, false );
				_fnInitComplete( settings );
			}
		}
	}
	
	
	/**
	 * Draw the table for the first time, adding all required features
	 *  @param {object} oSettings dataTables settings object
	 *  @param {object} [json] JSON from the server that completed the table, if using Ajax source
	 *    with client-side processing (optional)
	 *  @memberof DataTable#oApi
	 */
	function _fnInitComplete ( settings, json )
	{
		settings._bInitComplete = true;
	
		// When data was added after the initialisation (data or Ajax) we need to
		// calculate the column sizing
		if ( json || settings.oInit.aaData ) {
			_fnAdjustColumnSizing( settings );
		}
	
		_fnCallbackFire( settings, null, 'plugin-init', [settings, json] );
		_fnCallbackFire( settings, 'aoInitComplete', 'init', [settings, json] );
	}
	
	
	function _fnLengthChange ( settings, val )
	{
		var len = parseInt( val, 10 );
		settings._iDisplayLength = len;
	
		_fnLengthOverflow( settings );
	
		// Fire length change event
		_fnCallbackFire( settings, null, 'length', [settings, len] );
	}
	
	
	/**
	 * Generate the node required for user display length changing
	 *  @param {object} settings dataTables settings object
	 *  @returns {node} Display length feature node
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlLength ( settings )
	{
		var
			classes  = settings.oClasses,
			tableId  = settings.sTableId,
			menu     = settings.aLengthMenu,
			d2       = Array.isArray( menu[0] ),
			lengths  = d2 ? menu[0] : menu,
			language = d2 ? menu[1] : menu;
	
		var select = $('<select/>', {
			'name':          tableId+'_length',
			'aria-controls': tableId,
			'class':         classes.sLengthSelect
		} );
	
		for ( var i=0, ien=lengths.length ; i<ien ; i++ ) {
			select[0][ i ] = new Option(
				typeof language[i] === 'number' ?
					settings.fnFormatNumber( language[i] ) :
					language[i],
				lengths[i]
			);
		}
	
		var div = $('<div><label/></div>').addClass( classes.sLength );
		if ( ! settings.aanFeatures.l ) {
			div[0].id = tableId+'_length';
		}
	
		div.children().append(
			settings.oLanguage.sLengthMenu.replace( '_MENU_', select[0].outerHTML )
		);
	
		// Can't use `select` variable as user might provide their own and the
		// reference is broken by the use of outerHTML
		$('select', div)
			.val( settings._iDisplayLength )
			.on( 'change.DT', function(e) {
				_fnLengthChange( settings, $(this).val() );
				_fnDraw( settings );
			} );
	
		// Update node value whenever anything changes the table's length
		$(settings.nTable).on( 'length.dt.DT', function (e, s, len) {
			if ( settings === s ) {
				$('select', div).val( len );
			}
		} );
	
		return div[0];
	}
	
	
	
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 * Note that most of the paging logic is done in
	 * DataTable.ext.pager
	 */
	
	/**
	 * Generate the node required for default pagination
	 *  @param {object} oSettings dataTables settings object
	 *  @returns {node} Pagination feature node
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlPaginate ( settings )
	{
		var
			type   = settings.sPaginationType,
			plugin = DataTable.ext.pager[ type ],
			modern = typeof plugin === 'function',
			redraw = function( settings ) {
				_fnDraw( settings );
			},
			node = $('<div/>').addClass( settings.oClasses.sPaging + type )[0],
			features = settings.aanFeatures;
	
		if ( ! modern ) {
			plugin.fnInit( settings, node, redraw );
		}
	
		/* Add a draw callback for the pagination on first instance, to update the paging display */
		if ( ! features.p )
		{
			node.id = settings.sTableId+'_paginate';
	
			settings.aoDrawCallback.push( {
				"fn": function( settings ) {
					if ( modern ) {
						var
							start      = settings._iDisplayStart,
							len        = settings._iDisplayLength,
							visRecords = settings.fnRecordsDisplay(),
							all        = len === -1,
							page = all ? 0 : Math.ceil( start / len ),
							pages = all ? 1 : Math.ceil( visRecords / len ),
							buttons = plugin(page, pages),
							i, ien;
	
						for ( i=0, ien=features.p.length ; i<ien ; i++ ) {
							_fnRenderer( settings, 'pageButton' )(
								settings, features.p[i], i, buttons, page, pages
							);
						}
					}
					else {
						plugin.fnUpdate( settings, redraw );
					}
				},
				"sName": "pagination"
			} );
		}
	
		return node;
	}
	
	
	/**
	 * Alter the display settings to change the page
	 *  @param {object} settings DataTables settings object
	 *  @param {string|int} action Paging action to take: "first", "previous",
	 *    "next" or "last" or page number to jump to (integer)
	 *  @param [bool] redraw Automatically draw the update or not
	 *  @returns {bool} true page has changed, false - no change
	 *  @memberof DataTable#oApi
	 */
	function _fnPageChange ( settings, action, redraw )
	{
		var
			start     = settings._iDisplayStart,
			len       = settings._iDisplayLength,
			records   = settings.fnRecordsDisplay();
	
		if ( records === 0 || len === -1 )
		{
			start = 0;
		}
		else if ( typeof action === "number" )
		{
			start = action * len;
	
			if ( start > records )
			{
				start = 0;
			}
		}
		else if ( action == "first" )
		{
			start = 0;
		}
		else if ( action == "previous" )
		{
			start = len >= 0 ?
				start - len :
				0;
	
			if ( start < 0 )
			{
			  start = 0;
			}
		}
		else if ( action == "next" )
		{
			if ( start + len < records )
			{
				start += len;
			}
		}
		else if ( action == "last" )
		{
			start = Math.floor( (records-1) / len) * len;
		}
		else
		{
			_fnLog( settings, 0, "Unknown paging action: "+action, 5 );
		}
	
		var changed = settings._iDisplayStart !== start;
		settings._iDisplayStart = start;
	
		if ( changed ) {
			_fnCallbackFire( settings, null, 'page', [settings] );
	
			if ( redraw ) {
				_fnDraw( settings );
			}
		}
	
		return changed;
	}
	
	
	
	/**
	 * Generate the node required for the processing node
	 *  @param {object} settings dataTables settings object
	 *  @returns {node} Processing element
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlProcessing ( settings )
	{
		return $('<div/>', {
				'id': ! settings.aanFeatures.r ? settings.sTableId+'_processing' : null,
				'class': settings.oClasses.sProcessing
			} )
			.html( settings.oLanguage.sProcessing )
			.insertBefore( settings.nTable )[0];
	}
	
	
	/**
	 * Display or hide the processing indicator
	 *  @param {object} settings dataTables settings object
	 *  @param {bool} show Show the processing indicator (true) or not (false)
	 *  @memberof DataTable#oApi
	 */
	function _fnProcessingDisplay ( settings, show )
	{
		if ( settings.oFeatures.bProcessing ) {
			$(settings.aanFeatures.r).css( 'display', show ? 'block' : 'none' );
		}
	
		_fnCallbackFire( settings, null, 'processing', [settings, show] );
	}
	
	/**
	 * Add any control elements for the table - specifically scrolling
	 *  @param {object} settings dataTables settings object
	 *  @returns {node} Node to add to the DOM
	 *  @memberof DataTable#oApi
	 */
	function _fnFeatureHtmlTable ( settings )
	{
		var table = $(settings.nTable);
	
		// Add the ARIA grid role to the table
		table.attr( 'role', 'grid' );
	
		// Scrolling from here on in
		var scroll = settings.oScroll;
	
		if ( scroll.sX === '' && scroll.sY === '' ) {
			return settings.nTable;
		}
	
		var scrollX = scroll.sX;
		var scrollY = scroll.sY;
		var classes = settings.oClasses;
		var caption = table.children('caption');
		var captionSide = caption.length ? caption[0]._captionSide : null;
		var headerClone = $( table[0].cloneNode(false) );
		var footerClone = $( table[0].cloneNode(false) );
		var footer = table.children('tfoot');
		var _div = '<div/>';
		var size = function ( s ) {
			return !s ? null : _fnStringToCss( s );
		};
	
		if ( ! footer.length ) {
			footer = null;
		}
	
		/*
		 * The HTML structure that we want to generate in this function is:
		 *  div - scroller
		 *    div - scroll head
		 *      div - scroll head inner
		 *        table - scroll head table
		 *          thead - thead
		 *    div - scroll body
		 *      table - table (master table)
		 *        thead - thead clone for sizing
		 *        tbody - tbody
		 *    div - scroll foot
		 *      div - scroll foot inner
		 *        table - scroll foot table
		 *          tfoot - tfoot
		 */
		var scroller = $( _div, { 'class': classes.sScrollWrapper } )
			.append(
				$(_div, { 'class': classes.sScrollHead } )
					.css( {
						overflow: 'hidden',
						position: 'relative',
						border: 0,
						width: scrollX ? size(scrollX) : '100%'
					} )
					.append(
						$(_div, { 'class': classes.sScrollHeadInner } )
							.css( {
								'box-sizing': 'content-box',
								width: scroll.sXInner || '100%'
							} )
							.append(
								headerClone
									.removeAttr('id')
									.css( 'margin-left', 0 )
									.append( captionSide === 'top' ? caption : null )
									.append(
										table.children('thead')
									)
							)
					)
			)
			.append(
				$(_div, { 'class': classes.sScrollBody } )
					.css( {
						position: 'relative',
						overflow: 'auto',
						width: size( scrollX )
					} )
					.append( table )
			);
	
		if ( footer ) {
			scroller.append(
				$(_div, { 'class': classes.sScrollFoot } )
					.css( {
						overflow: 'hidden',
						border: 0,
						width: scrollX ? size(scrollX) : '100%'
					} )
					.append(
						$(_div, { 'class': classes.sScrollFootInner } )
							.append(
								footerClone
									.removeAttr('id')
									.css( 'margin-left', 0 )
									.append( captionSide === 'bottom' ? caption : null )
									.append(
										table.children('tfoot')
									)
							)
					)
			);
		}
	
		var children = scroller.children();
		var scrollHead = children[0];
		var scrollBody = children[1];
		var scrollFoot = footer ? children[2] : null;
	
		// When the body is scrolled, then we also want to scroll the headers
		if ( scrollX ) {
			$(scrollBody).on( 'scroll.DT', function (e) {
				var scrollLeft = this.scrollLeft;
	
				scrollHead.scrollLeft = scrollLeft;
	
				if ( footer ) {
					scrollFoot.scrollLeft = scrollLeft;
				}
			} );
		}
	
		$(scrollBody).css('max-height', scrollY);
		if (! scroll.bCollapse) {
			$(scrollBody).css('height', scrollY);
		}
	
		settings.nScrollHead = scrollHead;
		settings.nScrollBody = scrollBody;
		settings.nScrollFoot = scrollFoot;
	
		// On redraw - align columns
		settings.aoDrawCallback.push( {
			"fn": _fnScrollDraw,
			"sName": "scrolling"
		} );
	
		return scroller[0];
	}
	
	
	
	/**
	 * Update the header, footer and body tables for resizing - i.e. column
	 * alignment.
	 *
	 * Welcome to the most horrible function DataTables. The process that this
	 * function follows is basically:
	 *   1. Re-create the table inside the scrolling div
	 *   2. Take live measurements from the DOM
	 *   3. Apply the measurements to align the columns
	 *   4. Clean up
	 *
	 *  @param {object} settings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnScrollDraw ( settings )
	{
		// Given that this is such a monster function, a lot of variables are use
		// to try and keep the minimised size as small as possible
		var
			scroll         = settings.oScroll,
			scrollX        = scroll.sX,
			scrollXInner   = scroll.sXInner,
			scrollY        = scroll.sY,
			barWidth       = scroll.iBarWidth,
			divHeader      = $(settings.nScrollHead),
			divHeaderStyle = divHeader[0].style,
			divHeaderInner = divHeader.children('div'),
			divHeaderInnerStyle = divHeaderInner[0].style,
			divHeaderTable = divHeaderInner.children('table'),
			divBodyEl      = settings.nScrollBody,
			divBody        = $(divBodyEl),
			divBodyStyle   = divBodyEl.style,
			divFooter      = $(settings.nScrollFoot),
			divFooterInner = divFooter.children('div'),
			divFooterTable = divFooterInner.children('table'),
			header         = $(settings.nTHead),
			table          = $(settings.nTable),
			tableEl        = table[0],
			tableStyle     = tableEl.style,
			footer         = settings.nTFoot ? $(settings.nTFoot) : null,
			browser        = settings.oBrowser,
			ie67           = browser.bScrollOversize,
			dtHeaderCells  = _pluck( settings.aoColumns, 'nTh' ),
			headerTrgEls, footerTrgEls,
			headerSrcEls, footerSrcEls,
			headerCopy, footerCopy,
			headerWidths=[], footerWidths=[],
			headerContent=[], footerContent=[],
			idx, correction, sanityWidth,
			zeroOut = function(nSizer) {
				var style = nSizer.style;
				style.paddingTop = "0";
				style.paddingBottom = "0";
				style.borderTopWidth = "0";
				style.borderBottomWidth = "0";
				style.height = 0;
			};
	
		// If the scrollbar visibility has changed from the last draw, we need to
		// adjust the column sizes as the table width will have changed to account
		// for the scrollbar
		var scrollBarVis = divBodyEl.scrollHeight > divBodyEl.clientHeight;
		
		if ( settings.scrollBarVis !== scrollBarVis && settings.scrollBarVis !== undefined ) {
			settings.scrollBarVis = scrollBarVis;
			_fnAdjustColumnSizing( settings );
			return; // adjust column sizing will call this function again
		}
		else {
			settings.scrollBarVis = scrollBarVis;
		}
	
		/*
		 * 1. Re-create the table inside the scrolling div
		 */
	
		// Remove the old minimised thead and tfoot elements in the inner table
		table.children('thead, tfoot').remove();
	
		if ( footer ) {
			footerCopy = footer.clone().prependTo( table );
			footerTrgEls = footer.find('tr'); // the original tfoot is in its own table and must be sized
			footerSrcEls = footerCopy.find('tr');
		}
	
		// Clone the current header and footer elements and then place it into the inner table
		headerCopy = header.clone().prependTo( table );
		headerTrgEls = header.find('tr'); // original header is in its own table
		headerSrcEls = headerCopy.find('tr');
		headerCopy.find('th, td').removeAttr('tabindex');
	
	
		/*
		 * 2. Take live measurements from the DOM - do not alter the DOM itself!
		 */
	
		// Remove old sizing and apply the calculated column widths
		// Get the unique column headers in the newly created (cloned) header. We want to apply the
		// calculated sizes to this header
		if ( ! scrollX )
		{
			divBodyStyle.width = '100%';
			divHeader[0].style.width = '100%';
		}
	
		$.each( _fnGetUniqueThs( settings, headerCopy ), function ( i, el ) {
			idx = _fnVisibleToColumnIndex( settings, i );
			el.style.width = settings.aoColumns[idx].sWidth;
		} );
	
		if ( footer ) {
			_fnApplyToChildren( function(n) {
				n.style.width = "";
			}, footerSrcEls );
		}
	
		// Size the table as a whole
		sanityWidth = table.outerWidth();
		if ( scrollX === "" ) {
			// No x scrolling
			tableStyle.width = "100%";
	
			// IE7 will make the width of the table when 100% include the scrollbar
			// - which is shouldn't. When there is a scrollbar we need to take this
			// into account.
			if ( ie67 && (table.find('tbody').height() > divBodyEl.offsetHeight ||
				divBody.css('overflow-y') == "scroll")
			) {
				tableStyle.width = _fnStringToCss( table.outerWidth() - barWidth);
			}
	
			// Recalculate the sanity width
			sanityWidth = table.outerWidth();
		}
		else if ( scrollXInner !== "" ) {
			// legacy x scroll inner has been given - use it
			tableStyle.width = _fnStringToCss(scrollXInner);
	
			// Recalculate the sanity width
			sanityWidth = table.outerWidth();
		}
	
		// Hidden header should have zero height, so remove padding and borders. Then
		// set the width based on the real headers
	
		// Apply all styles in one pass
		_fnApplyToChildren( zeroOut, headerSrcEls );
	
		// Read all widths in next pass
		_fnApplyToChildren( function(nSizer) {
			headerContent.push( nSizer.innerHTML );
			headerWidths.push( _fnStringToCss( $(nSizer).css('width') ) );
		}, headerSrcEls );
	
		// Apply all widths in final pass
		_fnApplyToChildren( function(nToSize, i) {
			// Only apply widths to the DataTables detected header cells - this
			// prevents complex headers from having contradictory sizes applied
			if ( $.inArray( nToSize, dtHeaderCells ) !== -1 ) {
				nToSize.style.width = headerWidths[i];
			}
		}, headerTrgEls );
	
		$(headerSrcEls).height(0);
	
		/* Same again with the footer if we have one */
		if ( footer )
		{
			_fnApplyToChildren( zeroOut, footerSrcEls );
	
			_fnApplyToChildren( function(nSizer) {
				footerContent.push( nSizer.innerHTML );
				footerWidths.push( _fnStringToCss( $(nSizer).css('width') ) );
			}, footerSrcEls );
	
			_fnApplyToChildren( function(nToSize, i) {
				nToSize.style.width = footerWidths[i];
			}, footerTrgEls );
	
			$(footerSrcEls).height(0);
		}
	
	
		/*
		 * 3. Apply the measurements
		 */
	
		// "Hide" the header and footer that we used for the sizing. We need to keep
		// the content of the cell so that the width applied to the header and body
		// both match, but we want to hide it completely. We want to also fix their
		// width to what they currently are
		_fnApplyToChildren( function(nSizer, i) {
			nSizer.innerHTML = '<div class="dataTables_sizing">'+headerContent[i]+'</div>';
			nSizer.childNodes[0].style.height = "0";
			nSizer.childNodes[0].style.overflow = "hidden";
			nSizer.style.width = headerWidths[i];
		}, headerSrcEls );
	
		if ( footer )
		{
			_fnApplyToChildren( function(nSizer, i) {
				nSizer.innerHTML = '<div class="dataTables_sizing">'+footerContent[i]+'</div>';
				nSizer.childNodes[0].style.height = "0";
				nSizer.childNodes[0].style.overflow = "hidden";
				nSizer.style.width = footerWidths[i];
			}, footerSrcEls );
		}
	
		// Sanity check that the table is of a sensible width. If not then we are going to get
		// misalignment - try to prevent this by not allowing the table to shrink below its min width
		if ( table.outerWidth() < sanityWidth )
		{
			// The min width depends upon if we have a vertical scrollbar visible or not */
			correction = ((divBodyEl.scrollHeight > divBodyEl.offsetHeight ||
				divBody.css('overflow-y') == "scroll")) ?
					sanityWidth+barWidth :
					sanityWidth;
	
			// IE6/7 are a law unto themselves...
			if ( ie67 && (divBodyEl.scrollHeight >
				divBodyEl.offsetHeight || divBody.css('overflow-y') == "scroll")
			) {
				tableStyle.width = _fnStringToCss( correction-barWidth );
			}
	
			// And give the user a warning that we've stopped the table getting too small
			if ( scrollX === "" || scrollXInner !== "" ) {
				_fnLog( settings, 1, 'Possible column misalignment', 6 );
			}
		}
		else
		{
			correction = '100%';
		}
	
		// Apply to the container elements
		divBodyStyle.width = _fnStringToCss( correction );
		divHeaderStyle.width = _fnStringToCss( correction );
	
		if ( footer ) {
			settings.nScrollFoot.style.width = _fnStringToCss( correction );
		}
	
	
		/*
		 * 4. Clean up
		 */
		if ( ! scrollY ) {
			/* IE7< puts a vertical scrollbar in place (when it shouldn't be) due to subtracting
			 * the scrollbar height from the visible display, rather than adding it on. We need to
			 * set the height in order to sort this. Don't want to do it in any other browsers.
			 */
			if ( ie67 ) {
				divBodyStyle.height = _fnStringToCss( tableEl.offsetHeight+barWidth );
			}
		}
	
		/* Finally set the width's of the header and footer tables */
		var iOuterWidth = table.outerWidth();
		divHeaderTable[0].style.width = _fnStringToCss( iOuterWidth );
		divHeaderInnerStyle.width = _fnStringToCss( iOuterWidth );
	
		// Figure out if there are scrollbar present - if so then we need a the header and footer to
		// provide a bit more space to allow "overflow" scrolling (i.e. past the scrollbar)
		var bScrolling = table.height() > divBodyEl.clientHeight || divBody.css('overflow-y') == "scroll";
		var padding = 'padding' + (browser.bScrollbarLeft ? 'Left' : 'Right' );
		divHeaderInnerStyle[ padding ] = bScrolling ? barWidth+"px" : "0px";
	
		if ( footer ) {
			divFooterTable[0].style.width = _fnStringToCss( iOuterWidth );
			divFooterInner[0].style.width = _fnStringToCss( iOuterWidth );
			divFooterInner[0].style[padding] = bScrolling ? barWidth+"px" : "0px";
		}
	
		// Correct DOM ordering for colgroup - comes before the thead
		table.children('colgroup').insertBefore( table.children('thead') );
	
		/* Adjust the position of the header in case we loose the y-scrollbar */
		divBody.trigger('scroll');
	
		// If sorting or filtering has occurred, jump the scrolling back to the top
		// only if we aren't holding the position
		if ( (settings.bSorted || settings.bFiltered) && ! settings._drawHold ) {
			divBodyEl.scrollTop = 0;
		}
	}
	
	
	
	/**
	 * Apply a given function to the display child nodes of an element array (typically
	 * TD children of TR rows
	 *  @param {function} fn Method to apply to the objects
	 *  @param array {nodes} an1 List of elements to look through for display children
	 *  @param array {nodes} an2 Another list (identical structure to the first) - optional
	 *  @memberof DataTable#oApi
	 */
	function _fnApplyToChildren( fn, an1, an2 )
	{
		var index=0, i=0, iLen=an1.length;
		var nNode1, nNode2;
	
		while ( i < iLen ) {
			nNode1 = an1[i].firstChild;
			nNode2 = an2 ? an2[i].firstChild : null;
	
			while ( nNode1 ) {
				if ( nNode1.nodeType === 1 ) {
					if ( an2 ) {
						fn( nNode1, nNode2, index );
					}
					else {
						fn( nNode1, index );
					}
	
					index++;
				}
	
				nNode1 = nNode1.nextSibling;
				nNode2 = an2 ? nNode2.nextSibling : null;
			}
	
			i++;
		}
	}
	
	
	
	var __re_html_remove = /<.*?>/g;
	
	
	/**
	 * Calculate the width of columns for the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnCalculateColumnWidths ( oSettings )
	{
		var
			table = oSettings.nTable,
			columns = oSettings.aoColumns,
			scroll = oSettings.oScroll,
			scrollY = scroll.sY,
			scrollX = scroll.sX,
			scrollXInner = scroll.sXInner,
			columnCount = columns.length,
			visibleColumns = _fnGetColumns( oSettings, 'bVisible' ),
			headerCells = $('th', oSettings.nTHead),
			tableWidthAttr = table.getAttribute('width'), // from DOM element
			tableContainer = table.parentNode,
			userInputs = false,
			i, column, columnIdx, width, outerWidth,
			browser = oSettings.oBrowser,
			ie67 = browser.bScrollOversize;
	
		var styleWidth = table.style.width;
		if ( styleWidth && styleWidth.indexOf('%') !== -1 ) {
			tableWidthAttr = styleWidth;
		}
	
		/* Convert any user input sizes into pixel sizes */
		for ( i=0 ; i<visibleColumns.length ; i++ ) {
			column = columns[ visibleColumns[i] ];
	
			if ( column.sWidth !== null ) {
				column.sWidth = _fnConvertToWidth( column.sWidthOrig, tableContainer );
	
				userInputs = true;
			}
		}
	
		/* If the number of columns in the DOM equals the number that we have to
		 * process in DataTables, then we can use the offsets that are created by
		 * the web- browser. No custom sizes can be set in order for this to happen,
		 * nor scrolling used
		 */
		if ( ie67 || ! userInputs && ! scrollX && ! scrollY &&
		     columnCount == _fnVisbleColumns( oSettings ) &&
		     columnCount == headerCells.length
		) {
			for ( i=0 ; i<columnCount ; i++ ) {
				var colIdx = _fnVisibleToColumnIndex( oSettings, i );
	
				if ( colIdx !== null ) {
					columns[ colIdx ].sWidth = _fnStringToCss( headerCells.eq(i).width() );
				}
			}
		}
		else
		{
			// Otherwise construct a single row, worst case, table with the widest
			// node in the data, assign any user defined widths, then insert it into
			// the DOM and allow the browser to do all the hard work of calculating
			// table widths
			var tmpTable = $(table).clone() // don't use cloneNode - IE8 will remove events on the main table
				.css( 'visibility', 'hidden' )
				.removeAttr( 'id' );
	
			// Clean up the table body
			tmpTable.find('tbody tr').remove();
			var tr = $('<tr/>').appendTo( tmpTable.find('tbody') );
	
			// Clone the table header and footer - we can't use the header / footer
			// from the cloned table, since if scrolling is active, the table's
			// real header and footer are contained in different table tags
			tmpTable.find('thead, tfoot').remove();
			tmpTable
				.append( $(oSettings.nTHead).clone() )
				.append( $(oSettings.nTFoot).clone() );
	
			// Remove any assigned widths from the footer (from scrolling)
			tmpTable.find('tfoot th, tfoot td').css('width', '');
	
			// Apply custom sizing to the cloned header
			headerCells = _fnGetUniqueThs( oSettings, tmpTable.find('thead')[0] );
	
			for ( i=0 ; i<visibleColumns.length ; i++ ) {
				column = columns[ visibleColumns[i] ];
	
				headerCells[i].style.width = column.sWidthOrig !== null && column.sWidthOrig !== '' ?
					_fnStringToCss( column.sWidthOrig ) :
					'';
	
				// For scrollX we need to force the column width otherwise the
				// browser will collapse it. If this width is smaller than the
				// width the column requires, then it will have no effect
				if ( column.sWidthOrig && scrollX ) {
					$( headerCells[i] ).append( $('<div/>').css( {
						width: column.sWidthOrig,
						margin: 0,
						padding: 0,
						border: 0,
						height: 1
					} ) );
				}
			}
	
			// Find the widest cell for each column and put it into the table
			if ( oSettings.aoData.length ) {
				for ( i=0 ; i<visibleColumns.length ; i++ ) {
					columnIdx = visibleColumns[i];
					column = columns[ columnIdx ];
	
					$( _fnGetWidestNode( oSettings, columnIdx ) )
						.clone( false )
						.append( column.sContentPadding )
						.appendTo( tr );
				}
			}
	
			// Tidy the temporary table - remove name attributes so there aren't
			// duplicated in the dom (radio elements for example)
			$('[name]', tmpTable).removeAttr('name');
	
			// Table has been built, attach to the document so we can work with it.
			// A holding element is used, positioned at the top of the container
			// with minimal height, so it has no effect on if the container scrolls
			// or not. Otherwise it might trigger scrolling when it actually isn't
			// needed
			var holder = $('<div/>').css( scrollX || scrollY ?
					{
						position: 'absolute',
						top: 0,
						left: 0,
						height: 1,
						right: 0,
						overflow: 'hidden'
					} :
					{}
				)
				.append( tmpTable )
				.appendTo( tableContainer );
	
			// When scrolling (X or Y) we want to set the width of the table as 
			// appropriate. However, when not scrolling leave the table width as it
			// is. This results in slightly different, but I think correct behaviour
			if ( scrollX && scrollXInner ) {
				tmpTable.width( scrollXInner );
			}
			else if ( scrollX ) {
				tmpTable.css( 'width', 'auto' );
				tmpTable.removeAttr('width');
	
				// If there is no width attribute or style, then allow the table to
				// collapse
				if ( tmpTable.width() < tableContainer.clientWidth && tableWidthAttr ) {
					tmpTable.width( tableContainer.clientWidth );
				}
			}
			else if ( scrollY ) {
				tmpTable.width( tableContainer.clientWidth );
			}
			else if ( tableWidthAttr ) {
				tmpTable.width( tableWidthAttr );
			}
	
			// Get the width of each column in the constructed table - we need to
			// know the inner width (so it can be assigned to the other table's
			// cells) and the outer width so we can calculate the full width of the
			// table. This is safe since DataTables requires a unique cell for each
			// column, but if ever a header can span multiple columns, this will
			// need to be modified.
			var total = 0;
			for ( i=0 ; i<visibleColumns.length ; i++ ) {
				var cell = $(headerCells[i]);
				var border = cell.outerWidth() - cell.width();
	
				// Use getBounding... where possible (not IE8-) because it can give
				// sub-pixel accuracy, which we then want to round up!
				var bounding = browser.bBounding ?
					Math.ceil( headerCells[i].getBoundingClientRect().width ) :
					cell.outerWidth();
	
				// Total is tracked to remove any sub-pixel errors as the outerWidth
				// of the table might not equal the total given here (IE!).
				total += bounding;
	
				// Width for each column to use
				columns[ visibleColumns[i] ].sWidth = _fnStringToCss( bounding - border );
			}
	
			table.style.width = _fnStringToCss( total );
	
			// Finished with the table - ditch it
			holder.remove();
		}
	
		// If there is a width attr, we want to attach an event listener which
		// allows the table sizing to automatically adjust when the window is
		// resized. Use the width attr rather than CSS, since we can't know if the
		// CSS is a relative value or absolute - DOM read is always px.
		if ( tableWidthAttr ) {
			table.style.width = _fnStringToCss( tableWidthAttr );
		}
	
		if ( (tableWidthAttr || scrollX) && ! oSettings._reszEvt ) {
			var bindResize = function () {
				$(window).on('resize.DT-'+oSettings.sInstance, _fnThrottle( function () {
					_fnAdjustColumnSizing( oSettings );
				} ) );
			};
	
			// IE6/7 will crash if we bind a resize event handler on page load.
			// To be removed in 1.11 which drops IE6/7 support
			if ( ie67 ) {
				setTimeout( bindResize, 1000 );
			}
			else {
				bindResize();
			}
	
			oSettings._reszEvt = true;
		}
	}
	
	
	/**
	 * Throttle the calls to a function. Arguments and context are maintained for
	 * the throttled function
	 *  @param {function} fn Function to be called
	 *  @param {int} [freq=200] call frequency in mS
	 *  @returns {function} wrapped function
	 *  @memberof DataTable#oApi
	 */
	var _fnThrottle = DataTable.util.throttle;
	
	
	/**
	 * Convert a CSS unit width to pixels (e.g. 2em)
	 *  @param {string} width width to be converted
	 *  @param {node} parent parent to get the with for (required for relative widths) - optional
	 *  @returns {int} width in pixels
	 *  @memberof DataTable#oApi
	 */
	function _fnConvertToWidth ( width, parent )
	{
		if ( ! width ) {
			return 0;
		}
	
		var n = $('<div/>')
			.css( 'width', _fnStringToCss( width ) )
			.appendTo( parent || document.body );
	
		var val = n[0].offsetWidth;
		n.remove();
	
		return val;
	}
	
	
	/**
	 * Get the widest node
	 *  @param {object} settings dataTables settings object
	 *  @param {int} colIdx column of interest
	 *  @returns {node} widest table node
	 *  @memberof DataTable#oApi
	 */
	function _fnGetWidestNode( settings, colIdx )
	{
		var idx = _fnGetMaxLenString( settings, colIdx );
		if ( idx < 0 ) {
			return null;
		}
	
		var data = settings.aoData[ idx ];
		return ! data.nTr ? // Might not have been created when deferred rendering
			$('<td/>').html( _fnGetCellData( settings, idx, colIdx, 'display' ) )[0] :
			data.anCells[ colIdx ];
	}
	
	
	/**
	 * Get the maximum strlen for each data column
	 *  @param {object} settings dataTables settings object
	 *  @param {int} colIdx column of interest
	 *  @returns {string} max string length for each column
	 *  @memberof DataTable#oApi
	 */
	function _fnGetMaxLenString( settings, colIdx )
	{
		var s, max=-1, maxIdx = -1;
	
		for ( var i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
			s = _fnGetCellData( settings, i, colIdx, 'display' )+'';
			s = s.replace( __re_html_remove, '' );
			s = s.replace( /&nbsp;/g, ' ' );
	
			if ( s.length > max ) {
				max = s.length;
				maxIdx = i;
			}
		}
	
		return maxIdx;
	}
	
	
	/**
	 * Append a CSS unit (only if required) to a string
	 *  @param {string} value to css-ify
	 *  @returns {string} value with css unit
	 *  @memberof DataTable#oApi
	 */
	function _fnStringToCss( s )
	{
		if ( s === null ) {
			return '0px';
		}
	
		if ( typeof s == 'number' ) {
			return s < 0 ?
				'0px' :
				s+'px';
		}
	
		// Check it has a unit character already
		return s.match(/\d$/) ?
			s+'px' :
			s;
	}
	
	
	
	function _fnSortFlatten ( settings )
	{
		var
			i, iLen, k, kLen,
			aSort = [],
			aiOrig = [],
			aoColumns = settings.aoColumns,
			aDataSort, iCol, sType, srcCol,
			fixed = settings.aaSortingFixed,
			fixedObj = $.isPlainObject( fixed ),
			nestedSort = [],
			add = function ( a ) {
				if ( a.length && ! Array.isArray( a[0] ) ) {
					// 1D array
					nestedSort.push( a );
				}
				else {
					// 2D array
					$.merge( nestedSort, a );
				}
			};
	
		// Build the sort array, with pre-fix and post-fix options if they have been
		// specified
		if ( Array.isArray( fixed ) ) {
			add( fixed );
		}
	
		if ( fixedObj && fixed.pre ) {
			add( fixed.pre );
		}
	
		add( settings.aaSorting );
	
		if (fixedObj && fixed.post ) {
			add( fixed.post );
		}
	
		for ( i=0 ; i<nestedSort.length ; i++ )
		{
			srcCol = nestedSort[i][0];
			aDataSort = aoColumns[ srcCol ].aDataSort;
	
			for ( k=0, kLen=aDataSort.length ; k<kLen ; k++ )
			{
				iCol = aDataSort[k];
				sType = aoColumns[ iCol ].sType || 'string';
	
				if ( nestedSort[i]._idx === undefined ) {
					nestedSort[i]._idx = $.inArray( nestedSort[i][1], aoColumns[iCol].asSorting );
				}
	
				aSort.push( {
					src:       srcCol,
					col:       iCol,
					dir:       nestedSort[i][1],
					index:     nestedSort[i]._idx,
					type:      sType,
					formatter: DataTable.ext.type.order[ sType+"-pre" ]
				} );
			}
		}
	
		return aSort;
	}
	
	/**
	 * Change the order of the table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 *  @todo This really needs split up!
	 */
	function _fnSort ( oSettings )
	{
		var
			i, ien, iLen, j, jLen, k, kLen,
			sDataType, nTh,
			aiOrig = [],
			oExtSort = DataTable.ext.type.order,
			aoData = oSettings.aoData,
			aoColumns = oSettings.aoColumns,
			aDataSort, data, iCol, sType, oSort,
			formatters = 0,
			sortCol,
			displayMaster = oSettings.aiDisplayMaster,
			aSort;
	
		// Resolve any column types that are unknown due to addition or invalidation
		// @todo Can this be moved into a 'data-ready' handler which is called when
		//   data is going to be used in the table?
		_fnColumnTypes( oSettings );
	
		aSort = _fnSortFlatten( oSettings );
	
		for ( i=0, ien=aSort.length ; i<ien ; i++ ) {
			sortCol = aSort[i];
	
			// Track if we can use the fast sort algorithm
			if ( sortCol.formatter ) {
				formatters++;
			}
	
			// Load the data needed for the sort, for each cell
			_fnSortData( oSettings, sortCol.col );
		}
	
		/* No sorting required if server-side or no sorting array */
		if ( _fnDataSource( oSettings ) != 'ssp' && aSort.length !== 0 )
		{
			// Create a value - key array of the current row positions such that we can use their
			// current position during the sort, if values match, in order to perform stable sorting
			for ( i=0, iLen=displayMaster.length ; i<iLen ; i++ ) {
				aiOrig[ displayMaster[i] ] = i;
			}
	
			/* Do the sort - here we want multi-column sorting based on a given data source (column)
			 * and sorting function (from oSort) in a certain direction. It's reasonably complex to
			 * follow on it's own, but this is what we want (example two column sorting):
			 *  fnLocalSorting = function(a,b){
			 *    var iTest;
			 *    iTest = oSort['string-asc']('data11', 'data12');
			 *      if (iTest !== 0)
			 *        return iTest;
			 *    iTest = oSort['numeric-desc']('data21', 'data22');
			 *    if (iTest !== 0)
			 *      return iTest;
			 *    return oSort['numeric-asc']( aiOrig[a], aiOrig[b] );
			 *  }
			 * Basically we have a test for each sorting column, if the data in that column is equal,
			 * test the next column. If all columns match, then we use a numeric sort on the row
			 * positions in the original data array to provide a stable sort.
			 *
			 * Note - I know it seems excessive to have two sorting methods, but the first is around
			 * 15% faster, so the second is only maintained for backwards compatibility with sorting
			 * methods which do not have a pre-sort formatting function.
			 */
			if ( formatters === aSort.length ) {
				// All sort types have formatting functions
				displayMaster.sort( function ( a, b ) {
					var
						x, y, k, test, sort,
						len=aSort.length,
						dataA = aoData[a]._aSortData,
						dataB = aoData[b]._aSortData;
	
					for ( k=0 ; k<len ; k++ ) {
						sort = aSort[k];
	
						x = dataA[ sort.col ];
						y = dataB[ sort.col ];
	
						test = x<y ? -1 : x>y ? 1 : 0;
						if ( test !== 0 ) {
							return sort.dir === 'asc' ? test : -test;
						}
					}
	
					x = aiOrig[a];
					y = aiOrig[b];
					return x<y ? -1 : x>y ? 1 : 0;
				} );
			}
			else {
				// Depreciated - remove in 1.11 (providing a plug-in option)
				// Not all sort types have formatting methods, so we have to call their sorting
				// methods.
				displayMaster.sort( function ( a, b ) {
					var
						x, y, k, l, test, sort, fn,
						len=aSort.length,
						dataA = aoData[a]._aSortData,
						dataB = aoData[b]._aSortData;
	
					for ( k=0 ; k<len ; k++ ) {
						sort = aSort[k];
	
						x = dataA[ sort.col ];
						y = dataB[ sort.col ];
	
						fn = oExtSort[ sort.type+"-"+sort.dir ] || oExtSort[ "string-"+sort.dir ];
						test = fn( x, y );
						if ( test !== 0 ) {
							return test;
						}
					}
	
					x = aiOrig[a];
					y = aiOrig[b];
					return x<y ? -1 : x>y ? 1 : 0;
				} );
			}
		}
	
		/* Tell the draw function that we have sorted the data */
		oSettings.bSorted = true;
	}
	
	
	function _fnSortAria ( settings )
	{
		var label;
		var nextSort;
		var columns = settings.aoColumns;
		var aSort = _fnSortFlatten( settings );
		var oAria = settings.oLanguage.oAria;
	
		// ARIA attributes - need to loop all columns, to update all (removing old
		// attributes as needed)
		for ( var i=0, iLen=columns.length ; i<iLen ; i++ )
		{
			var col = columns[i];
			var asSorting = col.asSorting;
			var sTitle = col.sTitle.replace( /<.*?>/g, "" );
			var th = col.nTh;
	
			// IE7 is throwing an error when setting these properties with jQuery's
			// attr() and removeAttr() methods...
			th.removeAttribute('aria-sort');
	
			/* In ARIA only the first sorting column can be marked as sorting - no multi-sort option */
			if ( col.bSortable ) {
				if ( aSort.length > 0 && aSort[0].col == i ) {
					th.setAttribute('aria-sort', aSort[0].dir=="asc" ? "ascending" : "descending" );
					nextSort = asSorting[ aSort[0].index+1 ] || asSorting[0];
				}
				else {
					nextSort = asSorting[0];
				}
	
				label = sTitle + ( nextSort === "asc" ?
					oAria.sSortAscending :
					oAria.sSortDescending
				);
			}
			else {
				label = sTitle;
			}
	
			th.setAttribute('aria-label', label);
		}
	}
	
	
	/**
	 * Function to run on user sort request
	 *  @param {object} settings dataTables settings object
	 *  @param {node} attachTo node to attach the handler to
	 *  @param {int} colIdx column sorting index
	 *  @param {boolean} [append=false] Append the requested sort to the existing
	 *    sort if true (i.e. multi-column sort)
	 *  @param {function} [callback] callback function
	 *  @memberof DataTable#oApi
	 */
	function _fnSortListener ( settings, colIdx, append, callback )
	{
		var col = settings.aoColumns[ colIdx ];
		var sorting = settings.aaSorting;
		var asSorting = col.asSorting;
		var nextSortIdx;
		var next = function ( a, overflow ) {
			var idx = a._idx;
			if ( idx === undefined ) {
				idx = $.inArray( a[1], asSorting );
			}
	
			return idx+1 < asSorting.length ?
				idx+1 :
				overflow ?
					null :
					0;
		};
	
		// Convert to 2D array if needed
		if ( typeof sorting[0] === 'number' ) {
			sorting = settings.aaSorting = [ sorting ];
		}
	
		// If appending the sort then we are multi-column sorting
		if ( append && settings.oFeatures.bSortMulti ) {
			// Are we already doing some kind of sort on this column?
			var sortIdx = $.inArray( colIdx, _pluck(sorting, '0') );
	
			if ( sortIdx !== -1 ) {
				// Yes, modify the sort
				nextSortIdx = next( sorting[sortIdx], true );
	
				if ( nextSortIdx === null && sorting.length === 1 ) {
					nextSortIdx = 0; // can't remove sorting completely
				}
	
				if ( nextSortIdx === null ) {
					sorting.splice( sortIdx, 1 );
				}
				else {
					sorting[sortIdx][1] = asSorting[ nextSortIdx ];
					sorting[sortIdx]._idx = nextSortIdx;
				}
			}
			else {
				// No sort on this column yet
				sorting.push( [ colIdx, asSorting[0], 0 ] );
				sorting[sorting.length-1]._idx = 0;
			}
		}
		else if ( sorting.length && sorting[0][0] == colIdx ) {
			// Single column - already sorting on this column, modify the sort
			nextSortIdx = next( sorting[0] );
	
			sorting.length = 1;
			sorting[0][1] = asSorting[ nextSortIdx ];
			sorting[0]._idx = nextSortIdx;
		}
		else {
			// Single column - sort only on this column
			sorting.length = 0;
			sorting.push( [ colIdx, asSorting[0] ] );
			sorting[0]._idx = 0;
		}
	
		// Run the sort by calling a full redraw
		_fnReDraw( settings );
	
		// callback used for async user interaction
		if ( typeof callback == 'function' ) {
			callback( settings );
		}
	}
	
	
	/**
	 * Attach a sort handler (click) to a node
	 *  @param {object} settings dataTables settings object
	 *  @param {node} attachTo node to attach the handler to
	 *  @param {int} colIdx column sorting index
	 *  @param {function} [callback] callback function
	 *  @memberof DataTable#oApi
	 */
	function _fnSortAttachListener ( settings, attachTo, colIdx, callback )
	{
		var col = settings.aoColumns[ colIdx ];
	
		_fnBindAction( attachTo, {}, function (e) {
			/* If the column is not sortable - don't to anything */
			if ( col.bSortable === false ) {
				return;
			}
	
			// If processing is enabled use a timeout to allow the processing
			// display to be shown - otherwise to it synchronously
			if ( settings.oFeatures.bProcessing ) {
				_fnProcessingDisplay( settings, true );
	
				setTimeout( function() {
					_fnSortListener( settings, colIdx, e.shiftKey, callback );
	
					// In server-side processing, the draw callback will remove the
					// processing display
					if ( _fnDataSource( settings ) !== 'ssp' ) {
						_fnProcessingDisplay( settings, false );
					}
				}, 0 );
			}
			else {
				_fnSortListener( settings, colIdx, e.shiftKey, callback );
			}
		} );
	}
	
	
	/**
	 * Set the sorting classes on table's body, Note: it is safe to call this function
	 * when bSort and bSortClasses are false
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnSortingClasses( settings )
	{
		var oldSort = settings.aLastSort;
		var sortClass = settings.oClasses.sSortColumn;
		var sort = _fnSortFlatten( settings );
		var features = settings.oFeatures;
		var i, ien, colIdx;
	
		if ( features.bSort && features.bSortClasses ) {
			// Remove old sorting classes
			for ( i=0, ien=oldSort.length ; i<ien ; i++ ) {
				colIdx = oldSort[i].src;
	
				// Remove column sorting
				$( _pluck( settings.aoData, 'anCells', colIdx ) )
					.removeClass( sortClass + (i<2 ? i+1 : 3) );
			}
	
			// Add new column sorting
			for ( i=0, ien=sort.length ; i<ien ; i++ ) {
				colIdx = sort[i].src;
	
				$( _pluck( settings.aoData, 'anCells', colIdx ) )
					.addClass( sortClass + (i<2 ? i+1 : 3) );
			}
		}
	
		settings.aLastSort = sort;
	}
	
	
	// Get the data to sort a column, be it from cache, fresh (populating the
	// cache), or from a sort formatter
	function _fnSortData( settings, idx )
	{
		// Custom sorting function - provided by the sort data type
		var column = settings.aoColumns[ idx ];
		var customSort = DataTable.ext.order[ column.sSortDataType ];
		var customData;
	
		if ( customSort ) {
			customData = customSort.call( settings.oInstance, settings, idx,
				_fnColumnIndexToVisible( settings, idx )
			);
		}
	
		// Use / populate cache
		var row, cellData;
		var formatter = DataTable.ext.type.order[ column.sType+"-pre" ];
	
		for ( var i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
			row = settings.aoData[i];
	
			if ( ! row._aSortData ) {
				row._aSortData = [];
			}
	
			if ( ! row._aSortData[idx] || customSort ) {
				cellData = customSort ?
					customData[i] : // If there was a custom sort function, use data from there
					_fnGetCellData( settings, i, idx, 'sort' );
	
				row._aSortData[ idx ] = formatter ?
					formatter( cellData ) :
					cellData;
			}
		}
	}
	
	
	
	/**
	 * Save the state of a table
	 *  @param {object} oSettings dataTables settings object
	 *  @memberof DataTable#oApi
	 */
	function _fnSaveState ( settings )
	{
		if ( !settings.oFeatures.bStateSave || settings.bDestroying )
		{
			return;
		}
	
		/* Store the interesting variables */
		var state = {
			time:    +new Date(),
			start:   settings._iDisplayStart,
			length:  settings._iDisplayLength,
			order:   $.extend( true, [], settings.aaSorting ),
			search:  _fnSearchToCamel( settings.oPreviousSearch ),
			columns: $.map( settings.aoColumns, function ( col, i ) {
				return {
					visible: col.bVisible,
					search: _fnSearchToCamel( settings.aoPreSearchCols[i] )
				};
			} )
		};
	
		_fnCallbackFire( settings, "aoStateSaveParams", 'stateSaveParams', [settings, state] );
	
		settings.oSavedState = state;
		settings.fnStateSaveCallback.call( settings.oInstance, settings, state );
	}
	
	
	/**
	 * Attempt to load a saved table state
	 *  @param {object} oSettings dataTables settings object
	 *  @param {object} oInit DataTables init object so we can override settings
	 *  @param {function} callback Callback to execute when the state has been loaded
	 *  @memberof DataTable#oApi
	 */
	function _fnLoadState ( settings, oInit, callback )
	{
		var i, ien;
		var columns = settings.aoColumns;
		var loaded = function ( s ) {
			if ( ! s || ! s.time ) {
				callback();
				return;
			}
	
			// Allow custom and plug-in manipulation functions to alter the saved data set and
			// cancelling of loading by returning false
			var abStateLoad = _fnCallbackFire( settings, 'aoStateLoadParams', 'stateLoadParams', [settings, s] );
			if ( $.inArray( false, abStateLoad ) !== -1 ) {
				callback();
				return;
			}
	
			// Reject old data
			var duration = settings.iStateDuration;
			if ( duration > 0 && s.time < +new Date() - (duration*1000) ) {
				callback();
				return;
			}
	
			// Number of columns have changed - all bets are off, no restore of settings
			if ( s.columns && columns.length !== s.columns.length ) {
				callback();
				return;
			}
	
			// Store the saved state so it might be accessed at any time
			settings.oLoadedState = $.extend( true, {}, s );
	
			// Restore key features - todo - for 1.11 this needs to be done by
			// subscribed events
			if ( s.start !== undefined ) {
				settings._iDisplayStart    = s.start;
				settings.iInitDisplayStart = s.start;
			}
			if ( s.length !== undefined ) {
				settings._iDisplayLength   = s.length;
			}
	
			// Order
			if ( s.order !== undefined ) {
				settings.aaSorting = [];
				$.each( s.order, function ( i, col ) {
					settings.aaSorting.push( col[0] >= columns.length ?
						[ 0, col[1] ] :
						col
					);
				} );
			}
	
			// Search
			if ( s.search !== undefined ) {
				$.extend( settings.oPreviousSearch, _fnSearchToHung( s.search ) );
			}
	
			// Columns
			//
			if ( s.columns ) {
				for ( i=0, ien=s.columns.length ; i<ien ; i++ ) {
					var col = s.columns[i];
	
					// Visibility
					if ( col.visible !== undefined ) {
						columns[i].bVisible = col.visible;
					}
	
					// Search
					if ( col.search !== undefined ) {
						$.extend( settings.aoPreSearchCols[i], _fnSearchToHung( col.search ) );
					}
				}
			}
	
			_fnCallbackFire( settings, 'aoStateLoaded', 'stateLoaded', [settings, s] );
			callback();
		};
	
		if ( ! settings.oFeatures.bStateSave ) {
			callback();
			return;
		}
	
		var state = settings.fnStateLoadCallback.call( settings.oInstance, settings, loaded );
	
		if ( state !== undefined ) {
			loaded( state );
		}
		// otherwise, wait for the loaded callback to be executed
	}
	
	
	/**
	 * Return the settings object for a particular table
	 *  @param {node} table table we are using as a dataTable
	 *  @returns {object} Settings object - or null if not found
	 *  @memberof DataTable#oApi
	 */
	function _fnSettingsFromNode ( table )
	{
		var settings = DataTable.settings;
		var idx = $.inArray( table, _pluck( settings, 'nTable' ) );
	
		return idx !== -1 ?
			settings[ idx ] :
			null;
	}
	
	
	/**
	 * Log an error message
	 *  @param {object} settings dataTables settings object
	 *  @param {int} level log error messages, or display them to the user
	 *  @param {string} msg error message
	 *  @param {int} tn Technical note id to get more information about the error.
	 *  @memberof DataTable#oApi
	 */
	function _fnLog( settings, level, msg, tn )
	{
		msg = 'DataTables warning: '+
			(settings ? 'table id='+settings.sTableId+' - ' : '')+msg;
	
		if ( tn ) {
			msg += '. For more information about this error, please see '+
			'http://datatables.net/tn/'+tn;
		}
	
		if ( ! level  ) {
			// Backwards compatibility pre 1.10
			var ext = DataTable.ext;
			var type = ext.sErrMode || ext.errMode;
	
			if ( settings ) {
				_fnCallbackFire( settings, null, 'error', [ settings, tn, msg ] );
			}
	
			if ( type == 'alert' ) {
				alert( msg );
			}
			else if ( type == 'throw' ) {
				throw new Error(msg);
			}
			else if ( typeof type == 'function' ) {
				type( settings, tn, msg );
			}
		}
		else if ( window.console && console.log ) {
			console.log( msg );
		}
	}
	
	
	/**
	 * See if a property is defined on one object, if so assign it to the other object
	 *  @param {object} ret target object
	 *  @param {object} src source object
	 *  @param {string} name property
	 *  @param {string} [mappedName] name to map too - optional, name used if not given
	 *  @memberof DataTable#oApi
	 */
	function _fnMap( ret, src, name, mappedName )
	{
		if ( Array.isArray( name ) ) {
			$.each( name, function (i, val) {
				if ( Array.isArray( val ) ) {
					_fnMap( ret, src, val[0], val[1] );
				}
				else {
					_fnMap( ret, src, val );
				}
			} );
	
			return;
		}
	
		if ( mappedName === undefined ) {
			mappedName = name;
		}
	
		if ( src[name] !== undefined ) {
			ret[mappedName] = src[name];
		}
	}
	
	
	/**
	 * Extend objects - very similar to jQuery.extend, but deep copy objects, and
	 * shallow copy arrays. The reason we need to do this, is that we don't want to
	 * deep copy array init values (such as aaSorting) since the dev wouldn't be
	 * able to override them, but we do want to deep copy arrays.
	 *  @param {object} out Object to extend
	 *  @param {object} extender Object from which the properties will be applied to
	 *      out
	 *  @param {boolean} breakRefs If true, then arrays will be sliced to take an
	 *      independent copy with the exception of the `data` or `aaData` parameters
	 *      if they are present. This is so you can pass in a collection to
	 *      DataTables and have that used as your data source without breaking the
	 *      references
	 *  @returns {object} out Reference, just for convenience - out === the return.
	 *  @memberof DataTable#oApi
	 *  @todo This doesn't take account of arrays inside the deep copied objects.
	 */
	function _fnExtend( out, extender, breakRefs )
	{
		var val;
	
		for ( var prop in extender ) {
			if ( extender.hasOwnProperty(prop) ) {
				val = extender[prop];
	
				if ( $.isPlainObject( val ) ) {
					if ( ! $.isPlainObject( out[prop] ) ) {
						out[prop] = {};
					}
					$.extend( true, out[prop], val );
				}
				else if ( breakRefs && prop !== 'data' && prop !== 'aaData' && Array.isArray(val) ) {
					out[prop] = val.slice();
				}
				else {
					out[prop] = val;
				}
			}
		}
	
		return out;
	}
	
	
	/**
	 * Bind an event handers to allow a click or return key to activate the callback.
	 * This is good for accessibility since a return on the keyboard will have the
	 * same effect as a click, if the element has focus.
	 *  @param {element} n Element to bind the action to
	 *  @param {object} oData Data object to pass to the triggered function
	 *  @param {function} fn Callback function for when the event is triggered
	 *  @memberof DataTable#oApi
	 */
	function _fnBindAction( n, oData, fn )
	{
		$(n)
			.on( 'click.DT', oData, function (e) {
					$(n).trigger('blur'); // Remove focus outline for mouse users
					fn(e);
				} )
			.on( 'keypress.DT', oData, function (e){
					if ( e.which === 13 ) {
						e.preventDefault();
						fn(e);
					}
				} )
			.on( 'selectstart.DT', function () {
					/* Take the brutal approach to cancelling text selection */
					return false;
				} );
	}
	
	
	/**
	 * Register a callback function. Easily allows a callback function to be added to
	 * an array store of callback functions that can then all be called together.
	 *  @param {object} oSettings dataTables settings object
	 *  @param {string} sStore Name of the array storage for the callbacks in oSettings
	 *  @param {function} fn Function to be called back
	 *  @param {string} sName Identifying name for the callback (i.e. a label)
	 *  @memberof DataTable#oApi
	 */
	function _fnCallbackReg( oSettings, sStore, fn, sName )
	{
		if ( fn )
		{
			oSettings[sStore].push( {
				"fn": fn,
				"sName": sName
			} );
		}
	}
	
	
	/**
	 * Fire callback functions and trigger events. Note that the loop over the
	 * callback array store is done backwards! Further note that you do not want to
	 * fire off triggers in time sensitive applications (for example cell creation)
	 * as its slow.
	 *  @param {object} settings dataTables settings object
	 *  @param {string} callbackArr Name of the array storage for the callbacks in
	 *      oSettings
	 *  @param {string} eventName Name of the jQuery custom event to trigger. If
	 *      null no trigger is fired
	 *  @param {array} args Array of arguments to pass to the callback function /
	 *      trigger
	 *  @memberof DataTable#oApi
	 */
	function _fnCallbackFire( settings, callbackArr, eventName, args )
	{
		var ret = [];
	
		if ( callbackArr ) {
			ret = $.map( settings[callbackArr].slice().reverse(), function (val, i) {
				return val.fn.apply( settings.oInstance, args );
			} );
		}
	
		if ( eventName !== null ) {
			var e = $.Event( eventName+'.dt' );
	
			$(settings.nTable).trigger( e, args );
	
			ret.push( e.result );
		}
	
		return ret;
	}
	
	
	function _fnLengthOverflow ( settings )
	{
		var
			start = settings._iDisplayStart,
			end = settings.fnDisplayEnd(),
			len = settings._iDisplayLength;
	
		/* If we have space to show extra rows (backing up from the end point - then do so */
		if ( start >= end )
		{
			start = end - len;
		}
	
		// Keep the start record on the current page
		start -= (start % len);
	
		if ( len === -1 || start < 0 )
		{
			start = 0;
		}
	
		settings._iDisplayStart = start;
	}
	
	
	function _fnRenderer( settings, type )
	{
		var renderer = settings.renderer;
		var host = DataTable.ext.renderer[type];
	
		if ( $.isPlainObject( renderer ) && renderer[type] ) {
			// Specific renderer for this type. If available use it, otherwise use
			// the default.
			return host[renderer[type]] || host._;
		}
		else if ( typeof renderer === 'string' ) {
			// Common renderer - if there is one available for this type use it,
			// otherwise use the default
			return host[renderer] || host._;
		}
	
		// Use the default
		return host._;
	}
	
	
	/**
	 * Detect the data source being used for the table. Used to simplify the code
	 * a little (ajax) and to make it compress a little smaller.
	 *
	 *  @param {object} settings dataTables settings object
	 *  @returns {string} Data source
	 *  @memberof DataTable#oApi
	 */
	function _fnDataSource ( settings )
	{
		if ( settings.oFeatures.bServerSide ) {
			return 'ssp';
		}
		else if ( settings.ajax || settings.sAjaxSource ) {
			return 'ajax';
		}
		return 'dom';
	}
	

	
	
	/**
	 * Computed structure of the DataTables API, defined by the options passed to
	 * `DataTable.Api.register()` when building the API.
	 *
	 * The structure is built in order to speed creation and extension of the Api
	 * objects since the extensions are effectively pre-parsed.
	 *
	 * The array is an array of objects with the following structure, where this
	 * base array represents the Api prototype base:
	 *
	 *     [
	 *       {
	 *         name:      'data'                -- string   - Property name
	 *         val:       function () {},       -- function - Api method (or undefined if just an object
	 *         methodExt: [ ... ],              -- array    - Array of Api object definitions to extend the method result
	 *         propExt:   [ ... ]               -- array    - Array of Api object definitions to extend the property
	 *       },
	 *       {
	 *         name:     'row'
	 *         val:       {},
	 *         methodExt: [ ... ],
	 *         propExt:   [
	 *           {
	 *             name:      'data'
	 *             val:       function () {},
	 *             methodExt: [ ... ],
	 *             propExt:   [ ... ]
	 *           },
	 *           ...
	 *         ]
	 *       }
	 *     ]
	 *
	 * @type {Array}
	 * @ignore
	 */
	var __apiStruct = [];
	
	
	/**
	 * `Array.prototype` reference.
	 *
	 * @type object
	 * @ignore
	 */
	var __arrayProto = Array.prototype;
	
	
	/**
	 * Abstraction for `context` parameter of the `Api` constructor to allow it to
	 * take several different forms for ease of use.
	 *
	 * Each of the input parameter types will be converted to a DataTables settings
	 * object where possible.
	 *
	 * @param  {string|node|jQuery|object} mixed DataTable identifier. Can be one
	 *   of:
	 *
	 *   * `string` - jQuery selector. Any DataTables' matching the given selector
	 *     with be found and used.
	 *   * `node` - `TABLE` node which has already been formed into a DataTable.
	 *   * `jQuery` - A jQuery object of `TABLE` nodes.
	 *   * `object` - DataTables settings object
	 *   * `DataTables.Api` - API instance
	 * @return {array|null} Matching DataTables settings objects. `null` or
	 *   `undefined` is returned if no matching DataTable is found.
	 * @ignore
	 */
	var _toSettings = function ( mixed )
	{
		var idx, jq;
		var settings = DataTable.settings;
		var tables = $.map( settings, function (el, i) {
			return el.nTable;
		} );
	
		if ( ! mixed ) {
			return [];
		}
		else if ( mixed.nTable && mixed.oApi ) {
			// DataTables settings object
			return [ mixed ];
		}
		else if ( mixed.nodeName && mixed.nodeName.toLowerCase() === 'table' ) {
			// Table node
			idx = $.inArray( mixed, tables );
			return idx !== -1 ? [ settings[idx] ] : null;
		}
		else if ( mixed && typeof mixed.settings === 'function' ) {
			return mixed.settings().toArray();
		}
		else if ( typeof mixed === 'string' ) {
			// jQuery selector
			jq = $(mixed);
		}
		else if ( mixed instanceof $ ) {
			// jQuery object (also DataTables instance)
			jq = mixed;
		}
	
		if ( jq ) {
			return jq.map( function(i) {
				idx = $.inArray( this, tables );
				return idx !== -1 ? settings[idx] : null;
			} ).toArray();
		}
	};
	
	
	/**
	 * DataTables API class - used to control and interface with  one or more
	 * DataTables enhanced tables.
	 *
	 * The API class is heavily based on jQuery, presenting a chainable interface
	 * that you can use to interact with tables. Each instance of the API class has
	 * a "context" - i.e. the tables that it will operate on. This could be a single
	 * table, all tables on a page or a sub-set thereof.
	 *
	 * Additionally the API is designed to allow you to easily work with the data in
	 * the tables, retrieving and manipulating it as required. This is done by
	 * presenting the API class as an array like interface. The contents of the
	 * array depend upon the actions requested by each method (for example
	 * `rows().nodes()` will return an array of nodes, while `rows().data()` will
	 * return an array of objects or arrays depending upon your table's
	 * configuration). The API object has a number of array like methods (`push`,
	 * `pop`, `reverse` etc) as well as additional helper methods (`each`, `pluck`,
	 * `unique` etc) to assist your working with the data held in a table.
	 *
	 * Most methods (those which return an Api instance) are chainable, which means
	 * the return from a method call also has all of the methods available that the
	 * top level object had. For example, these two calls are equivalent:
	 *
	 *     // Not chained
	 *     api.row.add( {...} );
	 *     api.draw();
	 *
	 *     // Chained
	 *     api.row.add( {...} ).draw();
	 *
	 * @class DataTable.Api
	 * @param {array|object|string|jQuery} context DataTable identifier. This is
	 *   used to define which DataTables enhanced tables this API will operate on.
	 *   Can be one of:
	 *
	 *   * `string` - jQuery selector. Any DataTables' matching the given selector
	 *     with be found and used.
	 *   * `node` - `TABLE` node which has already been formed into a DataTable.
	 *   * `jQuery` - A jQuery object of `TABLE` nodes.
	 *   * `object` - DataTables settings object
	 * @param {array} [data] Data to initialise the Api instance with.
	 *
	 * @example
	 *   // Direct initialisation during DataTables construction
	 *   var api = $('#example').DataTable();
	 *
	 * @example
	 *   // Initialisation using a DataTables jQuery object
	 *   var api = $('#example').dataTable().api();
	 *
	 * @example
	 *   // Initialisation as a constructor
	 *   var api = new $.fn.DataTable.Api( 'table.dataTable' );
	 */
	_Api = function ( context, data )
	{
		if ( ! (this instanceof _Api) ) {
			return new _Api( context, data );
		}
	
		var settings = [];
		var ctxSettings = function ( o ) {
			var a = _toSettings( o );
			if ( a ) {
				settings.push.apply( settings, a );
			}
		};
	
		if ( Array.isArray( context ) ) {
			for ( var i=0, ien=context.length ; i<ien ; i++ ) {
				ctxSettings( context[i] );
			}
		}
		else {
			ctxSettings( context );
		}
	
		// Remove duplicates
		this.context = _unique( settings );
	
		// Initial data
		if ( data ) {
			$.merge( this, data );
		}
	
		// selector
		this.selector = {
			rows: null,
			cols: null,
			opts: null
		};
	
		_Api.extend( this, this, __apiStruct );
	};
	
	DataTable.Api = _Api;
	
	// Don't destroy the existing prototype, just extend it. Required for jQuery 2's
	// isPlainObject.
	$.extend( _Api.prototype, {
		any: function ()
		{
			return this.count() !== 0;
		},
	
	
		concat:  __arrayProto.concat,
	
	
		context: [], // array of table settings objects
	
	
		count: function ()
		{
			return this.flatten().length;
		},
	
	
		each: function ( fn )
		{
			for ( var i=0, ien=this.length ; i<ien; i++ ) {
				fn.call( this, this[i], i, this );
			}
	
			return this;
		},
	
	
		eq: function ( idx )
		{
			var ctx = this.context;
	
			return ctx.length > idx ?
				new _Api( ctx[idx], this[idx] ) :
				null;
		},
	
	
		filter: function ( fn )
		{
			var a = [];
	
			if ( __arrayProto.filter ) {
				a = __arrayProto.filter.call( this, fn, this );
			}
			else {
				// Compatibility for browsers without EMCA-252-5 (JS 1.6)
				for ( var i=0, ien=this.length ; i<ien ; i++ ) {
					if ( fn.call( this, this[i], i, this ) ) {
						a.push( this[i] );
					}
				}
			}
	
			return new _Api( this.context, a );
		},
	
	
		flatten: function ()
		{
			var a = [];
			return new _Api( this.context, a.concat.apply( a, this.toArray() ) );
		},
	
	
		join:    __arrayProto.join,
	
	
		indexOf: __arrayProto.indexOf || function (obj, start)
		{
			for ( var i=(start || 0), ien=this.length ; i<ien ; i++ ) {
				if ( this[i] === obj ) {
					return i;
				}
			}
			return -1;
		},
	
		iterator: function ( flatten, type, fn, alwaysNew ) {
			var
				a = [], ret,
				i, ien, j, jen,
				context = this.context,
				rows, items, item,
				selector = this.selector;
	
			// Argument shifting
			if ( typeof flatten === 'string' ) {
				alwaysNew = fn;
				fn = type;
				type = flatten;
				flatten = false;
			}
	
			for ( i=0, ien=context.length ; i<ien ; i++ ) {
				var apiInst = new _Api( context[i] );
	
				if ( type === 'table' ) {
					ret = fn.call( apiInst, context[i], i );
	
					if ( ret !== undefined ) {
						a.push( ret );
					}
				}
				else if ( type === 'columns' || type === 'rows' ) {
					// this has same length as context - one entry for each table
					ret = fn.call( apiInst, context[i], this[i], i );
	
					if ( ret !== undefined ) {
						a.push( ret );
					}
				}
				else if ( type === 'column' || type === 'column-rows' || type === 'row' || type === 'cell' ) {
					// columns and rows share the same structure.
					// 'this' is an array of column indexes for each context
					items = this[i];
	
					if ( type === 'column-rows' ) {
						rows = _selector_row_indexes( context[i], selector.opts );
					}
	
					for ( j=0, jen=items.length ; j<jen ; j++ ) {
						item = items[j];
	
						if ( type === 'cell' ) {
							ret = fn.call( apiInst, context[i], item.row, item.column, i, j );
						}
						else {
							ret = fn.call( apiInst, context[i], item, i, j, rows );
						}
	
						if ( ret !== undefined ) {
							a.push( ret );
						}
					}
				}
			}
	
			if ( a.length || alwaysNew ) {
				var api = new _Api( context, flatten ? a.concat.apply( [], a ) : a );
				var apiSelector = api.selector;
				apiSelector.rows = selector.rows;
				apiSelector.cols = selector.cols;
				apiSelector.opts = selector.opts;
				return api;
			}
			return this;
		},
	
	
		lastIndexOf: __arrayProto.lastIndexOf || function (obj, start)
		{
			// Bit cheeky...
			return this.indexOf.apply( this.toArray.reverse(), arguments );
		},
	
	
		length:  0,
	
	
		map: function ( fn )
		{
			var a = [];
	
			if ( __arrayProto.map ) {
				a = __arrayProto.map.call( this, fn, this );
			}
			else {
				// Compatibility for browsers without EMCA-252-5 (JS 1.6)
				for ( var i=0, ien=this.length ; i<ien ; i++ ) {
					a.push( fn.call( this, this[i], i ) );
				}
			}
	
			return new _Api( this.context, a );
		},
	
	
		pluck: function ( prop )
		{
			return this.map( function ( el ) {
				return el[ prop ];
			} );
		},
	
		pop:     __arrayProto.pop,
	
	
		push:    __arrayProto.push,
	
	
		// Does not return an API instance
		reduce: __arrayProto.reduce || function ( fn, init )
		{
			return _fnReduce( this, fn, init, 0, this.length, 1 );
		},
	
	
		reduceRight: __arrayProto.reduceRight || function ( fn, init )
		{
			return _fnReduce( this, fn, init, this.length-1, -1, -1 );
		},
	
	
		reverse: __arrayProto.reverse,
	
	
		// Object with rows, columns and opts
		selector: null,
	
	
		shift:   __arrayProto.shift,
	
	
		slice: function () {
			return new _Api( this.context, this );
		},
	
	
		sort:    __arrayProto.sort, // ? name - order?
	
	
		splice:  __arrayProto.splice,
	
	
		toArray: function ()
		{
			return __arrayProto.slice.call( this );
		},
	
	
		to$: function ()
		{
			return $( this );
		},
	
	
		toJQuery: function ()
		{
			return $( this );
		},
	
	
		unique: function ()
		{
			return new _Api( this.context, _unique(this) );
		},
	
	
		unshift: __arrayProto.unshift
	} );
	
	
	_Api.extend = function ( scope, obj, ext )
	{
		// Only extend API instances and static properties of the API
		if ( ! ext.length || ! obj || ( ! (obj instanceof _Api) && ! obj.__dt_wrapper ) ) {
			return;
		}
	
		var
			i, ien,
			struct,
			methodScoping = function ( scope, fn, struc ) {
				return function () {
					var ret = fn.apply( scope, arguments );
	
					// Method extension
					_Api.extend( ret, ret, struc.methodExt );
					return ret;
				};
			};
	
		for ( i=0, ien=ext.length ; i<ien ; i++ ) {
			struct = ext[i];
	
			// Value
			obj[ struct.name ] = struct.type === 'function' ?
				methodScoping( scope, struct.val, struct ) :
				struct.type === 'object' ?
					{} :
					struct.val;
	
			obj[ struct.name ].__dt_wrapper = true;
	
			// Property extension
			_Api.extend( scope, obj[ struct.name ], struct.propExt );
		}
	};
	
	
	// @todo - Is there need for an augment function?
	// _Api.augment = function ( inst, name )
	// {
	// 	// Find src object in the structure from the name
	// 	var parts = name.split('.');
	
	// 	_Api.extend( inst, obj );
	// };
	
	
	//     [
	//       {
	//         name:      'data'                -- string   - Property name
	//         val:       function () {},       -- function - Api method (or undefined if just an object
	//         methodExt: [ ... ],              -- array    - Array of Api object definitions to extend the method result
	//         propExt:   [ ... ]               -- array    - Array of Api object definitions to extend the property
	//       },
	//       {
	//         name:     'row'
	//         val:       {},
	//         methodExt: [ ... ],
	//         propExt:   [
	//           {
	//             name:      'data'
	//             val:       function () {},
	//             methodExt: [ ... ],
	//             propExt:   [ ... ]
	//           },
	//           ...
	//         ]
	//       }
	//     ]
	
	_Api.register = _api_register = function ( name, val )
	{
		if ( Array.isArray( name ) ) {
			for ( var j=0, jen=name.length ; j<jen ; j++ ) {
				_Api.register( name[j], val );
			}
			return;
		}
	
		var
			i, ien,
			heir = name.split('.'),
			struct = __apiStruct,
			key, method;
	
		var find = function ( src, name ) {
			for ( var i=0, ien=src.length ; i<ien ; i++ ) {
				if ( src[i].name === name ) {
					return src[i];
				}
			}
			return null;
		};
	
		for ( i=0, ien=heir.length ; i<ien ; i++ ) {
			method = heir[i].indexOf('()') !== -1;
			key = method ?
				heir[i].replace('()', '') :
				heir[i];
	
			var src = find( struct, key );
			if ( ! src ) {
				src = {
					name:      key,
					val:       {},
					methodExt: [],
					propExt:   [],
					type:      'object'
				};
				struct.push( src );
			}
	
			if ( i === ien-1 ) {
				src.val = val;
				src.type = typeof val === 'function' ?
					'function' :
					$.isPlainObject( val ) ?
						'object' :
						'other';
			}
			else {
				struct = method ?
					src.methodExt :
					src.propExt;
			}
		}
	};
	
	_Api.registerPlural = _api_registerPlural = function ( pluralName, singularName, val ) {
		_Api.register( pluralName, val );
	
		_Api.register( singularName, function () {
			var ret = val.apply( this, arguments );
	
			if ( ret === this ) {
				// Returned item is the API instance that was passed in, return it
				return this;
			}
			else if ( ret instanceof _Api ) {
				// New API instance returned, want the value from the first item
				// in the returned array for the singular result.
				return ret.length ?
					Array.isArray( ret[0] ) ?
						new _Api( ret.context, ret[0] ) : // Array results are 'enhanced'
						ret[0] :
					undefined;
			}
	
			// Non-API return - just fire it back
			return ret;
		} );
	};
	
	
	/**
	 * Selector for HTML tables. Apply the given selector to the give array of
	 * DataTables settings objects.
	 *
	 * @param {string|integer} [selector] jQuery selector string or integer
	 * @param  {array} Array of DataTables settings objects to be filtered
	 * @return {array}
	 * @ignore
	 */
	var __table_selector = function ( selector, a )
	{
		if ( Array.isArray(selector) ) {
			return $.map( selector, function (item) {
				return __table_selector(item, a);
			} );
		}
	
		// Integer is used to pick out a table by index
		if ( typeof selector === 'number' ) {
			return [ a[ selector ] ];
		}
	
		// Perform a jQuery selector on the table nodes
		var nodes = $.map( a, function (el, i) {
			return el.nTable;
		} );
	
		return $(nodes)
			.filter( selector )
			.map( function (i) {
				// Need to translate back from the table node to the settings
				var idx = $.inArray( this, nodes );
				return a[ idx ];
			} )
			.toArray();
	};
	
	
	
	/**
	 * Context selector for the API's context (i.e. the tables the API instance
	 * refers to.
	 *
	 * @name    DataTable.Api#tables
	 * @param {string|integer} [selector] Selector to pick which tables the iterator
	 *   should operate on. If not given, all tables in the current context are
	 *   used. This can be given as a jQuery selector (for example `':gt(0)'`) to
	 *   select multiple tables or as an integer to select a single table.
	 * @returns {DataTable.Api} Returns a new API instance if a selector is given.
	 */
	_api_register( 'tables()', function ( selector ) {
		// A new instance is created if there was a selector specified
		return selector !== undefined && selector !== null ?
			new _Api( __table_selector( selector, this.context ) ) :
			this;
	} );
	
	
	_api_register( 'table()', function ( selector ) {
		var tables = this.tables( selector );
		var ctx = tables.context;
	
		// Truncate to the first matched table
		return ctx.length ?
			new _Api( ctx[0] ) :
			tables;
	} );
	
	
	_api_registerPlural( 'tables().nodes()', 'table().node()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTable;
		}, 1 );
	} );
	
	
	_api_registerPlural( 'tables().body()', 'table().body()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTBody;
		}, 1 );
	} );
	
	
	_api_registerPlural( 'tables().header()', 'table().header()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTHead;
		}, 1 );
	} );
	
	
	_api_registerPlural( 'tables().footer()', 'table().footer()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTFoot;
		}, 1 );
	} );
	
	
	_api_registerPlural( 'tables().containers()', 'table().container()' , function () {
		return this.iterator( 'table', function ( ctx ) {
			return ctx.nTableWrapper;
		}, 1 );
	} );
	
	
	
	/**
	 * Redraw the tables in the current context.
	 */
	_api_register( 'draw()', function ( paging ) {
		return this.iterator( 'table', function ( settings ) {
			if ( paging === 'page' ) {
				_fnDraw( settings );
			}
			else {
				if ( typeof paging === 'string' ) {
					paging = paging === 'full-hold' ?
						false :
						true;
				}
	
				_fnReDraw( settings, paging===false );
			}
		} );
	} );
	
	
	
	/**
	 * Get the current page index.
	 *
	 * @return {integer} Current page index (zero based)
	 *//**
	 * Set the current page.
	 *
	 * Note that if you attempt to show a page which does not exist, DataTables will
	 * not throw an error, but rather reset the paging.
	 *
	 * @param {integer|string} action The paging action to take. This can be one of:
	 *  * `integer` - The page index to jump to
	 *  * `string` - An action to take:
	 *    * `first` - Jump to first page.
	 *    * `next` - Jump to the next page
	 *    * `previous` - Jump to previous page
	 *    * `last` - Jump to the last page.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'page()', function ( action ) {
		if ( action === undefined ) {
			return this.page.info().page; // not an expensive call
		}
	
		// else, have an action to take on all tables
		return this.iterator( 'table', function ( settings ) {
			_fnPageChange( settings, action );
		} );
	} );
	
	
	/**
	 * Paging information for the first table in the current context.
	 *
	 * If you require paging information for another table, use the `table()` method
	 * with a suitable selector.
	 *
	 * @return {object} Object with the following properties set:
	 *  * `page` - Current page index (zero based - i.e. the first page is `0`)
	 *  * `pages` - Total number of pages
	 *  * `start` - Display index for the first record shown on the current page
	 *  * `end` - Display index for the last record shown on the current page
	 *  * `length` - Display length (number of records). Note that generally `start
	 *    + length = end`, but this is not always true, for example if there are
	 *    only 2 records to show on the final page, with a length of 10.
	 *  * `recordsTotal` - Full data set length
	 *  * `recordsDisplay` - Data set length once the current filtering criterion
	 *    are applied.
	 */
	_api_register( 'page.info()', function ( action ) {
		if ( this.context.length === 0 ) {
			return undefined;
		}
	
		var
			settings   = this.context[0],
			start      = settings._iDisplayStart,
			len        = settings.oFeatures.bPaginate ? settings._iDisplayLength : -1,
			visRecords = settings.fnRecordsDisplay(),
			all        = len === -1;
	
		return {
			"page":           all ? 0 : Math.floor( start / len ),
			"pages":          all ? 1 : Math.ceil( visRecords / len ),
			"start":          start,
			"end":            settings.fnDisplayEnd(),
			"length":         len,
			"recordsTotal":   settings.fnRecordsTotal(),
			"recordsDisplay": visRecords,
			"serverSide":     _fnDataSource( settings ) === 'ssp'
		};
	} );
	
	
	/**
	 * Get the current page length.
	 *
	 * @return {integer} Current page length. Note `-1` indicates that all records
	 *   are to be shown.
	 *//**
	 * Set the current page length.
	 *
	 * @param {integer} Page length to set. Use `-1` to show all records.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'page.len()', function ( len ) {
		// Note that we can't call this function 'length()' because `length`
		// is a Javascript property of functions which defines how many arguments
		// the function expects.
		if ( len === undefined ) {
			return this.context.length !== 0 ?
				this.context[0]._iDisplayLength :
				undefined;
		}
	
		// else, set the page length
		return this.iterator( 'table', function ( settings ) {
			_fnLengthChange( settings, len );
		} );
	} );
	
	
	
	var __reload = function ( settings, holdPosition, callback ) {
		// Use the draw event to trigger a callback
		if ( callback ) {
			var api = new _Api( settings );
	
			api.one( 'draw', function () {
				callback( api.ajax.json() );
			} );
		}
	
		if ( _fnDataSource( settings ) == 'ssp' ) {
			_fnReDraw( settings, holdPosition );
		}
		else {
			_fnProcessingDisplay( settings, true );
	
			// Cancel an existing request
			var xhr = settings.jqXHR;
			if ( xhr && xhr.readyState !== 4 ) {
				xhr.abort();
			}
	
			// Trigger xhr
			_fnBuildAjax( settings, [], function( json ) {
				_fnClearTable( settings );
	
				var data = _fnAjaxDataSrc( settings, json );
				for ( var i=0, ien=data.length ; i<ien ; i++ ) {
					_fnAddData( settings, data[i] );
				}
	
				_fnReDraw( settings, holdPosition );
				_fnProcessingDisplay( settings, false );
			} );
		}
	};
	
	
	/**
	 * Get the JSON response from the last Ajax request that DataTables made to the
	 * server. Note that this returns the JSON from the first table in the current
	 * context.
	 *
	 * @return {object} JSON received from the server.
	 */
	_api_register( 'ajax.json()', function () {
		var ctx = this.context;
	
		if ( ctx.length > 0 ) {
			return ctx[0].json;
		}
	
		// else return undefined;
	} );
	
	
	/**
	 * Get the data submitted in the last Ajax request
	 */
	_api_register( 'ajax.params()', function () {
		var ctx = this.context;
	
		if ( ctx.length > 0 ) {
			return ctx[0].oAjaxData;
		}
	
		// else return undefined;
	} );
	
	
	/**
	 * Reload tables from the Ajax data source. Note that this function will
	 * automatically re-draw the table when the remote data has been loaded.
	 *
	 * @param {boolean} [reset=true] Reset (default) or hold the current paging
	 *   position. A full re-sort and re-filter is performed when this method is
	 *   called, which is why the pagination reset is the default action.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'ajax.reload()', function ( callback, resetPaging ) {
		return this.iterator( 'table', function (settings) {
			__reload( settings, resetPaging===false, callback );
		} );
	} );
	
	
	/**
	 * Get the current Ajax URL. Note that this returns the URL from the first
	 * table in the current context.
	 *
	 * @return {string} Current Ajax source URL
	 *//**
	 * Set the Ajax URL. Note that this will set the URL for all tables in the
	 * current context.
	 *
	 * @param {string} url URL to set.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'ajax.url()', function ( url ) {
		var ctx = this.context;
	
		if ( url === undefined ) {
			// get
			if ( ctx.length === 0 ) {
				return undefined;
			}
			ctx = ctx[0];
	
			return ctx.ajax ?
				$.isPlainObject( ctx.ajax ) ?
					ctx.ajax.url :
					ctx.ajax :
				ctx.sAjaxSource;
		}
	
		// set
		return this.iterator( 'table', function ( settings ) {
			if ( $.isPlainObject( settings.ajax ) ) {
				settings.ajax.url = url;
			}
			else {
				settings.ajax = url;
			}
			// No need to consider sAjaxSource here since DataTables gives priority
			// to `ajax` over `sAjaxSource`. So setting `ajax` here, renders any
			// value of `sAjaxSource` redundant.
		} );
	} );
	
	
	/**
	 * Load data from the newly set Ajax URL. Note that this method is only
	 * available when `ajax.url()` is used to set a URL. Additionally, this method
	 * has the same effect as calling `ajax.reload()` but is provided for
	 * convenience when setting a new URL. Like `ajax.reload()` it will
	 * automatically redraw the table once the remote data has been loaded.
	 *
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'ajax.url().load()', function ( callback, resetPaging ) {
		// Same as a reload, but makes sense to present it for easy access after a
		// url change
		return this.iterator( 'table', function ( ctx ) {
			__reload( ctx, resetPaging===false, callback );
		} );
	} );
	
	
	
	
	var _selector_run = function ( type, selector, selectFn, settings, opts )
	{
		var
			out = [], res,
			a, i, ien, j, jen,
			selectorType = typeof selector;
	
		// Can't just check for isArray here, as an API or jQuery instance might be
		// given with their array like look
		if ( ! selector || selectorType === 'string' || selectorType === 'function' || selector.length === undefined ) {
			selector = [ selector ];
		}
	
		for ( i=0, ien=selector.length ; i<ien ; i++ ) {
			// Only split on simple strings - complex expressions will be jQuery selectors
			a = selector[i] && selector[i].split && ! selector[i].match(/[\[\(:]/) ?
				selector[i].split(',') :
				[ selector[i] ];
	
			for ( j=0, jen=a.length ; j<jen ; j++ ) {
				res = selectFn( typeof a[j] === 'string' ? (a[j]).trim() : a[j] );
	
				if ( res && res.length ) {
					out = out.concat( res );
				}
			}
		}
	
		// selector extensions
		var ext = _ext.selector[ type ];
		if ( ext.length ) {
			for ( i=0, ien=ext.length ; i<ien ; i++ ) {
				out = ext[i]( settings, opts, out );
			}
		}
	
		return _unique( out );
	};
	
	
	var _selector_opts = function ( opts )
	{
		if ( ! opts ) {
			opts = {};
		}
	
		// Backwards compatibility for 1.9- which used the terminology filter rather
		// than search
		if ( opts.filter && opts.search === undefined ) {
			opts.search = opts.filter;
		}
	
		return $.extend( {
			search: 'none',
			order: 'current',
			page: 'all'
		}, opts );
	};
	
	
	var _selector_first = function ( inst )
	{
		// Reduce the API instance to the first item found
		for ( var i=0, ien=inst.length ; i<ien ; i++ ) {
			if ( inst[i].length > 0 ) {
				// Assign the first element to the first item in the instance
				// and truncate the instance and context
				inst[0] = inst[i];
				inst[0].length = 1;
				inst.length = 1;
				inst.context = [ inst.context[i] ];
	
				return inst;
			}
		}
	
		// Not found - return an empty instance
		inst.length = 0;
		return inst;
	};
	
	
	var _selector_row_indexes = function ( settings, opts )
	{
		var
			i, ien, tmp, a=[],
			displayFiltered = settings.aiDisplay,
			displayMaster = settings.aiDisplayMaster;
	
		var
			search = opts.search,  // none, applied, removed
			order  = opts.order,   // applied, current, index (original - compatibility with 1.9)
			page   = opts.page;    // all, current
	
		if ( _fnDataSource( settings ) == 'ssp' ) {
			// In server-side processing mode, most options are irrelevant since
			// rows not shown don't exist and the index order is the applied order
			// Removed is a special case - for consistency just return an empty
			// array
			return search === 'removed' ?
				[] :
				_range( 0, displayMaster.length );
		}
		else if ( page == 'current' ) {
			// Current page implies that order=current and fitler=applied, since it is
			// fairly senseless otherwise, regardless of what order and search actually
			// are
			for ( i=settings._iDisplayStart, ien=settings.fnDisplayEnd() ; i<ien ; i++ ) {
				a.push( displayFiltered[i] );
			}
		}
		else if ( order == 'current' || order == 'applied' ) {
			if ( search == 'none') {
				a = displayMaster.slice();
			}
			else if ( search == 'applied' ) {
				a = displayFiltered.slice();
			}
			else if ( search == 'removed' ) {
				// O(n+m) solution by creating a hash map
				var displayFilteredMap = {};
	
				for ( var i=0, ien=displayFiltered.length ; i<ien ; i++ ) {
					displayFilteredMap[displayFiltered[i]] = null;
				}
	
				a = $.map( displayMaster, function (el) {
					return ! displayFilteredMap.hasOwnProperty(el) ?
						el :
						null;
				} );
			}
		}
		else if ( order == 'index' || order == 'original' ) {
			for ( i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
				if ( search == 'none' ) {
					a.push( i );
				}
				else { // applied | removed
					tmp = $.inArray( i, displayFiltered );
	
					if ((tmp === -1 && search == 'removed') ||
						(tmp >= 0   && search == 'applied') )
					{
						a.push( i );
					}
				}
			}
		}
	
		return a;
	};
	
	
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 * Rows
	 *
	 * {}          - no selector - use all available rows
	 * {integer}   - row aoData index
	 * {node}      - TR node
	 * {string}    - jQuery selector to apply to the TR elements
	 * {array}     - jQuery array of nodes, or simply an array of TR nodes
	 *
	 */
	var __row_selector = function ( settings, selector, opts )
	{
		var rows;
		var run = function ( sel ) {
			var selInt = _intVal( sel );
			var i, ien;
			var aoData = settings.aoData;
	
			// Short cut - selector is a number and no options provided (default is
			// all records, so no need to check if the index is in there, since it
			// must be - dev error if the index doesn't exist).
			if ( selInt !== null && ! opts ) {
				return [ selInt ];
			}
	
			if ( ! rows ) {
				rows = _selector_row_indexes( settings, opts );
			}
	
			if ( selInt !== null && $.inArray( selInt, rows ) !== -1 ) {
				// Selector - integer
				return [ selInt ];
			}
			else if ( sel === null || sel === undefined || sel === '' ) {
				// Selector - none
				return rows;
			}
	
			// Selector - function
			if ( typeof sel === 'function' ) {
				return $.map( rows, function (idx) {
					var row = aoData[ idx ];
					return sel( idx, row._aData, row.nTr ) ? idx : null;
				} );
			}
	
			// Selector - node
			if ( sel.nodeName ) {
				var rowIdx = sel._DT_RowIndex;  // Property added by DT for fast lookup
				var cellIdx = sel._DT_CellIndex;
	
				if ( rowIdx !== undefined ) {
					// Make sure that the row is actually still present in the table
					return aoData[ rowIdx ] && aoData[ rowIdx ].nTr === sel ?
						[ rowIdx ] :
						[];
				}
				else if ( cellIdx ) {
					return aoData[ cellIdx.row ] && aoData[ cellIdx.row ].nTr === sel.parentNode ?
						[ cellIdx.row ] :
						[];
				}
				else {
					var host = $(sel).closest('*[data-dt-row]');
					return host.length ?
						[ host.data('dt-row') ] :
						[];
				}
			}
	
			// ID selector. Want to always be able to select rows by id, regardless
			// of if the tr element has been created or not, so can't rely upon
			// jQuery here - hence a custom implementation. This does not match
			// Sizzle's fast selector or HTML4 - in HTML5 the ID can be anything,
			// but to select it using a CSS selector engine (like Sizzle or
			// querySelect) it would need to need to be escaped for some characters.
			// DataTables simplifies this for row selectors since you can select
			// only a row. A # indicates an id any anything that follows is the id -
			// unescaped.
			if ( typeof sel === 'string' && sel.charAt(0) === '#' ) {
				// get row index from id
				var rowObj = settings.aIds[ sel.replace( /^#/, '' ) ];
				if ( rowObj !== undefined ) {
					return [ rowObj.idx ];
				}
	
				// need to fall through to jQuery in case there is DOM id that
				// matches
			}
			
			// Get nodes in the order from the `rows` array with null values removed
			var nodes = _removeEmpty(
				_pluck_order( settings.aoData, rows, 'nTr' )
			);
	
			// Selector - jQuery selector string, array of nodes or jQuery object/
			// As jQuery's .filter() allows jQuery objects to be passed in filter,
			// it also allows arrays, so this will cope with all three options
			return $(nodes)
				.filter( sel )
				.map( function () {
					return this._DT_RowIndex;
				} )
				.toArray();
		};
	
		return _selector_run( 'row', selector, run, settings, opts );
	};
	
	
	_api_register( 'rows()', function ( selector, opts ) {
		// argument shifting
		if ( selector === undefined ) {
			selector = '';
		}
		else if ( $.isPlainObject( selector ) ) {
			opts = selector;
			selector = '';
		}
	
		opts = _selector_opts( opts );
	
		var inst = this.iterator( 'table', function ( settings ) {
			return __row_selector( settings, selector, opts );
		}, 1 );
	
		// Want argument shifting here and in __row_selector?
		inst.selector.rows = selector;
		inst.selector.opts = opts;
	
		return inst;
	} );
	
	_api_register( 'rows().nodes()', function () {
		return this.iterator( 'row', function ( settings, row ) {
			return settings.aoData[ row ].nTr || undefined;
		}, 1 );
	} );
	
	_api_register( 'rows().data()', function () {
		return this.iterator( true, 'rows', function ( settings, rows ) {
			return _pluck_order( settings.aoData, rows, '_aData' );
		}, 1 );
	} );
	
	_api_registerPlural( 'rows().cache()', 'row().cache()', function ( type ) {
		return this.iterator( 'row', function ( settings, row ) {
			var r = settings.aoData[ row ];
			return type === 'search' ? r._aFilterData : r._aSortData;
		}, 1 );
	} );
	
	_api_registerPlural( 'rows().invalidate()', 'row().invalidate()', function ( src ) {
		return this.iterator( 'row', function ( settings, row ) {
			_fnInvalidate( settings, row, src );
		} );
	} );
	
	_api_registerPlural( 'rows().indexes()', 'row().index()', function () {
		return this.iterator( 'row', function ( settings, row ) {
			return row;
		}, 1 );
	} );
	
	_api_registerPlural( 'rows().ids()', 'row().id()', function ( hash ) {
		var a = [];
		var context = this.context;
	
		// `iterator` will drop undefined values, but in this case we want them
		for ( var i=0, ien=context.length ; i<ien ; i++ ) {
			for ( var j=0, jen=this[i].length ; j<jen ; j++ ) {
				var id = context[i].rowIdFn( context[i].aoData[ this[i][j] ]._aData );
				a.push( (hash === true ? '#' : '' )+ id );
			}
		}
	
		return new _Api( context, a );
	} );
	
	_api_registerPlural( 'rows().remove()', 'row().remove()', function () {
		var that = this;
	
		this.iterator( 'row', function ( settings, row, thatIdx ) {
			var data = settings.aoData;
			var rowData = data[ row ];
			var i, ien, j, jen;
			var loopRow, loopCells;
	
			data.splice( row, 1 );
	
			// Update the cached indexes
			for ( i=0, ien=data.length ; i<ien ; i++ ) {
				loopRow = data[i];
				loopCells = loopRow.anCells;
	
				// Rows
				if ( loopRow.nTr !== null ) {
					loopRow.nTr._DT_RowIndex = i;
				}
	
				// Cells
				if ( loopCells !== null ) {
					for ( j=0, jen=loopCells.length ; j<jen ; j++ ) {
						loopCells[j]._DT_CellIndex.row = i;
					}
				}
			}
	
			// Delete from the display arrays
			_fnDeleteIndex( settings.aiDisplayMaster, row );
			_fnDeleteIndex( settings.aiDisplay, row );
			_fnDeleteIndex( that[ thatIdx ], row, false ); // maintain local indexes
	
			// For server-side processing tables - subtract the deleted row from the count
			if ( settings._iRecordsDisplay > 0 ) {
				settings._iRecordsDisplay--;
			}
	
			// Check for an 'overflow' they case for displaying the table
			_fnLengthOverflow( settings );
	
			// Remove the row's ID reference if there is one
			var id = settings.rowIdFn( rowData._aData );
			if ( id !== undefined ) {
				delete settings.aIds[ id ];
			}
		} );
	
		this.iterator( 'table', function ( settings ) {
			for ( var i=0, ien=settings.aoData.length ; i<ien ; i++ ) {
				settings.aoData[i].idx = i;
			}
		} );
	
		return this;
	} );
	
	
	_api_register( 'rows.add()', function ( rows ) {
		var newRows = this.iterator( 'table', function ( settings ) {
				var row, i, ien;
				var out = [];
	
				for ( i=0, ien=rows.length ; i<ien ; i++ ) {
					row = rows[i];
	
					if ( row.nodeName && row.nodeName.toUpperCase() === 'TR' ) {
						out.push( _fnAddTr( settings, row )[0] );
					}
					else {
						out.push( _fnAddData( settings, row ) );
					}
				}
	
				return out;
			}, 1 );
	
		// Return an Api.rows() extended instance, so rows().nodes() etc can be used
		var modRows = this.rows( -1 );
		modRows.pop();
		$.merge( modRows, newRows );
	
		return modRows;
	} );
	
	
	
	
	
	/**
	 *
	 */
	_api_register( 'row()', function ( selector, opts ) {
		return _selector_first( this.rows( selector, opts ) );
	} );
	
	
	_api_register( 'row().data()', function ( data ) {
		var ctx = this.context;
	
		if ( data === undefined ) {
			// Get
			return ctx.length && this.length ?
				ctx[0].aoData[ this[0] ]._aData :
				undefined;
		}
	
		// Set
		var row = ctx[0].aoData[ this[0] ];
		row._aData = data;
	
		// If the DOM has an id, and the data source is an array
		if ( Array.isArray( data ) && row.nTr && row.nTr.id ) {
			_fnSetObjectDataFn( ctx[0].rowId )( data, row.nTr.id );
		}
	
		// Automatically invalidate
		_fnInvalidate( ctx[0], this[0], 'data' );
	
		return this;
	} );
	
	
	_api_register( 'row().node()', function () {
		var ctx = this.context;
	
		return ctx.length && this.length ?
			ctx[0].aoData[ this[0] ].nTr || null :
			null;
	} );
	
	
	_api_register( 'row.add()', function ( row ) {
		// Allow a jQuery object to be passed in - only a single row is added from
		// it though - the first element in the set
		if ( row instanceof $ && row.length ) {
			row = row[0];
		}
	
		var rows = this.iterator( 'table', function ( settings ) {
			if ( row.nodeName && row.nodeName.toUpperCase() === 'TR' ) {
				return _fnAddTr( settings, row )[0];
			}
			return _fnAddData( settings, row );
		} );
	
		// Return an Api.rows() extended instance, with the newly added row selected
		return this.row( rows[0] );
	} );
	
	
	
	var __details_add = function ( ctx, row, data, klass )
	{
		// Convert to array of TR elements
		var rows = [];
		var addRow = function ( r, k ) {
			// Recursion to allow for arrays of jQuery objects
			if ( Array.isArray( r ) || r instanceof $ ) {
				for ( var i=0, ien=r.length ; i<ien ; i++ ) {
					addRow( r[i], k );
				}
				return;
			}
	
			// If we get a TR element, then just add it directly - up to the dev
			// to add the correct number of columns etc
			if ( r.nodeName && r.nodeName.toLowerCase() === 'tr' ) {
				rows.push( r );
			}
			else {
				// Otherwise create a row with a wrapper
				var created = $('<tr><td></td></tr>').addClass( k );
				$('td', created)
					.addClass( k )
					.html( r )
					[0].colSpan = _fnVisbleColumns( ctx );
	
				rows.push( created[0] );
			}
		};
	
		addRow( data, klass );
	
		if ( row._details ) {
			row._details.detach();
		}
	
		row._details = $(rows);
	
		// If the children were already shown, that state should be retained
		if ( row._detailsShow ) {
			row._details.insertAfter( row.nTr );
		}
	};
	
	
	var __details_remove = function ( api, idx )
	{
		var ctx = api.context;
	
		if ( ctx.length ) {
			var row = ctx[0].aoData[ idx !== undefined ? idx : api[0] ];
	
			if ( row && row._details ) {
				row._details.remove();
	
				row._detailsShow = undefined;
				row._details = undefined;
			}
		}
	};
	
	
	var __details_display = function ( api, show ) {
		var ctx = api.context;
	
		if ( ctx.length && api.length ) {
			var row = ctx[0].aoData[ api[0] ];
	
			if ( row._details ) {
				row._detailsShow = show;
	
				if ( show ) {
					row._details.insertAfter( row.nTr );
				}
				else {
					row._details.detach();
				}
	
				__details_events( ctx[0] );
			}
		}
	};
	
	
	var __details_events = function ( settings )
	{
		var api = new _Api( settings );
		var namespace = '.dt.DT_details';
		var drawEvent = 'draw'+namespace;
		var colvisEvent = 'column-visibility'+namespace;
		var destroyEvent = 'destroy'+namespace;
		var data = settings.aoData;
	
		api.off( drawEvent +' '+ colvisEvent +' '+ destroyEvent );
	
		if ( _pluck( data, '_details' ).length > 0 ) {
			// On each draw, insert the required elements into the document
			api.on( drawEvent, function ( e, ctx ) {
				if ( settings !== ctx ) {
					return;
				}
	
				api.rows( {page:'current'} ).eq(0).each( function (idx) {
					// Internal data grab
					var row = data[ idx ];
	
					if ( row._detailsShow ) {
						row._details.insertAfter( row.nTr );
					}
				} );
			} );
	
			// Column visibility change - update the colspan
			api.on( colvisEvent, function ( e, ctx, idx, vis ) {
				if ( settings !== ctx ) {
					return;
				}
	
				// Update the colspan for the details rows (note, only if it already has
				// a colspan)
				var row, visible = _fnVisbleColumns( ctx );
	
				for ( var i=0, ien=data.length ; i<ien ; i++ ) {
					row = data[i];
	
					if ( row._details ) {
						row._details.children('td[colspan]').attr('colspan', visible );
					}
				}
			} );
	
			// Table destroyed - nuke any child rows
			api.on( destroyEvent, function ( e, ctx ) {
				if ( settings !== ctx ) {
					return;
				}
	
				for ( var i=0, ien=data.length ; i<ien ; i++ ) {
					if ( data[i]._details ) {
						__details_remove( api, i );
					}
				}
			} );
		}
	};
	
	// Strings for the method names to help minification
	var _emp = '';
	var _child_obj = _emp+'row().child';
	var _child_mth = _child_obj+'()';
	
	// data can be:
	//  tr
	//  string
	//  jQuery or array of any of the above
	_api_register( _child_mth, function ( data, klass ) {
		var ctx = this.context;
	
		if ( data === undefined ) {
			// get
			return ctx.length && this.length ?
				ctx[0].aoData[ this[0] ]._details :
				undefined;
		}
		else if ( data === true ) {
			// show
			this.child.show();
		}
		else if ( data === false ) {
			// remove
			__details_remove( this );
		}
		else if ( ctx.length && this.length ) {
			// set
			__details_add( ctx[0], ctx[0].aoData[ this[0] ], data, klass );
		}
	
		return this;
	} );
	
	
	_api_register( [
		_child_obj+'.show()',
		_child_mth+'.show()' // only when `child()` was called with parameters (without
	], function ( show ) {   // it returns an object and this method is not executed)
		__details_display( this, true );
		return this;
	} );
	
	
	_api_register( [
		_child_obj+'.hide()',
		_child_mth+'.hide()' // only when `child()` was called with parameters (without
	], function () {         // it returns an object and this method is not executed)
		__details_display( this, false );
		return this;
	} );
	
	
	_api_register( [
		_child_obj+'.remove()',
		_child_mth+'.remove()' // only when `child()` was called with parameters (without
	], function () {           // it returns an object and this method is not executed)
		__details_remove( this );
		return this;
	} );
	
	
	_api_register( _child_obj+'.isShown()', function () {
		var ctx = this.context;
	
		if ( ctx.length && this.length ) {
			// _detailsShown as false or undefined will fall through to return false
			return ctx[0].aoData[ this[0] ]._detailsShow || false;
		}
		return false;
	} );
	
	
	
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 * Columns
	 *
	 * {integer}           - column index (>=0 count from left, <0 count from right)
	 * "{integer}:visIdx"  - visible column index (i.e. translate to column index)  (>=0 count from left, <0 count from right)
	 * "{integer}:visible" - alias for {integer}:visIdx  (>=0 count from left, <0 count from right)
	 * "{string}:name"     - column name
	 * "{string}"          - jQuery selector on column header nodes
	 *
	 */
	
	// can be an array of these items, comma separated list, or an array of comma
	// separated lists
	
	var __re_column_selector = /^([^:]+):(name|visIdx|visible)$/;
	
	
	// r1 and r2 are redundant - but it means that the parameters match for the
	// iterator callback in columns().data()
	var __columnData = function ( settings, column, r1, r2, rows ) {
		var a = [];
		for ( var row=0, ien=rows.length ; row<ien ; row++ ) {
			a.push( _fnGetCellData( settings, rows[row], column ) );
		}
		return a;
	};
	
	
	var __column_selector = function ( settings, selector, opts )
	{
		var
			columns = settings.aoColumns,
			names = _pluck( columns, 'sName' ),
			nodes = _pluck( columns, 'nTh' );
	
		var run = function ( s ) {
			var selInt = _intVal( s );
	
			// Selector - all
			if ( s === '' ) {
				return _range( columns.length );
			}
	
			// Selector - index
			if ( selInt !== null ) {
				return [ selInt >= 0 ?
					selInt : // Count from left
					columns.length + selInt // Count from right (+ because its a negative value)
				];
			}
	
			// Selector = function
			if ( typeof s === 'function' ) {
				var rows = _selector_row_indexes( settings, opts );
	
				return $.map( columns, function (col, idx) {
					return s(
							idx,
							__columnData( settings, idx, 0, 0, rows ),
							nodes[ idx ]
						) ? idx : null;
				} );
			}
	
			// jQuery or string selector
			var match = typeof s === 'string' ?
				s.match( __re_column_selector ) :
				'';
	
			if ( match ) {
				switch( match[2] ) {
					case 'visIdx':
					case 'visible':
						var idx = parseInt( match[1], 10 );
						// Visible index given, convert to column index
						if ( idx < 0 ) {
							// Counting from the right
							var visColumns = $.map( columns, function (col,i) {
								return col.bVisible ? i : null;
							} );
							return [ visColumns[ visColumns.length + idx ] ];
						}
						// Counting from the left
						return [ _fnVisibleToColumnIndex( settings, idx ) ];
	
					case 'name':
						// match by name. `names` is column index complete and in order
						return $.map( names, function (name, i) {
							return name === match[1] ? i : null;
						} );
	
					default:
						return [];
				}
			}
	
			// Cell in the table body
			if ( s.nodeName && s._DT_CellIndex ) {
				return [ s._DT_CellIndex.column ];
			}
	
			// jQuery selector on the TH elements for the columns
			var jqResult = $( nodes )
				.filter( s )
				.map( function () {
					return $.inArray( this, nodes ); // `nodes` is column index complete and in order
				} )
				.toArray();
	
			if ( jqResult.length || ! s.nodeName ) {
				return jqResult;
			}
	
			// Otherwise a node which might have a `dt-column` data attribute, or be
			// a child or such an element
			var host = $(s).closest('*[data-dt-column]');
			return host.length ?
				[ host.data('dt-column') ] :
				[];
		};
	
		return _selector_run( 'column', selector, run, settings, opts );
	};
	
	
	var __setColumnVis = function ( settings, column, vis ) {
		var
			cols = settings.aoColumns,
			col  = cols[ column ],
			data = settings.aoData,
			row, cells, i, ien, tr;
	
		// Get
		if ( vis === undefined ) {
			return col.bVisible;
		}
	
		// Set
		// No change
		if ( col.bVisible === vis ) {
			return;
		}
	
		if ( vis ) {
			// Insert column
			// Need to decide if we should use appendChild or insertBefore
			var insertBefore = $.inArray( true, _pluck(cols, 'bVisible'), column+1 );
	
			for ( i=0, ien=data.length ; i<ien ; i++ ) {
				tr = data[i].nTr;
				cells = data[i].anCells;
	
				if ( tr ) {
					// insertBefore can act like appendChild if 2nd arg is null
					tr.insertBefore( cells[ column ], cells[ insertBefore ] || null );
				}
			}
		}
		else {
			// Remove column
			$( _pluck( settings.aoData, 'anCells', column ) ).detach();
		}
	
		// Common actions
		col.bVisible = vis;
	};
	
	
	_api_register( 'columns()', function ( selector, opts ) {
		// argument shifting
		if ( selector === undefined ) {
			selector = '';
		}
		else if ( $.isPlainObject( selector ) ) {
			opts = selector;
			selector = '';
		}
	
		opts = _selector_opts( opts );
	
		var inst = this.iterator( 'table', function ( settings ) {
			return __column_selector( settings, selector, opts );
		}, 1 );
	
		// Want argument shifting here and in _row_selector?
		inst.selector.cols = selector;
		inst.selector.opts = opts;
	
		return inst;
	} );
	
	_api_registerPlural( 'columns().header()', 'column().header()', function ( selector, opts ) {
		return this.iterator( 'column', function ( settings, column ) {
			return settings.aoColumns[column].nTh;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().footer()', 'column().footer()', function ( selector, opts ) {
		return this.iterator( 'column', function ( settings, column ) {
			return settings.aoColumns[column].nTf;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().data()', 'column().data()', function () {
		return this.iterator( 'column-rows', __columnData, 1 );
	} );
	
	_api_registerPlural( 'columns().dataSrc()', 'column().dataSrc()', function () {
		return this.iterator( 'column', function ( settings, column ) {
			return settings.aoColumns[column].mData;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().cache()', 'column().cache()', function ( type ) {
		return this.iterator( 'column-rows', function ( settings, column, i, j, rows ) {
			return _pluck_order( settings.aoData, rows,
				type === 'search' ? '_aFilterData' : '_aSortData', column
			);
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().nodes()', 'column().nodes()', function () {
		return this.iterator( 'column-rows', function ( settings, column, i, j, rows ) {
			return _pluck_order( settings.aoData, rows, 'anCells', column ) ;
		}, 1 );
	} );
	
	_api_registerPlural( 'columns().visible()', 'column().visible()', function ( vis, calc ) {
		var that = this;
		var ret = this.iterator( 'column', function ( settings, column ) {
			if ( vis === undefined ) {
				return settings.aoColumns[ column ].bVisible;
			} // else
			__setColumnVis( settings, column, vis );
		} );
	
		// Group the column visibility changes
		if ( vis !== undefined ) {
			this.iterator( 'table', function ( settings ) {
				// Redraw the header after changes
				_fnDrawHead( settings, settings.aoHeader );
				_fnDrawHead( settings, settings.aoFooter );
		
				// Update colspan for no records display. Child rows and extensions will use their own
				// listeners to do this - only need to update the empty table item here
				if ( ! settings.aiDisplay.length ) {
					$(settings.nTBody).find('td[colspan]').attr('colspan', _fnVisbleColumns(settings));
				}
		
				_fnSaveState( settings );
	
				// Second loop once the first is done for events
				that.iterator( 'column', function ( settings, column ) {
					_fnCallbackFire( settings, null, 'column-visibility', [settings, column, vis, calc] );
				} );
	
				if ( calc === undefined || calc ) {
					that.columns.adjust();
				}
			});
		}
	
		return ret;
	} );
	
	_api_registerPlural( 'columns().indexes()', 'column().index()', function ( type ) {
		return this.iterator( 'column', function ( settings, column ) {
			return type === 'visible' ?
				_fnColumnIndexToVisible( settings, column ) :
				column;
		}, 1 );
	} );
	
	_api_register( 'columns.adjust()', function () {
		return this.iterator( 'table', function ( settings ) {
			_fnAdjustColumnSizing( settings );
		}, 1 );
	} );
	
	_api_register( 'column.index()', function ( type, idx ) {
		if ( this.context.length !== 0 ) {
			var ctx = this.context[0];
	
			if ( type === 'fromVisible' || type === 'toData' ) {
				return _fnVisibleToColumnIndex( ctx, idx );
			}
			else if ( type === 'fromData' || type === 'toVisible' ) {
				return _fnColumnIndexToVisible( ctx, idx );
			}
		}
	} );
	
	_api_register( 'column()', function ( selector, opts ) {
		return _selector_first( this.columns( selector, opts ) );
	} );
	
	var __cell_selector = function ( settings, selector, opts )
	{
		var data = settings.aoData;
		var rows = _selector_row_indexes( settings, opts );
		var cells = _removeEmpty( _pluck_order( data, rows, 'anCells' ) );
		var allCells = $(_flatten( [], cells ));
		var row;
		var columns = settings.aoColumns.length;
		var a, i, ien, j, o, host;
	
		var run = function ( s ) {
			var fnSelector = typeof s === 'function';
	
			if ( s === null || s === undefined || fnSelector ) {
				// All cells and function selectors
				a = [];
	
				for ( i=0, ien=rows.length ; i<ien ; i++ ) {
					row = rows[i];
	
					for ( j=0 ; j<columns ; j++ ) {
						o = {
							row: row,
							column: j
						};
	
						if ( fnSelector ) {
							// Selector - function
							host = data[ row ];
	
							if ( s( o, _fnGetCellData(settings, row, j), host.anCells ? host.anCells[j] : null ) ) {
								a.push( o );
							}
						}
						else {
							// Selector - all
							a.push( o );
						}
					}
				}
	
				return a;
			}
			
			// Selector - index
			if ( $.isPlainObject( s ) ) {
				// Valid cell index and its in the array of selectable rows
				return s.column !== undefined && s.row !== undefined && $.inArray( s.row, rows ) !== -1 ?
					[s] :
					[];
			}
	
			// Selector - jQuery filtered cells
			var jqResult = allCells
				.filter( s )
				.map( function (i, el) {
					return { // use a new object, in case someone changes the values
						row:    el._DT_CellIndex.row,
						column: el._DT_CellIndex.column
	 				};
				} )
				.toArray();
	
			if ( jqResult.length || ! s.nodeName ) {
				return jqResult;
			}
	
			// Otherwise the selector is a node, and there is one last option - the
			// element might be a child of an element which has dt-row and dt-column
			// data attributes
			host = $(s).closest('*[data-dt-row]');
			return host.length ?
				[ {
					row: host.data('dt-row'),
					column: host.data('dt-column')
				} ] :
				[];
		};
	
		return _selector_run( 'cell', selector, run, settings, opts );
	};
	
	
	
	
	_api_register( 'cells()', function ( rowSelector, columnSelector, opts ) {
		// Argument shifting
		if ( $.isPlainObject( rowSelector ) ) {
			// Indexes
			if ( rowSelector.row === undefined ) {
				// Selector options in first parameter
				opts = rowSelector;
				rowSelector = null;
			}
			else {
				// Cell index objects in first parameter
				opts = columnSelector;
				columnSelector = null;
			}
		}
		if ( $.isPlainObject( columnSelector ) ) {
			opts = columnSelector;
			columnSelector = null;
		}
	
		// Cell selector
		if ( columnSelector === null || columnSelector === undefined ) {
			return this.iterator( 'table', function ( settings ) {
				return __cell_selector( settings, rowSelector, _selector_opts( opts ) );
			} );
		}
	
		// The default built in options need to apply to row and columns
		var internalOpts = opts ? {
			page: opts.page,
			order: opts.order,
			search: opts.search
		} : {};
	
		// Row + column selector
		var columns = this.columns( columnSelector, internalOpts );
		var rows = this.rows( rowSelector, internalOpts );
		var i, ien, j, jen;
	
		var cellsNoOpts = this.iterator( 'table', function ( settings, idx ) {
			var a = [];
	
			for ( i=0, ien=rows[idx].length ; i<ien ; i++ ) {
				for ( j=0, jen=columns[idx].length ; j<jen ; j++ ) {
					a.push( {
						row:    rows[idx][i],
						column: columns[idx][j]
					} );
				}
			}
	
			return a;
		}, 1 );
	
		// There is currently only one extension which uses a cell selector extension
		// It is a _major_ performance drag to run this if it isn't needed, so this is
		// an extension specific check at the moment
		var cells = opts && opts.selected ?
			this.cells( cellsNoOpts, opts ) :
			cellsNoOpts;
	
		$.extend( cells.selector, {
			cols: columnSelector,
			rows: rowSelector,
			opts: opts
		} );
	
		return cells;
	} );
	
	
	_api_registerPlural( 'cells().nodes()', 'cell().node()', function () {
		return this.iterator( 'cell', function ( settings, row, column ) {
			var data = settings.aoData[ row ];
	
			return data && data.anCells ?
				data.anCells[ column ] :
				undefined;
		}, 1 );
	} );
	
	
	_api_register( 'cells().data()', function () {
		return this.iterator( 'cell', function ( settings, row, column ) {
			return _fnGetCellData( settings, row, column );
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().cache()', 'cell().cache()', function ( type ) {
		type = type === 'search' ? '_aFilterData' : '_aSortData';
	
		return this.iterator( 'cell', function ( settings, row, column ) {
			return settings.aoData[ row ][ type ][ column ];
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().render()', 'cell().render()', function ( type ) {
		return this.iterator( 'cell', function ( settings, row, column ) {
			return _fnGetCellData( settings, row, column, type );
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().indexes()', 'cell().index()', function () {
		return this.iterator( 'cell', function ( settings, row, column ) {
			return {
				row: row,
				column: column,
				columnVisible: _fnColumnIndexToVisible( settings, column )
			};
		}, 1 );
	} );
	
	
	_api_registerPlural( 'cells().invalidate()', 'cell().invalidate()', function ( src ) {
		return this.iterator( 'cell', function ( settings, row, column ) {
			_fnInvalidate( settings, row, src, column );
		} );
	} );
	
	
	
	_api_register( 'cell()', function ( rowSelector, columnSelector, opts ) {
		return _selector_first( this.cells( rowSelector, columnSelector, opts ) );
	} );
	
	
	_api_register( 'cell().data()', function ( data ) {
		var ctx = this.context;
		var cell = this[0];
	
		if ( data === undefined ) {
			// Get
			return ctx.length && cell.length ?
				_fnGetCellData( ctx[0], cell[0].row, cell[0].column ) :
				undefined;
		}
	
		// Set
		_fnSetCellData( ctx[0], cell[0].row, cell[0].column, data );
		_fnInvalidate( ctx[0], cell[0].row, 'data', cell[0].column );
	
		return this;
	} );
	
	
	
	/**
	 * Get current ordering (sorting) that has been applied to the table.
	 *
	 * @returns {array} 2D array containing the sorting information for the first
	 *   table in the current context. Each element in the parent array represents
	 *   a column being sorted upon (i.e. multi-sorting with two columns would have
	 *   2 inner arrays). The inner arrays may have 2 or 3 elements. The first is
	 *   the column index that the sorting condition applies to, the second is the
	 *   direction of the sort (`desc` or `asc`) and, optionally, the third is the
	 *   index of the sorting order from the `column.sorting` initialisation array.
	 *//**
	 * Set the ordering for the table.
	 *
	 * @param {integer} order Column index to sort upon.
	 * @param {string} direction Direction of the sort to be applied (`asc` or `desc`)
	 * @returns {DataTables.Api} this
	 *//**
	 * Set the ordering for the table.
	 *
	 * @param {array} order 1D array of sorting information to be applied.
	 * @param {array} [...] Optional additional sorting conditions
	 * @returns {DataTables.Api} this
	 *//**
	 * Set the ordering for the table.
	 *
	 * @param {array} order 2D array of sorting information to be applied.
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'order()', function ( order, dir ) {
		var ctx = this.context;
	
		if ( order === undefined ) {
			// get
			return ctx.length !== 0 ?
				ctx[0].aaSorting :
				undefined;
		}
	
		// set
		if ( typeof order === 'number' ) {
			// Simple column / direction passed in
			order = [ [ order, dir ] ];
		}
		else if ( order.length && ! Array.isArray( order[0] ) ) {
			// Arguments passed in (list of 1D arrays)
			order = Array.prototype.slice.call( arguments );
		}
		// otherwise a 2D array was passed in
	
		return this.iterator( 'table', function ( settings ) {
			settings.aaSorting = order.slice();
		} );
	} );
	
	
	/**
	 * Attach a sort listener to an element for a given column
	 *
	 * @param {node|jQuery|string} node Identifier for the element(s) to attach the
	 *   listener to. This can take the form of a single DOM node, a jQuery
	 *   collection of nodes or a jQuery selector which will identify the node(s).
	 * @param {integer} column the column that a click on this node will sort on
	 * @param {function} [callback] callback function when sort is run
	 * @returns {DataTables.Api} this
	 */
	_api_register( 'order.listener()', function ( node, column, callback ) {
		return this.iterator( 'table', function ( settings ) {
			_fnSortAttachListener( settings, node, column, callback );
		} );
	} );
	
	
	_api_register( 'order.fixed()', function ( set ) {
		if ( ! set ) {
			var ctx = this.context;
			var fixed = ctx.length ?
				ctx[0].aaSortingFixed :
				undefined;
	
			return Array.isArray( fixed ) ?
				{ pre: fixed } :
				fixed;
		}
	
		return this.iterator( 'table', function ( settings ) {
			settings.aaSortingFixed = $.extend( true, {}, set );
		} );
	} );
	
	
	// Order by the selected column(s)
	_api_register( [
		'columns().order()',
		'column().order()'
	], function ( dir ) {
		var that = this;
	
		return this.iterator( 'table', function ( settings, i ) {
			var sort = [];
	
			$.each( that[i], function (j, col) {
				sort.push( [ col, dir ] );
			} );
	
			settings.aaSorting = sort;
		} );
	} );
	
	
	
	_api_register( 'search()', function ( input, regex, smart, caseInsen ) {
		var ctx = this.context;
	
		if ( input === undefined ) {
			// get
			return ctx.length !== 0 ?
				ctx[0].oPreviousSearch.sSearch :
				undefined;
		}
	
		// set
		return this.iterator( 'table', function ( settings ) {
			if ( ! settings.oFeatures.bFilter ) {
				return;
			}
	
			_fnFilterComplete( settings, $.extend( {}, settings.oPreviousSearch, {
				"sSearch": input+"",
				"bRegex":  regex === null ? false : regex,
				"bSmart":  smart === null ? true  : smart,
				"bCaseInsensitive": caseInsen === null ? true : caseInsen
			} ), 1 );
		} );
	} );
	
	
	_api_registerPlural(
		'columns().search()',
		'column().search()',
		function ( input, regex, smart, caseInsen ) {
			return this.iterator( 'column', function ( settings, column ) {
				var preSearch = settings.aoPreSearchCols;
	
				if ( input === undefined ) {
					// get
					return preSearch[ column ].sSearch;
				}
	
				// set
				if ( ! settings.oFeatures.bFilter ) {
					return;
				}
	
				$.extend( preSearch[ column ], {
					"sSearch": input+"",
					"bRegex":  regex === null ? false : regex,
					"bSmart":  smart === null ? true  : smart,
					"bCaseInsensitive": caseInsen === null ? true : caseInsen
				} );
	
				_fnFilterComplete( settings, settings.oPreviousSearch, 1 );
			} );
		}
	);
	
	/*
	 * State API methods
	 */
	
	_api_register( 'state()', function () {
		return this.context.length ?
			this.context[0].oSavedState :
			null;
	} );
	
	
	_api_register( 'state.clear()', function () {
		return this.iterator( 'table', function ( settings ) {
			// Save an empty object
			settings.fnStateSaveCallback.call( settings.oInstance, settings, {} );
		} );
	} );
	
	
	_api_register( 'state.loaded()', function () {
		return this.context.length ?
			this.context[0].oLoadedState :
			null;
	} );
	
	
	_api_register( 'state.save()', function () {
		return this.iterator( 'table', function ( settings ) {
			_fnSaveState( settings );
		} );
	} );
	
	
	
	/**
	 * Provide a common method for plug-ins to check the version of DataTables being
	 * used, in order to ensure compatibility.
	 *
	 *  @param {string} version Version string to check for, in the format "X.Y.Z".
	 *    Note that the formats "X" and "X.Y" are also acceptable.
	 *  @returns {boolean} true if this version of DataTables is greater or equal to
	 *    the required version, or false if this version of DataTales is not
	 *    suitable
	 *  @static
	 *  @dtopt API-Static
	 *
	 *  @example
	 *    alert( $.fn.dataTable.versionCheck( '1.9.0' ) );
	 */
	DataTable.versionCheck = DataTable.fnVersionCheck = function( version )
	{
		var aThis = DataTable.version.split('.');
		var aThat = version.split('.');
		var iThis, iThat;
	
		for ( var i=0, iLen=aThat.length ; i<iLen ; i++ ) {
			iThis = parseInt( aThis[i], 10 ) || 0;
			iThat = parseInt( aThat[i], 10 ) || 0;
	
			// Parts are the same, keep comparing
			if (iThis === iThat) {
				continue;
			}
	
			// Parts are different, return immediately
			return iThis > iThat;
		}
	
		return true;
	};
	
	
	/**
	 * Check if a `<table>` node is a DataTable table already or not.
	 *
	 *  @param {node|jquery|string} table Table node, jQuery object or jQuery
	 *      selector for the table to test. Note that if more than more than one
	 *      table is passed on, only the first will be checked
	 *  @returns {boolean} true the table given is a DataTable, or false otherwise
	 *  @static
	 *  @dtopt API-Static
	 *
	 *  @example
	 *    if ( ! $.fn.DataTable.isDataTable( '#example' ) ) {
	 *      $('#example').dataTable();
	 *    }
	 */
	DataTable.isDataTable = DataTable.fnIsDataTable = function ( table )
	{
		var t = $(table).get(0);
		var is = false;
	
		if ( table instanceof DataTable.Api ) {
			return true;
		}
	
		$.each( DataTable.settings, function (i, o) {
			var head = o.nScrollHead ? $('table', o.nScrollHead)[0] : null;
			var foot = o.nScrollFoot ? $('table', o.nScrollFoot)[0] : null;
	
			if ( o.nTable === t || head === t || foot === t ) {
				is = true;
			}
		} );
	
		return is;
	};
	
	
	/**
	 * Get all DataTable tables that have been initialised - optionally you can
	 * select to get only currently visible tables.
	 *
	 *  @param {boolean} [visible=false] Flag to indicate if you want all (default)
	 *    or visible tables only.
	 *  @returns {array} Array of `table` nodes (not DataTable instances) which are
	 *    DataTables
	 *  @static
	 *  @dtopt API-Static
	 *
	 *  @example
	 *    $.each( $.fn.dataTable.tables(true), function () {
	 *      $(table).DataTable().columns.adjust();
	 *    } );
	 */
	DataTable.tables = DataTable.fnTables = function ( visible )
	{
		var api = false;
	
		if ( $.isPlainObject( visible ) ) {
			api = visible.api;
			visible = visible.visible;
		}
	
		var a = $.map( DataTable.settings, function (o) {
			if ( !visible || (visible && $(o.nTable).is(':visible')) ) {
				return o.nTable;
			}
		} );
	
		return api ?
			new _Api( a ) :
			a;
	};
	
	
	/**
	 * Convert from camel case parameters to Hungarian notation. This is made public
	 * for the extensions to provide the same ability as DataTables core to accept
	 * either the 1.9 style Hungarian notation, or the 1.10+ style camelCase
	 * parameters.
	 *
	 *  @param {object} src The model object which holds all parameters that can be
	 *    mapped.
	 *  @param {object} user The object to convert from camel case to Hungarian.
	 *  @param {boolean} force When set to `true`, properties which already have a
	 *    Hungarian value in the `user` object will be overwritten. Otherwise they
	 *    won't be.
	 */
	DataTable.camelToHungarian = _fnCamelToHungarian;
	
	
	
	/**
	 *
	 */
	_api_register( '$()', function ( selector, opts ) {
		var
			rows   = this.rows( opts ).nodes(), // Get all rows
			jqRows = $(rows);
	
		return $( [].concat(
			jqRows.filter( selector ).toArray(),
			jqRows.find( selector ).toArray()
		) );
	} );
	
	
	// jQuery functions to operate on the tables
	$.each( [ 'on', 'one', 'off' ], function (i, key) {
		_api_register( key+'()', function ( /* event, handler */ ) {
			var args = Array.prototype.slice.call(arguments);
	
			// Add the `dt` namespace automatically if it isn't already present
			args[0] = $.map( args[0].split( /\s/ ), function ( e ) {
				return ! e.match(/\.dt\b/) ?
					e+'.dt' :
					e;
				} ).join( ' ' );
	
			var inst = $( this.tables().nodes() );
			inst[key].apply( inst, args );
			return this;
		} );
	} );
	
	
	_api_register( 'clear()', function () {
		return this.iterator( 'table', function ( settings ) {
			_fnClearTable( settings );
		} );
	} );
	
	
	_api_register( 'settings()', function () {
		return new _Api( this.context, this.context );
	} );
	
	
	_api_register( 'init()', function () {
		var ctx = this.context;
		return ctx.length ? ctx[0].oInit : null;
	} );
	
	
	_api_register( 'data()', function () {
		return this.iterator( 'table', function ( settings ) {
			return _pluck( settings.aoData, '_aData' );
		} ).flatten();
	} );
	
	
	_api_register( 'destroy()', function ( remove ) {
		remove = remove || false;
	
		return this.iterator( 'table', function ( settings ) {
			var orig      = settings.nTableWrapper.parentNode;
			var classes   = settings.oClasses;
			var table     = settings.nTable;
			var tbody     = settings.nTBody;
			var thead     = settings.nTHead;
			var tfoot     = settings.nTFoot;
			var jqTable   = $(table);
			var jqTbody   = $(tbody);
			var jqWrapper = $(settings.nTableWrapper);
			var rows      = $.map( settings.aoData, function (r) { return r.nTr; } );
			var i, ien;
	
			// Flag to note that the table is currently being destroyed - no action
			// should be taken
			settings.bDestroying = true;
	
			// Fire off the destroy callbacks for plug-ins etc
			_fnCallbackFire( settings, "aoDestroyCallback", "destroy", [settings] );
	
			// If not being removed from the document, make all columns visible
			if ( ! remove ) {
				new _Api( settings ).columns().visible( true );
			}
	
			// Blitz all `DT` namespaced events (these are internal events, the
			// lowercase, `dt` events are user subscribed and they are responsible
			// for removing them
			jqWrapper.off('.DT').find(':not(tbody *)').off('.DT');
			$(window).off('.DT-'+settings.sInstance);
	
			// When scrolling we had to break the table up - restore it
			if ( table != thead.parentNode ) {
				jqTable.children('thead').detach();
				jqTable.append( thead );
			}
	
			if ( tfoot && table != tfoot.parentNode ) {
				jqTable.children('tfoot').detach();
				jqTable.append( tfoot );
			}
	
			settings.aaSorting = [];
			settings.aaSortingFixed = [];
			_fnSortingClasses( settings );
	
			$( rows ).removeClass( settings.asStripeClasses.join(' ') );
	
			$('th, td', thead).removeClass( classes.sSortable+' '+
				classes.sSortableAsc+' '+classes.sSortableDesc+' '+classes.sSortableNone
			);
	
			// Add the TR elements back into the table in their original order
			jqTbody.children().detach();
			jqTbody.append( rows );
	
			// Remove the DataTables generated nodes, events and classes
			var removedMethod = remove ? 'remove' : 'detach';
			jqTable[ removedMethod ]();
			jqWrapper[ removedMethod ]();
	
			// If we need to reattach the table to the document
			if ( ! remove && orig ) {
				// insertBefore acts like appendChild if !arg[1]
				orig.insertBefore( table, settings.nTableReinsertBefore );
	
				// Restore the width of the original table - was read from the style property,
				// so we can restore directly to that
				jqTable
					.css( 'width', settings.sDestroyWidth )
					.removeClass( classes.sTable );
	
				// If the were originally stripe classes - then we add them back here.
				// Note this is not fool proof (for example if not all rows had stripe
				// classes - but it's a good effort without getting carried away
				ien = settings.asDestroyStripes.length;
	
				if ( ien ) {
					jqTbody.children().each( function (i) {
						$(this).addClass( settings.asDestroyStripes[i % ien] );
					} );
				}
			}
	
			/* Remove the settings object from the settings array */
			var idx = $.inArray( settings, DataTable.settings );
			if ( idx !== -1 ) {
				DataTable.settings.splice( idx, 1 );
			}
		} );
	} );
	
	
	// Add the `every()` method for rows, columns and cells in a compact form
	$.each( [ 'column', 'row', 'cell' ], function ( i, type ) {
		_api_register( type+'s().every()', function ( fn ) {
			var opts = this.selector.opts;
			var api = this;
	
			return this.iterator( type, function ( settings, arg1, arg2, arg3, arg4 ) {
				// Rows and columns:
				//  arg1 - index
				//  arg2 - table counter
				//  arg3 - loop counter
				//  arg4 - undefined
				// Cells:
				//  arg1 - row index
				//  arg2 - column index
				//  arg3 - table counter
				//  arg4 - loop counter
				fn.call(
					api[ type ](
						arg1,
						type==='cell' ? arg2 : opts,
						type==='cell' ? opts : undefined
					),
					arg1, arg2, arg3, arg4
				);
			} );
		} );
	} );
	
	
	// i18n method for extensions to be able to use the language object from the
	// DataTable
	_api_register( 'i18n()', function ( token, def, plural ) {
		var ctx = this.context[0];
		var resolved = _fnGetObjectDataFn( token )( ctx.oLanguage );
	
		if ( resolved === undefined ) {
			resolved = def;
		}
	
		if ( plural !== undefined && $.isPlainObject( resolved ) ) {
			resolved = resolved[ plural ] !== undefined ?
				resolved[ plural ] :
				resolved._;
		}
	
		return resolved.replace( '%d', plural ); // nb: plural might be undefined,
	} );
	/**
	 * Version string for plug-ins to check compatibility. Allowed format is
	 * `a.b.c-d` where: a:int, b:int, c:int, d:string(dev|beta|alpha). `d` is used
	 * only for non-release builds. See http://semver.org/ for more information.
	 *  @member
	 *  @type string
	 *  @default Version number
	 */
	DataTable.version = "1.10.24";

	/**
	 * Private data store, containing all of the settings objects that are
	 * created for the tables on a given page.
	 *
	 * Note that the `DataTable.settings` object is aliased to
	 * `jQuery.fn.dataTableExt` through which it may be accessed and
	 * manipulated, or `jQuery.fn.dataTable.settings`.
	 *  @member
	 *  @type array
	 *  @default []
	 *  @private
	 */
	DataTable.settings = [];

	/**
	 * Object models container, for the various models that DataTables has
	 * available to it. These models define the objects that are used to hold
	 * the active state and configuration of the table.
	 *  @namespace
	 */
	DataTable.models = {};
	
	
	
	/**
	 * Template object for the way in which DataTables holds information about
	 * search information for the global filter and individual column filters.
	 *  @namespace
	 */
	DataTable.models.oSearch = {
		/**
		 * Flag to indicate if the filtering should be case insensitive or not
		 *  @type boolean
		 *  @default true
		 */
		"bCaseInsensitive": true,
	
		/**
		 * Applied search term
		 *  @type string
		 *  @default <i>Empty string</i>
		 */
		"sSearch": "",
	
		/**
		 * Flag to indicate if the search term should be interpreted as a
		 * regular expression (true) or not (false) and therefore and special
		 * regex characters escaped.
		 *  @type boolean
		 *  @default false
		 */
		"bRegex": false,
	
		/**
		 * Flag to indicate if DataTables is to use its smart filtering or not.
		 *  @type boolean
		 *  @default true
		 */
		"bSmart": true
	};
	
	
	
	
	/**
	 * Template object for the way in which DataTables holds information about
	 * each individual row. This is the object format used for the settings
	 * aoData array.
	 *  @namespace
	 */
	DataTable.models.oRow = {
		/**
		 * TR element for the row
		 *  @type node
		 *  @default null
		 */
		"nTr": null,
	
		/**
		 * Array of TD elements for each row. This is null until the row has been
		 * created.
		 *  @type array nodes
		 *  @default []
		 */
		"anCells": null,
	
		/**
		 * Data object from the original data source for the row. This is either
		 * an array if using the traditional form of DataTables, or an object if
		 * using mData options. The exact type will depend on the passed in
		 * data from the data source, or will be an array if using DOM a data
		 * source.
		 *  @type array|object
		 *  @default []
		 */
		"_aData": [],
	
		/**
		 * Sorting data cache - this array is ostensibly the same length as the
		 * number of columns (although each index is generated only as it is
		 * needed), and holds the data that is used for sorting each column in the
		 * row. We do this cache generation at the start of the sort in order that
		 * the formatting of the sort data need be done only once for each cell
		 * per sort. This array should not be read from or written to by anything
		 * other than the master sorting methods.
		 *  @type array
		 *  @default null
		 *  @private
		 */
		"_aSortData": null,
	
		/**
		 * Per cell filtering data cache. As per the sort data cache, used to
		 * increase the performance of the filtering in DataTables
		 *  @type array
		 *  @default null
		 *  @private
		 */
		"_aFilterData": null,
	
		/**
		 * Filtering data cache. This is the same as the cell filtering cache, but
		 * in this case a string rather than an array. This is easily computed with
		 * a join on `_aFilterData`, but is provided as a cache so the join isn't
		 * needed on every search (memory traded for performance)
		 *  @type array
		 *  @default null
		 *  @private
		 */
		"_sFilterRow": null,
	
		/**
		 * Cache of the class name that DataTables has applied to the row, so we
		 * can quickly look at this variable rather than needing to do a DOM check
		 * on className for the nTr property.
		 *  @type string
		 *  @default <i>Empty string</i>
		 *  @private
		 */
		"_sRowStripe": "",
	
		/**
		 * Denote if the original data source was from the DOM, or the data source
		 * object. This is used for invalidating data, so DataTables can
		 * automatically read data from the original source, unless uninstructed
		 * otherwise.
		 *  @type string
		 *  @default null
		 *  @private
		 */
		"src": null,
	
		/**
		 * Index in the aoData array. This saves an indexOf lookup when we have the
		 * object, but want to know the index
		 *  @type integer
		 *  @default -1
		 *  @private
		 */
		"idx": -1
	};
	
	
	/**
	 * Template object for the column information object in DataTables. This object
	 * is held in the settings aoColumns array and contains all the information that
	 * DataTables needs about each individual column.
	 *
	 * Note that this object is related to {@link DataTable.defaults.column}
	 * but this one is the internal data store for DataTables's cache of columns.
	 * It should NOT be manipulated outside of DataTables. Any configuration should
	 * be done through the initialisation options.
	 *  @namespace
	 */
	DataTable.models.oColumn = {
		/**
		 * Column index. This could be worked out on-the-fly with $.inArray, but it
		 * is faster to just hold it as a variable
		 *  @type integer
		 *  @default null
		 */
		"idx": null,
	
		/**
		 * A list of the columns that sorting should occur on when this column
		 * is sorted. That this property is an array allows multi-column sorting
		 * to be defined for a column (for example first name / last name columns
		 * would benefit from this). The values are integers pointing to the
		 * columns to be sorted on (typically it will be a single integer pointing
		 * at itself, but that doesn't need to be the case).
		 *  @type array
		 */
		"aDataSort": null,
	
		/**
		 * Define the sorting directions that are applied to the column, in sequence
		 * as the column is repeatedly sorted upon - i.e. the first value is used
		 * as the sorting direction when the column if first sorted (clicked on).
		 * Sort it again (click again) and it will move on to the next index.
		 * Repeat until loop.
		 *  @type array
		 */
		"asSorting": null,
	
		/**
		 * Flag to indicate if the column is searchable, and thus should be included
		 * in the filtering or not.
		 *  @type boolean
		 */
		"bSearchable": null,
	
		/**
		 * Flag to indicate if the column is sortable or not.
		 *  @type boolean
		 */
		"bSortable": null,
	
		/**
		 * Flag to indicate if the column is currently visible in the table or not
		 *  @type boolean
		 */
		"bVisible": null,
	
		/**
		 * Store for manual type assignment using the `column.type` option. This
		 * is held in store so we can manipulate the column's `sType` property.
		 *  @type string
		 *  @default null
		 *  @private
		 */
		"_sManualType": null,
	
		/**
		 * Flag to indicate if HTML5 data attributes should be used as the data
		 * source for filtering or sorting. True is either are.
		 *  @type boolean
		 *  @default false
		 *  @private
		 */
		"_bAttrSrc": false,
	
		/**
		 * Developer definable function that is called whenever a cell is created (Ajax source,
		 * etc) or processed for input (DOM source). This can be used as a compliment to mRender
		 * allowing you to modify the DOM element (add background colour for example) when the
		 * element is available.
		 *  @type function
		 *  @param {element} nTd The TD node that has been created
		 *  @param {*} sData The Data for the cell
		 *  @param {array|object} oData The data for the whole row
		 *  @param {int} iRow The row index for the aoData data store
		 *  @default null
		 */
		"fnCreatedCell": null,
	
		/**
		 * Function to get data from a cell in a column. You should <b>never</b>
		 * access data directly through _aData internally in DataTables - always use
		 * the method attached to this property. It allows mData to function as
		 * required. This function is automatically assigned by the column
		 * initialisation method
		 *  @type function
		 *  @param {array|object} oData The data array/object for the array
		 *    (i.e. aoData[]._aData)
		 *  @param {string} sSpecific The specific data type you want to get -
		 *    'display', 'type' 'filter' 'sort'
		 *  @returns {*} The data for the cell from the given row's data
		 *  @default null
		 */
		"fnGetData": null,
	
		/**
		 * Function to set data for a cell in the column. You should <b>never</b>
		 * set the data directly to _aData internally in DataTables - always use
		 * this method. It allows mData to function as required. This function
		 * is automatically assigned by the column initialisation method
		 *  @type function
		 *  @param {array|object} oData The data array/object for the array
		 *    (i.e. aoData[]._aData)
		 *  @param {*} sValue Value to set
		 *  @default null
		 */
		"fnSetData": null,
	
		/**
		 * Property to read the value for the cells in the column from the data
		 * source array / object. If null, then the default content is used, if a
		 * function is given then the return from the function is used.
		 *  @type function|int|string|null
		 *  @default null
		 */
		"mData": null,
	
		/**
		 * Partner property to mData which is used (only when defined) to get
		 * the data - i.e. it is basically the same as mData, but without the
		 * 'set' option, and also the data fed to it is the result from mData.
		 * This is the rendering method to match the data method of mData.
		 *  @type function|int|string|null
		 *  @default null
		 */
		"mRender": null,
	
		/**
		 * Unique header TH/TD element for this column - this is what the sorting
		 * listener is attached to (if sorting is enabled.)
		 *  @type node
		 *  @default null
		 */
		"nTh": null,
	
		/**
		 * Unique footer TH/TD element for this column (if there is one). Not used
		 * in DataTables as such, but can be used for plug-ins to reference the
		 * footer for each column.
		 *  @type node
		 *  @default null
		 */
		"nTf": null,
	
		/**
		 * The class to apply to all TD elements in the table's TBODY for the column
		 *  @type string
		 *  @default null
		 */
		"sClass": null,
	
		/**
		 * When DataTables calculates the column widths to assign to each column,
		 * it finds the longest string in each column and then constructs a
		 * temporary table and reads the widths from that. The problem with this
		 * is that "mmm" is much wider then "iiii", but the latter is a longer
		 * string - thus the calculation can go wrong (doing it properly and putting
		 * it into an DOM object and measuring that is horribly(!) slow). Thus as
		 * a "work around" we provide this option. It will append its value to the
		 * text that is found to be the longest string for the column - i.e. padding.
		 *  @type string
		 */
		"sContentPadding": null,
	
		/**
		 * Allows a default value to be given for a column's data, and will be used
		 * whenever a null data source is encountered (this can be because mData
		 * is set to null, or because the data source itself is null).
		 *  @type string
		 *  @default null
		 */
		"sDefaultContent": null,
	
		/**
		 * Name for the column, allowing reference to the column by name as well as
		 * by index (needs a lookup to work by name).
		 *  @type string
		 */
		"sName": null,
	
		/**
		 * Custom sorting data type - defines which of the available plug-ins in
		 * afnSortData the custom sorting will use - if any is defined.
		 *  @type string
		 *  @default std
		 */
		"sSortDataType": 'std',
	
		/**
		 * Class to be applied to the header element when sorting on this column
		 *  @type string
		 *  @default null
		 */
		"sSortingClass": null,
	
		/**
		 * Class to be applied to the header element when sorting on this column -
		 * when jQuery UI theming is used.
		 *  @type string
		 *  @default null
		 */
		"sSortingClassJUI": null,
	
		/**
		 * Title of the column - what is seen in the TH element (nTh).
		 *  @type string
		 */
		"sTitle": null,
	
		/**
		 * Column sorting and filtering type
		 *  @type string
		 *  @default null
		 */
		"sType": null,
	
		/**
		 * Width of the column
		 *  @type string
		 *  @default null
		 */
		"sWidth": null,
	
		/**
		 * Width of the column when it was first "encountered"
		 *  @type string
		 *  @default null
		 */
		"sWidthOrig": null
	};
	
	
	/*
	 * Developer note: The properties of the object below are given in Hungarian
	 * notation, that was used as the interface for DataTables prior to v1.10, however
	 * from v1.10 onwards the primary interface is camel case. In order to avoid
	 * breaking backwards compatibility utterly with this change, the Hungarian
	 * version is still, internally the primary interface, but is is not documented
	 * - hence the @name tags in each doc comment. This allows a Javascript function
	 * to create a map from Hungarian notation to camel case (going the other direction
	 * would require each property to be listed, which would add around 3K to the size
	 * of DataTables, while this method is about a 0.5K hit).
	 *
	 * Ultimately this does pave the way for Hungarian notation to be dropped
	 * completely, but that is a massive amount of work and will break current
	 * installs (therefore is on-hold until v2).
	 */
	
	/**
	 * Initialisation options that can be given to DataTables at initialisation
	 * time.
	 *  @namespace
	 */
	DataTable.defaults = {
		/**
		 * An array of data to use for the table, passed in at initialisation which
		 * will be used in preference to any data which is already in the DOM. This is
		 * particularly useful for constructing tables purely in Javascript, for
		 * example with a custom Ajax call.
		 *  @type array
		 *  @default null
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.data
		 *
		 *  @example
		 *    // Using a 2D array data source
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "data": [
		 *          ['Trident', 'Internet Explorer 4.0', 'Win 95+', 4, 'X'],
		 *          ['Trident', 'Internet Explorer 5.0', 'Win 95+', 5, 'C'],
		 *        ],
		 *        "columns": [
		 *          { "title": "Engine" },
		 *          { "title": "Browser" },
		 *          { "title": "Platform" },
		 *          { "title": "Version" },
		 *          { "title": "Grade" }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using an array of objects as a data source (`data`)
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "data": [
		 *          {
		 *            "engine":   "Trident",
		 *            "browser":  "Internet Explorer 4.0",
		 *            "platform": "Win 95+",
		 *            "version":  4,
		 *            "grade":    "X"
		 *          },
		 *          {
		 *            "engine":   "Trident",
		 *            "browser":  "Internet Explorer 5.0",
		 *            "platform": "Win 95+",
		 *            "version":  5,
		 *            "grade":    "C"
		 *          }
		 *        ],
		 *        "columns": [
		 *          { "title": "Engine",   "data": "engine" },
		 *          { "title": "Browser",  "data": "browser" },
		 *          { "title": "Platform", "data": "platform" },
		 *          { "title": "Version",  "data": "version" },
		 *          { "title": "Grade",    "data": "grade" }
		 *        ]
		 *      } );
		 *    } );
		 */
		"aaData": null,
	
	
		/**
		 * If ordering is enabled, then DataTables will perform a first pass sort on
		 * initialisation. You can define which column(s) the sort is performed
		 * upon, and the sorting direction, with this variable. The `sorting` array
		 * should contain an array for each column to be sorted initially containing
		 * the column's index and a direction string ('asc' or 'desc').
		 *  @type array
		 *  @default [[0,'asc']]
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.order
		 *
		 *  @example
		 *    // Sort by 3rd column first, and then 4th column
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "order": [[2,'asc'], [3,'desc']]
		 *      } );
		 *    } );
		 *
		 *    // No initial sorting
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "order": []
		 *      } );
		 *    } );
		 */
		"aaSorting": [[0,'asc']],
	
	
		/**
		 * This parameter is basically identical to the `sorting` parameter, but
		 * cannot be overridden by user interaction with the table. What this means
		 * is that you could have a column (visible or hidden) which the sorting
		 * will always be forced on first - any sorting after that (from the user)
		 * will then be performed as required. This can be useful for grouping rows
		 * together.
		 *  @type array
		 *  @default null
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.orderFixed
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "orderFixed": [[0,'asc']]
		 *      } );
		 *    } )
		 */
		"aaSortingFixed": [],
	
	
		/**
		 * DataTables can be instructed to load data to display in the table from a
		 * Ajax source. This option defines how that Ajax call is made and where to.
		 *
		 * The `ajax` property has three different modes of operation, depending on
		 * how it is defined. These are:
		 *
		 * * `string` - Set the URL from where the data should be loaded from.
		 * * `object` - Define properties for `jQuery.ajax`.
		 * * `function` - Custom data get function
		 *
		 * `string`
		 * --------
		 *
		 * As a string, the `ajax` property simply defines the URL from which
		 * DataTables will load data.
		 *
		 * `object`
		 * --------
		 *
		 * As an object, the parameters in the object are passed to
		 * [jQuery.ajax](http://api.jquery.com/jQuery.ajax/) allowing fine control
		 * of the Ajax request. DataTables has a number of default parameters which
		 * you can override using this option. Please refer to the jQuery
		 * documentation for a full description of the options available, although
		 * the following parameters provide additional options in DataTables or
		 * require special consideration:
		 *
		 * * `data` - As with jQuery, `data` can be provided as an object, but it
		 *   can also be used as a function to manipulate the data DataTables sends
		 *   to the server. The function takes a single parameter, an object of
		 *   parameters with the values that DataTables has readied for sending. An
		 *   object may be returned which will be merged into the DataTables
		 *   defaults, or you can add the items to the object that was passed in and
		 *   not return anything from the function. This supersedes `fnServerParams`
		 *   from DataTables 1.9-.
		 *
		 * * `dataSrc` - By default DataTables will look for the property `data` (or
		 *   `aaData` for compatibility with DataTables 1.9-) when obtaining data
		 *   from an Ajax source or for server-side processing - this parameter
		 *   allows that property to be changed. You can use Javascript dotted
		 *   object notation to get a data source for multiple levels of nesting, or
		 *   it my be used as a function. As a function it takes a single parameter,
		 *   the JSON returned from the server, which can be manipulated as
		 *   required, with the returned value being that used by DataTables as the
		 *   data source for the table. This supersedes `sAjaxDataProp` from
		 *   DataTables 1.9-.
		 *
		 * * `success` - Should not be overridden it is used internally in
		 *   DataTables. To manipulate / transform the data returned by the server
		 *   use `ajax.dataSrc`, or use `ajax` as a function (see below).
		 *
		 * `function`
		 * ----------
		 *
		 * As a function, making the Ajax call is left up to yourself allowing
		 * complete control of the Ajax request. Indeed, if desired, a method other
		 * than Ajax could be used to obtain the required data, such as Web storage
		 * or an AIR database.
		 *
		 * The function is given four parameters and no return is required. The
		 * parameters are:
		 *
		 * 1. _object_ - Data to send to the server
		 * 2. _function_ - Callback function that must be executed when the required
		 *    data has been obtained. That data should be passed into the callback
		 *    as the only parameter
		 * 3. _object_ - DataTables settings object for the table
		 *
		 * Note that this supersedes `fnServerData` from DataTables 1.9-.
		 *
		 *  @type string|object|function
		 *  @default null
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.ajax
		 *  @since 1.10.0
		 *
		 * @example
		 *   // Get JSON data from a file via Ajax.
		 *   // Note DataTables expects data in the form `{ data: [ ...data... ] }` by default).
		 *   $('#example').dataTable( {
		 *     "ajax": "data.json"
		 *   } );
		 *
		 * @example
		 *   // Get JSON data from a file via Ajax, using `dataSrc` to change
		 *   // `data` to `tableData` (i.e. `{ tableData: [ ...data... ] }`)
		 *   $('#example').dataTable( {
		 *     "ajax": {
		 *       "url": "data.json",
		 *       "dataSrc": "tableData"
		 *     }
		 *   } );
		 *
		 * @example
		 *   // Get JSON data from a file via Ajax, using `dataSrc` to read data
		 *   // from a plain array rather than an array in an object
		 *   $('#example').dataTable( {
		 *     "ajax": {
		 *       "url": "data.json",
		 *       "dataSrc": ""
		 *     }
		 *   } );
		 *
		 * @example
		 *   // Manipulate the data returned from the server - add a link to data
		 *   // (note this can, should, be done using `render` for the column - this
		 *   // is just a simple example of how the data can be manipulated).
		 *   $('#example').dataTable( {
		 *     "ajax": {
		 *       "url": "data.json",
		 *       "dataSrc": function ( json ) {
		 *         for ( var i=0, ien=json.length ; i<ien ; i++ ) {
		 *           json[i][0] = '<a href="/message/'+json[i][0]+'>View message</a>';
		 *         }
		 *         return json;
		 *       }
		 *     }
		 *   } );
		 *
		 * @example
		 *   // Add data to the request
		 *   $('#example').dataTable( {
		 *     "ajax": {
		 *       "url": "data.json",
		 *       "data": function ( d ) {
		 *         return {
		 *           "extra_search": $('#extra').val()
		 *         };
		 *       }
		 *     }
		 *   } );
		 *
		 * @example
		 *   // Send request as POST
		 *   $('#example').dataTable( {
		 *     "ajax": {
		 *       "url": "data.json",
		 *       "type": "POST"
		 *     }
		 *   } );
		 *
		 * @example
		 *   // Get the data from localStorage (could interface with a form for
		 *   // adding, editing and removing rows).
		 *   $('#example').dataTable( {
		 *     "ajax": function (data, callback, settings) {
		 *       callback(
		 *         JSON.parse( localStorage.getItem('dataTablesData') )
		 *       );
		 *     }
		 *   } );
		 */
		"ajax": null,
	
	
		/**
		 * This parameter allows you to readily specify the entries in the length drop
		 * down menu that DataTables shows when pagination is enabled. It can be
		 * either a 1D array of options which will be used for both the displayed
		 * option and the value, or a 2D array which will use the array in the first
		 * position as the value, and the array in the second position as the
		 * displayed options (useful for language strings such as 'All').
		 *
		 * Note that the `pageLength` property will be automatically set to the
		 * first value given in this array, unless `pageLength` is also provided.
		 *  @type array
		 *  @default [ 10, 25, 50, 100 ]
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.lengthMenu
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "lengthMenu": [[10, 25, 50, -1], [10, 25, 50, "All"]]
		 *      } );
		 *    } );
		 */
		"aLengthMenu": [ 10, 25, 50, 100 ],
	
	
		/**
		 * The `columns` option in the initialisation parameter allows you to define
		 * details about the way individual columns behave. For a full list of
		 * column options that can be set, please see
		 * {@link DataTable.defaults.column}. Note that if you use `columns` to
		 * define your columns, you must have an entry in the array for every single
		 * column that you have in your table (these can be null if you don't which
		 * to specify any options).
		 *  @member
		 *
		 *  @name DataTable.defaults.column
		 */
		"aoColumns": null,
	
		/**
		 * Very similar to `columns`, `columnDefs` allows you to target a specific
		 * column, multiple columns, or all columns, using the `targets` property of
		 * each object in the array. This allows great flexibility when creating
		 * tables, as the `columnDefs` arrays can be of any length, targeting the
		 * columns you specifically want. `columnDefs` may use any of the column
		 * options available: {@link DataTable.defaults.column}, but it _must_
		 * have `targets` defined in each object in the array. Values in the `targets`
		 * array may be:
		 *   <ul>
		 *     <li>a string - class name will be matched on the TH for the column</li>
		 *     <li>0 or a positive integer - column index counting from the left</li>
		 *     <li>a negative integer - column index counting from the right</li>
		 *     <li>the string "_all" - all columns (i.e. assign a default)</li>
		 *   </ul>
		 *  @member
		 *
		 *  @name DataTable.defaults.columnDefs
		 */
		"aoColumnDefs": null,
	
	
		/**
		 * Basically the same as `search`, this parameter defines the individual column
		 * filtering state at initialisation time. The array must be of the same size
		 * as the number of columns, and each element be an object with the parameters
		 * `search` and `escapeRegex` (the latter is optional). 'null' is also
		 * accepted and the default will be used.
		 *  @type array
		 *  @default []
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.searchCols
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "searchCols": [
		 *          null,
		 *          { "search": "My filter" },
		 *          null,
		 *          { "search": "^[0-9]", "escapeRegex": false }
		 *        ]
		 *      } );
		 *    } )
		 */
		"aoSearchCols": [],
	
	
		/**
		 * An array of CSS classes that should be applied to displayed rows. This
		 * array may be of any length, and DataTables will apply each class
		 * sequentially, looping when required.
		 *  @type array
		 *  @default null <i>Will take the values determined by the `oClasses.stripe*`
		 *    options</i>
		 *
		 *  @dtopt Option
		 *  @name DataTable.defaults.stripeClasses
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stripeClasses": [ 'strip1', 'strip2', 'strip3' ]
		 *      } );
		 *    } )
		 */
		"asStripeClasses": null,
	
	
		/**
		 * Enable or disable automatic column width calculation. This can be disabled
		 * as an optimisation (it takes some time to calculate the widths) if the
		 * tables widths are passed in using `columns`.
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.autoWidth
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "autoWidth": false
		 *      } );
		 *    } );
		 */
		"bAutoWidth": true,
	
	
		/**
		 * Deferred rendering can provide DataTables with a huge speed boost when you
		 * are using an Ajax or JS data source for the table. This option, when set to
		 * true, will cause DataTables to defer the creation of the table elements for
		 * each row until they are needed for a draw - saving a significant amount of
		 * time.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.deferRender
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "ajax": "sources/arrays.txt",
		 *        "deferRender": true
		 *      } );
		 *    } );
		 */
		"bDeferRender": false,
	
	
		/**
		 * Replace a DataTable which matches the given selector and replace it with
		 * one which has the properties of the new initialisation object passed. If no
		 * table matches the selector, then the new DataTable will be constructed as
		 * per normal.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.destroy
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "srollY": "200px",
		 *        "paginate": false
		 *      } );
		 *
		 *      // Some time later....
		 *      $('#example').dataTable( {
		 *        "filter": false,
		 *        "destroy": true
		 *      } );
		 *    } );
		 */
		"bDestroy": false,
	
	
		/**
		 * Enable or disable filtering of data. Filtering in DataTables is "smart" in
		 * that it allows the end user to input multiple words (space separated) and
		 * will match a row containing those words, even if not in the order that was
		 * specified (this allow matching across multiple columns). Note that if you
		 * wish to use filtering in DataTables this must remain 'true' - to remove the
		 * default filtering input box and retain filtering abilities, please use
		 * {@link DataTable.defaults.dom}.
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.searching
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "searching": false
		 *      } );
		 *    } );
		 */
		"bFilter": true,
	
	
		/**
		 * Enable or disable the table information display. This shows information
		 * about the data that is currently visible on the page, including information
		 * about filtered data if that action is being performed.
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.info
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "info": false
		 *      } );
		 *    } );
		 */
		"bInfo": true,
	
	
		/**
		 * Allows the end user to select the size of a formatted page from a select
		 * menu (sizes are 10, 25, 50 and 100). Requires pagination (`paginate`).
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.lengthChange
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "lengthChange": false
		 *      } );
		 *    } );
		 */
		"bLengthChange": true,
	
	
		/**
		 * Enable or disable pagination.
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.paging
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "paging": false
		 *      } );
		 *    } );
		 */
		"bPaginate": true,
	
	
		/**
		 * Enable or disable the display of a 'processing' indicator when the table is
		 * being processed (e.g. a sort). This is particularly useful for tables with
		 * large amounts of data where it can take a noticeable amount of time to sort
		 * the entries.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.processing
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "processing": true
		 *      } );
		 *    } );
		 */
		"bProcessing": false,
	
	
		/**
		 * Retrieve the DataTables object for the given selector. Note that if the
		 * table has already been initialised, this parameter will cause DataTables
		 * to simply return the object that has already been set up - it will not take
		 * account of any changes you might have made to the initialisation object
		 * passed to DataTables (setting this parameter to true is an acknowledgement
		 * that you understand this). `destroy` can be used to reinitialise a table if
		 * you need.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.retrieve
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      initTable();
		 *      tableActions();
		 *    } );
		 *
		 *    function initTable ()
		 *    {
		 *      return $('#example').dataTable( {
		 *        "scrollY": "200px",
		 *        "paginate": false,
		 *        "retrieve": true
		 *      } );
		 *    }
		 *
		 *    function tableActions ()
		 *    {
		 *      var table = initTable();
		 *      // perform API operations with oTable
		 *    }
		 */
		"bRetrieve": false,
	
	
		/**
		 * When vertical (y) scrolling is enabled, DataTables will force the height of
		 * the table's viewport to the given height at all times (useful for layout).
		 * However, this can look odd when filtering data down to a small data set,
		 * and the footer is left "floating" further down. This parameter (when
		 * enabled) will cause DataTables to collapse the table's viewport down when
		 * the result set will fit within the given Y height.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.scrollCollapse
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "scrollY": "200",
		 *        "scrollCollapse": true
		 *      } );
		 *    } );
		 */
		"bScrollCollapse": false,
	
	
		/**
		 * Configure DataTables to use server-side processing. Note that the
		 * `ajax` parameter must also be given in order to give DataTables a
		 * source to obtain the required data for each draw.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Features
		 *  @dtopt Server-side
		 *  @name DataTable.defaults.serverSide
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "serverSide": true,
		 *        "ajax": "xhr.php"
		 *      } );
		 *    } );
		 */
		"bServerSide": false,
	
	
		/**
		 * Enable or disable sorting of columns. Sorting of individual columns can be
		 * disabled by the `sortable` option for each column.
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.ordering
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "ordering": false
		 *      } );
		 *    } );
		 */
		"bSort": true,
	
	
		/**
		 * Enable or display DataTables' ability to sort multiple columns at the
		 * same time (activated by shift-click by the user).
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.orderMulti
		 *
		 *  @example
		 *    // Disable multiple column sorting ability
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "orderMulti": false
		 *      } );
		 *    } );
		 */
		"bSortMulti": true,
	
	
		/**
		 * Allows control over whether DataTables should use the top (true) unique
		 * cell that is found for a single column, or the bottom (false - default).
		 * This is useful when using complex headers.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.orderCellsTop
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "orderCellsTop": true
		 *      } );
		 *    } );
		 */
		"bSortCellsTop": false,
	
	
		/**
		 * Enable or disable the addition of the classes `sorting\_1`, `sorting\_2` and
		 * `sorting\_3` to the columns which are currently being sorted on. This is
		 * presented as a feature switch as it can increase processing time (while
		 * classes are removed and added) so for large data sets you might want to
		 * turn this off.
		 *  @type boolean
		 *  @default true
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.orderClasses
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "orderClasses": false
		 *      } );
		 *    } );
		 */
		"bSortClasses": true,
	
	
		/**
		 * Enable or disable state saving. When enabled HTML5 `localStorage` will be
		 * used to save table display information such as pagination information,
		 * display length, filtering and sorting. As such when the end user reloads
		 * the page the display display will match what thy had previously set up.
		 *
		 * Due to the use of `localStorage` the default state saving is not supported
		 * in IE6 or 7. If state saving is required in those browsers, use
		 * `stateSaveCallback` to provide a storage solution such as cookies.
		 *  @type boolean
		 *  @default false
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.stateSave
		 *
		 *  @example
		 *    $(document).ready( function () {
		 *      $('#example').dataTable( {
		 *        "stateSave": true
		 *      } );
		 *    } );
		 */
		"bStateSave": false,
	
	
		/**
		 * This function is called when a TR element is created (and all TD child
		 * elements have been inserted), or registered if using a DOM source, allowing
		 * manipulation of the TR element (adding classes etc).
		 *  @type function
		 *  @param {node} row "TR" element for the current row
		 *  @param {array} data Raw data array for this row
		 *  @param {int} dataIndex The index of this row in the internal aoData array
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.createdRow
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "createdRow": function( row, data, dataIndex ) {
		 *          // Bold the grade for all 'A' grade browsers
		 *          if ( data[4] == "A" )
		 *          {
		 *            $('td:eq(4)', row).html( '<b>A</b>' );
		 *          }
		 *        }
		 *      } );
		 *    } );
		 */
		"fnCreatedRow": null,
	
	
		/**
		 * This function is called on every 'draw' event, and allows you to
		 * dynamically modify any aspect you want about the created DOM.
		 *  @type function
		 *  @param {object} settings DataTables settings object
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.drawCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "drawCallback": function( settings ) {
		 *          alert( 'DataTables has redrawn the table' );
		 *        }
		 *      } );
		 *    } );
		 */
		"fnDrawCallback": null,
	
	
		/**
		 * Identical to fnHeaderCallback() but for the table footer this function
		 * allows you to modify the table footer on every 'draw' event.
		 *  @type function
		 *  @param {node} foot "TR" element for the footer
		 *  @param {array} data Full table data (as derived from the original HTML)
		 *  @param {int} start Index for the current display starting point in the
		 *    display array
		 *  @param {int} end Index for the current display ending point in the
		 *    display array
		 *  @param {array int} display Index array to translate the visual position
		 *    to the full data array
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.footerCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "footerCallback": function( tfoot, data, start, end, display ) {
		 *          tfoot.getElementsByTagName('th')[0].innerHTML = "Starting index is "+start;
		 *        }
		 *      } );
		 *    } )
		 */
		"fnFooterCallback": null,
	
	
		/**
		 * When rendering large numbers in the information element for the table
		 * (i.e. "Showing 1 to 10 of 57 entries") DataTables will render large numbers
		 * to have a comma separator for the 'thousands' units (e.g. 1 million is
		 * rendered as "1,000,000") to help readability for the end user. This
		 * function will override the default method DataTables uses.
		 *  @type function
		 *  @member
		 *  @param {int} toFormat number to be formatted
		 *  @returns {string} formatted string for DataTables to show the number
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.formatNumber
		 *
		 *  @example
		 *    // Format a number using a single quote for the separator (note that
		 *    // this can also be done with the language.thousands option)
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "formatNumber": function ( toFormat ) {
		 *          return toFormat.toString().replace(
		 *            /\B(?=(\d{3})+(?!\d))/g, "'"
		 *          );
		 *        };
		 *      } );
		 *    } );
		 */
		"fnFormatNumber": function ( toFormat ) {
			return toFormat.toString().replace(
				/\B(?=(\d{3})+(?!\d))/g,
				this.oLanguage.sThousands
			);
		},
	
	
		/**
		 * This function is called on every 'draw' event, and allows you to
		 * dynamically modify the header row. This can be used to calculate and
		 * display useful information about the table.
		 *  @type function
		 *  @param {node} head "TR" element for the header
		 *  @param {array} data Full table data (as derived from the original HTML)
		 *  @param {int} start Index for the current display starting point in the
		 *    display array
		 *  @param {int} end Index for the current display ending point in the
		 *    display array
		 *  @param {array int} display Index array to translate the visual position
		 *    to the full data array
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.headerCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "fheaderCallback": function( head, data, start, end, display ) {
		 *          head.getElementsByTagName('th')[0].innerHTML = "Displaying "+(end-start)+" records";
		 *        }
		 *      } );
		 *    } )
		 */
		"fnHeaderCallback": null,
	
	
		/**
		 * The information element can be used to convey information about the current
		 * state of the table. Although the internationalisation options presented by
		 * DataTables are quite capable of dealing with most customisations, there may
		 * be times where you wish to customise the string further. This callback
		 * allows you to do exactly that.
		 *  @type function
		 *  @param {object} oSettings DataTables settings object
		 *  @param {int} start Starting position in data for the draw
		 *  @param {int} end End position in data for the draw
		 *  @param {int} max Total number of rows in the table (regardless of
		 *    filtering)
		 *  @param {int} total Total number of rows in the data set, after filtering
		 *  @param {string} pre The string that DataTables has formatted using it's
		 *    own rules
		 *  @returns {string} The string to be displayed in the information element.
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.infoCallback
		 *
		 *  @example
		 *    $('#example').dataTable( {
		 *      "infoCallback": function( settings, start, end, max, total, pre ) {
		 *        return start +" to "+ end;
		 *      }
		 *    } );
		 */
		"fnInfoCallback": null,
	
	
		/**
		 * Called when the table has been initialised. Normally DataTables will
		 * initialise sequentially and there will be no need for this function,
		 * however, this does not hold true when using external language information
		 * since that is obtained using an async XHR call.
		 *  @type function
		 *  @param {object} settings DataTables settings object
		 *  @param {object} json The JSON object request from the server - only
		 *    present if client-side Ajax sourced data is used
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.initComplete
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "initComplete": function(settings, json) {
		 *          alert( 'DataTables has finished its initialisation.' );
		 *        }
		 *      } );
		 *    } )
		 */
		"fnInitComplete": null,
	
	
		/**
		 * Called at the very start of each table draw and can be used to cancel the
		 * draw by returning false, any other return (including undefined) results in
		 * the full draw occurring).
		 *  @type function
		 *  @param {object} settings DataTables settings object
		 *  @returns {boolean} False will cancel the draw, anything else (including no
		 *    return) will allow it to complete.
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.preDrawCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "preDrawCallback": function( settings ) {
		 *          if ( $('#test').val() == 1 ) {
		 *            return false;
		 *          }
		 *        }
		 *      } );
		 *    } );
		 */
		"fnPreDrawCallback": null,
	
	
		/**
		 * This function allows you to 'post process' each row after it have been
		 * generated for each table draw, but before it is rendered on screen. This
		 * function might be used for setting the row class name etc.
		 *  @type function
		 *  @param {node} row "TR" element for the current row
		 *  @param {array} data Raw data array for this row
		 *  @param {int} displayIndex The display index for the current table draw
		 *  @param {int} displayIndexFull The index of the data in the full list of
		 *    rows (after filtering)
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.rowCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "rowCallback": function( row, data, displayIndex, displayIndexFull ) {
		 *          // Bold the grade for all 'A' grade browsers
		 *          if ( data[4] == "A" ) {
		 *            $('td:eq(4)', row).html( '<b>A</b>' );
		 *          }
		 *        }
		 *      } );
		 *    } );
		 */
		"fnRowCallback": null,
	
	
		/**
		 * __Deprecated__ The functionality provided by this parameter has now been
		 * superseded by that provided through `ajax`, which should be used instead.
		 *
		 * This parameter allows you to override the default function which obtains
		 * the data from the server so something more suitable for your application.
		 * For example you could use POST data, or pull information from a Gears or
		 * AIR database.
		 *  @type function
		 *  @member
		 *  @param {string} source HTTP source to obtain the data from (`ajax`)
		 *  @param {array} data A key/value pair object containing the data to send
		 *    to the server
		 *  @param {function} callback to be called on completion of the data get
		 *    process that will draw the data on the page.
		 *  @param {object} settings DataTables settings object
		 *
		 *  @dtopt Callbacks
		 *  @dtopt Server-side
		 *  @name DataTable.defaults.serverData
		 *
		 *  @deprecated 1.10. Please use `ajax` for this functionality now.
		 */
		"fnServerData": null,
	
	
		/**
		 * __Deprecated__ The functionality provided by this parameter has now been
		 * superseded by that provided through `ajax`, which should be used instead.
		 *
		 *  It is often useful to send extra data to the server when making an Ajax
		 * request - for example custom filtering information, and this callback
		 * function makes it trivial to send extra information to the server. The
		 * passed in parameter is the data set that has been constructed by
		 * DataTables, and you can add to this or modify it as you require.
		 *  @type function
		 *  @param {array} data Data array (array of objects which are name/value
		 *    pairs) that has been constructed by DataTables and will be sent to the
		 *    server. In the case of Ajax sourced data with server-side processing
		 *    this will be an empty array, for server-side processing there will be a
		 *    significant number of parameters!
		 *  @returns {undefined} Ensure that you modify the data array passed in,
		 *    as this is passed by reference.
		 *
		 *  @dtopt Callbacks
		 *  @dtopt Server-side
		 *  @name DataTable.defaults.serverParams
		 *
		 *  @deprecated 1.10. Please use `ajax` for this functionality now.
		 */
		"fnServerParams": null,
	
	
		/**
		 * Load the table state. With this function you can define from where, and how, the
		 * state of a table is loaded. By default DataTables will load from `localStorage`
		 * but you might wish to use a server-side database or cookies.
		 *  @type function
		 *  @member
		 *  @param {object} settings DataTables settings object
		 *  @param {object} callback Callback that can be executed when done. It
		 *    should be passed the loaded state object.
		 *  @return {object} The DataTables state object to be loaded
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.stateLoadCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateSave": true,
		 *        "stateLoadCallback": function (settings, callback) {
		 *          $.ajax( {
		 *            "url": "/state_load",
		 *            "dataType": "json",
		 *            "success": function (json) {
		 *              callback( json );
		 *            }
		 *          } );
		 *        }
		 *      } );
		 *    } );
		 */
		"fnStateLoadCallback": function ( settings ) {
			try {
				return JSON.parse(
					(settings.iStateDuration === -1 ? sessionStorage : localStorage).getItem(
						'DataTables_'+settings.sInstance+'_'+location.pathname
					)
				);
			} catch (e) {
				return {};
			}
		},
	
	
		/**
		 * Callback which allows modification of the saved state prior to loading that state.
		 * This callback is called when the table is loading state from the stored data, but
		 * prior to the settings object being modified by the saved state. Note that for
		 * plug-in authors, you should use the `stateLoadParams` event to load parameters for
		 * a plug-in.
		 *  @type function
		 *  @param {object} settings DataTables settings object
		 *  @param {object} data The state object that is to be loaded
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.stateLoadParams
		 *
		 *  @example
		 *    // Remove a saved filter, so filtering is never loaded
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateSave": true,
		 *        "stateLoadParams": function (settings, data) {
		 *          data.oSearch.sSearch = "";
		 *        }
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Disallow state loading by returning false
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateSave": true,
		 *        "stateLoadParams": function (settings, data) {
		 *          return false;
		 *        }
		 *      } );
		 *    } );
		 */
		"fnStateLoadParams": null,
	
	
		/**
		 * Callback that is called when the state has been loaded from the state saving method
		 * and the DataTables settings object has been modified as a result of the loaded state.
		 *  @type function
		 *  @param {object} settings DataTables settings object
		 *  @param {object} data The state object that was loaded
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.stateLoaded
		 *
		 *  @example
		 *    // Show an alert with the filtering value that was saved
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateSave": true,
		 *        "stateLoaded": function (settings, data) {
		 *          alert( 'Saved filter was: '+data.oSearch.sSearch );
		 *        }
		 *      } );
		 *    } );
		 */
		"fnStateLoaded": null,
	
	
		/**
		 * Save the table state. This function allows you to define where and how the state
		 * information for the table is stored By default DataTables will use `localStorage`
		 * but you might wish to use a server-side database or cookies.
		 *  @type function
		 *  @member
		 *  @param {object} settings DataTables settings object
		 *  @param {object} data The state object to be saved
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.stateSaveCallback
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateSave": true,
		 *        "stateSaveCallback": function (settings, data) {
		 *          // Send an Ajax request to the server with the state object
		 *          $.ajax( {
		 *            "url": "/state_save",
		 *            "data": data,
		 *            "dataType": "json",
		 *            "method": "POST"
		 *            "success": function () {}
		 *          } );
		 *        }
		 *      } );
		 *    } );
		 */
		"fnStateSaveCallback": function ( settings, data ) {
			try {
				(settings.iStateDuration === -1 ? sessionStorage : localStorage).setItem(
					'DataTables_'+settings.sInstance+'_'+location.pathname,
					JSON.stringify( data )
				);
			} catch (e) {}
		},
	
	
		/**
		 * Callback which allows modification of the state to be saved. Called when the table
		 * has changed state a new state save is required. This method allows modification of
		 * the state saving object prior to actually doing the save, including addition or
		 * other state properties or modification. Note that for plug-in authors, you should
		 * use the `stateSaveParams` event to save parameters for a plug-in.
		 *  @type function
		 *  @param {object} settings DataTables settings object
		 *  @param {object} data The state object to be saved
		 *
		 *  @dtopt Callbacks
		 *  @name DataTable.defaults.stateSaveParams
		 *
		 *  @example
		 *    // Remove a saved filter, so filtering is never saved
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateSave": true,
		 *        "stateSaveParams": function (settings, data) {
		 *          data.oSearch.sSearch = "";
		 *        }
		 *      } );
		 *    } );
		 */
		"fnStateSaveParams": null,
	
	
		/**
		 * Duration for which the saved state information is considered valid. After this period
		 * has elapsed the state will be returned to the default.
		 * Value is given in seconds.
		 *  @type int
		 *  @default 7200 <i>(2 hours)</i>
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.stateDuration
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "stateDuration": 60*60*24; // 1 day
		 *      } );
		 *    } )
		 */
		"iStateDuration": 7200,
	
	
		/**
		 * When enabled DataTables will not make a request to the server for the first
		 * page draw - rather it will use the data already on the page (no sorting etc
		 * will be applied to it), thus saving on an XHR at load time. `deferLoading`
		 * is used to indicate that deferred loading is required, but it is also used
		 * to tell DataTables how many records there are in the full table (allowing
		 * the information element and pagination to be displayed correctly). In the case
		 * where a filtering is applied to the table on initial load, this can be
		 * indicated by giving the parameter as an array, where the first element is
		 * the number of records available after filtering and the second element is the
		 * number of records without filtering (allowing the table information element
		 * to be shown correctly).
		 *  @type int | array
		 *  @default null
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.deferLoading
		 *
		 *  @example
		 *    // 57 records available in the table, no filtering applied
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "serverSide": true,
		 *        "ajax": "scripts/server_processing.php",
		 *        "deferLoading": 57
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // 57 records after filtering, 100 without filtering (an initial filter applied)
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "serverSide": true,
		 *        "ajax": "scripts/server_processing.php",
		 *        "deferLoading": [ 57, 100 ],
		 *        "search": {
		 *          "search": "my_filter"
		 *        }
		 *      } );
		 *    } );
		 */
		"iDeferLoading": null,
	
	
		/**
		 * Number of rows to display on a single page when using pagination. If
		 * feature enabled (`lengthChange`) then the end user will be able to override
		 * this to a custom setting using a pop-up menu.
		 *  @type int
		 *  @default 10
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.pageLength
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "pageLength": 50
		 *      } );
		 *    } )
		 */
		"iDisplayLength": 10,
	
	
		/**
		 * Define the starting point for data display when using DataTables with
		 * pagination. Note that this parameter is the number of records, rather than
		 * the page number, so if you have 10 records per page and want to start on
		 * the third page, it should be "20".
		 *  @type int
		 *  @default 0
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.displayStart
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "displayStart": 20
		 *      } );
		 *    } )
		 */
		"iDisplayStart": 0,
	
	
		/**
		 * By default DataTables allows keyboard navigation of the table (sorting, paging,
		 * and filtering) by adding a `tabindex` attribute to the required elements. This
		 * allows you to tab through the controls and press the enter key to activate them.
		 * The tabindex is default 0, meaning that the tab follows the flow of the document.
		 * You can overrule this using this parameter if you wish. Use a value of -1 to
		 * disable built-in keyboard navigation.
		 *  @type int
		 *  @default 0
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.tabIndex
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "tabIndex": 1
		 *      } );
		 *    } );
		 */
		"iTabIndex": 0,
	
	
		/**
		 * Classes that DataTables assigns to the various components and features
		 * that it adds to the HTML table. This allows classes to be configured
		 * during initialisation in addition to through the static
		 * {@link DataTable.ext.oStdClasses} object).
		 *  @namespace
		 *  @name DataTable.defaults.classes
		 */
		"oClasses": {},
	
	
		/**
		 * All strings that DataTables uses in the user interface that it creates
		 * are defined in this object, allowing you to modified them individually or
		 * completely replace them all as required.
		 *  @namespace
		 *  @name DataTable.defaults.language
		 */
		"oLanguage": {
			/**
			 * Strings that are used for WAI-ARIA labels and controls only (these are not
			 * actually visible on the page, but will be read by screenreaders, and thus
			 * must be internationalised as well).
			 *  @namespace
			 *  @name DataTable.defaults.language.aria
			 */
			"oAria": {
				/**
				 * ARIA label that is added to the table headers when the column may be
				 * sorted ascending by activing the column (click or return when focused).
				 * Note that the column header is prefixed to this string.
				 *  @type string
				 *  @default : activate to sort column ascending
				 *
				 *  @dtopt Language
				 *  @name DataTable.defaults.language.aria.sortAscending
				 *
				 *  @example
				 *    $(document).ready( function() {
				 *      $('#example').dataTable( {
				 *        "language": {
				 *          "aria": {
				 *            "sortAscending": " - click/return to sort ascending"
				 *          }
				 *        }
				 *      } );
				 *    } );
				 */
				"sSortAscending": ": activate to sort column ascending",
	
				/**
				 * ARIA label that is added to the table headers when the column may be
				 * sorted descending by activing the column (click or return when focused).
				 * Note that the column header is prefixed to this string.
				 *  @type string
				 *  @default : activate to sort column ascending
				 *
				 *  @dtopt Language
				 *  @name DataTable.defaults.language.aria.sortDescending
				 *
				 *  @example
				 *    $(document).ready( function() {
				 *      $('#example').dataTable( {
				 *        "language": {
				 *          "aria": {
				 *            "sortDescending": " - click/return to sort descending"
				 *          }
				 *        }
				 *      } );
				 *    } );
				 */
				"sSortDescending": ": activate to sort column descending"
			},
	
			/**
			 * Pagination string used by DataTables for the built-in pagination
			 * control types.
			 *  @namespace
			 *  @name DataTable.defaults.language.paginate
			 */
			"oPaginate": {
				/**
				 * Text to use when using the 'full_numbers' type of pagination for the
				 * button to take the user to the first page.
				 *  @type string
				 *  @default First
				 *
				 *  @dtopt Language
				 *  @name DataTable.defaults.language.paginate.first
				 *
				 *  @example
				 *    $(document).ready( function() {
				 *      $('#example').dataTable( {
				 *        "language": {
				 *          "paginate": {
				 *            "first": "First page"
				 *          }
				 *        }
				 *      } );
				 *    } );
				 */
				"sFirst": "First",
	
	
				/**
				 * Text to use when using the 'full_numbers' type of pagination for the
				 * button to take the user to the last page.
				 *  @type string
				 *  @default Last
				 *
				 *  @dtopt Language
				 *  @name DataTable.defaults.language.paginate.last
				 *
				 *  @example
				 *    $(document).ready( function() {
				 *      $('#example').dataTable( {
				 *        "language": {
				 *          "paginate": {
				 *            "last": "Last page"
				 *          }
				 *        }
				 *      } );
				 *    } );
				 */
				"sLast": "Last",
	
	
				/**
				 * Text to use for the 'next' pagination button (to take the user to the
				 * next page).
				 *  @type string
				 *  @default Next
				 *
				 *  @dtopt Language
				 *  @name DataTable.defaults.language.paginate.next
				 *
				 *  @example
				 *    $(document).ready( function() {
				 *      $('#example').dataTable( {
				 *        "language": {
				 *          "paginate": {
				 *            "next": "Next page"
				 *          }
				 *        }
				 *      } );
				 *    } );
				 */
				"sNext": "Next",
	
	
				/**
				 * Text to use for the 'previous' pagination button (to take the user to
				 * the previous page).
				 *  @type string
				 *  @default Previous
				 *
				 *  @dtopt Language
				 *  @name DataTable.defaults.language.paginate.previous
				 *
				 *  @example
				 *    $(document).ready( function() {
				 *      $('#example').dataTable( {
				 *        "language": {
				 *          "paginate": {
				 *            "previous": "Previous page"
				 *          }
				 *        }
				 *      } );
				 *    } );
				 */
				"sPrevious": "Previous"
			},
	
			/**
			 * This string is shown in preference to `zeroRecords` when the table is
			 * empty of data (regardless of filtering). Note that this is an optional
			 * parameter - if it is not given, the value of `zeroRecords` will be used
			 * instead (either the default or given value).
			 *  @type string
			 *  @default No data available in table
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.emptyTable
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "emptyTable": "No data available in table"
			 *        }
			 *      } );
			 *    } );
			 */
			"sEmptyTable": "No data available in table",
	
	
			/**
			 * This string gives information to the end user about the information
			 * that is current on display on the page. The following tokens can be
			 * used in the string and will be dynamically replaced as the table
			 * display updates. This tokens can be placed anywhere in the string, or
			 * removed as needed by the language requires:
			 *
			 * * `\_START\_` - Display index of the first record on the current page
			 * * `\_END\_` - Display index of the last record on the current page
			 * * `\_TOTAL\_` - Number of records in the table after filtering
			 * * `\_MAX\_` - Number of records in the table without filtering
			 * * `\_PAGE\_` - Current page number
			 * * `\_PAGES\_` - Total number of pages of data in the table
			 *
			 *  @type string
			 *  @default Showing _START_ to _END_ of _TOTAL_ entries
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.info
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "info": "Showing page _PAGE_ of _PAGES_"
			 *        }
			 *      } );
			 *    } );
			 */
			"sInfo": "Showing _START_ to _END_ of _TOTAL_ entries",
	
	
			/**
			 * Display information string for when the table is empty. Typically the
			 * format of this string should match `info`.
			 *  @type string
			 *  @default Showing 0 to 0 of 0 entries
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.infoEmpty
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "infoEmpty": "No entries to show"
			 *        }
			 *      } );
			 *    } );
			 */
			"sInfoEmpty": "Showing 0 to 0 of 0 entries",
	
	
			/**
			 * When a user filters the information in a table, this string is appended
			 * to the information (`info`) to give an idea of how strong the filtering
			 * is. The variable _MAX_ is dynamically updated.
			 *  @type string
			 *  @default (filtered from _MAX_ total entries)
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.infoFiltered
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "infoFiltered": " - filtering from _MAX_ records"
			 *        }
			 *      } );
			 *    } );
			 */
			"sInfoFiltered": "(filtered from _MAX_ total entries)",
	
	
			/**
			 * If can be useful to append extra information to the info string at times,
			 * and this variable does exactly that. This information will be appended to
			 * the `info` (`infoEmpty` and `infoFiltered` in whatever combination they are
			 * being used) at all times.
			 *  @type string
			 *  @default <i>Empty string</i>
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.infoPostFix
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "infoPostFix": "All records shown are derived from real information."
			 *        }
			 *      } );
			 *    } );
			 */
			"sInfoPostFix": "",
	
	
			/**
			 * This decimal place operator is a little different from the other
			 * language options since DataTables doesn't output floating point
			 * numbers, so it won't ever use this for display of a number. Rather,
			 * what this parameter does is modify the sort methods of the table so
			 * that numbers which are in a format which has a character other than
			 * a period (`.`) as a decimal place will be sorted numerically.
			 *
			 * Note that numbers with different decimal places cannot be shown in
			 * the same table and still be sortable, the table must be consistent.
			 * However, multiple different tables on the page can use different
			 * decimal place characters.
			 *  @type string
			 *  @default 
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.decimal
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "decimal": ","
			 *          "thousands": "."
			 *        }
			 *      } );
			 *    } );
			 */
			"sDecimal": "",
	
	
			/**
			 * DataTables has a build in number formatter (`formatNumber`) which is
			 * used to format large numbers that are used in the table information.
			 * By default a comma is used, but this can be trivially changed to any
			 * character you wish with this parameter.
			 *  @type string
			 *  @default ,
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.thousands
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "thousands": "'"
			 *        }
			 *      } );
			 *    } );
			 */
			"sThousands": ",",
	
	
			/**
			 * Detail the action that will be taken when the drop down menu for the
			 * pagination length option is changed. The '_MENU_' variable is replaced
			 * with a default select list of 10, 25, 50 and 100, and can be replaced
			 * with a custom select box if required.
			 *  @type string
			 *  @default Show _MENU_ entries
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.lengthMenu
			 *
			 *  @example
			 *    // Language change only
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "lengthMenu": "Display _MENU_ records"
			 *        }
			 *      } );
			 *    } );
			 *
			 *  @example
			 *    // Language and options change
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "lengthMenu": 'Display <select>'+
			 *            '<option value="10">10</option>'+
			 *            '<option value="20">20</option>'+
			 *            '<option value="30">30</option>'+
			 *            '<option value="40">40</option>'+
			 *            '<option value="50">50</option>'+
			 *            '<option value="-1">All</option>'+
			 *            '</select> records'
			 *        }
			 *      } );
			 *    } );
			 */
			"sLengthMenu": "Show _MENU_ entries",
	
	
			/**
			 * When using Ajax sourced data and during the first draw when DataTables is
			 * gathering the data, this message is shown in an empty row in the table to
			 * indicate to the end user the the data is being loaded. Note that this
			 * parameter is not used when loading data by server-side processing, just
			 * Ajax sourced data with client-side processing.
			 *  @type string
			 *  @default Loading...
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.loadingRecords
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "loadingRecords": "Please wait - loading..."
			 *        }
			 *      } );
			 *    } );
			 */
			"sLoadingRecords": "Loading...",
	
	
			/**
			 * Text which is displayed when the table is processing a user action
			 * (usually a sort command or similar).
			 *  @type string
			 *  @default Processing...
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.processing
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "processing": "DataTables is currently busy"
			 *        }
			 *      } );
			 *    } );
			 */
			"sProcessing": "Processing...",
	
	
			/**
			 * Details the actions that will be taken when the user types into the
			 * filtering input text box. The variable "_INPUT_", if used in the string,
			 * is replaced with the HTML text box for the filtering input allowing
			 * control over where it appears in the string. If "_INPUT_" is not given
			 * then the input box is appended to the string automatically.
			 *  @type string
			 *  @default Search:
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.search
			 *
			 *  @example
			 *    // Input text box will be appended at the end automatically
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "search": "Filter records:"
			 *        }
			 *      } );
			 *    } );
			 *
			 *  @example
			 *    // Specify where the filter should appear
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "search": "Apply filter _INPUT_ to table"
			 *        }
			 *      } );
			 *    } );
			 */
			"sSearch": "Search:",
	
	
			/**
			 * Assign a `placeholder` attribute to the search `input` element
			 *  @type string
			 *  @default 
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.searchPlaceholder
			 */
			"sSearchPlaceholder": "",
	
	
			/**
			 * All of the language information can be stored in a file on the
			 * server-side, which DataTables will look up if this parameter is passed.
			 * It must store the URL of the language file, which is in a JSON format,
			 * and the object has the same properties as the oLanguage object in the
			 * initialiser object (i.e. the above parameters). Please refer to one of
			 * the example language files to see how this works in action.
			 *  @type string
			 *  @default <i>Empty string - i.e. disabled</i>
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.url
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "url": "http://www.sprymedia.co.uk/dataTables/lang.txt"
			 *        }
			 *      } );
			 *    } );
			 */
			"sUrl": "",
	
	
			/**
			 * Text shown inside the table records when the is no information to be
			 * displayed after filtering. `emptyTable` is shown when there is simply no
			 * information in the table at all (regardless of filtering).
			 *  @type string
			 *  @default No matching records found
			 *
			 *  @dtopt Language
			 *  @name DataTable.defaults.language.zeroRecords
			 *
			 *  @example
			 *    $(document).ready( function() {
			 *      $('#example').dataTable( {
			 *        "language": {
			 *          "zeroRecords": "No records to display"
			 *        }
			 *      } );
			 *    } );
			 */
			"sZeroRecords": "No matching records found"
		},
	
	
		/**
		 * This parameter allows you to have define the global filtering state at
		 * initialisation time. As an object the `search` parameter must be
		 * defined, but all other parameters are optional. When `regex` is true,
		 * the search string will be treated as a regular expression, when false
		 * (default) it will be treated as a straight string. When `smart`
		 * DataTables will use it's smart filtering methods (to word match at
		 * any point in the data), when false this will not be done.
		 *  @namespace
		 *  @extends DataTable.models.oSearch
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.search
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "search": {"search": "Initial search"}
		 *      } );
		 *    } )
		 */
		"oSearch": $.extend( {}, DataTable.models.oSearch ),
	
	
		/**
		 * __Deprecated__ The functionality provided by this parameter has now been
		 * superseded by that provided through `ajax`, which should be used instead.
		 *
		 * By default DataTables will look for the property `data` (or `aaData` for
		 * compatibility with DataTables 1.9-) when obtaining data from an Ajax
		 * source or for server-side processing - this parameter allows that
		 * property to be changed. You can use Javascript dotted object notation to
		 * get a data source for multiple levels of nesting.
		 *  @type string
		 *  @default data
		 *
		 *  @dtopt Options
		 *  @dtopt Server-side
		 *  @name DataTable.defaults.ajaxDataProp
		 *
		 *  @deprecated 1.10. Please use `ajax` for this functionality now.
		 */
		"sAjaxDataProp": "data",
	
	
		/**
		 * __Deprecated__ The functionality provided by this parameter has now been
		 * superseded by that provided through `ajax`, which should be used instead.
		 *
		 * You can instruct DataTables to load data from an external
		 * source using this parameter (use aData if you want to pass data in you
		 * already have). Simply provide a url a JSON object can be obtained from.
		 *  @type string
		 *  @default null
		 *
		 *  @dtopt Options
		 *  @dtopt Server-side
		 *  @name DataTable.defaults.ajaxSource
		 *
		 *  @deprecated 1.10. Please use `ajax` for this functionality now.
		 */
		"sAjaxSource": null,
	
	
		/**
		 * This initialisation variable allows you to specify exactly where in the
		 * DOM you want DataTables to inject the various controls it adds to the page
		 * (for example you might want the pagination controls at the top of the
		 * table). DIV elements (with or without a custom class) can also be added to
		 * aid styling. The follow syntax is used:
		 *   <ul>
		 *     <li>The following options are allowed:
		 *       <ul>
		 *         <li>'l' - Length changing</li>
		 *         <li>'f' - Filtering input</li>
		 *         <li>'t' - The table!</li>
		 *         <li>'i' - Information</li>
		 *         <li>'p' - Pagination</li>
		 *         <li>'r' - pRocessing</li>
		 *       </ul>
		 *     </li>
		 *     <li>The following constants are allowed:
		 *       <ul>
		 *         <li>'H' - jQueryUI theme "header" classes ('fg-toolbar ui-widget-header ui-corner-tl ui-corner-tr ui-helper-clearfix')</li>
		 *         <li>'F' - jQueryUI theme "footer" classes ('fg-toolbar ui-widget-header ui-corner-bl ui-corner-br ui-helper-clearfix')</li>
		 *       </ul>
		 *     </li>
		 *     <li>The following syntax is expected:
		 *       <ul>
		 *         <li>'&lt;' and '&gt;' - div elements</li>
		 *         <li>'&lt;"class" and '&gt;' - div with a class</li>
		 *         <li>'&lt;"#id" and '&gt;' - div with an ID</li>
		 *       </ul>
		 *     </li>
		 *     <li>Examples:
		 *       <ul>
		 *         <li>'&lt;"wrapper"flipt&gt;'</li>
		 *         <li>'&lt;lf&lt;t&gt;ip&gt;'</li>
		 *       </ul>
		 *     </li>
		 *   </ul>
		 *  @type string
		 *  @default lfrtip <i>(when `jQueryUI` is false)</i> <b>or</b>
		 *    <"H"lfr>t<"F"ip> <i>(when `jQueryUI` is true)</i>
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.dom
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "dom": '&lt;"top"i&gt;rt&lt;"bottom"flp&gt;&lt;"clear"&gt;'
		 *      } );
		 *    } );
		 */
		"sDom": "lfrtip",
	
	
		/**
		 * Search delay option. This will throttle full table searches that use the
		 * DataTables provided search input element (it does not effect calls to
		 * `dt-api search()`, providing a delay before the search is made.
		 *  @type integer
		 *  @default 0
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.searchDelay
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "searchDelay": 200
		 *      } );
		 *    } )
		 */
		"searchDelay": null,
	
	
		/**
		 * DataTables features six different built-in options for the buttons to
		 * display for pagination control:
		 *
		 * * `numbers` - Page number buttons only
		 * * `simple` - 'Previous' and 'Next' buttons only
		 * * 'simple_numbers` - 'Previous' and 'Next' buttons, plus page numbers
		 * * `full` - 'First', 'Previous', 'Next' and 'Last' buttons
		 * * `full_numbers` - 'First', 'Previous', 'Next' and 'Last' buttons, plus page numbers
		 * * `first_last_numbers` - 'First' and 'Last' buttons, plus page numbers
		 *  
		 * Further methods can be added using {@link DataTable.ext.oPagination}.
		 *  @type string
		 *  @default simple_numbers
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.pagingType
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "pagingType": "full_numbers"
		 *      } );
		 *    } )
		 */
		"sPaginationType": "simple_numbers",
	
	
		/**
		 * Enable horizontal scrolling. When a table is too wide to fit into a
		 * certain layout, or you have a large number of columns in the table, you
		 * can enable x-scrolling to show the table in a viewport, which can be
		 * scrolled. This property can be `true` which will allow the table to
		 * scroll horizontally when needed, or any CSS unit, or a number (in which
		 * case it will be treated as a pixel measurement). Setting as simply `true`
		 * is recommended.
		 *  @type boolean|string
		 *  @default <i>blank string - i.e. disabled</i>
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.scrollX
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "scrollX": true,
		 *        "scrollCollapse": true
		 *      } );
		 *    } );
		 */
		"sScrollX": "",
	
	
		/**
		 * This property can be used to force a DataTable to use more width than it
		 * might otherwise do when x-scrolling is enabled. For example if you have a
		 * table which requires to be well spaced, this parameter is useful for
		 * "over-sizing" the table, and thus forcing scrolling. This property can by
		 * any CSS unit, or a number (in which case it will be treated as a pixel
		 * measurement).
		 *  @type string
		 *  @default <i>blank string - i.e. disabled</i>
		 *
		 *  @dtopt Options
		 *  @name DataTable.defaults.scrollXInner
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "scrollX": "100%",
		 *        "scrollXInner": "110%"
		 *      } );
		 *    } );
		 */
		"sScrollXInner": "",
	
	
		/**
		 * Enable vertical scrolling. Vertical scrolling will constrain the DataTable
		 * to the given height, and enable scrolling for any data which overflows the
		 * current viewport. This can be used as an alternative to paging to display
		 * a lot of data in a small area (although paging and scrolling can both be
		 * enabled at the same time). This property can be any CSS unit, or a number
		 * (in which case it will be treated as a pixel measurement).
		 *  @type string
		 *  @default <i>blank string - i.e. disabled</i>
		 *
		 *  @dtopt Features
		 *  @name DataTable.defaults.scrollY
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "scrollY": "200px",
		 *        "paginate": false
		 *      } );
		 *    } );
		 */
		"sScrollY": "",
	
	
		/**
		 * __Deprecated__ The functionality provided by this parameter has now been
		 * superseded by that provided through `ajax`, which should be used instead.
		 *
		 * Set the HTTP method that is used to make the Ajax call for server-side
		 * processing or Ajax sourced data.
		 *  @type string
		 *  @default GET
		 *
		 *  @dtopt Options
		 *  @dtopt Server-side
		 *  @name DataTable.defaults.serverMethod
		 *
		 *  @deprecated 1.10. Please use `ajax` for this functionality now.
		 */
		"sServerMethod": "GET",
	
	
		/**
		 * DataTables makes use of renderers when displaying HTML elements for
		 * a table. These renderers can be added or modified by plug-ins to
		 * generate suitable mark-up for a site. For example the Bootstrap
		 * integration plug-in for DataTables uses a paging button renderer to
		 * display pagination buttons in the mark-up required by Bootstrap.
		 *
		 * For further information about the renderers available see
		 * DataTable.ext.renderer
		 *  @type string|object
		 *  @default null
		 *
		 *  @name DataTable.defaults.renderer
		 *
		 */
		"renderer": null,
	
	
		/**
		 * Set the data property name that DataTables should use to get a row's id
		 * to set as the `id` property in the node.
		 *  @type string
		 *  @default DT_RowId
		 *
		 *  @name DataTable.defaults.rowId
		 */
		"rowId": "DT_RowId"
	};
	
	_fnHungarianMap( DataTable.defaults );
	
	
	
	/*
	 * Developer note - See note in model.defaults.js about the use of Hungarian
	 * notation and camel case.
	 */
	
	/**
	 * Column options that can be given to DataTables at initialisation time.
	 *  @namespace
	 */
	DataTable.defaults.column = {
		/**
		 * Define which column(s) an order will occur on for this column. This
		 * allows a column's ordering to take multiple columns into account when
		 * doing a sort or use the data from a different column. For example first
		 * name / last name columns make sense to do a multi-column sort over the
		 * two columns.
		 *  @type array|int
		 *  @default null <i>Takes the value of the column index automatically</i>
		 *
		 *  @name DataTable.defaults.column.orderData
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "orderData": [ 0, 1 ], "targets": [ 0 ] },
		 *          { "orderData": [ 1, 0 ], "targets": [ 1 ] },
		 *          { "orderData": 2, "targets": [ 2 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "orderData": [ 0, 1 ] },
		 *          { "orderData": [ 1, 0 ] },
		 *          { "orderData": 2 },
		 *          null,
		 *          null
		 *        ]
		 *      } );
		 *    } );
		 */
		"aDataSort": null,
		"iDataSort": -1,
	
	
		/**
		 * You can control the default ordering direction, and even alter the
		 * behaviour of the sort handler (i.e. only allow ascending ordering etc)
		 * using this parameter.
		 *  @type array
		 *  @default [ 'asc', 'desc' ]
		 *
		 *  @name DataTable.defaults.column.orderSequence
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "orderSequence": [ "asc" ], "targets": [ 1 ] },
		 *          { "orderSequence": [ "desc", "asc", "asc" ], "targets": [ 2 ] },
		 *          { "orderSequence": [ "desc" ], "targets": [ 3 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          null,
		 *          { "orderSequence": [ "asc" ] },
		 *          { "orderSequence": [ "desc", "asc", "asc" ] },
		 *          { "orderSequence": [ "desc" ] },
		 *          null
		 *        ]
		 *      } );
		 *    } );
		 */
		"asSorting": [ 'asc', 'desc' ],
	
	
		/**
		 * Enable or disable filtering on the data in this column.
		 *  @type boolean
		 *  @default true
		 *
		 *  @name DataTable.defaults.column.searchable
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "searchable": false, "targets": [ 0 ] }
		 *        ] } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "searchable": false },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ] } );
		 *    } );
		 */
		"bSearchable": true,
	
	
		/**
		 * Enable or disable ordering on this column.
		 *  @type boolean
		 *  @default true
		 *
		 *  @name DataTable.defaults.column.orderable
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "orderable": false, "targets": [ 0 ] }
		 *        ] } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "orderable": false },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ] } );
		 *    } );
		 */
		"bSortable": true,
	
	
		/**
		 * Enable or disable the display of this column.
		 *  @type boolean
		 *  @default true
		 *
		 *  @name DataTable.defaults.column.visible
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "visible": false, "targets": [ 0 ] }
		 *        ] } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "visible": false },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ] } );
		 *    } );
		 */
		"bVisible": true,
	
	
		/**
		 * Developer definable function that is called whenever a cell is created (Ajax source,
		 * etc) or processed for input (DOM source). This can be used as a compliment to mRender
		 * allowing you to modify the DOM element (add background colour for example) when the
		 * element is available.
		 *  @type function
		 *  @param {element} td The TD node that has been created
		 *  @param {*} cellData The Data for the cell
		 *  @param {array|object} rowData The data for the whole row
		 *  @param {int} row The row index for the aoData data store
		 *  @param {int} col The column index for aoColumns
		 *
		 *  @name DataTable.defaults.column.createdCell
		 *  @dtopt Columns
		 *
		 *  @example
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [3],
		 *          "createdCell": function (td, cellData, rowData, row, col) {
		 *            if ( cellData == "1.7" ) {
		 *              $(td).css('color', 'blue')
		 *            }
		 *          }
		 *        } ]
		 *      });
		 *    } );
		 */
		"fnCreatedCell": null,
	
	
		/**
		 * This parameter has been replaced by `data` in DataTables to ensure naming
		 * consistency. `dataProp` can still be used, as there is backwards
		 * compatibility in DataTables for this option, but it is strongly
		 * recommended that you use `data` in preference to `dataProp`.
		 *  @name DataTable.defaults.column.dataProp
		 */
	
	
		/**
		 * This property can be used to read data from any data source property,
		 * including deeply nested objects / properties. `data` can be given in a
		 * number of different ways which effect its behaviour:
		 *
		 * * `integer` - treated as an array index for the data source. This is the
		 *   default that DataTables uses (incrementally increased for each column).
		 * * `string` - read an object property from the data source. There are
		 *   three 'special' options that can be used in the string to alter how
		 *   DataTables reads the data from the source object:
		 *    * `.` - Dotted Javascript notation. Just as you use a `.` in
		 *      Javascript to read from nested objects, so to can the options
		 *      specified in `data`. For example: `browser.version` or
		 *      `browser.name`. If your object parameter name contains a period, use
		 *      `\\` to escape it - i.e. `first\\.name`.
		 *    * `[]` - Array notation. DataTables can automatically combine data
		 *      from and array source, joining the data with the characters provided
		 *      between the two brackets. For example: `name[, ]` would provide a
		 *      comma-space separated list from the source array. If no characters
		 *      are provided between the brackets, the original array source is
		 *      returned.
		 *    * `()` - Function notation. Adding `()` to the end of a parameter will
		 *      execute a function of the name given. For example: `browser()` for a
		 *      simple function on the data source, `browser.version()` for a
		 *      function in a nested property or even `browser().version` to get an
		 *      object property if the function called returns an object. Note that
		 *      function notation is recommended for use in `render` rather than
		 *      `data` as it is much simpler to use as a renderer.
		 * * `null` - use the original data source for the row rather than plucking
		 *   data directly from it. This action has effects on two other
		 *   initialisation options:
		 *    * `defaultContent` - When null is given as the `data` option and
		 *      `defaultContent` is specified for the column, the value defined by
		 *      `defaultContent` will be used for the cell.
		 *    * `render` - When null is used for the `data` option and the `render`
		 *      option is specified for the column, the whole data source for the
		 *      row is used for the renderer.
		 * * `function` - the function given will be executed whenever DataTables
		 *   needs to set or get the data for a cell in the column. The function
		 *   takes three parameters:
		 *    * Parameters:
		 *      * `{array|object}` The data source for the row
		 *      * `{string}` The type call data requested - this will be 'set' when
		 *        setting data or 'filter', 'display', 'type', 'sort' or undefined
		 *        when gathering data. Note that when `undefined` is given for the
		 *        type DataTables expects to get the raw data for the object back<
		 *      * `{*}` Data to set when the second parameter is 'set'.
		 *    * Return:
		 *      * The return value from the function is not required when 'set' is
		 *        the type of call, but otherwise the return is what will be used
		 *        for the data requested.
		 *
		 * Note that `data` is a getter and setter option. If you just require
		 * formatting of data for output, you will likely want to use `render` which
		 * is simply a getter and thus simpler to use.
		 *
		 * Note that prior to DataTables 1.9.2 `data` was called `mDataProp`. The
		 * name change reflects the flexibility of this property and is consistent
		 * with the naming of mRender. If 'mDataProp' is given, then it will still
		 * be used by DataTables, as it automatically maps the old name to the new
		 * if required.
		 *
		 *  @type string|int|function|null
		 *  @default null <i>Use automatically calculated column index</i>
		 *
		 *  @name DataTable.defaults.column.data
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Read table data from objects
		 *    // JSON structure for each row:
		 *    //   {
		 *    //      "engine": {value},
		 *    //      "browser": {value},
		 *    //      "platform": {value},
		 *    //      "version": {value},
		 *    //      "grade": {value}
		 *    //   }
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "ajaxSource": "sources/objects.txt",
		 *        "columns": [
		 *          { "data": "engine" },
		 *          { "data": "browser" },
		 *          { "data": "platform" },
		 *          { "data": "version" },
		 *          { "data": "grade" }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Read information from deeply nested objects
		 *    // JSON structure for each row:
		 *    //   {
		 *    //      "engine": {value},
		 *    //      "browser": {value},
		 *    //      "platform": {
		 *    //         "inner": {value}
		 *    //      },
		 *    //      "details": [
		 *    //         {value}, {value}
		 *    //      ]
		 *    //   }
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "ajaxSource": "sources/deep.txt",
		 *        "columns": [
		 *          { "data": "engine" },
		 *          { "data": "browser" },
		 *          { "data": "platform.inner" },
		 *          { "data": "details.0" },
		 *          { "data": "details.1" }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `data` as a function to provide different information for
		 *    // sorting, filtering and display. In this case, currency (price)
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "data": function ( source, type, val ) {
		 *            if (type === 'set') {
		 *              source.price = val;
		 *              // Store the computed dislay and filter values for efficiency
		 *              source.price_display = val=="" ? "" : "$"+numberFormat(val);
		 *              source.price_filter  = val=="" ? "" : "$"+numberFormat(val)+" "+val;
		 *              return;
		 *            }
		 *            else if (type === 'display') {
		 *              return source.price_display;
		 *            }
		 *            else if (type === 'filter') {
		 *              return source.price_filter;
		 *            }
		 *            // 'sort', 'type' and undefined all just use the integer
		 *            return source.price;
		 *          }
		 *        } ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using default content
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "data": null,
		 *          "defaultContent": "Click to edit"
		 *        } ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using array notation - outputting a list from an array
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "data": "name[, ]"
		 *        } ]
		 *      } );
		 *    } );
		 *
		 */
		"mData": null,
	
	
		/**
		 * This property is the rendering partner to `data` and it is suggested that
		 * when you want to manipulate data for display (including filtering,
		 * sorting etc) without altering the underlying data for the table, use this
		 * property. `render` can be considered to be the the read only companion to
		 * `data` which is read / write (then as such more complex). Like `data`
		 * this option can be given in a number of different ways to effect its
		 * behaviour:
		 *
		 * * `integer` - treated as an array index for the data source. This is the
		 *   default that DataTables uses (incrementally increased for each column).
		 * * `string` - read an object property from the data source. There are
		 *   three 'special' options that can be used in the string to alter how
		 *   DataTables reads the data from the source object:
		 *    * `.` - Dotted Javascript notation. Just as you use a `.` in
		 *      Javascript to read from nested objects, so to can the options
		 *      specified in `data`. For example: `browser.version` or
		 *      `browser.name`. If your object parameter name contains a period, use
		 *      `\\` to escape it - i.e. `first\\.name`.
		 *    * `[]` - Array notation. DataTables can automatically combine data
		 *      from and array source, joining the data with the characters provided
		 *      between the two brackets. For example: `name[, ]` would provide a
		 *      comma-space separated list from the source array. If no characters
		 *      are provided between the brackets, the original array source is
		 *      returned.
		 *    * `()` - Function notation. Adding `()` to the end of a parameter will
		 *      execute a function of the name given. For example: `browser()` for a
		 *      simple function on the data source, `browser.version()` for a
		 *      function in a nested property or even `browser().version` to get an
		 *      object property if the function called returns an object.
		 * * `object` - use different data for the different data types requested by
		 *   DataTables ('filter', 'display', 'type' or 'sort'). The property names
		 *   of the object is the data type the property refers to and the value can
		 *   defined using an integer, string or function using the same rules as
		 *   `render` normally does. Note that an `_` option _must_ be specified.
		 *   This is the default value to use if you haven't specified a value for
		 *   the data type requested by DataTables.
		 * * `function` - the function given will be executed whenever DataTables
		 *   needs to set or get the data for a cell in the column. The function
		 *   takes three parameters:
		 *    * Parameters:
		 *      * {array|object} The data source for the row (based on `data`)
		 *      * {string} The type call data requested - this will be 'filter',
		 *        'display', 'type' or 'sort'.
		 *      * {array|object} The full data source for the row (not based on
		 *        `data`)
		 *    * Return:
		 *      * The return value from the function is what will be used for the
		 *        data requested.
		 *
		 *  @type string|int|function|object|null
		 *  @default null Use the data source value.
		 *
		 *  @name DataTable.defaults.column.render
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Create a comma separated list from an array of objects
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "ajaxSource": "sources/deep.txt",
		 *        "columns": [
		 *          { "data": "engine" },
		 *          { "data": "browser" },
		 *          {
		 *            "data": "platform",
		 *            "render": "[, ].name"
		 *          }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Execute a function to obtain data
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "data": null, // Use the full data source object for the renderer's source
		 *          "render": "browserName()"
		 *        } ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // As an object, extracting different data for the different types
		 *    // This would be used with a data source such as:
		 *    //   { "phone": 5552368, "phone_filter": "5552368 555-2368", "phone_display": "555-2368" }
		 *    // Here the `phone` integer is used for sorting and type detection, while `phone_filter`
		 *    // (which has both forms) is used for filtering for if a user inputs either format, while
		 *    // the formatted phone number is the one that is shown in the table.
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "data": null, // Use the full data source object for the renderer's source
		 *          "render": {
		 *            "_": "phone",
		 *            "filter": "phone_filter",
		 *            "display": "phone_display"
		 *          }
		 *        } ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Use as a function to create a link from the data source
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "data": "download_link",
		 *          "render": function ( data, type, full ) {
		 *            return '<a href="'+data+'">Download</a>';
		 *          }
		 *        } ]
		 *      } );
		 *    } );
		 */
		"mRender": null,
	
	
		/**
		 * Change the cell type created for the column - either TD cells or TH cells. This
		 * can be useful as TH cells have semantic meaning in the table body, allowing them
		 * to act as a header for a row (you may wish to add scope='row' to the TH elements).
		 *  @type string
		 *  @default td
		 *
		 *  @name DataTable.defaults.column.cellType
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Make the first column use TH cells
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [ {
		 *          "targets": [ 0 ],
		 *          "cellType": "th"
		 *        } ]
		 *      } );
		 *    } );
		 */
		"sCellType": "td",
	
	
		/**
		 * Class to give to each cell in this column.
		 *  @type string
		 *  @default <i>Empty string</i>
		 *
		 *  @name DataTable.defaults.column.class
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "class": "my_class", "targets": [ 0 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "class": "my_class" },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ]
		 *      } );
		 *    } );
		 */
		"sClass": "",
	
		/**
		 * When DataTables calculates the column widths to assign to each column,
		 * it finds the longest string in each column and then constructs a
		 * temporary table and reads the widths from that. The problem with this
		 * is that "mmm" is much wider then "iiii", but the latter is a longer
		 * string - thus the calculation can go wrong (doing it properly and putting
		 * it into an DOM object and measuring that is horribly(!) slow). Thus as
		 * a "work around" we provide this option. It will append its value to the
		 * text that is found to be the longest string for the column - i.e. padding.
		 * Generally you shouldn't need this!
		 *  @type string
		 *  @default <i>Empty string<i>
		 *
		 *  @name DataTable.defaults.column.contentPadding
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          null,
		 *          null,
		 *          null,
		 *          {
		 *            "contentPadding": "mmm"
		 *          }
		 *        ]
		 *      } );
		 *    } );
		 */
		"sContentPadding": "",
	
	
		/**
		 * Allows a default value to be given for a column's data, and will be used
		 * whenever a null data source is encountered (this can be because `data`
		 * is set to null, or because the data source itself is null).
		 *  @type string
		 *  @default null
		 *
		 *  @name DataTable.defaults.column.defaultContent
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          {
		 *            "data": null,
		 *            "defaultContent": "Edit",
		 *            "targets": [ -1 ]
		 *          }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          null,
		 *          null,
		 *          null,
		 *          {
		 *            "data": null,
		 *            "defaultContent": "Edit"
		 *          }
		 *        ]
		 *      } );
		 *    } );
		 */
		"sDefaultContent": null,
	
	
		/**
		 * This parameter is only used in DataTables' server-side processing. It can
		 * be exceptionally useful to know what columns are being displayed on the
		 * client side, and to map these to database fields. When defined, the names
		 * also allow DataTables to reorder information from the server if it comes
		 * back in an unexpected order (i.e. if you switch your columns around on the
		 * client-side, your server-side code does not also need updating).
		 *  @type string
		 *  @default <i>Empty string</i>
		 *
		 *  @name DataTable.defaults.column.name
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "name": "engine", "targets": [ 0 ] },
		 *          { "name": "browser", "targets": [ 1 ] },
		 *          { "name": "platform", "targets": [ 2 ] },
		 *          { "name": "version", "targets": [ 3 ] },
		 *          { "name": "grade", "targets": [ 4 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "name": "engine" },
		 *          { "name": "browser" },
		 *          { "name": "platform" },
		 *          { "name": "version" },
		 *          { "name": "grade" }
		 *        ]
		 *      } );
		 *    } );
		 */
		"sName": "",
	
	
		/**
		 * Defines a data source type for the ordering which can be used to read
		 * real-time information from the table (updating the internally cached
		 * version) prior to ordering. This allows ordering to occur on user
		 * editable elements such as form inputs.
		 *  @type string
		 *  @default std
		 *
		 *  @name DataTable.defaults.column.orderDataType
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "orderDataType": "dom-text", "targets": [ 2, 3 ] },
		 *          { "type": "numeric", "targets": [ 3 ] },
		 *          { "orderDataType": "dom-select", "targets": [ 4 ] },
		 *          { "orderDataType": "dom-checkbox", "targets": [ 5 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          null,
		 *          null,
		 *          { "orderDataType": "dom-text" },
		 *          { "orderDataType": "dom-text", "type": "numeric" },
		 *          { "orderDataType": "dom-select" },
		 *          { "orderDataType": "dom-checkbox" }
		 *        ]
		 *      } );
		 *    } );
		 */
		"sSortDataType": "std",
	
	
		/**
		 * The title of this column.
		 *  @type string
		 *  @default null <i>Derived from the 'TH' value for this column in the
		 *    original HTML table.</i>
		 *
		 *  @name DataTable.defaults.column.title
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "title": "My column title", "targets": [ 0 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "title": "My column title" },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ]
		 *      } );
		 *    } );
		 */
		"sTitle": null,
	
	
		/**
		 * The type allows you to specify how the data for this column will be
		 * ordered. Four types (string, numeric, date and html (which will strip
		 * HTML tags before ordering)) are currently available. Note that only date
		 * formats understood by Javascript's Date() object will be accepted as type
		 * date. For example: "Mar 26, 2008 5:03 PM". May take the values: 'string',
		 * 'numeric', 'date' or 'html' (by default). Further types can be adding
		 * through plug-ins.
		 *  @type string
		 *  @default null <i>Auto-detected from raw data</i>
		 *
		 *  @name DataTable.defaults.column.type
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "type": "html", "targets": [ 0 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "type": "html" },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ]
		 *      } );
		 *    } );
		 */
		"sType": null,
	
	
		/**
		 * Defining the width of the column, this parameter may take any CSS value
		 * (3em, 20px etc). DataTables applies 'smart' widths to columns which have not
		 * been given a specific width through this interface ensuring that the table
		 * remains readable.
		 *  @type string
		 *  @default null <i>Automatic</i>
		 *
		 *  @name DataTable.defaults.column.width
		 *  @dtopt Columns
		 *
		 *  @example
		 *    // Using `columnDefs`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columnDefs": [
		 *          { "width": "20%", "targets": [ 0 ] }
		 *        ]
		 *      } );
		 *    } );
		 *
		 *  @example
		 *    // Using `columns`
		 *    $(document).ready( function() {
		 *      $('#example').dataTable( {
		 *        "columns": [
		 *          { "width": "20%" },
		 *          null,
		 *          null,
		 *          null,
		 *          null
		 *        ]
		 *      } );
		 *    } );
		 */
		"sWidth": null
	};
	
	_fnHungarianMap( DataTable.defaults.column );
	
	
	
	/**
	 * DataTables settings object - this holds all the information needed for a
	 * given table, including configuration, data and current application of the
	 * table options. DataTables does not have a single instance for each DataTable
	 * with the settings attached to that instance, but rather instances of the
	 * DataTable "class" are created on-the-fly as needed (typically by a
	 * $().dataTable() call) and the settings object is then applied to that
	 * instance.
	 *
	 * Note that this object is related to {@link DataTable.defaults} but this
	 * one is the internal data store for DataTables's cache of columns. It should
	 * NOT be manipulated outside of DataTables. Any configuration should be done
	 * through the initialisation options.
	 *  @namespace
	 *  @todo Really should attach the settings object to individual instances so we
	 *    don't need to create new instances on each $().dataTable() call (if the
	 *    table already exists). It would also save passing oSettings around and
	 *    into every single function. However, this is a very significant
	 *    architecture change for DataTables and will almost certainly break
	 *    backwards compatibility with older installations. This is something that
	 *    will be done in 2.0.
	 */
	DataTable.models.oSettings = {
		/**
		 * Primary features of DataTables and their enablement state.
		 *  @namespace
		 */
		"oFeatures": {
	
			/**
			 * Flag to say if DataTables should automatically try to calculate the
			 * optimum table and columns widths (true) or not (false).
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bAutoWidth": null,
	
			/**
			 * Delay the creation of TR and TD elements until they are actually
			 * needed by a driven page draw. This can give a significant speed
			 * increase for Ajax source and Javascript source data, but makes no
			 * difference at all fro DOM and server-side processing tables.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bDeferRender": null,
	
			/**
			 * Enable filtering on the table or not. Note that if this is disabled
			 * then there is no filtering at all on the table, including fnFilter.
			 * To just remove the filtering input use sDom and remove the 'f' option.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bFilter": null,
	
			/**
			 * Table information element (the 'Showing x of y records' div) enable
			 * flag.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bInfo": null,
	
			/**
			 * Present a user control allowing the end user to change the page size
			 * when pagination is enabled.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bLengthChange": null,
	
			/**
			 * Pagination enabled or not. Note that if this is disabled then length
			 * changing must also be disabled.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bPaginate": null,
	
			/**
			 * Processing indicator enable flag whenever DataTables is enacting a
			 * user request - typically an Ajax request for server-side processing.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bProcessing": null,
	
			/**
			 * Server-side processing enabled flag - when enabled DataTables will
			 * get all data from the server for every draw - there is no filtering,
			 * sorting or paging done on the client-side.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bServerSide": null,
	
			/**
			 * Sorting enablement flag.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bSort": null,
	
			/**
			 * Multi-column sorting
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bSortMulti": null,
	
			/**
			 * Apply a class to the columns which are being sorted to provide a
			 * visual highlight or not. This can slow things down when enabled since
			 * there is a lot of DOM interaction.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bSortClasses": null,
	
			/**
			 * State saving enablement flag.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bStateSave": null
		},
	
	
		/**
		 * Scrolling settings for a table.
		 *  @namespace
		 */
		"oScroll": {
			/**
			 * When the table is shorter in height than sScrollY, collapse the
			 * table container down to the height of the table (when true).
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type boolean
			 */
			"bCollapse": null,
	
			/**
			 * Width of the scrollbar for the web-browser's platform. Calculated
			 * during table initialisation.
			 *  @type int
			 *  @default 0
			 */
			"iBarWidth": 0,
	
			/**
			 * Viewport width for horizontal scrolling. Horizontal scrolling is
			 * disabled if an empty string.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type string
			 */
			"sX": null,
	
			/**
			 * Width to expand the table to when using x-scrolling. Typically you
			 * should not need to use this.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type string
			 *  @deprecated
			 */
			"sXInner": null,
	
			/**
			 * Viewport height for vertical scrolling. Vertical scrolling is disabled
			 * if an empty string.
			 * Note that this parameter will be set by the initialisation routine. To
			 * set a default use {@link DataTable.defaults}.
			 *  @type string
			 */
			"sY": null
		},
	
		/**
		 * Language information for the table.
		 *  @namespace
		 *  @extends DataTable.defaults.oLanguage
		 */
		"oLanguage": {
			/**
			 * Information callback function. See
			 * {@link DataTable.defaults.fnInfoCallback}
			 *  @type function
			 *  @default null
			 */
			"fnInfoCallback": null
		},
	
		/**
		 * Browser support parameters
		 *  @namespace
		 */
		"oBrowser": {
			/**
			 * Indicate if the browser incorrectly calculates width:100% inside a
			 * scrolling element (IE6/7)
			 *  @type boolean
			 *  @default false
			 */
			"bScrollOversize": false,
	
			/**
			 * Determine if the vertical scrollbar is on the right or left of the
			 * scrolling container - needed for rtl language layout, although not
			 * all browsers move the scrollbar (Safari).
			 *  @type boolean
			 *  @default false
			 */
			"bScrollbarLeft": false,
	
			/**
			 * Flag for if `getBoundingClientRect` is fully supported or not
			 *  @type boolean
			 *  @default false
			 */
			"bBounding": false,
	
			/**
			 * Browser scrollbar width
			 *  @type integer
			 *  @default 0
			 */
			"barWidth": 0
		},
	
	
		"ajax": null,
	
	
		/**
		 * Array referencing the nodes which are used for the features. The
		 * parameters of this object match what is allowed by sDom - i.e.
		 *   <ul>
		 *     <li>'l' - Length changing</li>
		 *     <li>'f' - Filtering input</li>
		 *     <li>'t' - The table!</li>
		 *     <li>'i' - Information</li>
		 *     <li>'p' - Pagination</li>
		 *     <li>'r' - pRocessing</li>
		 *   </ul>
		 *  @type array
		 *  @default []
		 */
		"aanFeatures": [],
	
		/**
		 * Store data information - see {@link DataTable.models.oRow} for detailed
		 * information.
		 *  @type array
		 *  @default []
		 */
		"aoData": [],
	
		/**
		 * Array of indexes which are in the current display (after filtering etc)
		 *  @type array
		 *  @default []
		 */
		"aiDisplay": [],
	
		/**
		 * Array of indexes for display - no filtering
		 *  @type array
		 *  @default []
		 */
		"aiDisplayMaster": [],
	
		/**
		 * Map of row ids to data indexes
		 *  @type object
		 *  @default {}
		 */
		"aIds": {},
	
		/**
		 * Store information about each column that is in use
		 *  @type array
		 *  @default []
		 */
		"aoColumns": [],
	
		/**
		 * Store information about the table's header
		 *  @type array
		 *  @default []
		 */
		"aoHeader": [],
	
		/**
		 * Store information about the table's footer
		 *  @type array
		 *  @default []
		 */
		"aoFooter": [],
	
		/**
		 * Store the applied global search information in case we want to force a
		 * research or compare the old search to a new one.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @namespace
		 *  @extends DataTable.models.oSearch
		 */
		"oPreviousSearch": {},
	
		/**
		 * Store the applied search for each column - see
		 * {@link DataTable.models.oSearch} for the format that is used for the
		 * filtering information for each column.
		 *  @type array
		 *  @default []
		 */
		"aoPreSearchCols": [],
	
		/**
		 * Sorting that is applied to the table. Note that the inner arrays are
		 * used in the following manner:
		 * <ul>
		 *   <li>Index 0 - column number</li>
		 *   <li>Index 1 - current sorting direction</li>
		 * </ul>
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type array
		 *  @todo These inner arrays should really be objects
		 */
		"aaSorting": null,
	
		/**
		 * Sorting that is always applied to the table (i.e. prefixed in front of
		 * aaSorting).
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type array
		 *  @default []
		 */
		"aaSortingFixed": [],
	
		/**
		 * Classes to use for the striping of a table.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type array
		 *  @default []
		 */
		"asStripeClasses": null,
	
		/**
		 * If restoring a table - we should restore its striping classes as well
		 *  @type array
		 *  @default []
		 */
		"asDestroyStripes": [],
	
		/**
		 * If restoring a table - we should restore its width
		 *  @type int
		 *  @default 0
		 */
		"sDestroyWidth": 0,
	
		/**
		 * Callback functions array for every time a row is inserted (i.e. on a draw).
		 *  @type array
		 *  @default []
		 */
		"aoRowCallback": [],
	
		/**
		 * Callback functions for the header on each draw.
		 *  @type array
		 *  @default []
		 */
		"aoHeaderCallback": [],
	
		/**
		 * Callback function for the footer on each draw.
		 *  @type array
		 *  @default []
		 */
		"aoFooterCallback": [],
	
		/**
		 * Array of callback functions for draw callback functions
		 *  @type array
		 *  @default []
		 */
		"aoDrawCallback": [],
	
		/**
		 * Array of callback functions for row created function
		 *  @type array
		 *  @default []
		 */
		"aoRowCreatedCallback": [],
	
		/**
		 * Callback functions for just before the table is redrawn. A return of
		 * false will be used to cancel the draw.
		 *  @type array
		 *  @default []
		 */
		"aoPreDrawCallback": [],
	
		/**
		 * Callback functions for when the table has been initialised.
		 *  @type array
		 *  @default []
		 */
		"aoInitComplete": [],
	
	
		/**
		 * Callbacks for modifying the settings to be stored for state saving, prior to
		 * saving state.
		 *  @type array
		 *  @default []
		 */
		"aoStateSaveParams": [],
	
		/**
		 * Callbacks for modifying the settings that have been stored for state saving
		 * prior to using the stored values to restore the state.
		 *  @type array
		 *  @default []
		 */
		"aoStateLoadParams": [],
	
		/**
		 * Callbacks for operating on the settings object once the saved state has been
		 * loaded
		 *  @type array
		 *  @default []
		 */
		"aoStateLoaded": [],
	
		/**
		 * Cache the table ID for quick access
		 *  @type string
		 *  @default <i>Empty string</i>
		 */
		"sTableId": "",
	
		/**
		 * The TABLE node for the main table
		 *  @type node
		 *  @default null
		 */
		"nTable": null,
	
		/**
		 * Permanent ref to the thead element
		 *  @type node
		 *  @default null
		 */
		"nTHead": null,
	
		/**
		 * Permanent ref to the tfoot element - if it exists
		 *  @type node
		 *  @default null
		 */
		"nTFoot": null,
	
		/**
		 * Permanent ref to the tbody element
		 *  @type node
		 *  @default null
		 */
		"nTBody": null,
	
		/**
		 * Cache the wrapper node (contains all DataTables controlled elements)
		 *  @type node
		 *  @default null
		 */
		"nTableWrapper": null,
	
		/**
		 * Indicate if when using server-side processing the loading of data
		 * should be deferred until the second draw.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type boolean
		 *  @default false
		 */
		"bDeferLoading": false,
	
		/**
		 * Indicate if all required information has been read in
		 *  @type boolean
		 *  @default false
		 */
		"bInitialised": false,
	
		/**
		 * Information about open rows. Each object in the array has the parameters
		 * 'nTr' and 'nParent'
		 *  @type array
		 *  @default []
		 */
		"aoOpenRows": [],
	
		/**
		 * Dictate the positioning of DataTables' control elements - see
		 * {@link DataTable.model.oInit.sDom}.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type string
		 *  @default null
		 */
		"sDom": null,
	
		/**
		 * Search delay (in mS)
		 *  @type integer
		 *  @default null
		 */
		"searchDelay": null,
	
		/**
		 * Which type of pagination should be used.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type string
		 *  @default two_button
		 */
		"sPaginationType": "two_button",
	
		/**
		 * The state duration (for `stateSave`) in seconds.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type int
		 *  @default 0
		 */
		"iStateDuration": 0,
	
		/**
		 * Array of callback functions for state saving. Each array element is an
		 * object with the following parameters:
		 *   <ul>
		 *     <li>function:fn - function to call. Takes two parameters, oSettings
		 *       and the JSON string to save that has been thus far created. Returns
		 *       a JSON string to be inserted into a json object
		 *       (i.e. '"param": [ 0, 1, 2]')</li>
		 *     <li>string:sName - name of callback</li>
		 *   </ul>
		 *  @type array
		 *  @default []
		 */
		"aoStateSave": [],
	
		/**
		 * Array of callback functions for state loading. Each array element is an
		 * object with the following parameters:
		 *   <ul>
		 *     <li>function:fn - function to call. Takes two parameters, oSettings
		 *       and the object stored. May return false to cancel state loading</li>
		 *     <li>string:sName - name of callback</li>
		 *   </ul>
		 *  @type array
		 *  @default []
		 */
		"aoStateLoad": [],
	
		/**
		 * State that was saved. Useful for back reference
		 *  @type object
		 *  @default null
		 */
		"oSavedState": null,
	
		/**
		 * State that was loaded. Useful for back reference
		 *  @type object
		 *  @default null
		 */
		"oLoadedState": null,
	
		/**
		 * Source url for AJAX data for the table.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type string
		 *  @default null
		 */
		"sAjaxSource": null,
	
		/**
		 * Property from a given object from which to read the table data from. This
		 * can be an empty string (when not server-side processing), in which case
		 * it is  assumed an an array is given directly.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type string
		 */
		"sAjaxDataProp": null,
	
		/**
		 * Note if draw should be blocked while getting data
		 *  @type boolean
		 *  @default true
		 */
		"bAjaxDataGet": true,
	
		/**
		 * The last jQuery XHR object that was used for server-side data gathering.
		 * This can be used for working with the XHR information in one of the
		 * callbacks
		 *  @type object
		 *  @default null
		 */
		"jqXHR": null,
	
		/**
		 * JSON returned from the server in the last Ajax request
		 *  @type object
		 *  @default undefined
		 */
		"json": undefined,
	
		/**
		 * Data submitted as part of the last Ajax request
		 *  @type object
		 *  @default undefined
		 */
		"oAjaxData": undefined,
	
		/**
		 * Function to get the server-side data.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type function
		 */
		"fnServerData": null,
	
		/**
		 * Functions which are called prior to sending an Ajax request so extra
		 * parameters can easily be sent to the server
		 *  @type array
		 *  @default []
		 */
		"aoServerParams": [],
	
		/**
		 * Send the XHR HTTP method - GET or POST (could be PUT or DELETE if
		 * required).
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type string
		 */
		"sServerMethod": null,
	
		/**
		 * Format numbers for display.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type function
		 */
		"fnFormatNumber": null,
	
		/**
		 * List of options that can be used for the user selectable length menu.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type array
		 *  @default []
		 */
		"aLengthMenu": null,
	
		/**
		 * Counter for the draws that the table does. Also used as a tracker for
		 * server-side processing
		 *  @type int
		 *  @default 0
		 */
		"iDraw": 0,
	
		/**
		 * Indicate if a redraw is being done - useful for Ajax
		 *  @type boolean
		 *  @default false
		 */
		"bDrawing": false,
	
		/**
		 * Draw index (iDraw) of the last error when parsing the returned data
		 *  @type int
		 *  @default -1
		 */
		"iDrawError": -1,
	
		/**
		 * Paging display length
		 *  @type int
		 *  @default 10
		 */
		"_iDisplayLength": 10,
	
		/**
		 * Paging start point - aiDisplay index
		 *  @type int
		 *  @default 0
		 */
		"_iDisplayStart": 0,
	
		/**
		 * Server-side processing - number of records in the result set
		 * (i.e. before filtering), Use fnRecordsTotal rather than
		 * this property to get the value of the number of records, regardless of
		 * the server-side processing setting.
		 *  @type int
		 *  @default 0
		 *  @private
		 */
		"_iRecordsTotal": 0,
	
		/**
		 * Server-side processing - number of records in the current display set
		 * (i.e. after filtering). Use fnRecordsDisplay rather than
		 * this property to get the value of the number of records, regardless of
		 * the server-side processing setting.
		 *  @type boolean
		 *  @default 0
		 *  @private
		 */
		"_iRecordsDisplay": 0,
	
		/**
		 * The classes to use for the table
		 *  @type object
		 *  @default {}
		 */
		"oClasses": {},
	
		/**
		 * Flag attached to the settings object so you can check in the draw
		 * callback if filtering has been done in the draw. Deprecated in favour of
		 * events.
		 *  @type boolean
		 *  @default false
		 *  @deprecated
		 */
		"bFiltered": false,
	
		/**
		 * Flag attached to the settings object so you can check in the draw
		 * callback if sorting has been done in the draw. Deprecated in favour of
		 * events.
		 *  @type boolean
		 *  @default false
		 *  @deprecated
		 */
		"bSorted": false,
	
		/**
		 * Indicate that if multiple rows are in the header and there is more than
		 * one unique cell per column, if the top one (true) or bottom one (false)
		 * should be used for sorting / title by DataTables.
		 * Note that this parameter will be set by the initialisation routine. To
		 * set a default use {@link DataTable.defaults}.
		 *  @type boolean
		 */
		"bSortCellsTop": null,
	
		/**
		 * Initialisation object that is used for the table
		 *  @type object
		 *  @default null
		 */
		"oInit": null,
	
		/**
		 * Destroy callback functions - for plug-ins to attach themselves to the
		 * destroy so they can clean up markup and events.
		 *  @type array
		 *  @default []
		 */
		"aoDestroyCallback": [],
	
	
		/**
		 * Get the number of records in the current record set, before filtering
		 *  @type function
		 */
		"fnRecordsTotal": function ()
		{
			return _fnDataSource( this ) == 'ssp' ?
				this._iRecordsTotal * 1 :
				this.aiDisplayMaster.length;
		},
	
		/**
		 * Get the number of records in the current record set, after filtering
		 *  @type function
		 */
		"fnRecordsDisplay": function ()
		{
			return _fnDataSource( this ) == 'ssp' ?
				this._iRecordsDisplay * 1 :
				this.aiDisplay.length;
		},
	
		/**
		 * Get the display end point - aiDisplay index
		 *  @type function
		 */
		"fnDisplayEnd": function ()
		{
			var
				len      = this._iDisplayLength,
				start    = this._iDisplayStart,
				calc     = start + len,
				records  = this.aiDisplay.length,
				features = this.oFeatures,
				paginate = features.bPaginate;
	
			if ( features.bServerSide ) {
				return paginate === false || len === -1 ?
					start + records :
					Math.min( start+len, this._iRecordsDisplay );
			}
			else {
				return ! paginate || calc>records || len===-1 ?
					records :
					calc;
			}
		},
	
		/**
		 * The DataTables object for this table
		 *  @type object
		 *  @default null
		 */
		"oInstance": null,
	
		/**
		 * Unique identifier for each instance of the DataTables object. If there
		 * is an ID on the table node, then it takes that value, otherwise an
		 * incrementing internal counter is used.
		 *  @type string
		 *  @default null
		 */
		"sInstance": null,
	
		/**
		 * tabindex attribute value that is added to DataTables control elements, allowing
		 * keyboard navigation of the table and its controls.
		 */
		"iTabIndex": 0,
	
		/**
		 * DIV container for the footer scrolling table if scrolling
		 */
		"nScrollHead": null,
	
		/**
		 * DIV container for the footer scrolling table if scrolling
		 */
		"nScrollFoot": null,
	
		/**
		 * Last applied sort
		 *  @type array
		 *  @default []
		 */
		"aLastSort": [],
	
		/**
		 * Stored plug-in instances
		 *  @type object
		 *  @default {}
		 */
		"oPlugins": {},
	
		/**
		 * Function used to get a row's id from the row's data
		 *  @type function
		 *  @default null
		 */
		"rowIdFn": null,
	
		/**
		 * Data location where to store a row's id
		 *  @type string
		 *  @default null
		 */
		"rowId": null
	};

	/**
	 * Extension object for DataTables that is used to provide all extension
	 * options.
	 *
	 * Note that the `DataTable.ext` object is available through
	 * `jQuery.fn.dataTable.ext` where it may be accessed and manipulated. It is
	 * also aliased to `jQuery.fn.dataTableExt` for historic reasons.
	 *  @namespace
	 *  @extends DataTable.models.ext
	 */
	
	
	/**
	 * DataTables extensions
	 * 
	 * This namespace acts as a collection area for plug-ins that can be used to
	 * extend DataTables capabilities. Indeed many of the build in methods
	 * use this method to provide their own capabilities (sorting methods for
	 * example).
	 *
	 * Note that this namespace is aliased to `jQuery.fn.dataTableExt` for legacy
	 * reasons
	 *
	 *  @namespace
	 */
	DataTable.ext = _ext = {
		/**
		 * Buttons. For use with the Buttons extension for DataTables. This is
		 * defined here so other extensions can define buttons regardless of load
		 * order. It is _not_ used by DataTables core.
		 *
		 *  @type object
		 *  @default {}
		 */
		buttons: {},
	
	
		/**
		 * Element class names
		 *
		 *  @type object
		 *  @default {}
		 */
		classes: {},
	
	
		/**
		 * DataTables build type (expanded by the download builder)
		 *
		 *  @type string
		 */
		builder: "-source-",
	
	
		/**
		 * Error reporting.
		 * 
		 * How should DataTables report an error. Can take the value 'alert',
		 * 'throw', 'none' or a function.
		 *
		 *  @type string|function
		 *  @default alert
		 */
		errMode: "alert",
	
	
		/**
		 * Feature plug-ins.
		 * 
		 * This is an array of objects which describe the feature plug-ins that are
		 * available to DataTables. These feature plug-ins are then available for
		 * use through the `dom` initialisation option.
		 * 
		 * Each feature plug-in is described by an object which must have the
		 * following properties:
		 * 
		 * * `fnInit` - function that is used to initialise the plug-in,
		 * * `cFeature` - a character so the feature can be enabled by the `dom`
		 *   instillation option. This is case sensitive.
		 *
		 * The `fnInit` function has the following input parameters:
		 *
		 * 1. `{object}` DataTables settings object: see
		 *    {@link DataTable.models.oSettings}
		 *
		 * And the following return is expected:
		 * 
		 * * {node|null} The element which contains your feature. Note that the
		 *   return may also be void if your plug-in does not require to inject any
		 *   DOM elements into DataTables control (`dom`) - for example this might
		 *   be useful when developing a plug-in which allows table control via
		 *   keyboard entry
		 *
		 *  @type array
		 *
		 *  @example
		 *    $.fn.dataTable.ext.features.push( {
		 *      "fnInit": function( oSettings ) {
		 *        return new TableTools( { "oDTSettings": oSettings } );
		 *      },
		 *      "cFeature": "T"
		 *    } );
		 */
		feature: [],
	
	
		/**
		 * Row searching.
		 * 
		 * This method of searching is complimentary to the default type based
		 * searching, and a lot more comprehensive as it allows you complete control
		 * over the searching logic. Each element in this array is a function
		 * (parameters described below) that is called for every row in the table,
		 * and your logic decides if it should be included in the searching data set
		 * or not.
		 *
		 * Searching functions have the following input parameters:
		 *
		 * 1. `{object}` DataTables settings object: see
		 *    {@link DataTable.models.oSettings}
		 * 2. `{array|object}` Data for the row to be processed (same as the
		 *    original format that was passed in as the data source, or an array
		 *    from a DOM data source
		 * 3. `{int}` Row index ({@link DataTable.models.oSettings.aoData}), which
		 *    can be useful to retrieve the `TR` element if you need DOM interaction.
		 *
		 * And the following return is expected:
		 *
		 * * {boolean} Include the row in the searched result set (true) or not
		 *   (false)
		 *
		 * Note that as with the main search ability in DataTables, technically this
		 * is "filtering", since it is subtractive. However, for consistency in
		 * naming we call it searching here.
		 *
		 *  @type array
		 *  @default []
		 *
		 *  @example
		 *    // The following example shows custom search being applied to the
		 *    // fourth column (i.e. the data[3] index) based on two input values
		 *    // from the end-user, matching the data in a certain range.
		 *    $.fn.dataTable.ext.search.push(
		 *      function( settings, data, dataIndex ) {
		 *        var min = document.getElementById('min').value * 1;
		 *        var max = document.getElementById('max').value * 1;
		 *        var version = data[3] == "-" ? 0 : data[3]*1;
		 *
		 *        if ( min == "" && max == "" ) {
		 *          return true;
		 *        }
		 *        else if ( min == "" && version < max ) {
		 *          return true;
		 *        }
		 *        else if ( min < version && "" == max ) {
		 *          return true;
		 *        }
		 *        else if ( min < version && version < max ) {
		 *          return true;
		 *        }
		 *        return false;
		 *      }
		 *    );
		 */
		search: [],
	
	
		/**
		 * Selector extensions
		 *
		 * The `selector` option can be used to extend the options available for the
		 * selector modifier options (`selector-modifier` object data type) that
		 * each of the three built in selector types offer (row, column and cell +
		 * their plural counterparts). For example the Select extension uses this
		 * mechanism to provide an option to select only rows, columns and cells
		 * that have been marked as selected by the end user (`{selected: true}`),
		 * which can be used in conjunction with the existing built in selector
		 * options.
		 *
		 * Each property is an array to which functions can be pushed. The functions
		 * take three attributes:
		 *
		 * * Settings object for the host table
		 * * Options object (`selector-modifier` object type)
		 * * Array of selected item indexes
		 *
		 * The return is an array of the resulting item indexes after the custom
		 * selector has been applied.
		 *
		 *  @type object
		 */
		selector: {
			cell: [],
			column: [],
			row: []
		},
	
	
		/**
		 * Internal functions, exposed for used in plug-ins.
		 * 
		 * Please note that you should not need to use the internal methods for
		 * anything other than a plug-in (and even then, try to avoid if possible).
		 * The internal function may change between releases.
		 *
		 *  @type object
		 *  @default {}
		 */
		internal: {},
	
	
		/**
		 * Legacy configuration options. Enable and disable legacy options that
		 * are available in DataTables.
		 *
		 *  @type object
		 */
		legacy: {
			/**
			 * Enable / disable DataTables 1.9 compatible server-side processing
			 * requests
			 *
			 *  @type boolean
			 *  @default null
			 */
			ajax: null
		},
	
	
		/**
		 * Pagination plug-in methods.
		 * 
		 * Each entry in this object is a function and defines which buttons should
		 * be shown by the pagination rendering method that is used for the table:
		 * {@link DataTable.ext.renderer.pageButton}. The renderer addresses how the
		 * buttons are displayed in the document, while the functions here tell it
		 * what buttons to display. This is done by returning an array of button
		 * descriptions (what each button will do).
		 *
		 * Pagination types (the four built in options and any additional plug-in
		 * options defined here) can be used through the `paginationType`
		 * initialisation parameter.
		 *
		 * The functions defined take two parameters:
		 *
		 * 1. `{int} page` The current page index
		 * 2. `{int} pages` The number of pages in the table
		 *
		 * Each function is expected to return an array where each element of the
		 * array can be one of:
		 *
		 * * `first` - Jump to first page when activated
		 * * `last` - Jump to last page when activated
		 * * `previous` - Show previous page when activated
		 * * `next` - Show next page when activated
		 * * `{int}` - Show page of the index given
		 * * `{array}` - A nested array containing the above elements to add a
		 *   containing 'DIV' element (might be useful for styling).
		 *
		 * Note that DataTables v1.9- used this object slightly differently whereby
		 * an object with two functions would be defined for each plug-in. That
		 * ability is still supported by DataTables 1.10+ to provide backwards
		 * compatibility, but this option of use is now decremented and no longer
		 * documented in DataTables 1.10+.
		 *
		 *  @type object
		 *  @default {}
		 *
		 *  @example
		 *    // Show previous, next and current page buttons only
		 *    $.fn.dataTableExt.oPagination.current = function ( page, pages ) {
		 *      return [ 'previous', page, 'next' ];
		 *    };
		 */
		pager: {},
	
	
		renderer: {
			pageButton: {},
			header: {}
		},
	
	
		/**
		 * Ordering plug-ins - custom data source
		 * 
		 * The extension options for ordering of data available here is complimentary
		 * to the default type based ordering that DataTables typically uses. It
		 * allows much greater control over the the data that is being used to
		 * order a column, but is necessarily therefore more complex.
		 * 
		 * This type of ordering is useful if you want to do ordering based on data
		 * live from the DOM (for example the contents of an 'input' element) rather
		 * than just the static string that DataTables knows of.
		 * 
		 * The way these plug-ins work is that you create an array of the values you
		 * wish to be ordering for the column in question and then return that
		 * array. The data in the array much be in the index order of the rows in
		 * the table (not the currently ordering order!). Which order data gathering
		 * function is run here depends on the `dt-init columns.orderDataType`
		 * parameter that is used for the column (if any).
		 *
		 * The functions defined take two parameters:
		 *
		 * 1. `{object}` DataTables settings object: see
		 *    {@link DataTable.models.oSettings}
		 * 2. `{int}` Target column index
		 *
		 * Each function is expected to return an array:
		 *
		 * * `{array}` Data for the column to be ordering upon
		 *
		 *  @type array
		 *
		 *  @example
		 *    // Ordering using `input` node values
		 *    $.fn.dataTable.ext.order['dom-text'] = function  ( settings, col )
		 *    {
		 *      return this.api().column( col, {order:'index'} ).nodes().map( function ( td, i ) {
		 *        return $('input', td).val();
		 *      } );
		 *    }
		 */
		order: {},
	
	
		/**
		 * Type based plug-ins.
		 *
		 * Each column in DataTables has a type assigned to it, either by automatic
		 * detection or by direct assignment using the `type` option for the column.
		 * The type of a column will effect how it is ordering and search (plug-ins
		 * can also make use of the column type if required).
		 *
		 * @namespace
		 */
		type: {
			/**
			 * Type detection functions.
			 *
			 * The functions defined in this object are used to automatically detect
			 * a column's type, making initialisation of DataTables super easy, even
			 * when complex data is in the table.
			 *
			 * The functions defined take two parameters:
			 *
		     *  1. `{*}` Data from the column cell to be analysed
		     *  2. `{settings}` DataTables settings object. This can be used to
		     *     perform context specific type detection - for example detection
		     *     based on language settings such as using a comma for a decimal
		     *     place. Generally speaking the options from the settings will not
		     *     be required
			 *
			 * Each function is expected to return:
			 *
			 * * `{string|null}` Data type detected, or null if unknown (and thus
			 *   pass it on to the other type detection functions.
			 *
			 *  @type array
			 *
			 *  @example
			 *    // Currency type detection plug-in:
			 *    $.fn.dataTable.ext.type.detect.push(
			 *      function ( data, settings ) {
			 *        // Check the numeric part
			 *        if ( ! data.substring(1).match(/[0-9]/) ) {
			 *          return null;
			 *        }
			 *
			 *        // Check prefixed by currency
			 *        if ( data.charAt(0) == '$' || data.charAt(0) == '&pound;' ) {
			 *          return 'currency';
			 *        }
			 *        return null;
			 *      }
			 *    );
			 */
			detect: [],
	
	
			/**
			 * Type based search formatting.
			 *
			 * The type based searching functions can be used to pre-format the
			 * data to be search on. For example, it can be used to strip HTML
			 * tags or to de-format telephone numbers for numeric only searching.
			 *
			 * Note that is a search is not defined for a column of a given type,
			 * no search formatting will be performed.
			 * 
			 * Pre-processing of searching data plug-ins - When you assign the sType
			 * for a column (or have it automatically detected for you by DataTables
			 * or a type detection plug-in), you will typically be using this for
			 * custom sorting, but it can also be used to provide custom searching
			 * by allowing you to pre-processing the data and returning the data in
			 * the format that should be searched upon. This is done by adding
			 * functions this object with a parameter name which matches the sType
			 * for that target column. This is the corollary of <i>afnSortData</i>
			 * for searching data.
			 *
			 * The functions defined take a single parameter:
			 *
		     *  1. `{*}` Data from the column cell to be prepared for searching
			 *
			 * Each function is expected to return:
			 *
			 * * `{string|null}` Formatted string that will be used for the searching.
			 *
			 *  @type object
			 *  @default {}
			 *
			 *  @example
			 *    $.fn.dataTable.ext.type.search['title-numeric'] = function ( d ) {
			 *      return d.replace(/\n/g," ").replace( /<.*?>/g, "" );
			 *    }
			 */
			search: {},
	
	
			/**
			 * Type based ordering.
			 *
			 * The column type tells DataTables what ordering to apply to the table
			 * when a column is sorted upon. The order for each type that is defined,
			 * is defined by the functions available in this object.
			 *
			 * Each ordering option can be described by three properties added to
			 * this object:
			 *
			 * * `{type}-pre` - Pre-formatting function
			 * * `{type}-asc` - Ascending order function
			 * * `{type}-desc` - Descending order function
			 *
			 * All three can be used together, only `{type}-pre` or only
			 * `{type}-asc` and `{type}-desc` together. It is generally recommended
			 * that only `{type}-pre` is used, as this provides the optimal
			 * implementation in terms of speed, although the others are provided
			 * for compatibility with existing Javascript sort functions.
			 *
			 * `{type}-pre`: Functions defined take a single parameter:
			 *
		     *  1. `{*}` Data from the column cell to be prepared for ordering
			 *
			 * And return:
			 *
			 * * `{*}` Data to be sorted upon
			 *
			 * `{type}-asc` and `{type}-desc`: Functions are typical Javascript sort
			 * functions, taking two parameters:
			 *
		     *  1. `{*}` Data to compare to the second parameter
		     *  2. `{*}` Data to compare to the first parameter
			 *
			 * And returning:
			 *
			 * * `{*}` Ordering match: <0 if first parameter should be sorted lower
			 *   than the second parameter, ===0 if the two parameters are equal and
			 *   >0 if the first parameter should be sorted height than the second
			 *   parameter.
			 * 
			 *  @type object
			 *  @default {}
			 *
			 *  @example
			 *    // Numeric ordering of formatted numbers with a pre-formatter
			 *    $.extend( $.fn.dataTable.ext.type.order, {
			 *      "string-pre": function(x) {
			 *        a = (a === "-" || a === "") ? 0 : a.replace( /[^\d\-\.]/g, "" );
			 *        return parseFloat( a );
			 *      }
			 *    } );
			 *
			 *  @example
			 *    // Case-sensitive string ordering, with no pre-formatting method
			 *    $.extend( $.fn.dataTable.ext.order, {
			 *      "string-case-asc": function(x,y) {
			 *        return ((x < y) ? -1 : ((x > y) ? 1 : 0));
			 *      },
			 *      "string-case-desc": function(x,y) {
			 *        return ((x < y) ? 1 : ((x > y) ? -1 : 0));
			 *      }
			 *    } );
			 */
			order: {}
		},
	
		/**
		 * Unique DataTables instance counter
		 *
		 * @type int
		 * @private
		 */
		_unique: 0,
	
	
		//
		// Depreciated
		// The following properties are retained for backwards compatiblity only.
		// The should not be used in new projects and will be removed in a future
		// version
		//
	
		/**
		 * Version check function.
		 *  @type function
		 *  @depreciated Since 1.10
		 */
		fnVersionCheck: DataTable.fnVersionCheck,
	
	
		/**
		 * Index for what 'this' index API functions should use
		 *  @type int
		 *  @deprecated Since v1.10
		 */
		iApiIndex: 0,
	
	
		/**
		 * jQuery UI class container
		 *  @type object
		 *  @deprecated Since v1.10
		 */
		oJUIClasses: {},
	
	
		/**
		 * Software version
		 *  @type string
		 *  @deprecated Since v1.10
		 */
		sVersion: DataTable.version
	};
	
	
	//
	// Backwards compatibility. Alias to pre 1.10 Hungarian notation counter parts
	//
	$.extend( _ext, {
		afnFiltering: _ext.search,
		aTypes:       _ext.type.detect,
		ofnSearch:    _ext.type.search,
		oSort:        _ext.type.order,
		afnSortData:  _ext.order,
		aoFeatures:   _ext.feature,
		oApi:         _ext.internal,
		oStdClasses:  _ext.classes,
		oPagination:  _ext.pager
	} );
	
	
	$.extend( DataTable.ext.classes, {
		"sTable": "dataTable",
		"sNoFooter": "no-footer",
	
		/* Paging buttons */
		"sPageButton": "paginate_button",
		"sPageButtonActive": "current",
		"sPageButtonDisabled": "disabled",
	
		/* Striping classes */
		"sStripeOdd": "odd",
		"sStripeEven": "even",
	
		/* Empty row */
		"sRowEmpty": "dataTables_empty",
	
		/* Features */
		"sWrapper": "dataTables_wrapper",
		"sFilter": "dataTables_filter",
		"sInfo": "dataTables_info",
		"sPaging": "dataTables_paginate paging_", /* Note that the type is postfixed */
		"sLength": "dataTables_length",
		"sProcessing": "dataTables_processing",
	
		/* Sorting */
		"sSortAsc": "sorting_asc",
		"sSortDesc": "sorting_desc",
		"sSortable": "sorting", /* Sortable in both directions */
		"sSortableAsc": "sorting_desc_disabled",
		"sSortableDesc": "sorting_asc_disabled",
		"sSortableNone": "sorting_disabled",
		"sSortColumn": "sorting_", /* Note that an int is postfixed for the sorting order */
	
		/* Filtering */
		"sFilterInput": "",
	
		/* Page length */
		"sLengthSelect": "",
	
		/* Scrolling */
		"sScrollWrapper": "dataTables_scroll",
		"sScrollHead": "dataTables_scrollHead",
		"sScrollHeadInner": "dataTables_scrollHeadInner",
		"sScrollBody": "dataTables_scrollBody",
		"sScrollFoot": "dataTables_scrollFoot",
		"sScrollFootInner": "dataTables_scrollFootInner",
	
		/* Misc */
		"sHeaderTH": "",
		"sFooterTH": "",
	
		// Deprecated
		"sSortJUIAsc": "",
		"sSortJUIDesc": "",
		"sSortJUI": "",
		"sSortJUIAscAllowed": "",
		"sSortJUIDescAllowed": "",
		"sSortJUIWrapper": "",
		"sSortIcon": "",
		"sJUIHeader": "",
		"sJUIFooter": ""
	} );
	
	
	var extPagination = DataTable.ext.pager;
	
	function _numbers ( page, pages ) {
		var
			numbers = [],
			buttons = extPagination.numbers_length,
			half = Math.floor( buttons / 2 ),
			i = 1;
	
		if ( pages <= buttons ) {
			numbers = _range( 0, pages );
		}
		else if ( page <= half ) {
			numbers = _range( 0, buttons-2 );
			numbers.push( 'ellipsis' );
			numbers.push( pages-1 );
		}
		else if ( page >= pages - 1 - half ) {
			numbers = _range( pages-(buttons-2), pages );
			numbers.splice( 0, 0, 'ellipsis' ); // no unshift in ie6
			numbers.splice( 0, 0, 0 );
		}
		else {
			numbers = _range( page-half+2, page+half-1 );
			numbers.push( 'ellipsis' );
			numbers.push( pages-1 );
			numbers.splice( 0, 0, 'ellipsis' );
			numbers.splice( 0, 0, 0 );
		}
	
		numbers.DT_el = 'span';
		return numbers;
	}
	
	
	$.extend( extPagination, {
		simple: function ( page, pages ) {
			return [ 'previous', 'next' ];
		},
	
		full: function ( page, pages ) {
			return [  'first', 'previous', 'next', 'last' ];
		},
	
		numbers: function ( page, pages ) {
			return [ _numbers(page, pages) ];
		},
	
		simple_numbers: function ( page, pages ) {
			return [ 'previous', _numbers(page, pages), 'next' ];
		},
	
		full_numbers: function ( page, pages ) {
			return [ 'first', 'previous', _numbers(page, pages), 'next', 'last' ];
		},
		
		first_last_numbers: function (page, pages) {
	 		return ['first', _numbers(page, pages), 'last'];
	 	},
	
		// For testing and plug-ins to use
		_numbers: _numbers,
	
		// Number of number buttons (including ellipsis) to show. _Must be odd!_
		numbers_length: 7
	} );
	
	
	$.extend( true, DataTable.ext.renderer, {
		pageButton: {
			_: function ( settings, host, idx, buttons, page, pages ) {
				var classes = settings.oClasses;
				var lang = settings.oLanguage.oPaginate;
				var aria = settings.oLanguage.oAria.paginate || {};
				var btnDisplay, btnClass, counter=0;
	
				var attach = function( container, buttons ) {
					var i, ien, node, button, tabIndex;
					var disabledClass = classes.sPageButtonDisabled;
					var clickHandler = function ( e ) {
						_fnPageChange( settings, e.data.action, true );
					};
	
					for ( i=0, ien=buttons.length ; i<ien ; i++ ) {
						button = buttons[i];
	
						if ( Array.isArray( button ) ) {
							var inner = $( '<'+(button.DT_el || 'div')+'/>' )
								.appendTo( container );
							attach( inner, button );
						}
						else {
							btnDisplay = null;
							btnClass = button;
							tabIndex = settings.iTabIndex;
	
							switch ( button ) {
								case 'ellipsis':
									container.append('<span class="ellipsis">&#x2026;</span>');
									break;
	
								case 'first':
									btnDisplay = lang.sFirst;
	
									if ( page === 0 ) {
										tabIndex = -1;
										btnClass += ' ' + disabledClass;
									}
									break;
	
								case 'previous':
									btnDisplay = lang.sPrevious;
	
									if ( page === 0 ) {
										tabIndex = -1;
										btnClass += ' ' + disabledClass;
									}
									break;
	
								case 'next':
									btnDisplay = lang.sNext;
	
									if ( pages === 0 || page === pages-1 ) {
										tabIndex = -1;
										btnClass += ' ' + disabledClass;
									}
									break;
	
								case 'last':
									btnDisplay = lang.sLast;
	
									if ( pages === 0 || page === pages-1 ) {
										tabIndex = -1;
										btnClass += ' ' + disabledClass;
									}
									break;
	
								default:
									btnDisplay = settings.fnFormatNumber( button + 1 );
									btnClass = page === button ?
										classes.sPageButtonActive : '';
									break;
							}
	
							if ( btnDisplay !== null ) {
								node = $('<a>', {
										'class': classes.sPageButton+' '+btnClass,
										'aria-controls': settings.sTableId,
										'aria-label': aria[ button ],
										'data-dt-idx': counter,
										'tabindex': tabIndex,
										'id': idx === 0 && typeof button === 'string' ?
											settings.sTableId +'_'+ button :
											null
									} )
									.html( btnDisplay )
									.appendTo( container );
	
								_fnBindAction(
									node, {action: button}, clickHandler
								);
	
								counter++;
							}
						}
					}
				};
	
				// IE9 throws an 'unknown error' if document.activeElement is used
				// inside an iframe or frame. Try / catch the error. Not good for
				// accessibility, but neither are frames.
				var activeEl;
	
				try {
					// Because this approach is destroying and recreating the paging
					// elements, focus is lost on the select button which is bad for
					// accessibility. So we want to restore focus once the draw has
					// completed
					activeEl = $(host).find(document.activeElement).data('dt-idx');
				}
				catch (e) {}
	
				attach( $(host).empty(), buttons );
	
				if ( activeEl !== undefined ) {
					$(host).find( '[data-dt-idx='+activeEl+']' ).trigger('focus');
				}
			}
		}
	} );
	
	
	
	// Built in type detection. See model.ext.aTypes for information about
	// what is required from this methods.
	$.extend( DataTable.ext.type.detect, [
		// Plain numbers - first since V8 detects some plain numbers as dates
		// e.g. Date.parse('55') (but not all, e.g. Date.parse('22')...).
		function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _isNumber( d, decimal ) ? 'num'+decimal : null;
		},
	
		// Dates (only those recognised by the browser's Date.parse)
		function ( d, settings )
		{
			// V8 tries _very_ hard to make a string passed into `Date.parse()`
			// valid, so we need to use a regex to restrict date formats. Use a
			// plug-in for anything other than ISO8601 style strings
			if ( d && !(d instanceof Date) && ! _re_date.test(d) ) {
				return null;
			}
			var parsed = Date.parse(d);
			return (parsed !== null && !isNaN(parsed)) || _empty(d) ? 'date' : null;
		},
	
		// Formatted numbers
		function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _isNumber( d, decimal, true ) ? 'num-fmt'+decimal : null;
		},
	
		// HTML numeric
		function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _htmlNumeric( d, decimal ) ? 'html-num'+decimal : null;
		},
	
		// HTML numeric, formatted
		function ( d, settings )
		{
			var decimal = settings.oLanguage.sDecimal;
			return _htmlNumeric( d, decimal, true ) ? 'html-num-fmt'+decimal : null;
		},
	
		// HTML (this is strict checking - there must be html)
		function ( d, settings )
		{
			return _empty( d ) || (typeof d === 'string' && d.indexOf('<') !== -1) ?
				'html' : null;
		}
	] );
	
	
	
	// Filter formatting functions. See model.ext.ofnSearch for information about
	// what is required from these methods.
	// 
	// Note that additional search methods are added for the html numbers and
	// html formatted numbers by `_addNumericSort()` when we know what the decimal
	// place is
	
	
	$.extend( DataTable.ext.type.search, {
		html: function ( data ) {
			return _empty(data) ?
				data :
				typeof data === 'string' ?
					data
						.replace( _re_new_lines, " " )
						.replace( _re_html, "" ) :
					'';
		},
	
		string: function ( data ) {
			return _empty(data) ?
				data :
				typeof data === 'string' ?
					data.replace( _re_new_lines, " " ) :
					data;
		}
	} );
	
	
	
	var __numericReplace = function ( d, decimalPlace, re1, re2 ) {
		if ( d !== 0 && (!d || d === '-') ) {
			return -Infinity;
		}
	
		// If a decimal place other than `.` is used, it needs to be given to the
		// function so we can detect it and replace with a `.` which is the only
		// decimal place Javascript recognises - it is not locale aware.
		if ( decimalPlace ) {
			d = _numToDecimal( d, decimalPlace );
		}
	
		if ( d.replace ) {
			if ( re1 ) {
				d = d.replace( re1, '' );
			}
	
			if ( re2 ) {
				d = d.replace( re2, '' );
			}
		}
	
		return d * 1;
	};
	
	
	// Add the numeric 'deformatting' functions for sorting and search. This is done
	// in a function to provide an easy ability for the language options to add
	// additional methods if a non-period decimal place is used.
	function _addNumericSort ( decimalPlace ) {
		$.each(
			{
				// Plain numbers
				"num": function ( d ) {
					return __numericReplace( d, decimalPlace );
				},
	
				// Formatted numbers
				"num-fmt": function ( d ) {
					return __numericReplace( d, decimalPlace, _re_formatted_numeric );
				},
	
				// HTML numeric
				"html-num": function ( d ) {
					return __numericReplace( d, decimalPlace, _re_html );
				},
	
				// HTML numeric, formatted
				"html-num-fmt": function ( d ) {
					return __numericReplace( d, decimalPlace, _re_html, _re_formatted_numeric );
				}
			},
			function ( key, fn ) {
				// Add the ordering method
				_ext.type.order[ key+decimalPlace+'-pre' ] = fn;
	
				// For HTML types add a search formatter that will strip the HTML
				if ( key.match(/^html\-/) ) {
					_ext.type.search[ key+decimalPlace ] = _ext.type.search.html;
				}
			}
		);
	}
	
	
	// Default sort methods
	$.extend( _ext.type.order, {
		// Dates
		"date-pre": function ( d ) {
			var ts = Date.parse( d );
			return isNaN(ts) ? -Infinity : ts;
		},
	
		// html
		"html-pre": function ( a ) {
			return _empty(a) ?
				'' :
				a.replace ?
					a.replace( /<.*?>/g, "" ).toLowerCase() :
					a+'';
		},
	
		// string
		"string-pre": function ( a ) {
			// This is a little complex, but faster than always calling toString,
			// http://jsperf.com/tostring-v-check
			return _empty(a) ?
				'' :
				typeof a === 'string' ?
					a.toLowerCase() :
					! a.toString ?
						'' :
						a.toString();
		},
	
		// string-asc and -desc are retained only for compatibility with the old
		// sort methods
		"string-asc": function ( x, y ) {
			return ((x < y) ? -1 : ((x > y) ? 1 : 0));
		},
	
		"string-desc": function ( x, y ) {
			return ((x < y) ? 1 : ((x > y) ? -1 : 0));
		}
	} );
	
	
	// Numeric sorting types - order doesn't matter here
	_addNumericSort( '' );
	
	
	$.extend( true, DataTable.ext.renderer, {
		header: {
			_: function ( settings, cell, column, classes ) {
				// No additional mark-up required
				// Attach a sort listener to update on sort - note that using the
				// `DT` namespace will allow the event to be removed automatically
				// on destroy, while the `dt` namespaced event is the one we are
				// listening for
				$(settings.nTable).on( 'order.dt.DT', function ( e, ctx, sorting, columns ) {
					if ( settings !== ctx ) { // need to check this this is the host
						return;               // table, not a nested one
					}
	
					var colIdx = column.idx;
	
					cell
						.removeClass(
							classes.sSortAsc +' '+
							classes.sSortDesc
						)
						.addClass( columns[ colIdx ] == 'asc' ?
							classes.sSortAsc : columns[ colIdx ] == 'desc' ?
								classes.sSortDesc :
								column.sSortingClass
						);
				} );
			},
	
			jqueryui: function ( settings, cell, column, classes ) {
				$('<div/>')
					.addClass( classes.sSortJUIWrapper )
					.append( cell.contents() )
					.append( $('<span/>')
						.addClass( classes.sSortIcon+' '+column.sSortingClassJUI )
					)
					.appendTo( cell );
	
				// Attach a sort listener to update on sort
				$(settings.nTable).on( 'order.dt.DT', function ( e, ctx, sorting, columns ) {
					if ( settings !== ctx ) {
						return;
					}
	
					var colIdx = column.idx;
	
					cell
						.removeClass( classes.sSortAsc +" "+classes.sSortDesc )
						.addClass( columns[ colIdx ] == 'asc' ?
							classes.sSortAsc : columns[ colIdx ] == 'desc' ?
								classes.sSortDesc :
								column.sSortingClass
						);
	
					cell
						.find( 'span.'+classes.sSortIcon )
						.removeClass(
							classes.sSortJUIAsc +" "+
							classes.sSortJUIDesc +" "+
							classes.sSortJUI +" "+
							classes.sSortJUIAscAllowed +" "+
							classes.sSortJUIDescAllowed
						)
						.addClass( columns[ colIdx ] == 'asc' ?
							classes.sSortJUIAsc : columns[ colIdx ] == 'desc' ?
								classes.sSortJUIDesc :
								column.sSortingClassJUI
						);
				} );
			}
		}
	} );
	
	/*
	 * Public helper functions. These aren't used internally by DataTables, or
	 * called by any of the options passed into DataTables, but they can be used
	 * externally by developers working with DataTables. They are helper functions
	 * to make working with DataTables a little bit easier.
	 */
	
	var __htmlEscapeEntities = function ( d ) {
		return typeof d === 'string' ?
			d
				.replace(/&/g, '&amp;')
				.replace(/</g, '&lt;')
				.replace(/>/g, '&gt;')
				.replace(/"/g, '&quot;') :
			d;
	};
	
	/**
	 * Helpers for `columns.render`.
	 *
	 * The options defined here can be used with the `columns.render` initialisation
	 * option to provide a display renderer. The following functions are defined:
	 *
	 * * `number` - Will format numeric data (defined by `columns.data`) for
	 *   display, retaining the original unformatted data for sorting and filtering.
	 *   It takes 5 parameters:
	 *   * `string` - Thousands grouping separator
	 *   * `string` - Decimal point indicator
	 *   * `integer` - Number of decimal points to show
	 *   * `string` (optional) - Prefix.
	 *   * `string` (optional) - Postfix (/suffix).
	 * * `text` - Escape HTML to help prevent XSS attacks. It has no optional
	 *   parameters.
	 *
	 * @example
	 *   // Column definition using the number renderer
	 *   {
	 *     data: "salary",
	 *     render: $.fn.dataTable.render.number( '\'', '.', 0, '$' )
	 *   }
	 *
	 * @namespace
	 */
	DataTable.render = {
		number: function ( thousands, decimal, precision, prefix, postfix ) {
			return {
				display: function ( d ) {
					if ( typeof d !== 'number' && typeof d !== 'string' ) {
						return d;
					}
	
					var negative = d < 0 ? '-' : '';
					var flo = parseFloat( d );
	
					// If NaN then there isn't much formatting that we can do - just
					// return immediately, escaping any HTML (this was supposed to
					// be a number after all)
					if ( isNaN( flo ) ) {
						return __htmlEscapeEntities( d );
					}
	
					flo = flo.toFixed( precision );
					d = Math.abs( flo );
	
					var intPart = parseInt( d, 10 );
					var floatPart = precision ?
						decimal+(d - intPart).toFixed( precision ).substring( 2 ):
						'';
	
					return negative + (prefix||'') +
						intPart.toString().replace(
							/\B(?=(\d{3})+(?!\d))/g, thousands
						) +
						floatPart +
						(postfix||'');
				}
			};
		},
	
		text: function () {
			return {
				display: __htmlEscapeEntities,
				filter: __htmlEscapeEntities
			};
		}
	};
	
	
	/*
	 * This is really a good bit rubbish this method of exposing the internal methods
	 * publicly... - To be fixed in 2.0 using methods on the prototype
	 */
	
	
	/**
	 * Create a wrapper function for exporting an internal functions to an external API.
	 *  @param {string} fn API function name
	 *  @returns {function} wrapped function
	 *  @memberof DataTable#internal
	 */
	function _fnExternApiFunc (fn)
	{
		return function() {
			var args = [_fnSettingsFromNode( this[DataTable.ext.iApiIndex] )].concat(
				Array.prototype.slice.call(arguments)
			);
			return DataTable.ext.internal[fn].apply( this, args );
		};
	}
	
	
	/**
	 * Reference to internal functions for use by plug-in developers. Note that
	 * these methods are references to internal functions and are considered to be
	 * private. If you use these methods, be aware that they are liable to change
	 * between versions.
	 *  @namespace
	 */
	$.extend( DataTable.ext.internal, {
		_fnExternApiFunc: _fnExternApiFunc,
		_fnBuildAjax: _fnBuildAjax,
		_fnAjaxUpdate: _fnAjaxUpdate,
		_fnAjaxParameters: _fnAjaxParameters,
		_fnAjaxUpdateDraw: _fnAjaxUpdateDraw,
		_fnAjaxDataSrc: _fnAjaxDataSrc,
		_fnAddColumn: _fnAddColumn,
		_fnColumnOptions: _fnColumnOptions,
		_fnAdjustColumnSizing: _fnAdjustColumnSizing,
		_fnVisibleToColumnIndex: _fnVisibleToColumnIndex,
		_fnColumnIndexToVisible: _fnColumnIndexToVisible,
		_fnVisbleColumns: _fnVisbleColumns,
		_fnGetColumns: _fnGetColumns,
		_fnColumnTypes: _fnColumnTypes,
		_fnApplyColumnDefs: _fnApplyColumnDefs,
		_fnHungarianMap: _fnHungarianMap,
		_fnCamelToHungarian: _fnCamelToHungarian,
		_fnLanguageCompat: _fnLanguageCompat,
		_fnBrowserDetect: _fnBrowserDetect,
		_fnAddData: _fnAddData,
		_fnAddTr: _fnAddTr,
		_fnNodeToDataIndex: _fnNodeToDataIndex,
		_fnNodeToColumnIndex: _fnNodeToColumnIndex,
		_fnGetCellData: _fnGetCellData,
		_fnSetCellData: _fnSetCellData,
		_fnSplitObjNotation: _fnSplitObjNotation,
		_fnGetObjectDataFn: _fnGetObjectDataFn,
		_fnSetObjectDataFn: _fnSetObjectDataFn,
		_fnGetDataMaster: _fnGetDataMaster,
		_fnClearTable: _fnClearTable,
		_fnDeleteIndex: _fnDeleteIndex,
		_fnInvalidate: _fnInvalidate,
		_fnGetRowElements: _fnGetRowElements,
		_fnCreateTr: _fnCreateTr,
		_fnBuildHead: _fnBuildHead,
		_fnDrawHead: _fnDrawHead,
		_fnDraw: _fnDraw,
		_fnReDraw: _fnReDraw,
		_fnAddOptionsHtml: _fnAddOptionsHtml,
		_fnDetectHeader: _fnDetectHeader,
		_fnGetUniqueThs: _fnGetUniqueThs,
		_fnFeatureHtmlFilter: _fnFeatureHtmlFilter,
		_fnFilterComplete: _fnFilterComplete,
		_fnFilterCustom: _fnFilterCustom,
		_fnFilterColumn: _fnFilterColumn,
		_fnFilter: _fnFilter,
		_fnFilterCreateSearch: _fnFilterCreateSearch,
		_fnEscapeRegex: _fnEscapeRegex,
		_fnFilterData: _fnFilterData,
		_fnFeatureHtmlInfo: _fnFeatureHtmlInfo,
		_fnUpdateInfo: _fnUpdateInfo,
		_fnInfoMacros: _fnInfoMacros,
		_fnInitialise: _fnInitialise,
		_fnInitComplete: _fnInitComplete,
		_fnLengthChange: _fnLengthChange,
		_fnFeatureHtmlLength: _fnFeatureHtmlLength,
		_fnFeatureHtmlPaginate: _fnFeatureHtmlPaginate,
		_fnPageChange: _fnPageChange,
		_fnFeatureHtmlProcessing: _fnFeatureHtmlProcessing,
		_fnProcessingDisplay: _fnProcessingDisplay,
		_fnFeatureHtmlTable: _fnFeatureHtmlTable,
		_fnScrollDraw: _fnScrollDraw,
		_fnApplyToChildren: _fnApplyToChildren,
		_fnCalculateColumnWidths: _fnCalculateColumnWidths,
		_fnThrottle: _fnThrottle,
		_fnConvertToWidth: _fnConvertToWidth,
		_fnGetWidestNode: _fnGetWidestNode,
		_fnGetMaxLenString: _fnGetMaxLenString,
		_fnStringToCss: _fnStringToCss,
		_fnSortFlatten: _fnSortFlatten,
		_fnSort: _fnSort,
		_fnSortAria: _fnSortAria,
		_fnSortListener: _fnSortListener,
		_fnSortAttachListener: _fnSortAttachListener,
		_fnSortingClasses: _fnSortingClasses,
		_fnSortData: _fnSortData,
		_fnSaveState: _fnSaveState,
		_fnLoadState: _fnLoadState,
		_fnSettingsFromNode: _fnSettingsFromNode,
		_fnLog: _fnLog,
		_fnMap: _fnMap,
		_fnBindAction: _fnBindAction,
		_fnCallbackReg: _fnCallbackReg,
		_fnCallbackFire: _fnCallbackFire,
		_fnLengthOverflow: _fnLengthOverflow,
		_fnRenderer: _fnRenderer,
		_fnDataSource: _fnDataSource,
		_fnRowAttributes: _fnRowAttributes,
		_fnExtend: _fnExtend,
		_fnCalculateEnd: function () {} // Used by a lot of plug-ins, but redundant
		                                // in 1.10, so this dead-end function is
		                                // added to prevent errors
	} );
	

	// jQuery access
	$.fn.dataTable = DataTable;

	// Provide access to the host jQuery object (circular reference)
	DataTable.$ = $;

	// Legacy aliases
	$.fn.dataTableSettings = DataTable.settings;
	$.fn.dataTableExt = DataTable.ext;

	// With a capital `D` we return a DataTables API instance rather than a
	// jQuery object
	$.fn.DataTable = function ( opts ) {
		return $(this).dataTable( opts ).api();
	};

	// All properties that are available to $.fn.dataTable should also be
	// available on $.fn.DataTable
	$.each( DataTable, function ( prop, val ) {
		$.fn.DataTable[ prop ] = val;
	} );


	// Information about events fired by DataTables - for documentation.
	/**
	 * Draw event, fired whenever the table is redrawn on the page, at the same
	 * point as fnDrawCallback. This may be useful for binding events or
	 * performing calculations when the table is altered at all.
	 *  @name DataTable#draw.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} o DataTables settings object {@link DataTable.models.oSettings}
	 */

	/**
	 * Search event, fired when the searching applied to the table (using the
	 * built-in global search, or column filters) is altered.
	 *  @name DataTable#search.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} o DataTables settings object {@link DataTable.models.oSettings}
	 */

	/**
	 * Page change event, fired when the paging of the table is altered.
	 *  @name DataTable#page.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} o DataTables settings object {@link DataTable.models.oSettings}
	 */

	/**
	 * Order event, fired when the ordering applied to the table is altered.
	 *  @name DataTable#order.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} o DataTables settings object {@link DataTable.models.oSettings}
	 */

	/**
	 * DataTables initialisation complete event, fired when the table is fully
	 * drawn, including Ajax data loaded, if Ajax data is required.
	 *  @name DataTable#init.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} oSettings DataTables settings object
	 *  @param {object} json The JSON object request from the server - only
	 *    present if client-side Ajax sourced data is used</li></ol>
	 */

	/**
	 * State save event, fired when the table has changed state a new state save
	 * is required. This event allows modification of the state saving object
	 * prior to actually doing the save, including addition or other state
	 * properties (for plug-ins) or modification of a DataTables core property.
	 *  @name DataTable#stateSaveParams.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} oSettings DataTables settings object
	 *  @param {object} json The state information to be saved
	 */

	/**
	 * State load event, fired when the table is loading state from the stored
	 * data, but prior to the settings object being modified by the saved state
	 * - allowing modification of the saved state is required or loading of
	 * state for a plug-in.
	 *  @name DataTable#stateLoadParams.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} oSettings DataTables settings object
	 *  @param {object} json The saved state information
	 */

	/**
	 * State loaded event, fired when state has been loaded from stored data and
	 * the settings object has been modified by the loaded data.
	 *  @name DataTable#stateLoaded.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} oSettings DataTables settings object
	 *  @param {object} json The saved state information
	 */

	/**
	 * Processing event, fired when DataTables is doing some kind of processing
	 * (be it, order, search or anything else). It can be used to indicate to
	 * the end user that there is something happening, or that something has
	 * finished.
	 *  @name DataTable#processing.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} oSettings DataTables settings object
	 *  @param {boolean} bShow Flag for if DataTables is doing processing or not
	 */

	/**
	 * Ajax (XHR) event, fired whenever an Ajax request is completed from a
	 * request to made to the server for new data. This event is called before
	 * DataTables processed the returned data, so it can also be used to pre-
	 * process the data returned from the server, if needed.
	 *
	 * Note that this trigger is called in `fnServerData`, if you override
	 * `fnServerData` and which to use this event, you need to trigger it in you
	 * success function.
	 *  @name DataTable#xhr.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} o DataTables settings object {@link DataTable.models.oSettings}
	 *  @param {object} json JSON returned from the server
	 *
	 *  @example
	 *     // Use a custom property returned from the server in another DOM element
	 *     $('#table').dataTable().on('xhr.dt', function (e, settings, json) {
	 *       $('#status').html( json.status );
	 *     } );
	 *
	 *  @example
	 *     // Pre-process the data returned from the server
	 *     $('#table').dataTable().on('xhr.dt', function (e, settings, json) {
	 *       for ( var i=0, ien=json.aaData.length ; i<ien ; i++ ) {
	 *         json.aaData[i].sum = json.aaData[i].one + json.aaData[i].two;
	 *       }
	 *       // Note no return - manipulate the data directly in the JSON object.
	 *     } );
	 */

	/**
	 * Destroy event, fired when the DataTable is destroyed by calling fnDestroy
	 * or passing the bDestroy:true parameter in the initialisation object. This
	 * can be used to remove bound events, added DOM nodes, etc.
	 *  @name DataTable#destroy.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} o DataTables settings object {@link DataTable.models.oSettings}
	 */

	/**
	 * Page length change event, fired when number of records to show on each
	 * page (the length) is changed.
	 *  @name DataTable#length.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} o DataTables settings object {@link DataTable.models.oSettings}
	 *  @param {integer} len New length
	 */

	/**
	 * Column sizing has changed.
	 *  @name DataTable#column-sizing.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} o DataTables settings object {@link DataTable.models.oSettings}
	 */

	/**
	 * Column visibility has changed.
	 *  @name DataTable#column-visibility.dt
	 *  @event
	 *  @param {event} e jQuery event object
	 *  @param {object} o DataTables settings object {@link DataTable.models.oSettings}
	 *  @param {int} column Column index
	 *  @param {bool} vis `false` if column now hidden, or `true` if visible
	 */

	return $.fn.dataTable;
}));

/*! DataTables Bootstrap 4 integration
 * ©2011-2017 SpryMedia Ltd - datatables.net/license
 */

/**
 * DataTables integration for Bootstrap 4. This requires Bootstrap 4 and
 * DataTables 1.10 or newer.
 *
 * This file sets the defaults and adds options to DataTables to style its
 * controls using Bootstrap. See http://datatables.net/manual/styling/bootstrap
 * for further information.
 */
(function( factory ){
	if ( typeof define === 'function' && define.amd ) {
		// AMD
		define( ['jquery', 'datatables.net'], function ( $ ) {
			return factory( $, window, document );
		} );
	}
	else if ( typeof exports === 'object' ) {
		// CommonJS
		module.exports = function (root, $) {
			if ( ! root ) {
				root = window;
			}

			if ( ! $ || ! $.fn.dataTable ) {
				// Require DataTables, which attaches to jQuery, including
				// jQuery if needed and have a $ property so we can access the
				// jQuery object that is used
				$ = require('datatables.net')(root, $).$;
			}

			return factory( $, root, root.document );
		};
	}
	else {
		// Browser
		factory( jQuery, window, document );
	}
}(function( $, window, document, undefined ) {
'use strict';
var DataTable = $.fn.dataTable;


/* Set the defaults for DataTables initialisation */
$.extend( true, DataTable.defaults, {
	dom:
		"<'row'<'col-sm-12 col-md-6'l><'col-sm-12 col-md-6'f>>" +
		"<'row'<'col-sm-12'tr>>" +
		"<'row'<'col-sm-12 col-md-5'i><'col-sm-12 col-md-7'p>>",
	renderer: 'bootstrap'
} );


/* Default class modification */
$.extend( DataTable.ext.classes, {
	sWrapper:      "dataTables_wrapper dt-bootstrap4",
	sFilterInput:  "form-control form-control-sm",
	sLengthSelect: "custom-select custom-select-sm form-control form-control-sm",
	sProcessing:   "dataTables_processing card",
	sPageButton:   "paginate_button page-item"
} );


/* Bootstrap paging button renderer */
DataTable.ext.renderer.pageButton.bootstrap = function ( settings, host, idx, buttons, page, pages ) {
	var api     = new DataTable.Api( settings );
	var classes = settings.oClasses;
	var lang    = settings.oLanguage.oPaginate;
	var aria = settings.oLanguage.oAria.paginate || {};
	var btnDisplay, btnClass, counter=0;

	var attach = function( container, buttons ) {
		var i, ien, node, button;
		var clickHandler = function ( e ) {
			e.preventDefault();
			if ( !$(e.currentTarget).hasClass('disabled') && api.page() != e.data.action ) {
				api.page( e.data.action ).draw( 'page' );
			}
		};

		for ( i=0, ien=buttons.length ; i<ien ; i++ ) {
			button = buttons[i];

			if ( Array.isArray( button ) ) {
				attach( container, button );
			}
			else {
				btnDisplay = '';
				btnClass = '';

				switch ( button ) {
					case 'ellipsis':
						btnDisplay = '&#x2026;';
						btnClass = 'disabled';
						break;

					case 'first':
						btnDisplay = lang.sFirst;
						btnClass = button + (page > 0 ?
							'' : ' disabled');
						break;

					case 'previous':
						btnDisplay = lang.sPrevious;
						btnClass = button + (page > 0 ?
							'' : ' disabled');
						break;

					case 'next':
						btnDisplay = lang.sNext;
						btnClass = button + (page < pages-1 ?
							'' : ' disabled');
						break;

					case 'last':
						btnDisplay = lang.sLast;
						btnClass = button + (page < pages-1 ?
							'' : ' disabled');
						break;

					default:
						btnDisplay = button + 1;
						btnClass = page === button ?
							'active' : '';
						break;
				}

				if ( btnDisplay ) {
					node = $('<li>', {
							'class': classes.sPageButton+' '+btnClass,
							'id': idx === 0 && typeof button === 'string' ?
								settings.sTableId +'_'+ button :
								null
						} )
						.append( $('<a>', {
								'href': '#',
								'aria-controls': settings.sTableId,
								'aria-label': aria[ button ],
								'data-dt-idx': counter,
								'tabindex': settings.iTabIndex,
								'class': 'page-link'
							} )
							.html( btnDisplay )
						)
						.appendTo( container );

					settings.oApi._fnBindAction(
						node, {action: button}, clickHandler
					);

					counter++;
				}
			}
		}
	};

	// IE9 throws an 'unknown error' if document.activeElement is used
	// inside an iframe or frame. 
	var activeEl;

	try {
		// Because this approach is destroying and recreating the paging
		// elements, focus is lost on the select button which is bad for
		// accessibility. So we want to restore focus once the draw has
		// completed
		activeEl = $(host).find(document.activeElement).data('dt-idx');
	}
	catch (e) {}

	attach(
		$(host).empty().html('<ul class="pagination"/>').children('ul'),
		buttons
	);

	if ( activeEl !== undefined ) {
		$(host).find( '[data-dt-idx='+activeEl+']' ).trigger('focus');
	}
};


return DataTable;
}));

"use strict";
var defaults = {
	"language": {
		"paginate": {
			"first": '<i class="ki ki-double-arrow-back"></i>',
			"last": '<i class="ki ki-double-arrow-next"></i>',
			"next": '<i class="ki ki-arrow-next"></i>',
			"previous": '<i class="ki ki-arrow-back"></i>'
		}
	}
};

if (KTUtil.isRTL()) {
	defaults = {
		"language": {
			"paginate": {
				"first": '<i class="ki ki-double-arrow-next"></i>',
				"last": '<i class="ki ki-double-arrow-back"></i>',
				"next": '<i class="ki ki-arrow-back"></i>',
				"previous": '<i class="ki ki-arrow-next"></i>'
			}
		}
	}
}

$.extend(true, $.fn.dataTable.defaults, defaults);

// fix dropdown overflow inside datatable
KTApp.initAbsoluteDropdown('.dataTables_wrapper');

/*!
 AutoFill 2.3.5
 ©2008-2020 SpryMedia Ltd - datatables.net/license
*/
(function(e){"function"===typeof define&&define.amd?define(["jquery","datatables.net"],function(g){return e(g,window,document)}):"object"===typeof exports?module.exports=function(g,i){g||(g=window);if(!i||!i.fn.dataTable)i=require("datatables.net")(g,i).$;return e(i,g,g.document)}:e(jQuery,window,document)})(function(e,g,i,r){var l=e.fn.dataTable,t=0,k=function(b,c){if(!l.versionCheck||!l.versionCheck("1.10.8"))throw"Warning: AutoFill requires DataTables 1.10.8 or greater";this.c=e.extend(!0,{},l.defaults.autoFill,
k.defaults,c);this.s={dt:new l.Api(b),namespace:".autoFill"+t++,scroll:{},scrollInterval:null,handle:{height:0,width:0},enabled:!1};this.dom={handle:e('<div class="dt-autofill-handle"/>'),select:{top:e('<div class="dt-autofill-select top"/>'),right:e('<div class="dt-autofill-select right"/>'),bottom:e('<div class="dt-autofill-select bottom"/>'),left:e('<div class="dt-autofill-select left"/>')},background:e('<div class="dt-autofill-background"/>'),list:e('<div class="dt-autofill-list">'+this.s.dt.i18n("autoFill.info",
"")+"<ul/></div>"),dtScroll:null,offsetParent:null};this._constructor()};e.extend(k.prototype,{enabled:function(){return this.s.enabled},enable:function(b){var c=this;if(!1===b)return this.disable();this.s.enabled=!0;this._focusListener();this.dom.handle.on("mousedown",function(a){c._mousedown(a);return!1});return this},disable:function(){this.s.enabled=!1;this._focusListenerRemove();return this},_constructor:function(){var b=this,c=this.s.dt,a=e("div.dataTables_scrollBody",this.s.dt.table().container());
c.settings()[0].autoFill=this;a.length&&(this.dom.dtScroll=a,"static"===a.css("position")&&a.css("position","relative"));!1!==this.c.enable&&this.enable();c.on("destroy.autoFill",function(){b._focusListenerRemove()})},_attach:function(b){var c=this.s.dt,a=c.cell(b).index(),d=this.dom.handle,f=this.s.handle;if(!a||-1===c.columns(this.c.columns).indexes().indexOf(a.column))this._detach();else{this.dom.offsetParent||(this.dom.offsetParent=e(c.table().node()).offsetParent());if(!f.height||!f.width)d.appendTo("body"),
f.height=d.outerHeight(),f.width=d.outerWidth();c=this._getPosition(b,this.dom.offsetParent);this.dom.attachedTo=b;d.css({top:c.top+b.offsetHeight-f.height,left:c.left+b.offsetWidth-f.width}).appendTo(this.dom.offsetParent)}},_actionSelector:function(b){var c=this,a=this.s.dt,d=k.actions,f=[];e.each(d,function(c,d){d.available(a,b)&&f.push(c)});if(1===f.length&&!1===this.c.alwaysAsk){var j=d[f[0]].execute(a,b);this._update(j,b)}else if(1<f.length){var h=this.dom.list.children("ul").empty();f.push("cancel");
e.each(f,function(f,j){h.append(e("<li/>").append('<div class="dt-autofill-question">'+d[j].option(a,b)+"<div>").append(e('<div class="dt-autofill-button">').append(e('<button class="'+k.classes.btn+'">'+a.i18n("autoFill.button","&gt;")+"</button>").on("click",function(){var f=d[j].execute(a,b,e(this).closest("li"));c._update(f,b);c.dom.background.remove();c.dom.list.remove()}))))});this.dom.background.appendTo("body");this.dom.list.appendTo("body");this.dom.list.css("margin-top",-1*(this.dom.list.outerHeight()/
2))}},_detach:function(){this.dom.attachedTo=null;this.dom.handle.detach()},_drawSelection:function(b){var c=this.s.dt,a=this.s.start,d=e(this.dom.start),f={row:this.c.vertical?c.rows({page:"current"}).nodes().indexOf(b.parentNode):a.row,column:this.c.horizontal?e(b).index():a.column},b=c.column.index("toData",f.column),j=c.row(":eq("+f.row+")",{page:"current"}),j=e(c.cell(j.index(),b).node());if(c.cell(j).any()&&-1!==c.columns(this.c.columns).indexes().indexOf(b)){this.s.end=f;var h,c=a.row<f.row?
d:j;h=a.row<f.row?j:d;b=a.column<f.column?d:j;d=a.column<f.column?j:d;c=this._getPosition(c.get(0)).top;b=this._getPosition(b.get(0)).left;a=this._getPosition(h.get(0)).top+h.outerHeight()-c;d=this._getPosition(d.get(0)).left+d.outerWidth()-b;f=this.dom.select;f.top.css({top:c,left:b,width:d});f.left.css({top:c,left:b,height:a});f.bottom.css({top:c+a,left:b,width:d});f.right.css({top:c,left:b+d,height:a})}},_editor:function(b){var c=this.s.dt,a=this.c.editor;if(a){for(var d={},f=[],e=a.fields(),h=
0,i=b.length;h<i;h++)for(var p=0,k=b[h].length;p<k;p++){var n=b[h][p],g=c.settings()[0].aoColumns[n.index.column],o=g.editField;if(o===r)for(var g=g.mData,q=0,l=e.length;q<l;q++){var s=a.field(e[q]);if(s.dataSrc()===g){o=s.name();break}}if(!o)throw"Could not automatically determine field data. Please see https://datatables.net/tn/11";d[o]||(d[o]={});g=c.row(n.index.row).id();d[o][g]=n.set;f.push(n.index)}a.bubble(f,!1).multiSet(d).submit()}},_emitEvent:function(b,c){this.s.dt.iterator("table",function(a){e(a.nTable).triggerHandler(b+
".dt",c)})},_focusListener:function(){var b=this,c=this.s.dt,a=this.s.namespace,d=null!==this.c.focus?this.c.focus:c.init().keys||c.settings()[0].keytable?"focus":"hover";if("focus"===d)c.on("key-focus.autoFill",function(a,c,d){b._attach(d.node())}).on("key-blur.autoFill",function(){b._detach()});else if("click"===d)e(c.table().body()).on("click"+a,"td, th",function(){b._attach(this)}),e(i.body).on("click"+a,function(a){e(a.target).parents().filter(c.table().body()).length||b._detach()});else e(c.table().body()).on("mouseenter"+
a,"td, th",function(){b._attach(this)}).on("mouseleave"+a,function(a){e(a.relatedTarget).hasClass("dt-autofill-handle")||b._detach()})},_focusListenerRemove:function(){var b=this.s.dt;b.off(".autoFill");e(b.table().body()).off(this.s.namespace);e(i.body).off(this.s.namespace)},_getPosition:function(b,c){var a=b,d,f=0,j=0;c||(c=e(e(this.s.dt.table().node())[0].offsetParent));do{var h=a.offsetTop,i=a.offsetLeft;d=e(a.offsetParent);f+=h+1*parseInt(d.css("border-top-width"));j+=i+1*parseInt(d.css("border-left-width"));
if("body"===a.nodeName.toLowerCase())break;a=d.get(0)}while(d.get(0)!==c.get(0));return{top:f,left:j}},_mousedown:function(b){var c=this,a=this.s.dt;this.dom.start=this.dom.attachedTo;this.s.start={row:a.rows({page:"current"}).nodes().indexOf(e(this.dom.start).parent()[0]),column:e(this.dom.start).index()};e(i.body).on("mousemove.autoFill",function(a){c._mousemove(a)}).on("mouseup.autoFill",function(a){c._mouseup(a)});var d=this.dom.select,a=e(a.table().node()).offsetParent();d.top.appendTo(a);d.left.appendTo(a);
d.right.appendTo(a);d.bottom.appendTo(a);this._drawSelection(this.dom.start,b);this.dom.handle.css("display","none");b=this.dom.dtScroll;this.s.scroll={windowHeight:e(g).height(),windowWidth:e(g).width(),dtTop:b?b.offset().top:null,dtLeft:b?b.offset().left:null,dtHeight:b?b.outerHeight():null,dtWidth:b?b.outerWidth():null}},_mousemove:function(b){var c=b.target.nodeName.toLowerCase();"td"!==c&&"th"!==c||(this._drawSelection(b.target,b),this._shiftScroll(b))},_mouseup:function(b){e(i.body).off(".autoFill");
var c=this,a=this.s.dt,d=this.dom.select;d.top.remove();d.left.remove();d.right.remove();d.bottom.remove();this.dom.handle.css("display","block");var d=this.s.start,f=this.s.end;if(!(d.row===f.row&&d.column===f.column)){var j=a.cell(":eq("+d.row+")",d.column+":visible",{page:"current"});if(e("div.DTE",j.node()).length){var h=a.editor();h.on("submitSuccess.dtaf close.dtaf",function(){h.off(".dtaf");setTimeout(function(){c._mouseup(b)},100)}).on("submitComplete.dtaf preSubmitCancelled.dtaf close.dtaf",
function(){h.off(".dtaf")});h.submit()}else{for(var g=this._range(d.row,f.row),d=this._range(d.column,f.column),f=[],k=a.settings()[0],l=k.aoColumns,n=a.columns(this.c.columns).indexes(),m=0;m<g.length;m++)f.push(e.map(d,function(b){var c=a.row(":eq("+g[m]+")",{page:"current"}),b=a.cell(c.index(),b+":visible"),c=b.data(),d=b.index(),f=l[d.column].editField;f!==r&&(c=k.oApi._fnGetObjectDataFn(f)(a.row(d.row).data()));if(-1!==n.indexOf(d.column))return{cell:b,data:c,label:b.data(),index:d}}));this._actionSelector(f);
clearInterval(this.s.scrollInterval);this.s.scrollInterval=null}}},_range:function(b,c){var a=[],d;if(b<=c)for(d=b;d<=c;d++)a.push(d);else for(d=b;d>=c;d--)a.push(d);return a},_shiftScroll:function(b){var c=this,a=this.s.scroll,d=!1,f=b.pageY-i.body.scrollTop,e=b.pageX-i.body.scrollLeft,h,g,k,l;65>f?h=-5:f>a.windowHeight-65&&(h=5);65>e?g=-5:e>a.windowWidth-65&&(g=5);null!==a.dtTop&&b.pageY<a.dtTop+65?k=-5:null!==a.dtTop&&b.pageY>a.dtTop+a.dtHeight-65&&(k=5);null!==a.dtLeft&&b.pageX<a.dtLeft+65?l=
-5:null!==a.dtLeft&&b.pageX>a.dtLeft+a.dtWidth-65&&(l=5);h||g||k||l?(a.windowVert=h,a.windowHoriz=g,a.dtVert=k,a.dtHoriz=l,d=!0):this.s.scrollInterval&&(clearInterval(this.s.scrollInterval),this.s.scrollInterval=null);!this.s.scrollInterval&&d&&(this.s.scrollInterval=setInterval(function(){if(a.windowVert)i.body.scrollTop=i.body.scrollTop+a.windowVert;if(a.windowHoriz)i.body.scrollLeft=i.body.scrollLeft+a.windowHoriz;if(a.dtVert||a.dtHoriz){var b=c.dom.dtScroll[0];if(a.dtVert)b.scrollTop=b.scrollTop+
a.dtVert;if(a.dtHoriz)b.scrollLeft=b.scrollLeft+a.dtHoriz}},20))},_update:function(b,c){if(!1!==b){var a=this.s.dt,d,f=a.columns(this.c.columns).indexes();this._emitEvent("preAutoFill",[a,c]);this._editor(c);if(null!==this.c.update?this.c.update:!this.c.editor){for(var e=0,h=c.length;e<h;e++)for(var g=0,i=c[e].length;g<i;g++)d=c[e][g],-1!==f.indexOf(d.index.column)&&d.cell.data(d.set);a.draw(!1)}this._emitEvent("autoFill",[a,c])}}});k.actions={increment:{available:function(b,c){var a=c[0][0].label;
return!isNaN(a-parseFloat(a))},option:function(b){return b.i18n("autoFill.increment",'Increment / decrement each cell by: <input type="number" value="1">')},execute:function(b,c,a){for(var b=1*c[0][0].data,a=1*e("input",a).val(),d=0,f=c.length;d<f;d++)for(var j=0,g=c[d].length;j<g;j++)c[d][j].set=b,b+=a}},fill:{available:function(){return!0},option:function(b,c){return b.i18n("autoFill.fill","Fill all cells with <i>"+c[0][0].label+"</i>")},execute:function(b,c){for(var a=c[0][0].data,d=0,f=c.length;d<
f;d++)for(var e=0,g=c[d].length;e<g;e++)c[d][e].set=a}},fillHorizontal:{available:function(b,c){return 1<c.length&&1<c[0].length},option:function(b){return b.i18n("autoFill.fillHorizontal","Fill cells horizontally")},execute:function(b,c){for(var a=0,d=c.length;a<d;a++)for(var f=0,e=c[a].length;f<e;f++)c[a][f].set=c[a][0].data}},fillVertical:{available:function(b,c){return 1<c.length},option:function(b){return b.i18n("autoFill.fillVertical","Fill cells vertically")},execute:function(b,c){for(var a=
0,d=c.length;a<d;a++)for(var e=0,g=c[a].length;e<g;e++)c[a][e].set=c[0][e].data}},cancel:{available:function(){return!1},option:function(b){return b.i18n("autoFill.cancel","Cancel")},execute:function(){return!1}}};k.version="2.3.5";k.defaults={alwaysAsk:!1,focus:null,columns:"",enable:!0,update:null,editor:null,vertical:!0,horizontal:!0};k.classes={btn:"btn"};var m=e.fn.dataTable.Api;m.register("autoFill()",function(){return this});m.register("autoFill().enabled()",function(){var b=this.context[0];
return b.autoFill?b.autoFill.enabled():!1});m.register("autoFill().enable()",function(b){return this.iterator("table",function(c){c.autoFill&&c.autoFill.enable(b)})});m.register("autoFill().disable()",function(){return this.iterator("table",function(b){b.autoFill&&b.autoFill.disable()})});e(i).on("preInit.dt.autofill",function(b,c){if("dt"===b.namespace){var a=c.oInit.autoFill,d=l.defaults.autoFill;if(a||d)d=e.extend({},a,d),!1!==a&&new k(c,d)}});l.AutoFill=k;return l.AutoFill=k});

/*!
 Bootstrap integration for DataTables' AutoFill
 ©2015 SpryMedia Ltd - datatables.net/license
*/
(function(a){"function"===typeof define&&define.amd?define(["jquery","datatables.net-bs4","datatables.net-autofill"],function(b){return a(b,window,document)}):"object"===typeof exports?module.exports=function(b,c){b||(b=window);if(!c||!c.fn.dataTable)c=require("datatables.net-bs4")(b,c).$;c.fn.dataTable.AutoFill||require("datatables.net-autofill")(b,c);return a(c,b,b.document)}:a(jQuery,window,document)})(function(a){a=a.fn.dataTable;a.AutoFill.classes.btn="btn btn-primary";return a});

/*!

JSZip v3.6.0 - A JavaScript class for generating and reading zip files
<http://stuartk.com/jszip>

(c) 2009-2016 Stuart Knightley <stuart [at] stuartk.com>
Dual licenced under the MIT license or GPLv3. See https://raw.github.com/Stuk/jszip/master/LICENSE.markdown.

JSZip uses the library pako released under the MIT license :
https://github.com/nodeca/pako/blob/master/LICENSE
*/

!function(e){if("object"==typeof exports&&"undefined"!=typeof module)module.exports=e();else if("function"==typeof define&&define.amd)define([],e);else{("undefined"!=typeof window?window:"undefined"!=typeof global?global:"undefined"!=typeof self?self:this).JSZip=e()}}(function(){return function s(a,o,u){function h(r,e){if(!o[r]){if(!a[r]){var t="function"==typeof require&&require;if(!e&&t)return t(r,!0);if(f)return f(r,!0);var n=new Error("Cannot find module '"+r+"'");throw n.code="MODULE_NOT_FOUND",n}var i=o[r]={exports:{}};a[r][0].call(i.exports,function(e){var t=a[r][1][e];return h(t||e)},i,i.exports,s,a,o,u)}return o[r].exports}for(var f="function"==typeof require&&require,e=0;e<u.length;e++)h(u[e]);return h}({1:[function(l,t,n){(function(r){!function(e){"object"==typeof n&&void 0!==t?t.exports=e():("undefined"!=typeof window?window:void 0!==r?r:"undefined"!=typeof self?self:this).JSZip=e()}(function(){return function s(a,o,u){function h(t,e){if(!o[t]){if(!a[t]){var r="function"==typeof l&&l;if(!e&&r)return r(t,!0);if(f)return f(t,!0);var n=new Error("Cannot find module '"+t+"'");throw n.code="MODULE_NOT_FOUND",n}var i=o[t]={exports:{}};a[t][0].call(i.exports,function(e){return h(a[t][1][e]||e)},i,i.exports,s,a,o,u)}return o[t].exports}for(var f="function"==typeof l&&l,e=0;e<u.length;e++)h(u[e]);return h}({1:[function(l,t,n){(function(r){!function(e){"object"==typeof n&&void 0!==t?t.exports=e():("undefined"!=typeof window?window:void 0!==r?r:"undefined"!=typeof self?self:this).JSZip=e()}(function(){return function s(a,o,u){function h(t,e){if(!o[t]){if(!a[t]){var r="function"==typeof l&&l;if(!e&&r)return r(t,!0);if(f)return f(t,!0);var n=new Error("Cannot find module '"+t+"'");throw n.code="MODULE_NOT_FOUND",n}var i=o[t]={exports:{}};a[t][0].call(i.exports,function(e){return h(a[t][1][e]||e)},i,i.exports,s,a,o,u)}return o[t].exports}for(var f="function"==typeof l&&l,e=0;e<u.length;e++)h(u[e]);return h}({1:[function(l,t,n){(function(r){!function(e){"object"==typeof n&&void 0!==t?t.exports=e():("undefined"!=typeof window?window:void 0!==r?r:"undefined"!=typeof self?self:this).JSZip=e()}(function(){return function s(a,o,u){function h(t,e){if(!o[t]){if(!a[t]){var r="function"==typeof l&&l;if(!e&&r)return r(t,!0);if(f)return f(t,!0);var n=new Error("Cannot find module '"+t+"'");throw n.code="MODULE_NOT_FOUND",n}var i=o[t]={exports:{}};a[t][0].call(i.exports,function(e){return h(a[t][1][e]||e)},i,i.exports,s,a,o,u)}return o[t].exports}for(var f="function"==typeof l&&l,e=0;e<u.length;e++)h(u[e]);return h}({1:[function(l,t,n){(function(r){!function(e){"object"==typeof n&&void 0!==t?t.exports=e():("undefined"!=typeof window?window:void 0!==r?r:"undefined"!=typeof self?self:this).JSZip=e()}(function(){return function s(a,o,u){function h(t,e){if(!o[t]){if(!a[t]){var r="function"==typeof l&&l;if(!e&&r)return r(t,!0);if(f)return f(t,!0);var n=new Error("Cannot find module '"+t+"'");throw n.code="MODULE_NOT_FOUND",n}var i=o[t]={exports:{}};a[t][0].call(i.exports,function(e){return h(a[t][1][e]||e)},i,i.exports,s,a,o,u)}return o[t].exports}for(var f="function"==typeof l&&l,e=0;e<u.length;e++)h(u[e]);return h}({1:[function(l,t,n){(function(r){!function(e){"object"==typeof n&&void 0!==t?t.exports=e():("undefined"!=typeof window?window:void 0!==r?r:"undefined"!=typeof self?self:this).JSZip=e()}(function(){return function s(a,o,u){function h(t,e){if(!o[t]){if(!a[t]){var r="function"==typeof l&&l;if(!e&&r)return r(t,!0);if(f)return f(t,!0);var n=new Error("Cannot find module '"+t+"'");throw n.code="MODULE_NOT_FOUND",n}var i=o[t]={exports:{}};a[t][0].call(i.exports,function(e){return h(a[t][1][e]||e)},i,i.exports,s,a,o,u)}return o[t].exports}for(var f="function"==typeof l&&l,e=0;e<u.length;e++)h(u[e]);return h}({1:[function(e,t,r){"use strict";var c=e("./utils"),l=e("./support"),p="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";r.encode=function(e){for(var t,r,n,i,s,a,o,u=[],h=0,f=e.length,l=f,d="string"!==c.getTypeOf(e);h<e.length;)l=f-h,n=d?(t=e[h++],r=h<f?e[h++]:0,h<f?e[h++]:0):(t=e.charCodeAt(h++),r=h<f?e.charCodeAt(h++):0,h<f?e.charCodeAt(h++):0),i=t>>2,s=(3&t)<<4|r>>4,a=1<l?(15&r)<<2|n>>6:64,o=2<l?63&n:64,u.push(p.charAt(i)+p.charAt(s)+p.charAt(a)+p.charAt(o));return u.join("")},r.decode=function(e){var t,r,n,i,s,a,o=0,u=0;if("data:"===e.substr(0,"data:".length))throw new Error("Invalid base64 input, it looks like a data url.");var h,f=3*(e=e.replace(/[^A-Za-z0-9\+\/\=]/g,"")).length/4;if(e.charAt(e.length-1)===p.charAt(64)&&f--,e.charAt(e.length-2)===p.charAt(64)&&f--,f%1!=0)throw new Error("Invalid base64 input, bad content length.");for(h=l.uint8array?new Uint8Array(0|f):new Array(0|f);o<e.length;)t=p.indexOf(e.charAt(o++))<<2|(i=p.indexOf(e.charAt(o++)))>>4,r=(15&i)<<4|(s=p.indexOf(e.charAt(o++)))>>2,n=(3&s)<<6|(a=p.indexOf(e.charAt(o++))),h[u++]=t,64!==s&&(h[u++]=r),64!==a&&(h[u++]=n);return h}},{"./support":30,"./utils":32}],2:[function(e,t,r){"use strict";var n=e("./external"),i=e("./stream/DataWorker"),s=e("./stream/Crc32Probe"),a=e("./stream/DataLengthProbe");function o(e,t,r,n,i){this.compressedSize=e,this.uncompressedSize=t,this.crc32=r,this.compression=n,this.compressedContent=i}o.prototype={getContentWorker:function(){var e=new i(n.Promise.resolve(this.compressedContent)).pipe(this.compression.uncompressWorker()).pipe(new a("data_length")),t=this;return e.on("end",function(){if(this.streamInfo.data_length!==t.uncompressedSize)throw new Error("Bug : uncompressed data size mismatch")}),e},getCompressedWorker:function(){return new i(n.Promise.resolve(this.compressedContent)).withStreamInfo("compressedSize",this.compressedSize).withStreamInfo("uncompressedSize",this.uncompressedSize).withStreamInfo("crc32",this.crc32).withStreamInfo("compression",this.compression)}},o.createWorkerFrom=function(e,t,r){return e.pipe(new s).pipe(new a("uncompressedSize")).pipe(t.compressWorker(r)).pipe(new a("compressedSize")).withStreamInfo("compression",t)},t.exports=o},{"./external":6,"./stream/Crc32Probe":25,"./stream/DataLengthProbe":26,"./stream/DataWorker":27}],3:[function(e,t,r){"use strict";var n=e("./stream/GenericWorker");r.STORE={magic:"\0\0",compressWorker:function(e){return new n("STORE compression")},uncompressWorker:function(){return new n("STORE decompression")}},r.DEFLATE=e("./flate")},{"./flate":7,"./stream/GenericWorker":28}],4:[function(e,t,r){"use strict";var n=e("./utils"),a=function(){for(var e,t=[],r=0;r<256;r++){e=r;for(var n=0;n<8;n++)e=1&e?3988292384^e>>>1:e>>>1;t[r]=e}return t}();t.exports=function(e,t){return void 0!==e&&e.length?"string"!==n.getTypeOf(e)?function(e,t,r){var n=a,i=0+r;e^=-1;for(var s=0;s<i;s++)e=e>>>8^n[255&(e^t[s])];return-1^e}(0|t,e,e.length):function(e,t,r){var n=a,i=0+r;e^=-1;for(var s=0;s<i;s++)e=e>>>8^n[255&(e^t.charCodeAt(s))];return-1^e}(0|t,e,e.length):0}},{"./utils":32}],5:[function(e,t,r){"use strict";r.base64=!1,r.binary=!1,r.dir=!1,r.createFolders=!0,r.date=null,r.compression=null,r.compressionOptions=null,r.comment=null,r.unixPermissions=null,r.dosPermissions=null},{}],6:[function(e,t,r){"use strict";var n;n="undefined"!=typeof Promise?Promise:e("lie"),t.exports={Promise:n}},{lie:37}],7:[function(e,t,r){"use strict";var n="undefined"!=typeof Uint8Array&&"undefined"!=typeof Uint16Array&&"undefined"!=typeof Uint32Array,i=e("pako"),s=e("./utils"),a=e("./stream/GenericWorker"),o=n?"uint8array":"array";function u(e,t){a.call(this,"FlateWorker/"+e),this._pako=null,this._pakoAction=e,this._pakoOptions=t,this.meta={}}r.magic="\b\0",s.inherits(u,a),u.prototype.processChunk=function(e){this.meta=e.meta,null===this._pako&&this._createPako(),this._pako.push(s.transformTo(o,e.data),!1)},u.prototype.flush=function(){a.prototype.flush.call(this),null===this._pako&&this._createPako(),this._pako.push([],!0)},u.prototype.cleanUp=function(){a.prototype.cleanUp.call(this),this._pako=null},u.prototype._createPako=function(){this._pako=new i[this._pakoAction]({raw:!0,level:this._pakoOptions.level||-1});var t=this;this._pako.onData=function(e){t.push({data:e,meta:t.meta})}},r.compressWorker=function(e){return new u("Deflate",e)},r.uncompressWorker=function(){return new u("Inflate",{})}},{"./stream/GenericWorker":28,"./utils":32,pako:38}],8:[function(e,t,r){"use strict";function I(e,t){var r,n="";for(r=0;r<t;r++)n+=String.fromCharCode(255&e),e>>>=8;return n}function i(e,t,r,n,i,s){var a,o,u=e.file,h=e.compression,f=s!==B.utf8encode,l=O.transformTo("string",s(u.name)),d=O.transformTo("string",B.utf8encode(u.name)),c=u.comment,p=O.transformTo("string",s(c)),m=O.transformTo("string",B.utf8encode(c)),_=d.length!==u.name.length,g=m.length!==c.length,v="",b="",w="",y=u.dir,k=u.date,x={crc32:0,compressedSize:0,uncompressedSize:0};t&&!r||(x.crc32=e.crc32,x.compressedSize=e.compressedSize,x.uncompressedSize=e.uncompressedSize);var S=0;t&&(S|=8),f||!_&&!g||(S|=2048);var z,E=0,C=0;y&&(E|=16),"UNIX"===i?(C=798,E|=((z=u.unixPermissions)||(z=y?16893:33204),(65535&z)<<16)):(C=20,E|=63&(u.dosPermissions||0)),a=k.getUTCHours(),a<<=6,a|=k.getUTCMinutes(),a<<=5,a|=k.getUTCSeconds()/2,o=k.getUTCFullYear()-1980,o<<=4,o|=k.getUTCMonth()+1,o<<=5,o|=k.getUTCDate(),_&&(v+="up"+I((b=I(1,1)+I(T(l),4)+d).length,2)+b),g&&(v+="uc"+I((w=I(1,1)+I(T(p),4)+m).length,2)+w);var A="";return A+="\n\0",A+=I(S,2),A+=h.magic,A+=I(a,2),A+=I(o,2),A+=I(x.crc32,4),A+=I(x.compressedSize,4),A+=I(x.uncompressedSize,4),A+=I(l.length,2),A+=I(v.length,2),{fileRecord:R.LOCAL_FILE_HEADER+A+l+v,dirRecord:R.CENTRAL_FILE_HEADER+I(C,2)+A+I(p.length,2)+"\0\0\0\0"+I(E,4)+I(n,4)+l+v+p}}var O=e("../utils"),s=e("../stream/GenericWorker"),B=e("../utf8"),T=e("../crc32"),R=e("../signature");function n(e,t,r,n){s.call(this,"ZipFileWorker"),this.bytesWritten=0,this.zipComment=t,this.zipPlatform=r,this.encodeFileName=n,this.streamFiles=e,this.accumulate=!1,this.contentBuffer=[],this.dirRecords=[],this.currentSourceOffset=0,this.entriesCount=0,this.currentFile=null,this._sources=[]}O.inherits(n,s),n.prototype.push=function(e){var t=e.meta.percent||0,r=this.entriesCount,n=this._sources.length;this.accumulate?this.contentBuffer.push(e):(this.bytesWritten+=e.data.length,s.prototype.push.call(this,{data:e.data,meta:{currentFile:this.currentFile,percent:r?(t+100*(r-n-1))/r:100}}))},n.prototype.openedSource=function(e){this.currentSourceOffset=this.bytesWritten,this.currentFile=e.file.name;var t=this.streamFiles&&!e.file.dir;if(t){var r=i(e,t,!1,this.currentSourceOffset,this.zipPlatform,this.encodeFileName);this.push({data:r.fileRecord,meta:{percent:0}})}else this.accumulate=!0},n.prototype.closedSource=function(e){this.accumulate=!1;var t,r=this.streamFiles&&!e.file.dir,n=i(e,r,!0,this.currentSourceOffset,this.zipPlatform,this.encodeFileName);if(this.dirRecords.push(n.dirRecord),r)this.push({data:(t=e,R.DATA_DESCRIPTOR+I(t.crc32,4)+I(t.compressedSize,4)+I(t.uncompressedSize,4)),meta:{percent:100}});else for(this.push({data:n.fileRecord,meta:{percent:0}});this.contentBuffer.length;)this.push(this.contentBuffer.shift());this.currentFile=null},n.prototype.flush=function(){for(var e=this.bytesWritten,t=0;t<this.dirRecords.length;t++)this.push({data:this.dirRecords[t],meta:{percent:100}});var r,n,i,s,a,o,u=this.bytesWritten-e,h=(r=this.dirRecords.length,n=u,i=e,s=this.zipComment,a=this.encodeFileName,o=O.transformTo("string",a(s)),R.CENTRAL_DIRECTORY_END+"\0\0\0\0"+I(r,2)+I(r,2)+I(n,4)+I(i,4)+I(o.length,2)+o);this.push({data:h,meta:{percent:100}})},n.prototype.prepareNextSource=function(){this.previous=this._sources.shift(),this.openedSource(this.previous.streamInfo),this.isPaused?this.previous.pause():this.previous.resume()},n.prototype.registerPrevious=function(e){this._sources.push(e);var t=this;return e.on("data",function(e){t.processChunk(e)}),e.on("end",function(){t.closedSource(t.previous.streamInfo),t._sources.length?t.prepareNextSource():t.end()}),e.on("error",function(e){t.error(e)}),this},n.prototype.resume=function(){return!!s.prototype.resume.call(this)&&(!this.previous&&this._sources.length?(this.prepareNextSource(),!0):this.previous||this._sources.length||this.generatedError?void 0:(this.end(),!0))},n.prototype.error=function(e){var t=this._sources;if(!s.prototype.error.call(this,e))return!1;for(var r=0;r<t.length;r++)try{t[r].error(e)}catch(e){}return!0},n.prototype.lock=function(){s.prototype.lock.call(this);for(var e=this._sources,t=0;t<e.length;t++)e[t].lock()},t.exports=n},{"../crc32":4,"../signature":23,"../stream/GenericWorker":28,"../utf8":31,"../utils":32}],9:[function(e,t,r){"use strict";var h=e("../compressions"),n=e("./ZipFileWorker");r.generateWorker=function(e,a,t){var o=new n(a.streamFiles,t,a.platform,a.encodeFileName),u=0;try{e.forEach(function(e,t){u++;var r=function(e,t){var r=e||t,n=h[r];if(!n)throw new Error(r+" is not a valid compression method !");return n}(t.options.compression,a.compression),n=t.options.compressionOptions||a.compressionOptions||{},i=t.dir,s=t.date;t._compressWorker(r,n).withStreamInfo("file",{name:e,dir:i,date:s,comment:t.comment||"",unixPermissions:t.unixPermissions,dosPermissions:t.dosPermissions}).pipe(o)}),o.entriesCount=u}catch(e){o.error(e)}return o}},{"../compressions":3,"./ZipFileWorker":8}],10:[function(e,t,r){"use strict";function n(){if(!(this instanceof n))return new n;if(arguments.length)throw new Error("The constructor with parameters has been removed in JSZip 3.0, please check the upgrade guide.");this.files={},this.comment=null,this.root="",this.clone=function(){var e=new n;for(var t in this)"function"!=typeof this[t]&&(e[t]=this[t]);return e}}(n.prototype=e("./object")).loadAsync=e("./load"),n.support=e("./support"),n.defaults=e("./defaults"),n.version="3.5.0",n.loadAsync=function(e,t){return(new n).loadAsync(e,t)},n.external=e("./external"),t.exports=n},{"./defaults":5,"./external":6,"./load":11,"./object":15,"./support":30}],11:[function(e,t,r){"use strict";var n=e("./utils"),i=e("./external"),o=e("./utf8"),u=e("./zipEntries"),s=e("./stream/Crc32Probe"),h=e("./nodejsUtils");function f(n){return new i.Promise(function(e,t){var r=n.decompressed.getContentWorker().pipe(new s);r.on("error",function(e){t(e)}).on("end",function(){r.streamInfo.crc32!==n.decompressed.crc32?t(new Error("Corrupted zip : CRC32 mismatch")):e()}).resume()})}t.exports=function(e,s){var a=this;return s=n.extend(s||{},{base64:!1,checkCRC32:!1,optimizedBinaryString:!1,createFolders:!1,decodeFileName:o.utf8decode}),h.isNode&&h.isStream(e)?i.Promise.reject(new Error("JSZip can't accept a stream when loading a zip file.")):n.prepareContent("the loaded zip file",e,!0,s.optimizedBinaryString,s.base64).then(function(e){var t=new u(s);return t.load(e),t}).then(function(e){var t=[i.Promise.resolve(e)],r=e.files;if(s.checkCRC32)for(var n=0;n<r.length;n++)t.push(f(r[n]));return i.Promise.all(t)}).then(function(e){for(var t=e.shift(),r=t.files,n=0;n<r.length;n++){var i=r[n];a.file(i.fileNameStr,i.decompressed,{binary:!0,optimizedBinaryString:!0,date:i.date,dir:i.dir,comment:i.fileCommentStr.length?i.fileCommentStr:null,unixPermissions:i.unixPermissions,dosPermissions:i.dosPermissions,createFolders:s.createFolders})}return t.zipComment.length&&(a.comment=t.zipComment),a})}},{"./external":6,"./nodejsUtils":14,"./stream/Crc32Probe":25,"./utf8":31,"./utils":32,"./zipEntries":33}],12:[function(e,t,r){"use strict";var n=e("../utils"),i=e("../stream/GenericWorker");function s(e,t){i.call(this,"Nodejs stream input adapter for "+e),this._upstreamEnded=!1,this._bindStream(t)}n.inherits(s,i),s.prototype._bindStream=function(e){var t=this;(this._stream=e).pause(),e.on("data",function(e){t.push({data:e,meta:{percent:0}})}).on("error",function(e){t.isPaused?this.generatedError=e:t.error(e)}).on("end",function(){t.isPaused?t._upstreamEnded=!0:t.end()})},s.prototype.pause=function(){return!!i.prototype.pause.call(this)&&(this._stream.pause(),!0)},s.prototype.resume=function(){return!!i.prototype.resume.call(this)&&(this._upstreamEnded?this.end():this._stream.resume(),!0)},t.exports=s},{"../stream/GenericWorker":28,"../utils":32}],13:[function(e,t,r){"use strict";var i=e("readable-stream").Readable;function n(e,t,r){i.call(this,t),this._helper=e;var n=this;e.on("data",function(e,t){n.push(e)||n._helper.pause(),r&&r(t)}).on("error",function(e){n.emit("error",e)}).on("end",function(){n.push(null)})}e("../utils").inherits(n,i),n.prototype._read=function(){this._helper.resume()},t.exports=n},{"../utils":32,"readable-stream":16}],14:[function(e,t,r){"use strict";t.exports={isNode:"undefined"!=typeof Buffer,newBufferFrom:function(e,t){if(Buffer.from&&Buffer.from!==Uint8Array.from)return Buffer.from(e,t);if("number"==typeof e)throw new Error('The "data" argument must not be a number');return new Buffer(e,t)},allocBuffer:function(e){if(Buffer.alloc)return Buffer.alloc(e);var t=new Buffer(e);return t.fill(0),t},isBuffer:function(e){return Buffer.isBuffer(e)},isStream:function(e){return e&&"function"==typeof e.on&&"function"==typeof e.pause&&"function"==typeof e.resume}}},{}],15:[function(e,t,r){"use strict";function s(e,t,r){var n,i=f.getTypeOf(t),s=f.extend(r||{},d);s.date=s.date||new Date,null!==s.compression&&(s.compression=s.compression.toUpperCase()),"string"==typeof s.unixPermissions&&(s.unixPermissions=parseInt(s.unixPermissions,8)),s.unixPermissions&&16384&s.unixPermissions&&(s.dir=!0),s.dosPermissions&&16&s.dosPermissions&&(s.dir=!0),s.dir&&(e=h(e)),s.createFolders&&(n=function(e){"/"===e.slice(-1)&&(e=e.substring(0,e.length-1));var t=e.lastIndexOf("/");return 0<t?e.substring(0,t):""}(e))&&g.call(this,n,!0);var a,o="string"===i&&!1===s.binary&&!1===s.base64;r&&void 0!==r.binary||(s.binary=!o),(t instanceof c&&0===t.uncompressedSize||s.dir||!t||0===t.length)&&(s.base64=!1,s.binary=!0,t="",s.compression="STORE",i="string"),a=t instanceof c||t instanceof l?t:m.isNode&&m.isStream(t)?new _(e,t):f.prepareContent(e,t,s.binary,s.optimizedBinaryString,s.base64);var u=new p(e,a,s);this.files[e]=u}function h(e){return"/"!==e.slice(-1)&&(e+="/"),e}var i=e("./utf8"),f=e("./utils"),l=e("./stream/GenericWorker"),a=e("./stream/StreamHelper"),d=e("./defaults"),c=e("./compressedObject"),p=e("./zipObject"),o=e("./generate"),m=e("./nodejsUtils"),_=e("./nodejs/NodejsStreamInputAdapter"),g=function(e,t){return t=void 0!==t?t:d.createFolders,e=h(e),this.files[e]||s.call(this,e,null,{dir:!0,createFolders:t}),this.files[e]};function u(e){return"[object RegExp]"===Object.prototype.toString.call(e)}var n={load:function(){throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.")},forEach:function(e){var t,r,n;for(t in this.files)this.files.hasOwnProperty(t)&&(n=this.files[t],(r=t.slice(this.root.length,t.length))&&t.slice(0,this.root.length)===this.root&&e(r,n))},filter:function(r){var n=[];return this.forEach(function(e,t){r(e,t)&&n.push(t)}),n},file:function(e,t,r){if(1!==arguments.length)return e=this.root+e,s.call(this,e,t,r),this;if(u(e)){var n=e;return this.filter(function(e,t){return!t.dir&&n.test(e)})}var i=this.files[this.root+e];return i&&!i.dir?i:null},folder:function(r){if(!r)return this;if(u(r))return this.filter(function(e,t){return t.dir&&r.test(e)});var e=this.root+r,t=g.call(this,e),n=this.clone();return n.root=t.name,n},remove:function(r){r=this.root+r;var e=this.files[r];if(e||("/"!==r.slice(-1)&&(r+="/"),e=this.files[r]),e&&!e.dir)delete this.files[r];else for(var t=this.filter(function(e,t){return t.name.slice(0,r.length)===r}),n=0;n<t.length;n++)delete this.files[t[n].name];return this},generate:function(e){throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.")},generateInternalStream:function(e){var t,r={};try{if((r=f.extend(e||{},{streamFiles:!1,compression:"STORE",compressionOptions:null,type:"",platform:"DOS",comment:null,mimeType:"application/zip",encodeFileName:i.utf8encode})).type=r.type.toLowerCase(),r.compression=r.compression.toUpperCase(),"binarystring"===r.type&&(r.type="string"),!r.type)throw new Error("No output type specified.");f.checkSupport(r.type),"darwin"!==r.platform&&"freebsd"!==r.platform&&"linux"!==r.platform&&"sunos"!==r.platform||(r.platform="UNIX"),"win32"===r.platform&&(r.platform="DOS");var n=r.comment||this.comment||"";t=o.generateWorker(this,r,n)}catch(e){(t=new l("error")).error(e)}return new a(t,r.type||"string",r.mimeType)},generateAsync:function(e,t){return this.generateInternalStream(e).accumulate(t)},generateNodeStream:function(e,t){return(e=e||{}).type||(e.type="nodebuffer"),this.generateInternalStream(e).toNodejsStream(t)}};t.exports=n},{"./compressedObject":2,"./defaults":5,"./generate":9,"./nodejs/NodejsStreamInputAdapter":12,"./nodejsUtils":14,"./stream/GenericWorker":28,"./stream/StreamHelper":29,"./utf8":31,"./utils":32,"./zipObject":35}],16:[function(e,t,r){t.exports=e("stream")},{stream:void 0}],17:[function(e,t,r){"use strict";var n=e("./DataReader");function i(e){n.call(this,e);for(var t=0;t<this.data.length;t++)e[t]=255&e[t]}e("../utils").inherits(i,n),i.prototype.byteAt=function(e){return this.data[this.zero+e]},i.prototype.lastIndexOfSignature=function(e){for(var t=e.charCodeAt(0),r=e.charCodeAt(1),n=e.charCodeAt(2),i=e.charCodeAt(3),s=this.length-4;0<=s;--s)if(this.data[s]===t&&this.data[s+1]===r&&this.data[s+2]===n&&this.data[s+3]===i)return s-this.zero;return-1},i.prototype.readAndCheckSignature=function(e){var t=e.charCodeAt(0),r=e.charCodeAt(1),n=e.charCodeAt(2),i=e.charCodeAt(3),s=this.readData(4);return t===s[0]&&r===s[1]&&n===s[2]&&i===s[3]},i.prototype.readData=function(e){if(this.checkOffset(e),0===e)return[];var t=this.data.slice(this.zero+this.index,this.zero+this.index+e);return this.index+=e,t},t.exports=i},{"../utils":32,"./DataReader":18}],18:[function(e,t,r){"use strict";var n=e("../utils");function i(e){this.data=e,this.length=e.length,this.index=0,this.zero=0}i.prototype={checkOffset:function(e){this.checkIndex(this.index+e)},checkIndex:function(e){if(this.length<this.zero+e||e<0)throw new Error("End of data reached (data length = "+this.length+", asked index = "+e+"). Corrupted zip ?")},setIndex:function(e){this.checkIndex(e),this.index=e},skip:function(e){this.setIndex(this.index+e)},byteAt:function(e){},readInt:function(e){var t,r=0;for(this.checkOffset(e),t=this.index+e-1;t>=this.index;t--)r=(r<<8)+this.byteAt(t);return this.index+=e,r},readString:function(e){return n.transformTo("string",this.readData(e))},readData:function(e){},lastIndexOfSignature:function(e){},readAndCheckSignature:function(e){},readDate:function(){var e=this.readInt(4);return new Date(Date.UTC(1980+(e>>25&127),(e>>21&15)-1,e>>16&31,e>>11&31,e>>5&63,(31&e)<<1))}},t.exports=i},{"../utils":32}],19:[function(e,t,r){"use strict";var n=e("./Uint8ArrayReader");function i(e){n.call(this,e)}e("../utils").inherits(i,n),i.prototype.readData=function(e){this.checkOffset(e);var t=this.data.slice(this.zero+this.index,this.zero+this.index+e);return this.index+=e,t},t.exports=i},{"../utils":32,"./Uint8ArrayReader":21}],20:[function(e,t,r){"use strict";var n=e("./DataReader");function i(e){n.call(this,e)}e("../utils").inherits(i,n),i.prototype.byteAt=function(e){return this.data.charCodeAt(this.zero+e)},i.prototype.lastIndexOfSignature=function(e){return this.data.lastIndexOf(e)-this.zero},i.prototype.readAndCheckSignature=function(e){return e===this.readData(4)},i.prototype.readData=function(e){this.checkOffset(e);var t=this.data.slice(this.zero+this.index,this.zero+this.index+e);return this.index+=e,t},t.exports=i},{"../utils":32,"./DataReader":18}],21:[function(e,t,r){"use strict";var n=e("./ArrayReader");function i(e){n.call(this,e)}e("../utils").inherits(i,n),i.prototype.readData=function(e){if(this.checkOffset(e),0===e)return new Uint8Array(0);var t=this.data.subarray(this.zero+this.index,this.zero+this.index+e);return this.index+=e,t},t.exports=i},{"../utils":32,"./ArrayReader":17}],22:[function(e,t,r){"use strict";var n=e("../utils"),i=e("../support"),s=e("./ArrayReader"),a=e("./StringReader"),o=e("./NodeBufferReader"),u=e("./Uint8ArrayReader");t.exports=function(e){var t=n.getTypeOf(e);return n.checkSupport(t),"string"!==t||i.uint8array?"nodebuffer"===t?new o(e):i.uint8array?new u(n.transformTo("uint8array",e)):new s(n.transformTo("array",e)):new a(e)}},{"../support":30,"../utils":32,"./ArrayReader":17,"./NodeBufferReader":19,"./StringReader":20,"./Uint8ArrayReader":21}],23:[function(e,t,r){"use strict";r.LOCAL_FILE_HEADER="PK",r.CENTRAL_FILE_HEADER="PK",r.CENTRAL_DIRECTORY_END="PK",r.ZIP64_CENTRAL_DIRECTORY_LOCATOR="PK",r.ZIP64_CENTRAL_DIRECTORY_END="PK",r.DATA_DESCRIPTOR="PK\b"},{}],24:[function(e,t,r){"use strict";var n=e("./GenericWorker"),i=e("../utils");function s(e){n.call(this,"ConvertWorker to "+e),this.destType=e}i.inherits(s,n),s.prototype.processChunk=function(e){this.push({data:i.transformTo(this.destType,e.data),meta:e.meta})},t.exports=s},{"../utils":32,"./GenericWorker":28}],25:[function(e,t,r){"use strict";var n=e("./GenericWorker"),i=e("../crc32");function s(){n.call(this,"Crc32Probe"),this.withStreamInfo("crc32",0)}e("../utils").inherits(s,n),s.prototype.processChunk=function(e){this.streamInfo.crc32=i(e.data,this.streamInfo.crc32||0),this.push(e)},t.exports=s},{"../crc32":4,"../utils":32,"./GenericWorker":28}],26:[function(e,t,r){"use strict";var n=e("../utils"),i=e("./GenericWorker");function s(e){i.call(this,"DataLengthProbe for "+e),this.propName=e,this.withStreamInfo(e,0)}n.inherits(s,i),s.prototype.processChunk=function(e){if(e){var t=this.streamInfo[this.propName]||0;this.streamInfo[this.propName]=t+e.data.length}i.prototype.processChunk.call(this,e)},t.exports=s},{"../utils":32,"./GenericWorker":28}],27:[function(e,t,r){"use strict";var n=e("../utils"),i=e("./GenericWorker");function s(e){i.call(this,"DataWorker");var t=this;this.dataIsReady=!1,this.index=0,this.max=0,this.data=null,this.type="",this._tickScheduled=!1,e.then(function(e){t.dataIsReady=!0,t.data=e,t.max=e&&e.length||0,t.type=n.getTypeOf(e),t.isPaused||t._tickAndRepeat()},function(e){t.error(e)})}n.inherits(s,i),s.prototype.cleanUp=function(){i.prototype.cleanUp.call(this),this.data=null},s.prototype.resume=function(){return!!i.prototype.resume.call(this)&&(!this._tickScheduled&&this.dataIsReady&&(this._tickScheduled=!0,n.delay(this._tickAndRepeat,[],this)),!0)},s.prototype._tickAndRepeat=function(){this._tickScheduled=!1,this.isPaused||this.isFinished||(this._tick(),this.isFinished||(n.delay(this._tickAndRepeat,[],this),this._tickScheduled=!0))},s.prototype._tick=function(){if(this.isPaused||this.isFinished)return!1;var e=null,t=Math.min(this.max,this.index+16384);if(this.index>=this.max)return this.end();switch(this.type){case"string":e=this.data.substring(this.index,t);break;case"uint8array":e=this.data.subarray(this.index,t);break;case"array":case"nodebuffer":e=this.data.slice(this.index,t)}return this.index=t,this.push({data:e,meta:{percent:this.max?this.index/this.max*100:0}})},t.exports=s},{"../utils":32,"./GenericWorker":28}],28:[function(e,t,r){"use strict";function n(e){this.name=e||"default",this.streamInfo={},this.generatedError=null,this.extraStreamInfo={},this.isPaused=!0,this.isFinished=!1,this.isLocked=!1,this._listeners={data:[],end:[],error:[]},this.previous=null}n.prototype={push:function(e){this.emit("data",e)},end:function(){if(this.isFinished)return!1;this.flush();try{this.emit("end"),this.cleanUp(),this.isFinished=!0}catch(e){this.emit("error",e)}return!0},error:function(e){return!this.isFinished&&(this.isPaused?this.generatedError=e:(this.isFinished=!0,this.emit("error",e),this.previous&&this.previous.error(e),this.cleanUp()),!0)},on:function(e,t){return this._listeners[e].push(t),this},cleanUp:function(){this.streamInfo=this.generatedError=this.extraStreamInfo=null,this._listeners=[]},emit:function(e,t){if(this._listeners[e])for(var r=0;r<this._listeners[e].length;r++)this._listeners[e][r].call(this,t)},pipe:function(e){return e.registerPrevious(this)},registerPrevious:function(e){if(this.isLocked)throw new Error("The stream '"+this+"' has already been used.");this.streamInfo=e.streamInfo,this.mergeStreamInfo(),this.previous=e;var t=this;return e.on("data",function(e){t.processChunk(e)}),e.on("end",function(){t.end()}),e.on("error",function(e){t.error(e)}),this},pause:function(){return!this.isPaused&&!this.isFinished&&(this.isPaused=!0,this.previous&&this.previous.pause(),!0)},resume:function(){if(!this.isPaused||this.isFinished)return!1;var e=this.isPaused=!1;return this.generatedError&&(this.error(this.generatedError),e=!0),this.previous&&this.previous.resume(),!e},flush:function(){},processChunk:function(e){this.push(e)},withStreamInfo:function(e,t){return this.extraStreamInfo[e]=t,this.mergeStreamInfo(),this},mergeStreamInfo:function(){for(var e in this.extraStreamInfo)this.extraStreamInfo.hasOwnProperty(e)&&(this.streamInfo[e]=this.extraStreamInfo[e])},lock:function(){if(this.isLocked)throw new Error("The stream '"+this+"' has already been used.");this.isLocked=!0,this.previous&&this.previous.lock()},toString:function(){var e="Worker "+this.name;return this.previous?this.previous+" -> "+e:e}},t.exports=n},{}],29:[function(e,t,r){"use strict";var h=e("../utils"),i=e("./ConvertWorker"),s=e("./GenericWorker"),f=e("../base64"),n=e("../support"),a=e("../external"),o=null;if(n.nodestream)try{o=e("../nodejs/NodejsStreamOutputAdapter")}catch(e){}function u(e,t,r){var n=t;switch(t){case"blob":case"arraybuffer":n="uint8array";break;case"base64":n="string"}try{this._internalType=n,this._outputType=t,this._mimeType=r,h.checkSupport(n),this._worker=e.pipe(new i(n)),e.lock()}catch(e){this._worker=new s("error"),this._worker.error(e)}}u.prototype={accumulate:function(e){return o=this,u=e,new a.Promise(function(t,r){var n=[],i=o._internalType,s=o._outputType,a=o._mimeType;o.on("data",function(e,t){n.push(e),u&&u(t)}).on("error",function(e){n=[],r(e)}).on("end",function(){try{var e=function(e,t,r){switch(e){case"blob":return h.newBlob(h.transformTo("arraybuffer",t),r);case"base64":return f.encode(t);default:return h.transformTo(e,t)}}(s,function(e,t){var r,n=0,i=null,s=0;for(r=0;r<t.length;r++)s+=t[r].length;switch(e){case"string":return t.join("");case"array":return Array.prototype.concat.apply([],t);case"uint8array":for(i=new Uint8Array(s),r=0;r<t.length;r++)i.set(t[r],n),n+=t[r].length;return i;case"nodebuffer":return Buffer.concat(t);default:throw new Error("concat : unsupported type '"+e+"'")}}(i,n),a);t(e)}catch(e){r(e)}n=[]}).resume()});var o,u},on:function(e,t){var r=this;return"data"===e?this._worker.on(e,function(e){t.call(r,e.data,e.meta)}):this._worker.on(e,function(){h.delay(t,arguments,r)}),this},resume:function(){return h.delay(this._worker.resume,[],this._worker),this},pause:function(){return this._worker.pause(),this},toNodejsStream:function(e){if(h.checkSupport("nodestream"),"nodebuffer"!==this._outputType)throw new Error(this._outputType+" is not supported by this method");return new o(this,{objectMode:"nodebuffer"!==this._outputType},e)}},t.exports=u},{"../base64":1,"../external":6,"../nodejs/NodejsStreamOutputAdapter":13,"../support":30,"../utils":32,"./ConvertWorker":24,"./GenericWorker":28}],30:[function(e,t,r){"use strict";if(r.base64=!0,r.array=!0,r.string=!0,r.arraybuffer="undefined"!=typeof ArrayBuffer&&"undefined"!=typeof Uint8Array,r.nodebuffer="undefined"!=typeof Buffer,r.uint8array="undefined"!=typeof Uint8Array,"undefined"==typeof ArrayBuffer)r.blob=!1;else{var n=new ArrayBuffer(0);try{r.blob=0===new Blob([n],{type:"application/zip"}).size}catch(e){try{var i=new(self.BlobBuilder||self.WebKitBlobBuilder||self.MozBlobBuilder||self.MSBlobBuilder);i.append(n),r.blob=0===i.getBlob("application/zip").size}catch(e){r.blob=!1}}}try{r.nodestream=!!e("readable-stream").Readable}catch(e){r.nodestream=!1}},{"readable-stream":16}],31:[function(e,t,s){"use strict";for(var o=e("./utils"),u=e("./support"),r=e("./nodejsUtils"),n=e("./stream/GenericWorker"),h=new Array(256),i=0;i<256;i++)h[i]=252<=i?6:248<=i?5:240<=i?4:224<=i?3:192<=i?2:1;function a(){n.call(this,"utf-8 decode"),this.leftOver=null}function f(){n.call(this,"utf-8 encode")}h[254]=h[254]=1,s.utf8encode=function(e){return u.nodebuffer?r.newBufferFrom(e,"utf-8"):function(e){var t,r,n,i,s,a=e.length,o=0;for(i=0;i<a;i++)55296==(64512&(r=e.charCodeAt(i)))&&i+1<a&&56320==(64512&(n=e.charCodeAt(i+1)))&&(r=65536+(r-55296<<10)+(n-56320),i++),o+=r<128?1:r<2048?2:r<65536?3:4;for(t=u.uint8array?new Uint8Array(o):new Array(o),i=s=0;s<o;i++)55296==(64512&(r=e.charCodeAt(i)))&&i+1<a&&56320==(64512&(n=e.charCodeAt(i+1)))&&(r=65536+(r-55296<<10)+(n-56320),i++),r<128?t[s++]=r:(r<2048?t[s++]=192|r>>>6:(r<65536?t[s++]=224|r>>>12:(t[s++]=240|r>>>18,t[s++]=128|r>>>12&63),t[s++]=128|r>>>6&63),t[s++]=128|63&r);return t}(e)},s.utf8decode=function(e){return u.nodebuffer?o.transformTo("nodebuffer",e).toString("utf-8"):function(e){var t,r,n,i,s=e.length,a=new Array(2*s);for(t=r=0;t<s;)if((n=e[t++])<128)a[r++]=n;else if(4<(i=h[n]))a[r++]=65533,t+=i-1;else{for(n&=2===i?31:3===i?15:7;1<i&&t<s;)n=n<<6|63&e[t++],i--;1<i?a[r++]=65533:n<65536?a[r++]=n:(n-=65536,a[r++]=55296|n>>10&1023,a[r++]=56320|1023&n)}return a.length!==r&&(a.subarray?a=a.subarray(0,r):a.length=r),o.applyFromCharCode(a)}(e=o.transformTo(u.uint8array?"uint8array":"array",e))},o.inherits(a,n),a.prototype.processChunk=function(e){var t=o.transformTo(u.uint8array?"uint8array":"array",e.data);if(this.leftOver&&this.leftOver.length){if(u.uint8array){var r=t;(t=new Uint8Array(r.length+this.leftOver.length)).set(this.leftOver,0),t.set(r,this.leftOver.length)}else t=this.leftOver.concat(t);this.leftOver=null}var n=function(e,t){var r;for((t=t||e.length)>e.length&&(t=e.length),r=t-1;0<=r&&128==(192&e[r]);)r--;return r<0?t:0===r?t:r+h[e[r]]>t?r:t}(t),i=t;n!==t.length&&(u.uint8array?(i=t.subarray(0,n),this.leftOver=t.subarray(n,t.length)):(i=t.slice(0,n),this.leftOver=t.slice(n,t.length))),this.push({data:s.utf8decode(i),meta:e.meta})},a.prototype.flush=function(){this.leftOver&&this.leftOver.length&&(this.push({data:s.utf8decode(this.leftOver),meta:{}}),this.leftOver=null)},s.Utf8DecodeWorker=a,o.inherits(f,n),f.prototype.processChunk=function(e){this.push({data:s.utf8encode(e.data),meta:e.meta})},s.Utf8EncodeWorker=f},{"./nodejsUtils":14,"./stream/GenericWorker":28,"./support":30,"./utils":32}],32:[function(e,t,o){"use strict";var u=e("./support"),h=e("./base64"),r=e("./nodejsUtils"),n=e("set-immediate-shim"),f=e("./external");function i(e){return e}function l(e,t){for(var r=0;r<e.length;++r)t[r]=255&e.charCodeAt(r);return t}o.newBlob=function(t,r){o.checkSupport("blob");try{return new Blob([t],{type:r})}catch(e){try{var n=new(self.BlobBuilder||self.WebKitBlobBuilder||self.MozBlobBuilder||self.MSBlobBuilder);return n.append(t),n.getBlob(r)}catch(e){throw new Error("Bug : can't construct the Blob.")}}};var s={stringifyByChunk:function(e,t,r){var n=[],i=0,s=e.length;if(s<=r)return String.fromCharCode.apply(null,e);for(;i<s;)"array"===t||"nodebuffer"===t?n.push(String.fromCharCode.apply(null,e.slice(i,Math.min(i+r,s)))):n.push(String.fromCharCode.apply(null,e.subarray(i,Math.min(i+r,s)))),i+=r;return n.join("")},stringifyByChar:function(e){for(var t="",r=0;r<e.length;r++)t+=String.fromCharCode(e[r]);return t},applyCanBeUsed:{uint8array:function(){try{return u.uint8array&&1===String.fromCharCode.apply(null,new Uint8Array(1)).length}catch(e){return!1}}(),nodebuffer:function(){try{return u.nodebuffer&&1===String.fromCharCode.apply(null,r.allocBuffer(1)).length}catch(e){return!1}}()}};function a(e){var t=65536,r=o.getTypeOf(e),n=!0;if("uint8array"===r?n=s.applyCanBeUsed.uint8array:"nodebuffer"===r&&(n=s.applyCanBeUsed.nodebuffer),n)for(;1<t;)try{return s.stringifyByChunk(e,r,t)}catch(e){t=Math.floor(t/2)}return s.stringifyByChar(e)}function d(e,t){for(var r=0;r<e.length;r++)t[r]=e[r];return t}o.applyFromCharCode=a;var c={};c.string={string:i,array:function(e){return l(e,new Array(e.length))},arraybuffer:function(e){return c.string.uint8array(e).buffer},uint8array:function(e){return l(e,new Uint8Array(e.length))},nodebuffer:function(e){return l(e,r.allocBuffer(e.length))}},c.array={string:a,array:i,arraybuffer:function(e){return new Uint8Array(e).buffer},uint8array:function(e){return new Uint8Array(e)},nodebuffer:function(e){return r.newBufferFrom(e)}},c.arraybuffer={string:function(e){return a(new Uint8Array(e))},array:function(e){return d(new Uint8Array(e),new Array(e.byteLength))},arraybuffer:i,uint8array:function(e){return new Uint8Array(e)},nodebuffer:function(e){return r.newBufferFrom(new Uint8Array(e))}},c.uint8array={string:a,array:function(e){return d(e,new Array(e.length))},arraybuffer:function(e){return e.buffer},uint8array:i,nodebuffer:function(e){return r.newBufferFrom(e)}},c.nodebuffer={string:a,array:function(e){return d(e,new Array(e.length))},arraybuffer:function(e){return c.nodebuffer.uint8array(e).buffer},uint8array:function(e){return d(e,new Uint8Array(e.length))},nodebuffer:i},o.transformTo=function(e,t){if(t=t||"",!e)return t;o.checkSupport(e);var r=o.getTypeOf(t);return c[r][e](t)},o.getTypeOf=function(e){return"string"==typeof e?"string":"[object Array]"===Object.prototype.toString.call(e)?"array":u.nodebuffer&&r.isBuffer(e)?"nodebuffer":u.uint8array&&e instanceof Uint8Array?"uint8array":u.arraybuffer&&e instanceof ArrayBuffer?"arraybuffer":void 0},o.checkSupport=function(e){if(!u[e.toLowerCase()])throw new Error(e+" is not supported by this platform")},o.MAX_VALUE_16BITS=65535,o.MAX_VALUE_32BITS=-1,o.pretty=function(e){var t,r,n="";for(r=0;r<(e||"").length;r++)n+="\\x"+((t=e.charCodeAt(r))<16?"0":"")+t.toString(16).toUpperCase();return n},o.delay=function(e,t,r){n(function(){e.apply(r||null,t||[])})},o.inherits=function(e,t){function r(){}r.prototype=t.prototype,e.prototype=new r},o.extend=function(){var e,t,r={};for(e=0;e<arguments.length;e++)for(t in arguments[e])arguments[e].hasOwnProperty(t)&&void 0===r[t]&&(r[t]=arguments[e][t]);return r},o.prepareContent=function(n,e,i,s,a){return f.Promise.resolve(e).then(function(n){return u.blob&&(n instanceof Blob||-1!==["[object File]","[object Blob]"].indexOf(Object.prototype.toString.call(n)))&&"undefined"!=typeof FileReader?new f.Promise(function(t,r){var e=new FileReader;e.onload=function(e){t(e.target.result)},e.onerror=function(e){r(e.target.error)},e.readAsArrayBuffer(n)}):n}).then(function(e){var t,r=o.getTypeOf(e);return r?("arraybuffer"===r?e=o.transformTo("uint8array",e):"string"===r&&(a?e=h.decode(e):i&&!0!==s&&(e=l(t=e,u.uint8array?new Uint8Array(t.length):new Array(t.length)))),e):f.Promise.reject(new Error("Can't read the data of '"+n+"'. Is it in a supported JavaScript type (String, Blob, ArrayBuffer, etc) ?"))})}},{"./base64":1,"./external":6,"./nodejsUtils":14,"./support":30,"set-immediate-shim":54}],33:[function(e,t,r){"use strict";var n=e("./reader/readerFor"),i=e("./utils"),s=e("./signature"),a=e("./zipEntry"),o=(e("./utf8"),e("./support"));function u(e){this.files=[],this.loadOptions=e}u.prototype={checkSignature:function(e){if(!this.reader.readAndCheckSignature(e)){this.reader.index-=4;var t=this.reader.readString(4);throw new Error("Corrupted zip or bug: unexpected signature ("+i.pretty(t)+", expected "+i.pretty(e)+")")}},isSignature:function(e,t){var r=this.reader.index;this.reader.setIndex(e);var n=this.reader.readString(4)===t;return this.reader.setIndex(r),n},readBlockEndOfCentral:function(){this.diskNumber=this.reader.readInt(2),this.diskWithCentralDirStart=this.reader.readInt(2),this.centralDirRecordsOnThisDisk=this.reader.readInt(2),this.centralDirRecords=this.reader.readInt(2),this.centralDirSize=this.reader.readInt(4),this.centralDirOffset=this.reader.readInt(4),this.zipCommentLength=this.reader.readInt(2);var e=this.reader.readData(this.zipCommentLength),t=o.uint8array?"uint8array":"array",r=i.transformTo(t,e);this.zipComment=this.loadOptions.decodeFileName(r)},readBlockZip64EndOfCentral:function(){this.zip64EndOfCentralSize=this.reader.readInt(8),this.reader.skip(4),this.diskNumber=this.reader.readInt(4),this.diskWithCentralDirStart=this.reader.readInt(4),this.centralDirRecordsOnThisDisk=this.reader.readInt(8),this.centralDirRecords=this.reader.readInt(8),this.centralDirSize=this.reader.readInt(8),this.centralDirOffset=this.reader.readInt(8),this.zip64ExtensibleData={};for(var e,t,r,n=this.zip64EndOfCentralSize-44;0<n;)e=this.reader.readInt(2),t=this.reader.readInt(4),r=this.reader.readData(t),this.zip64ExtensibleData[e]={id:e,length:t,value:r}},readBlockZip64EndOfCentralLocator:function(){if(this.diskWithZip64CentralDirStart=this.reader.readInt(4),this.relativeOffsetEndOfZip64CentralDir=this.reader.readInt(8),this.disksCount=this.reader.readInt(4),1<this.disksCount)throw new Error("Multi-volumes zip are not supported")},readLocalFiles:function(){var e,t;for(e=0;e<this.files.length;e++)t=this.files[e],this.reader.setIndex(t.localHeaderOffset),this.checkSignature(s.LOCAL_FILE_HEADER),t.readLocalPart(this.reader),t.handleUTF8(),t.processAttributes()},readCentralDir:function(){var e;for(this.reader.setIndex(this.centralDirOffset);this.reader.readAndCheckSignature(s.CENTRAL_FILE_HEADER);)(e=new a({zip64:this.zip64},this.loadOptions)).readCentralPart(this.reader),this.files.push(e);if(this.centralDirRecords!==this.files.length&&0!==this.centralDirRecords&&0===this.files.length)throw new Error("Corrupted zip or bug: expected "+this.centralDirRecords+" records in central dir, got "+this.files.length)},readEndOfCentral:function(){var e=this.reader.lastIndexOfSignature(s.CENTRAL_DIRECTORY_END);if(e<0)throw this.isSignature(0,s.LOCAL_FILE_HEADER)?new Error("Corrupted zip: can't find end of central directory"):new Error("Can't find end of central directory : is this a zip file ? If it is, see https://stuk.github.io/jszip/documentation/howto/read_zip.html");this.reader.setIndex(e);var t=e;if(this.checkSignature(s.CENTRAL_DIRECTORY_END),this.readBlockEndOfCentral(),this.diskNumber===i.MAX_VALUE_16BITS||this.diskWithCentralDirStart===i.MAX_VALUE_16BITS||this.centralDirRecordsOnThisDisk===i.MAX_VALUE_16BITS||this.centralDirRecords===i.MAX_VALUE_16BITS||this.centralDirSize===i.MAX_VALUE_32BITS||this.centralDirOffset===i.MAX_VALUE_32BITS){if(this.zip64=!0,(e=this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR))<0)throw new Error("Corrupted zip: can't find the ZIP64 end of central directory locator");if(this.reader.setIndex(e),this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_LOCATOR),this.readBlockZip64EndOfCentralLocator(),!this.isSignature(this.relativeOffsetEndOfZip64CentralDir,s.ZIP64_CENTRAL_DIRECTORY_END)&&(this.relativeOffsetEndOfZip64CentralDir=this.reader.lastIndexOfSignature(s.ZIP64_CENTRAL_DIRECTORY_END),this.relativeOffsetEndOfZip64CentralDir<0))throw new Error("Corrupted zip: can't find the ZIP64 end of central directory");this.reader.setIndex(this.relativeOffsetEndOfZip64CentralDir),this.checkSignature(s.ZIP64_CENTRAL_DIRECTORY_END),this.readBlockZip64EndOfCentral()}var r=this.centralDirOffset+this.centralDirSize;this.zip64&&(r+=20,r+=12+this.zip64EndOfCentralSize);var n=t-r;if(0<n)this.isSignature(t,s.CENTRAL_FILE_HEADER)||(this.reader.zero=n);else if(n<0)throw new Error("Corrupted zip: missing "+Math.abs(n)+" bytes.")},prepareReader:function(e){this.reader=n(e)},load:function(e){this.prepareReader(e),this.readEndOfCentral(),this.readCentralDir(),this.readLocalFiles()}},t.exports=u},{"./reader/readerFor":22,"./signature":23,"./support":30,"./utf8":31,"./utils":32,"./zipEntry":34}],34:[function(e,t,r){"use strict";var n=e("./reader/readerFor"),s=e("./utils"),i=e("./compressedObject"),a=e("./crc32"),o=e("./utf8"),u=e("./compressions"),h=e("./support");function f(e,t){this.options=e,this.loadOptions=t}f.prototype={isEncrypted:function(){return 1==(1&this.bitFlag)},useUTF8:function(){return 2048==(2048&this.bitFlag)},readLocalPart:function(e){var t,r;if(e.skip(22),this.fileNameLength=e.readInt(2),r=e.readInt(2),this.fileName=e.readData(this.fileNameLength),e.skip(r),-1===this.compressedSize||-1===this.uncompressedSize)throw new Error("Bug or corrupted zip : didn't get enough information from the central directory (compressedSize === -1 || uncompressedSize === -1)");if(null===(t=function(e){for(var t in u)if(u.hasOwnProperty(t)&&u[t].magic===e)return u[t];return null}(this.compressionMethod)))throw new Error("Corrupted zip : compression "+s.pretty(this.compressionMethod)+" unknown (inner file : "+s.transformTo("string",this.fileName)+")");this.decompressed=new i(this.compressedSize,this.uncompressedSize,this.crc32,t,e.readData(this.compressedSize))},readCentralPart:function(e){this.versionMadeBy=e.readInt(2),e.skip(2),this.bitFlag=e.readInt(2),this.compressionMethod=e.readString(2),this.date=e.readDate(),this.crc32=e.readInt(4),this.compressedSize=e.readInt(4),this.uncompressedSize=e.readInt(4);var t=e.readInt(2);if(this.extraFieldsLength=e.readInt(2),this.fileCommentLength=e.readInt(2),this.diskNumberStart=e.readInt(2),this.internalFileAttributes=e.readInt(2),this.externalFileAttributes=e.readInt(4),this.localHeaderOffset=e.readInt(4),this.isEncrypted())throw new Error("Encrypted zip are not supported");e.skip(t),this.readExtraFields(e),this.parseZIP64ExtraField(e),this.fileComment=e.readData(this.fileCommentLength)},processAttributes:function(){this.unixPermissions=null,this.dosPermissions=null;var e=this.versionMadeBy>>8;this.dir=!!(16&this.externalFileAttributes),0==e&&(this.dosPermissions=63&this.externalFileAttributes),3==e&&(this.unixPermissions=this.externalFileAttributes>>16&65535),this.dir||"/"!==this.fileNameStr.slice(-1)||(this.dir=!0)},parseZIP64ExtraField:function(e){if(this.extraFields[1]){var t=n(this.extraFields[1].value);this.uncompressedSize===s.MAX_VALUE_32BITS&&(this.uncompressedSize=t.readInt(8)),this.compressedSize===s.MAX_VALUE_32BITS&&(this.compressedSize=t.readInt(8)),this.localHeaderOffset===s.MAX_VALUE_32BITS&&(this.localHeaderOffset=t.readInt(8)),this.diskNumberStart===s.MAX_VALUE_32BITS&&(this.diskNumberStart=t.readInt(4))}},readExtraFields:function(e){var t,r,n,i=e.index+this.extraFieldsLength;for(this.extraFields||(this.extraFields={});e.index+4<i;)t=e.readInt(2),r=e.readInt(2),n=e.readData(r),this.extraFields[t]={id:t,length:r,value:n};e.setIndex(i)},handleUTF8:function(){var e=h.uint8array?"uint8array":"array";if(this.useUTF8())this.fileNameStr=o.utf8decode(this.fileName),this.fileCommentStr=o.utf8decode(this.fileComment);else{var t=this.findExtraFieldUnicodePath();if(null!==t)this.fileNameStr=t;else{var r=s.transformTo(e,this.fileName);this.fileNameStr=this.loadOptions.decodeFileName(r)}var n=this.findExtraFieldUnicodeComment();if(null!==n)this.fileCommentStr=n;else{var i=s.transformTo(e,this.fileComment);this.fileCommentStr=this.loadOptions.decodeFileName(i)}}},findExtraFieldUnicodePath:function(){var e=this.extraFields[28789];if(e){var t=n(e.value);return 1!==t.readInt(1)?null:a(this.fileName)!==t.readInt(4)?null:o.utf8decode(t.readData(e.length-5))}return null},findExtraFieldUnicodeComment:function(){var e=this.extraFields[25461];if(e){var t=n(e.value);return 1!==t.readInt(1)?null:a(this.fileComment)!==t.readInt(4)?null:o.utf8decode(t.readData(e.length-5))}return null}},t.exports=f},{"./compressedObject":2,"./compressions":3,"./crc32":4,"./reader/readerFor":22,"./support":30,"./utf8":31,"./utils":32}],35:[function(e,t,r){"use strict";function n(e,t,r){this.name=e,this.dir=r.dir,this.date=r.date,this.comment=r.comment,this.unixPermissions=r.unixPermissions,this.dosPermissions=r.dosPermissions,this._data=t,this._dataBinary=r.binary,this.options={compression:r.compression,compressionOptions:r.compressionOptions}}var s=e("./stream/StreamHelper"),i=e("./stream/DataWorker"),a=e("./utf8"),o=e("./compressedObject"),u=e("./stream/GenericWorker");n.prototype={internalStream:function(e){var t=null,r="string";try{if(!e)throw new Error("No output type specified.");var n="string"===(r=e.toLowerCase())||"text"===r;"binarystring"!==r&&"text"!==r||(r="string"),t=this._decompressWorker();var i=!this._dataBinary;i&&!n&&(t=t.pipe(new a.Utf8EncodeWorker)),!i&&n&&(t=t.pipe(new a.Utf8DecodeWorker))}catch(e){(t=new u("error")).error(e)}return new s(t,r,"")},async:function(e,t){return this.internalStream(e).accumulate(t)},nodeStream:function(e,t){return this.internalStream(e||"nodebuffer").toNodejsStream(t)},_compressWorker:function(e,t){if(this._data instanceof o&&this._data.compression.magic===e.magic)return this._data.getCompressedWorker();var r=this._decompressWorker();return this._dataBinary||(r=r.pipe(new a.Utf8EncodeWorker)),o.createWorkerFrom(r,e,t)},_decompressWorker:function(){return this._data instanceof o?this._data.getContentWorker():this._data instanceof u?this._data:new i(this._data)}};for(var h=["asText","asBinary","asNodeBuffer","asUint8Array","asArrayBuffer"],f=function(){throw new Error("This method has been removed in JSZip 3.0, please check the upgrade guide.")},l=0;l<h.length;l++)n.prototype[h[l]]=f;t.exports=n},{"./compressedObject":2,"./stream/DataWorker":27,"./stream/GenericWorker":28,"./stream/StreamHelper":29,"./utf8":31}],36:[function(e,f,t){(function(t){"use strict";var r,n,e=t.MutationObserver||t.WebKitMutationObserver;if(e){var i=0,s=new e(h),a=t.document.createTextNode("");s.observe(a,{characterData:!0}),r=function(){a.data=i=++i%2}}else if(t.setImmediate||void 0===t.MessageChannel)r="document"in t&&"onreadystatechange"in t.document.createElement("script")?function(){var e=t.document.createElement("script");e.onreadystatechange=function(){h(),e.onreadystatechange=null,e.parentNode.removeChild(e),e=null},t.document.documentElement.appendChild(e)}:function(){setTimeout(h,0)};else{var o=new t.MessageChannel;o.port1.onmessage=h,r=function(){o.port2.postMessage(0)}}var u=[];function h(){var e,t;n=!0;for(var r=u.length;r;){for(t=u,u=[],e=-1;++e<r;)t[e]();r=u.length}n=!1}f.exports=function(e){1!==u.push(e)||n||r()}}).call(this,void 0!==r?r:"undefined"!=typeof self?self:"undefined"!=typeof window?window:{})},{}],37:[function(e,t,r){"use strict";var i=e("immediate");function h(){}var f={},s=["REJECTED"],a=["FULFILLED"],n=["PENDING"];function o(e){if("function"!=typeof e)throw new TypeError("resolver must be a function");this.state=n,this.queue=[],this.outcome=void 0,e!==h&&c(this,e)}function u(e,t,r){this.promise=e,"function"==typeof t&&(this.onFulfilled=t,this.callFulfilled=this.otherCallFulfilled),"function"==typeof r&&(this.onRejected=r,this.callRejected=this.otherCallRejected)}function l(t,r,n){i(function(){var e;try{e=r(n)}catch(e){return f.reject(t,e)}e===t?f.reject(t,new TypeError("Cannot resolve promise with itself")):f.resolve(t,e)})}function d(e){var t=e&&e.then;if(e&&("object"==typeof e||"function"==typeof e)&&"function"==typeof t)return function(){t.apply(e,arguments)}}function c(t,e){var r=!1;function n(e){r||(r=!0,f.reject(t,e))}function i(e){r||(r=!0,f.resolve(t,e))}var s=p(function(){e(i,n)});"error"===s.status&&n(s.value)}function p(e,t){var r={};try{r.value=e(t),r.status="success"}catch(e){r.status="error",r.value=e}return r}(t.exports=o).prototype.finally=function(t){if("function"!=typeof t)return this;var r=this.constructor;return this.then(function(e){return r.resolve(t()).then(function(){return e})},function(e){return r.resolve(t()).then(function(){throw e})})},o.prototype.catch=function(e){return this.then(null,e)},o.prototype.then=function(e,t){if("function"!=typeof e&&this.state===a||"function"!=typeof t&&this.state===s)return this;var r=new this.constructor(h);return this.state!==n?l(r,this.state===a?e:t,this.outcome):this.queue.push(new u(r,e,t)),r},u.prototype.callFulfilled=function(e){f.resolve(this.promise,e)},u.prototype.otherCallFulfilled=function(e){l(this.promise,this.onFulfilled,e)},u.prototype.callRejected=function(e){f.reject(this.promise,e)},u.prototype.otherCallRejected=function(e){l(this.promise,this.onRejected,e)},f.resolve=function(e,t){var r=p(d,t);if("error"===r.status)return f.reject(e,r.value);var n=r.value;if(n)c(e,n);else{e.state=a,e.outcome=t;for(var i=-1,s=e.queue.length;++i<s;)e.queue[i].callFulfilled(t)}return e},f.reject=function(e,t){e.state=s,e.outcome=t;for(var r=-1,n=e.queue.length;++r<n;)e.queue[r].callRejected(t);return e},o.resolve=function(e){return e instanceof this?e:f.resolve(new this(h),e)},o.reject=function(e){var t=new this(h);return f.reject(t,e)},o.all=function(e){var r=this;if("[object Array]"!==Object.prototype.toString.call(e))return this.reject(new TypeError("must be an array"));var n=e.length,i=!1;if(!n)return this.resolve([]);for(var s=new Array(n),a=0,t=-1,o=new this(h);++t<n;)u(e[t],t);return o;function u(e,t){r.resolve(e).then(function(e){s[t]=e,++a!==n||i||(i=!0,f.resolve(o,s))},function(e){i||(i=!0,f.reject(o,e))})}},o.race=function(e){if("[object Array]"!==Object.prototype.toString.call(e))return this.reject(new TypeError("must be an array"));var t=e.length,r=!1;if(!t)return this.resolve([]);for(var n,i=-1,s=new this(h);++i<t;)n=e[i],this.resolve(n).then(function(e){r||(r=!0,f.resolve(s,e))},function(e){r||(r=!0,f.reject(s,e))});return s}},{immediate:36}],38:[function(e,t,r){"use strict";var n={};(0,e("./lib/utils/common").assign)(n,e("./lib/deflate"),e("./lib/inflate"),e("./lib/zlib/constants")),t.exports=n},{"./lib/deflate":39,"./lib/inflate":40,"./lib/utils/common":41,"./lib/zlib/constants":44}],39:[function(e,t,r){"use strict";var a=e("./zlib/deflate"),o=e("./utils/common"),u=e("./utils/strings"),i=e("./zlib/messages"),s=e("./zlib/zstream"),h=Object.prototype.toString,f=0,l=-1,d=0,c=8;function p(e){if(!(this instanceof p))return new p(e);this.options=o.assign({level:l,method:c,chunkSize:16384,windowBits:15,memLevel:8,strategy:d,to:""},e||{});var t=this.options;t.raw&&0<t.windowBits?t.windowBits=-t.windowBits:t.gzip&&0<t.windowBits&&t.windowBits<16&&(t.windowBits+=16),this.err=0,this.msg="",this.ended=!1,this.chunks=[],this.strm=new s,this.strm.avail_out=0;var r=a.deflateInit2(this.strm,t.level,t.method,t.windowBits,t.memLevel,t.strategy);if(r!==f)throw new Error(i[r]);if(t.header&&a.deflateSetHeader(this.strm,t.header),t.dictionary){var n;if(n="string"==typeof t.dictionary?u.string2buf(t.dictionary):"[object ArrayBuffer]"===h.call(t.dictionary)?new Uint8Array(t.dictionary):t.dictionary,(r=a.deflateSetDictionary(this.strm,n))!==f)throw new Error(i[r]);this._dict_set=!0}}function n(e,t){var r=new p(t);if(r.push(e,!0),r.err)throw r.msg||i[r.err];return r.result}p.prototype.push=function(e,t){var r,n,i=this.strm,s=this.options.chunkSize;if(this.ended)return!1;n=t===~~t?t:!0===t?4:0,"string"==typeof e?i.input=u.string2buf(e):"[object ArrayBuffer]"===h.call(e)?i.input=new Uint8Array(e):i.input=e,i.next_in=0,i.avail_in=i.input.length;do{if(0===i.avail_out&&(i.output=new o.Buf8(s),i.next_out=0,i.avail_out=s),1!==(r=a.deflate(i,n))&&r!==f)return this.onEnd(r),!(this.ended=!0);0!==i.avail_out&&(0!==i.avail_in||4!==n&&2!==n)||("string"===this.options.to?this.onData(u.buf2binstring(o.shrinkBuf(i.output,i.next_out))):this.onData(o.shrinkBuf(i.output,i.next_out)))}while((0<i.avail_in||0===i.avail_out)&&1!==r);return 4===n?(r=a.deflateEnd(this.strm),this.onEnd(r),this.ended=!0,r===f):2!==n||(this.onEnd(f),!(i.avail_out=0))},p.prototype.onData=function(e){this.chunks.push(e)},p.prototype.onEnd=function(e){e===f&&("string"===this.options.to?this.result=this.chunks.join(""):this.result=o.flattenChunks(this.chunks)),this.chunks=[],this.err=e,this.msg=this.strm.msg},r.Deflate=p,r.deflate=n,r.deflateRaw=function(e,t){return(t=t||{}).raw=!0,n(e,t)},r.gzip=function(e,t){return(t=t||{}).gzip=!0,n(e,t)}},{"./utils/common":41,"./utils/strings":42,"./zlib/deflate":46,"./zlib/messages":51,"./zlib/zstream":53}],40:[function(e,t,r){"use strict";var d=e("./zlib/inflate"),c=e("./utils/common"),p=e("./utils/strings"),m=e("./zlib/constants"),n=e("./zlib/messages"),i=e("./zlib/zstream"),s=e("./zlib/gzheader"),_=Object.prototype.toString;function a(e){if(!(this instanceof a))return new a(e);this.options=c.assign({chunkSize:16384,windowBits:0,to:""},e||{});var t=this.options;t.raw&&0<=t.windowBits&&t.windowBits<16&&(t.windowBits=-t.windowBits,0===t.windowBits&&(t.windowBits=-15)),!(0<=t.windowBits&&t.windowBits<16)||e&&e.windowBits||(t.windowBits+=32),15<t.windowBits&&t.windowBits<48&&0==(15&t.windowBits)&&(t.windowBits|=15),this.err=0,this.msg="",this.ended=!1,this.chunks=[],this.strm=new i,this.strm.avail_out=0;var r=d.inflateInit2(this.strm,t.windowBits);if(r!==m.Z_OK)throw new Error(n[r]);this.header=new s,d.inflateGetHeader(this.strm,this.header)}function o(e,t){var r=new a(t);if(r.push(e,!0),r.err)throw r.msg||n[r.err];return r.result}a.prototype.push=function(e,t){var r,n,i,s,a,o,u=this.strm,h=this.options.chunkSize,f=this.options.dictionary,l=!1;if(this.ended)return!1;n=t===~~t?t:!0===t?m.Z_FINISH:m.Z_NO_FLUSH,"string"==typeof e?u.input=p.binstring2buf(e):"[object ArrayBuffer]"===_.call(e)?u.input=new Uint8Array(e):u.input=e,u.next_in=0,u.avail_in=u.input.length;do{if(0===u.avail_out&&(u.output=new c.Buf8(h),u.next_out=0,u.avail_out=h),(r=d.inflate(u,m.Z_NO_FLUSH))===m.Z_NEED_DICT&&f&&(o="string"==typeof f?p.string2buf(f):"[object ArrayBuffer]"===_.call(f)?new Uint8Array(f):f,r=d.inflateSetDictionary(this.strm,o)),r===m.Z_BUF_ERROR&&!0===l&&(r=m.Z_OK,l=!1),r!==m.Z_STREAM_END&&r!==m.Z_OK)return this.onEnd(r),!(this.ended=!0);u.next_out&&(0!==u.avail_out&&r!==m.Z_STREAM_END&&(0!==u.avail_in||n!==m.Z_FINISH&&n!==m.Z_SYNC_FLUSH)||("string"===this.options.to?(i=p.utf8border(u.output,u.next_out),s=u.next_out-i,a=p.buf2string(u.output,i),u.next_out=s,u.avail_out=h-s,s&&c.arraySet(u.output,u.output,i,s,0),this.onData(a)):this.onData(c.shrinkBuf(u.output,u.next_out)))),0===u.avail_in&&0===u.avail_out&&(l=!0)}while((0<u.avail_in||0===u.avail_out)&&r!==m.Z_STREAM_END);return r===m.Z_STREAM_END&&(n=m.Z_FINISH),n===m.Z_FINISH?(r=d.inflateEnd(this.strm),this.onEnd(r),this.ended=!0,r===m.Z_OK):n!==m.Z_SYNC_FLUSH||(this.onEnd(m.Z_OK),!(u.avail_out=0))},a.prototype.onData=function(e){this.chunks.push(e)},a.prototype.onEnd=function(e){e===m.Z_OK&&("string"===this.options.to?this.result=this.chunks.join(""):this.result=c.flattenChunks(this.chunks)),this.chunks=[],this.err=e,this.msg=this.strm.msg},r.Inflate=a,r.inflate=o,r.inflateRaw=function(e,t){return(t=t||{}).raw=!0,o(e,t)},r.ungzip=o},{"./utils/common":41,"./utils/strings":42,"./zlib/constants":44,"./zlib/gzheader":47,"./zlib/inflate":49,"./zlib/messages":51,"./zlib/zstream":53}],41:[function(e,t,r){"use strict";var n="undefined"!=typeof Uint8Array&&"undefined"!=typeof Uint16Array&&"undefined"!=typeof Int32Array;r.assign=function(e){for(var t=Array.prototype.slice.call(arguments,1);t.length;){var r=t.shift();if(r){if("object"!=typeof r)throw new TypeError(r+"must be non-object");for(var n in r)r.hasOwnProperty(n)&&(e[n]=r[n])}}return e},r.shrinkBuf=function(e,t){return e.length===t?e:e.subarray?e.subarray(0,t):(e.length=t,e)};var i={arraySet:function(e,t,r,n,i){if(t.subarray&&e.subarray)e.set(t.subarray(r,r+n),i);else for(var s=0;s<n;s++)e[i+s]=t[r+s]},flattenChunks:function(e){var t,r,n,i,s,a;for(t=n=0,r=e.length;t<r;t++)n+=e[t].length;for(a=new Uint8Array(n),t=i=0,r=e.length;t<r;t++)s=e[t],a.set(s,i),i+=s.length;return a}},s={arraySet:function(e,t,r,n,i){for(var s=0;s<n;s++)e[i+s]=t[r+s]},flattenChunks:function(e){return[].concat.apply([],e)}};r.setTyped=function(e){e?(r.Buf8=Uint8Array,r.Buf16=Uint16Array,r.Buf32=Int32Array,r.assign(r,i)):(r.Buf8=Array,r.Buf16=Array,r.Buf32=Array,r.assign(r,s))},r.setTyped(n)},{}],42:[function(e,t,r){"use strict";var u=e("./common"),i=!0,s=!0;try{String.fromCharCode.apply(null,[0])}catch(e){i=!1}try{String.fromCharCode.apply(null,new Uint8Array(1))}catch(e){s=!1}for(var h=new u.Buf8(256),n=0;n<256;n++)h[n]=252<=n?6:248<=n?5:240<=n?4:224<=n?3:192<=n?2:1;function f(e,t){if(t<65537&&(e.subarray&&s||!e.subarray&&i))return String.fromCharCode.apply(null,u.shrinkBuf(e,t));for(var r="",n=0;n<t;n++)r+=String.fromCharCode(e[n]);return r}h[254]=h[254]=1,r.string2buf=function(e){var t,r,n,i,s,a=e.length,o=0;for(i=0;i<a;i++)55296==(64512&(r=e.charCodeAt(i)))&&i+1<a&&56320==(64512&(n=e.charCodeAt(i+1)))&&(r=65536+(r-55296<<10)+(n-56320),i++),o+=r<128?1:r<2048?2:r<65536?3:4;for(t=new u.Buf8(o),i=s=0;s<o;i++)55296==(64512&(r=e.charCodeAt(i)))&&i+1<a&&56320==(64512&(n=e.charCodeAt(i+1)))&&(r=65536+(r-55296<<10)+(n-56320),i++),r<128?t[s++]=r:(r<2048?t[s++]=192|r>>>6:(r<65536?t[s++]=224|r>>>12:(t[s++]=240|r>>>18,t[s++]=128|r>>>12&63),t[s++]=128|r>>>6&63),t[s++]=128|63&r);return t},r.buf2binstring=function(e){return f(e,e.length)},r.binstring2buf=function(e){for(var t=new u.Buf8(e.length),r=0,n=t.length;r<n;r++)t[r]=e.charCodeAt(r);return t},r.buf2string=function(e,t){var r,n,i,s,a=t||e.length,o=new Array(2*a);for(r=n=0;r<a;)if((i=e[r++])<128)o[n++]=i;else if(4<(s=h[i]))o[n++]=65533,r+=s-1;else{for(i&=2===s?31:3===s?15:7;1<s&&r<a;)i=i<<6|63&e[r++],s--;1<s?o[n++]=65533:i<65536?o[n++]=i:(i-=65536,o[n++]=55296|i>>10&1023,o[n++]=56320|1023&i)}return f(o,n)},r.utf8border=function(e,t){var r;for((t=t||e.length)>e.length&&(t=e.length),r=t-1;0<=r&&128==(192&e[r]);)r--;return r<0?t:0===r?t:r+h[e[r]]>t?r:t}},{"./common":41}],43:[function(e,t,r){"use strict";t.exports=function(e,t,r,n){for(var i=65535&e|0,s=e>>>16&65535|0,a=0;0!==r;){for(r-=a=2e3<r?2e3:r;s=s+(i=i+t[n++]|0)|0,--a;);i%=65521,s%=65521}return i|s<<16|0}},{}],44:[function(e,t,r){"use strict";t.exports={Z_NO_FLUSH:0,Z_PARTIAL_FLUSH:1,Z_SYNC_FLUSH:2,Z_FULL_FLUSH:3,Z_FINISH:4,Z_BLOCK:5,Z_TREES:6,Z_OK:0,Z_STREAM_END:1,Z_NEED_DICT:2,Z_ERRNO:-1,Z_STREAM_ERROR:-2,Z_DATA_ERROR:-3,Z_BUF_ERROR:-5,Z_NO_COMPRESSION:0,Z_BEST_SPEED:1,Z_BEST_COMPRESSION:9,Z_DEFAULT_COMPRESSION:-1,Z_FILTERED:1,Z_HUFFMAN_ONLY:2,Z_RLE:3,Z_FIXED:4,Z_DEFAULT_STRATEGY:0,Z_BINARY:0,Z_TEXT:1,Z_UNKNOWN:2,Z_DEFLATED:8}},{}],45:[function(e,t,r){"use strict";var o=function(){for(var e,t=[],r=0;r<256;r++){e=r;for(var n=0;n<8;n++)e=1&e?3988292384^e>>>1:e>>>1;t[r]=e}return t}();t.exports=function(e,t,r,n){var i=o,s=n+r;e^=-1;for(var a=n;a<s;a++)e=e>>>8^i[255&(e^t[a])];return-1^e}},{}],46:[function(e,t,r){"use strict";var u,d=e("../utils/common"),h=e("./trees"),c=e("./adler32"),p=e("./crc32"),n=e("./messages"),f=0,l=0,m=-2,i=2,_=8,s=286,a=30,o=19,g=2*s+1,v=15,b=3,w=258,y=w+b+1,k=42,x=113;function S(e,t){return e.msg=n[t],t}function z(e){return(e<<1)-(4<e?9:0)}function E(e){for(var t=e.length;0<=--t;)e[t]=0}function C(e){var t=e.state,r=t.pending;r>e.avail_out&&(r=e.avail_out),0!==r&&(d.arraySet(e.output,t.pending_buf,t.pending_out,r,e.next_out),e.next_out+=r,t.pending_out+=r,e.total_out+=r,e.avail_out-=r,t.pending-=r,0===t.pending&&(t.pending_out=0))}function A(e,t){h._tr_flush_block(e,0<=e.block_start?e.block_start:-1,e.strstart-e.block_start,t),e.block_start=e.strstart,C(e.strm)}function I(e,t){e.pending_buf[e.pending++]=t}function O(e,t){e.pending_buf[e.pending++]=t>>>8&255,e.pending_buf[e.pending++]=255&t}function B(e,t){var r,n,i=e.max_chain_length,s=e.strstart,a=e.prev_length,o=e.nice_match,u=e.strstart>e.w_size-y?e.strstart-(e.w_size-y):0,h=e.window,f=e.w_mask,l=e.prev,d=e.strstart+w,c=h[s+a-1],p=h[s+a];e.prev_length>=e.good_match&&(i>>=2),o>e.lookahead&&(o=e.lookahead);do{if(h[(r=t)+a]===p&&h[r+a-1]===c&&h[r]===h[s]&&h[++r]===h[s+1]){s+=2,r++;do{}while(h[++s]===h[++r]&&h[++s]===h[++r]&&h[++s]===h[++r]&&h[++s]===h[++r]&&h[++s]===h[++r]&&h[++s]===h[++r]&&h[++s]===h[++r]&&h[++s]===h[++r]&&s<d);if(n=w-(d-s),s=d-w,a<n){if(e.match_start=t,o<=(a=n))break;c=h[s+a-1],p=h[s+a]}}}while((t=l[t&f])>u&&0!=--i);return a<=e.lookahead?a:e.lookahead}function T(e){var t,r,n,i,s,a,o,u,h,f,l=e.w_size;do{if(i=e.window_size-e.lookahead-e.strstart,e.strstart>=l+(l-y)){for(d.arraySet(e.window,e.window,l,l,0),e.match_start-=l,e.strstart-=l,e.block_start-=l,t=r=e.hash_size;n=e.head[--t],e.head[t]=l<=n?n-l:0,--r;);for(t=r=l;n=e.prev[--t],e.prev[t]=l<=n?n-l:0,--r;);i+=l}if(0===e.strm.avail_in)break;if(a=e.strm,o=e.window,u=e.strstart+e.lookahead,f=void 0,(h=i)<(f=a.avail_in)&&(f=h),r=0===f?0:(a.avail_in-=f,d.arraySet(o,a.input,a.next_in,f,u),1===a.state.wrap?a.adler=c(a.adler,o,f,u):2===a.state.wrap&&(a.adler=p(a.adler,o,f,u)),a.next_in+=f,a.total_in+=f,f),e.lookahead+=r,e.lookahead+e.insert>=b)for(s=e.strstart-e.insert,e.ins_h=e.window[s],e.ins_h=(e.ins_h<<e.hash_shift^e.window[s+1])&e.hash_mask;e.insert&&(e.ins_h=(e.ins_h<<e.hash_shift^e.window[s+b-1])&e.hash_mask,e.prev[s&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=s,s++,e.insert--,!(e.lookahead+e.insert<b)););}while(e.lookahead<y&&0!==e.strm.avail_in)}function R(e,t){for(var r,n;;){if(e.lookahead<y){if(T(e),e.lookahead<y&&t===f)return 1;if(0===e.lookahead)break}if(r=0,e.lookahead>=b&&(e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+b-1])&e.hash_mask,r=e.prev[e.strstart&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=e.strstart),0!==r&&e.strstart-r<=e.w_size-y&&(e.match_length=B(e,r)),e.match_length>=b)if(n=h._tr_tally(e,e.strstart-e.match_start,e.match_length-b),e.lookahead-=e.match_length,e.match_length<=e.max_lazy_match&&e.lookahead>=b){for(e.match_length--;e.strstart++,e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+b-1])&e.hash_mask,r=e.prev[e.strstart&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=e.strstart,0!=--e.match_length;);e.strstart++}else e.strstart+=e.match_length,e.match_length=0,e.ins_h=e.window[e.strstart],e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+1])&e.hash_mask;else n=h._tr_tally(e,0,e.window[e.strstart]),e.lookahead--,e.strstart++;if(n&&(A(e,!1),0===e.strm.avail_out))return 1}return e.insert=e.strstart<b-1?e.strstart:b-1,4===t?(A(e,!0),0===e.strm.avail_out?3:4):e.last_lit&&(A(e,!1),0===e.strm.avail_out)?1:2}function D(e,t){for(var r,n,i;;){if(e.lookahead<y){if(T(e),e.lookahead<y&&t===f)return 1;if(0===e.lookahead)break}if(r=0,e.lookahead>=b&&(e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+b-1])&e.hash_mask,r=e.prev[e.strstart&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=e.strstart),e.prev_length=e.match_length,e.prev_match=e.match_start,e.match_length=b-1,0!==r&&e.prev_length<e.max_lazy_match&&e.strstart-r<=e.w_size-y&&(e.match_length=B(e,r),e.match_length<=5&&(1===e.strategy||e.match_length===b&&4096<e.strstart-e.match_start)&&(e.match_length=b-1)),e.prev_length>=b&&e.match_length<=e.prev_length){for(i=e.strstart+e.lookahead-b,n=h._tr_tally(e,e.strstart-1-e.prev_match,e.prev_length-b),e.lookahead-=e.prev_length-1,e.prev_length-=2;++e.strstart<=i&&(e.ins_h=(e.ins_h<<e.hash_shift^e.window[e.strstart+b-1])&e.hash_mask,r=e.prev[e.strstart&e.w_mask]=e.head[e.ins_h],e.head[e.ins_h]=e.strstart),0!=--e.prev_length;);if(e.match_available=0,e.match_length=b-1,e.strstart++,n&&(A(e,!1),0===e.strm.avail_out))return 1}else if(e.match_available){if((n=h._tr_tally(e,0,e.window[e.strstart-1]))&&A(e,!1),e.strstart++,e.lookahead--,0===e.strm.avail_out)return 1}else e.match_available=1,e.strstart++,e.lookahead--}return e.match_available&&(n=h._tr_tally(e,0,e.window[e.strstart-1]),e.match_available=0),e.insert=e.strstart<b-1?e.strstart:b-1,4===t?(A(e,!0),0===e.strm.avail_out?3:4):e.last_lit&&(A(e,!1),0===e.strm.avail_out)?1:2}function F(e,t,r,n,i){this.good_length=e,this.max_lazy=t,this.nice_length=r,this.max_chain=n,this.func=i}function N(){this.strm=null,this.status=0,this.pending_buf=null,this.pending_buf_size=0,this.pending_out=0,this.pending=0,this.wrap=0,this.gzhead=null,this.gzindex=0,this.method=_,this.last_flush=-1,this.w_size=0,this.w_bits=0,this.w_mask=0,this.window=null,this.window_size=0,this.prev=null,this.head=null,this.ins_h=0,this.hash_size=0,this.hash_bits=0,this.hash_mask=0,this.hash_shift=0,this.block_start=0,this.match_length=0,this.prev_match=0,this.match_available=0,this.strstart=0,this.match_start=0,this.lookahead=0,this.prev_length=0,this.max_chain_length=0,this.max_lazy_match=0,this.level=0,this.strategy=0,this.good_match=0,this.nice_match=0,this.dyn_ltree=new d.Buf16(2*g),this.dyn_dtree=new d.Buf16(2*(2*a+1)),this.bl_tree=new d.Buf16(2*(2*o+1)),E(this.dyn_ltree),E(this.dyn_dtree),E(this.bl_tree),this.l_desc=null,this.d_desc=null,this.bl_desc=null,this.bl_count=new d.Buf16(v+1),this.heap=new d.Buf16(2*s+1),E(this.heap),this.heap_len=0,this.heap_max=0,this.depth=new d.Buf16(2*s+1),E(this.depth),this.l_buf=0,this.lit_bufsize=0,this.last_lit=0,this.d_buf=0,this.opt_len=0,this.static_len=0,this.matches=0,this.insert=0,this.bi_buf=0,this.bi_valid=0}function U(e){var t;return e&&e.state?(e.total_in=e.total_out=0,e.data_type=i,(t=e.state).pending=0,t.pending_out=0,t.wrap<0&&(t.wrap=-t.wrap),t.status=t.wrap?k:x,e.adler=2===t.wrap?0:1,t.last_flush=f,h._tr_init(t),l):S(e,m)}function P(e){var t,r=U(e);return r===l&&((t=e.state).window_size=2*t.w_size,E(t.head),t.max_lazy_match=u[t.level].max_lazy,t.good_match=u[t.level].good_length,t.nice_match=u[t.level].nice_length,t.max_chain_length=u[t.level].max_chain,t.strstart=0,t.block_start=0,t.lookahead=0,t.insert=0,t.match_length=t.prev_length=b-1,t.match_available=0,t.ins_h=0),r}function L(e,t,r,n,i,s){if(!e)return m;var a=1;if(-1===t&&(t=6),n<0?(a=0,n=-n):15<n&&(a=2,n-=16),i<1||9<i||r!==_||n<8||15<n||t<0||9<t||s<0||4<s)return S(e,m);8===n&&(n=9);var o=new N;return(e.state=o).strm=e,o.wrap=a,o.gzhead=null,o.w_bits=n,o.w_size=1<<o.w_bits,o.w_mask=o.w_size-1,o.hash_bits=i+7,o.hash_size=1<<o.hash_bits,o.hash_mask=o.hash_size-1,o.hash_shift=~~((o.hash_bits+b-1)/b),o.window=new d.Buf8(2*o.w_size),o.head=new d.Buf16(o.hash_size),o.prev=new d.Buf16(o.w_size),o.lit_bufsize=1<<i+6,o.pending_buf_size=4*o.lit_bufsize,o.pending_buf=new d.Buf8(o.pending_buf_size),o.d_buf=1*o.lit_bufsize,o.l_buf=3*o.lit_bufsize,o.level=t,o.strategy=s,o.method=r,P(e)}u=[new F(0,0,0,0,function(e,t){var r=65535;for(r>e.pending_buf_size-5&&(r=e.pending_buf_size-5);;){if(e.lookahead<=1){if(T(e),0===e.lookahead&&t===f)return 1;if(0===e.lookahead)break}e.strstart+=e.lookahead,e.lookahead=0;var n=e.block_start+r;if((0===e.strstart||e.strstart>=n)&&(e.lookahead=e.strstart-n,e.strstart=n,A(e,!1),0===e.strm.avail_out))return 1;if(e.strstart-e.block_start>=e.w_size-y&&(A(e,!1),0===e.strm.avail_out))return 1}return e.insert=0,4===t?(A(e,!0),0===e.strm.avail_out?3:4):(e.strstart>e.block_start&&(A(e,!1),e.strm.avail_out),1)}),new F(4,4,8,4,R),new F(4,5,16,8,R),new F(4,6,32,32,R),new F(4,4,16,16,D),new F(8,16,32,32,D),new F(8,16,128,128,D),new F(8,32,128,256,D),new F(32,128,258,1024,D),new F(32,258,258,4096,D)],r.deflateInit=function(e,t){return L(e,t,_,15,8,0)},r.deflateInit2=L,r.deflateReset=P,r.deflateResetKeep=U,r.deflateSetHeader=function(e,t){return e&&e.state?2!==e.state.wrap?m:(e.state.gzhead=t,l):m},r.deflate=function(e,t){var r,n,i,s;if(!e||!e.state||5<t||t<0)return e?S(e,m):m;if(n=e.state,!e.output||!e.input&&0!==e.avail_in||666===n.status&&4!==t)return S(e,0===e.avail_out?-5:m);if(n.strm=e,r=n.last_flush,n.last_flush=t,n.status===k)if(2===n.wrap)e.adler=0,I(n,31),I(n,139),I(n,8),n.gzhead?(I(n,(n.gzhead.text?1:0)+(n.gzhead.hcrc?2:0)+(n.gzhead.extra?4:0)+(n.gzhead.name?8:0)+(n.gzhead.comment?16:0)),I(n,255&n.gzhead.time),I(n,n.gzhead.time>>8&255),I(n,n.gzhead.time>>16&255),I(n,n.gzhead.time>>24&255),I(n,9===n.level?2:2<=n.strategy||n.level<2?4:0),I(n,255&n.gzhead.os),n.gzhead.extra&&n.gzhead.extra.length&&(I(n,255&n.gzhead.extra.length),I(n,n.gzhead.extra.length>>8&255)),n.gzhead.hcrc&&(e.adler=p(e.adler,n.pending_buf,n.pending,0)),n.gzindex=0,n.status=69):(I(n,0),I(n,0),I(n,0),I(n,0),I(n,0),I(n,9===n.level?2:2<=n.strategy||n.level<2?4:0),I(n,3),n.status=x);else{var a=_+(n.w_bits-8<<4)<<8;a|=(2<=n.strategy||n.level<2?0:n.level<6?1:6===n.level?2:3)<<6,0!==n.strstart&&(a|=32),a+=31-a%31,n.status=x,O(n,a),0!==n.strstart&&(O(n,e.adler>>>16),O(n,65535&e.adler)),e.adler=1}if(69===n.status)if(n.gzhead.extra){for(i=n.pending;n.gzindex<(65535&n.gzhead.extra.length)&&(n.pending!==n.pending_buf_size||(n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),C(e),i=n.pending,n.pending!==n.pending_buf_size));)I(n,255&n.gzhead.extra[n.gzindex]),n.gzindex++;n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),n.gzindex===n.gzhead.extra.length&&(n.gzindex=0,n.status=73)}else n.status=73;if(73===n.status)if(n.gzhead.name){i=n.pending;do{if(n.pending===n.pending_buf_size&&(n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),C(e),i=n.pending,n.pending===n.pending_buf_size)){s=1;break}s=n.gzindex<n.gzhead.name.length?255&n.gzhead.name.charCodeAt(n.gzindex++):0,I(n,s)}while(0!==s);n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),0===s&&(n.gzindex=0,n.status=91)}else n.status=91;if(91===n.status)if(n.gzhead.comment){i=n.pending;do{if(n.pending===n.pending_buf_size&&(n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),C(e),i=n.pending,n.pending===n.pending_buf_size)){s=1;break}s=n.gzindex<n.gzhead.comment.length?255&n.gzhead.comment.charCodeAt(n.gzindex++):0,I(n,s)}while(0!==s);n.gzhead.hcrc&&n.pending>i&&(e.adler=p(e.adler,n.pending_buf,n.pending-i,i)),0===s&&(n.status=103)}else n.status=103;if(103===n.status&&(n.gzhead.hcrc?(n.pending+2>n.pending_buf_size&&C(e),n.pending+2<=n.pending_buf_size&&(I(n,255&e.adler),I(n,e.adler>>8&255),e.adler=0,n.status=x)):n.status=x),0!==n.pending){if(C(e),0===e.avail_out)return n.last_flush=-1,l}else if(0===e.avail_in&&z(t)<=z(r)&&4!==t)return S(e,-5);if(666===n.status&&0!==e.avail_in)return S(e,-5);if(0!==e.avail_in||0!==n.lookahead||t!==f&&666!==n.status){var o=2===n.strategy?function(e,t){for(var r;;){if(0===e.lookahead&&(T(e),0===e.lookahead)){if(t===f)return 1;break}if(e.match_length=0,r=h._tr_tally(e,0,e.window[e.strstart]),e.lookahead--,e.strstart++,r&&(A(e,!1),0===e.strm.avail_out))return 1}return e.insert=0,4===t?(A(e,!0),0===e.strm.avail_out?3:4):e.last_lit&&(A(e,!1),0===e.strm.avail_out)?1:2}(n,t):3===n.strategy?function(e,t){for(var r,n,i,s,a=e.window;;){if(e.lookahead<=w){if(T(e),e.lookahead<=w&&t===f)return 1;if(0===e.lookahead)break}if(e.match_length=0,e.lookahead>=b&&0<e.strstart&&(n=a[i=e.strstart-1])===a[++i]&&n===a[++i]&&n===a[++i]){s=e.strstart+w;do{}while(n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&n===a[++i]&&i<s);e.match_length=w-(s-i),e.match_length>e.lookahead&&(e.match_length=e.lookahead)}if(e.match_length>=b?(r=h._tr_tally(e,1,e.match_length-b),e.lookahead-=e.match_length,e.strstart+=e.match_length,e.match_length=0):(r=h._tr_tally(e,0,e.window[e.strstart]),e.lookahead--,e.strstart++),r&&(A(e,!1),0===e.strm.avail_out))return 1}return e.insert=0,4===t?(A(e,!0),0===e.strm.avail_out?3:4):e.last_lit&&(A(e,!1),0===e.strm.avail_out)?1:2}(n,t):u[n.level].func(n,t);if(3!==o&&4!==o||(n.status=666),1===o||3===o)return 0===e.avail_out&&(n.last_flush=-1),l;if(2===o&&(1===t?h._tr_align(n):5!==t&&(h._tr_stored_block(n,0,0,!1),3===t&&(E(n.head),0===n.lookahead&&(n.strstart=0,n.block_start=0,n.insert=0))),C(e),0===e.avail_out))return n.last_flush=-1,l}return 4!==t?l:n.wrap<=0?1:(2===n.wrap?(I(n,255&e.adler),I(n,e.adler>>8&255),I(n,e.adler>>16&255),I(n,e.adler>>24&255),I(n,255&e.total_in),I(n,e.total_in>>8&255),I(n,e.total_in>>16&255),I(n,e.total_in>>24&255)):(O(n,e.adler>>>16),O(n,65535&e.adler)),C(e),0<n.wrap&&(n.wrap=-n.wrap),0!==n.pending?l:1)},r.deflateEnd=function(e){var t;return e&&e.state?(t=e.state.status)!==k&&69!==t&&73!==t&&91!==t&&103!==t&&t!==x&&666!==t?S(e,m):(e.state=null,t===x?S(e,-3):l):m},r.deflateSetDictionary=function(e,t){var r,n,i,s,a,o,u,h,f=t.length;if(!e||!e.state)return m;if(2===(s=(r=e.state).wrap)||1===s&&r.status!==k||r.lookahead)return m;for(1===s&&(e.adler=c(e.adler,t,f,0)),r.wrap=0,f>=r.w_size&&(0===s&&(E(r.head),r.strstart=0,r.block_start=0,r.insert=0),h=new d.Buf8(r.w_size),d.arraySet(h,t,f-r.w_size,r.w_size,0),t=h,f=r.w_size),a=e.avail_in,o=e.next_in,u=e.input,e.avail_in=f,e.next_in=0,e.input=t,T(r);r.lookahead>=b;){for(n=r.strstart,i=r.lookahead-(b-1);r.ins_h=(r.ins_h<<r.hash_shift^r.window[n+b-1])&r.hash_mask,r.prev[n&r.w_mask]=r.head[r.ins_h],r.head[r.ins_h]=n,n++,--i;);r.strstart=n,r.lookahead=b-1,T(r)}return r.strstart+=r.lookahead,r.block_start=r.strstart,r.insert=r.lookahead,r.lookahead=0,r.match_length=r.prev_length=b-1,r.match_available=0,e.next_in=o,e.input=u,e.avail_in=a,r.wrap=s,l},r.deflateInfo="pako deflate (from Nodeca project)"},{"../utils/common":41,"./adler32":43,"./crc32":45,"./messages":51,"./trees":52}],47:[function(e,t,r){"use strict";t.exports=function(){this.text=0,this.time=0,this.xflags=0,this.os=0,this.extra=null,this.extra_len=0,this.name="",this.comment="",this.hcrc=0,this.done=!1}},{}],48:[function(e,t,r){"use strict";t.exports=function(e,t){var r,n,i,s,a,o,u,h,f,l,d,c,p,m,_,g,v,b,w,y,k,x,S,z,E;r=e.state,n=e.next_in,z=e.input,i=n+(e.avail_in-5),s=e.next_out,E=e.output,a=s-(t-e.avail_out),o=s+(e.avail_out-257),u=r.dmax,h=r.wsize,f=r.whave,l=r.wnext,d=r.window,c=r.hold,p=r.bits,m=r.lencode,_=r.distcode,g=(1<<r.lenbits)-1,v=(1<<r.distbits)-1;e:do{p<15&&(c+=z[n++]<<p,p+=8,c+=z[n++]<<p,p+=8),b=m[c&g];t:for(;;){if(c>>>=w=b>>>24,p-=w,0==(w=b>>>16&255))E[s++]=65535&b;else{if(!(16&w)){if(0==(64&w)){b=m[(65535&b)+(c&(1<<w)-1)];continue t}if(32&w){r.mode=12;break e}e.msg="invalid literal/length code",r.mode=30;break e}y=65535&b,(w&=15)&&(p<w&&(c+=z[n++]<<p,p+=8),y+=c&(1<<w)-1,c>>>=w,p-=w),p<15&&(c+=z[n++]<<p,p+=8,c+=z[n++]<<p,p+=8),b=_[c&v];r:for(;;){if(c>>>=w=b>>>24,p-=w,!(16&(w=b>>>16&255))){if(0==(64&w)){b=_[(65535&b)+(c&(1<<w)-1)];continue r}e.msg="invalid distance code",r.mode=30;break e}if(k=65535&b,p<(w&=15)&&(c+=z[n++]<<p,(p+=8)<w&&(c+=z[n++]<<p,p+=8)),u<(k+=c&(1<<w)-1)){e.msg="invalid distance too far back",r.mode=30;break e}if(c>>>=w,p-=w,(w=s-a)<k){if(f<(w=k-w)&&r.sane){e.msg="invalid distance too far back",r.mode=30;break e}if(S=d,(x=0)===l){if(x+=h-w,w<y){for(y-=w;E[s++]=d[x++],--w;);x=s-k,S=E}}else if(l<w){if(x+=h+l-w,(w-=l)<y){for(y-=w;E[s++]=d[x++],--w;);if(x=0,l<y){for(y-=w=l;E[s++]=d[x++],--w;);x=s-k,S=E}}}else if(x+=l-w,w<y){for(y-=w;E[s++]=d[x++],--w;);x=s-k,S=E}for(;2<y;)E[s++]=S[x++],E[s++]=S[x++],E[s++]=S[x++],y-=3;y&&(E[s++]=S[x++],1<y&&(E[s++]=S[x++]))}else{for(x=s-k;E[s++]=E[x++],E[s++]=E[x++],E[s++]=E[x++],2<(y-=3););y&&(E[s++]=E[x++],1<y&&(E[s++]=E[x++]))}break}}break}}while(n<i&&s<o);n-=y=p>>3,c&=(1<<(p-=y<<3))-1,e.next_in=n,e.next_out=s,e.avail_in=n<i?i-n+5:5-(n-i),e.avail_out=s<o?o-s+257:257-(s-o),r.hold=c,r.bits=p}},{}],49:[function(e,t,r){"use strict";var I=e("../utils/common"),O=e("./adler32"),B=e("./crc32"),T=e("./inffast"),R=e("./inftrees"),D=1,F=2,N=0,U=-2,P=1,n=852,i=592;function L(e){return(e>>>24&255)+(e>>>8&65280)+((65280&e)<<8)+((255&e)<<24)}function s(){this.mode=0,this.last=!1,this.wrap=0,this.havedict=!1,this.flags=0,this.dmax=0,this.check=0,this.total=0,this.head=null,this.wbits=0,this.wsize=0,this.whave=0,this.wnext=0,this.window=null,this.hold=0,this.bits=0,this.length=0,this.offset=0,this.extra=0,this.lencode=null,this.distcode=null,this.lenbits=0,this.distbits=0,this.ncode=0,this.nlen=0,this.ndist=0,this.have=0,this.next=null,this.lens=new I.Buf16(320),this.work=new I.Buf16(288),this.lendyn=null,this.distdyn=null,this.sane=0,this.back=0,this.was=0}function a(e){var t;return e&&e.state?(t=e.state,e.total_in=e.total_out=t.total=0,e.msg="",t.wrap&&(e.adler=1&t.wrap),t.mode=P,t.last=0,t.havedict=0,t.dmax=32768,t.head=null,t.hold=0,t.bits=0,t.lencode=t.lendyn=new I.Buf32(n),t.distcode=t.distdyn=new I.Buf32(i),t.sane=1,t.back=-1,N):U}function o(e){var t;return e&&e.state?((t=e.state).wsize=0,t.whave=0,t.wnext=0,a(e)):U}function u(e,t){var r,n;return e&&e.state?(n=e.state,t<0?(r=0,t=-t):(r=1+(t>>4),t<48&&(t&=15)),t&&(t<8||15<t)?U:(null!==n.window&&n.wbits!==t&&(n.window=null),n.wrap=r,n.wbits=t,o(e))):U}function h(e,t){var r,n;return e?(n=new s,(e.state=n).window=null,(r=u(e,t))!==N&&(e.state=null),r):U}var f,l,d=!0;function j(e){if(d){var t;for(f=new I.Buf32(512),l=new I.Buf32(32),t=0;t<144;)e.lens[t++]=8;for(;t<256;)e.lens[t++]=9;for(;t<280;)e.lens[t++]=7;for(;t<288;)e.lens[t++]=8;for(R(D,e.lens,0,288,f,0,e.work,{bits:9}),t=0;t<32;)e.lens[t++]=5;R(F,e.lens,0,32,l,0,e.work,{bits:5}),d=!1}e.lencode=f,e.lenbits=9,e.distcode=l,e.distbits=5}function Z(e,t,r,n){var i,s=e.state;return null===s.window&&(s.wsize=1<<s.wbits,s.wnext=0,s.whave=0,s.window=new I.Buf8(s.wsize)),n>=s.wsize?(I.arraySet(s.window,t,r-s.wsize,s.wsize,0),s.wnext=0,s.whave=s.wsize):(n<(i=s.wsize-s.wnext)&&(i=n),I.arraySet(s.window,t,r-n,i,s.wnext),(n-=i)?(I.arraySet(s.window,t,r-n,n,0),s.wnext=n,s.whave=s.wsize):(s.wnext+=i,s.wnext===s.wsize&&(s.wnext=0),s.whave<s.wsize&&(s.whave+=i))),0}r.inflateReset=o,r.inflateReset2=u,r.inflateResetKeep=a,r.inflateInit=function(e){return h(e,15)},r.inflateInit2=h,r.inflate=function(e,t){var r,n,i,s,a,o,u,h,f,l,d,c,p,m,_,g,v,b,w,y,k,x,S,z,E=0,C=new I.Buf8(4),A=[16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15];if(!e||!e.state||!e.output||!e.input&&0!==e.avail_in)return U;12===(r=e.state).mode&&(r.mode=13),a=e.next_out,i=e.output,u=e.avail_out,s=e.next_in,n=e.input,o=e.avail_in,h=r.hold,f=r.bits,l=o,d=u,x=N;e:for(;;)switch(r.mode){case P:if(0===r.wrap){r.mode=13;break}for(;f<16;){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}if(2&r.wrap&&35615===h){C[r.check=0]=255&h,C[1]=h>>>8&255,r.check=B(r.check,C,2,0),f=h=0,r.mode=2;break}if(r.flags=0,r.head&&(r.head.done=!1),!(1&r.wrap)||(((255&h)<<8)+(h>>8))%31){e.msg="incorrect header check",r.mode=30;break}if(8!=(15&h)){e.msg="unknown compression method",r.mode=30;break}if(f-=4,k=8+(15&(h>>>=4)),0===r.wbits)r.wbits=k;else if(k>r.wbits){e.msg="invalid window size",r.mode=30;break}r.dmax=1<<k,e.adler=r.check=1,r.mode=512&h?10:12,f=h=0;break;case 2:for(;f<16;){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}if(r.flags=h,8!=(255&r.flags)){e.msg="unknown compression method",r.mode=30;break}if(57344&r.flags){e.msg="unknown header flags set",r.mode=30;break}r.head&&(r.head.text=h>>8&1),512&r.flags&&(C[0]=255&h,C[1]=h>>>8&255,r.check=B(r.check,C,2,0)),f=h=0,r.mode=3;case 3:for(;f<32;){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}r.head&&(r.head.time=h),512&r.flags&&(C[0]=255&h,C[1]=h>>>8&255,C[2]=h>>>16&255,C[3]=h>>>24&255,r.check=B(r.check,C,4,0)),f=h=0,r.mode=4;case 4:for(;f<16;){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}r.head&&(r.head.xflags=255&h,r.head.os=h>>8),512&r.flags&&(C[0]=255&h,C[1]=h>>>8&255,r.check=B(r.check,C,2,0)),f=h=0,r.mode=5;case 5:if(1024&r.flags){for(;f<16;){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}r.length=h,r.head&&(r.head.extra_len=h),512&r.flags&&(C[0]=255&h,C[1]=h>>>8&255,r.check=B(r.check,C,2,0)),f=h=0}else r.head&&(r.head.extra=null);r.mode=6;case 6:if(1024&r.flags&&(o<(c=r.length)&&(c=o),c&&(r.head&&(k=r.head.extra_len-r.length,r.head.extra||(r.head.extra=new Array(r.head.extra_len)),I.arraySet(r.head.extra,n,s,c,k)),512&r.flags&&(r.check=B(r.check,n,c,s)),o-=c,s+=c,r.length-=c),r.length))break e;r.length=0,r.mode=7;case 7:if(2048&r.flags){if(0===o)break e;for(c=0;k=n[s+c++],r.head&&k&&r.length<65536&&(r.head.name+=String.fromCharCode(k)),k&&c<o;);if(512&r.flags&&(r.check=B(r.check,n,c,s)),o-=c,s+=c,k)break e}else r.head&&(r.head.name=null);r.length=0,r.mode=8;case 8:if(4096&r.flags){if(0===o)break e;for(c=0;k=n[s+c++],r.head&&k&&r.length<65536&&(r.head.comment+=String.fromCharCode(k)),k&&c<o;);if(512&r.flags&&(r.check=B(r.check,n,c,s)),o-=c,s+=c,k)break e}else r.head&&(r.head.comment=null);r.mode=9;case 9:if(512&r.flags){for(;f<16;){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}if(h!==(65535&r.check)){e.msg="header crc mismatch",r.mode=30;break}f=h=0}r.head&&(r.head.hcrc=r.flags>>9&1,r.head.done=!0),e.adler=r.check=0,r.mode=12;break;case 10:for(;f<32;){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}e.adler=r.check=L(h),f=h=0,r.mode=11;case 11:if(0===r.havedict)return e.next_out=a,e.avail_out=u,e.next_in=s,e.avail_in=o,r.hold=h,r.bits=f,2;e.adler=r.check=1,r.mode=12;case 12:if(5===t||6===t)break e;case 13:if(r.last){h>>>=7&f,f-=7&f,r.mode=27;break}for(;f<3;){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}switch(r.last=1&h,f-=1,3&(h>>>=1)){case 0:r.mode=14;break;case 1:if(j(r),r.mode=20,6!==t)break;h>>>=2,f-=2;break e;case 2:r.mode=17;break;case 3:e.msg="invalid block type",r.mode=30}h>>>=2,f-=2;break;case 14:for(h>>>=7&f,f-=7&f;f<32;){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}if((65535&h)!=(h>>>16^65535)){e.msg="invalid stored block lengths",r.mode=30;break}if(r.length=65535&h,f=h=0,r.mode=15,6===t)break e;case 15:r.mode=16;case 16:if(c=r.length){if(o<c&&(c=o),u<c&&(c=u),0===c)break e;I.arraySet(i,n,s,c,a),o-=c,s+=c,u-=c,a+=c,r.length-=c;break}r.mode=12;break;case 17:for(;f<14;){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}if(r.nlen=257+(31&h),h>>>=5,f-=5,r.ndist=1+(31&h),h>>>=5,f-=5,r.ncode=4+(15&h),h>>>=4,f-=4,286<r.nlen||30<r.ndist){e.msg="too many length or distance symbols",r.mode=30;break}r.have=0,r.mode=18;case 18:for(;r.have<r.ncode;){for(;f<3;){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}r.lens[A[r.have++]]=7&h,h>>>=3,f-=3}for(;r.have<19;)r.lens[A[r.have++]]=0;if(r.lencode=r.lendyn,r.lenbits=7,S={bits:r.lenbits},x=R(0,r.lens,0,19,r.lencode,0,r.work,S),r.lenbits=S.bits,x){e.msg="invalid code lengths set",r.mode=30;break}r.have=0,r.mode=19;case 19:for(;r.have<r.nlen+r.ndist;){for(;g=(E=r.lencode[h&(1<<r.lenbits)-1])>>>16&255,v=65535&E,!((_=E>>>24)<=f);){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}if(v<16)h>>>=_,f-=_,r.lens[r.have++]=v;else{if(16===v){for(z=_+2;f<z;){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}if(h>>>=_,f-=_,0===r.have){e.msg="invalid bit length repeat",r.mode=30;break}k=r.lens[r.have-1],c=3+(3&h),h>>>=2,f-=2}else if(17===v){for(z=_+3;f<z;){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}f-=_,k=0,c=3+(7&(h>>>=_)),h>>>=3,f-=3}else{for(z=_+7;f<z;){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}f-=_,k=0,c=11+(127&(h>>>=_)),h>>>=7,f-=7}if(r.have+c>r.nlen+r.ndist){e.msg="invalid bit length repeat",r.mode=30;break}for(;c--;)r.lens[r.have++]=k}}if(30===r.mode)break;if(0===r.lens[256]){e.msg="invalid code -- missing end-of-block",r.mode=30;break}if(r.lenbits=9,S={bits:r.lenbits},x=R(D,r.lens,0,r.nlen,r.lencode,0,r.work,S),r.lenbits=S.bits,x){e.msg="invalid literal/lengths set",r.mode=30;break}if(r.distbits=6,r.distcode=r.distdyn,S={bits:r.distbits},x=R(F,r.lens,r.nlen,r.ndist,r.distcode,0,r.work,S),r.distbits=S.bits,x){e.msg="invalid distances set",r.mode=30;break}if(r.mode=20,6===t)break e;case 20:r.mode=21;case 21:if(6<=o&&258<=u){e.next_out=a,e.avail_out=u,e.next_in=s,e.avail_in=o,r.hold=h,r.bits=f,T(e,d),a=e.next_out,i=e.output,u=e.avail_out,s=e.next_in,n=e.input,o=e.avail_in,h=r.hold,f=r.bits,12===r.mode&&(r.back=-1);break}for(r.back=0;g=(E=r.lencode[h&(1<<r.lenbits)-1])>>>16&255,v=65535&E,!((_=E>>>24)<=f);){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}if(g&&0==(240&g)){for(b=_,w=g,y=v;g=(E=r.lencode[y+((h&(1<<b+w)-1)>>b)])>>>16&255,v=65535&E,!(b+(_=E>>>24)<=f);){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}h>>>=b,f-=b,r.back+=b}if(h>>>=_,f-=_,r.back+=_,r.length=v,0===g){r.mode=26;break}if(32&g){r.back=-1,r.mode=12;break}if(64&g){e.msg="invalid literal/length code",r.mode=30;break}r.extra=15&g,r.mode=22;case 22:if(r.extra){for(z=r.extra;f<z;){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}r.length+=h&(1<<r.extra)-1,h>>>=r.extra,f-=r.extra,r.back+=r.extra}r.was=r.length,r.mode=23;case 23:for(;g=(E=r.distcode[h&(1<<r.distbits)-1])>>>16&255,v=65535&E,!((_=E>>>24)<=f);){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}if(0==(240&g)){for(b=_,w=g,y=v;g=(E=r.distcode[y+((h&(1<<b+w)-1)>>b)])>>>16&255,v=65535&E,!(b+(_=E>>>24)<=f);){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}h>>>=b,f-=b,r.back+=b}if(h>>>=_,f-=_,r.back+=_,64&g){e.msg="invalid distance code",r.mode=30;break}r.offset=v,r.extra=15&g,r.mode=24;case 24:if(r.extra){for(z=r.extra;f<z;){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}r.offset+=h&(1<<r.extra)-1,h>>>=r.extra,f-=r.extra,r.back+=r.extra}if(r.offset>r.dmax){e.msg="invalid distance too far back",r.mode=30;break}r.mode=25;case 25:if(0===u)break e;if(c=d-u,r.offset>c){if((c=r.offset-c)>r.whave&&r.sane){e.msg="invalid distance too far back",r.mode=30;break}p=c>r.wnext?(c-=r.wnext,r.wsize-c):r.wnext-c,c>r.length&&(c=r.length),m=r.window}else m=i,p=a-r.offset,c=r.length;for(u<c&&(c=u),u-=c,r.length-=c;i[a++]=m[p++],--c;);0===r.length&&(r.mode=21);break;case 26:if(0===u)break e;i[a++]=r.length,u--,r.mode=21;break;case 27:if(r.wrap){for(;f<32;){if(0===o)break e;o--,h|=n[s++]<<f,f+=8}if(d-=u,e.total_out+=d,r.total+=d,d&&(e.adler=r.check=r.flags?B(r.check,i,d,a-d):O(r.check,i,d,a-d)),d=u,(r.flags?h:L(h))!==r.check){e.msg="incorrect data check",r.mode=30;break}f=h=0}r.mode=28;case 28:if(r.wrap&&r.flags){for(;f<32;){if(0===o)break e;o--,h+=n[s++]<<f,f+=8}if(h!==(4294967295&r.total)){e.msg="incorrect length check",r.mode=30;break}f=h=0}r.mode=29;case 29:x=1;break e;case 30:x=-3;break e;case 31:return-4;case 32:default:return U}return e.next_out=a,e.avail_out=u,e.next_in=s,e.avail_in=o,r.hold=h,r.bits=f,(r.wsize||d!==e.avail_out&&r.mode<30&&(r.mode<27||4!==t))&&Z(e,e.output,e.next_out,d-e.avail_out)?(r.mode=31,-4):(l-=e.avail_in,d-=e.avail_out,e.total_in+=l,e.total_out+=d,r.total+=d,r.wrap&&d&&(e.adler=r.check=r.flags?B(r.check,i,d,e.next_out-d):O(r.check,i,d,e.next_out-d)),e.data_type=r.bits+(r.last?64:0)+(12===r.mode?128:0)+(20===r.mode||15===r.mode?256:0),(0==l&&0===d||4===t)&&x===N&&(x=-5),x)},r.inflateEnd=function(e){if(!e||!e.state)return U;var t=e.state;return t.window&&(t.window=null),e.state=null,N},r.inflateGetHeader=function(e,t){var r;return e&&e.state?0==(2&(r=e.state).wrap)?U:((r.head=t).done=!1,N):U},r.inflateSetDictionary=function(e,t){var r,n=t.length;return e&&e.state?0!==(r=e.state).wrap&&11!==r.mode?U:11===r.mode&&O(1,t,n,0)!==r.check?-3:Z(e,t,n,n)?(r.mode=31,-4):(r.havedict=1,N):U},r.inflateInfo="pako inflate (from Nodeca project)"},{"../utils/common":41,"./adler32":43,"./crc32":45,"./inffast":48,"./inftrees":50}],50:[function(e,t,r){"use strict";var D=e("../utils/common"),F=[3,4,5,6,7,8,9,10,11,13,15,17,19,23,27,31,35,43,51,59,67,83,99,115,131,163,195,227,258,0,0],N=[16,16,16,16,16,16,16,16,17,17,17,17,18,18,18,18,19,19,19,19,20,20,20,20,21,21,21,21,16,72,78],U=[1,2,3,4,5,7,9,13,17,25,33,49,65,97,129,193,257,385,513,769,1025,1537,2049,3073,4097,6145,8193,12289,16385,24577,0,0],P=[16,16,16,16,17,17,18,18,19,19,20,20,21,21,22,22,23,23,24,24,25,25,26,26,27,27,28,28,29,29,64,64];t.exports=function(e,t,r,n,i,s,a,o){var u,h,f,l,d,c,p,m,_,g=o.bits,v=0,b=0,w=0,y=0,k=0,x=0,S=0,z=0,E=0,C=0,A=null,I=0,O=new D.Buf16(16),B=new D.Buf16(16),T=null,R=0;for(v=0;v<=15;v++)O[v]=0;for(b=0;b<n;b++)O[t[r+b]]++;for(k=g,y=15;1<=y&&0===O[y];y--);if(y<k&&(k=y),0===y)return i[s++]=20971520,i[s++]=20971520,o.bits=1,0;for(w=1;w<y&&0===O[w];w++);for(k<w&&(k=w),v=z=1;v<=15;v++)if(z<<=1,(z-=O[v])<0)return-1;if(0<z&&(0===e||1!==y))return-1;for(B[1]=0,v=1;v<15;v++)B[v+1]=B[v]+O[v];for(b=0;b<n;b++)0!==t[r+b]&&(a[B[t[r+b]]++]=b);if(c=0===e?(A=T=a,19):1===e?(A=F,I-=257,T=N,R-=257,256):(A=U,T=P,-1),v=w,d=s,S=b=C=0,f=-1,l=(E=1<<(x=k))-1,1===e&&852<E||2===e&&592<E)return 1;for(;;){for(p=v-S,_=a[b]<c?(m=0,a[b]):a[b]>c?(m=T[R+a[b]],A[I+a[b]]):(m=96,0),u=1<<v-S,w=h=1<<x;i[d+(C>>S)+(h-=u)]=p<<24|m<<16|_|0,0!==h;);for(u=1<<v-1;C&u;)u>>=1;if(0!==u?(C&=u-1,C+=u):C=0,b++,0==--O[v]){if(v===y)break;v=t[r+a[b]]}if(k<v&&(C&l)!==f){for(0===S&&(S=k),d+=w,z=1<<(x=v-S);x+S<y&&!((z-=O[x+S])<=0);)x++,z<<=1;if(E+=1<<x,1===e&&852<E||2===e&&592<E)return 1;i[f=C&l]=k<<24|x<<16|d-s|0}}return 0!==C&&(i[d+C]=v-S<<24|64<<16|0),o.bits=k,0}},{"../utils/common":41}],51:[function(e,t,r){"use strict";t.exports={2:"need dictionary",1:"stream end",0:"","-1":"file error","-2":"stream error","-3":"data error","-4":"insufficient memory","-5":"buffer error","-6":"incompatible version"}},{}],52:[function(e,t,r){"use strict";var o=e("../utils/common");function n(e){for(var t=e.length;0<=--t;)e[t]=0}var _=15,i=16,u=[0,0,0,0,0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5,0],h=[0,0,0,0,1,1,2,2,3,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10,11,11,12,12,13,13],a=[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,7],f=[16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15],l=new Array(576);n(l);var d=new Array(60);n(d);var c=new Array(512);n(c);var p=new Array(256);n(p);var m=new Array(29);n(m);var g,v,b,w=new Array(30);function y(e,t,r,n,i){this.static_tree=e,this.extra_bits=t,this.extra_base=r,this.elems=n,this.max_length=i,this.has_stree=e&&e.length}function s(e,t){this.dyn_tree=e,this.max_code=0,this.stat_desc=t}function k(e){return e<256?c[e]:c[256+(e>>>7)]}function x(e,t){e.pending_buf[e.pending++]=255&t,e.pending_buf[e.pending++]=t>>>8&255}function S(e,t,r){e.bi_valid>i-r?(e.bi_buf|=t<<e.bi_valid&65535,x(e,e.bi_buf),e.bi_buf=t>>i-e.bi_valid,e.bi_valid+=r-i):(e.bi_buf|=t<<e.bi_valid&65535,e.bi_valid+=r)}function z(e,t,r){S(e,r[2*t],r[2*t+1])}function E(e,t){for(var r=0;r|=1&e,e>>>=1,r<<=1,0<--t;);return r>>>1}function C(e,t,r){var n,i,s=new Array(_+1),a=0;for(n=1;n<=_;n++)s[n]=a=a+r[n-1]<<1;for(i=0;i<=t;i++){var o=e[2*i+1];0!==o&&(e[2*i]=E(s[o]++,o))}}function A(e){var t;for(t=0;t<286;t++)e.dyn_ltree[2*t]=0;for(t=0;t<30;t++)e.dyn_dtree[2*t]=0;for(t=0;t<19;t++)e.bl_tree[2*t]=0;e.dyn_ltree[512]=1,e.opt_len=e.static_len=0,e.last_lit=e.matches=0}function I(e){8<e.bi_valid?x(e,e.bi_buf):0<e.bi_valid&&(e.pending_buf[e.pending++]=e.bi_buf),e.bi_buf=0,e.bi_valid=0}function O(e,t,r,n){var i=2*t,s=2*r;return e[i]<e[s]||e[i]===e[s]&&n[t]<=n[r]}function B(e,t,r){for(var n=e.heap[r],i=r<<1;i<=e.heap_len&&(i<e.heap_len&&O(t,e.heap[i+1],e.heap[i],e.depth)&&i++,!O(t,n,e.heap[i],e.depth));)e.heap[r]=e.heap[i],r=i,i<<=1;e.heap[r]=n}function T(e,t,r){var n,i,s,a,o=0;if(0!==e.last_lit)for(;n=e.pending_buf[e.d_buf+2*o]<<8|e.pending_buf[e.d_buf+2*o+1],i=e.pending_buf[e.l_buf+o],o++,0===n?z(e,i,t):(z(e,(s=p[i])+256+1,t),0!==(a=u[s])&&S(e,i-=m[s],a),z(e,s=k(--n),r),0!==(a=h[s])&&S(e,n-=w[s],a)),o<e.last_lit;);z(e,256,t)}function R(e,t){var r,n,i,s=t.dyn_tree,a=t.stat_desc.static_tree,o=t.stat_desc.has_stree,u=t.stat_desc.elems,h=-1;for(e.heap_len=0,e.heap_max=573,r=0;r<u;r++)0!==s[2*r]?(e.heap[++e.heap_len]=h=r,e.depth[r]=0):s[2*r+1]=0;for(;e.heap_len<2;)s[2*(i=e.heap[++e.heap_len]=h<2?++h:0)]=1,e.depth[i]=0,e.opt_len--,o&&(e.static_len-=a[2*i+1]);for(t.max_code=h,r=e.heap_len>>1;1<=r;r--)B(e,s,r);for(i=u;r=e.heap[1],e.heap[1]=e.heap[e.heap_len--],B(e,s,1),n=e.heap[1],e.heap[--e.heap_max]=r,e.heap[--e.heap_max]=n,s[2*i]=s[2*r]+s[2*n],e.depth[i]=(e.depth[r]>=e.depth[n]?e.depth[r]:e.depth[n])+1,s[2*r+1]=s[2*n+1]=i,e.heap[1]=i++,B(e,s,1),2<=e.heap_len;);e.heap[--e.heap_max]=e.heap[1],function(e,t){var r,n,i,s,a,o,u=t.dyn_tree,h=t.max_code,f=t.stat_desc.static_tree,l=t.stat_desc.has_stree,d=t.stat_desc.extra_bits,c=t.stat_desc.extra_base,p=t.stat_desc.max_length,m=0;for(s=0;s<=_;s++)e.bl_count[s]=0;for(u[2*e.heap[e.heap_max]+1]=0,r=e.heap_max+1;r<573;r++)p<(s=u[2*u[2*(n=e.heap[r])+1]+1]+1)&&(s=p,m++),u[2*n+1]=s,h<n||(e.bl_count[s]++,a=0,c<=n&&(a=d[n-c]),o=u[2*n],e.opt_len+=o*(s+a),l&&(e.static_len+=o*(f[2*n+1]+a)));if(0!==m){do{for(s=p-1;0===e.bl_count[s];)s--;e.bl_count[s]--,e.bl_count[s+1]+=2,e.bl_count[p]--,m-=2}while(0<m);for(s=p;0!==s;s--)for(n=e.bl_count[s];0!==n;)h<(i=e.heap[--r])||(u[2*i+1]!==s&&(e.opt_len+=(s-u[2*i+1])*u[2*i],u[2*i+1]=s),n--)}}(e,t),C(s,h,e.bl_count)}function D(e,t,r){var n,i,s=-1,a=t[1],o=0,u=7,h=4;for(0===a&&(u=138,h=3),t[2*(r+1)+1]=65535,n=0;n<=r;n++)i=a,a=t[2*(n+1)+1],++o<u&&i===a||(o<h?e.bl_tree[2*i]+=o:0!==i?(i!==s&&e.bl_tree[2*i]++,e.bl_tree[32]++):o<=10?e.bl_tree[34]++:e.bl_tree[36]++,s=i,h=(o=0)===a?(u=138,3):i===a?(u=6,3):(u=7,4))}function F(e,t,r){var n,i,s=-1,a=t[1],o=0,u=7,h=4;for(0===a&&(u=138,h=3),n=0;n<=r;n++)if(i=a,a=t[2*(n+1)+1],!(++o<u&&i===a)){if(o<h)for(;z(e,i,e.bl_tree),0!=--o;);else 0!==i?(i!==s&&(z(e,i,e.bl_tree),o--),z(e,16,e.bl_tree),S(e,o-3,2)):o<=10?(z(e,17,e.bl_tree),S(e,o-3,3)):(z(e,18,e.bl_tree),S(e,o-11,7));s=i,h=(o=0)===a?(u=138,3):i===a?(u=6,3):(u=7,4)}}n(w);var N=!1;function U(e,t,r,n){var i,s,a;S(e,0+(n?1:0),3),s=t,a=r,I(i=e),x(i,a),x(i,~a),o.arraySet(i.pending_buf,i.window,s,a,i.pending),i.pending+=a}r._tr_init=function(e){N||(function(){var e,t,r,n,i,s=new Array(_+1);for(n=r=0;n<28;n++)for(m[n]=r,e=0;e<1<<u[n];e++)p[r++]=n;for(p[r-1]=n,n=i=0;n<16;n++)for(w[n]=i,e=0;e<1<<h[n];e++)c[i++]=n;for(i>>=7;n<30;n++)for(w[n]=i<<7,e=0;e<1<<h[n]-7;e++)c[256+i++]=n;for(t=0;t<=_;t++)s[t]=0;for(e=0;e<=143;)l[2*e+1]=8,e++,s[8]++;for(;e<=255;)l[2*e+1]=9,e++,s[9]++;for(;e<=279;)l[2*e+1]=7,e++,s[7]++;for(;e<=287;)l[2*e+1]=8,e++,s[8]++;for(C(l,287,s),e=0;e<30;e++)d[2*e+1]=5,d[2*e]=E(e,5);g=new y(l,u,257,286,_),v=new y(d,h,0,30,_),b=new y(new Array(0),a,0,19,7)}(),N=!0),e.l_desc=new s(e.dyn_ltree,g),e.d_desc=new s(e.dyn_dtree,v),e.bl_desc=new s(e.bl_tree,b),e.bi_buf=0,e.bi_valid=0,A(e)},r._tr_stored_block=U,r._tr_flush_block=function(e,t,r,n){var i,s,a=0;0<e.level?(2===e.strm.data_type&&(e.strm.data_type=function(e){var t,r=4093624447;for(t=0;t<=31;t++,r>>>=1)if(1&r&&0!==e.dyn_ltree[2*t])return 0;if(0!==e.dyn_ltree[18]||0!==e.dyn_ltree[20]||0!==e.dyn_ltree[26])return 1;for(t=32;t<256;t++)if(0!==e.dyn_ltree[2*t])return 1;return 0}(e)),R(e,e.l_desc),R(e,e.d_desc),a=function(e){var t;for(D(e,e.dyn_ltree,e.l_desc.max_code),D(e,e.dyn_dtree,e.d_desc.max_code),R(e,e.bl_desc),t=18;3<=t&&0===e.bl_tree[2*f[t]+1];t--);return e.opt_len+=3*(t+1)+5+5+4,t}(e),i=e.opt_len+3+7>>>3,(s=e.static_len+3+7>>>3)<=i&&(i=s)):i=s=r+5,r+4<=i&&-1!==t?U(e,t,r,n):4===e.strategy||s===i?(S(e,2+(n?1:0),3),T(e,l,d)):(S(e,4+(n?1:0),3),function(e,t,r,n){var i;for(S(e,t-257,5),S(e,r-1,5),S(e,n-4,4),i=0;i<n;i++)S(e,e.bl_tree[2*f[i]+1],3);F(e,e.dyn_ltree,t-1),F(e,e.dyn_dtree,r-1)}(e,e.l_desc.max_code+1,e.d_desc.max_code+1,a+1),T(e,e.dyn_ltree,e.dyn_dtree)),A(e),n&&I(e)},r._tr_tally=function(e,t,r){return e.pending_buf[e.d_buf+2*e.last_lit]=t>>>8&255,e.pending_buf[e.d_buf+2*e.last_lit+1]=255&t,e.pending_buf[e.l_buf+e.last_lit]=255&r,e.last_lit++,0===t?e.dyn_ltree[2*r]++:(e.matches++,t--,e.dyn_ltree[2*(p[r]+256+1)]++,e.dyn_dtree[2*k(t)]++),e.last_lit===e.lit_bufsize-1},r._tr_align=function(e){var t;S(e,2,3),z(e,256,l),16===(t=e).bi_valid?(x(t,t.bi_buf),t.bi_buf=0,t.bi_valid=0):8<=t.bi_valid&&(t.pending_buf[t.pending++]=255&t.bi_buf,t.bi_buf>>=8,t.bi_valid-=8)}},{"../utils/common":41}],53:[function(e,t,r){"use strict";t.exports=function(){this.input=null,this.next_in=0,this.avail_in=0,this.total_in=0,this.output=null,this.next_out=0,this.avail_out=0,this.total_out=0,this.msg="",this.state=null,this.data_type=2,this.adler=0}},{}],54:[function(e,t,r){"use strict";t.exports="function"==typeof setImmediate?setImmediate:function(){var e=[].slice.apply(arguments);e.splice(1,0,0),setTimeout.apply(null,e)}},{}]},{},[10])(10)})}).call(this,void 0!==r?r:"undefined"!=typeof self?self:"undefined"!=typeof window?window:{})},{}]},{},[1])(1)})}).call(this,void 0!==r?r:"undefined"!=typeof self?self:"undefined"!=typeof window?window:{})},{}]},{},[1])(1)})}).call(this,void 0!==r?r:"undefined"!=typeof self?self:"undefined"!=typeof window?window:{})},{}]},{},[1])(1)})}).call(this,void 0!==r?r:"undefined"!=typeof self?self:"undefined"!=typeof window?window:{})},{}]},{},[1])(1)})}).call(this,"undefined"!=typeof global?global:"undefined"!=typeof self?self:"undefined"!=typeof window?window:{})},{}]},{},[1])(1)});
/*! pdfmake v0.1.71, @license MIT, @link http://pdfmake.org */
!function webpackUniversalModuleDefinition(i,o){if("object"==typeof exports&&"object"==typeof module)module.exports=o();else if("function"==typeof define&&define.amd)define([],o);else{var u=o();for(var p in u)("object"==typeof exports?exports:i)[p]=u[p]}}("undefined"!=typeof self?self:this,(function(){return function(i){var o={};function __webpack_require__(u){if(o[u])return o[u].exports;var p=o[u]={i:u,l:!1,exports:{}};return i[u].call(p.exports,p,p.exports,__webpack_require__),p.l=!0,p.exports}return __webpack_require__.m=i,__webpack_require__.c=o,__webpack_require__.d=function(i,o,u){__webpack_require__.o(i,o)||Object.defineProperty(i,o,{enumerable:!0,get:u})},__webpack_require__.r=function(i){"undefined"!=typeof Symbol&&Symbol.toStringTag&&Object.defineProperty(i,Symbol.toStringTag,{value:"Module"}),Object.defineProperty(i,"__esModule",{value:!0})},__webpack_require__.t=function(i,o){if(1&o&&(i=__webpack_require__(i)),8&o)return i;if(4&o&&"object"==typeof i&&i&&i.__esModule)return i;var u=Object.create(null);if(__webpack_require__.r(u),Object.defineProperty(u,"default",{enumerable:!0,value:i}),2&o&&"string"!=typeof i)for(var p in i)__webpack_require__.d(u,p,function(o){return i[o]}.bind(null,p));return u},__webpack_require__.n=function(i){var o=i&&i.__esModule?function getDefault(){return i.default}:function getModuleExports(){return i};return __webpack_require__.d(o,"a",o),o},__webpack_require__.o=function(i,o){return Object.prototype.hasOwnProperty.call(i,o)},__webpack_require__.p="",__webpack_require__(__webpack_require__.s=302)}([function(i,o,u){"use strict";function isArray(i){return Array.isArray(i)}i.exports={isString:function isString(i){return"string"==typeof i||i instanceof String},isNumber:function isNumber(i){return"number"==typeof i||i instanceof Number},isBoolean:function isBoolean(i){return"boolean"==typeof i},isArray:isArray,isFunction:function isFunction(i){return"function"==typeof i},isObject:function isObject(i){return null!==i&&"object"==typeof i},isNull:function isNull(i){return null===i},isUndefined:function isUndefined(i){return void 0===i},pack:function pack(){for(var i={},o=0,u=arguments.length;o<u;o++){var p=arguments[o];if(p)for(var g in p)p.hasOwnProperty(g)&&(i[g]=p[g])}return i},fontStringify:function fontStringify(i,o){return"font"===i?"font":o},offsetVector:function offsetVector(i,o,u){switch(i.type){case"ellipse":case"rect":i.x+=o,i.y+=u;break;case"line":i.x1+=o,i.x2+=o,i.y1+=u,i.y2+=u;break;case"polyline":for(var p=0,g=i.points.length;p<g;p++)i.points[p].x+=o,i.points[p].y+=u}},getNodeId:function getNodeId(i){if(i.id)return i.id;if(isArray(i.text))for(var o=0,u=i.text.length;o<u;o++){var p=getNodeId(i.text[o]);if(p)return p}return null}}},function(i,o,u){var p=u(4),g=u(34).f,y=u(18),w=u(25),_=u(141),x=u(204),k=u(146);i.exports=function(i,o){var u,P,E,O,I,B=i.target,D=i.global,R=i.stat;if(u=D?p:R?p[B]||_(B,{}):(p[B]||{}).prototype)for(P in o){if(O=o[P],E=i.noTargetGet?(I=g(u,P))&&I.value:u[P],!k(D?P:B+(R?".":"#")+P,i.forced)&&void 0!==E){if(typeof O==typeof E)continue;x(O,E)}(i.sham||E&&E.sham)&&y(O,"sham",!0),w(u,P,O,i)}}},function(i,o,u){var p;i.exports=(p=p||function(i,o){var u=Object.create||function(){function F(){}return function(i){var o;return F.prototype=i,o=new F,F.prototype=null,o}}(),p={},g=p.lib={},y=g.Base={extend:function(i){var o=u(this);return i&&o.mixIn(i),o.hasOwnProperty("init")&&this.init!==o.init||(o.init=function(){o.$super.init.apply(this,arguments)}),o.init.prototype=o,o.$super=this,o},create:function(){var i=this.extend();return i.init.apply(i,arguments),i},init:function(){},mixIn:function(i){for(var o in i)i.hasOwnProperty(o)&&(this[o]=i[o]);i.hasOwnProperty("toString")&&(this.toString=i.toString)},clone:function(){return this.init.prototype.extend(this)}},w=g.WordArray=y.extend({init:function(i,o){i=this.words=i||[],this.sigBytes=null!=o?o:4*i.length},toString:function(i){return(i||x).stringify(this)},concat:function(i){var o=this.words,u=i.words,p=this.sigBytes,g=i.sigBytes;if(this.clamp(),p%4)for(var y=0;y<g;y++){var w=u[y>>>2]>>>24-y%4*8&255;o[p+y>>>2]|=w<<24-(p+y)%4*8}else for(y=0;y<g;y+=4)o[p+y>>>2]=u[y>>>2];return this.sigBytes+=g,this},clamp:function(){var o=this.words,u=this.sigBytes;o[u>>>2]&=4294967295<<32-u%4*8,o.length=i.ceil(u/4)},clone:function(){var i=y.clone.call(this);return i.words=this.words.slice(0),i},random:function(o){for(var u,p=[],r=function(o){o=o;var u=987654321,p=4294967295;return function(){var g=((u=36969*(65535&u)+(u>>16)&p)<<16)+(o=18e3*(65535&o)+(o>>16)&p)&p;return g/=4294967296,(g+=.5)*(i.random()>.5?1:-1)}},g=0;g<o;g+=4){var y=r(4294967296*(u||i.random()));u=987654071*y(),p.push(4294967296*y()|0)}return new w.init(p,o)}}),_=p.enc={},x=_.Hex={stringify:function(i){for(var o=i.words,u=i.sigBytes,p=[],g=0;g<u;g++){var y=o[g>>>2]>>>24-g%4*8&255;p.push((y>>>4).toString(16)),p.push((15&y).toString(16))}return p.join("")},parse:function(i){for(var o=i.length,u=[],p=0;p<o;p+=2)u[p>>>3]|=parseInt(i.substr(p,2),16)<<24-p%8*4;return new w.init(u,o/2)}},k=_.Latin1={stringify:function(i){for(var o=i.words,u=i.sigBytes,p=[],g=0;g<u;g++){var y=o[g>>>2]>>>24-g%4*8&255;p.push(String.fromCharCode(y))}return p.join("")},parse:function(i){for(var o=i.length,u=[],p=0;p<o;p++)u[p>>>2]|=(255&i.charCodeAt(p))<<24-p%4*8;return new w.init(u,o)}},P=_.Utf8={stringify:function(i){try{return decodeURIComponent(escape(k.stringify(i)))}catch(i){throw new Error("Malformed UTF-8 data")}},parse:function(i){return k.parse(unescape(encodeURIComponent(i)))}},E=g.BufferedBlockAlgorithm=y.extend({reset:function(){this._data=new w.init,this._nDataBytes=0},_append:function(i){"string"==typeof i&&(i=P.parse(i)),this._data.concat(i),this._nDataBytes+=i.sigBytes},_process:function(o){var u=this._data,p=u.words,g=u.sigBytes,y=this.blockSize,_=g/(4*y),x=(_=o?i.ceil(_):i.max((0|_)-this._minBufferSize,0))*y,k=i.min(4*x,g);if(x){for(var P=0;P<x;P+=y)this._doProcessBlock(p,P);var E=p.splice(0,x);u.sigBytes-=k}return new w.init(E,k)},clone:function(){var i=y.clone.call(this);return i._data=this._data.clone(),i},_minBufferSize:0}),O=(g.Hasher=E.extend({cfg:y.extend(),init:function(i){this.cfg=this.cfg.extend(i),this.reset()},reset:function(){E.reset.call(this),this._doReset()},update:function(i){return this._append(i),this._process(),this},finalize:function(i){return i&&this._append(i),this._doFinalize()},blockSize:16,_createHelper:function(i){return function(o,u){return new i.init(u).finalize(o)}},_createHmacHelper:function(i){return function(o,u){return new O.HMAC.init(i,u).finalize(o)}}}),p.algo={});return p}(Math),p)},function(i,o){i.exports=function(i){try{return!!i()}catch(i){return!0}}},function(i,o,u){(function(o){var check=function(i){return i&&i.Math==Math&&i};i.exports=check("object"==typeof globalThis&&globalThis)||check("object"==typeof window&&window)||check("object"==typeof self&&self)||check("object"==typeof o&&o)||function(){return this}()||Function("return this")()}).call(this,u(27))},function(i,o,u){"use strict";var p,g=u(232),y=u(13),w=u(4),_=u(11),x=u(15),k=u(119),P=u(18),E=u(25),O=u(16).f,I=u(58),B=u(46),D=u(6),R=u(110),N=w.Int8Array,U=N&&N.prototype,W=w.Uint8ClampedArray,G=W&&W.prototype,j=N&&I(N),X=U&&I(U),K=Object.prototype,Y=K.isPrototypeOf,J=D("toStringTag"),$=R("TYPED_ARRAY_TAG"),tt=g&&!!B&&"Opera"!==k(w.opera),et=!1,rt={Int8Array:1,Uint8Array:1,Uint8ClampedArray:1,Int16Array:2,Uint16Array:2,Int32Array:4,Uint32Array:4,Float32Array:4,Float64Array:8},nt={BigInt64Array:8,BigUint64Array:8},isTypedArray=function(i){if(!_(i))return!1;var o=k(i);return x(rt,o)||x(nt,o)};for(p in rt)w[p]||(tt=!1);if((!tt||"function"!=typeof j||j===Function.prototype)&&(j=function TypedArray(){throw TypeError("Incorrect invocation")},tt))for(p in rt)w[p]&&B(w[p],j);if((!tt||!X||X===K)&&(X=j.prototype,tt))for(p in rt)w[p]&&B(w[p].prototype,X);if(tt&&I(G)!==X&&B(G,X),y&&!x(X,J))for(p in et=!0,O(X,J,{get:function(){return _(this)?this[$]:void 0}}),rt)w[p]&&P(w[p],$,p);i.exports={NATIVE_ARRAY_BUFFER_VIEWS:tt,TYPED_ARRAY_TAG:et&&$,aTypedArray:function(i){if(isTypedArray(i))return i;throw TypeError("Target is not a typed array")},aTypedArrayConstructor:function(i){if(B){if(Y.call(j,i))return i}else for(var o in rt)if(x(rt,p)){var u=w[o];if(u&&(i===u||Y.call(u,i)))return i}throw TypeError("Target is not a typed array constructor")},exportTypedArrayMethod:function(i,o,u){if(y){if(u)for(var p in rt){var g=w[p];g&&x(g.prototype,i)&&delete g.prototype[i]}X[i]&&!u||E(X,i,u?o:tt&&U[i]||o)}},exportTypedArrayStaticMethod:function(i,o,u){var p,g;if(y){if(B){if(u)for(p in rt)(g=w[p])&&x(g,i)&&delete g[i];if(j[i]&&!u)return;try{return E(j,i,u?o:tt&&N[i]||o)}catch(i){}}for(p in rt)!(g=w[p])||g[i]&&!u||E(g,i,o)}},isView:function isView(i){if(!_(i))return!1;var o=k(i);return"DataView"===o||x(rt,o)||x(nt,o)},isTypedArray:isTypedArray,TypedArray:j,TypedArrayPrototype:X}},function(i,o,u){var p=u(4),g=u(109),y=u(15),w=u(110),_=u(147),x=u(208),k=g("wks"),P=p.Symbol,E=x?P:P&&P.withoutSetter||w;i.exports=function(i){return y(k,i)&&(_||"string"==typeof k[i])||(_&&y(P,i)?k[i]=P[i]:k[i]=E("Symbol."+i)),k[i]}},function(i,o){var u=i.exports={version:"2.6.12"};"number"==typeof __e&&(__e=u)},function(i,o,u){var p=u(36),g=Math.min;i.exports=function(i){return i>0?g(p(i),9007199254740991):0}},function(i,o,u){var p=u(17),g=u(7),y=u(33),w=u(40),_=u(52),$export=function(i,o,u){var x,k,P,E=i&$export.F,O=i&$export.G,I=i&$export.S,B=i&$export.P,D=i&$export.B,R=i&$export.W,N=O?g:g[o]||(g[o]={}),U=N.prototype,W=O?p:I?p[o]:(p[o]||{}).prototype;for(x in O&&(u=o),u)(k=!E&&W&&void 0!==W[x])&&_(N,x)||(P=k?W[x]:u[x],N[x]=O&&"function"!=typeof W[x]?u[x]:D&&k?y(P,p):R&&W[x]==P?function(i){var F=function(o,u,p){if(this instanceof i){switch(arguments.length){case 0:return new i;case 1:return new i(o);case 2:return new i(o,u)}return new i(o,u,p)}return i.apply(this,arguments)};return F.prototype=i.prototype,F}(P):B&&"function"==typeof P?y(Function.call,P):P,B&&((N.virtual||(N.virtual={}))[x]=P,i&$export.R&&U&&!U[x]&&w(U,x,P)))};$export.F=1,$export.G=2,$export.S=4,$export.P=8,$export.B=16,$export.W=32,$export.U=64,$export.R=128,i.exports=$export},function(i,o,u){"use strict";(function(i){var p=u(304),g=u(305),y=u(201);function kMaxLength(){return Buffer.TYPED_ARRAY_SUPPORT?2147483647:1073741823}function createBuffer(i,o){if(kMaxLength()<o)throw new RangeError("Invalid typed array length");return Buffer.TYPED_ARRAY_SUPPORT?(i=new Uint8Array(o)).__proto__=Buffer.prototype:(null===i&&(i=new Buffer(o)),i.length=o),i}function Buffer(i,o,u){if(!(Buffer.TYPED_ARRAY_SUPPORT||this instanceof Buffer))return new Buffer(i,o,u);if("number"==typeof i){if("string"==typeof o)throw new Error("If encoding is specified then the first argument must be a string");return allocUnsafe(this,i)}return from(this,i,o,u)}function from(i,o,u,p){if("number"==typeof o)throw new TypeError('"value" argument must not be a number');return"undefined"!=typeof ArrayBuffer&&o instanceof ArrayBuffer?function fromArrayBuffer(i,o,u,p){if(o.byteLength,u<0||o.byteLength<u)throw new RangeError("'offset' is out of bounds");if(o.byteLength<u+(p||0))throw new RangeError("'length' is out of bounds");o=void 0===u&&void 0===p?new Uint8Array(o):void 0===p?new Uint8Array(o,u):new Uint8Array(o,u,p);Buffer.TYPED_ARRAY_SUPPORT?(i=o).__proto__=Buffer.prototype:i=fromArrayLike(i,o);return i}(i,o,u,p):"string"==typeof o?function fromString(i,o,u){"string"==typeof u&&""!==u||(u="utf8");if(!Buffer.isEncoding(u))throw new TypeError('"encoding" must be a valid string encoding');var p=0|byteLength(o,u),g=(i=createBuffer(i,p)).write(o,u);g!==p&&(i=i.slice(0,g));return i}(i,o,u):function fromObject(i,o){if(Buffer.isBuffer(o)){var u=0|checked(o.length);return 0===(i=createBuffer(i,u)).length||o.copy(i,0,0,u),i}if(o){if("undefined"!=typeof ArrayBuffer&&o.buffer instanceof ArrayBuffer||"length"in o)return"number"!=typeof o.length||function isnan(i){return i!=i}(o.length)?createBuffer(i,0):fromArrayLike(i,o);if("Buffer"===o.type&&y(o.data))return fromArrayLike(i,o.data)}throw new TypeError("First argument must be a string, Buffer, ArrayBuffer, Array, or array-like object.")}(i,o)}function assertSize(i){if("number"!=typeof i)throw new TypeError('"size" argument must be a number');if(i<0)throw new RangeError('"size" argument must not be negative')}function allocUnsafe(i,o){if(assertSize(o),i=createBuffer(i,o<0?0:0|checked(o)),!Buffer.TYPED_ARRAY_SUPPORT)for(var u=0;u<o;++u)i[u]=0;return i}function fromArrayLike(i,o){var u=o.length<0?0:0|checked(o.length);i=createBuffer(i,u);for(var p=0;p<u;p+=1)i[p]=255&o[p];return i}function checked(i){if(i>=kMaxLength())throw new RangeError("Attempt to allocate Buffer larger than maximum size: 0x"+kMaxLength().toString(16)+" bytes");return 0|i}function byteLength(i,o){if(Buffer.isBuffer(i))return i.length;if("undefined"!=typeof ArrayBuffer&&"function"==typeof ArrayBuffer.isView&&(ArrayBuffer.isView(i)||i instanceof ArrayBuffer))return i.byteLength;"string"!=typeof i&&(i=""+i);var u=i.length;if(0===u)return 0;for(var p=!1;;)switch(o){case"ascii":case"latin1":case"binary":return u;case"utf8":case"utf-8":case void 0:return utf8ToBytes(i).length;case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return 2*u;case"hex":return u>>>1;case"base64":return base64ToBytes(i).length;default:if(p)return utf8ToBytes(i).length;o=(""+o).toLowerCase(),p=!0}}function slowToString(i,o,u){var p=!1;if((void 0===o||o<0)&&(o=0),o>this.length)return"";if((void 0===u||u>this.length)&&(u=this.length),u<=0)return"";if((u>>>=0)<=(o>>>=0))return"";for(i||(i="utf8");;)switch(i){case"hex":return hexSlice(this,o,u);case"utf8":case"utf-8":return utf8Slice(this,o,u);case"ascii":return asciiSlice(this,o,u);case"latin1":case"binary":return latin1Slice(this,o,u);case"base64":return base64Slice(this,o,u);case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return utf16leSlice(this,o,u);default:if(p)throw new TypeError("Unknown encoding: "+i);i=(i+"").toLowerCase(),p=!0}}function swap(i,o,u){var p=i[o];i[o]=i[u],i[u]=p}function bidirectionalIndexOf(i,o,u,p,g){if(0===i.length)return-1;if("string"==typeof u?(p=u,u=0):u>2147483647?u=2147483647:u<-2147483648&&(u=-2147483648),u=+u,isNaN(u)&&(u=g?0:i.length-1),u<0&&(u=i.length+u),u>=i.length){if(g)return-1;u=i.length-1}else if(u<0){if(!g)return-1;u=0}if("string"==typeof o&&(o=Buffer.from(o,p)),Buffer.isBuffer(o))return 0===o.length?-1:arrayIndexOf(i,o,u,p,g);if("number"==typeof o)return o&=255,Buffer.TYPED_ARRAY_SUPPORT&&"function"==typeof Uint8Array.prototype.indexOf?g?Uint8Array.prototype.indexOf.call(i,o,u):Uint8Array.prototype.lastIndexOf.call(i,o,u):arrayIndexOf(i,[o],u,p,g);throw new TypeError("val must be string, number or Buffer")}function arrayIndexOf(i,o,u,p,g){var y,w=1,_=i.length,x=o.length;if(void 0!==p&&("ucs2"===(p=String(p).toLowerCase())||"ucs-2"===p||"utf16le"===p||"utf-16le"===p)){if(i.length<2||o.length<2)return-1;w=2,_/=2,x/=2,u/=2}function read(i,o){return 1===w?i[o]:i.readUInt16BE(o*w)}if(g){var k=-1;for(y=u;y<_;y++)if(read(i,y)===read(o,-1===k?0:y-k)){if(-1===k&&(k=y),y-k+1===x)return k*w}else-1!==k&&(y-=y-k),k=-1}else for(u+x>_&&(u=_-x),y=u;y>=0;y--){for(var P=!0,E=0;E<x;E++)if(read(i,y+E)!==read(o,E)){P=!1;break}if(P)return y}return-1}function hexWrite(i,o,u,p){u=Number(u)||0;var g=i.length-u;p?(p=Number(p))>g&&(p=g):p=g;var y=o.length;if(y%2!=0)throw new TypeError("Invalid hex string");p>y/2&&(p=y/2);for(var w=0;w<p;++w){var _=parseInt(o.substr(2*w,2),16);if(isNaN(_))return w;i[u+w]=_}return w}function utf8Write(i,o,u,p){return blitBuffer(utf8ToBytes(o,i.length-u),i,u,p)}function asciiWrite(i,o,u,p){return blitBuffer(function asciiToBytes(i){for(var o=[],u=0;u<i.length;++u)o.push(255&i.charCodeAt(u));return o}(o),i,u,p)}function latin1Write(i,o,u,p){return asciiWrite(i,o,u,p)}function base64Write(i,o,u,p){return blitBuffer(base64ToBytes(o),i,u,p)}function ucs2Write(i,o,u,p){return blitBuffer(function utf16leToBytes(i,o){for(var u,p,g,y=[],w=0;w<i.length&&!((o-=2)<0);++w)u=i.charCodeAt(w),p=u>>8,g=u%256,y.push(g),y.push(p);return y}(o,i.length-u),i,u,p)}function base64Slice(i,o,u){return 0===o&&u===i.length?p.fromByteArray(i):p.fromByteArray(i.slice(o,u))}function utf8Slice(i,o,u){u=Math.min(i.length,u);for(var p=[],g=o;g<u;){var y,w,_,x,k=i[g],P=null,E=k>239?4:k>223?3:k>191?2:1;if(g+E<=u)switch(E){case 1:k<128&&(P=k);break;case 2:128==(192&(y=i[g+1]))&&(x=(31&k)<<6|63&y)>127&&(P=x);break;case 3:y=i[g+1],w=i[g+2],128==(192&y)&&128==(192&w)&&(x=(15&k)<<12|(63&y)<<6|63&w)>2047&&(x<55296||x>57343)&&(P=x);break;case 4:y=i[g+1],w=i[g+2],_=i[g+3],128==(192&y)&&128==(192&w)&&128==(192&_)&&(x=(15&k)<<18|(63&y)<<12|(63&w)<<6|63&_)>65535&&x<1114112&&(P=x)}null===P?(P=65533,E=1):P>65535&&(P-=65536,p.push(P>>>10&1023|55296),P=56320|1023&P),p.push(P),g+=E}return function decodeCodePointsArray(i){var o=i.length;if(o<=4096)return String.fromCharCode.apply(String,i);var u="",p=0;for(;p<o;)u+=String.fromCharCode.apply(String,i.slice(p,p+=4096));return u}(p)}o.Buffer=Buffer,o.SlowBuffer=function SlowBuffer(i){+i!=i&&(i=0);return Buffer.alloc(+i)},o.INSPECT_MAX_BYTES=50,Buffer.TYPED_ARRAY_SUPPORT=void 0!==i.TYPED_ARRAY_SUPPORT?i.TYPED_ARRAY_SUPPORT:function typedArraySupport(){try{var i=new Uint8Array(1);return i.__proto__={__proto__:Uint8Array.prototype,foo:function(){return 42}},42===i.foo()&&"function"==typeof i.subarray&&0===i.subarray(1,1).byteLength}catch(i){return!1}}(),o.kMaxLength=kMaxLength(),Buffer.poolSize=8192,Buffer._augment=function(i){return i.__proto__=Buffer.prototype,i},Buffer.from=function(i,o,u){return from(null,i,o,u)},Buffer.TYPED_ARRAY_SUPPORT&&(Buffer.prototype.__proto__=Uint8Array.prototype,Buffer.__proto__=Uint8Array,"undefined"!=typeof Symbol&&Symbol.species&&Buffer[Symbol.species]===Buffer&&Object.defineProperty(Buffer,Symbol.species,{value:null,configurable:!0})),Buffer.alloc=function(i,o,u){return function alloc(i,o,u,p){return assertSize(o),o<=0?createBuffer(i,o):void 0!==u?"string"==typeof p?createBuffer(i,o).fill(u,p):createBuffer(i,o).fill(u):createBuffer(i,o)}(null,i,o,u)},Buffer.allocUnsafe=function(i){return allocUnsafe(null,i)},Buffer.allocUnsafeSlow=function(i){return allocUnsafe(null,i)},Buffer.isBuffer=function isBuffer(i){return!(null==i||!i._isBuffer)},Buffer.compare=function compare(i,o){if(!Buffer.isBuffer(i)||!Buffer.isBuffer(o))throw new TypeError("Arguments must be Buffers");if(i===o)return 0;for(var u=i.length,p=o.length,g=0,y=Math.min(u,p);g<y;++g)if(i[g]!==o[g]){u=i[g],p=o[g];break}return u<p?-1:p<u?1:0},Buffer.isEncoding=function isEncoding(i){switch(String(i).toLowerCase()){case"hex":case"utf8":case"utf-8":case"ascii":case"latin1":case"binary":case"base64":case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return!0;default:return!1}},Buffer.concat=function concat(i,o){if(!y(i))throw new TypeError('"list" argument must be an Array of Buffers');if(0===i.length)return Buffer.alloc(0);var u;if(void 0===o)for(o=0,u=0;u<i.length;++u)o+=i[u].length;var p=Buffer.allocUnsafe(o),g=0;for(u=0;u<i.length;++u){var w=i[u];if(!Buffer.isBuffer(w))throw new TypeError('"list" argument must be an Array of Buffers');w.copy(p,g),g+=w.length}return p},Buffer.byteLength=byteLength,Buffer.prototype._isBuffer=!0,Buffer.prototype.swap16=function swap16(){var i=this.length;if(i%2!=0)throw new RangeError("Buffer size must be a multiple of 16-bits");for(var o=0;o<i;o+=2)swap(this,o,o+1);return this},Buffer.prototype.swap32=function swap32(){var i=this.length;if(i%4!=0)throw new RangeError("Buffer size must be a multiple of 32-bits");for(var o=0;o<i;o+=4)swap(this,o,o+3),swap(this,o+1,o+2);return this},Buffer.prototype.swap64=function swap64(){var i=this.length;if(i%8!=0)throw new RangeError("Buffer size must be a multiple of 64-bits");for(var o=0;o<i;o+=8)swap(this,o,o+7),swap(this,o+1,o+6),swap(this,o+2,o+5),swap(this,o+3,o+4);return this},Buffer.prototype.toString=function toString(){var i=0|this.length;return 0===i?"":0===arguments.length?utf8Slice(this,0,i):slowToString.apply(this,arguments)},Buffer.prototype.equals=function equals(i){if(!Buffer.isBuffer(i))throw new TypeError("Argument must be a Buffer");return this===i||0===Buffer.compare(this,i)},Buffer.prototype.inspect=function inspect(){var i="",u=o.INSPECT_MAX_BYTES;return this.length>0&&(i=this.toString("hex",0,u).match(/.{2}/g).join(" "),this.length>u&&(i+=" ... ")),"<Buffer "+i+">"},Buffer.prototype.compare=function compare(i,o,u,p,g){if(!Buffer.isBuffer(i))throw new TypeError("Argument must be a Buffer");if(void 0===o&&(o=0),void 0===u&&(u=i?i.length:0),void 0===p&&(p=0),void 0===g&&(g=this.length),o<0||u>i.length||p<0||g>this.length)throw new RangeError("out of range index");if(p>=g&&o>=u)return 0;if(p>=g)return-1;if(o>=u)return 1;if(this===i)return 0;for(var y=(g>>>=0)-(p>>>=0),w=(u>>>=0)-(o>>>=0),_=Math.min(y,w),x=this.slice(p,g),k=i.slice(o,u),P=0;P<_;++P)if(x[P]!==k[P]){y=x[P],w=k[P];break}return y<w?-1:w<y?1:0},Buffer.prototype.includes=function includes(i,o,u){return-1!==this.indexOf(i,o,u)},Buffer.prototype.indexOf=function indexOf(i,o,u){return bidirectionalIndexOf(this,i,o,u,!0)},Buffer.prototype.lastIndexOf=function lastIndexOf(i,o,u){return bidirectionalIndexOf(this,i,o,u,!1)},Buffer.prototype.write=function write(i,o,u,p){if(void 0===o)p="utf8",u=this.length,o=0;else if(void 0===u&&"string"==typeof o)p=o,u=this.length,o=0;else{if(!isFinite(o))throw new Error("Buffer.write(string, encoding, offset[, length]) is no longer supported");o|=0,isFinite(u)?(u|=0,void 0===p&&(p="utf8")):(p=u,u=void 0)}var g=this.length-o;if((void 0===u||u>g)&&(u=g),i.length>0&&(u<0||o<0)||o>this.length)throw new RangeError("Attempt to write outside buffer bounds");p||(p="utf8");for(var y=!1;;)switch(p){case"hex":return hexWrite(this,i,o,u);case"utf8":case"utf-8":return utf8Write(this,i,o,u);case"ascii":return asciiWrite(this,i,o,u);case"latin1":case"binary":return latin1Write(this,i,o,u);case"base64":return base64Write(this,i,o,u);case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return ucs2Write(this,i,o,u);default:if(y)throw new TypeError("Unknown encoding: "+p);p=(""+p).toLowerCase(),y=!0}},Buffer.prototype.toJSON=function toJSON(){return{type:"Buffer",data:Array.prototype.slice.call(this._arr||this,0)}};function asciiSlice(i,o,u){var p="";u=Math.min(i.length,u);for(var g=o;g<u;++g)p+=String.fromCharCode(127&i[g]);return p}function latin1Slice(i,o,u){var p="";u=Math.min(i.length,u);for(var g=o;g<u;++g)p+=String.fromCharCode(i[g]);return p}function hexSlice(i,o,u){var p=i.length;(!o||o<0)&&(o=0),(!u||u<0||u>p)&&(u=p);for(var g="",y=o;y<u;++y)g+=toHex(i[y]);return g}function utf16leSlice(i,o,u){for(var p=i.slice(o,u),g="",y=0;y<p.length;y+=2)g+=String.fromCharCode(p[y]+256*p[y+1]);return g}function checkOffset(i,o,u){if(i%1!=0||i<0)throw new RangeError("offset is not uint");if(i+o>u)throw new RangeError("Trying to access beyond buffer length")}function checkInt(i,o,u,p,g,y){if(!Buffer.isBuffer(i))throw new TypeError('"buffer" argument must be a Buffer instance');if(o>g||o<y)throw new RangeError('"value" argument is out of bounds');if(u+p>i.length)throw new RangeError("Index out of range")}function objectWriteUInt16(i,o,u,p){o<0&&(o=65535+o+1);for(var g=0,y=Math.min(i.length-u,2);g<y;++g)i[u+g]=(o&255<<8*(p?g:1-g))>>>8*(p?g:1-g)}function objectWriteUInt32(i,o,u,p){o<0&&(o=4294967295+o+1);for(var g=0,y=Math.min(i.length-u,4);g<y;++g)i[u+g]=o>>>8*(p?g:3-g)&255}function checkIEEE754(i,o,u,p,g,y){if(u+p>i.length)throw new RangeError("Index out of range");if(u<0)throw new RangeError("Index out of range")}function writeFloat(i,o,u,p,y){return y||checkIEEE754(i,0,u,4),g.write(i,o,u,p,23,4),u+4}function writeDouble(i,o,u,p,y){return y||checkIEEE754(i,0,u,8),g.write(i,o,u,p,52,8),u+8}Buffer.prototype.slice=function slice(i,o){var u,p=this.length;if((i=~~i)<0?(i+=p)<0&&(i=0):i>p&&(i=p),(o=void 0===o?p:~~o)<0?(o+=p)<0&&(o=0):o>p&&(o=p),o<i&&(o=i),Buffer.TYPED_ARRAY_SUPPORT)(u=this.subarray(i,o)).__proto__=Buffer.prototype;else{var g=o-i;u=new Buffer(g,void 0);for(var y=0;y<g;++y)u[y]=this[y+i]}return u},Buffer.prototype.readUIntLE=function readUIntLE(i,o,u){i|=0,o|=0,u||checkOffset(i,o,this.length);for(var p=this[i],g=1,y=0;++y<o&&(g*=256);)p+=this[i+y]*g;return p},Buffer.prototype.readUIntBE=function readUIntBE(i,o,u){i|=0,o|=0,u||checkOffset(i,o,this.length);for(var p=this[i+--o],g=1;o>0&&(g*=256);)p+=this[i+--o]*g;return p},Buffer.prototype.readUInt8=function readUInt8(i,o){return o||checkOffset(i,1,this.length),this[i]},Buffer.prototype.readUInt16LE=function readUInt16LE(i,o){return o||checkOffset(i,2,this.length),this[i]|this[i+1]<<8},Buffer.prototype.readUInt16BE=function readUInt16BE(i,o){return o||checkOffset(i,2,this.length),this[i]<<8|this[i+1]},Buffer.prototype.readUInt32LE=function readUInt32LE(i,o){return o||checkOffset(i,4,this.length),(this[i]|this[i+1]<<8|this[i+2]<<16)+16777216*this[i+3]},Buffer.prototype.readUInt32BE=function readUInt32BE(i,o){return o||checkOffset(i,4,this.length),16777216*this[i]+(this[i+1]<<16|this[i+2]<<8|this[i+3])},Buffer.prototype.readIntLE=function readIntLE(i,o,u){i|=0,o|=0,u||checkOffset(i,o,this.length);for(var p=this[i],g=1,y=0;++y<o&&(g*=256);)p+=this[i+y]*g;return p>=(g*=128)&&(p-=Math.pow(2,8*o)),p},Buffer.prototype.readIntBE=function readIntBE(i,o,u){i|=0,o|=0,u||checkOffset(i,o,this.length);for(var p=o,g=1,y=this[i+--p];p>0&&(g*=256);)y+=this[i+--p]*g;return y>=(g*=128)&&(y-=Math.pow(2,8*o)),y},Buffer.prototype.readInt8=function readInt8(i,o){return o||checkOffset(i,1,this.length),128&this[i]?-1*(255-this[i]+1):this[i]},Buffer.prototype.readInt16LE=function readInt16LE(i,o){o||checkOffset(i,2,this.length);var u=this[i]|this[i+1]<<8;return 32768&u?4294901760|u:u},Buffer.prototype.readInt16BE=function readInt16BE(i,o){o||checkOffset(i,2,this.length);var u=this[i+1]|this[i]<<8;return 32768&u?4294901760|u:u},Buffer.prototype.readInt32LE=function readInt32LE(i,o){return o||checkOffset(i,4,this.length),this[i]|this[i+1]<<8|this[i+2]<<16|this[i+3]<<24},Buffer.prototype.readInt32BE=function readInt32BE(i,o){return o||checkOffset(i,4,this.length),this[i]<<24|this[i+1]<<16|this[i+2]<<8|this[i+3]},Buffer.prototype.readFloatLE=function readFloatLE(i,o){return o||checkOffset(i,4,this.length),g.read(this,i,!0,23,4)},Buffer.prototype.readFloatBE=function readFloatBE(i,o){return o||checkOffset(i,4,this.length),g.read(this,i,!1,23,4)},Buffer.prototype.readDoubleLE=function readDoubleLE(i,o){return o||checkOffset(i,8,this.length),g.read(this,i,!0,52,8)},Buffer.prototype.readDoubleBE=function readDoubleBE(i,o){return o||checkOffset(i,8,this.length),g.read(this,i,!1,52,8)},Buffer.prototype.writeUIntLE=function writeUIntLE(i,o,u,p){(i=+i,o|=0,u|=0,p)||checkInt(this,i,o,u,Math.pow(2,8*u)-1,0);var g=1,y=0;for(this[o]=255&i;++y<u&&(g*=256);)this[o+y]=i/g&255;return o+u},Buffer.prototype.writeUIntBE=function writeUIntBE(i,o,u,p){(i=+i,o|=0,u|=0,p)||checkInt(this,i,o,u,Math.pow(2,8*u)-1,0);var g=u-1,y=1;for(this[o+g]=255&i;--g>=0&&(y*=256);)this[o+g]=i/y&255;return o+u},Buffer.prototype.writeUInt8=function writeUInt8(i,o,u){return i=+i,o|=0,u||checkInt(this,i,o,1,255,0),Buffer.TYPED_ARRAY_SUPPORT||(i=Math.floor(i)),this[o]=255&i,o+1},Buffer.prototype.writeUInt16LE=function writeUInt16LE(i,o,u){return i=+i,o|=0,u||checkInt(this,i,o,2,65535,0),Buffer.TYPED_ARRAY_SUPPORT?(this[o]=255&i,this[o+1]=i>>>8):objectWriteUInt16(this,i,o,!0),o+2},Buffer.prototype.writeUInt16BE=function writeUInt16BE(i,o,u){return i=+i,o|=0,u||checkInt(this,i,o,2,65535,0),Buffer.TYPED_ARRAY_SUPPORT?(this[o]=i>>>8,this[o+1]=255&i):objectWriteUInt16(this,i,o,!1),o+2},Buffer.prototype.writeUInt32LE=function writeUInt32LE(i,o,u){return i=+i,o|=0,u||checkInt(this,i,o,4,4294967295,0),Buffer.TYPED_ARRAY_SUPPORT?(this[o+3]=i>>>24,this[o+2]=i>>>16,this[o+1]=i>>>8,this[o]=255&i):objectWriteUInt32(this,i,o,!0),o+4},Buffer.prototype.writeUInt32BE=function writeUInt32BE(i,o,u){return i=+i,o|=0,u||checkInt(this,i,o,4,4294967295,0),Buffer.TYPED_ARRAY_SUPPORT?(this[o]=i>>>24,this[o+1]=i>>>16,this[o+2]=i>>>8,this[o+3]=255&i):objectWriteUInt32(this,i,o,!1),o+4},Buffer.prototype.writeIntLE=function writeIntLE(i,o,u,p){if(i=+i,o|=0,!p){var g=Math.pow(2,8*u-1);checkInt(this,i,o,u,g-1,-g)}var y=0,w=1,_=0;for(this[o]=255&i;++y<u&&(w*=256);)i<0&&0===_&&0!==this[o+y-1]&&(_=1),this[o+y]=(i/w>>0)-_&255;return o+u},Buffer.prototype.writeIntBE=function writeIntBE(i,o,u,p){if(i=+i,o|=0,!p){var g=Math.pow(2,8*u-1);checkInt(this,i,o,u,g-1,-g)}var y=u-1,w=1,_=0;for(this[o+y]=255&i;--y>=0&&(w*=256);)i<0&&0===_&&0!==this[o+y+1]&&(_=1),this[o+y]=(i/w>>0)-_&255;return o+u},Buffer.prototype.writeInt8=function writeInt8(i,o,u){return i=+i,o|=0,u||checkInt(this,i,o,1,127,-128),Buffer.TYPED_ARRAY_SUPPORT||(i=Math.floor(i)),i<0&&(i=255+i+1),this[o]=255&i,o+1},Buffer.prototype.writeInt16LE=function writeInt16LE(i,o,u){return i=+i,o|=0,u||checkInt(this,i,o,2,32767,-32768),Buffer.TYPED_ARRAY_SUPPORT?(this[o]=255&i,this[o+1]=i>>>8):objectWriteUInt16(this,i,o,!0),o+2},Buffer.prototype.writeInt16BE=function writeInt16BE(i,o,u){return i=+i,o|=0,u||checkInt(this,i,o,2,32767,-32768),Buffer.TYPED_ARRAY_SUPPORT?(this[o]=i>>>8,this[o+1]=255&i):objectWriteUInt16(this,i,o,!1),o+2},Buffer.prototype.writeInt32LE=function writeInt32LE(i,o,u){return i=+i,o|=0,u||checkInt(this,i,o,4,2147483647,-2147483648),Buffer.TYPED_ARRAY_SUPPORT?(this[o]=255&i,this[o+1]=i>>>8,this[o+2]=i>>>16,this[o+3]=i>>>24):objectWriteUInt32(this,i,o,!0),o+4},Buffer.prototype.writeInt32BE=function writeInt32BE(i,o,u){return i=+i,o|=0,u||checkInt(this,i,o,4,2147483647,-2147483648),i<0&&(i=4294967295+i+1),Buffer.TYPED_ARRAY_SUPPORT?(this[o]=i>>>24,this[o+1]=i>>>16,this[o+2]=i>>>8,this[o+3]=255&i):objectWriteUInt32(this,i,o,!1),o+4},Buffer.prototype.writeFloatLE=function writeFloatLE(i,o,u){return writeFloat(this,i,o,!0,u)},Buffer.prototype.writeFloatBE=function writeFloatBE(i,o,u){return writeFloat(this,i,o,!1,u)},Buffer.prototype.writeDoubleLE=function writeDoubleLE(i,o,u){return writeDouble(this,i,o,!0,u)},Buffer.prototype.writeDoubleBE=function writeDoubleBE(i,o,u){return writeDouble(this,i,o,!1,u)},Buffer.prototype.copy=function copy(i,o,u,p){if(u||(u=0),p||0===p||(p=this.length),o>=i.length&&(o=i.length),o||(o=0),p>0&&p<u&&(p=u),p===u)return 0;if(0===i.length||0===this.length)return 0;if(o<0)throw new RangeError("targetStart out of bounds");if(u<0||u>=this.length)throw new RangeError("sourceStart out of bounds");if(p<0)throw new RangeError("sourceEnd out of bounds");p>this.length&&(p=this.length),i.length-o<p-u&&(p=i.length-o+u);var g,y=p-u;if(this===i&&u<o&&o<p)for(g=y-1;g>=0;--g)i[g+o]=this[g+u];else if(y<1e3||!Buffer.TYPED_ARRAY_SUPPORT)for(g=0;g<y;++g)i[g+o]=this[g+u];else Uint8Array.prototype.set.call(i,this.subarray(u,u+y),o);return y},Buffer.prototype.fill=function fill(i,o,u,p){if("string"==typeof i){if("string"==typeof o?(p=o,o=0,u=this.length):"string"==typeof u&&(p=u,u=this.length),1===i.length){var g=i.charCodeAt(0);g<256&&(i=g)}if(void 0!==p&&"string"!=typeof p)throw new TypeError("encoding must be a string");if("string"==typeof p&&!Buffer.isEncoding(p))throw new TypeError("Unknown encoding: "+p)}else"number"==typeof i&&(i&=255);if(o<0||this.length<o||this.length<u)throw new RangeError("Out of range index");if(u<=o)return this;var y;if(o>>>=0,u=void 0===u?this.length:u>>>0,i||(i=0),"number"==typeof i)for(y=o;y<u;++y)this[y]=i;else{var w=Buffer.isBuffer(i)?i:utf8ToBytes(new Buffer(i,p).toString()),_=w.length;for(y=0;y<u-o;++y)this[y+o]=w[y%_]}return this};var w=/[^+\/0-9A-Za-z-_]/g;function toHex(i){return i<16?"0"+i.toString(16):i.toString(16)}function utf8ToBytes(i,o){var u;o=o||1/0;for(var p=i.length,g=null,y=[],w=0;w<p;++w){if((u=i.charCodeAt(w))>55295&&u<57344){if(!g){if(u>56319){(o-=3)>-1&&y.push(239,191,189);continue}if(w+1===p){(o-=3)>-1&&y.push(239,191,189);continue}g=u;continue}if(u<56320){(o-=3)>-1&&y.push(239,191,189),g=u;continue}u=65536+(g-55296<<10|u-56320)}else g&&(o-=3)>-1&&y.push(239,191,189);if(g=null,u<128){if((o-=1)<0)break;y.push(u)}else if(u<2048){if((o-=2)<0)break;y.push(u>>6|192,63&u|128)}else if(u<65536){if((o-=3)<0)break;y.push(u>>12|224,u>>6&63|128,63&u|128)}else{if(!(u<1114112))throw new Error("Invalid code point");if((o-=4)<0)break;y.push(u>>18|240,u>>12&63|128,u>>6&63|128,63&u|128)}}return y}function base64ToBytes(i){return p.toByteArray(function base64clean(i){if((i=function stringtrim(i){return i.trim?i.trim():i.replace(/^\s+|\s+$/g,"")}(i).replace(w,"")).length<2)return"";for(;i.length%4!=0;)i+="=";return i}(i))}function blitBuffer(i,o,u,p){for(var g=0;g<p&&!(g+u>=o.length||g>=i.length);++g)o[g+u]=i[g];return g}}).call(this,u(27))},function(i,o){i.exports=function(i){return"object"==typeof i?null!==i:"function"==typeof i}},function(i,o,u){var p=u(11);i.exports=function(i){if(!p(i))throw TypeError(String(i)+" is not an object");return i}},function(i,o,u){var p=u(3);i.exports=!p((function(){return 7!=Object.defineProperty({},1,{get:function(){return 7}})[1]}))},function(i,o,u){var p,g,y,w,_,x,k,P,E,O,I,B,D,R,N,U,W,G,j;i.exports=(p=u(2),u(50),void(p.lib.Cipher||(g=p,y=g.lib,w=y.Base,_=y.WordArray,x=y.BufferedBlockAlgorithm,k=g.enc,k.Utf8,P=k.Base64,E=g.algo.EvpKDF,O=y.Cipher=x.extend({cfg:w.extend(),createEncryptor:function(i,o){return this.create(this._ENC_XFORM_MODE,i,o)},createDecryptor:function(i,o){return this.create(this._DEC_XFORM_MODE,i,o)},init:function(i,o,u){this.cfg=this.cfg.extend(u),this._xformMode=i,this._key=o,this.reset()},reset:function(){x.reset.call(this),this._doReset()},process:function(i){return this._append(i),this._process()},finalize:function(i){return i&&this._append(i),this._doFinalize()},keySize:4,ivSize:4,_ENC_XFORM_MODE:1,_DEC_XFORM_MODE:2,_createHelper:function(){function selectCipherStrategy(i){return"string"==typeof i?j:W}return function(i){return{encrypt:function(o,u,p){return selectCipherStrategy(u).encrypt(i,o,u,p)},decrypt:function(o,u,p){return selectCipherStrategy(u).decrypt(i,o,u,p)}}}}()}),y.StreamCipher=O.extend({_doFinalize:function(){return this._process(!0)},blockSize:1}),I=g.mode={},B=y.BlockCipherMode=w.extend({createEncryptor:function(i,o){return this.Encryptor.create(i,o)},createDecryptor:function(i,o){return this.Decryptor.create(i,o)},init:function(i,o){this._cipher=i,this._iv=o}}),D=I.CBC=function(){var i=B.extend();function xorBlock(i,o,u){var p=this._iv;if(p){var g=p;this._iv=void 0}else g=this._prevBlock;for(var y=0;y<u;y++)i[o+y]^=g[y]}return i.Encryptor=i.extend({processBlock:function(i,o){var u=this._cipher,p=u.blockSize;xorBlock.call(this,i,o,p),u.encryptBlock(i,o),this._prevBlock=i.slice(o,o+p)}}),i.Decryptor=i.extend({processBlock:function(i,o){var u=this._cipher,p=u.blockSize,g=i.slice(o,o+p);u.decryptBlock(i,o),xorBlock.call(this,i,o,p),this._prevBlock=g}}),i}(),R=(g.pad={}).Pkcs7={pad:function(i,o){for(var u=4*o,p=u-i.sigBytes%u,g=p<<24|p<<16|p<<8|p,y=[],w=0;w<p;w+=4)y.push(g);var x=_.create(y,p);i.concat(x)},unpad:function(i){var o=255&i.words[i.sigBytes-1>>>2];i.sigBytes-=o}},y.BlockCipher=O.extend({cfg:O.cfg.extend({mode:D,padding:R}),reset:function(){O.reset.call(this);var i=this.cfg,o=i.iv,u=i.mode;if(this._xformMode==this._ENC_XFORM_MODE)var p=u.createEncryptor;else p=u.createDecryptor,this._minBufferSize=1;this._mode&&this._mode.__creator==p?this._mode.init(this,o&&o.words):(this._mode=p.call(u,this,o&&o.words),this._mode.__creator=p)},_doProcessBlock:function(i,o){this._mode.processBlock(i,o)},_doFinalize:function(){var i=this.cfg.padding;if(this._xformMode==this._ENC_XFORM_MODE){i.pad(this._data,this.blockSize);var o=this._process(!0)}else o=this._process(!0),i.unpad(o);return o},blockSize:4}),N=y.CipherParams=w.extend({init:function(i){this.mixIn(i)},toString:function(i){return(i||this.formatter).stringify(this)}}),U=(g.format={}).OpenSSL={stringify:function(i){var o=i.ciphertext,u=i.salt;if(u)var p=_.create([1398893684,1701076831]).concat(u).concat(o);else p=o;return p.toString(P)},parse:function(i){var o=P.parse(i),u=o.words;if(1398893684==u[0]&&1701076831==u[1]){var p=_.create(u.slice(2,4));u.splice(0,4),o.sigBytes-=16}return N.create({ciphertext:o,salt:p})}},W=y.SerializableCipher=w.extend({cfg:w.extend({format:U}),encrypt:function(i,o,u,p){p=this.cfg.extend(p);var g=i.createEncryptor(u,p),y=g.finalize(o),w=g.cfg;return N.create({ciphertext:y,key:u,iv:w.iv,algorithm:i,mode:w.mode,padding:w.padding,blockSize:i.blockSize,formatter:p.format})},decrypt:function(i,o,u,p){return p=this.cfg.extend(p),o=this._parse(o,p.format),i.createDecryptor(u,p).finalize(o.ciphertext)},_parse:function(i,o){return"string"==typeof i?o.parse(i,this):i}}),G=(g.kdf={}).OpenSSL={execute:function(i,o,u,p){p||(p=_.random(8));var g=E.create({keySize:o+u}).compute(i,p),y=_.create(g.words.slice(o),4*u);return g.sigBytes=4*o,N.create({key:g,iv:y,salt:p})}},j=y.PasswordBasedCipher=W.extend({cfg:W.cfg.extend({kdf:G}),encrypt:function(i,o,u,p){var g=(p=this.cfg.extend(p)).kdf.execute(u,i.keySize,i.ivSize);p.iv=g.iv;var y=W.encrypt.call(this,i,o,g.key,p);return y.mixIn(g),y},decrypt:function(i,o,u,p){p=this.cfg.extend(p),o=this._parse(o,p.format);var g=p.kdf.execute(u,i.keySize,i.ivSize,o.salt);return p.iv=g.iv,W.decrypt.call(this,i,o,g.key,p)}}))))},function(i,o){var u={}.hasOwnProperty;i.exports=function(i,o){return u.call(i,o)}},function(i,o,u){var p=u(13),g=u(203),y=u(12),w=u(55),_=Object.defineProperty;o.f=p?_:function defineProperty(i,o,u){if(y(i),o=w(o,!0),y(u),g)try{return _(i,o,u)}catch(i){}if("get"in u||"set"in u)throw TypeError("Accessors not supported");return"value"in u&&(i[o]=u.value),i}},function(i,o){var u=i.exports="undefined"!=typeof window&&window.Math==Math?window:"undefined"!=typeof self&&self.Math==Math?self:Function("return this")();"number"==typeof __g&&(__g=u)},function(i,o,u){var p=u(13),g=u(16),y=u(41);i.exports=p?function(i,o,u){return g.f(i,o,y(1,u))}:function(i,o,u){return i[o]=u,i}},function(i,o,u){var p=u(35);i.exports=function(i){return Object(p(i))}},function(i,o,u){var p=u(185)("wks"),g=u(133),y=u(17).Symbol,w="function"==typeof y;(i.exports=function(i){return p[i]||(p[i]=w&&y[i]||(w?y:g)("Symbol."+i))}).store=p},function(i,o,u){var p=u(90),g=u(35);i.exports=function(i){return p(g(i))}},function(i,o,u){var p=u(93),g=u(90),y=u(19),w=u(8),_=u(212),x=[].push,createMethod=function(i){var o=1==i,u=2==i,k=3==i,P=4==i,E=6==i,O=7==i,I=5==i||E;return function(B,D,R,N){for(var U,W,G=y(B),j=g(G),X=p(D,R,3),K=w(j.length),Y=0,J=N||_,$=o?J(B,K):u||O?J(B,0):void 0;K>Y;Y++)if((I||Y in j)&&(W=X(U=j[Y],Y,G),i))if(o)$[Y]=W;else if(W)switch(i){case 3:return!0;case 5:return U;case 6:return Y;case 2:x.call($,U)}else switch(i){case 4:return!1;case 7:x.call($,U)}return E?-1:k||P?P:$}};i.exports={forEach:createMethod(0),map:createMethod(1),filter:createMethod(2),some:createMethod(3),every:createMethod(4),find:createMethod(5),findIndex:createMethod(6),filterOut:createMethod(7)}},function(i,o){i.exports=function(i){return"object"==typeof i?null!==i:"function"==typeof i}},function(i,o,u){i.exports=!u(53)((function(){return 7!=Object.defineProperty({},"a",{get:function(){return 7}}).a}))},function(i,o,u){var p=u(4),g=u(18),y=u(15),w=u(141),_=u(142),x=u(43),k=x.get,P=x.enforce,E=String(String).split("String");(i.exports=function(i,o,u,_){var x,k=!!_&&!!_.unsafe,O=!!_&&!!_.enumerable,I=!!_&&!!_.noTargetGet;"function"==typeof u&&("string"!=typeof o||y(u,"name")||g(u,"name",o),(x=P(u)).source||(x.source=E.join("string"==typeof o?o:""))),i!==p?(k?!I&&i[o]&&(O=!0):delete i[o],O?i[o]=u:g(i,o,u)):O?i[o]=u:w(o,u)})(Function.prototype,"toString",(function toString(){return"function"==typeof this&&k(this).source||_(this)}))},function(i,o,u){var p=u(29),g=u(254),y=u(179),w=Object.defineProperty;o.f=u(24)?Object.defineProperty:function defineProperty(i,o,u){if(p(i),o=y(o,!0),p(u),g)try{return w(i,o,u)}catch(i){}if("get"in u||"set"in u)throw TypeError("Accessors not supported!");return"value"in u&&(i[o]=u.value),i}},function(i,o){var u;u=function(){return this}();try{u=u||new Function("return this")()}catch(i){"object"==typeof window&&(u=window)}i.exports=u},function(i,o,u){"use strict";var p=u(21),g=u(153),y=u(94),w=u(43),_=u(217),x=w.set,k=w.getterFor("Array Iterator");i.exports=_(Array,"Array",(function(i,o){x(this,{type:"Array Iterator",target:p(i),index:0,kind:o})}),(function(){var i=k(this),o=i.target,u=i.kind,p=i.index++;return!o||p>=o.length?(i.target=void 0,{value:void 0,done:!0}):"keys"==u?{value:p,done:!1}:"values"==u?{value:o[p],done:!1}:{value:[p,o[p]],done:!1}}),"values"),y.Arguments=y.Array,g("keys"),g("values"),g("entries")},function(i,o,u){var p=u(23);i.exports=function(i){if(!p(i))throw TypeError(i+" is not an object!");return i}},function(i,o,u){var p=u(112),g=u(4),aFunction=function(i){return"function"==typeof i?i:void 0};i.exports=function(i,o){return arguments.length<2?aFunction(p[i])||aFunction(g[i]):p[i]&&p[i][o]||g[i]&&g[i][o]}},function(i,o){i.exports=function(i){if("function"!=typeof i)throw TypeError(String(i)+" is not a function");return i}},function(i,o,u){var p=u(150),g=u(25),y=u(323);p||g(Object.prototype,"toString",y,{unsafe:!0})},function(i,o,u){var p=u(98);i.exports=function(i,o,u){if(p(i),void 0===o)return i;switch(u){case 1:return function(u){return i.call(o,u)};case 2:return function(u,p){return i.call(o,u,p)};case 3:return function(u,p,g){return i.call(o,u,p,g)}}return function(){return i.apply(o,arguments)}}},function(i,o,u){var p=u(13),g=u(107),y=u(41),w=u(21),_=u(55),x=u(15),k=u(203),P=Object.getOwnPropertyDescriptor;o.f=p?P:function getOwnPropertyDescriptor(i,o){if(i=w(i),o=_(o,!0),k)try{return P(i,o)}catch(i){}if(x(i,o))return y(!g.f.call(i,o),i[o])}},function(i,o){i.exports=function(i){if(null==i)throw TypeError("Can't call method on "+i);return i}},function(i,o){var u=Math.ceil,p=Math.floor;i.exports=function(i){return isNaN(i=+i)?0:(i>0?p:u)(i)}},function(i,o,u){var p=u(12),g=u(31),y=u(6)("species");i.exports=function(i,o){var u,w=p(i).constructor;return void 0===w||null==(u=p(w)[y])?o:g(u)}},function(i,o,u){"use strict";(function(o){var p,g=u(10),y=g.Buffer,w={};for(p in g)g.hasOwnProperty(p)&&"SlowBuffer"!==p&&"Buffer"!==p&&(w[p]=g[p]);var _=w.Buffer={};for(p in y)y.hasOwnProperty(p)&&"allocUnsafe"!==p&&"allocUnsafeSlow"!==p&&(_[p]=y[p]);if(w.Buffer.prototype=y.prototype,_.from&&_.from!==Uint8Array.from||(_.from=function(i,o,u){if("number"==typeof i)throw new TypeError('The "value" argument must not be of type number. Received type '+typeof i);if(i&&void 0===i.length)throw new TypeError("The first argument must be one of type string, Buffer, ArrayBuffer, Array, or Array-like Object. Received type "+typeof i);return y(i,o,u)}),_.alloc||(_.alloc=function(i,o,u){if("number"!=typeof i)throw new TypeError('The "size" argument must be of type number. Received type '+typeof i);if(i<0||i>=2*(1<<30))throw new RangeError('The value "'+i+'" is invalid for option "size"');var p=y(i);return o&&0!==o.length?"string"==typeof u?p.fill(o,u):p.fill(o):p.fill(0),p}),!w.kStringMaxLength)try{w.kStringMaxLength=o.binding("buffer").kStringMaxLength}catch(i){}w.constants||(w.constants={MAX_LENGTH:w.kMaxLength},w.kStringMaxLength&&(w.constants.MAX_STRING_LENGTH=w.kStringMaxLength)),i.exports=w}).call(this,u(48))},function(i,o,u){(function(){var i,p;i=u(87).Number,o.resolveLength=function(o,u,p){var g;if("number"==typeof o?g=o:"function"==typeof o?g=o.call(p,p):p&&"string"==typeof o?g=p[o]:u&&o instanceof i&&(g=o.decode(u)),isNaN(g))throw new Error("Not a fixed size");return g},p=function p(i){var o,u;for(o in null==i&&(i={}),this.enumerable=!0,this.configurable=!0,i)u=i[o],this[o]=u},o.PropertyDescriptor=p}).call(this)},function(i,o,u){var p=u(26),g=u(97);i.exports=u(24)?function(i,o,u){return p.f(i,o,g(1,u))}:function(i,o,u){return i[o]=u,i}},function(i,o){i.exports=function(i,o){return{enumerable:!(1&i),configurable:!(2&i),writable:!(4&i),value:o}}},function(i,o){var u={}.toString;i.exports=function(i){return u.call(i).slice(8,-1)}},function(i,o,u){var p,g,y,w=u(310),_=u(4),x=u(11),k=u(18),P=u(15),E=u(143),O=u(108),I=u(111),B=_.WeakMap;if(w){var D=E.state||(E.state=new B),R=D.get,N=D.has,U=D.set;p=function(i,o){return o.facade=i,U.call(D,i,o),o},g=function(i){return R.call(D,i)||{}},y=function(i){return N.call(D,i)}}else{var W=O("state");I[W]=!0,p=function(i,o){return o.facade=i,k(i,W,o),o},g=function(i){return P(i,W)?i[W]:{}},y=function(i){return P(i,W)}}i.exports={set:p,get:g,has:y,enforce:function(i){return y(i)?g(i):p(i,{})},getterFor:function(i){return function(o){var u;if(!x(o)||(u=g(o)).type!==i)throw TypeError("Incompatible receiver, "+i+" required");return u}}}},function(i,o,u){var p=u(36),g=Math.max,y=Math.min;i.exports=function(i,o){var u=p(i);return u<0?g(u+o,0):y(u,o)}},function(i,o,u){var p,g=u(12),y=u(311),w=u(144),_=u(111),x=u(209),k=u(140),P=u(108),E=P("IE_PROTO"),EmptyConstructor=function(){},scriptTag=function(i){return"<script>"+i+"<\/script>"},NullProtoObject=function(){try{p=document.domain&&new ActiveXObject("htmlfile")}catch(i){}var i,o;NullProtoObject=p?function(i){i.write(scriptTag("")),i.close();var o=i.parentWindow.Object;return i=null,o}(p):((o=k("iframe")).style.display="none",x.appendChild(o),o.src=String("javascript:"),(i=o.contentWindow.document).open(),i.write(scriptTag("document.F=Object")),i.close(),i.F);for(var u=w.length;u--;)delete NullProtoObject.prototype[w[u]];return NullProtoObject()};_[E]=!0,i.exports=Object.create||function create(i,o){var u;return null!==i?(EmptyConstructor.prototype=g(i),u=new EmptyConstructor,EmptyConstructor.prototype=null,u[E]=i):u=NullProtoObject(),void 0===o?u:y(u,o)}},function(i,o,u){var p=u(12),g=u(319);i.exports=Object.setPrototypeOf||("__proto__"in{}?function(){var i,o=!1,u={};try{(i=Object.getOwnPropertyDescriptor(Object.prototype,"__proto__").set).call(u,[]),o=u instanceof Array}catch(i){}return function setPrototypeOf(u,y){return p(u),g(y),o?i.call(u,y):u.__proto__=y,u}}():void 0)},function(i,o,u){"use strict";var p=u(1),g=u(11),y=u(116),w=u(44),_=u(8),x=u(21),k=u(118),P=u(6),E=u(117)("slice"),O=P("species"),I=[].slice,B=Math.max;p({target:"Array",proto:!0,forced:!E},{slice:function slice(i,o){var u,p,P,E=x(this),D=_(E.length),R=w(i,D),N=w(void 0===o?D:o,D);if(y(E)&&("function"!=typeof(u=E.constructor)||u!==Array&&!y(u.prototype)?g(u)&&null===(u=u[O])&&(u=void 0):u=void 0,u===Array||void 0===u))return I.call(E,R,N);for(p=new(void 0===u?Array:u)(B(N-R,0)),P=0;R<N;R++,P++)R in E&&k(p,P,E[R]);return p.length=P,p}})},function(i,o){var u,p,g=i.exports={};function defaultSetTimout(){throw new Error("setTimeout has not been defined")}function defaultClearTimeout(){throw new Error("clearTimeout has not been defined")}function runTimeout(i){if(u===setTimeout)return setTimeout(i,0);if((u===defaultSetTimout||!u)&&setTimeout)return u=setTimeout,setTimeout(i,0);try{return u(i,0)}catch(o){try{return u.call(null,i,0)}catch(o){return u.call(this,i,0)}}}!function(){try{u="function"==typeof setTimeout?setTimeout:defaultSetTimout}catch(i){u=defaultSetTimout}try{p="function"==typeof clearTimeout?clearTimeout:defaultClearTimeout}catch(i){p=defaultClearTimeout}}();var y,w=[],_=!1,x=-1;function cleanUpNextTick(){_&&y&&(_=!1,y.length?w=y.concat(w):x=-1,w.length&&drainQueue())}function drainQueue(){if(!_){var i=runTimeout(cleanUpNextTick);_=!0;for(var o=w.length;o;){for(y=w,w=[];++x<o;)y&&y[x].run();x=-1,o=w.length}y=null,_=!1,function runClearTimeout(i){if(p===clearTimeout)return clearTimeout(i);if((p===defaultClearTimeout||!p)&&clearTimeout)return p=clearTimeout,clearTimeout(i);try{return p(i)}catch(o){try{return p.call(null,i)}catch(o){return p.call(this,i)}}}(i)}}function Item(i,o){this.fun=i,this.array=o}function noop(){}g.nextTick=function(i){var o=new Array(arguments.length-1);if(arguments.length>1)for(var u=1;u<arguments.length;u++)o[u-1]=arguments[u];w.push(new Item(i,o)),1!==w.length||_||runTimeout(drainQueue)},Item.prototype.run=function(){this.fun.apply(null,this.array)},g.title="browser",g.browser=!0,g.env={},g.argv=[],g.version="",g.versions={},g.on=noop,g.addListener=noop,g.once=noop,g.off=noop,g.removeListener=noop,g.removeAllListeners=noop,g.emit=noop,g.prependListener=noop,g.prependOnceListener=noop,g.listeners=function(i){return[]},g.binding=function(i){throw new Error("process.binding is not supported")},g.cwd=function(){return"/"},g.chdir=function(i){throw new Error("process.chdir is not supported")},g.umask=function(){return 0}},function(i,o,u){"use strict";var p=u(125),g=Object.keys||function(i){var o=[];for(var u in i)o.push(u);return o};i.exports=Duplex;var y=Object.create(u(95));y.inherits=u(84);var w=u(240),_=u(169);y.inherits(Duplex,w);for(var x=g(_.prototype),k=0;k<x.length;k++){var P=x[k];Duplex.prototype[P]||(Duplex.prototype[P]=_.prototype[P])}function Duplex(i){if(!(this instanceof Duplex))return new Duplex(i);w.call(this,i),_.call(this,i),i&&!1===i.readable&&(this.readable=!1),i&&!1===i.writable&&(this.writable=!1),this.allowHalfOpen=!0,i&&!1===i.allowHalfOpen&&(this.allowHalfOpen=!1),this.once("end",onend)}function onend(){this.allowHalfOpen||this._writableState.ended||p.nextTick(onEndNT,this)}function onEndNT(i){i.end()}Object.defineProperty(Duplex.prototype,"writableHighWaterMark",{enumerable:!1,get:function(){return this._writableState.highWaterMark}}),Object.defineProperty(Duplex.prototype,"destroyed",{get:function(){return void 0!==this._readableState&&void 0!==this._writableState&&(this._readableState.destroyed&&this._writableState.destroyed)},set:function(i){void 0!==this._readableState&&void 0!==this._writableState&&(this._readableState.destroyed=i,this._writableState.destroyed=i)}}),Duplex.prototype._destroy=function(i,o){this.push(null),this.end(),p.nextTick(o,i)}},function(i,o,u){var p,g,y,w,_,x,k,P;i.exports=(P=u(2),u(171),u(172),g=(p=P).lib,y=g.Base,w=g.WordArray,_=p.algo,x=_.MD5,k=_.EvpKDF=y.extend({cfg:y.extend({keySize:4,hasher:x,iterations:1}),init:function(i){this.cfg=this.cfg.extend(i)},compute:function(i,o){for(var u=this.cfg,p=u.hasher.create(),g=w.create(),y=g.words,_=u.keySize,x=u.iterations;y.length<_;){k&&p.update(k);var k=p.update(i).finalize(o);p.reset();for(var P=1;P<x;P++)k=p.finalize(k),p.reset();g.concat(k)}return g.sigBytes=4*_,g}}),p.EvpKDF=function(i,o,u){return k.create(u).compute(i,o)},P.EvpKDF)},function(i,o,u){var p=u(176),g=u(177);i.exports=function(i){return p(g(i))}},function(i,o){var u={}.hasOwnProperty;i.exports=function(i,o){return u.call(i,o)}},function(i,o){i.exports=function(i){try{return!!i()}catch(i){return!0}}},function(i,o,u){"use strict";(function(o,u){function VirtualFileSystem(){this.fileSystem={},this.dataSystem={}}function fixFilename(i){return 0===i.indexOf(u)&&(i=i.substring(u.length)),0===i.indexOf("/")&&(i=i.substring(1)),i}VirtualFileSystem.prototype.existsSync=function(i){return i=fixFilename(i),void 0!==this.fileSystem[i]||void 0!==this.dataSystem[i]},VirtualFileSystem.prototype.readFileSync=function(i,u){i=fixFilename(i);var p=this.dataSystem[i];if("string"==typeof p&&"utf8"===u)return p;if(p)return new o(p,"string"==typeof p?"base64":void 0);var g=this.fileSystem[i];if(g)return g;throw"File '"+i+"' not found in virtual file system"},VirtualFileSystem.prototype.writeFileSync=function(i,o){this.fileSystem[fixFilename(i)]=o},VirtualFileSystem.prototype.bindFS=function(i){this.dataSystem=i||{}},i.exports=new VirtualFileSystem}).call(this,u(10).Buffer,"/")},function(i,o,u){var p=u(11);i.exports=function(i,o){if(!p(i))return i;var u,g;if(o&&"function"==typeof(u=i.toString)&&!p(g=u.call(i)))return g;if("function"==typeof(u=i.valueOf)&&!p(g=u.call(i)))return g;if(!o&&"function"==typeof(u=i.toString)&&!p(g=u.call(i)))return g;throw TypeError("Can't convert object to primitive value")}},function(i,o){i.exports=!1},function(i,o,u){var p=u(206),g=u(144).concat("length","prototype");o.f=Object.getOwnPropertyNames||function getOwnPropertyNames(i){return p(i,g)}},function(i,o,u){var p=u(15),g=u(19),y=u(108),w=u(214),_=y("IE_PROTO"),x=Object.prototype;i.exports=w?Object.getPrototypeOf:function(i){return i=g(i),p(i,_)?i[_]:"function"==typeof i.constructor&&i instanceof i.constructor?i.constructor.prototype:i instanceof Object?x:null}},function(i,o,u){u(231)("Uint8",(function(i){return function Uint8Array(o,u,p){return i(this,o,u,p)}}))},function(i,o,u){"use strict";var p=u(5),g=u(341),y=p.aTypedArray;(0,p.exportTypedArrayMethod)("copyWithin",(function copyWithin(i,o){return g.call(y(this),i,o,arguments.length>2?arguments[2]:void 0)}))},function(i,o,u){"use strict";var p=u(5),g=u(22).every,y=p.aTypedArray;(0,p.exportTypedArrayMethod)("every",(function every(i){return g(y(this),i,arguments.length>1?arguments[1]:void 0)}))},function(i,o,u){"use strict";var p=u(5),g=u(166),y=p.aTypedArray;(0,p.exportTypedArrayMethod)("fill",(function fill(i){return g.apply(y(this),arguments)}))},function(i,o,u){"use strict";var p=u(5),g=u(22).filter,y=u(342),w=p.aTypedArray;(0,p.exportTypedArrayMethod)("filter",(function filter(i){var o=g(w(this),i,arguments.length>1?arguments[1]:void 0);return y(this,o)}))},function(i,o,u){"use strict";var p=u(5),g=u(22).find,y=p.aTypedArray;(0,p.exportTypedArrayMethod)("find",(function find(i){return g(y(this),i,arguments.length>1?arguments[1]:void 0)}))},function(i,o,u){"use strict";var p=u(5),g=u(22).findIndex,y=p.aTypedArray;(0,p.exportTypedArrayMethod)("findIndex",(function findIndex(i){return g(y(this),i,arguments.length>1?arguments[1]:void 0)}))},function(i,o,u){"use strict";var p=u(5),g=u(22).forEach,y=p.aTypedArray;(0,p.exportTypedArrayMethod)("forEach",(function forEach(i){g(y(this),i,arguments.length>1?arguments[1]:void 0)}))},function(i,o,u){"use strict";var p=u(5),g=u(113).includes,y=p.aTypedArray;(0,p.exportTypedArrayMethod)("includes",(function includes(i){return g(y(this),i,arguments.length>1?arguments[1]:void 0)}))},function(i,o,u){"use strict";var p=u(5),g=u(113).indexOf,y=p.aTypedArray;(0,p.exportTypedArrayMethod)("indexOf",(function indexOf(i){return g(y(this),i,arguments.length>1?arguments[1]:void 0)}))},function(i,o,u){"use strict";var p=u(4),g=u(5),y=u(28),w=u(6)("iterator"),_=p.Uint8Array,x=y.values,k=y.keys,P=y.entries,E=g.aTypedArray,O=g.exportTypedArrayMethod,I=_&&_.prototype[w],B=!!I&&("values"==I.name||null==I.name),D=function values(){return x.call(E(this))};O("entries",(function entries(){return P.call(E(this))})),O("keys",(function keys(){return k.call(E(this))})),O("values",D,!B),O(w,D,!B)},function(i,o,u){"use strict";var p=u(5),g=p.aTypedArray,y=p.exportTypedArrayMethod,w=[].join;y("join",(function join(i){return w.apply(g(this),arguments)}))},function(i,o,u){"use strict";var p=u(5),g=u(343),y=p.aTypedArray;(0,p.exportTypedArrayMethod)("lastIndexOf",(function lastIndexOf(i){return g.apply(y(this),arguments)}))},function(i,o,u){"use strict";var p=u(5),g=u(22).map,y=u(37),w=p.aTypedArray,_=p.aTypedArrayConstructor;(0,p.exportTypedArrayMethod)("map",(function map(i){return g(w(this),i,arguments.length>1?arguments[1]:void 0,(function(i,o){return new(_(y(i,i.constructor)))(o)}))}))},function(i,o,u){"use strict";var p=u(5),g=u(236).left,y=p.aTypedArray;(0,p.exportTypedArrayMethod)("reduce",(function reduce(i){return g(y(this),i,arguments.length,arguments.length>1?arguments[1]:void 0)}))},function(i,o,u){"use strict";var p=u(5),g=u(236).right,y=p.aTypedArray;(0,p.exportTypedArrayMethod)("reduceRight",(function reduceRight(i){return g(y(this),i,arguments.length,arguments.length>1?arguments[1]:void 0)}))},function(i,o,u){"use strict";var p=u(5),g=p.aTypedArray,y=p.exportTypedArrayMethod,w=Math.floor;y("reverse",(function reverse(){for(var i,o=g(this).length,u=w(o/2),p=0;p<u;)i=this[p],this[p++]=this[--o],this[o]=i;return this}))},function(i,o,u){"use strict";var p=u(5),g=u(8),y=u(235),w=u(19),_=u(3),x=p.aTypedArray;(0,p.exportTypedArrayMethod)("set",(function set(i){x(this);var o=y(arguments.length>1?arguments[1]:void 0,1),u=this.length,p=w(i),_=g(p.length),k=0;if(_+o>u)throw RangeError("Wrong length");for(;k<_;)this[o+k]=p[k++]}),_((function(){new Int8Array(1).set({})})))},function(i,o,u){"use strict";var p=u(5),g=u(37),y=u(3),w=p.aTypedArray,_=p.aTypedArrayConstructor,x=p.exportTypedArrayMethod,k=[].slice;x("slice",(function slice(i,o){for(var u=k.call(w(this),i,o),p=g(this,this.constructor),y=0,x=u.length,P=new(_(p))(x);x>y;)P[y]=u[y++];return P}),y((function(){new Int8Array(1).slice()})))},function(i,o,u){"use strict";var p=u(5),g=u(22).some,y=p.aTypedArray;(0,p.exportTypedArrayMethod)("some",(function some(i){return g(y(this),i,arguments.length>1?arguments[1]:void 0)}))},function(i,o,u){"use strict";var p=u(5),g=p.aTypedArray,y=p.exportTypedArrayMethod,w=[].sort;y("sort",(function sort(i){return w.call(g(this),i)}))},function(i,o,u){"use strict";var p=u(5),g=u(8),y=u(44),w=u(37),_=p.aTypedArray;(0,p.exportTypedArrayMethod)("subarray",(function subarray(i,o){var u=_(this),p=u.length,x=y(i,p);return new(w(u,u.constructor))(u.buffer,u.byteOffset+x*u.BYTES_PER_ELEMENT,g((void 0===o?p:y(o,p))-x))}))},function(i,o,u){"use strict";var p=u(4),g=u(5),y=u(3),w=p.Int8Array,_=g.aTypedArray,x=g.exportTypedArrayMethod,k=[].toLocaleString,P=[].slice,E=!!w&&y((function(){k.call(new w(1))}));x("toLocaleString",(function toLocaleString(){return k.apply(E?P.call(_(this)):_(this),arguments)}),y((function(){return[1,2].toLocaleString()!=new w([1,2]).toLocaleString()}))||!y((function(){w.prototype.toLocaleString.call([1,2])})))},function(i,o,u){"use strict";var p=u(5).exportTypedArrayMethod,g=u(3),y=u(4).Uint8Array,w=y&&y.prototype||{},_=[].toString,x=[].join;g((function(){_.call({})}))&&(_=function toString(){return x.call(this)});var k=w.toString!=_;p("toString",_,k)},function(i,o,u){"use strict";var p=u(1),g=u(3),y=u(164),w=u(12),_=u(44),x=u(8),k=u(37),P=y.ArrayBuffer,E=y.DataView,O=P.prototype.slice;p({target:"ArrayBuffer",proto:!0,unsafe:!0,forced:g((function(){return!new P(2).slice(1,void 0).byteLength}))},{slice:function slice(i,o){if(void 0!==O&&void 0===o)return O.call(w(this),i);for(var u=w(this).byteLength,p=_(i,u),g=_(void 0===o?u:o,u),y=new(k(this,P))(x(g-p)),I=new E(this),B=new E(y),D=0;p<g;)B.setUint8(D++,I.getUint8(p++));return y}})},function(i,o){"function"==typeof Object.create?i.exports=function inherits(i,o){o&&(i.super_=o,i.prototype=Object.create(o.prototype,{constructor:{value:i,enumerable:!1,writable:!0,configurable:!0}}))}:i.exports=function inherits(i,o){if(o){i.super_=o;var TempCtor=function(){};TempCtor.prototype=o.prototype,i.prototype=new TempCtor,i.prototype.constructor=i}}},function(i,o,u){var p,g,y;i.exports=(y=u(2),g=(p=y).lib.WordArray,p.enc.Base64={stringify:function(i){var o=i.words,u=i.sigBytes,p=this._map;i.clamp();for(var g=[],y=0;y<u;y+=3)for(var w=(o[y>>>2]>>>24-y%4*8&255)<<16|(o[y+1>>>2]>>>24-(y+1)%4*8&255)<<8|o[y+2>>>2]>>>24-(y+2)%4*8&255,_=0;_<4&&y+.75*_<u;_++)g.push(p.charAt(w>>>6*(3-_)&63));var x=p.charAt(64);if(x)for(;g.length%4;)g.push(x);return g.join("")},parse:function(i){var o=i.length,u=this._map,p=this._reverseMap;if(!p){p=this._reverseMap=[];for(var y=0;y<u.length;y++)p[u.charCodeAt(y)]=y}var w=u.charAt(64);if(w){var _=i.indexOf(w);-1!==_&&(o=_)}return function parseLoop(i,o,u){for(var p=[],y=0,w=0;w<o;w++)if(w%4){var _=u[i.charCodeAt(w-1)]<<w%4*2,x=u[i.charCodeAt(w)]>>>6-w%4*2;p[y>>>2]|=(_|x)<<24-y%4*8,y++}return g.create(p,y)}(i,o,p)},_map:"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/="},y.enc.Base64)},function(i,o,u){var p;i.exports=(p=u(2),function(i){var o=p,u=o.lib,g=u.WordArray,y=u.Hasher,w=o.algo,_=[];!function(){for(var o=0;o<64;o++)_[o]=4294967296*i.abs(i.sin(o+1))|0}();var x=w.MD5=y.extend({_doReset:function(){this._hash=new g.init([1732584193,4023233417,2562383102,271733878])},_doProcessBlock:function(i,o){for(var u=0;u<16;u++){var p=o+u,g=i[p];i[p]=16711935&(g<<8|g>>>24)|4278255360&(g<<24|g>>>8)}var y=this._hash.words,w=i[o+0],x=i[o+1],k=i[o+2],P=i[o+3],E=i[o+4],O=i[o+5],I=i[o+6],B=i[o+7],D=i[o+8],R=i[o+9],N=i[o+10],U=i[o+11],W=i[o+12],G=i[o+13],j=i[o+14],X=i[o+15],K=y[0],Y=y[1],J=y[2],$=y[3];K=FF(K,Y,J,$,w,7,_[0]),$=FF($,K,Y,J,x,12,_[1]),J=FF(J,$,K,Y,k,17,_[2]),Y=FF(Y,J,$,K,P,22,_[3]),K=FF(K,Y,J,$,E,7,_[4]),$=FF($,K,Y,J,O,12,_[5]),J=FF(J,$,K,Y,I,17,_[6]),Y=FF(Y,J,$,K,B,22,_[7]),K=FF(K,Y,J,$,D,7,_[8]),$=FF($,K,Y,J,R,12,_[9]),J=FF(J,$,K,Y,N,17,_[10]),Y=FF(Y,J,$,K,U,22,_[11]),K=FF(K,Y,J,$,W,7,_[12]),$=FF($,K,Y,J,G,12,_[13]),J=FF(J,$,K,Y,j,17,_[14]),K=GG(K,Y=FF(Y,J,$,K,X,22,_[15]),J,$,x,5,_[16]),$=GG($,K,Y,J,I,9,_[17]),J=GG(J,$,K,Y,U,14,_[18]),Y=GG(Y,J,$,K,w,20,_[19]),K=GG(K,Y,J,$,O,5,_[20]),$=GG($,K,Y,J,N,9,_[21]),J=GG(J,$,K,Y,X,14,_[22]),Y=GG(Y,J,$,K,E,20,_[23]),K=GG(K,Y,J,$,R,5,_[24]),$=GG($,K,Y,J,j,9,_[25]),J=GG(J,$,K,Y,P,14,_[26]),Y=GG(Y,J,$,K,D,20,_[27]),K=GG(K,Y,J,$,G,5,_[28]),$=GG($,K,Y,J,k,9,_[29]),J=GG(J,$,K,Y,B,14,_[30]),K=HH(K,Y=GG(Y,J,$,K,W,20,_[31]),J,$,O,4,_[32]),$=HH($,K,Y,J,D,11,_[33]),J=HH(J,$,K,Y,U,16,_[34]),Y=HH(Y,J,$,K,j,23,_[35]),K=HH(K,Y,J,$,x,4,_[36]),$=HH($,K,Y,J,E,11,_[37]),J=HH(J,$,K,Y,B,16,_[38]),Y=HH(Y,J,$,K,N,23,_[39]),K=HH(K,Y,J,$,G,4,_[40]),$=HH($,K,Y,J,w,11,_[41]),J=HH(J,$,K,Y,P,16,_[42]),Y=HH(Y,J,$,K,I,23,_[43]),K=HH(K,Y,J,$,R,4,_[44]),$=HH($,K,Y,J,W,11,_[45]),J=HH(J,$,K,Y,X,16,_[46]),K=II(K,Y=HH(Y,J,$,K,k,23,_[47]),J,$,w,6,_[48]),$=II($,K,Y,J,B,10,_[49]),J=II(J,$,K,Y,j,15,_[50]),Y=II(Y,J,$,K,O,21,_[51]),K=II(K,Y,J,$,W,6,_[52]),$=II($,K,Y,J,P,10,_[53]),J=II(J,$,K,Y,N,15,_[54]),Y=II(Y,J,$,K,x,21,_[55]),K=II(K,Y,J,$,D,6,_[56]),$=II($,K,Y,J,X,10,_[57]),J=II(J,$,K,Y,I,15,_[58]),Y=II(Y,J,$,K,G,21,_[59]),K=II(K,Y,J,$,E,6,_[60]),$=II($,K,Y,J,U,10,_[61]),J=II(J,$,K,Y,k,15,_[62]),Y=II(Y,J,$,K,R,21,_[63]),y[0]=y[0]+K|0,y[1]=y[1]+Y|0,y[2]=y[2]+J|0,y[3]=y[3]+$|0},_doFinalize:function(){var o=this._data,u=o.words,p=8*this._nDataBytes,g=8*o.sigBytes;u[g>>>5]|=128<<24-g%32;var y=i.floor(p/4294967296),w=p;u[15+(g+64>>>9<<4)]=16711935&(y<<8|y>>>24)|4278255360&(y<<24|y>>>8),u[14+(g+64>>>9<<4)]=16711935&(w<<8|w>>>24)|4278255360&(w<<24|w>>>8),o.sigBytes=4*(u.length+1),this._process();for(var _=this._hash,x=_.words,k=0;k<4;k++){var P=x[k];x[k]=16711935&(P<<8|P>>>24)|4278255360&(P<<24|P>>>8)}return _},clone:function(){var i=y.clone.call(this);return i._hash=this._hash.clone(),i}});function FF(i,o,u,p,g,y,w){var _=i+(o&u|~o&p)+g+w;return(_<<y|_>>>32-y)+o}function GG(i,o,u,p,g,y,w){var _=i+(o&p|u&~p)+g+w;return(_<<y|_>>>32-y)+o}function HH(i,o,u,p,g,y,w){var _=i+(o^u^p)+g+w;return(_<<y|_>>>32-y)+o}function II(i,o,u,p,g,y,w){var _=i+(u^(o|~p))+g+w;return(_<<y|_>>>32-y)+o}o.MD5=y._createHelper(x),o.HmacMD5=y._createHmacHelper(x)}(Math),p.MD5)},function(i,o,u){(function(){var i,p,g,y={}.hasOwnProperty;i=u(173),g=function(){function NumberT(i,o){this.type=i,this.endian=null!=o?o:"BE",this.fn=this.type,"8"!==this.type[this.type.length-1]&&(this.fn+=this.endian)}return NumberT.prototype.size=function(){return i.TYPES[this.type]},NumberT.prototype.decode=function(i){return i["read"+this.fn]()},NumberT.prototype.encode=function(i,o){return i["write"+this.fn](o)},NumberT}(),o.Number=g,o.uint8=new g("UInt8"),o.uint16be=o.uint16=new g("UInt16","BE"),o.uint16le=new g("UInt16","LE"),o.uint24be=o.uint24=new g("UInt24","BE"),o.uint24le=new g("UInt24","LE"),o.uint32be=o.uint32=new g("UInt32","BE"),o.uint32le=new g("UInt32","LE"),o.int8=new g("Int8"),o.int16be=o.int16=new g("Int16","BE"),o.int16le=new g("Int16","LE"),o.int24be=o.int24=new g("Int24","BE"),o.int24le=new g("Int24","LE"),o.int32be=o.int32=new g("Int32","BE"),o.int32le=new g("Int32","LE"),o.floatbe=o.float=new g("Float","BE"),o.floatle=new g("Float","LE"),o.doublebe=o.double=new g("Double","BE"),o.doublele=new g("Double","LE"),p=function(i){function Fixed(i,o,u){null==u&&(u=i>>1),Fixed.__super__.constructor.call(this,"Int"+i,o),this._point=1<<u}return function(i,o){for(var u in o)y.call(o,u)&&(i[u]=o[u]);function ctor(){this.constructor=i}ctor.prototype=o.prototype,i.prototype=new ctor,i.__super__=o.prototype}(Fixed,i),Fixed.prototype.decode=function(i){return Fixed.__super__.decode.call(this,i)/this._point},Fixed.prototype.encode=function(i,o){return Fixed.__super__.encode.call(this,i,o*this._point|0)},Fixed}(g),o.Fixed=p,o.fixed16be=o.fixed16=new p(16,"BE"),o.fixed16le=new p(16,"LE"),o.fixed32be=o.fixed32=new p(32,"BE"),o.fixed32le=new p(32,"LE")}).call(this)},function(i,o,u){var p=u(177);i.exports=function(i){return Object(p(i))}},function(i,o,u){"use strict";var p=u(435)(!0);u(182)(String,"String",(function(i){this._t=String(i),this._i=0}),(function(){var i,o=this._t,u=this._i;return u>=o.length?{value:void 0,done:!0}:(i=p(o,u),this._i+=i.length,{value:i,done:!1})}))},function(i,o,u){var p=u(3),g=u(42),y="".split;i.exports=p((function(){return!Object("z").propertyIsEnumerable(0)}))?function(i){return"String"==g(i)?y.call(i,""):Object(i)}:Object},function(i,o,u){var p=u(206),g=u(144);i.exports=Object.keys||function keys(i){return p(i,g)}},function(i,o,u){var p=u(16).f,g=u(15),y=u(6)("toStringTag");i.exports=function(i,o,u){i&&!g(i=u?i:i.prototype,y)&&p(i,y,{configurable:!0,value:o})}},function(i,o,u){var p=u(31);i.exports=function(i,o,u){if(p(i),void 0===o)return i;switch(u){case 0:return function(){return i.call(o)};case 1:return function(u){return i.call(o,u)};case 2:return function(u,p){return i.call(o,u,p)};case 3:return function(u,p,g){return i.call(o,u,p,g)}}return function(){return i.apply(o,arguments)}}},function(i,o){i.exports={}},function(i,o,u){(function(i){function objectToString(i){return Object.prototype.toString.call(i)}o.isArray=function isArray(i){return Array.isArray?Array.isArray(i):"[object Array]"===objectToString(i)},o.isBoolean=function isBoolean(i){return"boolean"==typeof i},o.isNull=function isNull(i){return null===i},o.isNullOrUndefined=function isNullOrUndefined(i){return null==i},o.isNumber=function isNumber(i){return"number"==typeof i},o.isString=function isString(i){return"string"==typeof i},o.isSymbol=function isSymbol(i){return"symbol"==typeof i},o.isUndefined=function isUndefined(i){return void 0===i},o.isRegExp=function isRegExp(i){return"[object RegExp]"===objectToString(i)},o.isObject=function isObject(i){return"object"==typeof i&&null!==i},o.isDate=function isDate(i){return"[object Date]"===objectToString(i)},o.isError=function isError(i){return"[object Error]"===objectToString(i)||i instanceof Error},o.isFunction=function isFunction(i){return"function"==typeof i},o.isPrimitive=function isPrimitive(i){return null===i||"boolean"==typeof i||"number"==typeof i||"string"==typeof i||"symbol"==typeof i||void 0===i},o.isBuffer=i.isBuffer}).call(this,u(10).Buffer)},function(i,o){var u={}.toString;i.exports=function(i){return u.call(i).slice(8,-1)}},function(i,o){i.exports=function(i,o){return{enumerable:!(1&i),configurable:!(2&i),writable:!(4&i),value:o}}},function(i,o){i.exports=function(i){if("function"!=typeof i)throw TypeError(i+" is not a function!");return i}},function(i,o,u){u(430);for(var p=u(17),g=u(40),y=u(100),w=u(20)("toStringTag"),_="CSSRuleList,CSSStyleDeclaration,CSSValueList,ClientRectList,DOMRectList,DOMStringList,DOMTokenList,DataTransferItemList,FileList,HTMLAllCollection,HTMLCollection,HTMLFormElement,HTMLSelectElement,MediaList,MimeTypeArray,NamedNodeMap,NodeList,PaintRequestList,Plugin,PluginArray,SVGLengthList,SVGNumberList,SVGPathSegList,SVGPointList,SVGStringList,SVGTransformList,SourceBufferList,StyleSheetList,TextTrackCueList,TextTrackList,TouchList".split(","),x=0;x<_.length;x++){var k=_[x],P=p[k],E=P&&P.prototype;E&&!E[w]&&g(E,w,k),y[k]=y.Array}},function(i,o){i.exports={}},function(i,o){i.exports=!0},function(i,o,u){var p=u(258),g=u(186);i.exports=Object.keys||function keys(i){return p(i,g)}},function(i,o,u){var p=u(26).f,g=u(52),y=u(20)("toStringTag");i.exports=function(i,o,u){i&&!g(i=u?i:i.prototype,y)&&p(i,y,{configurable:!0,value:o})}},function(i,o,u){var p=u(33),g=u(265),y=u(266),w=u(29),_=u(132),x=u(187),k={},P={};(o=i.exports=function(i,o,u,E,O){var I,B,D,R,N=O?function(){return i}:x(i),U=p(u,E,o?2:1),W=0;if("function"!=typeof N)throw TypeError(i+" is not iterable!");if(y(N)){for(I=_(i.length);I>W;W++)if((R=o?U(w(B=i[W])[0],B[1]):U(i[W]))===k||R===P)return R}else for(D=N.call(i);!(B=D.next()).done;)if((R=g(D,U,B.value,o))===k||R===P)return R}).BREAK=k,o.RETURN=P},function(i,o,u){"use strict";var p=u(272),g="function"==typeof Symbol&&"symbol"==typeof Symbol("foo"),y=Object.prototype.toString,w=Array.prototype.concat,_=Object.defineProperty,x=_&&function(){var i={};try{for(var o in _(i,"x",{enumerable:!1,value:i}),i)return!1;return i.x===i}catch(i){return!1}}(),defineProperty=function(i,o,u,p){var g;(!(o in i)||"function"==typeof(g=p)&&"[object Function]"===y.call(g)&&p())&&(x?_(i,o,{configurable:!0,enumerable:!1,value:u,writable:!0}):i[o]=u)},defineProperties=function(i,o){var u=arguments.length>2?arguments[2]:{},y=p(o);g&&(y=w.call(y,Object.getOwnPropertySymbols(o)));for(var _=0;_<y.length;_+=1)defineProperty(i,y[_],o[y[_]],u[y[_]])};defineProperties.supportsDescriptors=!!x,i.exports=defineProperties},function(i,o,u){"use strict";var p=u(31),PromiseCapability=function(i){var o,u;this.promise=new i((function(i,p){if(void 0!==o||void 0!==u)throw TypeError("Bad Promise constructor");o=i,u=p})),this.resolve=p(o),this.reject=p(u)};i.exports.f=function(i){return new PromiseCapability(i)}},function(i,o,u){"use strict";var p={}.propertyIsEnumerable,g=Object.getOwnPropertyDescriptor,y=g&&!p.call({1:2},1);o.f=y?function propertyIsEnumerable(i){var o=g(this,i);return!!o&&o.enumerable}:p},function(i,o,u){var p=u(109),g=u(110),y=p("keys");i.exports=function(i){return y[i]||(y[i]=g(i))}},function(i,o,u){var p=u(56),g=u(143);(i.exports=function(i,o){return g[i]||(g[i]=void 0!==o?o:{})})("versions",[]).push({version:"3.10.0",mode:p?"pure":"global",copyright:"© 2021 Denis Pushkarev (zloirock.ru)"})},function(i,o){var u=0,p=Math.random();i.exports=function(i){return"Symbol("+String(void 0===i?"":i)+")_"+(++u+p).toString(36)}},function(i,o){i.exports={}},function(i,o,u){var p=u(4);i.exports=p},function(i,o,u){var p=u(21),g=u(8),y=u(44),createMethod=function(i){return function(o,u,w){var _,x=p(o),k=g(x.length),P=y(w,k);if(i&&u!=u){for(;k>P;)if((_=x[P++])!=_)return!0}else for(;k>P;P++)if((i||P in x)&&x[P]===u)return i||P||0;return!i&&-1}};i.exports={includes:createMethod(!0),indexOf:createMethod(!1)}},function(i,o,u){var p=u(42),g=u(4);i.exports="process"==p(g.process)},function(i,o,u){var p,g,y=u(4),w=u(148),_=y.process,x=_&&_.versions,k=x&&x.v8;k?g=(p=k.split("."))[0]+p[1]:w&&(!(p=w.match(/Edge\/(\d+)/))||p[1]>=74)&&(p=w.match(/Chrome\/(\d+)/))&&(g=p[1]),i.exports=g&&+g},function(i,o,u){var p=u(42);i.exports=Array.isArray||function isArray(i){return"Array"==p(i)}},function(i,o,u){var p=u(3),g=u(6),y=u(115),w=g("species");i.exports=function(i){return y>=51||!p((function(){var o=[];return(o.constructor={})[w]=function(){return{foo:1}},1!==o[i](Boolean).foo}))}},function(i,o,u){"use strict";var p=u(55),g=u(16),y=u(41);i.exports=function(i,o,u){var w=p(o);w in i?g.f(i,w,y(0,u)):i[w]=u}},function(i,o,u){var p=u(150),g=u(42),y=u(6)("toStringTag"),w="Arguments"==g(function(){return arguments}());i.exports=p?g:function(i){var o,u,p;return void 0===i?"Undefined":null===i?"Null":"string"==typeof(u=function(i,o){try{return i[o]}catch(i){}}(o=Object(i),y))?u:w?g(o):"Object"==(p=g(o))&&"function"==typeof o.callee?"Arguments":p}},function(i,o,u){var p=u(4),g=u(213),y=u(28),w=u(18),_=u(6),x=_("iterator"),k=_("toStringTag"),P=y.values;for(var E in g){var O=p[E],I=O&&O.prototype;if(I){if(I[x]!==P)try{w(I,x,P)}catch(i){I[x]=P}if(I[k]||w(I,k,E),g[E])for(var B in y)if(I[B]!==y[B])try{w(I,B,y[B])}catch(i){I[B]=y[B]}}}},function(i,o,u){"use strict";var p=u(1),g=u(122);p({target:"RegExp",proto:!0,forced:/./.exec!==g},{exec:g})},function(i,o,u){"use strict";var p,g,y=u(223),w=u(328),_=u(109),x=RegExp.prototype.exec,k=_("native-string-replace",String.prototype.replace),P=x,E=(p=/a/,g=/b*/g,x.call(p,"a"),x.call(g,"a"),0!==p.lastIndex||0!==g.lastIndex),O=w.UNSUPPORTED_Y||w.BROKEN_CARET,I=void 0!==/()??/.exec("")[1];(E||I||O)&&(P=function exec(i){var o,u,p,g,w=this,_=O&&w.sticky,P=y.call(w),B=w.source,D=0,R=i;return _&&(-1===(P=P.replace("y","")).indexOf("g")&&(P+="g"),R=String(i).slice(w.lastIndex),w.lastIndex>0&&(!w.multiline||w.multiline&&"\n"!==i[w.lastIndex-1])&&(B="(?: "+B+")",R=" "+R,D++),u=new RegExp("^(?:"+B+")",P)),I&&(u=new RegExp("^"+B+"$(?!\\s)",P)),E&&(o=w.lastIndex),p=x.call(_?u:w,R),_?p?(p.input=p.input.slice(D),p[0]=p[0].slice(D),p.index=w.lastIndex,w.lastIndex+=p[0].length):w.lastIndex=0:E&&p&&(w.lastIndex=w.global?p.index+p[0].length:o),I&&p&&p.length>1&&k.call(p[0],u,(function(){for(g=1;g<arguments.length-2;g++)void 0===arguments[g]&&(p[g]=void 0)})),p}),i.exports=P},function(i,o,u){i.exports=Stream;var p=u(124).EventEmitter;function Stream(){p.call(this)}u(84)(Stream,p),Stream.Readable=u(168),Stream.Writable=u(354),Stream.Duplex=u(355),Stream.Transform=u(356),Stream.PassThrough=u(357),Stream.Stream=Stream,Stream.prototype.pipe=function(i,o){var u=this;function ondata(o){i.writable&&!1===i.write(o)&&u.pause&&u.pause()}function ondrain(){u.readable&&u.resume&&u.resume()}u.on("data",ondata),i.on("drain",ondrain),i._isStdio||o&&!1===o.end||(u.on("end",onend),u.on("close",onclose));var g=!1;function onend(){g||(g=!0,i.end())}function onclose(){g||(g=!0,"function"==typeof i.destroy&&i.destroy())}function onerror(i){if(cleanup(),0===p.listenerCount(this,"error"))throw i}function cleanup(){u.removeListener("data",ondata),i.removeListener("drain",ondrain),u.removeListener("end",onend),u.removeListener("close",onclose),u.removeListener("error",onerror),i.removeListener("error",onerror),u.removeListener("end",cleanup),u.removeListener("close",cleanup),i.removeListener("close",cleanup)}return u.on("error",onerror),i.on("error",onerror),u.on("end",cleanup),u.on("close",cleanup),i.on("close",cleanup),i.emit("pipe",u),i}},function(i,o,u){"use strict";var p,g="object"==typeof Reflect?Reflect:null,y=g&&"function"==typeof g.apply?g.apply:function ReflectApply(i,o,u){return Function.prototype.apply.call(i,o,u)};p=g&&"function"==typeof g.ownKeys?g.ownKeys:Object.getOwnPropertySymbols?function ReflectOwnKeys(i){return Object.getOwnPropertyNames(i).concat(Object.getOwnPropertySymbols(i))}:function ReflectOwnKeys(i){return Object.getOwnPropertyNames(i)};var w=Number.isNaN||function NumberIsNaN(i){return i!=i};function EventEmitter(){EventEmitter.init.call(this)}i.exports=EventEmitter,i.exports.once=function once(i,o){return new Promise((function(u,p){function errorListener(u){i.removeListener(o,resolver),p(u)}function resolver(){"function"==typeof i.removeListener&&i.removeListener("error",errorListener),u([].slice.call(arguments))}eventTargetAgnosticAddListener(i,o,resolver,{once:!0}),"error"!==o&&function addErrorHandlerIfEventEmitter(i,o,u){"function"==typeof i.on&&eventTargetAgnosticAddListener(i,"error",o,u)}(i,errorListener,{once:!0})}))},EventEmitter.EventEmitter=EventEmitter,EventEmitter.prototype._events=void 0,EventEmitter.prototype._eventsCount=0,EventEmitter.prototype._maxListeners=void 0;var _=10;function checkListener(i){if("function"!=typeof i)throw new TypeError('The "listener" argument must be of type Function. Received type '+typeof i)}function _getMaxListeners(i){return void 0===i._maxListeners?EventEmitter.defaultMaxListeners:i._maxListeners}function _addListener(i,o,u,p){var g,y,w;if(checkListener(u),void 0===(y=i._events)?(y=i._events=Object.create(null),i._eventsCount=0):(void 0!==y.newListener&&(i.emit("newListener",o,u.listener?u.listener:u),y=i._events),w=y[o]),void 0===w)w=y[o]=u,++i._eventsCount;else if("function"==typeof w?w=y[o]=p?[u,w]:[w,u]:p?w.unshift(u):w.push(u),(g=_getMaxListeners(i))>0&&w.length>g&&!w.warned){w.warned=!0;var _=new Error("Possible EventEmitter memory leak detected. "+w.length+" "+String(o)+" listeners added. Use emitter.setMaxListeners() to increase limit");_.name="MaxListenersExceededWarning",_.emitter=i,_.type=o,_.count=w.length,function ProcessEmitWarning(i){console&&console.warn}()}return i}function onceWrapper(){if(!this.fired)return this.target.removeListener(this.type,this.wrapFn),this.fired=!0,0===arguments.length?this.listener.call(this.target):this.listener.apply(this.target,arguments)}function _onceWrap(i,o,u){var p={fired:!1,wrapFn:void 0,target:i,type:o,listener:u},g=onceWrapper.bind(p);return g.listener=u,p.wrapFn=g,g}function _listeners(i,o,u){var p=i._events;if(void 0===p)return[];var g=p[o];return void 0===g?[]:"function"==typeof g?u?[g.listener||g]:[g]:u?function unwrapListeners(i){for(var o=new Array(i.length),u=0;u<o.length;++u)o[u]=i[u].listener||i[u];return o}(g):arrayClone(g,g.length)}function listenerCount(i){var o=this._events;if(void 0!==o){var u=o[i];if("function"==typeof u)return 1;if(void 0!==u)return u.length}return 0}function arrayClone(i,o){for(var u=new Array(o),p=0;p<o;++p)u[p]=i[p];return u}function eventTargetAgnosticAddListener(i,o,u,p){if("function"==typeof i.on)p.once?i.once(o,u):i.on(o,u);else{if("function"!=typeof i.addEventListener)throw new TypeError('The "emitter" argument must be of type EventEmitter. Received type '+typeof i);i.addEventListener(o,(function wrapListener(g){p.once&&i.removeEventListener(o,wrapListener),u(g)}))}}Object.defineProperty(EventEmitter,"defaultMaxListeners",{enumerable:!0,get:function(){return _},set:function(i){if("number"!=typeof i||i<0||w(i))throw new RangeError('The value of "defaultMaxListeners" is out of range. It must be a non-negative number. Received '+i+".");_=i}}),EventEmitter.init=function(){void 0!==this._events&&this._events!==Object.getPrototypeOf(this)._events||(this._events=Object.create(null),this._eventsCount=0),this._maxListeners=this._maxListeners||void 0},EventEmitter.prototype.setMaxListeners=function setMaxListeners(i){if("number"!=typeof i||i<0||w(i))throw new RangeError('The value of "n" is out of range. It must be a non-negative number. Received '+i+".");return this._maxListeners=i,this},EventEmitter.prototype.getMaxListeners=function getMaxListeners(){return _getMaxListeners(this)},EventEmitter.prototype.emit=function emit(i){for(var o=[],u=1;u<arguments.length;u++)o.push(arguments[u]);var p="error"===i,g=this._events;if(void 0!==g)p=p&&void 0===g.error;else if(!p)return!1;if(p){var w;if(o.length>0&&(w=o[0]),w instanceof Error)throw w;var _=new Error("Unhandled error."+(w?" ("+w.message+")":""));throw _.context=w,_}var x=g[i];if(void 0===x)return!1;if("function"==typeof x)y(x,this,o);else{var k=x.length,P=arrayClone(x,k);for(u=0;u<k;++u)y(P[u],this,o)}return!0},EventEmitter.prototype.addListener=function addListener(i,o){return _addListener(this,i,o,!1)},EventEmitter.prototype.on=EventEmitter.prototype.addListener,EventEmitter.prototype.prependListener=function prependListener(i,o){return _addListener(this,i,o,!0)},EventEmitter.prototype.once=function once(i,o){return checkListener(o),this.on(i,_onceWrap(this,i,o)),this},EventEmitter.prototype.prependOnceListener=function prependOnceListener(i,o){return checkListener(o),this.prependListener(i,_onceWrap(this,i,o)),this},EventEmitter.prototype.removeListener=function removeListener(i,o){var u,p,g,y,w;if(checkListener(o),void 0===(p=this._events))return this;if(void 0===(u=p[i]))return this;if(u===o||u.listener===o)0==--this._eventsCount?this._events=Object.create(null):(delete p[i],p.removeListener&&this.emit("removeListener",i,u.listener||o));else if("function"!=typeof u){for(g=-1,y=u.length-1;y>=0;y--)if(u[y]===o||u[y].listener===o){w=u[y].listener,g=y;break}if(g<0)return this;0===g?u.shift():function spliceOne(i,o){for(;o+1<i.length;o++)i[o]=i[o+1];i.pop()}(u,g),1===u.length&&(p[i]=u[0]),void 0!==p.removeListener&&this.emit("removeListener",i,w||o)}return this},EventEmitter.prototype.off=EventEmitter.prototype.removeListener,EventEmitter.prototype.removeAllListeners=function removeAllListeners(i){var o,u,p;if(void 0===(u=this._events))return this;if(void 0===u.removeListener)return 0===arguments.length?(this._events=Object.create(null),this._eventsCount=0):void 0!==u[i]&&(0==--this._eventsCount?this._events=Object.create(null):delete u[i]),this;if(0===arguments.length){var g,y=Object.keys(u);for(p=0;p<y.length;++p)"removeListener"!==(g=y[p])&&this.removeAllListeners(g);return this.removeAllListeners("removeListener"),this._events=Object.create(null),this._eventsCount=0,this}if("function"==typeof(o=u[i]))this.removeListener(i,o);else if(void 0!==o)for(p=o.length-1;p>=0;p--)this.removeListener(i,o[p]);return this},EventEmitter.prototype.listeners=function listeners(i){return _listeners(this,i,!0)},EventEmitter.prototype.rawListeners=function rawListeners(i){return _listeners(this,i,!1)},EventEmitter.listenerCount=function(i,o){return"function"==typeof i.listenerCount?i.listenerCount(o):listenerCount.call(i,o)},EventEmitter.prototype.listenerCount=listenerCount,EventEmitter.prototype.eventNames=function eventNames(){return this._eventsCount>0?p(this._events):[]}},function(i,o,u){"use strict";(function(o){void 0===o||!o.version||0===o.version.indexOf("v0.")||0===o.version.indexOf("v1.")&&0!==o.version.indexOf("v1.8.")?i.exports={nextTick:function nextTick(i,u,p,g){if("function"!=typeof i)throw new TypeError('"callback" argument must be a function');var y,w,_=arguments.length;switch(_){case 0:case 1:return o.nextTick(i);case 2:return o.nextTick((function afterTickOne(){i.call(null,u)}));case 3:return o.nextTick((function afterTickTwo(){i.call(null,u,p)}));case 4:return o.nextTick((function afterTickThree(){i.call(null,u,p,g)}));default:for(y=new Array(_-1),w=0;w<y.length;)y[w++]=arguments[w];return o.nextTick((function afterTick(){i.apply(null,y)}))}}}:i.exports=o}).call(this,u(48))},function(i,o,u){var p=u(10),g=p.Buffer;function copyProps(i,o){for(var u in i)o[u]=i[u]}function SafeBuffer(i,o,u){return g(i,o,u)}g.from&&g.alloc&&g.allocUnsafe&&g.allocUnsafeSlow?i.exports=p:(copyProps(p,o),o.Buffer=SafeBuffer),copyProps(g,SafeBuffer),SafeBuffer.from=function(i,o,u){if("number"==typeof i)throw new TypeError("Argument must not be a number");return g(i,o,u)},SafeBuffer.alloc=function(i,o,u){if("number"!=typeof i)throw new TypeError("Argument must be a number");var p=g(i);return void 0!==o?"string"==typeof u?p.fill(o,u):p.fill(o):p.fill(0),p},SafeBuffer.allocUnsafe=function(i){if("number"!=typeof i)throw new TypeError("Argument must be a number");return g(i)},SafeBuffer.allocUnsafeSlow=function(i){if("number"!=typeof i)throw new TypeError("Argument must be a number");return p.SlowBuffer(i)}},function(i,o,u){"use strict";var p=u(126).Buffer,g=p.isEncoding||function(i){switch((i=""+i)&&i.toLowerCase()){case"hex":case"utf8":case"utf-8":case"ascii":case"binary":case"base64":case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":case"raw":return!0;default:return!1}};function StringDecoder(i){var o;switch(this.encoding=function normalizeEncoding(i){var o=function _normalizeEncoding(i){if(!i)return"utf8";for(var o;;)switch(i){case"utf8":case"utf-8":return"utf8";case"ucs2":case"ucs-2":case"utf16le":case"utf-16le":return"utf16le";case"latin1":case"binary":return"latin1";case"base64":case"ascii":case"hex":return i;default:if(o)return;i=(""+i).toLowerCase(),o=!0}}(i);if("string"!=typeof o&&(p.isEncoding===g||!g(i)))throw new Error("Unknown encoding: "+i);return o||i}(i),this.encoding){case"utf16le":this.text=utf16Text,this.end=utf16End,o=4;break;case"utf8":this.fillLast=utf8FillLast,o=4;break;case"base64":this.text=base64Text,this.end=base64End,o=3;break;default:return this.write=simpleWrite,void(this.end=simpleEnd)}this.lastNeed=0,this.lastTotal=0,this.lastChar=p.allocUnsafe(o)}function utf8CheckByte(i){return i<=127?0:i>>5==6?2:i>>4==14?3:i>>3==30?4:i>>6==2?-1:-2}function utf8FillLast(i){var o=this.lastTotal-this.lastNeed,u=function utf8CheckExtraBytes(i,o,u){if(128!=(192&o[0]))return i.lastNeed=0,"�";if(i.lastNeed>1&&o.length>1){if(128!=(192&o[1]))return i.lastNeed=1,"�";if(i.lastNeed>2&&o.length>2&&128!=(192&o[2]))return i.lastNeed=2,"�"}}(this,i);return void 0!==u?u:this.lastNeed<=i.length?(i.copy(this.lastChar,o,0,this.lastNeed),this.lastChar.toString(this.encoding,0,this.lastTotal)):(i.copy(this.lastChar,o,0,i.length),void(this.lastNeed-=i.length))}function utf16Text(i,o){if((i.length-o)%2==0){var u=i.toString("utf16le",o);if(u){var p=u.charCodeAt(u.length-1);if(p>=55296&&p<=56319)return this.lastNeed=2,this.lastTotal=4,this.lastChar[0]=i[i.length-2],this.lastChar[1]=i[i.length-1],u.slice(0,-1)}return u}return this.lastNeed=1,this.lastTotal=2,this.lastChar[0]=i[i.length-1],i.toString("utf16le",o,i.length-1)}function utf16End(i){var o=i&&i.length?this.write(i):"";if(this.lastNeed){var u=this.lastTotal-this.lastNeed;return o+this.lastChar.toString("utf16le",0,u)}return o}function base64Text(i,o){var u=(i.length-o)%3;return 0===u?i.toString("base64",o):(this.lastNeed=3-u,this.lastTotal=3,1===u?this.lastChar[0]=i[i.length-1]:(this.lastChar[0]=i[i.length-2],this.lastChar[1]=i[i.length-1]),i.toString("base64",o,i.length-u))}function base64End(i){var o=i&&i.length?this.write(i):"";return this.lastNeed?o+this.lastChar.toString("base64",0,3-this.lastNeed):o}function simpleWrite(i){return i.toString(this.encoding)}function simpleEnd(i){return i&&i.length?this.write(i):""}o.StringDecoder=StringDecoder,StringDecoder.prototype.write=function(i){if(0===i.length)return"";var o,u;if(this.lastNeed){if(void 0===(o=this.fillLast(i)))return"";u=this.lastNeed,this.lastNeed=0}else u=0;return u<i.length?o?o+this.text(i,u):this.text(i,u):o||""},StringDecoder.prototype.end=function utf8End(i){var o=i&&i.length?this.write(i):"";return this.lastNeed?o+"�":o},StringDecoder.prototype.text=function utf8Text(i,o){var u=function utf8CheckIncomplete(i,o,u){var p=o.length-1;if(p<u)return 0;var g=utf8CheckByte(o[p]);if(g>=0)return g>0&&(i.lastNeed=g-1),g;if(--p<u||-2===g)return 0;if((g=utf8CheckByte(o[p]))>=0)return g>0&&(i.lastNeed=g-2),g;if(--p<u||-2===g)return 0;if((g=utf8CheckByte(o[p]))>=0)return g>0&&(2===g?g=0:i.lastNeed=g-3),g;return 0}(this,i,o);if(!this.lastNeed)return i.toString("utf8",o);this.lastTotal=u;var p=i.length-(u-this.lastNeed);return i.copy(this.lastChar,0,p),i.toString("utf8",o,p)},StringDecoder.prototype.fillLast=function(i){if(this.lastNeed<=i.length)return i.copy(this.lastChar,this.lastTotal-this.lastNeed,0,this.lastNeed),this.lastChar.toString(this.encoding,0,this.lastTotal);i.copy(this.lastChar,this.lastTotal-this.lastNeed,0,i.length),this.lastNeed-=i.length}},function(i,o,u){"use strict";var p="undefined"!=typeof Uint8Array&&"undefined"!=typeof Uint16Array&&"undefined"!=typeof Int32Array;function _has(i,o){return Object.prototype.hasOwnProperty.call(i,o)}o.assign=function(i){for(var o=Array.prototype.slice.call(arguments,1);o.length;){var u=o.shift();if(u){if("object"!=typeof u)throw new TypeError(u+"must be non-object");for(var p in u)_has(u,p)&&(i[p]=u[p])}}return i},o.shrinkBuf=function(i,o){return i.length===o?i:i.subarray?i.subarray(0,o):(i.length=o,i)};var g={arraySet:function(i,o,u,p,g){if(o.subarray&&i.subarray)i.set(o.subarray(u,u+p),g);else for(var y=0;y<p;y++)i[g+y]=o[u+y]},flattenChunks:function(i){var o,u,p,g,y,w;for(p=0,o=0,u=i.length;o<u;o++)p+=i[o].length;for(w=new Uint8Array(p),g=0,o=0,u=i.length;o<u;o++)y=i[o],w.set(y,g),g+=y.length;return w}},y={arraySet:function(i,o,u,p,g){for(var y=0;y<p;y++)i[g+y]=o[u+y]},flattenChunks:function(i){return[].concat.apply([],i)}};o.setTyped=function(i){i?(o.Buf8=Uint8Array,o.Buf16=Uint16Array,o.Buf32=Int32Array,o.assign(o,g)):(o.Buf8=Array,o.Buf16=Array,o.Buf32=Array,o.assign(o,y))},o.setTyped(p)},function(i,o,u){var p,g,y,w,_,x;i.exports=(x=u(2),g=(p=x).lib,y=g.Base,w=g.WordArray,(_=p.x64={}).Word=y.extend({init:function(i,o){this.high=i,this.low=o}}),_.WordArray=y.extend({init:function(i,o){i=this.words=i||[],this.sigBytes=null!=o?o:8*i.length},toX32:function(){for(var i=this.words,o=i.length,u=[],p=0;p<o;p++){var g=i[p];u.push(g.high),u.push(g.low)}return w.create(u,this.sigBytes)},clone:function(){for(var i=y.clone.call(this),o=i.words=this.words.slice(0),u=o.length,p=0;p<u;p++)o[p]=o[p].clone();return i}}),x)},function(i,o){o.f={}.propertyIsEnumerable},function(i,o,u){var p=u(29),g=u(257),y=u(186),w=u(184)("IE_PROTO"),Empty=function(){},createDict=function(){var i,o=u(180)("iframe"),p=y.length;for(o.style.display="none",u(260).appendChild(o),o.src="javascript:",(i=o.contentWindow.document).open(),i.write("<script>document.F=Object<\/script>"),i.close(),createDict=i.F;p--;)delete createDict.prototype[y[p]];return createDict()};i.exports=Object.create||function create(i,o){var u;return null!==i?(Empty.prototype=p(i),u=new Empty,Empty.prototype=null,u[w]=i):u=createDict(),void 0===o?u:g(u,o)}},function(i,o,u){var p=u(183),g=Math.min;i.exports=function(i){return i>0?g(p(i),9007199254740991):0}},function(i,o){var u=0,p=Math.random();i.exports=function(i){return"Symbol(".concat(void 0===i?"":i,")_",(++u+p).toString(36))}},function(i,o,u){var p=u(133)("meta"),g=u(23),y=u(52),w=u(26).f,_=0,x=Object.isExtensible||function(){return!0},k=!u(53)((function(){return x(Object.preventExtensions({}))})),setMeta=function(i){w(i,p,{value:{i:"O"+ ++_,w:{}}})},P=i.exports={KEY:p,NEED:!1,fastKey:function(i,o){if(!g(i))return"symbol"==typeof i?i:("string"==typeof i?"S":"P")+i;if(!y(i,p)){if(!x(i))return"F";if(!o)return"E";setMeta(i)}return i[p].i},getWeak:function(i,o){if(!y(i,p)){if(!x(i))return!0;if(!o)return!1;setMeta(i)}return i[p].w},onFreeze:function(i){return k&&P.NEED&&x(i)&&!y(i,p)&&setMeta(i),i}}},function(i,o){},function(i,o,u){u(231)("Uint32",(function(i){return function Uint32Array(o,u,p){return i(this,o,u,p)}}))},function(i,o){function Tree(){this.table=new Uint16Array(16),this.trans=new Uint16Array(288)}function Data(i,o){this.source=i,this.sourceIndex=0,this.tag=0,this.bitcount=0,this.dest=o,this.destLen=0,this.ltree=new Tree,this.dtree=new Tree}var u=new Tree,p=new Tree,g=new Uint8Array(30),y=new Uint16Array(30),w=new Uint8Array(30),_=new Uint16Array(30),x=new Uint8Array([16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15]),k=new Tree,P=new Uint8Array(320);function tinf_build_bits_base(i,o,u,p){var g,y;for(g=0;g<u;++g)i[g]=0;for(g=0;g<30-u;++g)i[g+u]=g/u|0;for(y=p,g=0;g<30;++g)o[g]=y,y+=1<<i[g]}var E=new Uint16Array(16);function tinf_build_tree(i,o,u,p){var g,y;for(g=0;g<16;++g)i.table[g]=0;for(g=0;g<p;++g)i.table[o[u+g]]++;for(i.table[0]=0,y=0,g=0;g<16;++g)E[g]=y,y+=i.table[g];for(g=0;g<p;++g)o[u+g]&&(i.trans[E[o[u+g]]++]=g)}function tinf_getbit(i){i.bitcount--||(i.tag=i.source[i.sourceIndex++],i.bitcount=7);var o=1&i.tag;return i.tag>>>=1,o}function tinf_read_bits(i,o,u){if(!o)return u;for(;i.bitcount<24;)i.tag|=i.source[i.sourceIndex++]<<i.bitcount,i.bitcount+=8;var p=i.tag&65535>>>16-o;return i.tag>>>=o,i.bitcount-=o,p+u}function tinf_decode_symbol(i,o){for(;i.bitcount<24;)i.tag|=i.source[i.sourceIndex++]<<i.bitcount,i.bitcount+=8;var u=0,p=0,g=0,y=i.tag;do{p=2*p+(1&y),y>>>=1,++g,u+=o.table[g],p-=o.table[g]}while(p>=0);return i.tag=y,i.bitcount-=g,o.trans[u+p]}function tinf_decode_trees(i,o,u){var p,g,y,w,_,E;for(p=tinf_read_bits(i,5,257),g=tinf_read_bits(i,5,1),y=tinf_read_bits(i,4,4),w=0;w<19;++w)P[w]=0;for(w=0;w<y;++w){var O=tinf_read_bits(i,3,0);P[x[w]]=O}for(tinf_build_tree(k,P,0,19),_=0;_<p+g;){var I=tinf_decode_symbol(i,k);switch(I){case 16:var B=P[_-1];for(E=tinf_read_bits(i,2,3);E;--E)P[_++]=B;break;case 17:for(E=tinf_read_bits(i,3,3);E;--E)P[_++]=0;break;case 18:for(E=tinf_read_bits(i,7,11);E;--E)P[_++]=0;break;default:P[_++]=I}}tinf_build_tree(o,P,0,p),tinf_build_tree(u,P,p,g)}function tinf_inflate_block_data(i,o,u){for(;;){var p,x,k,P,E=tinf_decode_symbol(i,o);if(256===E)return 0;if(E<256)i.dest[i.destLen++]=E;else for(p=tinf_read_bits(i,g[E-=257],y[E]),x=tinf_decode_symbol(i,u),P=k=i.destLen-tinf_read_bits(i,w[x],_[x]);P<k+p;++P)i.dest[i.destLen++]=i.dest[P]}}function tinf_inflate_uncompressed_block(i){for(var o,u;i.bitcount>8;)i.sourceIndex--,i.bitcount-=8;if((o=256*(o=i.source[i.sourceIndex+1])+i.source[i.sourceIndex])!==(65535&~(256*i.source[i.sourceIndex+3]+i.source[i.sourceIndex+2])))return-3;for(i.sourceIndex+=4,u=o;u;--u)i.dest[i.destLen++]=i.source[i.sourceIndex++];return i.bitcount=0,0}!function tinf_build_fixed_trees(i,o){var u;for(u=0;u<7;++u)i.table[u]=0;for(i.table[7]=24,i.table[8]=152,i.table[9]=112,u=0;u<24;++u)i.trans[u]=256+u;for(u=0;u<144;++u)i.trans[24+u]=u;for(u=0;u<8;++u)i.trans[168+u]=280+u;for(u=0;u<112;++u)i.trans[176+u]=144+u;for(u=0;u<5;++u)o.table[u]=0;for(o.table[5]=32,u=0;u<32;++u)o.trans[u]=u}(u,p),tinf_build_bits_base(g,y,4,3),tinf_build_bits_base(w,_,2,1),g[28]=0,y[28]=258,i.exports=function tinf_uncompress(i,o){var g,y,w=new Data(i,o);do{switch(g=tinf_getbit(w),tinf_read_bits(w,2,0)){case 0:y=tinf_inflate_uncompressed_block(w);break;case 1:y=tinf_inflate_block_data(w,u,p);break;case 2:tinf_decode_trees(w,w.ltree,w.dtree),y=tinf_inflate_block_data(w,w.ltree,w.dtree);break;default:y=-3}if(0!==y)throw new Error("Data error")}while(!g);return w.destLen<w.dest.length?"function"==typeof w.dest.slice?w.dest.slice(0,w.destLen):w.dest.subarray(0,w.destLen):w.dest}},function(i,o,u){var p=u(12),g=u(154),y=u(8),w=u(93),_=u(155),x=u(220),Result=function(i,o){this.stopped=i,this.result=o};i.exports=function(i,o,u){var k,P,E,O,I,B,D,R=u&&u.that,N=!(!u||!u.AS_ENTRIES),U=!(!u||!u.IS_ITERATOR),W=!(!u||!u.INTERRUPTED),G=w(o,R,1+N+W),stop=function(i){return k&&x(k),new Result(!0,i)},callFn=function(i){return N?(p(i),W?G(i[0],i[1],stop):G(i[0],i[1])):W?G(i,stop):G(i)};if(U)k=i;else{if("function"!=typeof(P=_(i)))throw TypeError("Target is not iterable");if(g(P)){for(E=0,O=y(i.length);O>E;E++)if((I=callFn(i[E]))&&I instanceof Result)return I;return new Result(!1)}k=P.call(i)}for(B=k.next;!(D=B.call(k)).done;){try{I=callFn(D.value)}catch(i){throw x(k),i}if("object"==typeof I&&I&&I instanceof Result)return I}return new Result(!1)}},function(i,o){i.exports=function(i){try{return{error:!1,value:i()}}catch(i){return{error:!0,value:i}}}},function(i,o,u){var p=u(4),g=u(11),y=p.document,w=g(y)&&g(y.createElement);i.exports=function(i){return w?y.createElement(i):{}}},function(i,o,u){var p=u(4),g=u(18);i.exports=function(i,o){try{g(p,i,o)}catch(u){p[i]=o}return o}},function(i,o,u){var p=u(143),g=Function.toString;"function"!=typeof p.inspectSource&&(p.inspectSource=function(i){return g.call(i)}),i.exports=p.inspectSource},function(i,o,u){var p=u(4),g=u(141),y=p["__core-js_shared__"]||g("__core-js_shared__",{});i.exports=y},function(i,o){i.exports=["constructor","hasOwnProperty","isPrototypeOf","propertyIsEnumerable","toLocaleString","toString","valueOf"]},function(i,o){o.f=Object.getOwnPropertySymbols},function(i,o,u){var p=u(3),g=/#|\.prototype\./,isForced=function(i,o){var u=w[y(i)];return u==x||u!=_&&("function"==typeof o?p(o):!!o)},y=isForced.normalize=function(i){return String(i).replace(g,".").toLowerCase()},w=isForced.data={},_=isForced.NATIVE="N",x=isForced.POLYFILL="P";i.exports=isForced},function(i,o,u){var p=u(114),g=u(115),y=u(3);i.exports=!!Object.getOwnPropertySymbols&&!y((function(){return!Symbol.sham&&(p?38===g:g>37&&g<41)}))},function(i,o,u){var p=u(30);i.exports=p("navigator","userAgent")||""},function(i,o,u){"use strict";var p=u(3);i.exports=function(i,o){var u=[][i];return!!u&&p((function(){u.call(null,o||function(){throw 1},1)}))}},function(i,o,u){var p={};p[u(6)("toStringTag")]="z",i.exports="[object z]"===String(p)},function(i,o,u){"use strict";var p=u(152).charAt,g=u(43),y=u(217),w=g.set,_=g.getterFor("String Iterator");y(String,"String",(function(i){w(this,{type:"String Iterator",string:String(i),index:0})}),(function next(){var i,o=_(this),u=o.string,g=o.index;return g>=u.length?{value:void 0,done:!0}:(i=p(u,g),o.index+=i.length,{value:i,done:!1})}))},function(i,o,u){var p=u(36),g=u(35),createMethod=function(i){return function(o,u){var y,w,_=String(g(o)),x=p(u),k=_.length;return x<0||x>=k?i?"":void 0:(y=_.charCodeAt(x))<55296||y>56319||x+1===k||(w=_.charCodeAt(x+1))<56320||w>57343?i?_.charAt(x):y:i?_.slice(x,x+2):w-56320+(y-55296<<10)+65536}};i.exports={codeAt:createMethod(!1),charAt:createMethod(!0)}},function(i,o,u){var p=u(6),g=u(45),y=u(16),w=p("unscopables"),_=Array.prototype;null==_[w]&&y.f(_,w,{configurable:!0,value:g(null)}),i.exports=function(i){_[w][i]=!0}},function(i,o,u){var p=u(6),g=u(94),y=p("iterator"),w=Array.prototype;i.exports=function(i){return void 0!==i&&(g.Array===i||w[y]===i)}},function(i,o,u){var p=u(119),g=u(94),y=u(6)("iterator");i.exports=function(i){if(null!=i)return i[y]||i["@@iterator"]||g[p(i)]}},function(i,o,u){var p=u(6)("iterator"),g=!1;try{var y=0,w={next:function(){return{done:!!y++}},return:function(){g=!0}};w[p]=function(){return this},Array.from(w,(function(){throw 2}))}catch(i){}i.exports=function(i,o){if(!o&&!g)return!1;var u=!1;try{var y={};y[p]=function(){return{next:function(){return{done:u=!0}}}},i(y)}catch(i){}return u}},function(i,o,u){var p=u(13),g=u(16).f,y=Function.prototype,w=y.toString,_=/^\s*function ([^ (]*)/;p&&!("name"in y)&&g(y,"name",{configurable:!0,get:function(){try{return w.call(this).match(_)[1]}catch(i){return""}}})},function(i,o,u){"use strict";u(121);var p=u(25),g=u(3),y=u(6),w=u(122),_=u(18),x=y("species"),k=!g((function(){var i=/./;return i.exec=function(){var i=[];return i.groups={a:"7"},i},"7"!=="".replace(i,"$<a>")})),P="$0"==="a".replace(/./,"$0"),E=y("replace"),O=!!/./[E]&&""===/./[E]("a","$0"),I=!g((function(){var i=/(?:)/,o=i.exec;i.exec=function(){return o.apply(this,arguments)};var u="ab".split(i);return 2!==u.length||"a"!==u[0]||"b"!==u[1]}));i.exports=function(i,o,u,E){var B=y(i),D=!g((function(){var o={};return o[B]=function(){return 7},7!=""[i](o)})),R=D&&!g((function(){var o=!1,u=/a/;return"split"===i&&((u={}).constructor={},u.constructor[x]=function(){return u},u.flags="",u[B]=/./[B]),u.exec=function(){return o=!0,null},u[B](""),!o}));if(!D||!R||"replace"===i&&(!k||!P||O)||"split"===i&&!I){var N=/./[B],U=u(B,""[i],(function(i,o,u,p,g){return o.exec===w?D&&!g?{done:!0,value:N.call(o,u,p)}:{done:!0,value:i.call(u,o,p)}:{done:!1}}),{REPLACE_KEEPS_$0:P,REGEXP_REPLACE_SUBSTITUTES_UNDEFINED_CAPTURE:O}),W=U[0],G=U[1];p(String.prototype,i,W),p(RegExp.prototype,B,2==o?function(i,o){return G.call(i,this,o)}:function(i){return G.call(i,this)})}E&&_(RegExp.prototype[B],"sham",!0)}},function(i,o,u){"use strict";var p=u(152).charAt;i.exports=function(i,o,u){return o+(u?p(i,o).length:1)}},function(i,o,u){var p=u(42),g=u(122);i.exports=function(i,o){var u=i.exec;if("function"==typeof u){var y=u.call(i,o);if("object"!=typeof y)throw TypeError("RegExp exec method returned something other than an Object or null");return y}if("RegExp"!==p(i))throw TypeError("RegExp#exec called on incompatible receiver");return g.call(i,o)}},function(i,o,u){"use strict";var p=u(158),g=u(12),y=u(8),w=u(35),_=u(159),x=u(160);p("match",1,(function(i,o,u){return[function match(o){var u=w(this),p=null==o?void 0:o[i];return void 0!==p?p.call(o,u):new RegExp(o)[i](String(u))},function(i){var p=u(o,i,this);if(p.done)return p.value;var w=g(i),k=String(this);if(!w.global)return x(w,k);var P=w.unicode;w.lastIndex=0;for(var E,O=[],I=0;null!==(E=x(w,k));){var B=String(E[0]);O[I]=B,""===B&&(w.lastIndex=_(k,y(w.lastIndex),P)),I++}return 0===I?null:O}]}))},function(i,o,u){var p=u(35),g=/"/g;i.exports=function(i,o,u,y){var w=String(p(i)),_="<"+o;return""!==u&&(_+=" "+u+'="'+String(y).replace(g,"&quot;")+'"'),_+">"+w+"</"+o+">"}},function(i,o,u){var p=u(3);i.exports=function(i){return p((function(){var o=""[i]('"');return o!==o.toLowerCase()||o.split('"').length>3}))}},function(i,o,u){"use strict";var p=u(4),g=u(13),y=u(232),w=u(18),_=u(233),x=u(3),k=u(165),P=u(36),E=u(8),O=u(234),I=u(338),B=u(58),D=u(46),R=u(57).f,N=u(16).f,U=u(166),W=u(92),G=u(43),j=G.get,X=G.set,K=p.ArrayBuffer,Y=K,J=p.DataView,$=J&&J.prototype,tt=Object.prototype,et=p.RangeError,rt=I.pack,nt=I.unpack,packInt8=function(i){return[255&i]},packInt16=function(i){return[255&i,i>>8&255]},packInt32=function(i){return[255&i,i>>8&255,i>>16&255,i>>24&255]},unpackInt32=function(i){return i[3]<<24|i[2]<<16|i[1]<<8|i[0]},packFloat32=function(i){return rt(i,23,4)},packFloat64=function(i){return rt(i,52,8)},addGetter=function(i,o){N(i.prototype,o,{get:function(){return j(this)[o]}})},get=function(i,o,u,p){var g=O(u),y=j(i);if(g+o>y.byteLength)throw et("Wrong index");var w=j(y.buffer).bytes,_=g+y.byteOffset,x=w.slice(_,_+o);return p?x:x.reverse()},set=function(i,o,u,p,g,y){var w=O(u),_=j(i);if(w+o>_.byteLength)throw et("Wrong index");for(var x=j(_.buffer).bytes,k=w+_.byteOffset,P=p(+g),E=0;E<o;E++)x[k+E]=P[y?E:o-E-1]};if(y){if(!x((function(){K(1)}))||!x((function(){new K(-1)}))||x((function(){return new K,new K(1.5),new K(NaN),"ArrayBuffer"!=K.name}))){for(var it,ot=(Y=function ArrayBuffer(i){return k(this,Y),new K(O(i))}).prototype=K.prototype,at=R(K),st=0;at.length>st;)(it=at[st++])in Y||w(Y,it,K[it]);ot.constructor=Y}D&&B($)!==tt&&D($,tt);var lt=new J(new Y(2)),ct=$.setInt8;lt.setInt8(0,2147483648),lt.setInt8(1,2147483649),!lt.getInt8(0)&&lt.getInt8(1)||_($,{setInt8:function setInt8(i,o){ct.call(this,i,o<<24>>24)},setUint8:function setUint8(i,o){ct.call(this,i,o<<24>>24)}},{unsafe:!0})}else Y=function ArrayBuffer(i){k(this,Y,"ArrayBuffer");var o=O(i);X(this,{bytes:U.call(new Array(o),0),byteLength:o}),g||(this.byteLength=o)},J=function DataView(i,o,u){k(this,J,"DataView"),k(i,Y,"DataView");var p=j(i).byteLength,y=P(o);if(y<0||y>p)throw et("Wrong offset");if(y+(u=void 0===u?p-y:E(u))>p)throw et("Wrong length");X(this,{buffer:i,byteLength:u,byteOffset:y}),g||(this.buffer=i,this.byteLength=u,this.byteOffset=y)},g&&(addGetter(Y,"byteLength"),addGetter(J,"buffer"),addGetter(J,"byteLength"),addGetter(J,"byteOffset")),_(J.prototype,{getInt8:function getInt8(i){return get(this,1,i)[0]<<24>>24},getUint8:function getUint8(i){return get(this,1,i)[0]},getInt16:function getInt16(i){var o=get(this,2,i,arguments.length>1?arguments[1]:void 0);return(o[1]<<8|o[0])<<16>>16},getUint16:function getUint16(i){var o=get(this,2,i,arguments.length>1?arguments[1]:void 0);return o[1]<<8|o[0]},getInt32:function getInt32(i){return unpackInt32(get(this,4,i,arguments.length>1?arguments[1]:void 0))},getUint32:function getUint32(i){return unpackInt32(get(this,4,i,arguments.length>1?arguments[1]:void 0))>>>0},getFloat32:function getFloat32(i){return nt(get(this,4,i,arguments.length>1?arguments[1]:void 0),23)},getFloat64:function getFloat64(i){return nt(get(this,8,i,arguments.length>1?arguments[1]:void 0),52)},setInt8:function setInt8(i,o){set(this,1,i,packInt8,o)},setUint8:function setUint8(i,o){set(this,1,i,packInt8,o)},setInt16:function setInt16(i,o){set(this,2,i,packInt16,o,arguments.length>2?arguments[2]:void 0)},setUint16:function setUint16(i,o){set(this,2,i,packInt16,o,arguments.length>2?arguments[2]:void 0)},setInt32:function setInt32(i,o){set(this,4,i,packInt32,o,arguments.length>2?arguments[2]:void 0)},setUint32:function setUint32(i,o){set(this,4,i,packInt32,o,arguments.length>2?arguments[2]:void 0)},setFloat32:function setFloat32(i,o){set(this,4,i,packFloat32,o,arguments.length>2?arguments[2]:void 0)},setFloat64:function setFloat64(i,o){set(this,8,i,packFloat64,o,arguments.length>2?arguments[2]:void 0)}});W(Y,"ArrayBuffer"),W(J,"DataView"),i.exports={ArrayBuffer:Y,DataView:J}},function(i,o){i.exports=function(i,o,u){if(!(i instanceof o))throw TypeError("Incorrect "+(u?u+" ":"")+"invocation");return i}},function(i,o,u){"use strict";var p=u(19),g=u(44),y=u(8);i.exports=function fill(i){for(var o=p(this),u=y(o.length),w=arguments.length,_=g(w>1?arguments[1]:void 0,u),x=w>2?arguments[2]:void 0,k=void 0===x?u:g(x,u);k>_;)o[_++]=i;return o}},function(i,o,u){"use strict";var p=u(30),g=u(16),y=u(6),w=u(13),_=y("species");i.exports=function(i){var o=p(i),u=g.f;w&&o&&!o[_]&&u(o,_,{configurable:!0,get:function(){return this}})}},function(i,o,u){(o=i.exports=u(240)).Stream=o,o.Readable=o,o.Writable=u(169),o.Duplex=u(49),o.Transform=u(243),o.PassThrough=u(353)},function(i,o,u){"use strict";(function(o,p){var g=u(125);function CorkedRequest(i){var o=this;this.next=null,this.entry=null,this.finish=function(){!function onCorkedFinish(i,o,u){var p=i.entry;i.entry=null;for(;p;){var g=p.callback;o.pendingcb--,g(u),p=p.next}o.corkedRequestsFree?o.corkedRequestsFree.next=i:o.corkedRequestsFree=i}(o,i)}}i.exports=Writable;var y,w=!o.browser&&["v0.10","v0.9."].indexOf(o.version.slice(0,5))>-1?setImmediate:g.nextTick;Writable.WritableState=WritableState;var _=Object.create(u(95));_.inherits=u(84);var x={deprecate:u(352)},k=u(241),P=u(126).Buffer,E=p.Uint8Array||function(){};var O,I=u(242);function nop(){}function WritableState(i,o){y=y||u(49),i=i||{};var p=o instanceof y;this.objectMode=!!i.objectMode,p&&(this.objectMode=this.objectMode||!!i.writableObjectMode);var _=i.highWaterMark,x=i.writableHighWaterMark,k=this.objectMode?16:16384;this.highWaterMark=_||0===_?_:p&&(x||0===x)?x:k,this.highWaterMark=Math.floor(this.highWaterMark),this.finalCalled=!1,this.needDrain=!1,this.ending=!1,this.ended=!1,this.finished=!1,this.destroyed=!1;var P=!1===i.decodeStrings;this.decodeStrings=!P,this.defaultEncoding=i.defaultEncoding||"utf8",this.length=0,this.writing=!1,this.corked=0,this.sync=!0,this.bufferProcessing=!1,this.onwrite=function(i){!function onwrite(i,o){var u=i._writableState,p=u.sync,y=u.writecb;if(function onwriteStateUpdate(i){i.writing=!1,i.writecb=null,i.length-=i.writelen,i.writelen=0}(u),o)!function onwriteError(i,o,u,p,y){--o.pendingcb,u?(g.nextTick(y,p),g.nextTick(finishMaybe,i,o),i._writableState.errorEmitted=!0,i.emit("error",p)):(y(p),i._writableState.errorEmitted=!0,i.emit("error",p),finishMaybe(i,o))}(i,u,p,o,y);else{var _=needFinish(u);_||u.corked||u.bufferProcessing||!u.bufferedRequest||clearBuffer(i,u),p?w(afterWrite,i,u,_,y):afterWrite(i,u,_,y)}}(o,i)},this.writecb=null,this.writelen=0,this.bufferedRequest=null,this.lastBufferedRequest=null,this.pendingcb=0,this.prefinished=!1,this.errorEmitted=!1,this.bufferedRequestCount=0,this.corkedRequestsFree=new CorkedRequest(this)}function Writable(i){if(y=y||u(49),!(O.call(Writable,this)||this instanceof y))return new Writable(i);this._writableState=new WritableState(i,this),this.writable=!0,i&&("function"==typeof i.write&&(this._write=i.write),"function"==typeof i.writev&&(this._writev=i.writev),"function"==typeof i.destroy&&(this._destroy=i.destroy),"function"==typeof i.final&&(this._final=i.final)),k.call(this)}function doWrite(i,o,u,p,g,y,w){o.writelen=p,o.writecb=w,o.writing=!0,o.sync=!0,u?i._writev(g,o.onwrite):i._write(g,y,o.onwrite),o.sync=!1}function afterWrite(i,o,u,p){u||function onwriteDrain(i,o){0===o.length&&o.needDrain&&(o.needDrain=!1,i.emit("drain"))}(i,o),o.pendingcb--,p(),finishMaybe(i,o)}function clearBuffer(i,o){o.bufferProcessing=!0;var u=o.bufferedRequest;if(i._writev&&u&&u.next){var p=o.bufferedRequestCount,g=new Array(p),y=o.corkedRequestsFree;y.entry=u;for(var w=0,_=!0;u;)g[w]=u,u.isBuf||(_=!1),u=u.next,w+=1;g.allBuffers=_,doWrite(i,o,!0,o.length,g,"",y.finish),o.pendingcb++,o.lastBufferedRequest=null,y.next?(o.corkedRequestsFree=y.next,y.next=null):o.corkedRequestsFree=new CorkedRequest(o),o.bufferedRequestCount=0}else{for(;u;){var x=u.chunk,k=u.encoding,P=u.callback;if(doWrite(i,o,!1,o.objectMode?1:x.length,x,k,P),u=u.next,o.bufferedRequestCount--,o.writing)break}null===u&&(o.lastBufferedRequest=null)}o.bufferedRequest=u,o.bufferProcessing=!1}function needFinish(i){return i.ending&&0===i.length&&null===i.bufferedRequest&&!i.finished&&!i.writing}function callFinal(i,o){i._final((function(u){o.pendingcb--,u&&i.emit("error",u),o.prefinished=!0,i.emit("prefinish"),finishMaybe(i,o)}))}function finishMaybe(i,o){var u=needFinish(o);return u&&(!function prefinish(i,o){o.prefinished||o.finalCalled||("function"==typeof i._final?(o.pendingcb++,o.finalCalled=!0,g.nextTick(callFinal,i,o)):(o.prefinished=!0,i.emit("prefinish")))}(i,o),0===o.pendingcb&&(o.finished=!0,i.emit("finish"))),u}_.inherits(Writable,k),WritableState.prototype.getBuffer=function getBuffer(){for(var i=this.bufferedRequest,o=[];i;)o.push(i),i=i.next;return o},function(){try{Object.defineProperty(WritableState.prototype,"buffer",{get:x.deprecate((function(){return this.getBuffer()}),"_writableState.buffer is deprecated. Use _writableState.getBuffer instead.","DEP0003")})}catch(i){}}(),"function"==typeof Symbol&&Symbol.hasInstance&&"function"==typeof Function.prototype[Symbol.hasInstance]?(O=Function.prototype[Symbol.hasInstance],Object.defineProperty(Writable,Symbol.hasInstance,{value:function(i){return!!O.call(this,i)||this===Writable&&(i&&i._writableState instanceof WritableState)}})):O=function(i){return i instanceof this},Writable.prototype.pipe=function(){this.emit("error",new Error("Cannot pipe, not readable"))},Writable.prototype.write=function(i,o,u){var p=this._writableState,y=!1,w=!p.objectMode&&function _isUint8Array(i){return P.isBuffer(i)||i instanceof E}(i);return w&&!P.isBuffer(i)&&(i=function _uint8ArrayToBuffer(i){return P.from(i)}(i)),"function"==typeof o&&(u=o,o=null),w?o="buffer":o||(o=p.defaultEncoding),"function"!=typeof u&&(u=nop),p.ended?function writeAfterEnd(i,o){var u=new Error("write after end");i.emit("error",u),g.nextTick(o,u)}(this,u):(w||function validChunk(i,o,u,p){var y=!0,w=!1;return null===u?w=new TypeError("May not write null values to stream"):"string"==typeof u||void 0===u||o.objectMode||(w=new TypeError("Invalid non-string/buffer chunk")),w&&(i.emit("error",w),g.nextTick(p,w),y=!1),y}(this,p,i,u))&&(p.pendingcb++,y=function writeOrBuffer(i,o,u,p,g,y){if(!u){var w=function decodeChunk(i,o,u){i.objectMode||!1===i.decodeStrings||"string"!=typeof o||(o=P.from(o,u));return o}(o,p,g);p!==w&&(u=!0,g="buffer",p=w)}var _=o.objectMode?1:p.length;o.length+=_;var x=o.length<o.highWaterMark;x||(o.needDrain=!0);if(o.writing||o.corked){var k=o.lastBufferedRequest;o.lastBufferedRequest={chunk:p,encoding:g,isBuf:u,callback:y,next:null},k?k.next=o.lastBufferedRequest:o.bufferedRequest=o.lastBufferedRequest,o.bufferedRequestCount+=1}else doWrite(i,o,!1,_,p,g,y);return x}(this,p,w,i,o,u)),y},Writable.prototype.cork=function(){this._writableState.corked++},Writable.prototype.uncork=function(){var i=this._writableState;i.corked&&(i.corked--,i.writing||i.corked||i.finished||i.bufferProcessing||!i.bufferedRequest||clearBuffer(this,i))},Writable.prototype.setDefaultEncoding=function setDefaultEncoding(i){if("string"==typeof i&&(i=i.toLowerCase()),!(["hex","utf8","utf-8","ascii","binary","base64","ucs2","ucs-2","utf16le","utf-16le","raw"].indexOf((i+"").toLowerCase())>-1))throw new TypeError("Unknown encoding: "+i);return this._writableState.defaultEncoding=i,this},Object.defineProperty(Writable.prototype,"writableHighWaterMark",{enumerable:!1,get:function(){return this._writableState.highWaterMark}}),Writable.prototype._write=function(i,o,u){u(new Error("_write() is not implemented"))},Writable.prototype._writev=null,Writable.prototype.end=function(i,o,u){var p=this._writableState;"function"==typeof i?(u=i,i=null,o=null):"function"==typeof o&&(u=o,o=null),null!=i&&this.write(i,o),p.corked&&(p.corked=1,this.uncork()),p.ending||p.finished||function endWritable(i,o,u){o.ending=!0,finishMaybe(i,o),u&&(o.finished?g.nextTick(u):i.once("finish",u));o.ended=!0,i.writable=!1}(this,p,u)},Object.defineProperty(Writable.prototype,"destroyed",{get:function(){return void 0!==this._writableState&&this._writableState.destroyed},set:function(i){this._writableState&&(this._writableState.destroyed=i)}}),Writable.prototype.destroy=I.destroy,Writable.prototype._undestroy=I.undestroy,Writable.prototype._destroy=function(i,o){this.end(),o(i)}}).call(this,u(48),u(27))},function(i,o,u){(function(i){var p=Object.getOwnPropertyDescriptors||function getOwnPropertyDescriptors(i){for(var o=Object.keys(i),u={},p=0;p<o.length;p++)u[o[p]]=Object.getOwnPropertyDescriptor(i,o[p]);return u},g=/%[sdj%]/g;o.format=function(i){if(!isString(i)){for(var o=[],u=0;u<arguments.length;u++)o.push(inspect(arguments[u]));return o.join(" ")}u=1;for(var p=arguments,y=p.length,w=String(i).replace(g,(function(i){if("%%"===i)return"%";if(u>=y)return i;switch(i){case"%s":return String(p[u++]);case"%d":return Number(p[u++]);case"%j":try{return JSON.stringify(p[u++])}catch(i){return"[Circular]"}default:return i}})),_=p[u];u<y;_=p[++u])isNull(_)||!isObject(_)?w+=" "+_:w+=" "+inspect(_);return w},o.deprecate=function(u,p){if(void 0!==i&&!0===i.noDeprecation)return u;if(void 0===i)return function(){return o.deprecate(u,p).apply(this,arguments)};var g=!1;return function deprecated(){if(!g){if(i.throwDeprecation)throw new Error(p);i.traceDeprecation,g=!0}return u.apply(this,arguments)}};var y,w={};function inspect(i,u){var p={seen:[],stylize:stylizeNoColor};return arguments.length>=3&&(p.depth=arguments[2]),arguments.length>=4&&(p.colors=arguments[3]),isBoolean(u)?p.showHidden=u:u&&o._extend(p,u),isUndefined(p.showHidden)&&(p.showHidden=!1),isUndefined(p.depth)&&(p.depth=2),isUndefined(p.colors)&&(p.colors=!1),isUndefined(p.customInspect)&&(p.customInspect=!0),p.colors&&(p.stylize=stylizeWithColor),formatValue(p,i,p.depth)}function stylizeWithColor(i,o){var u=inspect.styles[o];return u?"["+inspect.colors[u][0]+"m"+i+"["+inspect.colors[u][1]+"m":i}function stylizeNoColor(i,o){return i}function formatValue(i,u,p){if(i.customInspect&&u&&isFunction(u.inspect)&&u.inspect!==o.inspect&&(!u.constructor||u.constructor.prototype!==u)){var g=u.inspect(p,i);return isString(g)||(g=formatValue(i,g,p)),g}var y=function formatPrimitive(i,o){if(isUndefined(o))return i.stylize("undefined","undefined");if(isString(o)){var u="'"+JSON.stringify(o).replace(/^"|"$/g,"").replace(/'/g,"\\'").replace(/\\"/g,'"')+"'";return i.stylize(u,"string")}if(isNumber(o))return i.stylize(""+o,"number");if(isBoolean(o))return i.stylize(""+o,"boolean");if(isNull(o))return i.stylize("null","null")}(i,u);if(y)return y;var w=Object.keys(u),_=function arrayToHash(i){var o={};return i.forEach((function(i,u){o[i]=!0})),o}(w);if(i.showHidden&&(w=Object.getOwnPropertyNames(u)),isError(u)&&(w.indexOf("message")>=0||w.indexOf("description")>=0))return formatError(u);if(0===w.length){if(isFunction(u)){var x=u.name?": "+u.name:"";return i.stylize("[Function"+x+"]","special")}if(isRegExp(u))return i.stylize(RegExp.prototype.toString.call(u),"regexp");if(isDate(u))return i.stylize(Date.prototype.toString.call(u),"date");if(isError(u))return formatError(u)}var k,P="",E=!1,O=["{","}"];(isArray(u)&&(E=!0,O=["[","]"]),isFunction(u))&&(P=" [Function"+(u.name?": "+u.name:"")+"]");return isRegExp(u)&&(P=" "+RegExp.prototype.toString.call(u)),isDate(u)&&(P=" "+Date.prototype.toUTCString.call(u)),isError(u)&&(P=" "+formatError(u)),0!==w.length||E&&0!=u.length?p<0?isRegExp(u)?i.stylize(RegExp.prototype.toString.call(u),"regexp"):i.stylize("[Object]","special"):(i.seen.push(u),k=E?function formatArray(i,o,u,p,g){for(var y=[],w=0,_=o.length;w<_;++w)hasOwnProperty(o,String(w))?y.push(formatProperty(i,o,u,p,String(w),!0)):y.push("");return g.forEach((function(g){g.match(/^\d+$/)||y.push(formatProperty(i,o,u,p,g,!0))})),y}(i,u,p,_,w):w.map((function(o){return formatProperty(i,u,p,_,o,E)})),i.seen.pop(),function reduceToSingleString(i,o,u){if(i.reduce((function(i,o){return o.indexOf("\n")>=0&&0,i+o.replace(/\u001b\[\d\d?m/g,"").length+1}),0)>60)return u[0]+(""===o?"":o+"\n ")+" "+i.join(",\n  ")+" "+u[1];return u[0]+o+" "+i.join(", ")+" "+u[1]}(k,P,O)):O[0]+P+O[1]}function formatError(i){return"["+Error.prototype.toString.call(i)+"]"}function formatProperty(i,o,u,p,g,y){var w,_,x;if((x=Object.getOwnPropertyDescriptor(o,g)||{value:o[g]}).get?_=x.set?i.stylize("[Getter/Setter]","special"):i.stylize("[Getter]","special"):x.set&&(_=i.stylize("[Setter]","special")),hasOwnProperty(p,g)||(w="["+g+"]"),_||(i.seen.indexOf(x.value)<0?(_=isNull(u)?formatValue(i,x.value,null):formatValue(i,x.value,u-1)).indexOf("\n")>-1&&(_=y?_.split("\n").map((function(i){return"  "+i})).join("\n").substr(2):"\n"+_.split("\n").map((function(i){return"   "+i})).join("\n")):_=i.stylize("[Circular]","special")),isUndefined(w)){if(y&&g.match(/^\d+$/))return _;(w=JSON.stringify(""+g)).match(/^"([a-zA-Z_][a-zA-Z_0-9]*)"$/)?(w=w.substr(1,w.length-2),w=i.stylize(w,"name")):(w=w.replace(/'/g,"\\'").replace(/\\"/g,'"').replace(/(^"|"$)/g,"'"),w=i.stylize(w,"string"))}return w+": "+_}function isArray(i){return Array.isArray(i)}function isBoolean(i){return"boolean"==typeof i}function isNull(i){return null===i}function isNumber(i){return"number"==typeof i}function isString(i){return"string"==typeof i}function isUndefined(i){return void 0===i}function isRegExp(i){return isObject(i)&&"[object RegExp]"===objectToString(i)}function isObject(i){return"object"==typeof i&&null!==i}function isDate(i){return isObject(i)&&"[object Date]"===objectToString(i)}function isError(i){return isObject(i)&&("[object Error]"===objectToString(i)||i instanceof Error)}function isFunction(i){return"function"==typeof i}function objectToString(i){return Object.prototype.toString.call(i)}o.debuglog=function(u){if(isUndefined(y)&&(y=i.env.NODE_DEBUG||""),u=u.toUpperCase(),!w[u])if(new RegExp("\\b"+u+"\\b","i").test(y)){i.pid;w[u]=function(){o.format.apply(o,arguments)}}else w[u]=function(){};return w[u]},o.inspect=inspect,inspect.colors={bold:[1,22],italic:[3,23],underline:[4,24],inverse:[7,27],white:[37,39],grey:[90,39],black:[30,39],blue:[34,39],cyan:[36,39],green:[32,39],magenta:[35,39],red:[31,39],yellow:[33,39]},inspect.styles={special:"cyan",number:"yellow",boolean:"yellow",undefined:"grey",null:"bold",string:"green",date:"magenta",regexp:"red"},o.isArray=isArray,o.isBoolean=isBoolean,o.isNull=isNull,o.isNullOrUndefined=function isNullOrUndefined(i){return null==i},o.isNumber=isNumber,o.isString=isString,o.isSymbol=function isSymbol(i){return"symbol"==typeof i},o.isUndefined=isUndefined,o.isRegExp=isRegExp,o.isObject=isObject,o.isDate=isDate,o.isError=isError,o.isFunction=isFunction,o.isPrimitive=function isPrimitive(i){return null===i||"boolean"==typeof i||"number"==typeof i||"string"==typeof i||"symbol"==typeof i||void 0===i},o.isBuffer=u(360);function hasOwnProperty(i,o){return Object.prototype.hasOwnProperty.call(i,o)}o.log=function(){},o.inherits=u(361),o._extend=function(i,o){if(!o||!isObject(o))return i;for(var u=Object.keys(o),p=u.length;p--;)i[u[p]]=o[u[p]];return i};var _="undefined"!=typeof Symbol?Symbol("util.promisify.custom"):void 0;function callbackifyOnRejected(i,o){if(!i){var u=new Error("Promise was rejected with a falsy value");u.reason=i,i=u}return o(i)}o.promisify=function promisify(i){if("function"!=typeof i)throw new TypeError('The "original" argument must be of type Function');if(_&&i[_]){var o;if("function"!=typeof(o=i[_]))throw new TypeError('The "util.promisify.custom" argument must be of type Function');return Object.defineProperty(o,_,{value:o,enumerable:!1,writable:!1,configurable:!0}),o}function o(){for(var o,u,p=new Promise((function(i,p){o=i,u=p})),g=[],y=0;y<arguments.length;y++)g.push(arguments[y]);g.push((function(i,p){i?u(i):o(p)}));try{i.apply(this,g)}catch(i){u(i)}return p}return Object.setPrototypeOf(o,Object.getPrototypeOf(i)),_&&Object.defineProperty(o,_,{value:o,enumerable:!1,writable:!1,configurable:!0}),Object.defineProperties(o,p(i))},o.promisify.custom=_,o.callbackify=function callbackify(o){if("function"!=typeof o)throw new TypeError('The "original" argument must be of type Function');function callbackified(){for(var u=[],p=0;p<arguments.length;p++)u.push(arguments[p]);var g=u.pop();if("function"!=typeof g)throw new TypeError("The last argument must be of type Function");var y=this,cb=function(){return g.apply(y,arguments)};o.apply(this,u).then((function(o){i.nextTick(cb,null,o)}),(function(o){i.nextTick(callbackifyOnRejected,o,cb)}))}return Object.setPrototypeOf(callbackified,Object.getPrototypeOf(o)),Object.defineProperties(callbackified,p(o)),callbackified}}).call(this,u(48))},function(i,o,u){var p,g,y,w,_,x,k,P;i.exports=(P=u(2),g=(p=P).lib,y=g.WordArray,w=g.Hasher,_=p.algo,x=[],k=_.SHA1=w.extend({_doReset:function(){this._hash=new y.init([1732584193,4023233417,2562383102,271733878,3285377520])},_doProcessBlock:function(i,o){for(var u=this._hash.words,p=u[0],g=u[1],y=u[2],w=u[3],_=u[4],k=0;k<80;k++){if(k<16)x[k]=0|i[o+k];else{var P=x[k-3]^x[k-8]^x[k-14]^x[k-16];x[k]=P<<1|P>>>31}var E=(p<<5|p>>>27)+_+x[k];E+=k<20?1518500249+(g&y|~g&w):k<40?1859775393+(g^y^w):k<60?(g&y|g&w|y&w)-1894007588:(g^y^w)-899497514,_=w,w=y,y=g<<30|g>>>2,g=p,p=E}u[0]=u[0]+p|0,u[1]=u[1]+g|0,u[2]=u[2]+y|0,u[3]=u[3]+w|0,u[4]=u[4]+_|0},_doFinalize:function(){var i=this._data,o=i.words,u=8*this._nDataBytes,p=8*i.sigBytes;return o[p>>>5]|=128<<24-p%32,o[14+(p+64>>>9<<4)]=Math.floor(u/4294967296),o[15+(p+64>>>9<<4)]=u,i.sigBytes=4*o.length,this._process(),this._hash},clone:function(){var i=w.clone.call(this);return i._hash=this._hash.clone(),i}}),p.SHA1=w._createHelper(k),p.HmacSHA1=w._createHmacHelper(k),P.SHA1)},function(i,o,u){var p,g,y,w;i.exports=(p=u(2),y=(g=p).lib.Base,w=g.enc.Utf8,void(g.algo.HMAC=y.extend({init:function(i,o){i=this._hasher=new i.init,"string"==typeof o&&(o=w.parse(o));var u=i.blockSize,p=4*u;o.sigBytes>p&&(o=i.finalize(o)),o.clamp();for(var g=this._oKey=o.clone(),y=this._iKey=o.clone(),_=g.words,x=y.words,k=0;k<u;k++)_[k]^=1549556828,x[k]^=909522486;g.sigBytes=y.sigBytes=p,this.reset()},reset:function(){var i=this._hasher;i.reset(),i.update(this._iKey)},update:function(i){return this._hasher.update(i),this},finalize:function(i){var o=this._hasher,u=o.finalize(i);return o.reset(),o.finalize(this._oKey.clone().concat(u))}})))},function(i,o,u){(function(o){(function(){var p,g;try{g=u(174)}catch(i){}p=function(){var i;function DecodeStream(i){this.buffer=i,this.pos=0,this.length=this.buffer.length}for(i in DecodeStream.TYPES={UInt8:1,UInt16:2,UInt24:3,UInt32:4,Int8:1,Int16:2,Int24:3,Int32:4,Float:4,Double:8},o.prototype)"read"===i.slice(0,4)&&function(i){var o;o=DecodeStream.TYPES[i.replace(/read|[BL]E/g,"")],DecodeStream.prototype[i]=function(){var u;return u=this.buffer[i](this.pos),this.pos+=o,u}}(i);return DecodeStream.prototype.readString=function(i,u){var p,y,w,_,x;switch(null==u&&(u="ascii"),u){case"utf16le":case"ucs2":case"utf8":case"ascii":return this.buffer.toString(u,this.pos,this.pos+=i);case"utf16be":for(w=_=0,x=(p=new o(this.readBuffer(i))).length-1;_<x;w=_+=2)y=p[w],p[w]=p[w+1],p[w+1]=y;return p.toString("utf16le");default:if(p=this.readBuffer(i),g)try{return g.decode(p,u)}catch(i){}return p}},DecodeStream.prototype.readBuffer=function(i){return this.buffer.slice(this.pos,this.pos+=i)},DecodeStream.prototype.readUInt24BE=function(){return(this.readUInt16BE()<<8)+this.readUInt8()},DecodeStream.prototype.readUInt24LE=function(){return this.readUInt16LE()+(this.readUInt8()<<16)},DecodeStream.prototype.readInt24BE=function(){return(this.readInt16BE()<<8)+this.readUInt8()},DecodeStream.prototype.readInt24LE=function(){return this.readUInt16LE()+(this.readInt8()<<16)},DecodeStream}(),i.exports=p}).call(this)}).call(this,u(10).Buffer)},function(i,o,u){"use strict";var p,g=u(38).Buffer,y=u(397),w=i.exports;w.encodings=null,w.defaultCharUnicode="�",w.defaultCharSingleByte="?",w.encode=function encode(i,o,u){i=""+(i||"");var p=w.getEncoder(o,u),y=p.write(i),_=p.end();return _&&_.length>0?g.concat([y,_]):y},w.decode=function decode(i,o,u){"string"==typeof i&&(w.skipDecodeWarning||(w.skipDecodeWarning=!0),i=g.from(""+(i||""),"binary"));var p=w.getDecoder(o,u),y=p.write(i),_=p.end();return _?y+_:y},w.encodingExists=function encodingExists(i){try{return w.getCodec(i),!0}catch(i){return!1}},w.toEncoding=w.encode,w.fromEncoding=w.decode,w._codecDataCache={},w.getCodec=function getCodec(i){w.encodings||(w.encodings=u(398));for(var o=w._canonicalizeEncoding(i),p={};;){var g=w._codecDataCache[o];if(g)return g;var y=w.encodings[o];switch(typeof y){case"string":o=y;break;case"object":for(var _ in y)p[_]=y[_];p.encodingName||(p.encodingName=o),o=y.type;break;case"function":return p.encodingName||(p.encodingName=o),g=new y(p,w),w._codecDataCache[p.encodingName]=g,g;default:throw new Error("Encoding not recognized: '"+i+"' (searched as: '"+o+"')")}}},w._canonicalizeEncoding=function(i){return(""+i).toLowerCase().replace(/:\d{4}$|[^0-9a-z]/g,"")},w.getEncoder=function getEncoder(i,o){var u=w.getCodec(i),p=new u.encoder(o,u);return u.bomAware&&o&&o.addBOM&&(p=new y.PrependBOM(p,o)),p},w.getDecoder=function getDecoder(i,o){var u=w.getCodec(i),p=new u.decoder(o,u);return!u.bomAware||o&&!1===o.stripBOM||(p=new y.StripBOM(p,o)),p},w.enableStreamingAPI=function enableStreamingAPI(i){if(!w.supportsStreams){var o=u(413)(i);w.IconvLiteEncoderStream=o.IconvLiteEncoderStream,w.IconvLiteDecoderStream=o.IconvLiteDecoderStream,w.encodeStream=function encodeStream(i,o){return new w.IconvLiteEncoderStream(w.getEncoder(i,o),o)},w.decodeStream=function decodeStream(i,o){return new w.IconvLiteDecoderStream(w.getDecoder(i,o),o)},w.supportsStreams=!0}};try{p=u(414)}catch(i){}p&&p.Transform?w.enableStreamingAPI(p):w.encodeStream=w.decodeStream=function(){throw new Error("iconv-lite Streaming API is not enabled. Use iconv.enableStreamingAPI(require('stream')); to enable it.")}},function(i){i.exports=JSON.parse('[["0","\\u0000",127,"€"],["8140","丂丄丅丆丏丒丗丟丠両丣並丩丮丯丱丳丵丷丼乀乁乂乄乆乊乑乕乗乚乛乢乣乤乥乧乨乪",5,"乲乴",9,"乿",6,"亇亊"],["8180","亐亖亗亙亜亝亞亣亪亯亰亱亴亶亷亸亹亼亽亾仈仌仏仐仒仚仛仜仠仢仦仧仩仭仮仯仱仴仸仹仺仼仾伀伂",6,"伋伌伒",4,"伜伝伡伣伨伩伬伭伮伱伳伵伷伹伻伾",4,"佄佅佇",5,"佒佔佖佡佢佦佨佪佫佭佮佱佲併佷佸佹佺佽侀侁侂侅來侇侊侌侎侐侒侓侕侖侘侙侚侜侞侟価侢"],["8240","侤侫侭侰",4,"侶",8,"俀俁係俆俇俈俉俋俌俍俒",4,"俙俛俠俢俤俥俧俫俬俰俲俴俵俶俷俹俻俼俽俿",11],["8280","個倎倐們倓倕倖倗倛倝倞倠倢倣値倧倫倯",10,"倻倽倿偀偁偂偄偅偆偉偊偋偍偐",4,"偖偗偘偙偛偝",7,"偦",5,"偭",8,"偸偹偺偼偽傁傂傃傄傆傇傉傊傋傌傎",20,"傤傦傪傫傭",4,"傳",6,"傼"],["8340","傽",17,"僐",5,"僗僘僙僛",10,"僨僩僪僫僯僰僱僲僴僶",4,"僼",9,"儈"],["8380","儉儊儌",5,"儓",13,"儢",28,"兂兇兊兌兎兏児兒兓兗兘兙兛兝",4,"兣兤兦內兩兪兯兲兺兾兿冃冄円冇冊冋冎冏冐冑冓冔冘冚冝冞冟冡冣冦",4,"冭冮冴冸冹冺冾冿凁凂凃凅凈凊凍凎凐凒",5],["8440","凘凙凚凜凞凟凢凣凥",5,"凬凮凱凲凴凷凾刄刅刉刋刌刏刐刓刔刕刜刞刟刡刢刣別刦刧刪刬刯刱刲刴刵刼刾剄",5,"剋剎剏剒剓剕剗剘"],["8480","剙剚剛剝剟剠剢剣剤剦剨剫剬剭剮剰剱剳",9,"剾劀劃",4,"劉",6,"劑劒劔",6,"劜劤劥劦劧劮劯劰労",9,"勀勁勂勄勅勆勈勊勌勍勎勏勑勓勔動勗務",5,"勠勡勢勣勥",10,"勱",7,"勻勼勽匁匂匃匄匇匉匊匋匌匎"],["8540","匑匒匓匔匘匛匜匞匟匢匤匥匧匨匩匫匬匭匯",9,"匼匽區卂卄卆卋卌卍卐協単卙卛卝卥卨卪卬卭卲卶卹卻卼卽卾厀厁厃厇厈厊厎厏"],["8580","厐",4,"厖厗厙厛厜厞厠厡厤厧厪厫厬厭厯",6,"厷厸厹厺厼厽厾叀參",4,"収叏叐叒叓叕叚叜叝叞叡叢叧叴叺叾叿吀吂吅吇吋吔吘吙吚吜吢吤吥吪吰吳吶吷吺吽吿呁呂呄呅呇呉呌呍呎呏呑呚呝",4,"呣呥呧呩",7,"呴呹呺呾呿咁咃咅咇咈咉咊咍咑咓咗咘咜咞咟咠咡"],["8640","咢咥咮咰咲咵咶咷咹咺咼咾哃哅哊哋哖哘哛哠",4,"哫哬哯哰哱哴",5,"哻哾唀唂唃唄唅唈唊",4,"唒唓唕",5,"唜唝唞唟唡唥唦"],["8680","唨唩唫唭唲唴唵唶唸唹唺唻唽啀啂啅啇啈啋",4,"啑啒啓啔啗",4,"啝啞啟啠啢啣啨啩啫啯",5,"啹啺啽啿喅喆喌喍喎喐喒喓喕喖喗喚喛喞喠",6,"喨",8,"喲喴営喸喺喼喿",4,"嗆嗇嗈嗊嗋嗎嗏嗐嗕嗗",4,"嗞嗠嗢嗧嗩嗭嗮嗰嗱嗴嗶嗸",4,"嗿嘂嘃嘄嘅"],["8740","嘆嘇嘊嘋嘍嘐",7,"嘙嘚嘜嘝嘠嘡嘢嘥嘦嘨嘩嘪嘫嘮嘯嘰嘳嘵嘷嘸嘺嘼嘽嘾噀",11,"噏",4,"噕噖噚噛噝",4],["8780","噣噥噦噧噭噮噯噰噲噳噴噵噷噸噹噺噽",7,"嚇",6,"嚐嚑嚒嚔",14,"嚤",10,"嚰",6,"嚸嚹嚺嚻嚽",12,"囋",8,"囕囖囘囙囜団囥",5,"囬囮囯囲図囶囷囸囻囼圀圁圂圅圇國",6],["8840","園",9,"圝圞圠圡圢圤圥圦圧圫圱圲圴",4,"圼圽圿坁坃坄坅坆坈坉坋坒",4,"坘坙坢坣坥坧坬坮坰坱坲坴坵坸坹坺坽坾坿垀"],["8880","垁垇垈垉垊垍",4,"垔",6,"垜垝垞垟垥垨垪垬垯垰垱垳垵垶垷垹",8,"埄",6,"埌埍埐埑埓埖埗埛埜埞埡埢埣埥",7,"埮埰埱埲埳埵埶執埻埼埾埿堁堃堄堅堈堉堊堌堎堏堐堒堓堔堖堗堘堚堛堜堝堟堢堣堥",4,"堫",4,"報堲堳場堶",7],["8940","堾",5,"塅",6,"塎塏塐塒塓塕塖塗塙",4,"塟",5,"塦",4,"塭",16,"塿墂墄墆墇墈墊墋墌"],["8980","墍",4,"墔",4,"墛墜墝墠",7,"墪",17,"墽墾墿壀壂壃壄壆",10,"壒壓壔壖",13,"壥",5,"壭壯壱売壴壵壷壸壺",7,"夃夅夆夈",4,"夎夐夑夒夓夗夘夛夝夞夠夡夢夣夦夨夬夰夲夳夵夶夻"],["8a40","夽夾夿奀奃奅奆奊奌奍奐奒奓奙奛",4,"奡奣奤奦",12,"奵奷奺奻奼奾奿妀妅妉妋妌妎妏妐妑妔妕妘妚妛妜妝妟妠妡妢妦"],["8a80","妧妬妭妰妱妳",5,"妺妼妽妿",6,"姇姈姉姌姍姎姏姕姖姙姛姞",4,"姤姦姧姩姪姫姭",11,"姺姼姽姾娀娂娊娋娍娎娏娐娒娔娕娖娗娙娚娛娝娞娡娢娤娦娧娨娪",6,"娳娵娷",4,"娽娾娿婁",4,"婇婈婋",9,"婖婗婘婙婛",5],["8b40","婡婣婤婥婦婨婩婫",8,"婸婹婻婼婽婾媀",17,"媓",6,"媜",13,"媫媬"],["8b80","媭",4,"媴媶媷媹",4,"媿嫀嫃",5,"嫊嫋嫍",4,"嫓嫕嫗嫙嫚嫛嫝嫞嫟嫢嫤嫥嫧嫨嫪嫬",4,"嫲",22,"嬊",11,"嬘",25,"嬳嬵嬶嬸",7,"孁",6],["8c40","孈",7,"孒孖孞孠孡孧孨孫孭孮孯孲孴孶孷學孹孻孼孾孿宂宆宊宍宎宐宑宒宔宖実宧宨宩宬宭宮宯宱宲宷宺宻宼寀寁寃寈寉寊寋寍寎寏"],["8c80","寑寔",8,"寠寢寣實寧審",4,"寯寱",6,"寽対尀専尃尅將專尋尌對導尐尒尓尗尙尛尞尟尠尡尣尦尨尩尪尫尭尮尯尰尲尳尵尶尷屃屄屆屇屌屍屒屓屔屖屗屘屚屛屜屝屟屢層屧",6,"屰屲",6,"屻屼屽屾岀岃",4,"岉岊岋岎岏岒岓岕岝",4,"岤",4],["8d40","岪岮岯岰岲岴岶岹岺岻岼岾峀峂峃峅",5,"峌",5,"峓",5,"峚",6,"峢峣峧峩峫峬峮峯峱",9,"峼",4],["8d80","崁崄崅崈",5,"崏",4,"崕崗崘崙崚崜崝崟",4,"崥崨崪崫崬崯",4,"崵",7,"崿",7,"嵈嵉嵍",10,"嵙嵚嵜嵞",10,"嵪嵭嵮嵰嵱嵲嵳嵵",12,"嶃",21,"嶚嶛嶜嶞嶟嶠"],["8e40","嶡",21,"嶸",12,"巆",6,"巎",12,"巜巟巠巣巤巪巬巭"],["8e80","巰巵巶巸",4,"巿帀帄帇帉帊帋帍帎帒帓帗帞",7,"帨",4,"帯帰帲",4,"帹帺帾帿幀幁幃幆",5,"幍",6,"幖",4,"幜幝幟幠幣",14,"幵幷幹幾庁庂広庅庈庉庌庍庎庒庘庛庝庡庢庣庤庨",4,"庮",4,"庴庺庻庼庽庿",6],["8f40","廆廇廈廋",5,"廔廕廗廘廙廚廜",11,"廩廫",8,"廵廸廹廻廼廽弅弆弇弉弌弍弎弐弒弔弖弙弚弜弝弞弡弢弣弤"],["8f80","弨弫弬弮弰弲",6,"弻弽弾弿彁",14,"彑彔彙彚彛彜彞彟彠彣彥彧彨彫彮彯彲彴彵彶彸彺彽彾彿徃徆徍徎徏徑従徔徖徚徛徝從徟徠徢",5,"復徫徬徯",5,"徶徸徹徺徻徾",4,"忇忈忊忋忎忓忔忕忚忛応忞忟忢忣忥忦忨忩忬忯忰忲忳忴忶忷忹忺忼怇"],["9040","怈怉怋怌怐怑怓怗怘怚怞怟怢怣怤怬怭怮怰",4,"怶",4,"怽怾恀恄",6,"恌恎恏恑恓恔恖恗恘恛恜恞恟恠恡恥恦恮恱恲恴恵恷恾悀"],["9080","悁悂悅悆悇悈悊悋悎悏悐悑悓悕悗悘悙悜悞悡悢悤悥悧悩悪悮悰悳悵悶悷悹悺悽",7,"惇惈惉惌",4,"惒惓惔惖惗惙惛惞惡",4,"惪惱惲惵惷惸惻",4,"愂愃愄愅愇愊愋愌愐",4,"愖愗愘愙愛愜愝愞愡愢愥愨愩愪愬",18,"慀",6],["9140","慇慉態慍慏慐慒慓慔慖",6,"慞慟慠慡慣慤慥慦慩",6,"慱慲慳慴慶慸",18,"憌憍憏",4,"憕"],["9180","憖",6,"憞",8,"憪憫憭",9,"憸",5,"憿懀懁懃",4,"應懌",4,"懓懕",16,"懧",13,"懶",8,"戀",5,"戇戉戓戔戙戜戝戞戠戣戦戧戨戩戫戭戯戰戱戲戵戶戸",4,"扂扄扅扆扊"],["9240","扏扐払扖扗扙扚扜",6,"扤扥扨扱扲扴扵扷扸扺扻扽抁抂抃抅抆抇抈抋",5,"抔抙抜抝択抣抦抧抩抪抭抮抯抰抲抳抴抶抷抸抺抾拀拁"],["9280","拃拋拏拑拕拝拞拠拡拤拪拫拰拲拵拸拹拺拻挀挃挄挅挆挊挋挌挍挏挐挒挓挔挕挗挘挙挜挦挧挩挬挭挮挰挱挳",5,"挻挼挾挿捀捁捄捇捈捊捑捒捓捔捖",7,"捠捤捥捦捨捪捫捬捯捰捲捳捴捵捸捹捼捽捾捿掁掃掄掅掆掋掍掑掓掔掕掗掙",6,"採掤掦掫掯掱掲掵掶掹掻掽掿揀"],["9340","揁揂揃揅揇揈揊揋揌揑揓揔揕揗",6,"揟揢揤",4,"揫揬揮揯揰揱揳揵揷揹揺揻揼揾搃搄搆",4,"損搎搑搒搕",5,"搝搟搢搣搤"],["9380","搥搧搨搩搫搮",5,"搵",4,"搻搼搾摀摂摃摉摋",6,"摓摕摖摗摙",4,"摟",7,"摨摪摫摬摮",9,"摻",6,"撃撆撈",8,"撓撔撗撘撚撛撜撝撟",4,"撥撦撧撨撪撫撯撱撲撳撴撶撹撻撽撾撿擁擃擄擆",6,"擏擑擓擔擕擖擙據"],["9440","擛擜擝擟擠擡擣擥擧",24,"攁",7,"攊",7,"攓",4,"攙",8],["9480","攢攣攤攦",4,"攬攭攰攱攲攳攷攺攼攽敀",4,"敆敇敊敋敍敎敐敒敓敔敗敘敚敜敟敠敡敤敥敧敨敩敪敭敮敯敱敳敵敶數",14,"斈斉斊斍斎斏斒斔斕斖斘斚斝斞斠斢斣斦斨斪斬斮斱",7,"斺斻斾斿旀旂旇旈旉旊旍旐旑旓旔旕旘",7,"旡旣旤旪旫"],["9540","旲旳旴旵旸旹旻",4,"昁昄昅昇昈昉昋昍昐昑昒昖昗昘昚昛昜昞昡昢昣昤昦昩昪昫昬昮昰昲昳昷",4,"昽昿晀時晄",6,"晍晎晐晑晘"],["9580","晙晛晜晝晞晠晢晣晥晧晩",4,"晱晲晳晵晸晹晻晼晽晿暀暁暃暅暆暈暉暊暋暍暎暏暐暒暓暔暕暘",4,"暞",8,"暩",4,"暯",4,"暵暶暷暸暺暻暼暽暿",25,"曚曞",7,"曧曨曪",5,"曱曵曶書曺曻曽朁朂會"],["9640","朄朅朆朇朌朎朏朑朒朓朖朘朙朚朜朞朠",5,"朧朩朮朰朲朳朶朷朸朹朻朼朾朿杁杄杅杇杊杋杍杒杔杕杗",4,"杝杢杣杤杦杧杫杬杮東杴杶"],["9680","杸杹杺杻杽枀枂枃枅枆枈枊枌枍枎枏枑枒枓枔枖枙枛枟枠枡枤枦枩枬枮枱枲枴枹",7,"柂柅",9,"柕柖柗柛柟柡柣柤柦柧柨柪柫柭柮柲柵",7,"柾栁栂栃栄栆栍栐栒栔栕栘",4,"栞栟栠栢",6,"栫",6,"栴栵栶栺栻栿桇桋桍桏桒桖",5],["9740","桜桝桞桟桪桬",7,"桵桸",8,"梂梄梇",7,"梐梑梒梔梕梖梘",9,"梣梤梥梩梪梫梬梮梱梲梴梶梷梸"],["9780","梹",6,"棁棃",5,"棊棌棎棏棐棑棓棔棖棗棙棛",4,"棡棢棤",9,"棯棲棳棴棶棷棸棻棽棾棿椀椂椃椄椆",4,"椌椏椑椓",11,"椡椢椣椥",7,"椮椯椱椲椳椵椶椷椸椺椻椼椾楀楁楃",16,"楕楖楘楙楛楜楟"],["9840","楡楢楤楥楧楨楩楪楬業楯楰楲",4,"楺楻楽楾楿榁榃榅榊榋榌榎",5,"榖榗榙榚榝",9,"榩榪榬榮榯榰榲榳榵榶榸榹榺榼榽"],["9880","榾榿槀槂",7,"構槍槏槑槒槓槕",5,"槜槝槞槡",11,"槮槯槰槱槳",9,"槾樀",9,"樋",11,"標",5,"樠樢",5,"権樫樬樭樮樰樲樳樴樶",6,"樿",4,"橅橆橈",7,"橑",6,"橚"],["9940","橜",4,"橢橣橤橦",10,"橲",6,"橺橻橽橾橿檁檂檃檅",8,"檏檒",4,"檘",7,"檡",5],["9980","檧檨檪檭",114,"欥欦欨",6],["9a40","欯欰欱欳欴欵欶欸欻欼欽欿歀歁歂歄歅歈歊歋歍",11,"歚",7,"歨歩歫",13,"歺歽歾歿殀殅殈"],["9a80","殌殎殏殐殑殔殕殗殘殙殜",4,"殢",7,"殫",7,"殶殸",6,"毀毃毄毆",4,"毌毎毐毑毘毚毜",4,"毢",7,"毬毭毮毰毱毲毴毶毷毸毺毻毼毾",6,"氈",4,"氎氒気氜氝氞氠氣氥氫氬氭氱氳氶氷氹氺氻氼氾氿汃汄汅汈汋",4,"汑汒汓汖汘"],["9b40","汙汚汢汣汥汦汧汫",4,"汱汳汵汷汸決汻汼汿沀沄沇沊沋沍沎沑沒沕沖沗沘沚沜沝沞沠沢沨沬沯沰沴沵沶沷沺泀況泂泃泆泇泈泋泍泎泏泑泒泘"],["9b80","泙泚泜泝泟泤泦泧泩泬泭泲泴泹泿洀洂洃洅洆洈洉洊洍洏洐洑洓洔洕洖洘洜洝洟",5,"洦洨洩洬洭洯洰洴洶洷洸洺洿浀浂浄浉浌浐浕浖浗浘浛浝浟浡浢浤浥浧浨浫浬浭浰浱浲浳浵浶浹浺浻浽",4,"涃涄涆涇涊涋涍涏涐涒涖",4,"涜涢涥涬涭涰涱涳涴涶涷涹",5,"淁淂淃淈淉淊"],["9c40","淍淎淏淐淒淓淔淕淗淚淛淜淟淢淣淥淧淨淩淪淭淯淰淲淴淵淶淸淺淽",7,"渆渇済渉渋渏渒渓渕渘渙減渜渞渟渢渦渧渨渪測渮渰渱渳渵"],["9c80","渶渷渹渻",7,"湅",7,"湏湐湑湒湕湗湙湚湜湝湞湠",10,"湬湭湯",14,"満溁溂溄溇溈溊",4,"溑",6,"溙溚溛溝溞溠溡溣溤溦溨溩溫溬溭溮溰溳溵溸溹溼溾溿滀滃滄滅滆滈滉滊滌滍滎滐滒滖滘滙滛滜滝滣滧滪",5],["9d40","滰滱滲滳滵滶滷滸滺",7,"漃漄漅漇漈漊",4,"漐漑漒漖",9,"漡漢漣漥漦漧漨漬漮漰漲漴漵漷",6,"漿潀潁潂"],["9d80","潃潄潅潈潉潊潌潎",9,"潙潚潛潝潟潠潡潣潤潥潧",5,"潯潰潱潳潵潶潷潹潻潽",6,"澅澆澇澊澋澏",12,"澝澞澟澠澢",4,"澨",10,"澴澵澷澸澺",5,"濁濃",5,"濊",6,"濓",10,"濟濢濣濤濥"],["9e40","濦",7,"濰",32,"瀒",7,"瀜",6,"瀤",6],["9e80","瀫",9,"瀶瀷瀸瀺",17,"灍灎灐",13,"灟",11,"灮灱灲灳灴灷灹灺灻災炁炂炃炄炆炇炈炋炌炍炏炐炑炓炗炘炚炛炞",12,"炰炲炴炵炶為炾炿烄烅烆烇烉烋",12,"烚"],["9f40","烜烝烞烠烡烢烣烥烪烮烰",6,"烸烺烻烼烾",10,"焋",4,"焑焒焔焗焛",10,"焧",7,"焲焳焴"],["9f80","焵焷",13,"煆煇煈煉煋煍煏",12,"煝煟",4,"煥煩",4,"煯煰煱煴煵煶煷煹煻煼煾",5,"熅",4,"熋熌熍熎熐熑熒熓熕熖熗熚",4,"熡",6,"熩熪熫熭",5,"熴熶熷熸熺",8,"燄",9,"燏",4],["a040","燖",9,"燡燢燣燤燦燨",5,"燯",9,"燺",11,"爇",19],["a080","爛爜爞",9,"爩爫爭爮爯爲爳爴爺爼爾牀",6,"牉牊牋牎牏牐牑牓牔牕牗牘牚牜牞牠牣牤牥牨牪牫牬牭牰牱牳牴牶牷牸牻牼牽犂犃犅",4,"犌犎犐犑犓",11,"犠",11,"犮犱犲犳犵犺",6,"狅狆狇狉狊狋狌狏狑狓狔狕狖狘狚狛"],["a1a1","　、。·ˉˇ¨〃々—～‖…‘’“”〔〕〈",7,"〖〗【】±×÷∶∧∨∑∏∪∩∈∷√⊥∥∠⌒⊙∫∮≡≌≈∽∝≠≮≯≤≥∞∵∴♂♀°′″℃＄¤￠￡‰§№☆★○●◎◇◆□■△▲※→←↑↓〓"],["a2a1","ⅰ",9],["a2b1","⒈",19,"⑴",19,"①",9],["a2e5","㈠",9],["a2f1","Ⅰ",11],["a3a1","！＂＃￥％",88,"￣"],["a4a1","ぁ",82],["a5a1","ァ",85],["a6a1","Α",16,"Σ",6],["a6c1","α",16,"σ",6],["a6e0","︵︶︹︺︿﹀︽︾﹁﹂﹃﹄"],["a6ee","︻︼︷︸︱"],["a6f4","︳︴"],["a7a1","А",5,"ЁЖ",25],["a7d1","а",5,"ёж",25],["a840","ˊˋ˙–―‥‵℅℉↖↗↘↙∕∟∣≒≦≧⊿═",35,"▁",6],["a880","█",7,"▓▔▕▼▽◢◣◤◥☉⊕〒〝〞"],["a8a1","āáǎàēéěèīíǐìōóǒòūúǔùǖǘǚǜüêɑ"],["a8bd","ńň"],["a8c0","ɡ"],["a8c5","ㄅ",36],["a940","〡",8,"㊣㎎㎏㎜㎝㎞㎡㏄㏎㏑㏒㏕︰￢￤"],["a959","℡㈱"],["a95c","‐"],["a960","ー゛゜ヽヾ〆ゝゞ﹉",9,"﹔﹕﹖﹗﹙",8],["a980","﹢",4,"﹨﹩﹪﹫"],["a996","〇"],["a9a4","─",75],["aa40","狜狝狟狢",5,"狪狫狵狶狹狽狾狿猀猂猄",5,"猋猌猍猏猐猑猒猔猘猙猚猟猠猣猤猦猧猨猭猯猰猲猳猵猶猺猻猼猽獀",8],["aa80","獉獊獋獌獎獏獑獓獔獕獖獘",7,"獡",10,"獮獰獱"],["ab40","獲",11,"獿",4,"玅玆玈玊玌玍玏玐玒玓玔玕玗玘玙玚玜玝玞玠玡玣",5,"玪玬玭玱玴玵玶玸玹玼玽玾玿珁珃",4],["ab80","珋珌珎珒",6,"珚珛珜珝珟珡珢珣珤珦珨珪珫珬珮珯珰珱珳",4],["ac40","珸",10,"琄琇琈琋琌琍琎琑",8,"琜",5,"琣琤琧琩琫琭琯琱琲琷",4,"琽琾琿瑀瑂",11],["ac80","瑎",6,"瑖瑘瑝瑠",12,"瑮瑯瑱",4,"瑸瑹瑺"],["ad40","瑻瑼瑽瑿璂璄璅璆璈璉璊璌璍璏璑",10,"璝璟",7,"璪",15,"璻",12],["ad80","瓈",9,"瓓",8,"瓝瓟瓡瓥瓧",6,"瓰瓱瓲"],["ae40","瓳瓵瓸",6,"甀甁甂甃甅",7,"甎甐甒甔甕甖甗甛甝甞甠",4,"甦甧甪甮甴甶甹甼甽甿畁畂畃畄畆畇畉畊畍畐畑畒畓畕畖畗畘"],["ae80","畝",7,"畧畨畩畫",6,"畳畵當畷畺",4,"疀疁疂疄疅疇"],["af40","疈疉疊疌疍疎疐疓疕疘疛疜疞疢疦",4,"疭疶疷疺疻疿痀痁痆痋痌痎痏痐痑痓痗痙痚痜痝痟痠痡痥痩痬痭痮痯痲痳痵痶痷痸痺痻痽痾瘂瘄瘆瘇"],["af80","瘈瘉瘋瘍瘎瘏瘑瘒瘓瘔瘖瘚瘜瘝瘞瘡瘣瘧瘨瘬瘮瘯瘱瘲瘶瘷瘹瘺瘻瘽癁療癄"],["b040","癅",6,"癎",5,"癕癗",4,"癝癟癠癡癢癤",6,"癬癭癮癰",7,"癹発發癿皀皁皃皅皉皊皌皍皏皐皒皔皕皗皘皚皛"],["b080","皜",7,"皥",8,"皯皰皳皵",9,"盀盁盃啊阿埃挨哎唉哀皑癌蔼矮艾碍爱隘鞍氨安俺按暗岸胺案肮昂盎凹敖熬翱袄傲奥懊澳芭捌扒叭吧笆八疤巴拔跋靶把耙坝霸罢爸白柏百摆佰败拜稗斑班搬扳般颁板版扮拌伴瓣半办绊邦帮梆榜膀绑棒磅蚌镑傍谤苞胞包褒剥"],["b140","盄盇盉盋盌盓盕盙盚盜盝盞盠",4,"盦",7,"盰盳盵盶盷盺盻盽盿眀眂眃眅眆眊県眎",10,"眛眜眝眞眡眣眤眥眧眪眫"],["b180","眬眮眰",4,"眹眻眽眾眿睂睄睅睆睈",7,"睒",7,"睜薄雹保堡饱宝抱报暴豹鲍爆杯碑悲卑北辈背贝钡倍狈备惫焙被奔苯本笨崩绷甭泵蹦迸逼鼻比鄙笔彼碧蓖蔽毕毙毖币庇痹闭敝弊必辟壁臂避陛鞭边编贬扁便变卞辨辩辫遍标彪膘表鳖憋别瘪彬斌濒滨宾摈兵冰柄丙秉饼炳"],["b240","睝睞睟睠睤睧睩睪睭",11,"睺睻睼瞁瞂瞃瞆",5,"瞏瞐瞓",11,"瞡瞣瞤瞦瞨瞫瞭瞮瞯瞱瞲瞴瞶",4],["b280","瞼瞾矀",12,"矎",8,"矘矙矚矝",4,"矤病并玻菠播拨钵波博勃搏铂箔伯帛舶脖膊渤泊驳捕卜哺补埠不布步簿部怖擦猜裁材才财睬踩采彩菜蔡餐参蚕残惭惨灿苍舱仓沧藏操糙槽曹草厕策侧册测层蹭插叉茬茶查碴搽察岔差诧拆柴豺搀掺蝉馋谗缠铲产阐颤昌猖"],["b340","矦矨矪矯矰矱矲矴矵矷矹矺矻矼砃",5,"砊砋砎砏砐砓砕砙砛砞砠砡砢砤砨砪砫砮砯砱砲砳砵砶砽砿硁硂硃硄硆硈硉硊硋硍硏硑硓硔硘硙硚"],["b380","硛硜硞",11,"硯",7,"硸硹硺硻硽",6,"场尝常长偿肠厂敞畅唱倡超抄钞朝嘲潮巢吵炒车扯撤掣彻澈郴臣辰尘晨忱沉陈趁衬撑称城橙成呈乘程惩澄诚承逞骋秤吃痴持匙池迟弛驰耻齿侈尺赤翅斥炽充冲虫崇宠抽酬畴踌稠愁筹仇绸瞅丑臭初出橱厨躇锄雏滁除楚"],["b440","碄碅碆碈碊碋碏碐碒碔碕碖碙碝碞碠碢碤碦碨",7,"碵碶碷碸確碻碼碽碿磀磂磃磄磆磇磈磌磍磎磏磑磒磓磖磗磘磚",9],["b480","磤磥磦磧磩磪磫磭",4,"磳磵磶磸磹磻",5,"礂礃礄礆",6,"础储矗搐触处揣川穿椽传船喘串疮窗幢床闯创吹炊捶锤垂春椿醇唇淳纯蠢戳绰疵茨磁雌辞慈瓷词此刺赐次聪葱囱匆从丛凑粗醋簇促蹿篡窜摧崔催脆瘁粹淬翠村存寸磋撮搓措挫错搭达答瘩打大呆歹傣戴带殆代贷袋待逮"],["b540","礍",5,"礔",9,"礟",4,"礥",14,"礵",4,"礽礿祂祃祄祅祇祊",8,"祔祕祘祙祡祣"],["b580","祤祦祩祪祫祬祮祰",6,"祹祻",4,"禂禃禆禇禈禉禋禌禍禎禐禑禒怠耽担丹单郸掸胆旦氮但惮淡诞弹蛋当挡党荡档刀捣蹈倒岛祷导到稻悼道盗德得的蹬灯登等瞪凳邓堤低滴迪敌笛狄涤翟嫡抵底地蒂第帝弟递缔颠掂滇碘点典靛垫电佃甸店惦奠淀殿碉叼雕凋刁掉吊钓调跌爹碟蝶迭谍叠"],["b640","禓",6,"禛",11,"禨",10,"禴",4,"禼禿秂秄秅秇秈秊秌秎秏秐秓秔秖秗秙",5,"秠秡秢秥秨秪"],["b680","秬秮秱",6,"秹秺秼秾秿稁稄稅稇稈稉稊稌稏",4,"稕稖稘稙稛稜丁盯叮钉顶鼎锭定订丢东冬董懂动栋侗恫冻洞兜抖斗陡豆逗痘都督毒犊独读堵睹赌杜镀肚度渡妒端短锻段断缎堆兑队对墩吨蹲敦顿囤钝盾遁掇哆多夺垛躲朵跺舵剁惰堕蛾峨鹅俄额讹娥恶厄扼遏鄂饿恩而儿耳尔饵洱二"],["b740","稝稟稡稢稤",14,"稴稵稶稸稺稾穀",5,"穇",9,"穒",4,"穘",16],["b780","穩",6,"穱穲穳穵穻穼穽穾窂窅窇窉窊窋窌窎窏窐窓窔窙窚窛窞窡窢贰发罚筏伐乏阀法珐藩帆番翻樊矾钒繁凡烦反返范贩犯饭泛坊芳方肪房防妨仿访纺放菲非啡飞肥匪诽吠肺废沸费芬酚吩氛分纷坟焚汾粉奋份忿愤粪丰封枫蜂峰锋风疯烽逢冯缝讽奉凤佛否夫敷肤孵扶拂辐幅氟符伏俘服"],["b840","窣窤窧窩窪窫窮",4,"窴",10,"竀",10,"竌",9,"竗竘竚竛竜竝竡竢竤竧",5,"竮竰竱竲竳"],["b880","竴",4,"竻竼竾笀笁笂笅笇笉笌笍笎笐笒笓笖笗笘笚笜笝笟笡笢笣笧笩笭浮涪福袱弗甫抚辅俯釜斧脯腑府腐赴副覆赋复傅付阜父腹负富讣附妇缚咐噶嘎该改概钙盖溉干甘杆柑竿肝赶感秆敢赣冈刚钢缸肛纲岗港杠篙皋高膏羔糕搞镐稿告哥歌搁戈鸽胳疙割革葛格蛤阁隔铬个各给根跟耕更庚羹"],["b940","笯笰笲笴笵笶笷笹笻笽笿",5,"筆筈筊筍筎筓筕筗筙筜筞筟筡筣",10,"筯筰筳筴筶筸筺筼筽筿箁箂箃箄箆",6,"箎箏"],["b980","箑箒箓箖箘箙箚箛箞箟箠箣箤箥箮箯箰箲箳箵箶箷箹",7,"篂篃範埂耿梗工攻功恭龚供躬公宫弓巩汞拱贡共钩勾沟苟狗垢构购够辜菇咕箍估沽孤姑鼓古蛊骨谷股故顾固雇刮瓜剐寡挂褂乖拐怪棺关官冠观管馆罐惯灌贯光广逛瑰规圭硅归龟闺轨鬼诡癸桂柜跪贵刽辊滚棍锅郭国果裹过哈"],["ba40","篅篈築篊篋篍篎篏篐篒篔",4,"篛篜篞篟篠篢篣篤篧篨篩篫篬篭篯篰篲",4,"篸篹篺篻篽篿",7,"簈簉簊簍簎簐",5,"簗簘簙"],["ba80","簚",4,"簠",5,"簨簩簫",12,"簹",5,"籂骸孩海氦亥害骇酣憨邯韩含涵寒函喊罕翰撼捍旱憾悍焊汗汉夯杭航壕嚎豪毫郝好耗号浩呵喝荷菏核禾和何合盒貉阂河涸赫褐鹤贺嘿黑痕很狠恨哼亨横衡恒轰哄烘虹鸿洪宏弘红喉侯猴吼厚候后呼乎忽瑚壶葫胡蝴狐糊湖"],["bb40","籃",9,"籎",36,"籵",5,"籾",9],["bb80","粈粊",6,"粓粔粖粙粚粛粠粡粣粦粧粨粩粫粬粭粯粰粴",4,"粺粻弧虎唬护互沪户花哗华猾滑画划化话槐徊怀淮坏欢环桓还缓换患唤痪豢焕涣宦幻荒慌黄磺蝗簧皇凰惶煌晃幌恍谎灰挥辉徽恢蛔回毁悔慧卉惠晦贿秽会烩汇讳诲绘荤昏婚魂浑混豁活伙火获或惑霍货祸击圾基机畸稽积箕"],["bc40","粿糀糂糃糄糆糉糋糎",6,"糘糚糛糝糞糡",6,"糩",5,"糰",7,"糹糺糼",13,"紋",5],["bc80","紑",14,"紡紣紤紥紦紨紩紪紬紭紮細",6,"肌饥迹激讥鸡姬绩缉吉极棘辑籍集及急疾汲即嫉级挤几脊己蓟技冀季伎祭剂悸济寄寂计记既忌际妓继纪嘉枷夹佳家加荚颊贾甲钾假稼价架驾嫁歼监坚尖笺间煎兼肩艰奸缄茧检柬碱硷拣捡简俭剪减荐槛鉴践贱见键箭件"],["bd40","紷",54,"絯",7],["bd80","絸",32,"健舰剑饯渐溅涧建僵姜将浆江疆蒋桨奖讲匠酱降蕉椒礁焦胶交郊浇骄娇嚼搅铰矫侥脚狡角饺缴绞剿教酵轿较叫窖揭接皆秸街阶截劫节桔杰捷睫竭洁结解姐戒藉芥界借介疥诫届巾筋斤金今津襟紧锦仅谨进靳晋禁近烬浸"],["be40","継",12,"綧",6,"綯",42],["be80","線",32,"尽劲荆兢茎睛晶鲸京惊精粳经井警景颈静境敬镜径痉靖竟竞净炯窘揪究纠玖韭久灸九酒厩救旧臼舅咎就疚鞠拘狙疽居驹菊局咀矩举沮聚拒据巨具距踞锯俱句惧炬剧捐鹃娟倦眷卷绢撅攫抉掘倔爵觉决诀绝均菌钧军君峻"],["bf40","緻",62],["bf80","縺縼",4,"繂",4,"繈",21,"俊竣浚郡骏喀咖卡咯开揩楷凯慨刊堪勘坎砍看康慷糠扛抗亢炕考拷烤靠坷苛柯棵磕颗科壳咳可渴克刻客课肯啃垦恳坑吭空恐孔控抠口扣寇枯哭窟苦酷库裤夸垮挎跨胯块筷侩快宽款匡筐狂框矿眶旷况亏盔岿窥葵奎魁傀"],["c040","繞",35,"纃",23,"纜纝纞"],["c080","纮纴纻纼绖绤绬绹缊缐缞缷缹缻",6,"罃罆",9,"罒罓馈愧溃坤昆捆困括扩廓阔垃拉喇蜡腊辣啦莱来赖蓝婪栏拦篮阑兰澜谰揽览懒缆烂滥琅榔狼廊郎朗浪捞劳牢老佬姥酪烙涝勒乐雷镭蕾磊累儡垒擂肋类泪棱楞冷厘梨犁黎篱狸离漓理李里鲤礼莉荔吏栗丽厉励砾历利傈例俐"],["c140","罖罙罛罜罝罞罠罣",4,"罫罬罭罯罰罳罵罶罷罸罺罻罼罽罿羀羂",7,"羋羍羏",4,"羕",4,"羛羜羠羢羣羥羦羨",6,"羱"],["c180","羳",4,"羺羻羾翀翂翃翄翆翇翈翉翋翍翏",4,"翖翗翙",5,"翢翣痢立粒沥隶力璃哩俩联莲连镰廉怜涟帘敛脸链恋炼练粮凉梁粱良两辆量晾亮谅撩聊僚疗燎寥辽潦了撂镣廖料列裂烈劣猎琳林磷霖临邻鳞淋凛赁吝拎玲菱零龄铃伶羚凌灵陵岭领另令溜琉榴硫馏留刘瘤流柳六龙聋咙笼窿"],["c240","翤翧翨翪翫翬翭翯翲翴",6,"翽翾翿耂耇耈耉耊耎耏耑耓耚耛耝耞耟耡耣耤耫",5,"耲耴耹耺耼耾聀聁聄聅聇聈聉聎聏聐聑聓聕聖聗"],["c280","聙聛",13,"聫",5,"聲",11,"隆垄拢陇楼娄搂篓漏陋芦卢颅庐炉掳卤虏鲁麓碌露路赂鹿潞禄录陆戮驴吕铝侣旅履屡缕虑氯律率滤绿峦挛孪滦卵乱掠略抡轮伦仑沦纶论萝螺罗逻锣箩骡裸落洛骆络妈麻玛码蚂马骂嘛吗埋买麦卖迈脉瞒馒蛮满蔓曼慢漫"],["c340","聾肁肂肅肈肊肍",5,"肔肕肗肙肞肣肦肧肨肬肰肳肵肶肸肹肻胅胇",4,"胏",6,"胘胟胠胢胣胦胮胵胷胹胻胾胿脀脁脃脄脅脇脈脋"],["c380","脌脕脗脙脛脜脝脟",12,"脭脮脰脳脴脵脷脹",4,"脿谩芒茫盲氓忙莽猫茅锚毛矛铆卯茂冒帽貌贸么玫枚梅酶霉煤没眉媒镁每美昧寐妹媚门闷们萌蒙檬盟锰猛梦孟眯醚靡糜迷谜弥米秘觅泌蜜密幂棉眠绵冕免勉娩缅面苗描瞄藐秒渺庙妙蔑灭民抿皿敏悯闽明螟鸣铭名命谬摸"],["c440","腀",5,"腇腉腍腎腏腒腖腗腘腛",4,"腡腢腣腤腦腨腪腫腬腯腲腳腵腶腷腸膁膃",4,"膉膋膌膍膎膐膒",5,"膙膚膞",4,"膤膥"],["c480","膧膩膫",7,"膴",5,"膼膽膾膿臄臅臇臈臉臋臍",6,"摹蘑模膜磨摩魔抹末莫墨默沫漠寞陌谋牟某拇牡亩姆母墓暮幕募慕木目睦牧穆拿哪呐钠那娜纳氖乃奶耐奈南男难囊挠脑恼闹淖呢馁内嫩能妮霓倪泥尼拟你匿腻逆溺蔫拈年碾撵捻念娘酿鸟尿捏聂孽啮镊镍涅您柠狞凝宁"],["c540","臔",14,"臤臥臦臨臩臫臮",4,"臵",5,"臽臿舃與",4,"舎舏舑舓舕",5,"舝舠舤舥舦舧舩舮舲舺舼舽舿"],["c580","艀艁艂艃艅艆艈艊艌艍艎艐",7,"艙艛艜艝艞艠",7,"艩拧泞牛扭钮纽脓浓农弄奴努怒女暖虐疟挪懦糯诺哦欧鸥殴藕呕偶沤啪趴爬帕怕琶拍排牌徘湃派攀潘盘磐盼畔判叛乓庞旁耪胖抛咆刨炮袍跑泡呸胚培裴赔陪配佩沛喷盆砰抨烹澎彭蓬棚硼篷膨朋鹏捧碰坯砒霹批披劈琵毗"],["c640","艪艫艬艭艱艵艶艷艸艻艼芀芁芃芅芆芇芉芌芐芓芔芕芖芚芛芞芠芢芣芧芲芵芶芺芻芼芿苀苂苃苅苆苉苐苖苙苚苝苢苧苨苩苪苬苭苮苰苲苳苵苶苸"],["c680","苺苼",4,"茊茋茍茐茒茓茖茘茙茝",9,"茩茪茮茰茲茷茻茽啤脾疲皮匹痞僻屁譬篇偏片骗飘漂瓢票撇瞥拼频贫品聘乒坪苹萍平凭瓶评屏坡泼颇婆破魄迫粕剖扑铺仆莆葡菩蒲埔朴圃普浦谱曝瀑期欺栖戚妻七凄漆柒沏其棋奇歧畦崎脐齐旗祈祁骑起岂乞企启契砌器气迄弃汽泣讫掐"],["c740","茾茿荁荂荄荅荈荊",4,"荓荕",4,"荝荢荰",6,"荹荺荾",6,"莇莈莊莋莌莍莏莐莑莔莕莖莗莙莚莝莟莡",6,"莬莭莮"],["c780","莯莵莻莾莿菂菃菄菆菈菉菋菍菎菐菑菒菓菕菗菙菚菛菞菢菣菤菦菧菨菫菬菭恰洽牵扦钎铅千迁签仟谦乾黔钱钳前潜遣浅谴堑嵌欠歉枪呛腔羌墙蔷强抢橇锹敲悄桥瞧乔侨巧鞘撬翘峭俏窍切茄且怯窃钦侵亲秦琴勤芹擒禽寝沁青轻氢倾卿清擎晴氰情顷请庆琼穷秋丘邱球求囚酋泅趋区蛆曲躯屈驱渠"],["c840","菮華菳",4,"菺菻菼菾菿萀萂萅萇萈萉萊萐萒",5,"萙萚萛萞",5,"萩",7,"萲",5,"萹萺萻萾",7,"葇葈葉"],["c880","葊",6,"葒",4,"葘葝葞葟葠葢葤",4,"葪葮葯葰葲葴葷葹葻葼取娶龋趣去圈颧权醛泉全痊拳犬券劝缺炔瘸却鹊榷确雀裙群然燃冉染瓤壤攘嚷让饶扰绕惹热壬仁人忍韧任认刃妊纫扔仍日戎茸蓉荣融熔溶容绒冗揉柔肉茹蠕儒孺如辱乳汝入褥软阮蕊瑞锐闰润若弱撒洒萨腮鳃塞赛三叁"],["c940","葽",4,"蒃蒄蒅蒆蒊蒍蒏",7,"蒘蒚蒛蒝蒞蒟蒠蒢",12,"蒰蒱蒳蒵蒶蒷蒻蒼蒾蓀蓂蓃蓅蓆蓇蓈蓋蓌蓎蓏蓒蓔蓕蓗"],["c980","蓘",4,"蓞蓡蓢蓤蓧",4,"蓭蓮蓯蓱",10,"蓽蓾蔀蔁蔂伞散桑嗓丧搔骚扫嫂瑟色涩森僧莎砂杀刹沙纱傻啥煞筛晒珊苫杉山删煽衫闪陕擅赡膳善汕扇缮墒伤商赏晌上尚裳梢捎稍烧芍勺韶少哨邵绍奢赊蛇舌舍赦摄射慑涉社设砷申呻伸身深娠绅神沈审婶甚肾慎渗声生甥牲升绳"],["ca40","蔃",8,"蔍蔎蔏蔐蔒蔔蔕蔖蔘蔙蔛蔜蔝蔞蔠蔢",8,"蔭",9,"蔾",4,"蕄蕅蕆蕇蕋",10],["ca80","蕗蕘蕚蕛蕜蕝蕟",4,"蕥蕦蕧蕩",8,"蕳蕵蕶蕷蕸蕼蕽蕿薀薁省盛剩胜圣师失狮施湿诗尸虱十石拾时什食蚀实识史矢使屎驶始式示士世柿事拭誓逝势是嗜噬适仕侍释饰氏市恃室视试收手首守寿授售受瘦兽蔬枢梳殊抒输叔舒淑疏书赎孰熟薯暑曙署蜀黍鼠属术述树束戍竖墅庶数漱"],["cb40","薂薃薆薈",6,"薐",10,"薝",6,"薥薦薧薩薫薬薭薱",5,"薸薺",6,"藂",6,"藊",4,"藑藒"],["cb80","藔藖",5,"藝",6,"藥藦藧藨藪",14,"恕刷耍摔衰甩帅栓拴霜双爽谁水睡税吮瞬顺舜说硕朔烁斯撕嘶思私司丝死肆寺嗣四伺似饲巳松耸怂颂送宋讼诵搜艘擞嗽苏酥俗素速粟僳塑溯宿诉肃酸蒜算虽隋随绥髓碎岁穗遂隧祟孙损笋蓑梭唆缩琐索锁所塌他它她塔"],["cc40","藹藺藼藽藾蘀",4,"蘆",10,"蘒蘓蘔蘕蘗",15,"蘨蘪",13,"蘹蘺蘻蘽蘾蘿虀"],["cc80","虁",11,"虒虓處",4,"虛虜虝號虠虡虣",7,"獭挞蹋踏胎苔抬台泰酞太态汰坍摊贪瘫滩坛檀痰潭谭谈坦毯袒碳探叹炭汤塘搪堂棠膛唐糖倘躺淌趟烫掏涛滔绦萄桃逃淘陶讨套特藤腾疼誊梯剔踢锑提题蹄啼体替嚏惕涕剃屉天添填田甜恬舔腆挑条迢眺跳贴铁帖厅听烃"],["cd40","虭虯虰虲",6,"蚃",6,"蚎",4,"蚔蚖",5,"蚞",4,"蚥蚦蚫蚭蚮蚲蚳蚷蚸蚹蚻",4,"蛁蛂蛃蛅蛈蛌蛍蛒蛓蛕蛖蛗蛚蛜"],["cd80","蛝蛠蛡蛢蛣蛥蛦蛧蛨蛪蛫蛬蛯蛵蛶蛷蛺蛻蛼蛽蛿蜁蜄蜅蜆蜋蜌蜎蜏蜐蜑蜔蜖汀廷停亭庭挺艇通桐酮瞳同铜彤童桶捅筒统痛偷投头透凸秃突图徒途涂屠土吐兔湍团推颓腿蜕褪退吞屯臀拖托脱鸵陀驮驼椭妥拓唾挖哇蛙洼娃瓦袜歪外豌弯湾玩顽丸烷完碗挽晚皖惋宛婉万腕汪王亡枉网往旺望忘妄威"],["ce40","蜙蜛蜝蜟蜠蜤蜦蜧蜨蜪蜫蜬蜭蜯蜰蜲蜳蜵蜶蜸蜹蜺蜼蜽蝀",6,"蝊蝋蝍蝏蝐蝑蝒蝔蝕蝖蝘蝚",5,"蝡蝢蝦",7,"蝯蝱蝲蝳蝵"],["ce80","蝷蝸蝹蝺蝿螀螁螄螆螇螉螊螌螎",4,"螔螕螖螘",6,"螠",4,"巍微危韦违桅围唯惟为潍维苇萎委伟伪尾纬未蔚味畏胃喂魏位渭谓尉慰卫瘟温蚊文闻纹吻稳紊问嗡翁瓮挝蜗涡窝我斡卧握沃巫呜钨乌污诬屋无芜梧吾吴毋武五捂午舞伍侮坞戊雾晤物勿务悟误昔熙析西硒矽晰嘻吸锡牺"],["cf40","螥螦螧螩螪螮螰螱螲螴螶螷螸螹螻螼螾螿蟁",4,"蟇蟈蟉蟌",4,"蟔",6,"蟜蟝蟞蟟蟡蟢蟣蟤蟦蟧蟨蟩蟫蟬蟭蟯",9],["cf80","蟺蟻蟼蟽蟿蠀蠁蠂蠄",5,"蠋",7,"蠔蠗蠘蠙蠚蠜",4,"蠣稀息希悉膝夕惜熄烯溪汐犀檄袭席习媳喜铣洗系隙戏细瞎虾匣霞辖暇峡侠狭下厦夏吓掀锨先仙鲜纤咸贤衔舷闲涎弦嫌显险现献县腺馅羡宪陷限线相厢镶香箱襄湘乡翔祥详想响享项巷橡像向象萧硝霄削哮嚣销消宵淆晓"],["d040","蠤",13,"蠳",5,"蠺蠻蠽蠾蠿衁衂衃衆",5,"衎",5,"衕衖衘衚",6,"衦衧衪衭衯衱衳衴衵衶衸衹衺"],["d080","衻衼袀袃袆袇袉袊袌袎袏袐袑袓袔袕袗",4,"袝",4,"袣袥",5,"小孝校肖啸笑效楔些歇蝎鞋协挟携邪斜胁谐写械卸蟹懈泄泻谢屑薪芯锌欣辛新忻心信衅星腥猩惺兴刑型形邢行醒幸杏性姓兄凶胸匈汹雄熊休修羞朽嗅锈秀袖绣墟戌需虚嘘须徐许蓄酗叙旭序畜恤絮婿绪续轩喧宣悬旋玄"],["d140","袬袮袯袰袲",4,"袸袹袺袻袽袾袿裀裃裄裇裈裊裋裌裍裏裐裑裓裖裗裚",4,"裠裡裦裧裩",6,"裲裵裶裷裺裻製裿褀褁褃",5],["d180","褉褋",4,"褑褔",4,"褜",4,"褢褣褤褦褧褨褩褬褭褮褯褱褲褳褵褷选癣眩绚靴薛学穴雪血勋熏循旬询寻驯巡殉汛训讯逊迅压押鸦鸭呀丫芽牙蚜崖衙涯雅哑亚讶焉咽阉烟淹盐严研蜒岩延言颜阎炎沿奄掩眼衍演艳堰燕厌砚雁唁彦焰宴谚验殃央鸯秧杨扬佯疡羊洋阳氧仰痒养样漾邀腰妖瑶"],["d240","褸",8,"襂襃襅",24,"襠",5,"襧",19,"襼"],["d280","襽襾覀覂覄覅覇",26,"摇尧遥窑谣姚咬舀药要耀椰噎耶爷野冶也页掖业叶曳腋夜液一壹医揖铱依伊衣颐夷遗移仪胰疑沂宜姨彝椅蚁倚已乙矣以艺抑易邑屹亿役臆逸肄疫亦裔意毅忆义益溢诣议谊译异翼翌绎茵荫因殷音阴姻吟银淫寅饮尹引隐"],["d340","覢",30,"觃觍觓觔觕觗觘觙觛觝觟觠觡觢觤觧觨觩觪觬觭觮觰觱觲觴",6],["d380","觻",4,"訁",5,"計",21,"印英樱婴鹰应缨莹萤营荧蝇迎赢盈影颖硬映哟拥佣臃痈庸雍踊蛹咏泳涌永恿勇用幽优悠忧尤由邮铀犹油游酉有友右佑釉诱又幼迂淤于盂榆虞愚舆余俞逾鱼愉渝渔隅予娱雨与屿禹宇语羽玉域芋郁吁遇喻峪御愈欲狱育誉"],["d440","訞",31,"訿",8,"詉",21],["d480","詟",25,"詺",6,"浴寓裕预豫驭鸳渊冤元垣袁原援辕园员圆猿源缘远苑愿怨院曰约越跃钥岳粤月悦阅耘云郧匀陨允运蕴酝晕韵孕匝砸杂栽哉灾宰载再在咱攒暂赞赃脏葬遭糟凿藻枣早澡蚤躁噪造皂灶燥责择则泽贼怎增憎曾赠扎喳渣札轧"],["d540","誁",7,"誋",7,"誔",46],["d580","諃",32,"铡闸眨栅榨咋乍炸诈摘斋宅窄债寨瞻毡詹粘沾盏斩辗崭展蘸栈占战站湛绽樟章彰漳张掌涨杖丈帐账仗胀瘴障招昭找沼赵照罩兆肇召遮折哲蛰辙者锗蔗这浙珍斟真甄砧臻贞针侦枕疹诊震振镇阵蒸挣睁征狰争怔整拯正政"],["d640","諤",34,"謈",27],["d680","謤謥謧",30,"帧症郑证芝枝支吱蜘知肢脂汁之织职直植殖执值侄址指止趾只旨纸志挚掷至致置帜峙制智秩稚质炙痔滞治窒中盅忠钟衷终种肿重仲众舟周州洲诌粥轴肘帚咒皱宙昼骤珠株蛛朱猪诸诛逐竹烛煮拄瞩嘱主著柱助蛀贮铸筑"],["d740","譆",31,"譧",4,"譭",25],["d780","讇",24,"讬讱讻诇诐诪谉谞住注祝驻抓爪拽专砖转撰赚篆桩庄装妆撞壮状椎锥追赘坠缀谆准捉拙卓桌琢茁酌啄着灼浊兹咨资姿滋淄孜紫仔籽滓子自渍字鬃棕踪宗综总纵邹走奏揍租足卒族祖诅阻组钻纂嘴醉最罪尊遵昨左佐柞做作坐座"],["d840","谸",8,"豂豃豄豅豈豊豋豍",7,"豖豗豘豙豛",5,"豣",6,"豬",6,"豴豵豶豷豻",6,"貃貄貆貇"],["d880","貈貋貍",6,"貕貖貗貙",20,"亍丌兀丐廿卅丕亘丞鬲孬噩丨禺丿匕乇夭爻卮氐囟胤馗毓睾鼗丶亟鼐乜乩亓芈孛啬嘏仄厍厝厣厥厮靥赝匚叵匦匮匾赜卦卣刂刈刎刭刳刿剀剌剞剡剜蒯剽劂劁劐劓冂罔亻仃仉仂仨仡仫仞伛仳伢佤仵伥伧伉伫佞佧攸佚佝"],["d940","貮",62],["d980","賭",32,"佟佗伲伽佶佴侑侉侃侏佾佻侪佼侬侔俦俨俪俅俚俣俜俑俟俸倩偌俳倬倏倮倭俾倜倌倥倨偾偃偕偈偎偬偻傥傧傩傺僖儆僭僬僦僮儇儋仝氽佘佥俎龠汆籴兮巽黉馘冁夔勹匍訇匐凫夙兕亠兖亳衮袤亵脔裒禀嬴蠃羸冫冱冽冼"],["da40","贎",14,"贠赑赒赗赟赥赨赩赪赬赮赯赱赲赸",8,"趂趃趆趇趈趉趌",4,"趒趓趕",9,"趠趡"],["da80","趢趤",12,"趲趶趷趹趻趽跀跁跂跅跇跈跉跊跍跐跒跓跔凇冖冢冥讠讦讧讪讴讵讷诂诃诋诏诎诒诓诔诖诘诙诜诟诠诤诨诩诮诰诳诶诹诼诿谀谂谄谇谌谏谑谒谔谕谖谙谛谘谝谟谠谡谥谧谪谫谮谯谲谳谵谶卩卺阝阢阡阱阪阽阼陂陉陔陟陧陬陲陴隈隍隗隰邗邛邝邙邬邡邴邳邶邺"],["db40","跕跘跙跜跠跡跢跥跦跧跩跭跮跰跱跲跴跶跼跾",6,"踆踇踈踋踍踎踐踑踒踓踕",7,"踠踡踤",4,"踫踭踰踲踳踴踶踷踸踻踼踾"],["db80","踿蹃蹅蹆蹌",4,"蹓",5,"蹚",11,"蹧蹨蹪蹫蹮蹱邸邰郏郅邾郐郄郇郓郦郢郜郗郛郫郯郾鄄鄢鄞鄣鄱鄯鄹酃酆刍奂劢劬劭劾哿勐勖勰叟燮矍廴凵凼鬯厶弁畚巯坌垩垡塾墼壅壑圩圬圪圳圹圮圯坜圻坂坩垅坫垆坼坻坨坭坶坳垭垤垌垲埏垧垴垓垠埕埘埚埙埒垸埴埯埸埤埝"],["dc40","蹳蹵蹷",4,"蹽蹾躀躂躃躄躆躈",6,"躑躒躓躕",6,"躝躟",11,"躭躮躰躱躳",6,"躻",7],["dc80","軃",10,"軏",21,"堋堍埽埭堀堞堙塄堠塥塬墁墉墚墀馨鼙懿艹艽艿芏芊芨芄芎芑芗芙芫芸芾芰苈苊苣芘芷芮苋苌苁芩芴芡芪芟苄苎芤苡茉苷苤茏茇苜苴苒苘茌苻苓茑茚茆茔茕苠苕茜荑荛荜茈莒茼茴茱莛荞茯荏荇荃荟荀茗荠茭茺茳荦荥"],["dd40","軥",62],["dd80","輤",32,"荨茛荩荬荪荭荮莰荸莳莴莠莪莓莜莅荼莶莩荽莸荻莘莞莨莺莼菁萁菥菘堇萘萋菝菽菖萜萸萑萆菔菟萏萃菸菹菪菅菀萦菰菡葜葑葚葙葳蒇蒈葺蒉葸萼葆葩葶蒌蒎萱葭蓁蓍蓐蓦蒽蓓蓊蒿蒺蓠蒡蒹蒴蒗蓥蓣蔌甍蔸蓰蔹蔟蔺"],["de40","轅",32,"轪辀辌辒辝辠辡辢辤辥辦辧辪辬辭辮辯農辳辴辵辷辸辺辻込辿迀迃迆"],["de80","迉",4,"迏迒迖迗迚迠迡迣迧迬迯迱迲迴迵迶迺迻迼迾迿逇逈逌逎逓逕逘蕖蔻蓿蓼蕙蕈蕨蕤蕞蕺瞢蕃蕲蕻薤薨薇薏蕹薮薜薅薹薷薰藓藁藜藿蘧蘅蘩蘖蘼廾弈夼奁耷奕奚奘匏尢尥尬尴扌扪抟抻拊拚拗拮挢拶挹捋捃掭揶捱捺掎掴捭掬掊捩掮掼揲揸揠揿揄揞揎摒揆掾摅摁搋搛搠搌搦搡摞撄摭撖"],["df40","這逜連逤逥逧",5,"逰",4,"逷逹逺逽逿遀遃遅遆遈",4,"過達違遖遙遚遜",5,"遤遦遧適遪遫遬遯",4,"遶",6,"遾邁"],["df80","還邅邆邇邉邊邌",4,"邒邔邖邘邚邜邞邟邠邤邥邧邨邩邫邭邲邷邼邽邿郀摺撷撸撙撺擀擐擗擤擢攉攥攮弋忒甙弑卟叱叽叩叨叻吒吖吆呋呒呓呔呖呃吡呗呙吣吲咂咔呷呱呤咚咛咄呶呦咝哐咭哂咴哒咧咦哓哔呲咣哕咻咿哌哙哚哜咩咪咤哝哏哞唛哧唠哽唔哳唢唣唏唑唧唪啧喏喵啉啭啁啕唿啐唼"],["e040","郂郃郆郈郉郋郌郍郒郔郕郖郘郙郚郞郟郠郣郤郥郩郪郬郮郰郱郲郳郵郶郷郹郺郻郼郿鄀鄁鄃鄅",19,"鄚鄛鄜"],["e080","鄝鄟鄠鄡鄤",10,"鄰鄲",6,"鄺",8,"酄唷啖啵啶啷唳唰啜喋嗒喃喱喹喈喁喟啾嗖喑啻嗟喽喾喔喙嗪嗷嗉嘟嗑嗫嗬嗔嗦嗝嗄嗯嗥嗲嗳嗌嗍嗨嗵嗤辔嘞嘈嘌嘁嘤嘣嗾嘀嘧嘭噘嘹噗嘬噍噢噙噜噌噔嚆噤噱噫噻噼嚅嚓嚯囔囗囝囡囵囫囹囿圄圊圉圜帏帙帔帑帱帻帼"],["e140","酅酇酈酑酓酔酕酖酘酙酛酜酟酠酦酧酨酫酭酳酺酻酼醀",4,"醆醈醊醎醏醓",6,"醜",5,"醤",5,"醫醬醰醱醲醳醶醷醸醹醻"],["e180","醼",10,"釈釋釐釒",9,"針",8,"帷幄幔幛幞幡岌屺岍岐岖岈岘岙岑岚岜岵岢岽岬岫岱岣峁岷峄峒峤峋峥崂崃崧崦崮崤崞崆崛嵘崾崴崽嵬嵛嵯嵝嵫嵋嵊嵩嵴嶂嶙嶝豳嶷巅彳彷徂徇徉後徕徙徜徨徭徵徼衢彡犭犰犴犷犸狃狁狎狍狒狨狯狩狲狴狷猁狳猃狺"],["e240","釦",62],["e280","鈥",32,"狻猗猓猡猊猞猝猕猢猹猥猬猸猱獐獍獗獠獬獯獾舛夥飧夤夂饣饧",5,"饴饷饽馀馄馇馊馍馐馑馓馔馕庀庑庋庖庥庠庹庵庾庳赓廒廑廛廨廪膺忄忉忖忏怃忮怄忡忤忾怅怆忪忭忸怙怵怦怛怏怍怩怫怊怿怡恸恹恻恺恂"],["e340","鉆",45,"鉵",16],["e380","銆",7,"銏",24,"恪恽悖悚悭悝悃悒悌悛惬悻悱惝惘惆惚悴愠愦愕愣惴愀愎愫慊慵憬憔憧憷懔懵忝隳闩闫闱闳闵闶闼闾阃阄阆阈阊阋阌阍阏阒阕阖阗阙阚丬爿戕氵汔汜汊沣沅沐沔沌汨汩汴汶沆沩泐泔沭泷泸泱泗沲泠泖泺泫泮沱泓泯泾"],["e440","銨",5,"銯",24,"鋉",31],["e480","鋩",32,"洹洧洌浃浈洇洄洙洎洫浍洮洵洚浏浒浔洳涑浯涞涠浞涓涔浜浠浼浣渚淇淅淞渎涿淠渑淦淝淙渖涫渌涮渫湮湎湫溲湟溆湓湔渲渥湄滟溱溘滠漭滢溥溧溽溻溷滗溴滏溏滂溟潢潆潇漤漕滹漯漶潋潴漪漉漩澉澍澌潸潲潼潺濑"],["e540","錊",51,"錿",10],["e580","鍊",31,"鍫濉澧澹澶濂濡濮濞濠濯瀚瀣瀛瀹瀵灏灞宀宄宕宓宥宸甯骞搴寤寮褰寰蹇謇辶迓迕迥迮迤迩迦迳迨逅逄逋逦逑逍逖逡逵逶逭逯遄遑遒遐遨遘遢遛暹遴遽邂邈邃邋彐彗彖彘尻咫屐屙孱屣屦羼弪弩弭艴弼鬻屮妁妃妍妩妪妣"],["e640","鍬",34,"鎐",27],["e680","鎬",29,"鏋鏌鏍妗姊妫妞妤姒妲妯姗妾娅娆姝娈姣姘姹娌娉娲娴娑娣娓婀婧婊婕娼婢婵胬媪媛婷婺媾嫫媲嫒嫔媸嫠嫣嫱嫖嫦嫘嫜嬉嬗嬖嬲嬷孀尕尜孚孥孳孑孓孢驵驷驸驺驿驽骀骁骅骈骊骐骒骓骖骘骛骜骝骟骠骢骣骥骧纟纡纣纥纨纩"],["e740","鏎",7,"鏗",54],["e780","鐎",32,"纭纰纾绀绁绂绉绋绌绐绔绗绛绠绡绨绫绮绯绱绲缍绶绺绻绾缁缂缃缇缈缋缌缏缑缒缗缙缜缛缟缡",6,"缪缫缬缭缯",4,"缵幺畿巛甾邕玎玑玮玢玟珏珂珑玷玳珀珉珈珥珙顼琊珩珧珞玺珲琏琪瑛琦琥琨琰琮琬"],["e840","鐯",14,"鐿",43,"鑬鑭鑮鑯"],["e880","鑰",20,"钑钖钘铇铏铓铔铚铦铻锜锠琛琚瑁瑜瑗瑕瑙瑷瑭瑾璜璎璀璁璇璋璞璨璩璐璧瓒璺韪韫韬杌杓杞杈杩枥枇杪杳枘枧杵枨枞枭枋杷杼柰栉柘栊柩枰栌柙枵柚枳柝栀柃枸柢栎柁柽栲栳桠桡桎桢桄桤梃栝桕桦桁桧桀栾桊桉栩梵梏桴桷梓桫棂楮棼椟椠棹"],["e940","锧锳锽镃镈镋镕镚镠镮镴镵長",7,"門",42],["e980","閫",32,"椤棰椋椁楗棣椐楱椹楠楂楝榄楫榀榘楸椴槌榇榈槎榉楦楣楹榛榧榻榫榭槔榱槁槊槟榕槠榍槿樯槭樗樘橥槲橄樾檠橐橛樵檎橹樽樨橘橼檑檐檩檗檫猷獒殁殂殇殄殒殓殍殚殛殡殪轫轭轱轲轳轵轶轸轷轹轺轼轾辁辂辄辇辋"],["ea40","闌",27,"闬闿阇阓阘阛阞阠阣",6,"阫阬阭阯阰阷阸阹阺阾陁陃陊陎陏陑陒陓陖陗"],["ea80","陘陙陚陜陝陞陠陣陥陦陫陭",4,"陳陸",12,"隇隉隊辍辎辏辘辚軎戋戗戛戟戢戡戥戤戬臧瓯瓴瓿甏甑甓攴旮旯旰昊昙杲昃昕昀炅曷昝昴昱昶昵耆晟晔晁晏晖晡晗晷暄暌暧暝暾曛曜曦曩贲贳贶贻贽赀赅赆赈赉赇赍赕赙觇觊觋觌觎觏觐觑牮犟牝牦牯牾牿犄犋犍犏犒挈挲掰"],["eb40","隌階隑隒隓隕隖隚際隝",9,"隨",7,"隱隲隴隵隷隸隺隻隿雂雃雈雊雋雐雑雓雔雖",9,"雡",6,"雫"],["eb80","雬雭雮雰雱雲雴雵雸雺電雼雽雿霂霃霅霊霋霌霐霑霒霔霕霗",4,"霝霟霠搿擘耄毪毳毽毵毹氅氇氆氍氕氘氙氚氡氩氤氪氲攵敕敫牍牒牖爰虢刖肟肜肓肼朊肽肱肫肭肴肷胧胨胩胪胛胂胄胙胍胗朐胝胫胱胴胭脍脎胲胼朕脒豚脶脞脬脘脲腈腌腓腴腙腚腱腠腩腼腽腭腧塍媵膈膂膑滕膣膪臌朦臊膻"],["ec40","霡",8,"霫霬霮霯霱霳",4,"霺霻霼霽霿",18,"靔靕靗靘靚靜靝靟靣靤靦靧靨靪",7],["ec80","靲靵靷",4,"靽",7,"鞆",4,"鞌鞎鞏鞐鞓鞕鞖鞗鞙",4,"臁膦欤欷欹歃歆歙飑飒飓飕飙飚殳彀毂觳斐齑斓於旆旄旃旌旎旒旖炀炜炖炝炻烀炷炫炱烨烊焐焓焖焯焱煳煜煨煅煲煊煸煺熘熳熵熨熠燠燔燧燹爝爨灬焘煦熹戾戽扃扈扉礻祀祆祉祛祜祓祚祢祗祠祯祧祺禅禊禚禧禳忑忐"],["ed40","鞞鞟鞡鞢鞤",6,"鞬鞮鞰鞱鞳鞵",46],["ed80","韤韥韨韮",4,"韴韷",23,"怼恝恚恧恁恙恣悫愆愍慝憩憝懋懑戆肀聿沓泶淼矶矸砀砉砗砘砑斫砭砜砝砹砺砻砟砼砥砬砣砩硎硭硖硗砦硐硇硌硪碛碓碚碇碜碡碣碲碹碥磔磙磉磬磲礅磴礓礤礞礴龛黹黻黼盱眄眍盹眇眈眚眢眙眭眦眵眸睐睑睇睃睚睨"],["ee40","頏",62],["ee80","顎",32,"睢睥睿瞍睽瞀瞌瞑瞟瞠瞰瞵瞽町畀畎畋畈畛畲畹疃罘罡罟詈罨罴罱罹羁罾盍盥蠲钅钆钇钋钊钌钍钏钐钔钗钕钚钛钜钣钤钫钪钭钬钯钰钲钴钶",4,"钼钽钿铄铈",6,"铐铑铒铕铖铗铙铘铛铞铟铠铢铤铥铧铨铪"],["ef40","顯",5,"颋颎颒颕颙颣風",37,"飏飐飔飖飗飛飜飝飠",4],["ef80","飥飦飩",30,"铩铫铮铯铳铴铵铷铹铼铽铿锃锂锆锇锉锊锍锎锏锒",4,"锘锛锝锞锟锢锪锫锩锬锱锲锴锶锷锸锼锾锿镂锵镄镅镆镉镌镎镏镒镓镔镖镗镘镙镛镞镟镝镡镢镤",8,"镯镱镲镳锺矧矬雉秕秭秣秫稆嵇稃稂稞稔"],["f040","餈",4,"餎餏餑",28,"餯",26],["f080","饊",9,"饖",12,"饤饦饳饸饹饻饾馂馃馉稹稷穑黏馥穰皈皎皓皙皤瓞瓠甬鸠鸢鸨",4,"鸲鸱鸶鸸鸷鸹鸺鸾鹁鹂鹄鹆鹇鹈鹉鹋鹌鹎鹑鹕鹗鹚鹛鹜鹞鹣鹦",6,"鹱鹭鹳疒疔疖疠疝疬疣疳疴疸痄疱疰痃痂痖痍痣痨痦痤痫痧瘃痱痼痿瘐瘀瘅瘌瘗瘊瘥瘘瘕瘙"],["f140","馌馎馚",10,"馦馧馩",47],["f180","駙",32,"瘛瘼瘢瘠癀瘭瘰瘿瘵癃瘾瘳癍癞癔癜癖癫癯翊竦穸穹窀窆窈窕窦窠窬窨窭窳衤衩衲衽衿袂袢裆袷袼裉裢裎裣裥裱褚裼裨裾裰褡褙褓褛褊褴褫褶襁襦襻疋胥皲皴矜耒耔耖耜耠耢耥耦耧耩耨耱耋耵聃聆聍聒聩聱覃顸颀颃"],["f240","駺",62],["f280","騹",32,"颉颌颍颏颔颚颛颞颟颡颢颥颦虍虔虬虮虿虺虼虻蚨蚍蚋蚬蚝蚧蚣蚪蚓蚩蚶蛄蚵蛎蚰蚺蚱蚯蛉蛏蚴蛩蛱蛲蛭蛳蛐蜓蛞蛴蛟蛘蛑蜃蜇蛸蜈蜊蜍蜉蜣蜻蜞蜥蜮蜚蜾蝈蜴蜱蜩蜷蜿螂蜢蝽蝾蝻蝠蝰蝌蝮螋蝓蝣蝼蝤蝙蝥螓螯螨蟒"],["f340","驚",17,"驲骃骉骍骎骔骕骙骦骩",6,"骲骳骴骵骹骻骽骾骿髃髄髆",4,"髍髎髏髐髒體髕髖髗髙髚髛髜"],["f380","髝髞髠髢髣髤髥髧髨髩髪髬髮髰",8,"髺髼",6,"鬄鬅鬆蟆螈螅螭螗螃螫蟥螬螵螳蟋蟓螽蟑蟀蟊蟛蟪蟠蟮蠖蠓蟾蠊蠛蠡蠹蠼缶罂罄罅舐竺竽笈笃笄笕笊笫笏筇笸笪笙笮笱笠笥笤笳笾笞筘筚筅筵筌筝筠筮筻筢筲筱箐箦箧箸箬箝箨箅箪箜箢箫箴篑篁篌篝篚篥篦篪簌篾篼簏簖簋"],["f440","鬇鬉",5,"鬐鬑鬒鬔",10,"鬠鬡鬢鬤",10,"鬰鬱鬳",7,"鬽鬾鬿魀魆魊魋魌魎魐魒魓魕",5],["f480","魛",32,"簟簪簦簸籁籀臾舁舂舄臬衄舡舢舣舭舯舨舫舸舻舳舴舾艄艉艋艏艚艟艨衾袅袈裘裟襞羝羟羧羯羰羲籼敉粑粝粜粞粢粲粼粽糁糇糌糍糈糅糗糨艮暨羿翎翕翥翡翦翩翮翳糸絷綦綮繇纛麸麴赳趄趔趑趱赧赭豇豉酊酐酎酏酤"],["f540","魼",62],["f580","鮻",32,"酢酡酰酩酯酽酾酲酴酹醌醅醐醍醑醢醣醪醭醮醯醵醴醺豕鹾趸跫踅蹙蹩趵趿趼趺跄跖跗跚跞跎跏跛跆跬跷跸跣跹跻跤踉跽踔踝踟踬踮踣踯踺蹀踹踵踽踱蹉蹁蹂蹑蹒蹊蹰蹶蹼蹯蹴躅躏躔躐躜躞豸貂貊貅貘貔斛觖觞觚觜"],["f640","鯜",62],["f680","鰛",32,"觥觫觯訾謦靓雩雳雯霆霁霈霏霎霪霭霰霾龀龃龅",5,"龌黾鼋鼍隹隼隽雎雒瞿雠銎銮鋈錾鍪鏊鎏鐾鑫鱿鲂鲅鲆鲇鲈稣鲋鲎鲐鲑鲒鲔鲕鲚鲛鲞",5,"鲥",4,"鲫鲭鲮鲰",7,"鲺鲻鲼鲽鳄鳅鳆鳇鳊鳋"],["f740","鰼",62],["f780","鱻鱽鱾鲀鲃鲄鲉鲊鲌鲏鲓鲖鲗鲘鲙鲝鲪鲬鲯鲹鲾",4,"鳈鳉鳑鳒鳚鳛鳠鳡鳌",4,"鳓鳔鳕鳗鳘鳙鳜鳝鳟鳢靼鞅鞑鞒鞔鞯鞫鞣鞲鞴骱骰骷鹘骶骺骼髁髀髅髂髋髌髑魅魃魇魉魈魍魑飨餍餮饕饔髟髡髦髯髫髻髭髹鬈鬏鬓鬟鬣麽麾縻麂麇麈麋麒鏖麝麟黛黜黝黠黟黢黩黧黥黪黯鼢鼬鼯鼹鼷鼽鼾齄"],["f840","鳣",62],["f880","鴢",32],["f940","鵃",62],["f980","鶂",32],["fa40","鶣",62],["fa80","鷢",32],["fb40","鸃",27,"鸤鸧鸮鸰鸴鸻鸼鹀鹍鹐鹒鹓鹔鹖鹙鹝鹟鹠鹡鹢鹥鹮鹯鹲鹴",9,"麀"],["fb80","麁麃麄麅麆麉麊麌",5,"麔",8,"麞麠",5,"麧麨麩麪"],["fc40","麫",8,"麵麶麷麹麺麼麿",4,"黅黆黇黈黊黋黌黐黒黓黕黖黗黙黚點黡黣黤黦黨黫黬黭黮黰",8,"黺黽黿",6],["fc80","鼆",4,"鼌鼏鼑鼒鼔鼕鼖鼘鼚",5,"鼡鼣",8,"鼭鼮鼰鼱"],["fd40","鼲",4,"鼸鼺鼼鼿",4,"齅",10,"齒",38],["fd80","齹",5,"龁龂龍",11,"龜龝龞龡",4,"郎凉秊裏隣"],["fe40","兀嗀﨎﨏﨑﨓﨔礼﨟蘒﨡﨣﨤﨧﨨﨩"]]')},function(i,o,u){var p=u(96);i.exports=Object("z").propertyIsEnumerable(0)?Object:function(i){return"String"==p(i)?i.split(""):Object(i)}},function(i,o){i.exports=function(i){if(null==i)throw TypeError("Can't call method on  "+i);return i}},function(i,o,u){var p=u(130),g=u(97),y=u(51),w=u(179),_=u(52),x=u(254),k=Object.getOwnPropertyDescriptor;o.f=u(24)?k:function getOwnPropertyDescriptor(i,o){if(i=y(i),o=w(o,!0),x)try{return k(i,o)}catch(i){}if(_(i,o))return g(!p.f.call(i,o),i[o])}},function(i,o,u){var p=u(23);i.exports=function(i,o){if(!p(i))return i;var u,g;if(o&&"function"==typeof(u=i.toString)&&!p(g=u.call(i)))return g;if("function"==typeof(u=i.valueOf)&&!p(g=u.call(i)))return g;if(!o&&"function"==typeof(u=i.toString)&&!p(g=u.call(i)))return g;throw TypeError("Can't convert object to primitive value")}},function(i,o,u){var p=u(23),g=u(17).document,y=p(g)&&p(g.createElement);i.exports=function(i){return y?g.createElement(i):{}}},function(i,o,u){var p=u(9),g=u(7),y=u(53);i.exports=function(i,o){var u=(g.Object||{})[i]||Object[i],w={};w[i]=o(u),p(p.S+p.F*y((function(){u(1)})),"Object",w)}},function(i,o,u){"use strict";var p=u(101),g=u(9),y=u(256),w=u(40),_=u(100),x=u(432),k=u(103),P=u(434),E=u(20)("iterator"),O=!([].keys&&"next"in[].keys()),returnThis=function(){return this};i.exports=function(i,o,u,I,B,D,R){x(u,o,I);var N,U,W,getMethod=function(i){if(!O&&i in K)return K[i];switch(i){case"keys":return function keys(){return new u(this,i)};case"values":return function values(){return new u(this,i)}}return function entries(){return new u(this,i)}},G=o+" Iterator",j="values"==B,X=!1,K=i.prototype,Y=K[E]||K["@@iterator"]||B&&K[B],J=Y||getMethod(B),$=B?j?getMethod("entries"):J:void 0,tt="Array"==o&&K.entries||Y;if(tt&&(W=P(tt.call(new i)))!==Object.prototype&&W.next&&(k(W,G,!0),p||"function"==typeof W[E]||w(W,E,returnThis)),j&&Y&&"values"!==Y.name&&(X=!0,J=function values(){return Y.call(this)}),p&&!R||!O&&!X&&K[E]||w(K,E,J),_[o]=J,_[G]=returnThis,B)if(N={values:j?J:getMethod("values"),keys:D?J:getMethod("keys"),entries:$},R)for(U in N)U in K||y(K,U,N[U]);else g(g.P+g.F*(O||X),o,N);return N}},function(i,o){var u=Math.ceil,p=Math.floor;i.exports=function(i){return isNaN(i=+i)?0:(i>0?p:u)(i)}},function(i,o,u){var p=u(185)("keys"),g=u(133);i.exports=function(i){return p[i]||(p[i]=g(i))}},function(i,o,u){var p=u(7),g=u(17),y=g["__core-js_shared__"]||(g["__core-js_shared__"]={});(i.exports=function(i,o){return y[i]||(y[i]=void 0!==o?o:{})})("versions",[]).push({version:p.version,mode:u(101)?"pure":"global",copyright:"© 2020 Denis Pushkarev (zloirock.ru)"})},function(i,o){i.exports="constructor,hasOwnProperty,isPrototypeOf,propertyIsEnumerable,toLocaleString,toString,valueOf".split(",")},function(i,o,u){var p=u(188),g=u(20)("iterator"),y=u(100);i.exports=u(7).getIteratorMethod=function(i){if(null!=i)return i[g]||i["@@iterator"]||y[p(i)]}},function(i,o,u){var p=u(96),g=u(20)("toStringTag"),y="Arguments"==p(function(){return arguments}());i.exports=function(i){var o,u,w;return void 0===i?"Undefined":null===i?"Null":"string"==typeof(u=function(i,o){try{return i[o]}catch(i){}}(o=Object(i),g))?u:y?p(o):"Object"==(w=p(o))&&"function"==typeof o.callee?"Arguments":w}},function(i,o,u){"use strict";o.__esModule=!0;var p=_interopRequireDefault(u(440)),g=_interopRequireDefault(u(442)),y="function"==typeof g.default&&"symbol"==typeof p.default?function(i){return typeof i}:function(i){return i&&"function"==typeof g.default&&i.constructor===g.default&&i!==g.default.prototype?"symbol":typeof i};function _interopRequireDefault(i){return i&&i.__esModule?i:{default:i}}o.default="function"==typeof g.default&&"symbol"===y(p.default)?function(i){return void 0===i?"undefined":y(i)}:function(i){return i&&"function"==typeof g.default&&i.constructor===g.default&&i!==g.default.prototype?"symbol":void 0===i?"undefined":y(i)}},function(i,o,u){o.f=u(20)},function(i,o,u){var p=u(17),g=u(7),y=u(101),w=u(190),_=u(26).f;i.exports=function(i){var o=g.Symbol||(g.Symbol=y?{}:p.Symbol||{});"_"==i.charAt(0)||i in o||_(o,i,{value:w.f(i)})}},function(i,o){o.f=Object.getOwnPropertySymbols},function(i,o,u){var p=u(40);i.exports=function(i,o,u){for(var g in o)u&&i[g]?i[g]=o[g]:p(i,g,o[g]);return i}},function(i,o){i.exports=function(i,o,u,p){if(!(i instanceof o)||void 0!==p&&p in i)throw TypeError(u+": incorrect invocation!");return i}},function(i,o,u){var p=u(23);i.exports=function(i,o){if(!p(i)||i._t!==o)throw TypeError("Incompatible receiver, "+o+" required!");return i}},function(i,o,u){"use strict";var p=u(482);i.exports=Function.prototype.bind||p},function(i,o,u){"use strict";var p=u(196),g=u(275),y=g("%Function.prototype.apply%"),w=g("%Function.prototype.call%"),_=g("%Reflect.apply%",!0)||p.call(w,y),x=g("%Object.getOwnPropertyDescriptor%",!0),k=g("%Object.defineProperty%",!0),P=g("%Math.max%");if(k)try{k({},"a",{value:1})}catch(i){k=null}i.exports=function callBind(i){var o=_(p,w,arguments);if(x&&k){var u=x(o,"length");u.configurable&&k(o,"length",{value:1+P(0,i.length-(arguments.length-1))})}return o};var E=function applyBind(){return _(p,y,arguments)};k?k(i.exports,"apply",{value:E}):i.exports.apply=E},function(i,o,u){"use strict";var p=u(98);function PromiseCapability(i){var o,u;this.promise=new i((function(i,p){if(void 0!==o||void 0!==u)throw TypeError("Bad Promise constructor");o=i,u=p})),this.resolve=p(o),this.reject=p(u)}i.exports.f=function(i){return new PromiseCapability(i)}},function(i,o,u){"use strict";var p=u(0).isString,g=u(0).isNumber,y=u(0).isObject,w=u(0).isArray,_=u(0).isUndefined,x=u(290),k=/^(\s)+/g,P=/(\s)+$/g;function TextTools(i){this.fontProvider=i}function splitWords(i,o){var u=[];if(i=i.replace(/\t/g,"    "),o)return u.push({text:i}),u;for(var p,g=new x(i),y=0;p=g.nextBreak();){var w=i.slice(y,p.position);p.required||w.match(/\r?\n$|\r$/)?(w=w.replace(/\r?\n$|\r$/,""),u.push({text:w,lineEnd:!0})):u.push({text:w}),y=p.position}return u}function copyStyle(i,o){for(var u in o=o||{},i=i||{})"text"!=u&&i.hasOwnProperty(u)&&(o[u]=i[u]);return o}function normalizeString(i){return null==i?"":g(i)?i.toString():p(i)?i:i.toString()}function getStyleProperty(i,o,u,p){var g;return void 0!==i[u]&&null!==i[u]?i[u]:o?(o.auto(i,(function(){g=o.getProperty(u)})),null!=g?g:p):p}function widthOfString(i,o,u,p,g){return o.widthOfString(i,u,g)+(p||0)*(i.length-1)}TextTools.prototype.buildInlines=function(i,o){var u,p=function measure(i,o,u){var p=function normalizeTextArray(i,o){function getOneWord(i,o,u){if(_(o[i]))return null;if(o[i].lineEnd)return null;var p=o[i].text;if(u){var g=splitWords(normalizeString(p),!1);if(_(g[g.length-1]))return null;p=g[g.length-1].text}return p}var u=[];w(i)||(i=[i]);i=function flatten(i){return i.reduce((function(i,o){var u=w(o.text)?flatten(o.text):o,p=[].concat(u).some(Array.isArray);return i.concat(p?flatten(u):u)}),[])}(i);for(var p=null,g=0,x=i.length;g<x;g++){var k,P=i[g],E=null,O=getStyleProperty(P||{},o,"noWrap",!1);if(y(P)?(P._textRef&&P._textRef._textNodeRef.text&&(P.text=P._textRef._textNodeRef.text),k=splitWords(normalizeString(P.text),O),E=copyStyle(P)):k=splitWords(normalizeString(P),O),p&&k.length){var I=getOneWord(0,k,O);1===splitWords(normalizeString(p+I),!1).length&&(u[u.length-1].noNewLine=!0)}for(var B=0,D=k.length;B<D;B++){var R={text:k[B].text};k[B].lineEnd&&(R.lineEnd=!0),copyStyle(E,R),u.push(R)}p=null,g+1<x&&(p=getOneWord(k.length-1,k,O))}return u}(o,u);if(p.length){var g=getStyleProperty(p[0],u,"leadingIndent",0);g&&(p[0].leadingCut=-g,p[0].leadingIndent=g)}return p.forEach((function(o){var p=getStyleProperty(o,u,"font","Roboto"),g=getStyleProperty(o,u,"fontSize",12),y=getStyleProperty(o,u,"fontFeatures",null),w=getStyleProperty(o,u,"bold",!1),_=getStyleProperty(o,u,"italics",!1),x=getStyleProperty(o,u,"color","black"),E=getStyleProperty(o,u,"decoration",null),O=getStyleProperty(o,u,"decorationColor",null),I=getStyleProperty(o,u,"decorationStyle",null),B=getStyleProperty(o,u,"background",null),D=getStyleProperty(o,u,"lineHeight",1),R=getStyleProperty(o,u,"characterSpacing",0),N=getStyleProperty(o,u,"link",null),U=getStyleProperty(o,u,"linkToPage",null),W=getStyleProperty(o,u,"linkToDestination",null),G=getStyleProperty(o,u,"noWrap",null),j=getStyleProperty(o,u,"preserveLeadingSpaces",!1),X=getStyleProperty(o,u,"preserveTrailingSpaces",!1),K=getStyleProperty(o,u,"opacity",1),Y=getStyleProperty(o,u,"sup",!1),J=getStyleProperty(o,u,"sub",!1);(Y||J)&&void 0===o.fontSize&&(g*=.58);var $,tt,et=i.provideFont(p,w,_);o.width=widthOfString(o.text,et,g,R,y),o.height=et.lineHeight(g)*D,o.leadingCut||(o.leadingCut=0),!j&&($=o.text.match(k))&&(o.leadingCut+=widthOfString($[0],et,g,R,y)),!X&&(tt=o.text.match(P))?o.trailingCut=widthOfString(tt[0],et,g,R,y):o.trailingCut=0,o.alignment=getStyleProperty(o,u,"alignment","left"),o.font=et,o.fontSize=g,o.fontFeatures=y,o.characterSpacing=R,o.color=x,o.decoration=E,o.decorationColor=O,o.decorationStyle=I,o.background=B,o.link=N,o.linkToPage=U,o.linkToDestination=W,o.noWrap=G,o.opacity=K,o.sup=Y,o.sub=J})),p}(this.fontProvider,i,o),g=0,x=0;return p.forEach((function(i){g=Math.max(g,i.width-i.leadingCut-i.trailingCut),u||(u={width:0,leadingCut:i.leadingCut,trailingCut:0}),u.width+=i.width,u.trailingCut=i.trailingCut,x=Math.max(x,function getTrimmedWidth(i){return Math.max(0,i.width-i.leadingCut-i.trailingCut)}(u)),i.lineEnd&&(u=null)})),getStyleProperty({},o,"noWrap",!1)&&(g=x),{items:p,minWidth:g,maxWidth:x}},TextTools.prototype.sizeOfString=function(i,o){i=i?i.toString().replace(/\t/g,"    "):"";var u=getStyleProperty({},o,"font","Roboto"),p=getStyleProperty({},o,"fontSize",12),g=getStyleProperty({},o,"fontFeatures",null),y=getStyleProperty({},o,"bold",!1),w=getStyleProperty({},o,"italics",!1),_=getStyleProperty({},o,"lineHeight",1),x=getStyleProperty({},o,"characterSpacing",0),k=this.fontProvider.provideFont(u,y,w);return{width:widthOfString(i,k,p,x,g),height:k.lineHeight(p)*_,fontSize:p,lineHeight:_,ascender:k.ascender/1e3*p,descender:k.descender/1e3*p}},TextTools.prototype.sizeOfRotatedText=function(i,o,u){var p=o*Math.PI/-180,g=this.sizeOfString(i,u);return{width:Math.abs(g.height*Math.sin(p))+Math.abs(g.width*Math.cos(p)),height:Math.abs(g.width*Math.sin(p))+Math.abs(g.height*Math.cos(p))}},TextTools.prototype.widthOfString=function(i,o,u,p,g){return widthOfString(i,o,u,p,g)},i.exports=TextTools},function(i,o,u){"use strict";var p=u(0).isString;function isAutoColumn(i){return"auto"===i.width}function isStarColumn(i){return null===i.width||void 0===i.width||"*"===i.width||"star"===i.width}i.exports={buildColumnWidths:function buildColumnWidths(i,o){var u=[],g=0,y=0,w=[],_=0,x=0,k=[],P=o;i.forEach((function(i){isAutoColumn(i)?(u.push(i),g+=i._minWidth,y+=i._maxWidth):isStarColumn(i)?(w.push(i),_=Math.max(_,i._minWidth),x=Math.max(x,i._maxWidth)):k.push(i)})),k.forEach((function(i){p(i.width)&&/\d+%/.test(i.width)&&(i.width=parseFloat(i.width)*P/100),i.width<i._minWidth&&i.elasticWidth?i._calcWidth=i._minWidth:i._calcWidth=i.width,o-=i._calcWidth}));var E=g+_*w.length,O=y+x*w.length;if(E>=o)u.forEach((function(i){i._calcWidth=i._minWidth})),w.forEach((function(i){i._calcWidth=_}));else{if(O<o)u.forEach((function(i){i._calcWidth=i._maxWidth,o-=i._calcWidth}));else{var I=o-E,B=O-E;u.forEach((function(i){var u=i._maxWidth-i._minWidth;i._calcWidth=i._minWidth+u*I/B,o-=i._calcWidth}))}if(w.length>0){var D=o/w.length;w.forEach((function(i){i._calcWidth=D}))}}},measureMinMax:function measureMinMax(i){for(var o={min:0,max:0},u={min:0,max:0},p=0,g=0,y=i.length;g<y;g++){var w=i[g];isStarColumn(w)?(u.min=Math.max(u.min,w._minWidth),u.max=Math.max(u.max,w._maxWidth),p++):isAutoColumn(w)?(o.min+=w._minWidth,o.max+=w._maxWidth):(o.min+=void 0!==w.width&&w.width||w._minWidth,o.max+=void 0!==w.width&&w.width||w._maxWidth)}return p&&(o.min+=p*u.min,o.max+=p*u.max),o},isAutoColumn:isAutoColumn,isStarColumn:isStarColumn}},function(i,o){var u={}.toString;i.exports=Array.isArray||function(i){return"[object Array]"==u.call(i)}},function(i,o,u){var p=u(1),g=u(19),y=u(91);p({target:"Object",stat:!0,forced:u(3)((function(){y(1)}))},{keys:function keys(i){return y(g(i))}})},function(i,o,u){var p=u(13),g=u(3),y=u(140);i.exports=!p&&!g((function(){return 7!=Object.defineProperty(y("div"),"a",{get:function(){return 7}}).a}))},function(i,o,u){var p=u(15),g=u(205),y=u(34),w=u(16);i.exports=function(i,o){for(var u=g(o),_=w.f,x=y.f,k=0;k<u.length;k++){var P=u[k];p(i,P)||_(i,P,x(o,P))}}},function(i,o,u){var p=u(30),g=u(57),y=u(145),w=u(12);i.exports=p("Reflect","ownKeys")||function ownKeys(i){var o=g.f(w(i)),u=y.f;return u?o.concat(u(i)):o}},function(i,o,u){var p=u(15),g=u(21),y=u(113).indexOf,w=u(111);i.exports=function(i,o){var u,_=g(i),x=0,k=[];for(u in _)!p(w,u)&&p(_,u)&&k.push(u);for(;o.length>x;)p(_,u=o[x++])&&(~y(k,u)||k.push(u));return k}},function(i,o,u){"use strict";var p=u(1),g=u(4),y=u(30),w=u(56),_=u(13),x=u(147),k=u(208),P=u(3),E=u(15),O=u(116),I=u(11),B=u(12),D=u(19),R=u(21),N=u(55),U=u(41),W=u(45),G=u(91),j=u(57),X=u(312),K=u(145),Y=u(34),J=u(16),$=u(107),tt=u(18),et=u(25),rt=u(109),nt=u(108),it=u(111),ot=u(110),at=u(6),st=u(210),lt=u(211),ct=u(92),ut=u(43),ft=u(22).forEach,ht=nt("hidden"),dt=at("toPrimitive"),pt=ut.set,gt=ut.getterFor("Symbol"),yt=Object.prototype,vt=g.Symbol,mt=y("JSON","stringify"),bt=Y.f,wt=J.f,_t=X.f,xt=$.f,St=rt("symbols"),Ct=rt("op-symbols"),kt=rt("string-to-symbol-registry"),At=rt("symbol-to-string-registry"),Pt=rt("wks"),Tt=g.QObject,Et=!Tt||!Tt.prototype||!Tt.prototype.findChild,Ot=_&&P((function(){return 7!=W(wt({},"a",{get:function(){return wt(this,"a",{value:7}).a}})).a}))?function(i,o,u){var p=bt(yt,o);p&&delete yt[o],wt(i,o,u),p&&i!==yt&&wt(yt,o,p)}:wt,wrap=function(i,o){var u=St[i]=W(vt.prototype);return pt(u,{type:"Symbol",tag:i,description:o}),_||(u.description=o),u},It=k?function(i){return"symbol"==typeof i}:function(i){return Object(i)instanceof vt},Bt=function defineProperty(i,o,u){i===yt&&Bt(Ct,o,u),B(i);var p=N(o,!0);return B(u),E(St,p)?(u.enumerable?(E(i,ht)&&i[ht][p]&&(i[ht][p]=!1),u=W(u,{enumerable:U(0,!1)})):(E(i,ht)||wt(i,ht,U(1,{})),i[ht][p]=!0),Ot(i,p,u)):wt(i,p,u)},Ft=function defineProperties(i,o){B(i);var u=R(o),p=G(u).concat(Rt(u));return ft(p,(function(o){_&&!Dt.call(u,o)||Bt(i,o,u[o])})),i},Dt=function propertyIsEnumerable(i){var o=N(i,!0),u=xt.call(this,o);return!(this===yt&&E(St,o)&&!E(Ct,o))&&(!(u||!E(this,o)||!E(St,o)||E(this,ht)&&this[ht][o])||u)},Lt=function getOwnPropertyDescriptor(i,o){var u=R(i),p=N(o,!0);if(u!==yt||!E(St,p)||E(Ct,p)){var g=bt(u,p);return!g||!E(St,p)||E(u,ht)&&u[ht][p]||(g.enumerable=!0),g}},Mt=function getOwnPropertyNames(i){var o=_t(R(i)),u=[];return ft(o,(function(i){E(St,i)||E(it,i)||u.push(i)})),u},Rt=function getOwnPropertySymbols(i){var o=i===yt,u=_t(o?Ct:R(i)),p=[];return ft(u,(function(i){!E(St,i)||o&&!E(yt,i)||p.push(St[i])})),p};(x||(et((vt=function Symbol(){if(this instanceof vt)throw TypeError("Symbol is not a constructor");var i=arguments.length&&void 0!==arguments[0]?String(arguments[0]):void 0,o=ot(i),setter=function(i){this===yt&&setter.call(Ct,i),E(this,ht)&&E(this[ht],o)&&(this[ht][o]=!1),Ot(this,o,U(1,i))};return _&&Et&&Ot(yt,o,{configurable:!0,set:setter}),wrap(o,i)}).prototype,"toString",(function toString(){return gt(this).tag})),et(vt,"withoutSetter",(function(i){return wrap(ot(i),i)})),$.f=Dt,J.f=Bt,Y.f=Lt,j.f=X.f=Mt,K.f=Rt,st.f=function(i){return wrap(at(i),i)},_&&(wt(vt.prototype,"description",{configurable:!0,get:function description(){return gt(this).description}}),w||et(yt,"propertyIsEnumerable",Dt,{unsafe:!0}))),p({global:!0,wrap:!0,forced:!x,sham:!x},{Symbol:vt}),ft(G(Pt),(function(i){lt(i)})),p({target:"Symbol",stat:!0,forced:!x},{for:function(i){var o=String(i);if(E(kt,o))return kt[o];var u=vt(o);return kt[o]=u,At[u]=o,u},keyFor:function keyFor(i){if(!It(i))throw TypeError(i+" is not a symbol");if(E(At,i))return At[i]},useSetter:function(){Et=!0},useSimple:function(){Et=!1}}),p({target:"Object",stat:!0,forced:!x,sham:!_},{create:function create(i,o){return void 0===o?W(i):Ft(W(i),o)},defineProperty:Bt,defineProperties:Ft,getOwnPropertyDescriptor:Lt}),p({target:"Object",stat:!0,forced:!x},{getOwnPropertyNames:Mt,getOwnPropertySymbols:Rt}),p({target:"Object",stat:!0,forced:P((function(){K.f(1)}))},{getOwnPropertySymbols:function getOwnPropertySymbols(i){return K.f(D(i))}}),mt)&&p({target:"JSON",stat:!0,forced:!x||P((function(){var i=vt();return"[null]"!=mt([i])||"{}"!=mt({a:i})||"{}"!=mt(Object(i))}))},{stringify:function stringify(i,o,u){for(var p,g=[i],y=1;arguments.length>y;)g.push(arguments[y++]);if(p=o,(I(o)||void 0!==i)&&!It(i))return O(o)||(o=function(i,o){if("function"==typeof p&&(o=p.call(this,i,o)),!It(o))return o}),g[1]=o,mt.apply(null,g)}});vt.prototype[dt]||tt(vt.prototype,dt,vt.prototype.valueOf),ct(vt,"Symbol"),it[ht]=!0},function(i,o,u){var p=u(147);i.exports=p&&!Symbol.sham&&"symbol"==typeof Symbol.iterator},function(i,o,u){var p=u(30);i.exports=p("document","documentElement")},function(i,o,u){var p=u(6);o.f=p},function(i,o,u){var p=u(112),g=u(15),y=u(210),w=u(16).f;i.exports=function(i){var o=p.Symbol||(p.Symbol={});g(o,i)||w(o,i,{value:y.f(i)})}},function(i,o,u){var p=u(11),g=u(116),y=u(6)("species");i.exports=function(i,o){var u;return g(i)&&("function"!=typeof(u=i.constructor)||u!==Array&&!g(u.prototype)?p(u)&&null===(u=u[y])&&(u=void 0):u=void 0),new(void 0===u?Array:u)(0===o?0:o)}},function(i,o){i.exports={CSSRuleList:0,CSSStyleDeclaration:0,CSSValueList:0,ClientRectList:0,DOMRectList:0,DOMStringList:0,DOMTokenList:1,DataTransferItemList:0,FileList:0,HTMLAllCollection:0,HTMLCollection:0,HTMLFormElement:0,HTMLSelectElement:0,MediaList:0,MimeTypeArray:0,NamedNodeMap:0,NodeList:1,PaintRequestList:0,Plugin:0,PluginArray:0,SVGLengthList:0,SVGNumberList:0,SVGPathSegList:0,SVGPointList:0,SVGStringList:0,SVGTransformList:0,SourceBufferList:0,StyleSheetList:0,TextTrackCueList:0,TextTrackList:0,TouchList:0}},function(i,o,u){var p=u(3);i.exports=!p((function(){function F(){}return F.prototype.constructor=null,Object.getPrototypeOf(new F)!==F.prototype}))},function(i,o,u){"use strict";var p=u(1),g=u(13),y=u(4),w=u(15),_=u(11),x=u(16).f,k=u(204),P=y.Symbol;if(g&&"function"==typeof P&&(!("description"in P.prototype)||void 0!==P().description)){var E={},O=function Symbol(){var i=arguments.length<1||void 0===arguments[0]?void 0:String(arguments[0]),o=this instanceof O?new P(i):void 0===i?P():P(i);return""===i&&(E[o]=!0),o};k(O,P);var I=O.prototype=P.prototype;I.constructor=O;var B=I.toString,D="Symbol(test)"==String(P("test")),R=/^Symbol\((.*)\)[^)]+$/;x(I,"description",{configurable:!0,get:function description(){var i=_(this)?this.valueOf():this,o=B.call(i);if(w(E,i))return"";var u=D?o.slice(7,-1):o.replace(R,"$1");return""===u?void 0:u}}),p({global:!0,forced:!0},{Symbol:O})}},function(i,o,u){u(211)("iterator")},function(i,o,u){"use strict";var p=u(1),g=u(324),y=u(58),w=u(46),_=u(92),x=u(18),k=u(25),P=u(6),E=u(56),O=u(94),I=u(218),B=I.IteratorPrototype,D=I.BUGGY_SAFARI_ITERATORS,R=P("iterator"),returnThis=function(){return this};i.exports=function(i,o,u,P,I,N,U){g(u,o,P);var W,G,j,getIterationMethod=function(i){if(i===I&&$)return $;if(!D&&i in Y)return Y[i];switch(i){case"keys":return function keys(){return new u(this,i)};case"values":return function values(){return new u(this,i)};case"entries":return function entries(){return new u(this,i)}}return function(){return new u(this)}},X=o+" Iterator",K=!1,Y=i.prototype,J=Y[R]||Y["@@iterator"]||I&&Y[I],$=!D&&J||getIterationMethod(I),tt="Array"==o&&Y.entries||J;if(tt&&(W=y(tt.call(new i)),B!==Object.prototype&&W.next&&(E||y(W)===B||(w?w(W,B):"function"!=typeof W[R]&&x(W,R,returnThis)),_(W,X,!0,!0),E&&(O[X]=returnThis))),"values"==I&&J&&"values"!==J.name&&(K=!0,$=function values(){return J.call(this)}),E&&!U||Y[R]===$||x(Y,R,$),O[o]=$,I)if(G={values:getIterationMethod("values"),keys:N?$:getIterationMethod("keys"),entries:getIterationMethod("entries")},U)for(j in G)(D||K||!(j in Y))&&k(Y,j,G[j]);else p({target:o,proto:!0,forced:D||K},G);return G}},function(i,o,u){"use strict";var p,g,y,w=u(3),_=u(58),x=u(18),k=u(15),P=u(6),E=u(56),O=P("iterator"),I=!1;[].keys&&("next"in(y=[].keys())?(g=_(_(y)))!==Object.prototype&&(p=g):I=!0);var B=null==p||w((function(){var i={};return p[O].call(i)!==i}));B&&(p={}),E&&!B||k(p,O)||x(p,O,(function(){return this})),i.exports={IteratorPrototype:p,BUGGY_SAFARI_ITERATORS:I}},function(i,o,u){var p=u(1),g=u(325);p({target:"Array",stat:!0,forced:!u(156)((function(i){Array.from(i)}))},{from:g})},function(i,o,u){var p=u(12);i.exports=function(i){var o=i.return;if(void 0!==o)return p(o.call(i)).value}},function(i,o,u){"use strict";var p=u(1),g=u(3),y=u(116),w=u(11),_=u(19),x=u(8),k=u(118),P=u(212),E=u(117),O=u(6),I=u(115),B=O("isConcatSpreadable"),D=I>=51||!g((function(){var i=[];return i[B]=!1,i.concat()[0]!==i})),R=E("concat"),isConcatSpreadable=function(i){if(!w(i))return!1;var o=i[B];return void 0!==o?!!o:y(i)};p({target:"Array",proto:!0,forced:!D||!R},{concat:function concat(i){var o,u,p,g,y,w=_(this),E=P(w,0),O=0;for(o=-1,p=arguments.length;o<p;o++)if(isConcatSpreadable(y=-1===o?w:arguments[o])){if(O+(g=x(y.length))>9007199254740991)throw TypeError("Maximum allowed index exceeded");for(u=0;u<g;u++,O++)u in y&&k(E,O,y[u])}else{if(O>=9007199254740991)throw TypeError("Maximum allowed index exceeded");k(E,O++,y)}return E.length=O,E}})},function(i,o,u){"use strict";var p=u(1),g=u(90),y=u(21),w=u(149),_=[].join,x=g!=Object,k=w("join",",");p({target:"Array",proto:!0,forced:x||!k},{join:function join(i){return _.call(y(this),void 0===i?",":i)}})},function(i,o,u){"use strict";var p=u(12);i.exports=function(){var i=p(this),o="";return i.global&&(o+="g"),i.ignoreCase&&(o+="i"),i.multiline&&(o+="m"),i.dotAll&&(o+="s"),i.unicode&&(o+="u"),i.sticky&&(o+="y"),o}},function(i,o,u){"use strict";var p=u(158),g=u(12),y=u(8),w=u(36),_=u(35),x=u(159),k=u(329),P=u(160),E=Math.max,O=Math.min;p("replace",2,(function(i,o,u,p){var I=p.REGEXP_REPLACE_SUBSTITUTES_UNDEFINED_CAPTURE,B=p.REPLACE_KEEPS_$0,D=I?"$":"$0";return[function replace(u,p){var g=_(this),y=null==u?void 0:u[i];return void 0!==y?y.call(u,g,p):o.call(String(g),u,p)},function(i,p){if(!I&&B||"string"==typeof p&&-1===p.indexOf(D)){var _=u(o,i,this,p);if(_.done)return _.value}var R=g(i),N=String(this),U="function"==typeof p;U||(p=String(p));var W=R.global;if(W){var G=R.unicode;R.lastIndex=0}for(var j=[];;){var X=P(R,N);if(null===X)break;if(j.push(X),!W)break;""===String(X[0])&&(R.lastIndex=x(N,y(R.lastIndex),G))}for(var K,Y="",J=0,$=0;$<j.length;$++){X=j[$];for(var tt=String(X[0]),et=E(O(w(X.index),N.length),0),rt=[],nt=1;nt<X.length;nt++)rt.push(void 0===(K=X[nt])?K:String(K));var it=X.groups;if(U){var ot=[tt].concat(rt,et,N);void 0!==it&&ot.push(it);var at=String(p.apply(void 0,ot))}else at=k(tt,N,et,rt,it,p);et>=J&&(Y+=N.slice(J,et)+at,J=et+tt.length)}return Y+N.slice(J)}]}))},function(i,o,u){"use strict";var p=u(1),g=u(113).includes,y=u(153);p({target:"Array",proto:!0},{includes:function includes(i){return g(this,i,arguments.length>1?arguments[1]:void 0)}}),y("includes")},function(i,o,u){"use strict";var p=u(13),g=u(4),y=u(146),w=u(25),_=u(15),x=u(42),k=u(227),P=u(55),E=u(3),O=u(45),I=u(57).f,B=u(34).f,D=u(16).f,R=u(228).trim,N=g.Number,U=N.prototype,W="Number"==x(O(U)),toNumber=function(i){var o,u,p,g,y,w,_,x,k=P(i,!1);if("string"==typeof k&&k.length>2)if(43===(o=(k=R(k)).charCodeAt(0))||45===o){if(88===(u=k.charCodeAt(2))||120===u)return NaN}else if(48===o){switch(k.charCodeAt(1)){case 66:case 98:p=2,g=49;break;case 79:case 111:p=8,g=55;break;default:return+k}for(w=(y=k.slice(2)).length,_=0;_<w;_++)if((x=y.charCodeAt(_))<48||x>g)return NaN;return parseInt(y,p)}return+k};if(y("Number",!N(" 0o1")||!N("0b1")||N("+0x1"))){for(var G,j=function Number(i){var o=arguments.length<1?0:i,u=this;return u instanceof j&&(W?E((function(){U.valueOf.call(u)})):"Number"!=x(u))?k(new N(toNumber(o)),u,j):toNumber(o)},X=p?I(N):"MAX_VALUE,MIN_VALUE,NaN,NEGATIVE_INFINITY,POSITIVE_INFINITY,EPSILON,isFinite,isInteger,isNaN,isSafeInteger,MAX_SAFE_INTEGER,MIN_SAFE_INTEGER,parseFloat,parseInt,isInteger,fromString,range".split(","),K=0;X.length>K;K++)_(N,G=X[K])&&!_(j,G)&&D(j,G,B(N,G));j.prototype=U,U.constructor=j,w(g,"Number",j)}},function(i,o,u){var p=u(11),g=u(46);i.exports=function(i,o,u){var y,w;return g&&"function"==typeof(y=o.constructor)&&y!==u&&p(w=y.prototype)&&w!==u.prototype&&g(i,w),i}},function(i,o,u){var p=u(35),g="["+u(229)+"]",y=RegExp("^"+g+g+"*"),w=RegExp(g+g+"*$"),createMethod=function(i){return function(o){var u=String(p(o));return 1&i&&(u=u.replace(y,"")),2&i&&(u=u.replace(w,"")),u}};i.exports={start:createMethod(1),end:createMethod(2),trim:createMethod(3)}},function(i,o){i.exports="\t\n\v\f\r                　\u2028\u2029\ufeff"},function(i,o,u){"use strict";var p=u(158),g=u(335),y=u(12),w=u(35),_=u(37),x=u(159),k=u(8),P=u(160),E=u(122),O=u(3),I=[].push,B=Math.min,D=!O((function(){return!RegExp(4294967295,"y")}));p("split",2,(function(i,o,u){var p;return p="c"=="abbc".split(/(b)*/)[1]||4!="test".split(/(?:)/,-1).length||2!="ab".split(/(?:ab)*/).length||4!=".".split(/(.?)(.?)/).length||".".split(/()()/).length>1||"".split(/.?/).length?function(i,u){var p=String(w(this)),y=void 0===u?4294967295:u>>>0;if(0===y)return[];if(void 0===i)return[p];if(!g(i))return o.call(p,i,y);for(var _,x,k,P=[],O=(i.ignoreCase?"i":"")+(i.multiline?"m":"")+(i.unicode?"u":"")+(i.sticky?"y":""),B=0,D=new RegExp(i.source,O+"g");(_=E.call(D,p))&&!((x=D.lastIndex)>B&&(P.push(p.slice(B,_.index)),_.length>1&&_.index<p.length&&I.apply(P,_.slice(1)),k=_[0].length,B=x,P.length>=y));)D.lastIndex===_.index&&D.lastIndex++;return B===p.length?!k&&D.test("")||P.push(""):P.push(p.slice(B)),P.length>y?P.slice(0,y):P}:"0".split(void 0,0).length?function(i,u){return void 0===i&&0===u?[]:o.call(this,i,u)}:o,[function split(o,u){var g=w(this),y=null==o?void 0:o[i];return void 0!==y?y.call(o,g,u):p.call(String(g),o,u)},function(i,g){var w=u(p,i,this,g,p!==o);if(w.done)return w.value;var E=y(i),O=String(this),I=_(E,RegExp),R=E.unicode,N=(E.ignoreCase?"i":"")+(E.multiline?"m":"")+(E.unicode?"u":"")+(D?"y":"g"),U=new I(D?E:"^(?:"+E.source+")",N),W=void 0===g?4294967295:g>>>0;if(0===W)return[];if(0===O.length)return null===P(U,O)?[O]:[];for(var G=0,j=0,X=[];j<O.length;){U.lastIndex=D?j:0;var K,Y=P(U,D?O:O.slice(j));if(null===Y||(K=B(k(U.lastIndex+(D?0:j)),O.length))===G)j=x(O,j,R);else{if(X.push(O.slice(G,j)),X.length===W)return X;for(var J=1;J<=Y.length-1;J++)if(X.push(Y[J]),X.length===W)return X;j=G=K}}return X.push(O.slice(G)),X}]}),!D)},function(i,o,u){"use strict";var p=u(1),g=u(4),y=u(13),w=u(337),_=u(5),x=u(164),k=u(165),P=u(41),E=u(18),O=u(8),I=u(234),B=u(235),D=u(55),R=u(15),N=u(119),U=u(11),W=u(45),G=u(46),j=u(57).f,X=u(340),K=u(22).forEach,Y=u(167),J=u(16),$=u(34),tt=u(43),et=u(227),rt=tt.get,nt=tt.set,it=J.f,ot=$.f,at=Math.round,st=g.RangeError,lt=x.ArrayBuffer,ct=x.DataView,ut=_.NATIVE_ARRAY_BUFFER_VIEWS,ft=_.TYPED_ARRAY_TAG,ht=_.TypedArray,dt=_.TypedArrayPrototype,pt=_.aTypedArrayConstructor,gt=_.isTypedArray,fromList=function(i,o){for(var u=0,p=o.length,g=new(pt(i))(p);p>u;)g[u]=o[u++];return g},addGetter=function(i,o){it(i,o,{get:function(){return rt(this)[o]}})},isArrayBuffer=function(i){var o;return i instanceof lt||"ArrayBuffer"==(o=N(i))||"SharedArrayBuffer"==o},isTypedArrayIndex=function(i,o){return gt(i)&&"symbol"!=typeof o&&o in i&&String(+o)==String(o)},yt=function getOwnPropertyDescriptor(i,o){return isTypedArrayIndex(i,o=D(o,!0))?P(2,i[o]):ot(i,o)},vt=function defineProperty(i,o,u){return!(isTypedArrayIndex(i,o=D(o,!0))&&U(u)&&R(u,"value"))||R(u,"get")||R(u,"set")||u.configurable||R(u,"writable")&&!u.writable||R(u,"enumerable")&&!u.enumerable?it(i,o,u):(i[o]=u.value,i)};y?(ut||($.f=yt,J.f=vt,addGetter(dt,"buffer"),addGetter(dt,"byteOffset"),addGetter(dt,"byteLength"),addGetter(dt,"length")),p({target:"Object",stat:!0,forced:!ut},{getOwnPropertyDescriptor:yt,defineProperty:vt}),i.exports=function(i,o,u){var y=i.match(/\d+$/)[0]/8,_=i+(u?"Clamped":"")+"Array",x="get"+i,P="set"+i,D=g[_],R=D,N=R&&R.prototype,J={},addElement=function(i,o){it(i,o,{get:function(){return function(i,o){var u=rt(i);return u.view[x](o*y+u.byteOffset,!0)}(this,o)},set:function(i){return function(i,o,p){var g=rt(i);u&&(p=(p=at(p))<0?0:p>255?255:255&p),g.view[P](o*y+g.byteOffset,p,!0)}(this,o,i)},enumerable:!0})};ut?w&&(R=o((function(i,o,u,p){return k(i,R,_),et(U(o)?isArrayBuffer(o)?void 0!==p?new D(o,B(u,y),p):void 0!==u?new D(o,B(u,y)):new D(o):gt(o)?fromList(R,o):X.call(R,o):new D(I(o)),i,R)})),G&&G(R,ht),K(j(D),(function(i){i in R||E(R,i,D[i])})),R.prototype=N):(R=o((function(i,o,u,p){k(i,R,_);var g,w,x,P=0,E=0;if(U(o)){if(!isArrayBuffer(o))return gt(o)?fromList(R,o):X.call(R,o);g=o,E=B(u,y);var D=o.byteLength;if(void 0===p){if(D%y)throw st("Wrong length");if((w=D-E)<0)throw st("Wrong length")}else if((w=O(p)*y)+E>D)throw st("Wrong length");x=w/y}else x=I(o),g=new lt(w=x*y);for(nt(i,{buffer:g,byteOffset:E,byteLength:w,length:x,view:new ct(g)});P<x;)addElement(i,P++)})),G&&G(R,ht),N=R.prototype=W(dt)),N.constructor!==R&&E(N,"constructor",R),ft&&E(N,ft,_),J[_]=R,p({global:!0,forced:R!=D,sham:!ut},J),"BYTES_PER_ELEMENT"in R||E(R,"BYTES_PER_ELEMENT",y),"BYTES_PER_ELEMENT"in N||E(N,"BYTES_PER_ELEMENT",y),Y(_)}):i.exports=function(){}},function(i,o){i.exports="undefined"!=typeof ArrayBuffer&&"undefined"!=typeof DataView},function(i,o,u){var p=u(25);i.exports=function(i,o,u){for(var g in o)p(i,g,o[g],u);return i}},function(i,o,u){var p=u(36),g=u(8);i.exports=function(i){if(void 0===i)return 0;var o=p(i),u=g(o);if(o!==u)throw RangeError("Wrong length or index");return u}},function(i,o,u){var p=u(339);i.exports=function(i,o){var u=p(i);if(u%o)throw RangeError("Wrong offset");return u}},function(i,o,u){var p=u(31),g=u(19),y=u(90),w=u(8),createMethod=function(i){return function(o,u,_,x){p(u);var k=g(o),P=y(k),E=w(k.length),O=i?E-1:0,I=i?-1:1;if(_<2)for(;;){if(O in P){x=P[O],O+=I;break}if(O+=I,i?O<0:E<=O)throw TypeError("Reduce of empty array with no initial value")}for(;i?O>=0:E>O;O+=I)O in P&&(x=u(x,P[O],O,k));return x}};i.exports={left:createMethod(!1),right:createMethod(!0)}},function(i,o,u){var p=u(1),g=u(166),y=u(153);p({target:"Array",proto:!0},{fill:g}),y("fill")},function(i,o,u){"use strict";var p=u(1),g=u(228).trim;p({target:"String",proto:!0,forced:u(347)("trim")},{trim:function trim(){return g(this)}})},function(i,o,u){"use strict";var p=u(1),g=u(162);p({target:"String",proto:!0,forced:u(163)("link")},{link:function link(i){return g(this,"a","href",i)}})},function(i,o,u){"use strict";(function(o,p){var g=u(125);i.exports=Readable;var y,w=u(201);Readable.ReadableState=ReadableState;u(124).EventEmitter;var EElistenerCount=function(i,o){return i.listeners(o).length},_=u(241),x=u(126).Buffer,k=o.Uint8Array||function(){};var P=Object.create(u(95));P.inherits=u(84);var E=u(349),O=void 0;O=E&&E.debuglog?E.debuglog("stream"):function(){};var I,B=u(350),D=u(242);P.inherits(Readable,_);var R=["error","close","destroy","pause","resume"];function ReadableState(i,o){i=i||{};var p=o instanceof(y=y||u(49));this.objectMode=!!i.objectMode,p&&(this.objectMode=this.objectMode||!!i.readableObjectMode);var g=i.highWaterMark,w=i.readableHighWaterMark,_=this.objectMode?16:16384;this.highWaterMark=g||0===g?g:p&&(w||0===w)?w:_,this.highWaterMark=Math.floor(this.highWaterMark),this.buffer=new B,this.length=0,this.pipes=null,this.pipesCount=0,this.flowing=null,this.ended=!1,this.endEmitted=!1,this.reading=!1,this.sync=!0,this.needReadable=!1,this.emittedReadable=!1,this.readableListening=!1,this.resumeScheduled=!1,this.destroyed=!1,this.defaultEncoding=i.defaultEncoding||"utf8",this.awaitDrain=0,this.readingMore=!1,this.decoder=null,this.encoding=null,i.encoding&&(I||(I=u(127).StringDecoder),this.decoder=new I(i.encoding),this.encoding=i.encoding)}function Readable(i){if(y=y||u(49),!(this instanceof Readable))return new Readable(i);this._readableState=new ReadableState(i,this),this.readable=!0,i&&("function"==typeof i.read&&(this._read=i.read),"function"==typeof i.destroy&&(this._destroy=i.destroy)),_.call(this)}function readableAddChunk(i,o,u,p,g){var y,w=i._readableState;null===o?(w.reading=!1,function onEofChunk(i,o){if(o.ended)return;if(o.decoder){var u=o.decoder.end();u&&u.length&&(o.buffer.push(u),o.length+=o.objectMode?1:u.length)}o.ended=!0,emitReadable(i)}(i,w)):(g||(y=function chunkInvalid(i,o){var u;(function _isUint8Array(i){return x.isBuffer(i)||i instanceof k})(o)||"string"==typeof o||void 0===o||i.objectMode||(u=new TypeError("Invalid non-string/buffer chunk"));return u}(w,o)),y?i.emit("error",y):w.objectMode||o&&o.length>0?("string"==typeof o||w.objectMode||Object.getPrototypeOf(o)===x.prototype||(o=function _uint8ArrayToBuffer(i){return x.from(i)}(o)),p?w.endEmitted?i.emit("error",new Error("stream.unshift() after end event")):addChunk(i,w,o,!0):w.ended?i.emit("error",new Error("stream.push() after EOF")):(w.reading=!1,w.decoder&&!u?(o=w.decoder.write(o),w.objectMode||0!==o.length?addChunk(i,w,o,!1):maybeReadMore(i,w)):addChunk(i,w,o,!1))):p||(w.reading=!1));return function needMoreData(i){return!i.ended&&(i.needReadable||i.length<i.highWaterMark||0===i.length)}(w)}function addChunk(i,o,u,p){o.flowing&&0===o.length&&!o.sync?(i.emit("data",u),i.read(0)):(o.length+=o.objectMode?1:u.length,p?o.buffer.unshift(u):o.buffer.push(u),o.needReadable&&emitReadable(i)),maybeReadMore(i,o)}Object.defineProperty(Readable.prototype,"destroyed",{get:function(){return void 0!==this._readableState&&this._readableState.destroyed},set:function(i){this._readableState&&(this._readableState.destroyed=i)}}),Readable.prototype.destroy=D.destroy,Readable.prototype._undestroy=D.undestroy,Readable.prototype._destroy=function(i,o){this.push(null),o(i)},Readable.prototype.push=function(i,o){var u,p=this._readableState;return p.objectMode?u=!0:"string"==typeof i&&((o=o||p.defaultEncoding)!==p.encoding&&(i=x.from(i,o),o=""),u=!0),readableAddChunk(this,i,o,!1,u)},Readable.prototype.unshift=function(i){return readableAddChunk(this,i,null,!0,!1)},Readable.prototype.isPaused=function(){return!1===this._readableState.flowing},Readable.prototype.setEncoding=function(i){return I||(I=u(127).StringDecoder),this._readableState.decoder=new I(i),this._readableState.encoding=i,this};function howMuchToRead(i,o){return i<=0||0===o.length&&o.ended?0:o.objectMode?1:i!=i?o.flowing&&o.length?o.buffer.head.data.length:o.length:(i>o.highWaterMark&&(o.highWaterMark=function computeNewHighWaterMark(i){return i>=8388608?i=8388608:(i--,i|=i>>>1,i|=i>>>2,i|=i>>>4,i|=i>>>8,i|=i>>>16,i++),i}(i)),i<=o.length?i:o.ended?o.length:(o.needReadable=!0,0))}function emitReadable(i){var o=i._readableState;o.needReadable=!1,o.emittedReadable||(O("emitReadable",o.flowing),o.emittedReadable=!0,o.sync?g.nextTick(emitReadable_,i):emitReadable_(i))}function emitReadable_(i){O("emit readable"),i.emit("readable"),flow(i)}function maybeReadMore(i,o){o.readingMore||(o.readingMore=!0,g.nextTick(maybeReadMore_,i,o))}function maybeReadMore_(i,o){for(var u=o.length;!o.reading&&!o.flowing&&!o.ended&&o.length<o.highWaterMark&&(O("maybeReadMore read 0"),i.read(0),u!==o.length);)u=o.length;o.readingMore=!1}function nReadingNextTick(i){O("readable nexttick read 0"),i.read(0)}function resume_(i,o){o.reading||(O("resume read 0"),i.read(0)),o.resumeScheduled=!1,o.awaitDrain=0,i.emit("resume"),flow(i),o.flowing&&!o.reading&&i.read(0)}function flow(i){var o=i._readableState;for(O("flow",o.flowing);o.flowing&&null!==i.read(););}function fromList(i,o){return 0===o.length?null:(o.objectMode?u=o.buffer.shift():!i||i>=o.length?(u=o.decoder?o.buffer.join(""):1===o.buffer.length?o.buffer.head.data:o.buffer.concat(o.length),o.buffer.clear()):u=function fromListPartial(i,o,u){var p;i<o.head.data.length?(p=o.head.data.slice(0,i),o.head.data=o.head.data.slice(i)):p=i===o.head.data.length?o.shift():u?function copyFromBufferString(i,o){var u=o.head,p=1,g=u.data;i-=g.length;for(;u=u.next;){var y=u.data,w=i>y.length?y.length:i;if(w===y.length?g+=y:g+=y.slice(0,i),0===(i-=w)){w===y.length?(++p,u.next?o.head=u.next:o.head=o.tail=null):(o.head=u,u.data=y.slice(w));break}++p}return o.length-=p,g}(i,o):function copyFromBuffer(i,o){var u=x.allocUnsafe(i),p=o.head,g=1;p.data.copy(u),i-=p.data.length;for(;p=p.next;){var y=p.data,w=i>y.length?y.length:i;if(y.copy(u,u.length-i,0,w),0===(i-=w)){w===y.length?(++g,p.next?o.head=p.next:o.head=o.tail=null):(o.head=p,p.data=y.slice(w));break}++g}return o.length-=g,u}(i,o);return p}(i,o.buffer,o.decoder),u);var u}function endReadable(i){var o=i._readableState;if(o.length>0)throw new Error('"endReadable()" called on non-empty stream');o.endEmitted||(o.ended=!0,g.nextTick(endReadableNT,o,i))}function endReadableNT(i,o){i.endEmitted||0!==i.length||(i.endEmitted=!0,o.readable=!1,o.emit("end"))}function indexOf(i,o){for(var u=0,p=i.length;u<p;u++)if(i[u]===o)return u;return-1}Readable.prototype.read=function(i){O("read",i),i=parseInt(i,10);var o=this._readableState,u=i;if(0!==i&&(o.emittedReadable=!1),0===i&&o.needReadable&&(o.length>=o.highWaterMark||o.ended))return O("read: emitReadable",o.length,o.ended),0===o.length&&o.ended?endReadable(this):emitReadable(this),null;if(0===(i=howMuchToRead(i,o))&&o.ended)return 0===o.length&&endReadable(this),null;var p,g=o.needReadable;return O("need readable",g),(0===o.length||o.length-i<o.highWaterMark)&&O("length less than watermark",g=!0),o.ended||o.reading?O("reading or ended",g=!1):g&&(O("do read"),o.reading=!0,o.sync=!0,0===o.length&&(o.needReadable=!0),this._read(o.highWaterMark),o.sync=!1,o.reading||(i=howMuchToRead(u,o))),null===(p=i>0?fromList(i,o):null)?(o.needReadable=!0,i=0):o.length-=i,0===o.length&&(o.ended||(o.needReadable=!0),u!==i&&o.ended&&endReadable(this)),null!==p&&this.emit("data",p),p},Readable.prototype._read=function(i){this.emit("error",new Error("_read() is not implemented"))},Readable.prototype.pipe=function(i,o){var u=this,y=this._readableState;switch(y.pipesCount){case 0:y.pipes=i;break;case 1:y.pipes=[y.pipes,i];break;default:y.pipes.push(i)}y.pipesCount+=1,O("pipe count=%d opts=%j",y.pipesCount,o);var _=(!o||!1!==o.end)&&i!==p.stdout&&i!==p.stderr?onend:unpipe;function onunpipe(o,p){O("onunpipe"),o===u&&p&&!1===p.hasUnpiped&&(p.hasUnpiped=!0,function cleanup(){O("cleanup"),i.removeListener("close",onclose),i.removeListener("finish",onfinish),i.removeListener("drain",x),i.removeListener("error",onerror),i.removeListener("unpipe",onunpipe),u.removeListener("end",onend),u.removeListener("end",unpipe),u.removeListener("data",ondata),k=!0,!y.awaitDrain||i._writableState&&!i._writableState.needDrain||x()}())}function onend(){O("onend"),i.end()}y.endEmitted?g.nextTick(_):u.once("end",_),i.on("unpipe",onunpipe);var x=function pipeOnDrain(i){return function(){var o=i._readableState;O("pipeOnDrain",o.awaitDrain),o.awaitDrain&&o.awaitDrain--,0===o.awaitDrain&&EElistenerCount(i,"data")&&(o.flowing=!0,flow(i))}}(u);i.on("drain",x);var k=!1;var P=!1;function ondata(o){O("ondata"),P=!1,!1!==i.write(o)||P||((1===y.pipesCount&&y.pipes===i||y.pipesCount>1&&-1!==indexOf(y.pipes,i))&&!k&&(O("false write response, pause",u._readableState.awaitDrain),u._readableState.awaitDrain++,P=!0),u.pause())}function onerror(o){O("onerror",o),unpipe(),i.removeListener("error",onerror),0===EElistenerCount(i,"error")&&i.emit("error",o)}function onclose(){i.removeListener("finish",onfinish),unpipe()}function onfinish(){O("onfinish"),i.removeListener("close",onclose),unpipe()}function unpipe(){O("unpipe"),u.unpipe(i)}return u.on("data",ondata),function prependListener(i,o,u){if("function"==typeof i.prependListener)return i.prependListener(o,u);i._events&&i._events[o]?w(i._events[o])?i._events[o].unshift(u):i._events[o]=[u,i._events[o]]:i.on(o,u)}(i,"error",onerror),i.once("close",onclose),i.once("finish",onfinish),i.emit("pipe",u),y.flowing||(O("pipe resume"),u.resume()),i},Readable.prototype.unpipe=function(i){var o=this._readableState,u={hasUnpiped:!1};if(0===o.pipesCount)return this;if(1===o.pipesCount)return i&&i!==o.pipes||(i||(i=o.pipes),o.pipes=null,o.pipesCount=0,o.flowing=!1,i&&i.emit("unpipe",this,u)),this;if(!i){var p=o.pipes,g=o.pipesCount;o.pipes=null,o.pipesCount=0,o.flowing=!1;for(var y=0;y<g;y++)p[y].emit("unpipe",this,u);return this}var w=indexOf(o.pipes,i);return-1===w||(o.pipes.splice(w,1),o.pipesCount-=1,1===o.pipesCount&&(o.pipes=o.pipes[0]),i.emit("unpipe",this,u)),this},Readable.prototype.on=function(i,o){var u=_.prototype.on.call(this,i,o);if("data"===i)!1!==this._readableState.flowing&&this.resume();else if("readable"===i){var p=this._readableState;p.endEmitted||p.readableListening||(p.readableListening=p.needReadable=!0,p.emittedReadable=!1,p.reading?p.length&&emitReadable(this):g.nextTick(nReadingNextTick,this))}return u},Readable.prototype.addListener=Readable.prototype.on,Readable.prototype.resume=function(){var i=this._readableState;return i.flowing||(O("resume"),i.flowing=!0,function resume(i,o){o.resumeScheduled||(o.resumeScheduled=!0,g.nextTick(resume_,i,o))}(this,i)),this},Readable.prototype.pause=function(){return O("call pause flowing=%j",this._readableState.flowing),!1!==this._readableState.flowing&&(O("pause"),this._readableState.flowing=!1,this.emit("pause")),this},Readable.prototype.wrap=function(i){var o=this,u=this._readableState,p=!1;for(var g in i.on("end",(function(){if(O("wrapped end"),u.decoder&&!u.ended){var i=u.decoder.end();i&&i.length&&o.push(i)}o.push(null)})),i.on("data",(function(g){(O("wrapped data"),u.decoder&&(g=u.decoder.write(g)),u.objectMode&&null==g)||(u.objectMode||g&&g.length)&&(o.push(g)||(p=!0,i.pause()))})),i)void 0===this[g]&&"function"==typeof i[g]&&(this[g]=function(o){return function(){return i[o].apply(i,arguments)}}(g));for(var y=0;y<R.length;y++)i.on(R[y],this.emit.bind(this,R[y]));return this._read=function(o){O("wrapped _read",o),p&&(p=!1,i.resume())},this},Object.defineProperty(Readable.prototype,"readableHighWaterMark",{enumerable:!1,get:function(){return this._readableState.highWaterMark}}),Readable._fromList=fromList}).call(this,u(27),u(48))},function(i,o,u){i.exports=u(124).EventEmitter},function(i,o,u){"use strict";var p=u(125);function emitErrorNT(i,o){i.emit("error",o)}i.exports={destroy:function destroy(i,o){var u=this,g=this._readableState&&this._readableState.destroyed,y=this._writableState&&this._writableState.destroyed;return g||y?(o?o(i):!i||this._writableState&&this._writableState.errorEmitted||p.nextTick(emitErrorNT,this,i),this):(this._readableState&&(this._readableState.destroyed=!0),this._writableState&&(this._writableState.destroyed=!0),this._destroy(i||null,(function(i){!o&&i?(p.nextTick(emitErrorNT,u,i),u._writableState&&(u._writableState.errorEmitted=!0)):o&&o(i)})),this)},undestroy:function undestroy(){this._readableState&&(this._readableState.destroyed=!1,this._readableState.reading=!1,this._readableState.ended=!1,this._readableState.endEmitted=!1),this._writableState&&(this._writableState.destroyed=!1,this._writableState.ended=!1,this._writableState.ending=!1,this._writableState.finished=!1,this._writableState.errorEmitted=!1)}}},function(i,o,u){"use strict";i.exports=Transform;var p=u(49),g=Object.create(u(95));function afterTransform(i,o){var u=this._transformState;u.transforming=!1;var p=u.writecb;if(!p)return this.emit("error",new Error("write callback called multiple times"));u.writechunk=null,u.writecb=null,null!=o&&this.push(o),p(i);var g=this._readableState;g.reading=!1,(g.needReadable||g.length<g.highWaterMark)&&this._read(g.highWaterMark)}function Transform(i){if(!(this instanceof Transform))return new Transform(i);p.call(this,i),this._transformState={afterTransform:afterTransform.bind(this),needTransform:!1,transforming:!1,writecb:null,writechunk:null,writeencoding:null},this._readableState.needReadable=!0,this._readableState.sync=!1,i&&("function"==typeof i.transform&&(this._transform=i.transform),"function"==typeof i.flush&&(this._flush=i.flush)),this.on("prefinish",prefinish)}function prefinish(){var i=this;"function"==typeof this._flush?this._flush((function(o,u){done(i,o,u)})):done(this,null,null)}function done(i,o,u){if(o)return i.emit("error",o);if(null!=u&&i.push(u),i._writableState.length)throw new Error("Calling transform done when ws.length != 0");if(i._transformState.transforming)throw new Error("Calling transform done when still transforming");return i.push(null)}g.inherits=u(84),g.inherits(Transform,p),Transform.prototype.push=function(i,o){return this._transformState.needTransform=!1,p.prototype.push.call(this,i,o)},Transform.prototype._transform=function(i,o,u){throw new Error("_transform() is not implemented")},Transform.prototype._write=function(i,o,u){var p=this._transformState;if(p.writecb=u,p.writechunk=i,p.writeencoding=o,!p.transforming){var g=this._readableState;(p.needTransform||g.needReadable||g.length<g.highWaterMark)&&this._read(g.highWaterMark)}},Transform.prototype._read=function(i){var o=this._transformState;null!==o.writechunk&&o.writecb&&!o.transforming?(o.transforming=!0,this._transform(o.writechunk,o.writeencoding,o.afterTransform)):o.needTransform=!0},Transform.prototype._destroy=function(i,o){var u=this;p.prototype._destroy.call(this,i,(function(i){o(i),u.emit("close")}))}},function(i,o,u){"use strict";(function(i){var p=u(10).Buffer,g=u(123).Transform,y=u(358),w=u(170),_=u(245).ok,x=u(10).kMaxLength,k="Cannot create final Buffer. It would be larger than 0x"+x.toString(16)+" bytes";y.Z_MIN_WINDOWBITS=8,y.Z_MAX_WINDOWBITS=15,y.Z_DEFAULT_WINDOWBITS=15,y.Z_MIN_CHUNK=64,y.Z_MAX_CHUNK=1/0,y.Z_DEFAULT_CHUNK=16384,y.Z_MIN_MEMLEVEL=1,y.Z_MAX_MEMLEVEL=9,y.Z_DEFAULT_MEMLEVEL=8,y.Z_MIN_LEVEL=-1,y.Z_MAX_LEVEL=9,y.Z_DEFAULT_LEVEL=y.Z_DEFAULT_COMPRESSION;for(var P=Object.keys(y),E=0;E<P.length;E++){var O=P[E];O.match(/^Z/)&&Object.defineProperty(o,O,{enumerable:!0,value:y[O],writable:!1})}for(var I={Z_OK:y.Z_OK,Z_STREAM_END:y.Z_STREAM_END,Z_NEED_DICT:y.Z_NEED_DICT,Z_ERRNO:y.Z_ERRNO,Z_STREAM_ERROR:y.Z_STREAM_ERROR,Z_DATA_ERROR:y.Z_DATA_ERROR,Z_MEM_ERROR:y.Z_MEM_ERROR,Z_BUF_ERROR:y.Z_BUF_ERROR,Z_VERSION_ERROR:y.Z_VERSION_ERROR},B=Object.keys(I),D=0;D<B.length;D++){var R=B[D];I[I[R]]=R}function zlibBuffer(i,o,u){var g=[],y=0;function flow(){for(var o;null!==(o=i.read());)g.push(o),y+=o.length;i.once("readable",flow)}function onEnd(){var o,w=null;y>=x?w=new RangeError(k):o=p.concat(g,y),g=[],i.close(),u(w,o)}i.on("error",(function onError(o){i.removeListener("end",onEnd),i.removeListener("readable",flow),u(o)})),i.on("end",onEnd),i.end(o),flow()}function zlibBufferSync(i,o){if("string"==typeof o&&(o=p.from(o)),!p.isBuffer(o))throw new TypeError("Not a string or buffer");var u=i._finishFlushFlag;return i._processChunk(o,u)}function Deflate(i){if(!(this instanceof Deflate))return new Deflate(i);Zlib.call(this,i,y.DEFLATE)}function Inflate(i){if(!(this instanceof Inflate))return new Inflate(i);Zlib.call(this,i,y.INFLATE)}function Gzip(i){if(!(this instanceof Gzip))return new Gzip(i);Zlib.call(this,i,y.GZIP)}function Gunzip(i){if(!(this instanceof Gunzip))return new Gunzip(i);Zlib.call(this,i,y.GUNZIP)}function DeflateRaw(i){if(!(this instanceof DeflateRaw))return new DeflateRaw(i);Zlib.call(this,i,y.DEFLATERAW)}function InflateRaw(i){if(!(this instanceof InflateRaw))return new InflateRaw(i);Zlib.call(this,i,y.INFLATERAW)}function Unzip(i){if(!(this instanceof Unzip))return new Unzip(i);Zlib.call(this,i,y.UNZIP)}function isValidFlushFlag(i){return i===y.Z_NO_FLUSH||i===y.Z_PARTIAL_FLUSH||i===y.Z_SYNC_FLUSH||i===y.Z_FULL_FLUSH||i===y.Z_FINISH||i===y.Z_BLOCK}function Zlib(i,u){var w=this;if(this._opts=i=i||{},this._chunkSize=i.chunkSize||o.Z_DEFAULT_CHUNK,g.call(this,i),i.flush&&!isValidFlushFlag(i.flush))throw new Error("Invalid flush flag: "+i.flush);if(i.finishFlush&&!isValidFlushFlag(i.finishFlush))throw new Error("Invalid flush flag: "+i.finishFlush);if(this._flushFlag=i.flush||y.Z_NO_FLUSH,this._finishFlushFlag=void 0!==i.finishFlush?i.finishFlush:y.Z_FINISH,i.chunkSize&&(i.chunkSize<o.Z_MIN_CHUNK||i.chunkSize>o.Z_MAX_CHUNK))throw new Error("Invalid chunk size: "+i.chunkSize);if(i.windowBits&&(i.windowBits<o.Z_MIN_WINDOWBITS||i.windowBits>o.Z_MAX_WINDOWBITS))throw new Error("Invalid windowBits: "+i.windowBits);if(i.level&&(i.level<o.Z_MIN_LEVEL||i.level>o.Z_MAX_LEVEL))throw new Error("Invalid compression level: "+i.level);if(i.memLevel&&(i.memLevel<o.Z_MIN_MEMLEVEL||i.memLevel>o.Z_MAX_MEMLEVEL))throw new Error("Invalid memLevel: "+i.memLevel);if(i.strategy&&i.strategy!=o.Z_FILTERED&&i.strategy!=o.Z_HUFFMAN_ONLY&&i.strategy!=o.Z_RLE&&i.strategy!=o.Z_FIXED&&i.strategy!=o.Z_DEFAULT_STRATEGY)throw new Error("Invalid strategy: "+i.strategy);if(i.dictionary&&!p.isBuffer(i.dictionary))throw new Error("Invalid dictionary: it should be a Buffer instance");this._handle=new y.Zlib(u);var _=this;this._hadError=!1,this._handle.onerror=function(i,u){_close(_),_._hadError=!0;var p=new Error(i);p.errno=u,p.code=o.codes[u],_.emit("error",p)};var x=o.Z_DEFAULT_COMPRESSION;"number"==typeof i.level&&(x=i.level);var k=o.Z_DEFAULT_STRATEGY;"number"==typeof i.strategy&&(k=i.strategy),this._handle.init(i.windowBits||o.Z_DEFAULT_WINDOWBITS,x,i.memLevel||o.Z_DEFAULT_MEMLEVEL,k,i.dictionary),this._buffer=p.allocUnsafe(this._chunkSize),this._offset=0,this._level=x,this._strategy=k,this.once("end",this.close),Object.defineProperty(this,"_closed",{get:function(){return!w._handle},configurable:!0,enumerable:!0})}function _close(o,u){u&&i.nextTick(u),o._handle&&(o._handle.close(),o._handle=null)}function emitCloseNT(i){i.emit("close")}Object.defineProperty(o,"codes",{enumerable:!0,value:Object.freeze(I),writable:!1}),o.Deflate=Deflate,o.Inflate=Inflate,o.Gzip=Gzip,o.Gunzip=Gunzip,o.DeflateRaw=DeflateRaw,o.InflateRaw=InflateRaw,o.Unzip=Unzip,o.createDeflate=function(i){return new Deflate(i)},o.createInflate=function(i){return new Inflate(i)},o.createDeflateRaw=function(i){return new DeflateRaw(i)},o.createInflateRaw=function(i){return new InflateRaw(i)},o.createGzip=function(i){return new Gzip(i)},o.createGunzip=function(i){return new Gunzip(i)},o.createUnzip=function(i){return new Unzip(i)},o.deflate=function(i,o,u){return"function"==typeof o&&(u=o,o={}),zlibBuffer(new Deflate(o),i,u)},o.deflateSync=function(i,o){return zlibBufferSync(new Deflate(o),i)},o.gzip=function(i,o,u){return"function"==typeof o&&(u=o,o={}),zlibBuffer(new Gzip(o),i,u)},o.gzipSync=function(i,o){return zlibBufferSync(new Gzip(o),i)},o.deflateRaw=function(i,o,u){return"function"==typeof o&&(u=o,o={}),zlibBuffer(new DeflateRaw(o),i,u)},o.deflateRawSync=function(i,o){return zlibBufferSync(new DeflateRaw(o),i)},o.unzip=function(i,o,u){return"function"==typeof o&&(u=o,o={}),zlibBuffer(new Unzip(o),i,u)},o.unzipSync=function(i,o){return zlibBufferSync(new Unzip(o),i)},o.inflate=function(i,o,u){return"function"==typeof o&&(u=o,o={}),zlibBuffer(new Inflate(o),i,u)},o.inflateSync=function(i,o){return zlibBufferSync(new Inflate(o),i)},o.gunzip=function(i,o,u){return"function"==typeof o&&(u=o,o={}),zlibBuffer(new Gunzip(o),i,u)},o.gunzipSync=function(i,o){return zlibBufferSync(new Gunzip(o),i)},o.inflateRaw=function(i,o,u){return"function"==typeof o&&(u=o,o={}),zlibBuffer(new InflateRaw(o),i,u)},o.inflateRawSync=function(i,o){return zlibBufferSync(new InflateRaw(o),i)},w.inherits(Zlib,g),Zlib.prototype.params=function(u,p,g){if(u<o.Z_MIN_LEVEL||u>o.Z_MAX_LEVEL)throw new RangeError("Invalid compression level: "+u);if(p!=o.Z_FILTERED&&p!=o.Z_HUFFMAN_ONLY&&p!=o.Z_RLE&&p!=o.Z_FIXED&&p!=o.Z_DEFAULT_STRATEGY)throw new TypeError("Invalid strategy: "+p);if(this._level!==u||this._strategy!==p){var w=this;this.flush(y.Z_SYNC_FLUSH,(function(){_(w._handle,"zlib binding closed"),w._handle.params(u,p),w._hadError||(w._level=u,w._strategy=p,g&&g())}))}else i.nextTick(g)},Zlib.prototype.reset=function(){return _(this._handle,"zlib binding closed"),this._handle.reset()},Zlib.prototype._flush=function(i){this._transform(p.alloc(0),"",i)},Zlib.prototype.flush=function(o,u){var g=this,w=this._writableState;("function"==typeof o||void 0===o&&!u)&&(u=o,o=y.Z_FULL_FLUSH),w.ended?u&&i.nextTick(u):w.ending?u&&this.once("end",u):w.needDrain?u&&this.once("drain",(function(){return g.flush(o,u)})):(this._flushFlag=o,this.write(p.alloc(0),"",u))},Zlib.prototype.close=function(o){_close(this,o),i.nextTick(emitCloseNT,this)},Zlib.prototype._transform=function(i,o,u){var g,w=this._writableState,_=(w.ending||w.ended)&&(!i||w.length===i.length);return null===i||p.isBuffer(i)?this._handle?(_?g=this._finishFlushFlag:(g=this._flushFlag,i.length>=w.length&&(this._flushFlag=this._opts.flush||y.Z_NO_FLUSH)),void this._processChunk(i,g,u)):u(new Error("zlib binding closed")):u(new Error("invalid input"))},Zlib.prototype._processChunk=function(i,o,u){var g=i&&i.length,y=this._chunkSize-this._offset,w=0,P=this,E="function"==typeof u;if(!E){var O,I=[],B=0;this.on("error",(function(i){O=i})),_(this._handle,"zlib binding closed");do{var D=this._handle.writeSync(o,i,w,g,this._buffer,this._offset,y)}while(!this._hadError&&callback(D[0],D[1]));if(this._hadError)throw O;if(B>=x)throw _close(this),new RangeError(k);var R=p.concat(I,B);return _close(this),R}_(this._handle,"zlib binding closed");var N=this._handle.write(o,i,w,g,this._buffer,this._offset,y);function callback(x,k){if(this&&(this.buffer=null,this.callback=null),!P._hadError){var O=y-k;if(_(O>=0,"have should not go down"),O>0){var D=P._buffer.slice(P._offset,P._offset+O);P._offset+=O,E?P.push(D):(I.push(D),B+=D.length)}if((0===k||P._offset>=P._chunkSize)&&(y=P._chunkSize,P._offset=0,P._buffer=p.allocUnsafe(P._chunkSize)),0===k){if(w+=g-x,g=x,!E)return!0;var R=P._handle.write(o,i,w,g,P._buffer,P._offset,P._chunkSize);return R.callback=callback,void(R.buffer=i)}if(!E)return!1;u()}}N.buffer=i,N.callback=callback},w.inherits(Deflate,Zlib),w.inherits(Inflate,Zlib),w.inherits(Gzip,Zlib),w.inherits(Gunzip,Zlib),w.inherits(DeflateRaw,Zlib),w.inherits(InflateRaw,Zlib),w.inherits(Unzip,Zlib)}).call(this,u(48))},function(i,o,u){"use strict";(function(o){var p=u(359);function compare(i,o){if(i===o)return 0;for(var u=i.length,p=o.length,g=0,y=Math.min(u,p);g<y;++g)if(i[g]!==o[g]){u=i[g],p=o[g];break}return u<p?-1:p<u?1:0}function isBuffer(i){return o.Buffer&&"function"==typeof o.Buffer.isBuffer?o.Buffer.isBuffer(i):!(null==i||!i._isBuffer)}var g=u(170),y=Object.prototype.hasOwnProperty,w=Array.prototype.slice,_="foo"===function foo(){}.name;function pToString(i){return Object.prototype.toString.call(i)}function isView(i){return!isBuffer(i)&&("function"==typeof o.ArrayBuffer&&("function"==typeof ArrayBuffer.isView?ArrayBuffer.isView(i):!!i&&(i instanceof DataView||!!(i.buffer&&i.buffer instanceof ArrayBuffer))))}var x=i.exports=ok,k=/\s*function\s+([^\(\s]*)\s*/;function getName(i){if(g.isFunction(i)){if(_)return i.name;var o=i.toString().match(k);return o&&o[1]}}function truncate(i,o){return"string"==typeof i?i.length<o?i:i.slice(0,o):i}function inspect(i){if(_||!g.isFunction(i))return g.inspect(i);var o=getName(i);return"[Function"+(o?": "+o:"")+"]"}function fail(i,o,u,p,g){throw new x.AssertionError({message:u,actual:i,expected:o,operator:p,stackStartFunction:g})}function ok(i,o){i||fail(i,!0,o,"==",x.ok)}function _deepEqual(i,o,u,p){if(i===o)return!0;if(isBuffer(i)&&isBuffer(o))return 0===compare(i,o);if(g.isDate(i)&&g.isDate(o))return i.getTime()===o.getTime();if(g.isRegExp(i)&&g.isRegExp(o))return i.source===o.source&&i.global===o.global&&i.multiline===o.multiline&&i.lastIndex===o.lastIndex&&i.ignoreCase===o.ignoreCase;if(null!==i&&"object"==typeof i||null!==o&&"object"==typeof o){if(isView(i)&&isView(o)&&pToString(i)===pToString(o)&&!(i instanceof Float32Array||i instanceof Float64Array))return 0===compare(new Uint8Array(i.buffer),new Uint8Array(o.buffer));if(isBuffer(i)!==isBuffer(o))return!1;var y=(p=p||{actual:[],expected:[]}).actual.indexOf(i);return-1!==y&&y===p.expected.indexOf(o)||(p.actual.push(i),p.expected.push(o),function objEquiv(i,o,u,p){if(null==i||null==o)return!1;if(g.isPrimitive(i)||g.isPrimitive(o))return i===o;if(u&&Object.getPrototypeOf(i)!==Object.getPrototypeOf(o))return!1;var y=isArguments(i),_=isArguments(o);if(y&&!_||!y&&_)return!1;if(y)return i=w.call(i),o=w.call(o),_deepEqual(i,o,u);var x,k,E=P(i),O=P(o);if(E.length!==O.length)return!1;for(E.sort(),O.sort(),k=E.length-1;k>=0;k--)if(E[k]!==O[k])return!1;for(k=E.length-1;k>=0;k--)if(x=E[k],!_deepEqual(i[x],o[x],u,p))return!1;return!0}(i,o,u,p))}return u?i===o:i==o}function isArguments(i){return"[object Arguments]"==Object.prototype.toString.call(i)}function expectedException(i,o){if(!i||!o)return!1;if("[object RegExp]"==Object.prototype.toString.call(o))return o.test(i);try{if(i instanceof o)return!0}catch(i){}return!Error.isPrototypeOf(o)&&!0===o.call({},i)}function _throws(i,o,u,p){var y;if("function"!=typeof o)throw new TypeError('"block" argument must be a function');"string"==typeof u&&(p=u,u=null),y=function _tryBlock(i){var o;try{i()}catch(i){o=i}return o}(o),p=(u&&u.name?" ("+u.name+").":".")+(p?" "+p:"."),i&&!y&&fail(y,u,"Missing expected exception"+p);var w="string"==typeof p,_=!i&&y&&!u;if((!i&&g.isError(y)&&w&&expectedException(y,u)||_)&&fail(y,u,"Got unwanted exception"+p),i&&y&&u&&!expectedException(y,u)||!i&&y)throw y}x.AssertionError=function AssertionError(i){this.name="AssertionError",this.actual=i.actual,this.expected=i.expected,this.operator=i.operator,i.message?(this.message=i.message,this.generatedMessage=!1):(this.message=function getMessage(i){return truncate(inspect(i.actual),128)+" "+i.operator+" "+truncate(inspect(i.expected),128)}(this),this.generatedMessage=!0);var o=i.stackStartFunction||fail;if(Error.captureStackTrace)Error.captureStackTrace(this,o);else{var u=new Error;if(u.stack){var p=u.stack,g=getName(o),y=p.indexOf("\n"+g);if(y>=0){var w=p.indexOf("\n",y+1);p=p.substring(w+1)}this.stack=p}}},g.inherits(x.AssertionError,Error),x.fail=fail,x.ok=ok,x.equal=function equal(i,o,u){i!=o&&fail(i,o,u,"==",x.equal)},x.notEqual=function notEqual(i,o,u){i==o&&fail(i,o,u,"!=",x.notEqual)},x.deepEqual=function deepEqual(i,o,u){_deepEqual(i,o,!1)||fail(i,o,u,"deepEqual",x.deepEqual)},x.deepStrictEqual=function deepStrictEqual(i,o,u){_deepEqual(i,o,!0)||fail(i,o,u,"deepStrictEqual",x.deepStrictEqual)},x.notDeepEqual=function notDeepEqual(i,o,u){_deepEqual(i,o,!1)&&fail(i,o,u,"notDeepEqual",x.notDeepEqual)},x.notDeepStrictEqual=function notDeepStrictEqual(i,o,u){_deepEqual(i,o,!0)&&fail(i,o,u,"notDeepStrictEqual",notDeepStrictEqual)},x.strictEqual=function strictEqual(i,o,u){i!==o&&fail(i,o,u,"===",x.strictEqual)},x.notStrictEqual=function notStrictEqual(i,o,u){i===o&&fail(i,o,u,"!==",x.notStrictEqual)},x.throws=function(i,o,u){_throws(!0,i,o,u)},x.doesNotThrow=function(i,o,u){_throws(!1,i,o,u)},x.ifError=function(i){if(i)throw i},x.strict=p((function strict(i,o){i||fail(i,!0,o,"==",strict)}),x,{equal:x.strictEqual,deepEqual:x.deepStrictEqual,notEqual:x.notStrictEqual,notDeepEqual:x.notDeepStrictEqual}),x.strict.strict=x.strict;var P=Object.keys||function(i){var o=[];for(var u in i)y.call(i,u)&&o.push(u);return o}}).call(this,u(27))},function(i,o,u){"use strict";i.exports=function adler32(i,o,u,p){for(var g=65535&i|0,y=i>>>16&65535|0,w=0;0!==u;){u-=w=u>2e3?2e3:u;do{y=y+(g=g+o[p++]|0)|0}while(--w);g%=65521,y%=65521}return g|y<<16|0}},function(i,o,u){"use strict";var p=function makeTable(){for(var i,o=[],u=0;u<256;u++){i=u;for(var p=0;p<8;p++)i=1&i?3988292384^i>>>1:i>>>1;o[u]=i}return o}();i.exports=function crc32(i,o,u,g){var y=p,w=g+u;i^=-1;for(var _=g;_<w;_++)i=i>>>8^y[255&(i^o[_])];return-1^i}},function(i,o,u){var p;i.exports=(p=u(2),function(i){var o=p,u=o.lib,g=u.WordArray,y=u.Hasher,w=o.algo,_=[],x=[];!function(){function isPrime(o){for(var u=i.sqrt(o),p=2;p<=u;p++)if(!(o%p))return!1;return!0}function getFractionalBits(i){return 4294967296*(i-(0|i))|0}for(var o=2,u=0;u<64;)isPrime(o)&&(u<8&&(_[u]=getFractionalBits(i.pow(o,.5))),x[u]=getFractionalBits(i.pow(o,1/3)),u++),o++}();var k=[],P=w.SHA256=y.extend({_doReset:function(){this._hash=new g.init(_.slice(0))},_doProcessBlock:function(i,o){for(var u=this._hash.words,p=u[0],g=u[1],y=u[2],w=u[3],_=u[4],P=u[5],E=u[6],O=u[7],I=0;I<64;I++){if(I<16)k[I]=0|i[o+I];else{var B=k[I-15],D=(B<<25|B>>>7)^(B<<14|B>>>18)^B>>>3,R=k[I-2],N=(R<<15|R>>>17)^(R<<13|R>>>19)^R>>>10;k[I]=D+k[I-7]+N+k[I-16]}var U=p&g^p&y^g&y,W=(p<<30|p>>>2)^(p<<19|p>>>13)^(p<<10|p>>>22),G=O+((_<<26|_>>>6)^(_<<21|_>>>11)^(_<<7|_>>>25))+(_&P^~_&E)+x[I]+k[I];O=E,E=P,P=_,_=w+G|0,w=y,y=g,g=p,p=G+(W+U)|0}u[0]=u[0]+p|0,u[1]=u[1]+g|0,u[2]=u[2]+y|0,u[3]=u[3]+w|0,u[4]=u[4]+_|0,u[5]=u[5]+P|0,u[6]=u[6]+E|0,u[7]=u[7]+O|0},_doFinalize:function(){var o=this._data,u=o.words,p=8*this._nDataBytes,g=8*o.sigBytes;return u[g>>>5]|=128<<24-g%32,u[14+(g+64>>>9<<4)]=i.floor(p/4294967296),u[15+(g+64>>>9<<4)]=p,o.sigBytes=4*u.length,this._process(),this._hash},clone:function(){var i=y.clone.call(this);return i._hash=this._hash.clone(),i}});o.SHA256=y._createHelper(P),o.HmacSHA256=y._createHmacHelper(P)}(Math),p.SHA256)},function(i,o,u){var p;i.exports=(p=u(2),u(129),function(){var i=p,o=i.lib.Hasher,u=i.x64,g=u.Word,y=u.WordArray,w=i.algo;function X64Word_create(){return g.create.apply(g,arguments)}var _=[X64Word_create(1116352408,3609767458),X64Word_create(1899447441,602891725),X64Word_create(3049323471,3964484399),X64Word_create(3921009573,2173295548),X64Word_create(961987163,4081628472),X64Word_create(1508970993,3053834265),X64Word_create(2453635748,2937671579),X64Word_create(2870763221,3664609560),X64Word_create(3624381080,2734883394),X64Word_create(310598401,1164996542),X64Word_create(607225278,1323610764),X64Word_create(1426881987,3590304994),X64Word_create(1925078388,4068182383),X64Word_create(2162078206,991336113),X64Word_create(2614888103,633803317),X64Word_create(3248222580,3479774868),X64Word_create(3835390401,2666613458),X64Word_create(4022224774,944711139),X64Word_create(264347078,2341262773),X64Word_create(604807628,2007800933),X64Word_create(770255983,1495990901),X64Word_create(1249150122,1856431235),X64Word_create(1555081692,3175218132),X64Word_create(1996064986,2198950837),X64Word_create(2554220882,3999719339),X64Word_create(2821834349,766784016),X64Word_create(2952996808,2566594879),X64Word_create(3210313671,3203337956),X64Word_create(3336571891,1034457026),X64Word_create(3584528711,2466948901),X64Word_create(113926993,3758326383),X64Word_create(338241895,168717936),X64Word_create(666307205,1188179964),X64Word_create(773529912,1546045734),X64Word_create(1294757372,1522805485),X64Word_create(1396182291,2643833823),X64Word_create(1695183700,2343527390),X64Word_create(1986661051,1014477480),X64Word_create(2177026350,1206759142),X64Word_create(2456956037,344077627),X64Word_create(2730485921,1290863460),X64Word_create(2820302411,3158454273),X64Word_create(3259730800,3505952657),X64Word_create(3345764771,106217008),X64Word_create(3516065817,3606008344),X64Word_create(3600352804,1432725776),X64Word_create(4094571909,1467031594),X64Word_create(275423344,851169720),X64Word_create(430227734,3100823752),X64Word_create(506948616,1363258195),X64Word_create(659060556,3750685593),X64Word_create(883997877,3785050280),X64Word_create(958139571,3318307427),X64Word_create(1322822218,3812723403),X64Word_create(1537002063,2003034995),X64Word_create(1747873779,3602036899),X64Word_create(1955562222,1575990012),X64Word_create(2024104815,1125592928),X64Word_create(2227730452,2716904306),X64Word_create(2361852424,442776044),X64Word_create(2428436474,593698344),X64Word_create(2756734187,3733110249),X64Word_create(3204031479,2999351573),X64Word_create(3329325298,3815920427),X64Word_create(3391569614,3928383900),X64Word_create(3515267271,566280711),X64Word_create(3940187606,3454069534),X64Word_create(4118630271,4000239992),X64Word_create(116418474,1914138554),X64Word_create(174292421,2731055270),X64Word_create(289380356,3203993006),X64Word_create(460393269,320620315),X64Word_create(685471733,587496836),X64Word_create(852142971,1086792851),X64Word_create(1017036298,365543100),X64Word_create(1126000580,2618297676),X64Word_create(1288033470,3409855158),X64Word_create(1501505948,4234509866),X64Word_create(1607167915,987167468),X64Word_create(1816402316,1246189591)],x=[];!function(){for(var i=0;i<80;i++)x[i]=X64Word_create()}();var k=w.SHA512=o.extend({_doReset:function(){this._hash=new y.init([new g.init(1779033703,4089235720),new g.init(3144134277,2227873595),new g.init(1013904242,4271175723),new g.init(2773480762,1595750129),new g.init(1359893119,2917565137),new g.init(2600822924,725511199),new g.init(528734635,4215389547),new g.init(1541459225,327033209)])},_doProcessBlock:function(i,o){for(var u=this._hash.words,p=u[0],g=u[1],y=u[2],w=u[3],k=u[4],P=u[5],E=u[6],O=u[7],I=p.high,B=p.low,D=g.high,R=g.low,N=y.high,U=y.low,W=w.high,G=w.low,j=k.high,X=k.low,K=P.high,Y=P.low,J=E.high,$=E.low,tt=O.high,et=O.low,rt=I,nt=B,it=D,ot=R,at=N,st=U,lt=W,ct=G,ut=j,ft=X,ht=K,dt=Y,pt=J,gt=$,yt=tt,vt=et,mt=0;mt<80;mt++){var bt=x[mt];if(mt<16)var wt=bt.high=0|i[o+2*mt],_t=bt.low=0|i[o+2*mt+1];else{var xt=x[mt-15],St=xt.high,Ct=xt.low,kt=(St>>>1|Ct<<31)^(St>>>8|Ct<<24)^St>>>7,At=(Ct>>>1|St<<31)^(Ct>>>8|St<<24)^(Ct>>>7|St<<25),Pt=x[mt-2],Tt=Pt.high,Et=Pt.low,Ot=(Tt>>>19|Et<<13)^(Tt<<3|Et>>>29)^Tt>>>6,It=(Et>>>19|Tt<<13)^(Et<<3|Tt>>>29)^(Et>>>6|Tt<<26),Bt=x[mt-7],Ft=Bt.high,Dt=Bt.low,Lt=x[mt-16],Mt=Lt.high,Rt=Lt.low;wt=(wt=(wt=kt+Ft+((_t=At+Dt)>>>0<At>>>0?1:0))+Ot+((_t+=It)>>>0<It>>>0?1:0))+Mt+((_t+=Rt)>>>0<Rt>>>0?1:0),bt.high=wt,bt.low=_t}var Nt,Ut=ut&ht^~ut&pt,Wt=ft&dt^~ft&gt,zt=rt&it^rt&at^it&at,Gt=nt&ot^nt&st^ot&st,jt=(rt>>>28|nt<<4)^(rt<<30|nt>>>2)^(rt<<25|nt>>>7),Vt=(nt>>>28|rt<<4)^(nt<<30|rt>>>2)^(nt<<25|rt>>>7),Ht=(ut>>>14|ft<<18)^(ut>>>18|ft<<14)^(ut<<23|ft>>>9),qt=(ft>>>14|ut<<18)^(ft>>>18|ut<<14)^(ft<<23|ut>>>9),Xt=_[mt],Zt=Xt.high,Kt=Xt.low,Yt=yt+Ht+((Nt=vt+qt)>>>0<vt>>>0?1:0),Jt=Vt+Gt;yt=pt,vt=gt,pt=ht,gt=dt,ht=ut,dt=ft,ut=lt+(Yt=(Yt=(Yt=Yt+Ut+((Nt+=Wt)>>>0<Wt>>>0?1:0))+Zt+((Nt+=Kt)>>>0<Kt>>>0?1:0))+wt+((Nt+=_t)>>>0<_t>>>0?1:0))+((ft=ct+Nt|0)>>>0<ct>>>0?1:0)|0,lt=at,ct=st,at=it,st=ot,it=rt,ot=nt,rt=Yt+(jt+zt+(Jt>>>0<Vt>>>0?1:0))+((nt=Nt+Jt|0)>>>0<Nt>>>0?1:0)|0}B=p.low=B+nt,p.high=I+rt+(B>>>0<nt>>>0?1:0),R=g.low=R+ot,g.high=D+it+(R>>>0<ot>>>0?1:0),U=y.low=U+st,y.high=N+at+(U>>>0<st>>>0?1:0),G=w.low=G+ct,w.high=W+lt+(G>>>0<ct>>>0?1:0),X=k.low=X+ft,k.high=j+ut+(X>>>0<ft>>>0?1:0),Y=P.low=Y+dt,P.high=K+ht+(Y>>>0<dt>>>0?1:0),$=E.low=$+gt,E.high=J+pt+($>>>0<gt>>>0?1:0),et=O.low=et+vt,O.high=tt+yt+(et>>>0<vt>>>0?1:0)},_doFinalize:function(){var i=this._data,o=i.words,u=8*this._nDataBytes,p=8*i.sigBytes;return o[p>>>5]|=128<<24-p%32,o[30+(p+128>>>10<<5)]=Math.floor(u/4294967296),o[31+(p+128>>>10<<5)]=u,i.sigBytes=4*o.length,this._process(),this._hash.toX32()},clone:function(){var i=o.clone.call(this);return i._hash=this._hash.clone(),i},blockSize:32});i.SHA512=o._createHelper(k),i.HmacSHA512=o._createHmacHelper(k)}(),p.SHA512)},function(i){i.exports=JSON.parse('[["a140","",62],["a180","",32],["a240","",62],["a280","",32],["a2ab","",5],["a2e3","€"],["a2ef",""],["a2fd",""],["a340","",62],["a380","",31,"　"],["a440","",62],["a480","",32],["a4f4","",10],["a540","",62],["a580","",32],["a5f7","",7],["a640","",62],["a680","",32],["a6b9","",7],["a6d9","",6],["a6ec",""],["a6f3",""],["a6f6","",8],["a740","",62],["a780","",32],["a7c2","",14],["a7f2","",12],["a896","",10],["a8bc","ḿ"],["a8bf","ǹ"],["a8c1",""],["a8ea","",20],["a958",""],["a95b",""],["a95d",""],["a989","〾⿰",11],["a997","",12],["a9f0","",14],["aaa1","",93],["aba1","",93],["aca1","",93],["ada1","",93],["aea1","",93],["afa1","",93],["d7fa","",4],["f8a1","",93],["f9a1","",93],["faa1","",93],["fba1","",93],["fca1","",93],["fda1","",93],["fe50","⺁⺄㑳㑇⺈⺋㖞㘚㘎⺌⺗㥮㤘㧏㧟㩳㧐㭎㱮㳠⺧⺪䁖䅟⺮䌷⺳⺶⺷䎱䎬⺻䏝䓖䙡䙌"],["fe80","䜣䜩䝼䞍⻊䥇䥺䥽䦂䦃䦅䦆䦟䦛䦷䦶䲣䲟䲠䲡䱷䲢䴓",6,"䶮",93],["8135f437",""]]')},function(i){i.exports=JSON.parse('[["0","\\u0000",127],["a140","　，、。．‧；：？！︰…‥﹐﹑﹒·﹔﹕﹖﹗｜–︱—︳╴︴﹏（）︵︶｛｝︷︸〔〕︹︺【】︻︼《》︽︾〈〉︿﹀「」﹁﹂『』﹃﹄﹙﹚"],["a1a1","﹛﹜﹝﹞‘’“”〝〞‵′＃＆＊※§〃○●△▲◎☆★◇◆□■▽▼㊣℅¯￣＿ˍ﹉﹊﹍﹎﹋﹌﹟﹠﹡＋－×÷±√＜＞＝≦≧≠∞≒≡﹢",4,"～∩∪⊥∠∟⊿㏒㏑∫∮∵∴♀♂⊕⊙↑↓←→↖↗↙↘∥∣／"],["a240","＼∕﹨＄￥〒￠￡％＠℃℉﹩﹪﹫㏕㎜㎝㎞㏎㎡㎎㎏㏄°兙兛兞兝兡兣嗧瓩糎▁",7,"▏▎▍▌▋▊▉┼┴┬┤├▔─│▕┌┐└┘╭"],["a2a1","╮╰╯═╞╪╡◢◣◥◤╱╲╳０",9,"Ⅰ",9,"〡",8,"十卄卅Ａ",25,"ａ",21],["a340","ｗｘｙｚΑ",16,"Σ",6,"α",16,"σ",6,"ㄅ",10],["a3a1","ㄐ",25,"˙ˉˊˇˋ"],["a3e1","€"],["a440","一乙丁七乃九了二人儿入八几刀刁力匕十卜又三下丈上丫丸凡久么也乞于亡兀刃勺千叉口土士夕大女子孑孓寸小尢尸山川工己已巳巾干廾弋弓才"],["a4a1","丑丐不中丰丹之尹予云井互五亢仁什仃仆仇仍今介仄元允內六兮公冗凶分切刈勻勾勿化匹午升卅卞厄友及反壬天夫太夭孔少尤尺屯巴幻廿弔引心戈戶手扎支文斗斤方日曰月木欠止歹毋比毛氏水火爪父爻片牙牛犬王丙"],["a540","世丕且丘主乍乏乎以付仔仕他仗代令仙仞充兄冉冊冬凹出凸刊加功包匆北匝仟半卉卡占卯卮去可古右召叮叩叨叼司叵叫另只史叱台句叭叻四囚外"],["a5a1","央失奴奶孕它尼巨巧左市布平幼弁弘弗必戊打扔扒扑斥旦朮本未末札正母民氐永汁汀氾犯玄玉瓜瓦甘生用甩田由甲申疋白皮皿目矛矢石示禾穴立丞丟乒乓乩亙交亦亥仿伉伙伊伕伍伐休伏仲件任仰仳份企伋光兇兆先全"],["a640","共再冰列刑划刎刖劣匈匡匠印危吉吏同吊吐吁吋各向名合吃后吆吒因回囝圳地在圭圬圯圩夙多夷夸妄奸妃好她如妁字存宇守宅安寺尖屹州帆并年"],["a6a1","式弛忙忖戎戌戍成扣扛托收早旨旬旭曲曳有朽朴朱朵次此死氖汝汗汙江池汐汕污汛汍汎灰牟牝百竹米糸缶羊羽老考而耒耳聿肉肋肌臣自至臼舌舛舟艮色艾虫血行衣西阡串亨位住佇佗佞伴佛何估佐佑伽伺伸佃佔似但佣"],["a740","作你伯低伶余佝佈佚兌克免兵冶冷別判利刪刨劫助努劬匣即卵吝吭吞吾否呎吧呆呃吳呈呂君吩告吹吻吸吮吵吶吠吼呀吱含吟听囪困囤囫坊坑址坍"],["a7a1","均坎圾坐坏圻壯夾妝妒妨妞妣妙妖妍妤妓妊妥孝孜孚孛完宋宏尬局屁尿尾岐岑岔岌巫希序庇床廷弄弟彤形彷役忘忌志忍忱快忸忪戒我抄抗抖技扶抉扭把扼找批扳抒扯折扮投抓抑抆改攻攸旱更束李杏材村杜杖杞杉杆杠"],["a840","杓杗步每求汞沙沁沈沉沅沛汪決沐汰沌汨沖沒汽沃汲汾汴沆汶沍沔沘沂灶灼災灸牢牡牠狄狂玖甬甫男甸皂盯矣私秀禿究系罕肖肓肝肘肛肚育良芒"],["a8a1","芋芍見角言谷豆豕貝赤走足身車辛辰迂迆迅迄巡邑邢邪邦那酉釆里防阮阱阪阬並乖乳事些亞享京佯依侍佳使佬供例來侃佰併侈佩佻侖佾侏侑佺兔兒兕兩具其典冽函刻券刷刺到刮制剁劾劻卒協卓卑卦卷卸卹取叔受味呵"],["a940","咖呸咕咀呻呷咄咒咆呼咐呱呶和咚呢周咋命咎固垃坷坪坩坡坦坤坼夜奉奇奈奄奔妾妻委妹妮姑姆姐姍始姓姊妯妳姒姅孟孤季宗定官宜宙宛尚屈居"],["a9a1","屆岷岡岸岩岫岱岳帘帚帖帕帛帑幸庚店府底庖延弦弧弩往征彿彼忝忠忽念忿怏怔怯怵怖怪怕怡性怩怫怛或戕房戾所承拉拌拄抿拂抹拒招披拓拔拋拈抨抽押拐拙拇拍抵拚抱拘拖拗拆抬拎放斧於旺昔易昌昆昂明昀昏昕昊"],["aa40","昇服朋杭枋枕東果杳杷枇枝林杯杰板枉松析杵枚枓杼杪杲欣武歧歿氓氛泣注泳沱泌泥河沽沾沼波沫法泓沸泄油況沮泗泅泱沿治泡泛泊沬泯泜泖泠"],["aaa1","炕炎炒炊炙爬爭爸版牧物狀狎狙狗狐玩玨玟玫玥甽疝疙疚的盂盲直知矽社祀祁秉秈空穹竺糾罔羌羋者肺肥肢肱股肫肩肴肪肯臥臾舍芳芝芙芭芽芟芹花芬芥芯芸芣芰芾芷虎虱初表軋迎返近邵邸邱邶采金長門阜陀阿阻附"],["ab40","陂隹雨青非亟亭亮信侵侯便俠俑俏保促侶俘俟俊俗侮俐俄係俚俎俞侷兗冒冑冠剎剃削前剌剋則勇勉勃勁匍南卻厚叛咬哀咨哎哉咸咦咳哇哂咽咪品"],["aba1","哄哈咯咫咱咻咩咧咿囿垂型垠垣垢城垮垓奕契奏奎奐姜姘姿姣姨娃姥姪姚姦威姻孩宣宦室客宥封屎屏屍屋峙峒巷帝帥帟幽庠度建弈弭彥很待徊律徇後徉怒思怠急怎怨恍恰恨恢恆恃恬恫恪恤扁拜挖按拼拭持拮拽指拱拷"],["ac40","拯括拾拴挑挂政故斫施既春昭映昧是星昨昱昤曷柿染柱柔某柬架枯柵柩柯柄柑枴柚查枸柏柞柳枰柙柢柝柒歪殃殆段毒毗氟泉洋洲洪流津洌洱洞洗"],["aca1","活洽派洶洛泵洹洧洸洩洮洵洎洫炫為炳炬炯炭炸炮炤爰牲牯牴狩狠狡玷珊玻玲珍珀玳甚甭畏界畎畋疫疤疥疢疣癸皆皇皈盈盆盃盅省盹相眉看盾盼眇矜砂研砌砍祆祉祈祇禹禺科秒秋穿突竿竽籽紂紅紀紉紇約紆缸美羿耄"],["ad40","耐耍耑耶胖胥胚胃胄背胡胛胎胞胤胝致舢苧范茅苣苛苦茄若茂茉苒苗英茁苜苔苑苞苓苟苯茆虐虹虻虺衍衫要觔計訂訃貞負赴赳趴軍軌述迦迢迪迥"],["ada1","迭迫迤迨郊郎郁郃酋酊重閂限陋陌降面革韋韭音頁風飛食首香乘亳倌倍倣俯倦倥俸倩倖倆值借倚倒們俺倀倔倨俱倡個候倘俳修倭倪俾倫倉兼冤冥冢凍凌准凋剖剜剔剛剝匪卿原厝叟哨唐唁唷哼哥哲唆哺唔哩哭員唉哮哪"],["ae40","哦唧唇哽唏圃圄埂埔埋埃堉夏套奘奚娑娘娜娟娛娓姬娠娣娩娥娌娉孫屘宰害家宴宮宵容宸射屑展屐峭峽峻峪峨峰島崁峴差席師庫庭座弱徒徑徐恙"],["aea1","恣恥恐恕恭恩息悄悟悚悍悔悌悅悖扇拳挈拿捎挾振捕捂捆捏捉挺捐挽挪挫挨捍捌效敉料旁旅時晉晏晃晒晌晅晁書朔朕朗校核案框桓根桂桔栩梳栗桌桑栽柴桐桀格桃株桅栓栘桁殊殉殷氣氧氨氦氤泰浪涕消涇浦浸海浙涓"],["af40","浬涉浮浚浴浩涌涊浹涅浥涔烊烘烤烙烈烏爹特狼狹狽狸狷玆班琉珮珠珪珞畔畝畜畚留疾病症疲疳疽疼疹痂疸皋皰益盍盎眩真眠眨矩砰砧砸砝破砷"],["afa1","砥砭砠砟砲祕祐祠祟祖神祝祗祚秤秣秧租秦秩秘窄窈站笆笑粉紡紗紋紊素索純紐紕級紜納紙紛缺罟羔翅翁耆耘耕耙耗耽耿胱脂胰脅胭胴脆胸胳脈能脊胼胯臭臬舀舐航舫舨般芻茫荒荔荊茸荐草茵茴荏茲茹茶茗荀茱茨荃"],["b040","虔蚊蚪蚓蚤蚩蚌蚣蚜衰衷袁袂衽衹記訐討訌訕訊託訓訖訏訑豈豺豹財貢起躬軒軔軏辱送逆迷退迺迴逃追逅迸邕郡郝郢酒配酌釘針釗釜釙閃院陣陡"],["b0a1","陛陝除陘陞隻飢馬骨高鬥鬲鬼乾偺偽停假偃偌做偉健偶偎偕偵側偷偏倏偯偭兜冕凰剪副勒務勘動匐匏匙匿區匾參曼商啪啦啄啞啡啃啊唱啖問啕唯啤唸售啜唬啣唳啁啗圈國圉域堅堊堆埠埤基堂堵執培夠奢娶婁婉婦婪婀"],["b140","娼婢婚婆婊孰寇寅寄寂宿密尉專將屠屜屝崇崆崎崛崖崢崑崩崔崙崤崧崗巢常帶帳帷康庸庶庵庾張強彗彬彩彫得徙從徘御徠徜恿患悉悠您惋悴惦悽"],["b1a1","情悻悵惜悼惘惕惆惟悸惚惇戚戛扈掠控捲掖探接捷捧掘措捱掩掉掃掛捫推掄授掙採掬排掏掀捻捩捨捺敝敖救教敗啟敏敘敕敔斜斛斬族旋旌旎晝晚晤晨晦晞曹勗望梁梯梢梓梵桿桶梱梧梗械梃棄梭梆梅梔條梨梟梡梂欲殺"],["b240","毫毬氫涎涼淳淙液淡淌淤添淺清淇淋涯淑涮淞淹涸混淵淅淒渚涵淚淫淘淪深淮淨淆淄涪淬涿淦烹焉焊烽烯爽牽犁猜猛猖猓猙率琅琊球理現琍瓠瓶"],["b2a1","瓷甜產略畦畢異疏痔痕疵痊痍皎盔盒盛眷眾眼眶眸眺硫硃硎祥票祭移窒窕笠笨笛第符笙笞笮粒粗粕絆絃統紮紹紼絀細紳組累終紲紱缽羞羚翌翎習耜聊聆脯脖脣脫脩脰脤舂舵舷舶船莎莞莘荸莢莖莽莫莒莊莓莉莠荷荻荼"],["b340","莆莧處彪蛇蛀蚶蛄蚵蛆蛋蚱蚯蛉術袞袈被袒袖袍袋覓規訪訝訣訥許設訟訛訢豉豚販責貫貨貪貧赧赦趾趺軛軟這逍通逗連速逝逐逕逞造透逢逖逛途"],["b3a1","部郭都酗野釵釦釣釧釭釩閉陪陵陳陸陰陴陶陷陬雀雪雩章竟頂頃魚鳥鹵鹿麥麻傢傍傅備傑傀傖傘傚最凱割剴創剩勞勝勛博厥啻喀喧啼喊喝喘喂喜喪喔喇喋喃喳單喟唾喲喚喻喬喱啾喉喫喙圍堯堪場堤堰報堡堝堠壹壺奠"],["b440","婷媚婿媒媛媧孳孱寒富寓寐尊尋就嵌嵐崴嵇巽幅帽幀幃幾廊廁廂廄弼彭復循徨惑惡悲悶惠愜愣惺愕惰惻惴慨惱愎惶愉愀愒戟扉掣掌描揀揩揉揆揍"],["b4a1","插揣提握揖揭揮捶援揪換摒揚揹敞敦敢散斑斐斯普晰晴晶景暑智晾晷曾替期朝棺棕棠棘棗椅棟棵森棧棹棒棲棣棋棍植椒椎棉棚楮棻款欺欽殘殖殼毯氮氯氬港游湔渡渲湧湊渠渥渣減湛湘渤湖湮渭渦湯渴湍渺測湃渝渾滋"],["b540","溉渙湎湣湄湲湩湟焙焚焦焰無然煮焜牌犄犀猶猥猴猩琺琪琳琢琥琵琶琴琯琛琦琨甥甦畫番痢痛痣痙痘痞痠登發皖皓皴盜睏短硝硬硯稍稈程稅稀窘"],["b5a1","窗窖童竣等策筆筐筒答筍筋筏筑粟粥絞結絨絕紫絮絲絡給絢絰絳善翔翕耋聒肅腕腔腋腑腎脹腆脾腌腓腴舒舜菩萃菸萍菠菅萋菁華菱菴著萊菰萌菌菽菲菊萸萎萄菜萇菔菟虛蛟蛙蛭蛔蛛蛤蛐蛞街裁裂袱覃視註詠評詞証詁"],["b640","詔詛詐詆訴診訶詖象貂貯貼貳貽賁費賀貴買貶貿貸越超趁跎距跋跚跑跌跛跆軻軸軼辜逮逵週逸進逶鄂郵鄉郾酣酥量鈔鈕鈣鈉鈞鈍鈐鈇鈑閔閏開閑"],["b6a1","間閒閎隊階隋陽隅隆隍陲隄雁雅雄集雇雯雲韌項順須飧飪飯飩飲飭馮馭黃黍黑亂傭債傲傳僅傾催傷傻傯僇剿剷剽募勦勤勢勣匯嗟嗨嗓嗦嗎嗜嗇嗑嗣嗤嗯嗚嗡嗅嗆嗥嗉園圓塞塑塘塗塚塔填塌塭塊塢塒塋奧嫁嫉嫌媾媽媼"],["b740","媳嫂媲嵩嵯幌幹廉廈弒彙徬微愚意慈感想愛惹愁愈慎慌慄慍愾愴愧愍愆愷戡戢搓搾搞搪搭搽搬搏搜搔損搶搖搗搆敬斟新暗暉暇暈暖暄暘暍會榔業"],["b7a1","楚楷楠楔極椰概楊楨楫楞楓楹榆楝楣楛歇歲毀殿毓毽溢溯滓溶滂源溝滇滅溥溘溼溺溫滑準溜滄滔溪溧溴煎煙煩煤煉照煜煬煦煌煥煞煆煨煖爺牒猷獅猿猾瑯瑚瑕瑟瑞瑁琿瑙瑛瑜當畸瘀痰瘁痲痱痺痿痴痳盞盟睛睫睦睞督"],["b840","睹睪睬睜睥睨睢矮碎碰碗碘碌碉硼碑碓硿祺祿禁萬禽稜稚稠稔稟稞窟窠筷節筠筮筧粱粳粵經絹綑綁綏絛置罩罪署義羨群聖聘肆肄腱腰腸腥腮腳腫"],["b8a1","腹腺腦舅艇蒂葷落萱葵葦葫葉葬葛萼萵葡董葩葭葆虞虜號蛹蜓蜈蜇蜀蛾蛻蜂蜃蜆蜊衙裟裔裙補裘裝裡裊裕裒覜解詫該詳試詩詰誇詼詣誠話誅詭詢詮詬詹詻訾詨豢貊貉賊資賈賄貲賃賂賅跡跟跨路跳跺跪跤跦躲較載軾輊"],["b940","辟農運遊道遂達逼違遐遇遏過遍遑逾遁鄒鄗酬酪酩釉鈷鉗鈸鈽鉀鈾鉛鉋鉤鉑鈴鉉鉍鉅鈹鈿鉚閘隘隔隕雍雋雉雊雷電雹零靖靴靶預頑頓頊頒頌飼飴"],["b9a1","飽飾馳馱馴髡鳩麂鼎鼓鼠僧僮僥僖僭僚僕像僑僱僎僩兢凳劃劂匱厭嗾嘀嘛嘗嗽嘔嘆嘉嘍嘎嗷嘖嘟嘈嘐嗶團圖塵塾境墓墊塹墅塽壽夥夢夤奪奩嫡嫦嫩嫗嫖嫘嫣孵寞寧寡寥實寨寢寤察對屢嶄嶇幛幣幕幗幔廓廖弊彆彰徹慇"],["ba40","愿態慷慢慣慟慚慘慵截撇摘摔撤摸摟摺摑摧搴摭摻敲斡旗旖暢暨暝榜榨榕槁榮槓構榛榷榻榫榴槐槍榭槌榦槃榣歉歌氳漳演滾漓滴漩漾漠漬漏漂漢"],["baa1","滿滯漆漱漸漲漣漕漫漯澈漪滬漁滲滌滷熔熙煽熊熄熒爾犒犖獄獐瑤瑣瑪瑰瑭甄疑瘧瘍瘋瘉瘓盡監瞄睽睿睡磁碟碧碳碩碣禎福禍種稱窪窩竭端管箕箋筵算箝箔箏箸箇箄粹粽精綻綰綜綽綾綠緊綴網綱綺綢綿綵綸維緒緇綬"],["bb40","罰翠翡翟聞聚肇腐膀膏膈膊腿膂臧臺與舔舞艋蓉蒿蓆蓄蒙蒞蒲蒜蓋蒸蓀蓓蒐蒼蓑蓊蜿蜜蜻蜢蜥蜴蜘蝕蜷蜩裳褂裴裹裸製裨褚裯誦誌語誣認誡誓誤"],["bba1","說誥誨誘誑誚誧豪貍貌賓賑賒赫趙趕跼輔輒輕輓辣遠遘遜遣遙遞遢遝遛鄙鄘鄞酵酸酷酴鉸銀銅銘銖鉻銓銜銨鉼銑閡閨閩閣閥閤隙障際雌雒需靼鞅韶頗領颯颱餃餅餌餉駁骯骰髦魁魂鳴鳶鳳麼鼻齊億儀僻僵價儂儈儉儅凜"],["bc40","劇劈劉劍劊勰厲嘮嘻嘹嘲嘿嘴嘩噓噎噗噴嘶嘯嘰墀墟增墳墜墮墩墦奭嬉嫻嬋嫵嬌嬈寮寬審寫層履嶝嶔幢幟幡廢廚廟廝廣廠彈影德徵慶慧慮慝慕憂"],["bca1","慼慰慫慾憧憐憫憎憬憚憤憔憮戮摩摯摹撞撲撈撐撰撥撓撕撩撒撮播撫撚撬撙撢撳敵敷數暮暫暴暱樣樟槨樁樞標槽模樓樊槳樂樅槭樑歐歎殤毅毆漿潼澄潑潦潔澆潭潛潸潮澎潺潰潤澗潘滕潯潠潟熟熬熱熨牖犛獎獗瑩璋璃"],["bd40","瑾璀畿瘠瘩瘟瘤瘦瘡瘢皚皺盤瞎瞇瞌瞑瞋磋磅確磊碾磕碼磐稿稼穀稽稷稻窯窮箭箱範箴篆篇篁箠篌糊締練緯緻緘緬緝編緣線緞緩綞緙緲緹罵罷羯"],["bda1","翩耦膛膜膝膠膚膘蔗蔽蔚蓮蔬蔭蔓蔑蔣蔡蔔蓬蔥蓿蔆螂蝴蝶蝠蝦蝸蝨蝙蝗蝌蝓衛衝褐複褒褓褕褊誼諒談諄誕請諸課諉諂調誰論諍誶誹諛豌豎豬賠賞賦賤賬賭賢賣賜質賡赭趟趣踫踐踝踢踏踩踟踡踞躺輝輛輟輩輦輪輜輞"],["be40","輥適遮遨遭遷鄰鄭鄧鄱醇醉醋醃鋅銻銷鋪銬鋤鋁銳銼鋒鋇鋰銲閭閱霄霆震霉靠鞍鞋鞏頡頫頜颳養餓餒餘駝駐駟駛駑駕駒駙骷髮髯鬧魅魄魷魯鴆鴉"],["bea1","鴃麩麾黎墨齒儒儘儔儐儕冀冪凝劑劓勳噙噫噹噩噤噸噪器噥噱噯噬噢噶壁墾壇壅奮嬝嬴學寰導彊憲憑憩憊懍憶憾懊懈戰擅擁擋撻撼據擄擇擂操撿擒擔撾整曆曉暹曄曇暸樽樸樺橙橫橘樹橄橢橡橋橇樵機橈歙歷氅濂澱澡"],["bf40","濃澤濁澧澳激澹澶澦澠澴熾燉燐燒燈燕熹燎燙燜燃燄獨璜璣璘璟璞瓢甌甍瘴瘸瘺盧盥瞠瞞瞟瞥磨磚磬磧禦積穎穆穌穋窺篙簑築篤篛篡篩篦糕糖縊"],["bfa1","縑縈縛縣縞縝縉縐罹羲翰翱翮耨膳膩膨臻興艘艙蕊蕙蕈蕨蕩蕃蕉蕭蕪蕞螃螟螞螢融衡褪褲褥褫褡親覦諦諺諫諱謀諜諧諮諾謁謂諷諭諳諶諼豫豭貓賴蹄踱踴蹂踹踵輻輯輸輳辨辦遵遴選遲遼遺鄴醒錠錶鋸錳錯錢鋼錫錄錚"],["c040","錐錦錡錕錮錙閻隧隨險雕霎霑霖霍霓霏靛靜靦鞘頰頸頻頷頭頹頤餐館餞餛餡餚駭駢駱骸骼髻髭鬨鮑鴕鴣鴦鴨鴒鴛默黔龍龜優償儡儲勵嚎嚀嚐嚅嚇"],["c0a1","嚏壕壓壑壎嬰嬪嬤孺尷屨嶼嶺嶽嶸幫彌徽應懂懇懦懋戲戴擎擊擘擠擰擦擬擱擢擭斂斃曙曖檀檔檄檢檜櫛檣橾檗檐檠歜殮毚氈濘濱濟濠濛濤濫濯澀濬濡濩濕濮濰燧營燮燦燥燭燬燴燠爵牆獰獲璩環璦璨癆療癌盪瞳瞪瞰瞬"],["c140","瞧瞭矯磷磺磴磯礁禧禪穗窿簇簍篾篷簌篠糠糜糞糢糟糙糝縮績繆縷縲繃縫總縱繅繁縴縹繈縵縿縯罄翳翼聱聲聰聯聳臆臃膺臂臀膿膽臉膾臨舉艱薪"],["c1a1","薄蕾薜薑薔薯薛薇薨薊虧蟀蟑螳蟒蟆螫螻螺蟈蟋褻褶襄褸褽覬謎謗謙講謊謠謝謄謐豁谿豳賺賽購賸賻趨蹉蹋蹈蹊轄輾轂轅輿避遽還邁邂邀鄹醣醞醜鍍鎂錨鍵鍊鍥鍋錘鍾鍬鍛鍰鍚鍔闊闋闌闈闆隱隸雖霜霞鞠韓顆颶餵騁"],["c240","駿鮮鮫鮪鮭鴻鴿麋黏點黜黝黛鼾齋叢嚕嚮壙壘嬸彝懣戳擴擲擾攆擺擻擷斷曜朦檳檬櫃檻檸櫂檮檯歟歸殯瀉瀋濾瀆濺瀑瀏燻燼燾燸獷獵璧璿甕癖癘"],["c2a1","癒瞽瞿瞻瞼礎禮穡穢穠竄竅簫簧簪簞簣簡糧織繕繞繚繡繒繙罈翹翻職聶臍臏舊藏薩藍藐藉薰薺薹薦蟯蟬蟲蟠覆覲觴謨謹謬謫豐贅蹙蹣蹦蹤蹟蹕軀轉轍邇邃邈醫醬釐鎔鎊鎖鎢鎳鎮鎬鎰鎘鎚鎗闔闖闐闕離雜雙雛雞霤鞣鞦"],["c340","鞭韹額顏題顎顓颺餾餿餽餮馥騎髁鬃鬆魏魎魍鯊鯉鯽鯈鯀鵑鵝鵠黠鼕鼬儳嚥壞壟壢寵龐廬懲懷懶懵攀攏曠曝櫥櫝櫚櫓瀛瀟瀨瀚瀝瀕瀘爆爍牘犢獸"],["c3a1","獺璽瓊瓣疇疆癟癡矇礙禱穫穩簾簿簸簽簷籀繫繭繹繩繪羅繳羶羹羸臘藩藝藪藕藤藥藷蟻蠅蠍蟹蟾襠襟襖襞譁譜識證譚譎譏譆譙贈贊蹼蹲躇蹶蹬蹺蹴轔轎辭邊邋醱醮鏡鏑鏟鏃鏈鏜鏝鏖鏢鏍鏘鏤鏗鏨關隴難霪霧靡韜韻類"],["c440","願顛颼饅饉騖騙鬍鯨鯧鯖鯛鶉鵡鵲鵪鵬麒麗麓麴勸嚨嚷嚶嚴嚼壤孀孃孽寶巉懸懺攘攔攙曦朧櫬瀾瀰瀲爐獻瓏癢癥礦礪礬礫竇競籌籃籍糯糰辮繽繼"],["c4a1","纂罌耀臚艦藻藹蘑藺蘆蘋蘇蘊蠔蠕襤覺觸議譬警譯譟譫贏贍躉躁躅躂醴釋鐘鐃鏽闡霰飄饒饑馨騫騰騷騵鰓鰍鹹麵黨鼯齟齣齡儷儸囁囀囂夔屬巍懼懾攝攜斕曩櫻欄櫺殲灌爛犧瓖瓔癩矓籐纏續羼蘗蘭蘚蠣蠢蠡蠟襪襬覽譴"],["c540","護譽贓躊躍躋轟辯醺鐮鐳鐵鐺鐸鐲鐫闢霸霹露響顧顥饗驅驃驀騾髏魔魑鰭鰥鶯鶴鷂鶸麝黯鼙齜齦齧儼儻囈囊囉孿巔巒彎懿攤權歡灑灘玀瓤疊癮癬"],["c5a1","禳籠籟聾聽臟襲襯觼讀贖贗躑躓轡酈鑄鑑鑒霽霾韃韁顫饕驕驍髒鬚鱉鰱鰾鰻鷓鷗鼴齬齪龔囌巖戀攣攫攪曬欐瓚竊籤籣籥纓纖纔臢蘸蘿蠱變邐邏鑣鑠鑤靨顯饜驚驛驗髓體髑鱔鱗鱖鷥麟黴囑壩攬灞癱癲矗罐羈蠶蠹衢讓讒"],["c640","讖艷贛釀鑪靂靈靄韆顰驟鬢魘鱟鷹鷺鹼鹽鼇齷齲廳欖灣籬籮蠻觀躡釁鑲鑰顱饞髖鬣黌灤矚讚鑷韉驢驥纜讜躪釅鑽鑾鑼鱷鱸黷豔鑿鸚爨驪鬱鸛鸞籲"],["c940","乂乜凵匚厂万丌乇亍囗兀屮彳丏冇与丮亓仂仉仈冘勼卬厹圠夃夬尐巿旡殳毌气爿丱丼仨仜仩仡仝仚刌匜卌圢圣夗夯宁宄尒尻屴屳帄庀庂忉戉扐氕"],["c9a1","氶汃氿氻犮犰玊禸肊阞伎优伬仵伔仱伀价伈伝伂伅伢伓伄仴伒冱刓刉刐劦匢匟卍厊吇囡囟圮圪圴夼妀奼妅奻奾奷奿孖尕尥屼屺屻屾巟幵庄异弚彴忕忔忏扜扞扤扡扦扢扙扠扚扥旯旮朾朹朸朻机朿朼朳氘汆汒汜汏汊汔汋"],["ca40","汌灱牞犴犵玎甪癿穵网艸艼芀艽艿虍襾邙邗邘邛邔阢阤阠阣佖伻佢佉体佤伾佧佒佟佁佘伭伳伿佡冏冹刜刞刡劭劮匉卣卲厎厏吰吷吪呔呅吙吜吥吘"],["caa1","吽呏呁吨吤呇囮囧囥坁坅坌坉坋坒夆奀妦妘妠妗妎妢妐妏妧妡宎宒尨尪岍岏岈岋岉岒岊岆岓岕巠帊帎庋庉庌庈庍弅弝彸彶忒忑忐忭忨忮忳忡忤忣忺忯忷忻怀忴戺抃抌抎抏抔抇扱扻扺扰抁抈扷扽扲扴攷旰旴旳旲旵杅杇"],["cb40","杙杕杌杈杝杍杚杋毐氙氚汸汧汫沄沋沏汱汯汩沚汭沇沕沜汦汳汥汻沎灴灺牣犿犽狃狆狁犺狅玕玗玓玔玒町甹疔疕皁礽耴肕肙肐肒肜芐芏芅芎芑芓"],["cba1","芊芃芄豸迉辿邟邡邥邞邧邠阰阨阯阭丳侘佼侅佽侀侇佶佴侉侄佷佌侗佪侚佹侁佸侐侜侔侞侒侂侕佫佮冞冼冾刵刲刳剆刱劼匊匋匼厒厔咇呿咁咑咂咈呫呺呾呥呬呴呦咍呯呡呠咘呣呧呤囷囹坯坲坭坫坱坰坶垀坵坻坳坴坢"],["cc40","坨坽夌奅妵妺姏姎妲姌姁妶妼姃姖妱妽姀姈妴姇孢孥宓宕屄屇岮岤岠岵岯岨岬岟岣岭岢岪岧岝岥岶岰岦帗帔帙弨弢弣弤彔徂彾彽忞忥怭怦怙怲怋"],["cca1","怴怊怗怳怚怞怬怢怍怐怮怓怑怌怉怜戔戽抭抴拑抾抪抶拊抮抳抯抻抩抰抸攽斨斻昉旼昄昒昈旻昃昋昍昅旽昑昐曶朊枅杬枎枒杶杻枘枆构杴枍枌杺枟枑枙枃杽极杸杹枔欥殀歾毞氝沓泬泫泮泙沶泔沭泧沷泐泂沺泃泆泭泲"],["cd40","泒泝沴沊沝沀泞泀洰泍泇沰泹泏泩泑炔炘炅炓炆炄炑炖炂炚炃牪狖狋狘狉狜狒狔狚狌狑玤玡玭玦玢玠玬玝瓝瓨甿畀甾疌疘皯盳盱盰盵矸矼矹矻矺"],["cda1","矷祂礿秅穸穻竻籵糽耵肏肮肣肸肵肭舠芠苀芫芚芘芛芵芧芮芼芞芺芴芨芡芩苂芤苃芶芢虰虯虭虮豖迒迋迓迍迖迕迗邲邴邯邳邰阹阽阼阺陃俍俅俓侲俉俋俁俔俜俙侻侳俛俇俖侺俀侹俬剄剉勀勂匽卼厗厖厙厘咺咡咭咥哏"],["ce40","哃茍咷咮哖咶哅哆咠呰咼咢咾呲哞咰垵垞垟垤垌垗垝垛垔垘垏垙垥垚垕壴复奓姡姞姮娀姱姝姺姽姼姶姤姲姷姛姩姳姵姠姾姴姭宨屌峐峘峌峗峋峛"],["cea1","峞峚峉峇峊峖峓峔峏峈峆峎峟峸巹帡帢帣帠帤庰庤庢庛庣庥弇弮彖徆怷怹恔恲恞恅恓恇恉恛恌恀恂恟怤恄恘恦恮扂扃拏挍挋拵挎挃拫拹挏挌拸拶挀挓挔拺挕拻拰敁敃斪斿昶昡昲昵昜昦昢昳昫昺昝昴昹昮朏朐柁柲柈枺"],["cf40","柜枻柸柘柀枷柅柫柤柟枵柍枳柷柶柮柣柂枹柎柧柰枲柼柆柭柌枮柦柛柺柉柊柃柪柋欨殂殄殶毖毘毠氠氡洨洴洭洟洼洿洒洊泚洳洄洙洺洚洑洀洝浂"],["cfa1","洁洘洷洃洏浀洇洠洬洈洢洉洐炷炟炾炱炰炡炴炵炩牁牉牊牬牰牳牮狊狤狨狫狟狪狦狣玅珌珂珈珅玹玶玵玴珫玿珇玾珃珆玸珋瓬瓮甮畇畈疧疪癹盄眈眃眄眅眊盷盻盺矧矨砆砑砒砅砐砏砎砉砃砓祊祌祋祅祄秕种秏秖秎窀"],["d040","穾竑笀笁籺籸籹籿粀粁紃紈紁罘羑羍羾耇耎耏耔耷胘胇胠胑胈胂胐胅胣胙胜胊胕胉胏胗胦胍臿舡芔苙苾苹茇苨茀苕茺苫苖苴苬苡苲苵茌苻苶苰苪"],["d0a1","苤苠苺苳苭虷虴虼虳衁衎衧衪衩觓訄訇赲迣迡迮迠郱邽邿郕郅邾郇郋郈釔釓陔陏陑陓陊陎倞倅倇倓倢倰倛俵俴倳倷倬俶俷倗倜倠倧倵倯倱倎党冔冓凊凄凅凈凎剡剚剒剞剟剕剢勍匎厞唦哢唗唒哧哳哤唚哿唄唈哫唑唅哱"],["d140","唊哻哷哸哠唎唃唋圁圂埌堲埕埒垺埆垽垼垸垶垿埇埐垹埁夎奊娙娖娭娮娕娏娗娊娞娳孬宧宭宬尃屖屔峬峿峮峱峷崀峹帩帨庨庮庪庬弳弰彧恝恚恧"],["d1a1","恁悢悈悀悒悁悝悃悕悛悗悇悜悎戙扆拲挐捖挬捄捅挶捃揤挹捋捊挼挩捁挴捘捔捙挭捇挳捚捑挸捗捀捈敊敆旆旃旄旂晊晟晇晑朒朓栟栚桉栲栳栻桋桏栖栱栜栵栫栭栯桎桄栴栝栒栔栦栨栮桍栺栥栠欬欯欭欱欴歭肂殈毦毤"],["d240","毨毣毢毧氥浺浣浤浶洍浡涒浘浢浭浯涑涍淯浿涆浞浧浠涗浰浼浟涂涘洯浨涋浾涀涄洖涃浻浽浵涐烜烓烑烝烋缹烢烗烒烞烠烔烍烅烆烇烚烎烡牂牸"],["d2a1","牷牶猀狺狴狾狶狳狻猁珓珙珥珖玼珧珣珩珜珒珛珔珝珚珗珘珨瓞瓟瓴瓵甡畛畟疰痁疻痄痀疿疶疺皊盉眝眛眐眓眒眣眑眕眙眚眢眧砣砬砢砵砯砨砮砫砡砩砳砪砱祔祛祏祜祓祒祑秫秬秠秮秭秪秜秞秝窆窉窅窋窌窊窇竘笐"],["d340","笄笓笅笏笈笊笎笉笒粄粑粊粌粈粍粅紞紝紑紎紘紖紓紟紒紏紌罜罡罞罠罝罛羖羒翃翂翀耖耾耹胺胲胹胵脁胻脀舁舯舥茳茭荄茙荑茥荖茿荁茦茜茢"],["d3a1","荂荎茛茪茈茼荍茖茤茠茷茯茩荇荅荌荓茞茬荋茧荈虓虒蚢蚨蚖蚍蚑蚞蚇蚗蚆蚋蚚蚅蚥蚙蚡蚧蚕蚘蚎蚝蚐蚔衃衄衭衵衶衲袀衱衿衯袃衾衴衼訒豇豗豻貤貣赶赸趵趷趶軑軓迾迵适迿迻逄迼迶郖郠郙郚郣郟郥郘郛郗郜郤酐"],["d440","酎酏釕釢釚陜陟隼飣髟鬯乿偰偪偡偞偠偓偋偝偲偈偍偁偛偊偢倕偅偟偩偫偣偤偆偀偮偳偗偑凐剫剭剬剮勖勓匭厜啵啶唼啍啐唴唪啑啢唶唵唰啒啅"],["d4a1","唌唲啥啎唹啈唭唻啀啋圊圇埻堔埢埶埜埴堀埭埽堈埸堋埳埏堇埮埣埲埥埬埡堎埼堐埧堁堌埱埩埰堍堄奜婠婘婕婧婞娸娵婭婐婟婥婬婓婤婗婃婝婒婄婛婈媎娾婍娹婌婰婩婇婑婖婂婜孲孮寁寀屙崞崋崝崚崠崌崨崍崦崥崏"],["d540","崰崒崣崟崮帾帴庱庴庹庲庳弶弸徛徖徟悊悐悆悾悰悺惓惔惏惤惙惝惈悱惛悷惊悿惃惍惀挲捥掊掂捽掽掞掭掝掗掫掎捯掇掐据掯捵掜捭掮捼掤挻掟"],["d5a1","捸掅掁掑掍捰敓旍晥晡晛晙晜晢朘桹梇梐梜桭桮梮梫楖桯梣梬梩桵桴梲梏桷梒桼桫桲梪梀桱桾梛梖梋梠梉梤桸桻梑梌梊桽欶欳欷欸殑殏殍殎殌氪淀涫涴涳湴涬淩淢涷淶淔渀淈淠淟淖涾淥淜淝淛淴淊涽淭淰涺淕淂淏淉"],["d640","淐淲淓淽淗淍淣涻烺焍烷焗烴焌烰焄烳焐烼烿焆焓焀烸烶焋焂焎牾牻牼牿猝猗猇猑猘猊猈狿猏猞玈珶珸珵琄琁珽琇琀珺珼珿琌琋珴琈畤畣痎痒痏"],["d6a1","痋痌痑痐皏皉盓眹眯眭眱眲眴眳眽眥眻眵硈硒硉硍硊硌砦硅硐祤祧祩祪祣祫祡离秺秸秶秷窏窔窐笵筇笴笥笰笢笤笳笘笪笝笱笫笭笯笲笸笚笣粔粘粖粣紵紽紸紶紺絅紬紩絁絇紾紿絊紻紨罣羕羜羝羛翊翋翍翐翑翇翏翉耟"],["d740","耞耛聇聃聈脘脥脙脛脭脟脬脞脡脕脧脝脢舑舸舳舺舴舲艴莐莣莨莍荺荳莤荴莏莁莕莙荵莔莩荽莃莌莝莛莪莋荾莥莯莈莗莰荿莦莇莮荶莚虙虖蚿蚷"],["d7a1","蛂蛁蛅蚺蚰蛈蚹蚳蚸蛌蚴蚻蚼蛃蚽蚾衒袉袕袨袢袪袚袑袡袟袘袧袙袛袗袤袬袌袓袎覂觖觙觕訰訧訬訞谹谻豜豝豽貥赽赻赹趼跂趹趿跁軘軞軝軜軗軠軡逤逋逑逜逌逡郯郪郰郴郲郳郔郫郬郩酖酘酚酓酕釬釴釱釳釸釤釹釪"],["d840","釫釷釨釮镺閆閈陼陭陫陱陯隿靪頄飥馗傛傕傔傞傋傣傃傌傎傝偨傜傒傂傇兟凔匒匑厤厧喑喨喥喭啷噅喢喓喈喏喵喁喣喒喤啽喌喦啿喕喡喎圌堩堷"],["d8a1","堙堞堧堣堨埵塈堥堜堛堳堿堶堮堹堸堭堬堻奡媯媔媟婺媢媞婸媦婼媥媬媕媮娷媄媊媗媃媋媩婻婽媌媜媏媓媝寪寍寋寔寑寊寎尌尰崷嵃嵫嵁嵋崿崵嵑嵎嵕崳崺嵒崽崱嵙嵂崹嵉崸崼崲崶嵀嵅幄幁彘徦徥徫惉悹惌惢惎惄愔"],["d940","惲愊愖愅惵愓惸惼惾惁愃愘愝愐惿愄愋扊掔掱掰揎揥揨揯揃撝揳揊揠揶揕揲揵摡揟掾揝揜揄揘揓揂揇揌揋揈揰揗揙攲敧敪敤敜敨敥斌斝斞斮旐旒"],["d9a1","晼晬晻暀晱晹晪晲朁椌棓椄棜椪棬棪棱椏棖棷棫棤棶椓椐棳棡椇棌椈楰梴椑棯棆椔棸棐棽棼棨椋椊椗棎棈棝棞棦棴棑椆棔棩椕椥棇欹欻欿欼殔殗殙殕殽毰毲毳氰淼湆湇渟湉溈渼渽湅湢渫渿湁湝湳渜渳湋湀湑渻渃渮湞"],["da40","湨湜湡渱渨湠湱湫渹渢渰湓湥渧湸湤湷湕湹湒湦渵渶湚焠焞焯烻焮焱焣焥焢焲焟焨焺焛牋牚犈犉犆犅犋猒猋猰猢猱猳猧猲猭猦猣猵猌琮琬琰琫琖"],["daa1","琚琡琭琱琤琣琝琩琠琲瓻甯畯畬痧痚痡痦痝痟痤痗皕皒盚睆睇睄睍睅睊睎睋睌矞矬硠硤硥硜硭硱硪确硰硩硨硞硢祴祳祲祰稂稊稃稌稄窙竦竤筊笻筄筈筌筎筀筘筅粢粞粨粡絘絯絣絓絖絧絪絏絭絜絫絒絔絩絑絟絎缾缿罥"],["db40","罦羢羠羡翗聑聏聐胾胔腃腊腒腏腇脽腍脺臦臮臷臸臹舄舼舽舿艵茻菏菹萣菀菨萒菧菤菼菶萐菆菈菫菣莿萁菝菥菘菿菡菋菎菖菵菉萉萏菞萑萆菂菳"],["dba1","菕菺菇菑菪萓菃菬菮菄菻菗菢萛菛菾蛘蛢蛦蛓蛣蛚蛪蛝蛫蛜蛬蛩蛗蛨蛑衈衖衕袺裗袹袸裀袾袶袼袷袽袲褁裉覕覘覗觝觚觛詎詍訹詙詀詗詘詄詅詒詈詑詊詌詏豟貁貀貺貾貰貹貵趄趀趉跘跓跍跇跖跜跏跕跙跈跗跅軯軷軺"],["dc40","軹軦軮軥軵軧軨軶軫軱軬軴軩逭逴逯鄆鄬鄄郿郼鄈郹郻鄁鄀鄇鄅鄃酡酤酟酢酠鈁鈊鈥鈃鈚鈦鈏鈌鈀鈒釿釽鈆鈄鈧鈂鈜鈤鈙鈗鈅鈖镻閍閌閐隇陾隈"],["dca1","隉隃隀雂雈雃雱雰靬靰靮頇颩飫鳦黹亃亄亶傽傿僆傮僄僊傴僈僂傰僁傺傱僋僉傶傸凗剺剸剻剼嗃嗛嗌嗐嗋嗊嗝嗀嗔嗄嗩喿嗒喍嗏嗕嗢嗖嗈嗲嗍嗙嗂圔塓塨塤塏塍塉塯塕塎塝塙塥塛堽塣塱壼嫇嫄嫋媺媸媱媵媰媿嫈媻嫆"],["dd40","媷嫀嫊媴媶嫍媹媐寖寘寙尟尳嵱嵣嵊嵥嵲嵬嵞嵨嵧嵢巰幏幎幊幍幋廅廌廆廋廇彀徯徭惷慉慊愫慅愶愲愮慆愯慏愩慀戠酨戣戥戤揅揱揫搐搒搉搠搤"],["dda1","搳摃搟搕搘搹搷搢搣搌搦搰搨摁搵搯搊搚摀搥搧搋揧搛搮搡搎敯斒旓暆暌暕暐暋暊暙暔晸朠楦楟椸楎楢楱椿楅楪椹楂楗楙楺楈楉椵楬椳椽楥棰楸椴楩楀楯楄楶楘楁楴楌椻楋椷楜楏楑椲楒椯楻椼歆歅歃歂歈歁殛嗀毻毼"],["de40","毹毷毸溛滖滈溏滀溟溓溔溠溱溹滆滒溽滁溞滉溷溰滍溦滏溲溾滃滜滘溙溒溎溍溤溡溿溳滐滊溗溮溣煇煔煒煣煠煁煝煢煲煸煪煡煂煘煃煋煰煟煐煓"],["dea1","煄煍煚牏犍犌犑犐犎猼獂猻猺獀獊獉瑄瑊瑋瑒瑑瑗瑀瑏瑐瑎瑂瑆瑍瑔瓡瓿瓾瓽甝畹畷榃痯瘏瘃痷痾痼痹痸瘐痻痶痭痵痽皙皵盝睕睟睠睒睖睚睩睧睔睙睭矠碇碚碔碏碄碕碅碆碡碃硹碙碀碖硻祼禂祽祹稑稘稙稒稗稕稢稓"],["df40","稛稐窣窢窞竫筦筤筭筴筩筲筥筳筱筰筡筸筶筣粲粴粯綈綆綀綍絿綅絺綎絻綃絼綌綔綄絽綒罭罫罧罨罬羦羥羧翛翜耡腤腠腷腜腩腛腢腲朡腞腶腧腯"],["dfa1","腄腡舝艉艄艀艂艅蓱萿葖葶葹蒏蒍葥葑葀蒆葧萰葍葽葚葙葴葳葝蔇葞萷萺萴葺葃葸萲葅萩菙葋萯葂萭葟葰萹葎葌葒葯蓅蒎萻葇萶萳葨葾葄萫葠葔葮葐蜋蜄蛷蜌蛺蛖蛵蝍蛸蜎蜉蜁蛶蜍蜅裖裋裍裎裞裛裚裌裐覅覛觟觥觤"],["e040","觡觠觢觜触詶誆詿詡訿詷誂誄詵誃誁詴詺谼豋豊豥豤豦貆貄貅賌赨赩趑趌趎趏趍趓趔趐趒跰跠跬跱跮跐跩跣跢跧跲跫跴輆軿輁輀輅輇輈輂輋遒逿"],["e0a1","遄遉逽鄐鄍鄏鄑鄖鄔鄋鄎酮酯鉈鉒鈰鈺鉦鈳鉥鉞銃鈮鉊鉆鉭鉬鉏鉠鉧鉯鈶鉡鉰鈱鉔鉣鉐鉲鉎鉓鉌鉖鈲閟閜閞閛隒隓隑隗雎雺雽雸雵靳靷靸靲頏頍頎颬飶飹馯馲馰馵骭骫魛鳪鳭鳧麀黽僦僔僗僨僳僛僪僝僤僓僬僰僯僣僠"],["e140","凘劀劁勩勫匰厬嘧嘕嘌嘒嗼嘏嘜嘁嘓嘂嗺嘝嘄嗿嗹墉塼墐墘墆墁塿塴墋塺墇墑墎塶墂墈塻墔墏壾奫嫜嫮嫥嫕嫪嫚嫭嫫嫳嫢嫠嫛嫬嫞嫝嫙嫨嫟孷寠"],["e1a1","寣屣嶂嶀嵽嶆嵺嶁嵷嶊嶉嶈嵾嵼嶍嵹嵿幘幙幓廘廑廗廎廜廕廙廒廔彄彃彯徶愬愨慁慞慱慳慒慓慲慬憀慴慔慺慛慥愻慪慡慖戩戧戫搫摍摛摝摴摶摲摳摽摵摦撦摎撂摞摜摋摓摠摐摿搿摬摫摙摥摷敳斠暡暠暟朅朄朢榱榶槉"],["e240","榠槎榖榰榬榼榑榙榎榧榍榩榾榯榿槄榽榤槔榹槊榚槏榳榓榪榡榞槙榗榐槂榵榥槆歊歍歋殞殟殠毃毄毾滎滵滱漃漥滸漷滻漮漉潎漙漚漧漘漻漒滭漊"],["e2a1","漶潳滹滮漭潀漰漼漵滫漇漎潃漅滽滶漹漜滼漺漟漍漞漈漡熇熐熉熀熅熂熏煻熆熁熗牄牓犗犕犓獃獍獑獌瑢瑳瑱瑵瑲瑧瑮甀甂甃畽疐瘖瘈瘌瘕瘑瘊瘔皸瞁睼瞅瞂睮瞀睯睾瞃碲碪碴碭碨硾碫碞碥碠碬碢碤禘禊禋禖禕禔禓"],["e340","禗禈禒禐稫穊稰稯稨稦窨窫窬竮箈箜箊箑箐箖箍箌箛箎箅箘劄箙箤箂粻粿粼粺綧綷緂綣綪緁緀緅綝緎緄緆緋緌綯綹綖綼綟綦綮綩綡緉罳翢翣翥翞"],["e3a1","耤聝聜膉膆膃膇膍膌膋舕蒗蒤蒡蒟蒺蓎蓂蒬蒮蒫蒹蒴蓁蓍蒪蒚蒱蓐蒝蒧蒻蒢蒔蓇蓌蒛蒩蒯蒨蓖蒘蒶蓏蒠蓗蓔蓒蓛蒰蒑虡蜳蜣蜨蝫蝀蜮蜞蜡蜙蜛蝃蜬蝁蜾蝆蜠蜲蜪蜭蜼蜒蜺蜱蜵蝂蜦蜧蜸蜤蜚蜰蜑裷裧裱裲裺裾裮裼裶裻"],["e440","裰裬裫覝覡覟覞觩觫觨誫誙誋誒誏誖谽豨豩賕賏賗趖踉踂跿踍跽踊踃踇踆踅跾踀踄輐輑輎輍鄣鄜鄠鄢鄟鄝鄚鄤鄡鄛酺酲酹酳銥銤鉶銛鉺銠銔銪銍"],["e4a1","銦銚銫鉹銗鉿銣鋮銎銂銕銢鉽銈銡銊銆銌銙銧鉾銇銩銝銋鈭隞隡雿靘靽靺靾鞃鞀鞂靻鞄鞁靿韎韍頖颭颮餂餀餇馝馜駃馹馻馺駂馽駇骱髣髧鬾鬿魠魡魟鳱鳲鳵麧僿儃儰僸儆儇僶僾儋儌僽儊劋劌勱勯噈噂噌嘵噁噊噉噆噘"],["e540","噚噀嘳嘽嘬嘾嘸嘪嘺圚墫墝墱墠墣墯墬墥墡壿嫿嫴嫽嫷嫶嬃嫸嬂嫹嬁嬇嬅嬏屧嶙嶗嶟嶒嶢嶓嶕嶠嶜嶡嶚嶞幩幝幠幜緳廛廞廡彉徲憋憃慹憱憰憢憉"],["e5a1","憛憓憯憭憟憒憪憡憍慦憳戭摮摰撖撠撅撗撜撏撋撊撌撣撟摨撱撘敶敺敹敻斲斳暵暰暩暲暷暪暯樀樆樗槥槸樕槱槤樠槿槬槢樛樝槾樧槲槮樔槷槧橀樈槦槻樍槼槫樉樄樘樥樏槶樦樇槴樖歑殥殣殢殦氁氀毿氂潁漦潾澇濆澒"],["e640","澍澉澌潢潏澅潚澖潶潬澂潕潲潒潐潗澔澓潝漀潡潫潽潧澐潓澋潩潿澕潣潷潪潻熲熯熛熰熠熚熩熵熝熥熞熤熡熪熜熧熳犘犚獘獒獞獟獠獝獛獡獚獙"],["e6a1","獢璇璉璊璆璁瑽璅璈瑼瑹甈甇畾瘥瘞瘙瘝瘜瘣瘚瘨瘛皜皝皞皛瞍瞏瞉瞈磍碻磏磌磑磎磔磈磃磄磉禚禡禠禜禢禛歶稹窲窴窳箷篋箾箬篎箯箹篊箵糅糈糌糋緷緛緪緧緗緡縃緺緦緶緱緰緮緟罶羬羰羭翭翫翪翬翦翨聤聧膣膟"],["e740","膞膕膢膙膗舖艏艓艒艐艎艑蔤蔻蔏蔀蔩蔎蔉蔍蔟蔊蔧蔜蓻蔫蓺蔈蔌蓴蔪蓲蔕蓷蓫蓳蓼蔒蓪蓩蔖蓾蔨蔝蔮蔂蓽蔞蓶蔱蔦蓧蓨蓰蓯蓹蔘蔠蔰蔋蔙蔯虢"],["e7a1","蝖蝣蝤蝷蟡蝳蝘蝔蝛蝒蝡蝚蝑蝞蝭蝪蝐蝎蝟蝝蝯蝬蝺蝮蝜蝥蝏蝻蝵蝢蝧蝩衚褅褌褔褋褗褘褙褆褖褑褎褉覢覤覣觭觰觬諏諆誸諓諑諔諕誻諗誾諀諅諘諃誺誽諙谾豍貏賥賟賙賨賚賝賧趠趜趡趛踠踣踥踤踮踕踛踖踑踙踦踧"],["e840","踔踒踘踓踜踗踚輬輤輘輚輠輣輖輗遳遰遯遧遫鄯鄫鄩鄪鄲鄦鄮醅醆醊醁醂醄醀鋐鋃鋄鋀鋙銶鋏鋱鋟鋘鋩鋗鋝鋌鋯鋂鋨鋊鋈鋎鋦鋍鋕鋉鋠鋞鋧鋑鋓"],["e8a1","銵鋡鋆銴镼閬閫閮閰隤隢雓霅霈霂靚鞊鞎鞈韐韏頞頝頦頩頨頠頛頧颲餈飺餑餔餖餗餕駜駍駏駓駔駎駉駖駘駋駗駌骳髬髫髳髲髱魆魃魧魴魱魦魶魵魰魨魤魬鳼鳺鳽鳿鳷鴇鴀鳹鳻鴈鴅鴄麃黓鼏鼐儜儓儗儚儑凞匴叡噰噠噮"],["e940","噳噦噣噭噲噞噷圜圛壈墽壉墿墺壂墼壆嬗嬙嬛嬡嬔嬓嬐嬖嬨嬚嬠嬞寯嶬嶱嶩嶧嶵嶰嶮嶪嶨嶲嶭嶯嶴幧幨幦幯廩廧廦廨廥彋徼憝憨憖懅憴懆懁懌憺"],["e9a1","憿憸憌擗擖擐擏擉撽撉擃擛擳擙攳敿敼斢曈暾曀曊曋曏暽暻暺曌朣樴橦橉橧樲橨樾橝橭橶橛橑樨橚樻樿橁橪橤橐橏橔橯橩橠樼橞橖橕橍橎橆歕歔歖殧殪殫毈毇氄氃氆澭濋澣濇澼濎濈潞濄澽澞濊澨瀄澥澮澺澬澪濏澿澸"],["ea40","澢濉澫濍澯澲澰燅燂熿熸燖燀燁燋燔燊燇燏熽燘熼燆燚燛犝犞獩獦獧獬獥獫獪瑿璚璠璔璒璕璡甋疀瘯瘭瘱瘽瘳瘼瘵瘲瘰皻盦瞚瞝瞡瞜瞛瞢瞣瞕瞙"],["eaa1","瞗磝磩磥磪磞磣磛磡磢磭磟磠禤穄穈穇窶窸窵窱窷篞篣篧篝篕篥篚篨篹篔篪篢篜篫篘篟糒糔糗糐糑縒縡縗縌縟縠縓縎縜縕縚縢縋縏縖縍縔縥縤罃罻罼罺羱翯耪耩聬膱膦膮膹膵膫膰膬膴膲膷膧臲艕艖艗蕖蕅蕫蕍蕓蕡蕘"],["eb40","蕀蕆蕤蕁蕢蕄蕑蕇蕣蔾蕛蕱蕎蕮蕵蕕蕧蕠薌蕦蕝蕔蕥蕬虣虥虤螛螏螗螓螒螈螁螖螘蝹螇螣螅螐螑螝螄螔螜螚螉褞褦褰褭褮褧褱褢褩褣褯褬褟觱諠"],["eba1","諢諲諴諵諝謔諤諟諰諈諞諡諨諿諯諻貑貒貐賵賮賱賰賳赬赮趥趧踳踾踸蹀蹅踶踼踽蹁踰踿躽輶輮輵輲輹輷輴遶遹遻邆郺鄳鄵鄶醓醐醑醍醏錧錞錈錟錆錏鍺錸錼錛錣錒錁鍆錭錎錍鋋錝鋺錥錓鋹鋷錴錂錤鋿錩錹錵錪錔錌"],["ec40","錋鋾錉錀鋻錖閼闍閾閹閺閶閿閵閽隩雔霋霒霐鞙鞗鞔韰韸頵頯頲餤餟餧餩馞駮駬駥駤駰駣駪駩駧骹骿骴骻髶髺髹髷鬳鮀鮅鮇魼魾魻鮂鮓鮒鮐魺鮕"],["eca1","魽鮈鴥鴗鴠鴞鴔鴩鴝鴘鴢鴐鴙鴟麈麆麇麮麭黕黖黺鼒鼽儦儥儢儤儠儩勴嚓嚌嚍嚆嚄嚃噾嚂噿嚁壖壔壏壒嬭嬥嬲嬣嬬嬧嬦嬯嬮孻寱寲嶷幬幪徾徻懃憵憼懧懠懥懤懨懞擯擩擣擫擤擨斁斀斶旚曒檍檖檁檥檉檟檛檡檞檇檓檎"],["ed40","檕檃檨檤檑橿檦檚檅檌檒歛殭氉濌澩濴濔濣濜濭濧濦濞濲濝濢濨燡燱燨燲燤燰燢獳獮獯璗璲璫璐璪璭璱璥璯甐甑甒甏疄癃癈癉癇皤盩瞵瞫瞲瞷瞶"],["eda1","瞴瞱瞨矰磳磽礂磻磼磲礅磹磾礄禫禨穜穛穖穘穔穚窾竀竁簅簏篲簀篿篻簎篴簋篳簂簉簃簁篸篽簆篰篱簐簊糨縭縼繂縳顈縸縪繉繀繇縩繌縰縻縶繄縺罅罿罾罽翴翲耬膻臄臌臊臅臇膼臩艛艚艜薃薀薏薧薕薠薋薣蕻薤薚薞"],["ee40","蕷蕼薉薡蕺蕸蕗薎薖薆薍薙薝薁薢薂薈薅蕹蕶薘薐薟虨螾螪螭蟅螰螬螹螵螼螮蟉蟃蟂蟌螷螯蟄蟊螴螶螿螸螽蟞螲褵褳褼褾襁襒褷襂覭覯覮觲觳謞"],["eea1","謘謖謑謅謋謢謏謒謕謇謍謈謆謜謓謚豏豰豲豱豯貕貔賹赯蹎蹍蹓蹐蹌蹇轃轀邅遾鄸醚醢醛醙醟醡醝醠鎡鎃鎯鍤鍖鍇鍼鍘鍜鍶鍉鍐鍑鍠鍭鎏鍌鍪鍹鍗鍕鍒鍏鍱鍷鍻鍡鍞鍣鍧鎀鍎鍙闇闀闉闃闅閷隮隰隬霠霟霘霝霙鞚鞡鞜"],["ef40","鞞鞝韕韔韱顁顄顊顉顅顃餥餫餬餪餳餲餯餭餱餰馘馣馡騂駺駴駷駹駸駶駻駽駾駼騃骾髾髽鬁髼魈鮚鮨鮞鮛鮦鮡鮥鮤鮆鮢鮠鮯鴳鵁鵧鴶鴮鴯鴱鴸鴰"],["efa1","鵅鵂鵃鴾鴷鵀鴽翵鴭麊麉麍麰黈黚黻黿鼤鼣鼢齔龠儱儭儮嚘嚜嚗嚚嚝嚙奰嬼屩屪巀幭幮懘懟懭懮懱懪懰懫懖懩擿攄擽擸攁攃擼斔旛曚曛曘櫅檹檽櫡櫆檺檶檷櫇檴檭歞毉氋瀇瀌瀍瀁瀅瀔瀎濿瀀濻瀦濼濷瀊爁燿燹爃燽獶"],["f040","璸瓀璵瓁璾璶璻瓂甔甓癜癤癙癐癓癗癚皦皽盬矂瞺磿礌礓礔礉礐礒礑禭禬穟簜簩簙簠簟簭簝簦簨簢簥簰繜繐繖繣繘繢繟繑繠繗繓羵羳翷翸聵臑臒"],["f0a1","臐艟艞薴藆藀藃藂薳薵薽藇藄薿藋藎藈藅薱薶藒蘤薸薷薾虩蟧蟦蟢蟛蟫蟪蟥蟟蟳蟤蟔蟜蟓蟭蟘蟣螤蟗蟙蠁蟴蟨蟝襓襋襏襌襆襐襑襉謪謧謣謳謰謵譇謯謼謾謱謥謷謦謶謮謤謻謽謺豂豵貙貘貗賾贄贂贀蹜蹢蹠蹗蹖蹞蹥蹧"],["f140","蹛蹚蹡蹝蹩蹔轆轇轈轋鄨鄺鄻鄾醨醥醧醯醪鎵鎌鎒鎷鎛鎝鎉鎧鎎鎪鎞鎦鎕鎈鎙鎟鎍鎱鎑鎲鎤鎨鎴鎣鎥闒闓闑隳雗雚巂雟雘雝霣霢霥鞬鞮鞨鞫鞤鞪"],["f1a1","鞢鞥韗韙韖韘韺顐顑顒颸饁餼餺騏騋騉騍騄騑騊騅騇騆髀髜鬈鬄鬅鬩鬵魊魌魋鯇鯆鯃鮿鯁鮵鮸鯓鮶鯄鮹鮽鵜鵓鵏鵊鵛鵋鵙鵖鵌鵗鵒鵔鵟鵘鵚麎麌黟鼁鼀鼖鼥鼫鼪鼩鼨齌齕儴儵劖勷厴嚫嚭嚦嚧嚪嚬壚壝壛夒嬽嬾嬿巃幰"],["f240","徿懻攇攐攍攉攌攎斄旞旝曞櫧櫠櫌櫑櫙櫋櫟櫜櫐櫫櫏櫍櫞歠殰氌瀙瀧瀠瀖瀫瀡瀢瀣瀩瀗瀤瀜瀪爌爊爇爂爅犥犦犤犣犡瓋瓅璷瓃甖癠矉矊矄矱礝礛"],["f2a1","礡礜礗礞禰穧穨簳簼簹簬簻糬糪繶繵繸繰繷繯繺繲繴繨罋罊羃羆羷翽翾聸臗臕艤艡艣藫藱藭藙藡藨藚藗藬藲藸藘藟藣藜藑藰藦藯藞藢蠀蟺蠃蟶蟷蠉蠌蠋蠆蟼蠈蟿蠊蠂襢襚襛襗襡襜襘襝襙覈覷覶觶譐譈譊譀譓譖譔譋譕"],["f340","譑譂譒譗豃豷豶貚贆贇贉趬趪趭趫蹭蹸蹳蹪蹯蹻軂轒轑轏轐轓辴酀鄿醰醭鏞鏇鏏鏂鏚鏐鏹鏬鏌鏙鎩鏦鏊鏔鏮鏣鏕鏄鏎鏀鏒鏧镽闚闛雡霩霫霬霨霦"],["f3a1","鞳鞷鞶韝韞韟顜顙顝顗颿颽颻颾饈饇饃馦馧騚騕騥騝騤騛騢騠騧騣騞騜騔髂鬋鬊鬎鬌鬷鯪鯫鯠鯞鯤鯦鯢鯰鯔鯗鯬鯜鯙鯥鯕鯡鯚鵷鶁鶊鶄鶈鵱鶀鵸鶆鶋鶌鵽鵫鵴鵵鵰鵩鶅鵳鵻鶂鵯鵹鵿鶇鵨麔麑黀黼鼭齀齁齍齖齗齘匷嚲"],["f440","嚵嚳壣孅巆巇廮廯忀忁懹攗攖攕攓旟曨曣曤櫳櫰櫪櫨櫹櫱櫮櫯瀼瀵瀯瀷瀴瀱灂瀸瀿瀺瀹灀瀻瀳灁爓爔犨獽獼璺皫皪皾盭矌矎矏矍矲礥礣礧礨礤礩"],["f4a1","禲穮穬穭竷籉籈籊籇籅糮繻繾纁纀羺翿聹臛臙舋艨艩蘢藿蘁藾蘛蘀藶蘄蘉蘅蘌藽蠙蠐蠑蠗蠓蠖襣襦覹觷譠譪譝譨譣譥譧譭趮躆躈躄轙轖轗轕轘轚邍酃酁醷醵醲醳鐋鐓鏻鐠鐏鐔鏾鐕鐐鐨鐙鐍鏵鐀鏷鐇鐎鐖鐒鏺鐉鏸鐊鏿"],["f540","鏼鐌鏶鐑鐆闞闠闟霮霯鞹鞻韽韾顠顢顣顟飁飂饐饎饙饌饋饓騲騴騱騬騪騶騩騮騸騭髇髊髆鬐鬒鬑鰋鰈鯷鰅鰒鯸鱀鰇鰎鰆鰗鰔鰉鶟鶙鶤鶝鶒鶘鶐鶛"],["f5a1","鶠鶔鶜鶪鶗鶡鶚鶢鶨鶞鶣鶿鶩鶖鶦鶧麙麛麚黥黤黧黦鼰鼮齛齠齞齝齙龑儺儹劘劗囃嚽嚾孈孇巋巏廱懽攛欂櫼欃櫸欀灃灄灊灈灉灅灆爝爚爙獾甗癪矐礭礱礯籔籓糲纊纇纈纋纆纍罍羻耰臝蘘蘪蘦蘟蘣蘜蘙蘧蘮蘡蘠蘩蘞蘥"],["f640","蠩蠝蠛蠠蠤蠜蠫衊襭襩襮襫觺譹譸譅譺譻贐贔趯躎躌轞轛轝酆酄酅醹鐿鐻鐶鐩鐽鐼鐰鐹鐪鐷鐬鑀鐱闥闤闣霵霺鞿韡顤飉飆飀饘饖騹騽驆驄驂驁騺"],["f6a1","騿髍鬕鬗鬘鬖鬺魒鰫鰝鰜鰬鰣鰨鰩鰤鰡鶷鶶鶼鷁鷇鷊鷏鶾鷅鷃鶻鶵鷎鶹鶺鶬鷈鶱鶭鷌鶳鷍鶲鹺麜黫黮黭鼛鼘鼚鼱齎齥齤龒亹囆囅囋奱孋孌巕巑廲攡攠攦攢欋欈欉氍灕灖灗灒爞爟犩獿瓘瓕瓙瓗癭皭礵禴穰穱籗籜籙籛籚"],["f740","糴糱纑罏羇臞艫蘴蘵蘳蘬蘲蘶蠬蠨蠦蠪蠥襱覿覾觻譾讄讂讆讅譿贕躕躔躚躒躐躖躗轠轢酇鑌鑐鑊鑋鑏鑇鑅鑈鑉鑆霿韣顪顩飋饔饛驎驓驔驌驏驈驊"],["f7a1","驉驒驐髐鬙鬫鬻魖魕鱆鱈鰿鱄鰹鰳鱁鰼鰷鰴鰲鰽鰶鷛鷒鷞鷚鷋鷐鷜鷑鷟鷩鷙鷘鷖鷵鷕鷝麶黰鼵鼳鼲齂齫龕龢儽劙壨壧奲孍巘蠯彏戁戃戄攩攥斖曫欑欒欏毊灛灚爢玂玁玃癰矔籧籦纕艬蘺虀蘹蘼蘱蘻蘾蠰蠲蠮蠳襶襴襳觾"],["f840","讌讎讋讈豅贙躘轤轣醼鑢鑕鑝鑗鑞韄韅頀驖驙鬞鬟鬠鱒鱘鱐鱊鱍鱋鱕鱙鱌鱎鷻鷷鷯鷣鷫鷸鷤鷶鷡鷮鷦鷲鷰鷢鷬鷴鷳鷨鷭黂黐黲黳鼆鼜鼸鼷鼶齃齏"],["f8a1","齱齰齮齯囓囍孎屭攭曭曮欓灟灡灝灠爣瓛瓥矕礸禷禶籪纗羉艭虃蠸蠷蠵衋讔讕躞躟躠躝醾醽釂鑫鑨鑩雥靆靃靇韇韥驞髕魙鱣鱧鱦鱢鱞鱠鸂鷾鸇鸃鸆鸅鸀鸁鸉鷿鷽鸄麠鼞齆齴齵齶囔攮斸欘欙欗欚灢爦犪矘矙礹籩籫糶纚"],["f940","纘纛纙臠臡虆虇虈襹襺襼襻觿讘讙躥躤躣鑮鑭鑯鑱鑳靉顲饟鱨鱮鱭鸋鸍鸐鸏鸒鸑麡黵鼉齇齸齻齺齹圞灦籯蠼趲躦釃鑴鑸鑶鑵驠鱴鱳鱱鱵鸔鸓黶鼊"],["f9a1","龤灨灥糷虪蠾蠽蠿讞貜躩軉靋顳顴飌饡馫驤驦驧鬤鸕鸗齈戇欞爧虌躨钂钀钁驩驨鬮鸙爩虋讟钃鱹麷癵驫鱺鸝灩灪麤齾齉龘碁銹裏墻恒粧嫺╔╦╗╠╬╣╚╩╝╒╤╕╞╪╡╘╧╛╓╥╖╟╫╢╙╨╜║═╭╮╰╯▓"]]')},function(i,o,u){(function(){var o,p,g;p=u(87).Number,g=u(39),o=function(){function ArrayT(i,o,u){this.type=i,this.length=o,this.lengthType=null!=u?u:"count"}return ArrayT.prototype.decode=function(i,o){var u,y,w,_,x,k;if(w=i.pos,_=[],u=o,null!=this.length&&(y=g.resolveLength(this.length,i,o)),this.length instanceof p&&(Object.defineProperties(_,{parent:{value:o},_startOffset:{value:w},_currentOffset:{value:0,writable:!0},_length:{value:y}}),u=_),null==y||"bytes"===this.lengthType)for(x=null!=y?i.pos+y:(null!=o?o._length:void 0)?o._startOffset+o._length:i.length;i.pos<x;)_.push(this.type.decode(i,u));else for(k=0;k<y;k+=1)_.push(this.type.decode(i,u));return _},ArrayT.prototype.size=function(i,o){var u,y,w,_;if(!i)return this.type.size(null,o)*g.resolveLength(this.length,null,o);for(y=0,this.length instanceof p&&(y+=this.length.size(),o={parent:o}),w=0,_=i.length;w<_;w++)u=i[w],y+=this.type.size(u,o);return y},ArrayT.prototype.encode=function(i,o,u){var g,y,w,_,x,k;for(g=u,this.length instanceof p&&((g={pointers:[],startOffset:i.pos,parent:u}).pointerOffset=i.pos+this.size(o,g),this.length.encode(i,o.length)),x=0,k=o.length;x<k;x++)w=o[x],this.type.encode(i,w,g);if(this.length instanceof p)for(y=0;y<g.pointers.length;)(_=g.pointers[y++]).type.encode(i,_.val)},ArrayT}(),i.exports=o}).call(this)},function(i,o,u){(function(){var o,p;p=u(39),o=function(){function Struct(i){this.fields=null!=i?i:{}}return Struct.prototype.decode=function(i,o,u){var p,g;return null==u&&(u=0),p=this._setup(i,o,u),this._parseFields(i,p,this.fields),null!=(g=this.process)&&g.call(p,i),p},Struct.prototype._setup=function(i,o,u){var p;return p={},Object.defineProperties(p,{parent:{value:o},_startOffset:{value:i.pos},_currentOffset:{value:0,writable:!0},_length:{value:u}}),p},Struct.prototype._parseFields=function(i,o,u){var g,y,w;for(g in u)void 0!==(w="function"==typeof(y=u[g])?y.call(o,o):y.decode(i,o))&&(w instanceof p.PropertyDescriptor?Object.defineProperty(o,g,w):o[g]=w),o._currentOffset=i.pos-o._startOffset},Struct.prototype.size=function(i,o,u){var p,g,y,w,_;for(g in null==i&&(i={}),null==u&&(u=!0),p={parent:o,val:i,pointerSize:0},y=0,_=this.fields)null!=(w=_[g]).size&&(y+=w.size(i[g],p));return u&&(y+=p.pointerSize),y},Struct.prototype.encode=function(i,o,u){var p,g,y,w,_,x,k;for(y in null!=(x=this.preEncode)&&x.call(o,i),(p={pointers:[],startOffset:i.pos,parent:u,val:o,pointerSize:0}).pointerOffset=i.pos+this.size(o,p,!1),k=this.fields)null!=(_=k[y]).encode&&_.encode(i,o[y],p);for(g=0;g<p.pointers.length;)(w=p.pointers[g++]).type.encode(i,w.val,w.parent)},Struct}(),i.exports=o}).call(this)},function(i,o,u){i.exports=!u(24)&&!u(53)((function(){return 7!=Object.defineProperty(u(180)("div"),"a",{get:function(){return 7}}).a}))},function(i,o){i.exports=function(i,o){return{value:o,done:!!i}}},function(i,o,u){i.exports=u(40)},function(i,o,u){var p=u(26),g=u(29),y=u(102);i.exports=u(24)?Object.defineProperties:function defineProperties(i,o){g(i);for(var u,w=y(o),_=w.length,x=0;_>x;)p.f(i,u=w[x++],o[u]);return i}},function(i,o,u){var p=u(52),g=u(51),y=u(433)(!1),w=u(184)("IE_PROTO");i.exports=function(i,o){var u,_=g(i),x=0,k=[];for(u in _)u!=w&&p(_,u)&&k.push(u);for(;o.length>x;)p(_,u=o[x++])&&(~y(k,u)||k.push(u));return k}},function(i,o,u){var p=u(183),g=Math.max,y=Math.min;i.exports=function(i,o){return(i=p(i))<0?g(i+o,0):y(i,o)}},function(i,o,u){var p=u(17).document;i.exports=p&&p.documentElement},function(i,o,u){var p=u(96);i.exports=Array.isArray||function isArray(i){return"Array"==p(i)}},function(i,o,u){var p=u(258),g=u(186).concat("length","prototype");o.f=Object.getOwnPropertyNames||function getOwnPropertyNames(i){return p(i,g)}},function(i,o,u){i.exports={default:u(452),__esModule:!0}},function(i,o,u){"use strict";var p=u(26).f,g=u(131),y=u(193),w=u(33),_=u(194),x=u(104),k=u(182),P=u(255),E=u(267),O=u(24),I=u(134).fastKey,B=u(195),D=O?"_s":"size",getEntry=function(i,o){var u,p=I(o);if("F"!==p)return i._i[p];for(u=i._f;u;u=u.n)if(u.k==o)return u};i.exports={getConstructor:function(i,o,u,k){var P=i((function(i,p){_(i,P,o,"_i"),i._t=o,i._i=g(null),i._f=void 0,i._l=void 0,i[D]=0,null!=p&&x(p,u,i[k],i)}));return y(P.prototype,{clear:function clear(){for(var i=B(this,o),u=i._i,p=i._f;p;p=p.n)p.r=!0,p.p&&(p.p=p.p.n=void 0),delete u[p.i];i._f=i._l=void 0,i[D]=0},delete:function(i){var u=B(this,o),p=getEntry(u,i);if(p){var g=p.n,y=p.p;delete u._i[p.i],p.r=!0,y&&(y.n=g),g&&(g.p=y),u._f==p&&(u._f=g),u._l==p&&(u._l=y),u[D]--}return!!p},forEach:function forEach(i){B(this,o);for(var u,p=w(i,arguments.length>1?arguments[1]:void 0,3);u=u?u.n:this._f;)for(p(u.v,u.k,this);u&&u.r;)u=u.p},has:function has(i){return!!getEntry(B(this,o),i)}}),O&&p(P.prototype,"size",{get:function(){return B(this,o)[D]}}),P},def:function(i,o,u){var p,g,y=getEntry(i,o);return y?y.v=u:(i._l=y={i:g=I(o,!0),k:o,v:u,p:p=i._l,n:void 0,r:!1},i._f||(i._f=y),p&&(p.n=y),i[D]++,"F"!==g&&(i._i[g]=y)),i},getEntry:getEntry,setStrong:function(i,o,u){k(i,o,(function(i,u){this._t=B(i,o),this._k=u,this._l=void 0}),(function(){for(var i=this._k,o=this._l;o&&o.r;)o=o.p;return this._t&&(this._l=o=o?o.n:this._t._f)?P(0,"keys"==i?o.k:"values"==i?o.v:[o.k,o.v]):(this._t=void 0,P(1))}),u?"entries":"values",!u,!0),E(o)}}},function(i,o,u){var p=u(29);i.exports=function(i,o,u,g){try{return g?o(p(u)[0],u[1]):o(u)}catch(o){var y=i.return;throw void 0!==y&&p(y.call(i)),o}}},function(i,o,u){var p=u(100),g=u(20)("iterator"),y=Array.prototype;i.exports=function(i){return void 0!==i&&(p.Array===i||y[g]===i)}},function(i,o,u){"use strict";var p=u(17),g=u(7),y=u(26),w=u(24),_=u(20)("species");i.exports=function(i){var o="function"==typeof g[i]?g[i]:p[i];w&&o&&!o[_]&&y.f(o,_,{configurable:!0,get:function(){return this}})}},function(i,o,u){"use strict";var p=u(17),g=u(9),y=u(134),w=u(53),_=u(40),x=u(193),k=u(104),P=u(194),E=u(23),O=u(103),I=u(26).f,B=u(459)(0),D=u(24);i.exports=function(i,o,u,R,N,U){var W=p[i],G=W,j=N?"set":"add",X=G&&G.prototype,K={};return D&&"function"==typeof G&&(U||X.forEach&&!w((function(){(new G).entries().next()})))?(G=o((function(o,u){P(o,G,i,"_c"),o._c=new W,null!=u&&k(u,N,o[j],o)})),B("add,clear,delete,forEach,get,has,set,keys,values,entries,toJSON".split(","),(function(i){var o="add"==i||"set"==i;!(i in X)||U&&"clear"==i||_(G.prototype,i,(function(u,p){if(P(this,G,i),!o&&U&&!E(u))return"get"==i&&void 0;var g=this._c[i](0===u?0:u,p);return o?this:g}))})),U||I(G.prototype,"size",{get:function(){return this._c.size}})):(G=R.getConstructor(o,i,N,j),x(G.prototype,u),y.NEED=!0),O(G,i),K[i]=G,g(g.G+g.W+g.F,K),U||R.setStrong(G,i,N),G}},function(i,o,u){var p=u(188),g=u(463);i.exports=function(i){return function toJSON(){if(p(this)!=i)throw TypeError(i+"#toJSON isn't generic");return g(this)}}},function(i,o,u){"use strict";var p=u(9);i.exports=function(i){p(p.S,i,{of:function of(){for(var i=arguments.length,o=new Array(i);i--;)o[i]=arguments[i];return new this(o)}})}},function(i,o,u){"use strict";var p=u(9),g=u(98),y=u(33),w=u(104);i.exports=function(i){p(p.S,i,{from:function from(i){var o,u,p,_,x=arguments[1];return g(this),(o=void 0!==x)&&g(x),null==i?new this:(u=[],o?(p=0,_=y(x,arguments[2],2),w(i,!1,(function(i){u.push(_(i,p++))}))):w(i,!1,u.push,u),new this(u))}})}},function(i,o,u){"use strict";var p=Array.prototype.slice,g=u(273),y=Object.keys,w=y?function keys(i){return y(i)}:u(479),_=Object.keys;w.shim=function shimObjectKeys(){Object.keys?function(){var i=Object.keys(arguments);return i&&i.length===arguments.length}(1,2)||(Object.keys=function keys(i){return g(i)?_(p.call(i)):_(i)}):Object.keys=w;return Object.keys||w},i.exports=w},function(i,o,u){"use strict";var p=Object.prototype.toString;i.exports=function isArguments(i){var o=p.call(i),u="[object Arguments]"===o;return u||(u="[object Array]"!==o&&null!==i&&"object"==typeof i&&"number"==typeof i.length&&i.length>=0&&"[object Function]"===p.call(i.callee)),u}},function(i,o,u){"use strict";var p=u(275),g=u(197),y=g(p("String.prototype.indexOf"));i.exports=function callBoundIntrinsic(i,o){var u=p(i,!!o);return"function"==typeof u&&y(i,".prototype.")>-1?g(u):u}},function(i,o,u){"use strict";var p=SyntaxError,g=Function,y=TypeError,getEvalledConstructor=function(i){try{return g('"use strict"; return ('+i+").constructor;")()}catch(i){}},w=Object.getOwnPropertyDescriptor;if(w)try{w({},"")}catch(i){w=null}var throwTypeError=function(){throw new y},_=w?function(){try{return throwTypeError}catch(i){try{return w(arguments,"callee").get}catch(i){return throwTypeError}}}():throwTypeError,x=u(276)(),k=Object.getPrototypeOf||function(i){return i.__proto__},P={},E="undefined"==typeof Uint8Array?void 0:k(Uint8Array),O={"%AggregateError%":"undefined"==typeof AggregateError?void 0:AggregateError,"%Array%":Array,"%ArrayBuffer%":"undefined"==typeof ArrayBuffer?void 0:ArrayBuffer,"%ArrayIteratorPrototype%":x?k([][Symbol.iterator]()):void 0,"%AsyncFromSyncIteratorPrototype%":void 0,"%AsyncFunction%":P,"%AsyncGenerator%":P,"%AsyncGeneratorFunction%":P,"%AsyncIteratorPrototype%":P,"%Atomics%":"undefined"==typeof Atomics?void 0:Atomics,"%BigInt%":"undefined"==typeof BigInt?void 0:BigInt,"%Boolean%":Boolean,"%DataView%":"undefined"==typeof DataView?void 0:DataView,"%Date%":Date,"%decodeURI%":decodeURI,"%decodeURIComponent%":decodeURIComponent,"%encodeURI%":encodeURI,"%encodeURIComponent%":encodeURIComponent,"%Error%":Error,"%eval%":eval,"%EvalError%":EvalError,"%Float32Array%":"undefined"==typeof Float32Array?void 0:Float32Array,"%Float64Array%":"undefined"==typeof Float64Array?void 0:Float64Array,"%FinalizationRegistry%":"undefined"==typeof FinalizationRegistry?void 0:FinalizationRegistry,"%Function%":g,"%GeneratorFunction%":P,"%Int8Array%":"undefined"==typeof Int8Array?void 0:Int8Array,"%Int16Array%":"undefined"==typeof Int16Array?void 0:Int16Array,"%Int32Array%":"undefined"==typeof Int32Array?void 0:Int32Array,"%isFinite%":isFinite,"%isNaN%":isNaN,"%IteratorPrototype%":x?k(k([][Symbol.iterator]())):void 0,"%JSON%":"object"==typeof JSON?JSON:void 0,"%Map%":"undefined"==typeof Map?void 0:Map,"%MapIteratorPrototype%":"undefined"!=typeof Map&&x?k((new Map)[Symbol.iterator]()):void 0,"%Math%":Math,"%Number%":Number,"%Object%":Object,"%parseFloat%":parseFloat,"%parseInt%":parseInt,"%Promise%":"undefined"==typeof Promise?void 0:Promise,"%Proxy%":"undefined"==typeof Proxy?void 0:Proxy,"%RangeError%":RangeError,"%ReferenceError%":ReferenceError,"%Reflect%":"undefined"==typeof Reflect?void 0:Reflect,"%RegExp%":RegExp,"%Set%":"undefined"==typeof Set?void 0:Set,"%SetIteratorPrototype%":"undefined"!=typeof Set&&x?k((new Set)[Symbol.iterator]()):void 0,"%SharedArrayBuffer%":"undefined"==typeof SharedArrayBuffer?void 0:SharedArrayBuffer,"%String%":String,"%StringIteratorPrototype%":x?k(""[Symbol.iterator]()):void 0,"%Symbol%":x?Symbol:void 0,"%SyntaxError%":p,"%ThrowTypeError%":_,"%TypedArray%":E,"%TypeError%":y,"%Uint8Array%":"undefined"==typeof Uint8Array?void 0:Uint8Array,"%Uint8ClampedArray%":"undefined"==typeof Uint8ClampedArray?void 0:Uint8ClampedArray,"%Uint16Array%":"undefined"==typeof Uint16Array?void 0:Uint16Array,"%Uint32Array%":"undefined"==typeof Uint32Array?void 0:Uint32Array,"%URIError%":URIError,"%WeakMap%":"undefined"==typeof WeakMap?void 0:WeakMap,"%WeakRef%":"undefined"==typeof WeakRef?void 0:WeakRef,"%WeakSet%":"undefined"==typeof WeakSet?void 0:WeakSet},I={"%ArrayBufferPrototype%":["ArrayBuffer","prototype"],"%ArrayPrototype%":["Array","prototype"],"%ArrayProto_entries%":["Array","prototype","entries"],"%ArrayProto_forEach%":["Array","prototype","forEach"],"%ArrayProto_keys%":["Array","prototype","keys"],"%ArrayProto_values%":["Array","prototype","values"],"%AsyncFunctionPrototype%":["AsyncFunction","prototype"],"%AsyncGenerator%":["AsyncGeneratorFunction","prototype"],"%AsyncGeneratorPrototype%":["AsyncGeneratorFunction","prototype","prototype"],"%BooleanPrototype%":["Boolean","prototype"],"%DataViewPrototype%":["DataView","prototype"],"%DatePrototype%":["Date","prototype"],"%ErrorPrototype%":["Error","prototype"],"%EvalErrorPrototype%":["EvalError","prototype"],"%Float32ArrayPrototype%":["Float32Array","prototype"],"%Float64ArrayPrototype%":["Float64Array","prototype"],"%FunctionPrototype%":["Function","prototype"],"%Generator%":["GeneratorFunction","prototype"],"%GeneratorPrototype%":["GeneratorFunction","prototype","prototype"],"%Int8ArrayPrototype%":["Int8Array","prototype"],"%Int16ArrayPrototype%":["Int16Array","prototype"],"%Int32ArrayPrototype%":["Int32Array","prototype"],"%JSONParse%":["JSON","parse"],"%JSONStringify%":["JSON","stringify"],"%MapPrototype%":["Map","prototype"],"%NumberPrototype%":["Number","prototype"],"%ObjectPrototype%":["Object","prototype"],"%ObjProto_toString%":["Object","prototype","toString"],"%ObjProto_valueOf%":["Object","prototype","valueOf"],"%PromisePrototype%":["Promise","prototype"],"%PromiseProto_then%":["Promise","prototype","then"],"%Promise_all%":["Promise","all"],"%Promise_reject%":["Promise","reject"],"%Promise_resolve%":["Promise","resolve"],"%RangeErrorPrototype%":["RangeError","prototype"],"%ReferenceErrorPrototype%":["ReferenceError","prototype"],"%RegExpPrototype%":["RegExp","prototype"],"%SetPrototype%":["Set","prototype"],"%SharedArrayBufferPrototype%":["SharedArrayBuffer","prototype"],"%StringPrototype%":["String","prototype"],"%SymbolPrototype%":["Symbol","prototype"],"%SyntaxErrorPrototype%":["SyntaxError","prototype"],"%TypedArrayPrototype%":["TypedArray","prototype"],"%TypeErrorPrototype%":["TypeError","prototype"],"%Uint8ArrayPrototype%":["Uint8Array","prototype"],"%Uint8ClampedArrayPrototype%":["Uint8ClampedArray","prototype"],"%Uint16ArrayPrototype%":["Uint16Array","prototype"],"%Uint32ArrayPrototype%":["Uint32Array","prototype"],"%URIErrorPrototype%":["URIError","prototype"],"%WeakMapPrototype%":["WeakMap","prototype"],"%WeakSetPrototype%":["WeakSet","prototype"]},B=u(196),D=u(483),R=B.call(Function.call,Array.prototype.concat),N=B.call(Function.apply,Array.prototype.splice),U=B.call(Function.call,String.prototype.replace),W=B.call(Function.call,String.prototype.slice),G=/[^%.[\]]+|\[(?:(-?\d+(?:\.\d+)?)|(["'])((?:(?!\2)[^\\]|\\.)*?)\2)\]|(?=(?:\.|\[\])(?:\.|\[\]|%$))/g,j=/\\(\\)?/g,X=function stringToPath(i){var o=W(i,0,1),u=W(i,-1);if("%"===o&&"%"!==u)throw new p("invalid intrinsic syntax, expected closing `%`");if("%"===u&&"%"!==o)throw new p("invalid intrinsic syntax, expected opening `%`");var g=[];return U(i,G,(function(i,o,u,p){g[g.length]=u?U(p,j,"$1"):o||i})),g},K=function getBaseIntrinsic(i,o){var u,g=i;if(D(I,g)&&(g="%"+(u=I[g])[0]+"%"),D(O,g)){var w=O[g];if(w===P&&(w=function doEval(i){var o;if("%AsyncFunction%"===i)o=getEvalledConstructor("async function () {}");else if("%GeneratorFunction%"===i)o=getEvalledConstructor("function* () {}");else if("%AsyncGeneratorFunction%"===i)o=getEvalledConstructor("async function* () {}");else if("%AsyncGenerator%"===i){var u=doEval("%AsyncGeneratorFunction%");u&&(o=u.prototype)}else if("%AsyncIteratorPrototype%"===i){var p=doEval("%AsyncGenerator%");p&&(o=k(p.prototype))}return O[i]=o,o}(g)),void 0===w&&!o)throw new y("intrinsic "+i+" exists, but is not available. Please file an issue!");return{alias:u,name:g,value:w}}throw new p("intrinsic "+i+" does not exist!")};i.exports=function GetIntrinsic(i,o){if("string"!=typeof i||0===i.length)throw new y("intrinsic name must be a non-empty string");if(arguments.length>1&&"boolean"!=typeof o)throw new y('"allowMissing" argument must be a boolean');var u=X(i),g=u.length>0?u[0]:"",_=K("%"+g+"%",o),x=_.name,k=_.value,P=!1,E=_.alias;E&&(g=E[0],N(u,R([0,1],E)));for(var I=1,B=!0;I<u.length;I+=1){var U=u[I],G=W(U,0,1),j=W(U,-1);if(('"'===G||"'"===G||"`"===G||'"'===j||"'"===j||"`"===j)&&G!==j)throw new p("property names with quotes must have matching quotes");if("constructor"!==U&&B||(P=!0),D(O,x="%"+(g+="."+U)+"%"))k=O[x];else if(null!=k){if(!(U in k)){if(!o)throw new y("base intrinsic for "+i+" exists, but the property is not available.");return}if(w&&I+1>=u.length){var Y=w(k,U);k=(B=!!Y)&&"get"in Y&&!("originalValue"in Y.get)?Y.get:k[U]}else B=D(k,U),k=k[U];B&&!P&&(O[x]=k)}}return k}},function(i,o,u){"use strict";var p="undefined"!=typeof Symbol&&Symbol,g=u(481);i.exports=function hasNativeSymbols(){return"function"==typeof p&&("function"==typeof Symbol&&("symbol"==typeof p("foo")&&("symbol"==typeof Symbol("bar")&&g())))}},function(i,o,u){"use strict";var numberIsNaN=function(i){return i!=i};i.exports=function is(i,o){return 0===i&&0===o?1/i==1/o:i===o||!(!numberIsNaN(i)||!numberIsNaN(o))}},function(i,o,u){"use strict";var p=u(277);i.exports=function getPolyfill(){return"function"==typeof Object.is?Object.is:p}},function(i,o,u){"use strict";var p=Object,g=TypeError;i.exports=function flags(){if(null!=this&&this!==p(this))throw new g("RegExp.prototype.flags getter called on non-object");var i="";return this.global&&(i+="g"),this.ignoreCase&&(i+="i"),this.multiline&&(i+="m"),this.dotAll&&(i+="s"),this.unicode&&(i+="u"),this.sticky&&(i+="y"),i}},function(i,o,u){"use strict";var p=u(279),g=u(105).supportsDescriptors,y=Object.getOwnPropertyDescriptor,w=TypeError;i.exports=function getPolyfill(){if(!g)throw new w("RegExp.prototype.flags requires a true ES5 environment that supports property descriptors");if("gim"===/a/gim.flags){var i=y(RegExp.prototype,"flags");if(i&&"function"==typeof i.get&&"boolean"==typeof/a/.dotAll)return i.get}return p}},function(i,o,u){var p=u(20)("iterator"),g=!1;try{var y=[7][p]();y.return=function(){g=!0},Array.from(y,(function(){throw 2}))}catch(i){}i.exports=function(i,o){if(!o&&!g)return!1;var u=!1;try{var y=[7],w=y[p]();w.next=function(){return{done:u=!0}},y[p]=function(){return w},i(y)}catch(i){}return u}},function(i,o,u){var p=u(29),g=u(98),y=u(20)("species");i.exports=function(i,o){var u,w=p(i).constructor;return void 0===w||null==(u=p(w)[y])?o:g(u)}},function(i,o,u){var p,g,y,w=u(33),_=u(522),x=u(260),k=u(180),P=u(17),E=P.process,O=P.setImmediate,I=P.clearImmediate,B=P.MessageChannel,D=P.Dispatch,R=0,N={},run=function(){var i=+this;if(N.hasOwnProperty(i)){var o=N[i];delete N[i],o()}},listener=function(i){run.call(i.data)};O&&I||(O=function setImmediate(i){for(var o=[],u=1;arguments.length>u;)o.push(arguments[u++]);return N[++R]=function(){_("function"==typeof i?i:Function(i),o)},p(R),R},I=function clearImmediate(i){delete N[i]},"process"==u(96)(E)?p=function(i){E.nextTick(w(run,i,1))}:D&&D.now?p=function(i){D.now(w(run,i,1))}:B?(y=(g=new B).port2,g.port1.onmessage=listener,p=w(y.postMessage,y,1)):P.addEventListener&&"function"==typeof postMessage&&!P.importScripts?(p=function(i){P.postMessage(i+"","*")},P.addEventListener("message",listener,!1)):p="onreadystatechange"in k("script")?function(i){x.appendChild(k("script")).onreadystatechange=function(){x.removeChild(this),run.call(i)}}:function(i){setTimeout(w(run,i,1),0)}),i.exports={set:O,clear:I}},function(i,o){i.exports=function(i){try{return{e:!1,v:i()}}catch(i){return{e:!0,v:i}}}},function(i,o,u){var p=u(29),g=u(23),y=u(198);i.exports=function(i,o){if(p(i),g(o)&&o.constructor===i)return o;var u=y.f(i);return(0,u.resolve)(o),u.promise}},function(i,o,u){var p=u(287).BrotliInput,g=u(287).BrotliOutput,y=u(528),w=u(288),_=u(289).HuffmanCode,x=u(289).BrotliBuildHuffmanTable,k=u(532),P=u(533),E=u(534),O=new Uint8Array([1,2,3,4,0,5,17,6,16,7,8,9,10,11,12,13,14,15]),I=new Uint8Array([3,2,1,0,3,3,3,3,3,3,2,2,2,2,2,2]),B=new Int8Array([0,0,0,0,-1,1,-2,2,-3,3,-1,1,-2,2,-3,3]),D=new Uint16Array([256,402,436,468,500,534,566,598,630,662,694,726,758,790,822,854,886,920,952,984,1016,1048,1080]);function DecodeWindowBits(i){var o;return 0===i.readBits(1)?16:(o=i.readBits(3))>0?17+o:(o=i.readBits(3))>0?8+o:17}function DecodeVarLenUint8(i){if(i.readBits(1)){var o=i.readBits(3);return 0===o?1:i.readBits(o)+(1<<o)}return 0}function MetaBlockLength(){this.meta_block_length=0,this.input_end=0,this.is_uncompressed=0,this.is_metadata=!1}function DecodeMetaBlockLength(i){var o,u,p,g=new MetaBlockLength;if(g.input_end=i.readBits(1),g.input_end&&i.readBits(1))return g;if(7===(o=i.readBits(2)+4)){if(g.is_metadata=!0,0!==i.readBits(1))throw new Error("Invalid reserved bit");if(0===(u=i.readBits(2)))return g;for(p=0;p<u;p++){var y=i.readBits(8);if(p+1===u&&u>1&&0===y)throw new Error("Invalid size byte");g.meta_block_length|=y<<8*p}}else for(p=0;p<o;++p){var w=i.readBits(4);if(p+1===o&&o>4&&0===w)throw new Error("Invalid size nibble");g.meta_block_length|=w<<4*p}return++g.meta_block_length,g.input_end||g.is_metadata||(g.is_uncompressed=i.readBits(1)),g}function ReadSymbol(i,o,u){var p;return u.fillBitWindow(),(p=i[o+=u.val_>>>u.bit_pos_&255].bits-8)>0&&(u.bit_pos_+=8,o+=i[o].value,o+=u.val_>>>u.bit_pos_&(1<<p)-1),u.bit_pos_+=i[o].bits,i[o].value}function ReadHuffmanCode(i,o,u,p){var g,y,w=new Uint8Array(i);if(p.readMoreInput(),1===(y=p.readBits(2))){for(var k=i-1,P=0,E=new Int32Array(4),I=p.readBits(2)+1;k;)k>>=1,++P;for(B=0;B<I;++B)E[B]=p.readBits(P)%i,w[E[B]]=2;switch(w[E[0]]=1,I){case 1:break;case 3:if(E[0]===E[1]||E[0]===E[2]||E[1]===E[2])throw new Error("[ReadHuffmanCode] invalid symbols");break;case 2:if(E[0]===E[1])throw new Error("[ReadHuffmanCode] invalid symbols");w[E[1]]=1;break;case 4:if(E[0]===E[1]||E[0]===E[2]||E[0]===E[3]||E[1]===E[2]||E[1]===E[3]||E[2]===E[3])throw new Error("[ReadHuffmanCode] invalid symbols");p.readBits(1)?(w[E[2]]=3,w[E[3]]=3):w[E[0]]=2}}else{var B,D=new Uint8Array(18),R=32,N=0,U=[new _(2,0),new _(2,4),new _(2,3),new _(3,2),new _(2,0),new _(2,4),new _(2,3),new _(4,1),new _(2,0),new _(2,4),new _(2,3),new _(3,2),new _(2,0),new _(2,4),new _(2,3),new _(4,5)];for(B=y;B<18&&R>0;++B){var W,G=O[B],j=0;p.fillBitWindow(),j+=p.val_>>>p.bit_pos_&15,p.bit_pos_+=U[j].bits,W=U[j].value,D[G]=W,0!==W&&(R-=32>>W,++N)}if(1!==N&&0!==R)throw new Error("[ReadHuffmanCode] invalid num_codes or space");!function ReadHuffmanCodeLengths(i,o,u,p){for(var g=0,y=8,w=0,k=0,P=32768,E=[],O=0;O<32;O++)E.push(new _(0,0));for(x(E,0,5,i,18);g<o&&P>0;){var I,B=0;if(p.readMoreInput(),p.fillBitWindow(),B+=p.val_>>>p.bit_pos_&31,p.bit_pos_+=E[B].bits,(I=255&E[B].value)<16)w=0,u[g++]=I,0!==I&&(y=I,P-=32768>>I);else{var D,R,N=I-14,U=0;if(16===I&&(U=y),k!==U&&(w=0,k=U),D=w,w>0&&(w-=2,w<<=N),g+(R=(w+=p.readBits(N)+3)-D)>o)throw new Error("[ReadHuffmanCodeLengths] symbol + repeat_delta > num_symbols");for(var W=0;W<R;W++)u[g+W]=k;g+=R,0!==k&&(P-=R<<15-k)}}if(0!==P)throw new Error("[ReadHuffmanCodeLengths] space = "+P);for(;g<o;g++)u[g]=0}(D,i,w,p)}if(0===(g=x(o,u,8,w,i)))throw new Error("[ReadHuffmanCode] BuildHuffmanTable failed: ");return g}function ReadBlockLength(i,o,u){var p,g;return p=ReadSymbol(i,o,u),g=P.kBlockLengthPrefixCode[p].nbits,P.kBlockLengthPrefixCode[p].offset+u.readBits(g)}function TranslateShortCodes(i,o,u){var p;return i<16?(u+=I[i],p=o[u&=3]+B[i]):p=i-16+1,p}function MoveToFront(i,o){for(var u=i[o],p=o;p;--p)i[p]=i[p-1];i[0]=u}function HuffmanTreeGroup(i,o){this.alphabet_size=i,this.num_htrees=o,this.codes=new Array(o+o*D[i+31>>>5]),this.htrees=new Uint32Array(o)}function DecodeContextMap(i,o){var u,p,g={num_htrees:null,context_map:null},y=0;o.readMoreInput();var w=g.num_htrees=DecodeVarLenUint8(o)+1,x=g.context_map=new Uint8Array(i);if(w<=1)return g;for(o.readBits(1)&&(y=o.readBits(4)+1),u=[],p=0;p<1080;p++)u[p]=new _(0,0);for(ReadHuffmanCode(w+y,u,0,o),p=0;p<i;){var k;if(o.readMoreInput(),0===(k=ReadSymbol(u,0,o)))x[p]=0,++p;else if(k<=y)for(var P=1+(1<<k)+o.readBits(k);--P;){if(p>=i)throw new Error("[DecodeContextMap] i >= context_map_size");x[p]=0,++p}else x[p]=k-y,++p}return o.readBits(1)&&function InverseMoveToFrontTransform(i,o){var u,p=new Uint8Array(256);for(u=0;u<256;++u)p[u]=u;for(u=0;u<o;++u){var g=i[u];i[u]=p[g],g&&MoveToFront(p,g)}}(x,i),g}function DecodeBlockType(i,o,u,p,g,y,w){var _,x=2*u,k=u,P=ReadSymbol(o,1080*u,w);(_=0===P?g[x+(1&y[k])]:1===P?g[x+(y[k]-1&1)]+1:P-2)>=i&&(_-=i),p[u]=_,g[x+(1&y[k])]=_,++y[k]}function CopyUncompressedBlockToOutput(i,o,u,p,g,w){var _,x=g+1,k=u&g,P=w.pos_&y.IBUF_MASK;if(o<8||w.bit_pos_+(o<<3)<w.bit_end_pos_)for(;o-- >0;)w.readMoreInput(),p[k++]=w.readBits(8),k===x&&(i.write(p,x),k=0);else{if(w.bit_end_pos_<32)throw new Error("[CopyUncompressedBlockToOutput] br.bit_end_pos_ < 32");for(;w.bit_pos_<32;)p[k]=w.val_>>>w.bit_pos_,w.bit_pos_+=8,++k,--o;if(P+(_=w.bit_end_pos_-w.bit_pos_>>3)>y.IBUF_MASK){for(var E=y.IBUF_MASK+1-P,O=0;O<E;O++)p[k+O]=w.buf_[P+O];_-=E,k+=E,o-=E,P=0}for(O=0;O<_;O++)p[k+O]=w.buf_[P+O];if(o-=_,(k+=_)>=x){i.write(p,x),k-=x;for(O=0;O<k;O++)p[O]=p[x+O]}for(;k+o>=x;){if(_=x-k,w.input_.read(p,k,_)<_)throw new Error("[CopyUncompressedBlockToOutput] not enough bytes");i.write(p,x),o-=_,k=0}if(w.input_.read(p,k,o)<o)throw new Error("[CopyUncompressedBlockToOutput] not enough bytes");w.reset()}}function JumpToByteBoundary(i){var o=i.bit_pos_+7&-8;return 0==i.readBits(o-i.bit_pos_)}function BrotliDecompressedSize(i){var o=new p(i),u=new y(o);return DecodeWindowBits(u),DecodeMetaBlockLength(u).meta_block_length}function BrotliDecompress(i,o){var u,p,g,x,O,I,B,D,R,N,U=0,W=0,G=0,j=[16,15,11,4],X=0,K=0,Y=0,J=[new HuffmanTreeGroup(0,0),new HuffmanTreeGroup(0,0),new HuffmanTreeGroup(0,0)],$=128+y.READ_SIZE;g=(1<<(p=DecodeWindowBits(N=new y(i))))-16,O=(x=1<<p)-1,I=new Uint8Array(x+$+w.maxDictionaryWordLength),B=x,D=[],R=[];for(var tt=0;tt<3240;tt++)D[tt]=new _(0,0),R[tt]=new _(0,0);for(;!W;){var et,rt,nt,it,ot,at,st,lt,ct,ut,ft,ht=0,dt=[1<<28,1<<28,1<<28],pt=[0],gt=[1,1,1],yt=[0,1,0,1,0,1],vt=[0],mt=null,bt=null,wt=0,_t=null,xt=0,St=0,Ct=0;for(u=0;u<3;++u)J[u].codes=null,J[u].htrees=null;N.readMoreInput();var kt=DecodeMetaBlockLength(N);if(U+(ht=kt.meta_block_length)>o.buffer.length){var At=new Uint8Array(U+ht);At.set(o.buffer),o.buffer=At}if(W=kt.input_end,et=kt.is_uncompressed,kt.is_metadata)for(JumpToByteBoundary(N);ht>0;--ht)N.readMoreInput(),N.readBits(8);else if(0!==ht)if(et)N.bit_pos_=N.bit_pos_+7&-8,CopyUncompressedBlockToOutput(o,ht,U,I,O,N),U+=ht;else{for(u=0;u<3;++u)gt[u]=DecodeVarLenUint8(N)+1,gt[u]>=2&&(ReadHuffmanCode(gt[u]+2,D,1080*u,N),ReadHuffmanCode(26,R,1080*u,N),dt[u]=ReadBlockLength(R,1080*u,N),vt[u]=1);for(N.readMoreInput(),it=(1<<(rt=N.readBits(2)))-1,ot=(nt=16+(N.readBits(4)<<rt))+(48<<rt),mt=new Uint8Array(gt[0]),u=0;u<gt[0];++u)N.readMoreInput(),mt[u]=N.readBits(2)<<1;var Pt=DecodeContextMap(gt[0]<<6,N);st=Pt.num_htrees,at=Pt.context_map;var Tt=DecodeContextMap(gt[2]<<2,N);for(ct=Tt.num_htrees,lt=Tt.context_map,J[0]=new HuffmanTreeGroup(256,st),J[1]=new HuffmanTreeGroup(704,gt[1]),J[2]=new HuffmanTreeGroup(ot,ct),u=0;u<3;++u)J[u].decode(N);for(bt=0,_t=0,ut=mt[pt[0]],St=k.lookupOffsets[ut],Ct=k.lookupOffsets[ut+1],ft=J[1].htrees[0];ht>0;){var Et,Ot,It,Bt,Ft,Dt,Lt,Mt,Rt,Nt,Ut,Wt;for(N.readMoreInput(),0===dt[1]&&(DecodeBlockType(gt[1],D,1,pt,yt,vt,N),dt[1]=ReadBlockLength(R,1080,N),ft=J[1].htrees[pt[1]]),--dt[1],(Ot=(Et=ReadSymbol(J[1].codes,ft,N))>>6)>=2?(Ot-=2,Lt=-1):Lt=0,It=P.kInsertRangeLut[Ot]+(Et>>3&7),Bt=P.kCopyRangeLut[Ot]+(7&Et),Ft=P.kInsertLengthPrefixCode[It].offset+N.readBits(P.kInsertLengthPrefixCode[It].nbits),Dt=P.kCopyLengthPrefixCode[Bt].offset+N.readBits(P.kCopyLengthPrefixCode[Bt].nbits),K=I[U-1&O],Y=I[U-2&O],Rt=0;Rt<Ft;++Rt)N.readMoreInput(),0===dt[0]&&(DecodeBlockType(gt[0],D,0,pt,yt,vt,N),dt[0]=ReadBlockLength(R,0,N),bt=pt[0]<<6,ut=mt[pt[0]],St=k.lookupOffsets[ut],Ct=k.lookupOffsets[ut+1]),wt=at[bt+(k.lookup[St+K]|k.lookup[Ct+Y])],--dt[0],Y=K,K=ReadSymbol(J[0].codes,J[0].htrees[wt],N),I[U&O]=K,(U&O)===O&&o.write(I,x),++U;if((ht-=Ft)<=0)break;if(Lt<0)if(N.readMoreInput(),0===dt[2]&&(DecodeBlockType(gt[2],D,2,pt,yt,vt,N),dt[2]=ReadBlockLength(R,2160,N),_t=pt[2]<<2),--dt[2],xt=lt[_t+(255&(Dt>4?3:Dt-2))],(Lt=ReadSymbol(J[2].codes,J[2].htrees[xt],N))>=nt)Wt=(Lt-=nt)&it,Lt=nt+((zt=(2+(1&(Lt>>=rt))<<(Ut=1+(Lt>>1)))-4)+N.readBits(Ut)<<rt)+Wt;if((Mt=TranslateShortCodes(Lt,j,X))<0)throw new Error("[BrotliDecompress] invalid distance");if(Nt=U&O,Mt>(G=U<g&&G!==g?U:g)){if(!(Dt>=w.minDictionaryWordLength&&Dt<=w.maxDictionaryWordLength))throw new Error("Invalid backward reference. pos: "+U+" distance: "+Mt+" len: "+Dt+" bytes left: "+ht);var zt=w.offsetsByLength[Dt],Gt=Mt-G-1,jt=w.sizeBitsByLength[Dt],Vt=Gt>>jt;if(zt+=(Gt&(1<<jt)-1)*Dt,!(Vt<E.kNumTransforms))throw new Error("Invalid backward reference. pos: "+U+" distance: "+Mt+" len: "+Dt+" bytes left: "+ht);var Ht=E.transformDictionaryWord(I,Nt,zt,Dt,Vt);if(U+=Ht,ht-=Ht,(Nt+=Ht)>=B){o.write(I,x);for(var qt=0;qt<Nt-B;qt++)I[qt]=I[B+qt]}}else{if(Lt>0&&(j[3&X]=Mt,++X),Dt>ht)throw new Error("Invalid backward reference. pos: "+U+" distance: "+Mt+" len: "+Dt+" bytes left: "+ht);for(Rt=0;Rt<Dt;++Rt)I[U&O]=I[U-Mt&O],(U&O)===O&&o.write(I,x),++U,--ht}K=I[U-1&O],Y=I[U-2&O]}U&=1073741823}}o.write(I,U&O)}HuffmanTreeGroup.prototype.decode=function(i){var o,u=0;for(o=0;o<this.num_htrees;++o)this.htrees[o]=u,u+=ReadHuffmanCode(this.alphabet_size,this.codes,u,i)},o.BrotliDecompressedSize=BrotliDecompressedSize,o.BrotliDecompressBuffer=function BrotliDecompressBuffer(i,o){var u=new p(i);null==o&&(o=BrotliDecompressedSize(i));var y=new Uint8Array(o),w=new g(y);return BrotliDecompress(u,w),w.pos<w.buffer.length&&(w.buffer=w.buffer.subarray(0,w.pos)),w.buffer},o.BrotliDecompress=BrotliDecompress,w.init()},function(i,o){function BrotliInput(i){this.buffer=i,this.pos=0}function BrotliOutput(i){this.buffer=i,this.pos=0}BrotliInput.prototype.read=function(i,o,u){this.pos+u>this.buffer.length&&(u=this.buffer.length-this.pos);for(var p=0;p<u;p++)i[o+p]=this.buffer[this.pos+p];return this.pos+=u,u},o.BrotliInput=BrotliInput,BrotliOutput.prototype.write=function(i,o){if(this.pos+o>this.buffer.length)throw new Error("Output buffer is not large enough");return this.buffer.set(i.subarray(0,o),this.pos),this.pos+=o,o},o.BrotliOutput=BrotliOutput},function(i,o,u){var p=u(529);o.init=function(){o.dictionary=p.init()},o.offsetsByLength=new Uint32Array([0,0,0,0,0,4096,9216,21504,35840,44032,53248,63488,74752,87040,93696,100864,104704,106752,108928,113536,115968,118528,119872,121280,122016]),o.sizeBitsByLength=new Uint8Array([0,0,0,0,10,10,11,11,10,10,10,10,10,9,9,8,7,7,8,7,7,6,6,5,5]),o.minDictionaryWordLength=4,o.maxDictionaryWordLength=24},function(i,o){function HuffmanCode(i,o){this.bits=i,this.value=o}o.HuffmanCode=HuffmanCode;function GetNextKey(i,o){for(var u=1<<o-1;i&u;)u>>=1;return(i&u-1)+u}function ReplicateValue(i,o,u,p,g){do{i[o+(p-=u)]=new HuffmanCode(g.bits,g.value)}while(p>0)}function NextTableBitSize(i,o,u){for(var p=1<<o-u;o<15&&!((p-=i[o])<=0);)++o,p<<=1;return o-u}o.BrotliBuildHuffmanTable=function(i,o,u,p,g){var y,w,_,x,k,P,E,O,I,B,D=o,R=new Int32Array(16),N=new Int32Array(16);for(B=new Int32Array(g),w=0;w<g;w++)R[p[w]]++;for(N[1]=0,y=1;y<15;y++)N[y+1]=N[y]+R[y];for(w=0;w<g;w++)0!==p[w]&&(B[N[p[w]]++]=w);if(I=O=1<<(E=u),1===N[15]){for(_=0;_<I;++_)i[o+_]=new HuffmanCode(0,65535&B[0]);return I}for(_=0,w=0,y=1,x=2;y<=u;++y,x<<=1)for(;R[y]>0;--R[y])ReplicateValue(i,o+_,x,O,new HuffmanCode(255&y,65535&B[w++])),_=GetNextKey(_,y);for(P=I-1,k=-1,y=u+1,x=2;y<=15;++y,x<<=1)for(;R[y]>0;--R[y])(_&P)!==k&&(o+=O,I+=O=1<<(E=NextTableBitSize(R,y,u)),i[D+(k=_&P)]=new HuffmanCode(E+u&255,o-D-k&65535)),ReplicateValue(i,o+(_>>u),x,O,new HuffmanCode(y-u&255,65535&B[w++])),_=GetNextKey(_,y);return I}},function(i,o,u){"use strict";var p,g,y,w,_,x,k,P,E,O,I,B,D,R,N,U=u(535),W=u(536),G=u(537);G.BK,k=G.CR,G.LF,G.NL,_=G.CB,y=G.BA,G.SP,R=G.WJ,D=G.SP,w=G.BK,P=G.LF,E=G.NL,p=G.AI,g=G.AL,I=G.SA,B=G.SG,N=G.XX,x=G.CJ,G.ID,O=G.NS;var j=u(538),X=j.DI_BRK,K=j.IN_BRK,Y=j.CI_BRK,J=j.CP_BRK,$=(j.PR_BRK,j.pairTable),tt=new U(W.toByteArray("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")),et=function mapClass(i){switch(i){case p:return g;case I:case B:case N:return g;case x:return O;default:return i}},rt=function mapFirst(i){switch(i){case P:case E:return w;case _:return y;case D:return R;default:return i}},nt=function Break(i,o){void 0===o&&(o=!1),this.position=i,this.required=o},it=function(){function LineBreaker(i){this.string=i,this.pos=0,this.lastPos=0,this.curClass=null,this.nextClass=null}var i=LineBreaker.prototype;return i.nextCodePoint=function nextCodePoint(){var i=this.string.charCodeAt(this.pos++),o=this.string.charCodeAt(this.pos);return 55296<=i&&i<=56319&&56320<=o&&o<=57343?(this.pos++,1024*(i-55296)+(o-56320)+65536):i},i.nextCharClass=function nextCharClass(){return et(tt.get(this.nextCodePoint()))},i.nextBreak=function nextBreak(){for(null==this.curClass&&(this.curClass=rt(this.nextCharClass()));this.pos<this.string.length;){this.lastPos=this.pos;var i=this.nextClass;if(this.nextClass=this.nextCharClass(),this.curClass===w||this.curClass===k&&this.nextClass!==P)return this.curClass=rt(et(this.nextClass)),new nt(this.lastPos,!0);var o=void 0;switch(this.nextClass){case D:o=this.curClass;break;case w:case P:case E:o=w;break;case k:o=k;break;case _:o=y}if(null==o){var u=!1;switch($[this.curClass][this.nextClass]){case X:u=!0;break;case K:u=i===D;break;case Y:if(!(u=i===D))continue;break;case J:if(i!==D)continue}if(this.curClass=this.nextClass,u)return new nt(this.lastPos)}else if(this.curClass=o,this.nextClass===_)return new nt(this.lastPos)}if(this.pos>=this.string.length)return this.lastPos<this.string.length?(this.lastPos=this.string.length,new nt(this.string.length)):null},LineBreaker}();i.exports=it},function(i,o,u){"use strict";function TraversalTracker(){this.events={}}TraversalTracker.prototype.startTracking=function(i,o){var u=this.events[i]||(this.events[i]=[]);u.indexOf(o)<0&&u.push(o)},TraversalTracker.prototype.stopTracking=function(i,o){var u=this.events[i];if(u){var p=u.indexOf(o);p>=0&&u.splice(p,1)}},TraversalTracker.prototype.emit=function(i){var o=Array.prototype.slice.call(arguments,1),u=this.events[i];u&&u.forEach((function(i){i.apply(this,o)}))},TraversalTracker.prototype.auto=function(i,o,u){this.startTracking(i,o),u(),this.stopTracking(i,o)},i.exports=TraversalTracker},function(i,o,u){"use strict";var p=u(0).isString,g=u(0).isArray,y=u(0).isUndefined,w=u(0).isNull;function StyleContextStack(i,o){this.defaultStyle=o||{},this.styleDictionary=i,this.styleOverrides=[]}StyleContextStack.prototype.clone=function(){var i=new StyleContextStack(this.styleDictionary,this.defaultStyle);return this.styleOverrides.forEach((function(o){i.styleOverrides.push(o)})),i},StyleContextStack.prototype.push=function(i){this.styleOverrides.push(i)},StyleContextStack.prototype.pop=function(i){for(i=i||1;i-- >0;)this.styleOverrides.pop()},StyleContextStack.prototype.autopush=function(i){if(p(i))return 0;var o=[];i.style&&(o=g(i.style)?i.style:[i.style]);for(var u=0,_=o.length;u<_;u++)this.push(o[u]);var x={},k=!1;return["font","fontSize","fontFeatures","bold","italics","alignment","color","columnGap","fillColor","fillOpacity","decoration","decorationStyle","decorationColor","background","lineHeight","characterSpacing","noWrap","markerColor","leadingIndent","sup","sub"].forEach((function(o){y(i[o])||w(i[o])||(x[o]=i[o],k=!0)})),k&&this.push(x),o.length+(k?1:0)},StyleContextStack.prototype.auto=function(i,o){var u=this.autopush(i),p=o();return u>0&&this.pop(u),p},StyleContextStack.prototype.getProperty=function(i){if(this.styleOverrides)for(var o=this.styleOverrides.length-1;o>=0;o--){var u=this.styleOverrides[o];if(p(u)){var g=this.styleDictionary[u];if(g&&!y(g[i])&&!w(g[i]))return g[i]}else if(!y(u[i])&&!w(u[i]))return u[i]}return this.defaultStyle&&this.defaultStyle[i]},i.exports=StyleContextStack},function(i,o,u){"use strict";var p=u(291),g=u(0).isString;function DocumentContext(i,o){this.pages=[],this.pageMargins=o,this.x=o.left,this.availableWidth=i.width-o.left-o.right,this.availableHeight=0,this.page=-1,this.snapshots=[],this.endingCell=null,this.tracker=new p,this.backgroundLength=[],this.addPage(i)}DocumentContext.prototype.beginColumnGroup=function(){this.snapshots.push({x:this.x,y:this.y,availableHeight:this.availableHeight,availableWidth:this.availableWidth,page:this.page,bottomMost:{x:this.x,y:this.y,availableHeight:this.availableHeight,availableWidth:this.availableWidth,page:this.page},endingCell:this.endingCell,lastColumnWidth:this.lastColumnWidth}),this.lastColumnWidth=0},DocumentContext.prototype.beginColumn=function(i,o,u){var p=this.snapshots[this.snapshots.length-1];this.calculateBottomMost(p),this.endingCell=u,this.page=p.page,this.x=this.x+this.lastColumnWidth+(o||0),this.y=p.y,this.availableWidth=i,this.availableHeight=p.availableHeight,this.lastColumnWidth=i},DocumentContext.prototype.calculateBottomMost=function(i){this.endingCell?(this.saveContextInEndingCell(this.endingCell),this.endingCell=null):i.bottomMost=function bottomMostContext(i,o){var u;u=i.page>o.page?i:o.page>i.page?o:i.y>o.y?i:o;return{page:u.page,x:u.x,y:u.y,availableHeight:u.availableHeight,availableWidth:u.availableWidth}}(this,i.bottomMost)},DocumentContext.prototype.markEnding=function(i){this.page=i._columnEndingContext.page,this.x=i._columnEndingContext.x,this.y=i._columnEndingContext.y,this.availableWidth=i._columnEndingContext.availableWidth,this.availableHeight=i._columnEndingContext.availableHeight,this.lastColumnWidth=i._columnEndingContext.lastColumnWidth},DocumentContext.prototype.saveContextInEndingCell=function(i){i._columnEndingContext={page:this.page,x:this.x,y:this.y,availableHeight:this.availableHeight,availableWidth:this.availableWidth,lastColumnWidth:this.lastColumnWidth}},DocumentContext.prototype.completeColumnGroup=function(i){var o=this.snapshots.pop();this.calculateBottomMost(o),this.endingCell=null,this.x=o.x;var u=o.bottomMost.y;i&&(o.page===o.bottomMost.page?o.y+i>u&&(u=o.y+i):u+=i),this.y=u,this.page=o.bottomMost.page,this.availableWidth=o.availableWidth,this.availableHeight=o.bottomMost.availableHeight,i&&(this.availableHeight-=u-o.bottomMost.y),this.lastColumnWidth=o.lastColumnWidth},DocumentContext.prototype.addMargin=function(i,o){this.x+=i,this.availableWidth-=i+(o||0)},DocumentContext.prototype.moveDown=function(i){return this.y+=i,this.availableHeight-=i,this.availableHeight>0},DocumentContext.prototype.initializePage=function(){this.y=this.pageMargins.top,this.availableHeight=this.getCurrentPage().pageSize.height-this.pageMargins.top-this.pageMargins.bottom,this.pageSnapshot().availableWidth=this.getCurrentPage().pageSize.width-this.pageMargins.left-this.pageMargins.right},DocumentContext.prototype.pageSnapshot=function(){return this.snapshots[0]?this.snapshots[0]:this},DocumentContext.prototype.moveTo=function(i,o){null!=i&&(this.x=i,this.availableWidth=this.getCurrentPage().pageSize.width-this.x-this.pageMargins.right),null!=o&&(this.y=o,this.availableHeight=this.getCurrentPage().pageSize.height-this.y-this.pageMargins.bottom)},DocumentContext.prototype.moveToRelative=function(i,o){null!=i&&(this.x=this.x+i),null!=o&&(this.y=this.y+o)},DocumentContext.prototype.beginDetachedBlock=function(){this.snapshots.push({x:this.x,y:this.y,availableHeight:this.availableHeight,availableWidth:this.availableWidth,page:this.page,endingCell:this.endingCell,lastColumnWidth:this.lastColumnWidth})},DocumentContext.prototype.endDetachedBlock=function(){var i=this.snapshots.pop();this.x=i.x,this.y=i.y,this.availableWidth=i.availableWidth,this.availableHeight=i.availableHeight,this.page=i.page,this.endingCell=i.endingCell,this.lastColumnWidth=i.lastColumnWidth};var getPageSize=function(i,o){return(o=function pageOrientation(i,o){return void 0===i?o:g(i)&&"landscape"===i.toLowerCase()?"landscape":"portrait"}(o,i.pageSize.orientation))!==i.pageSize.orientation?{orientation:o,width:i.pageSize.height,height:i.pageSize.width}:{orientation:i.pageSize.orientation,width:i.pageSize.width,height:i.pageSize.height}};DocumentContext.prototype.moveToNextPage=function(i){var o=this.page+1,u=this.page,p=this.y,g=o>=this.pages.length;if(g){var y=this.availableWidth,w=this.getCurrentPage().pageSize.orientation,_=getPageSize(this.getCurrentPage(),i);this.addPage(_),w===_.orientation&&(this.availableWidth=y)}else this.page=o,this.initializePage();return{newPageCreated:g,prevPage:u,prevY:p,y:this.y}},DocumentContext.prototype.addPage=function(i){var o={items:[],pageSize:i};return this.pages.push(o),this.backgroundLength.push(0),this.page=this.pages.length-1,this.initializePage(),this.tracker.emit("pageAdded"),o},DocumentContext.prototype.getCurrentPage=function(){return this.page<0||this.page>=this.pages.length?null:this.pages[this.page]},DocumentContext.prototype.getCurrentPosition=function(){var i=this.getCurrentPage().pageSize,o=i.height-this.pageMargins.top-this.pageMargins.bottom,u=i.width-this.pageMargins.left-this.pageMargins.right;return{pageNumber:this.page+1,pageOrientation:i.orientation,pageInnerHeight:o,pageInnerWidth:u,left:this.x,top:this.y,verticalRatio:(this.y-this.pageMargins.top)/o,horizontalRatio:(this.x-this.pageMargins.left)/u}},i.exports=DocumentContext},function(i,o,u){"use strict";function Line(i){this.maxWidth=i,this.leadingCut=0,this.trailingCut=0,this.inlineWidths=0,this.inlines=[]}Line.prototype.getAscenderHeight=function(){var i=0;return this.inlines.forEach((function(o){i=Math.max(i,o.font.ascender/1e3*o.fontSize)})),i},Line.prototype.hasEnoughSpaceForInline=function(i,o){if(o=o||[],0===this.inlines.length)return!0;if(this.newLineForced)return!1;var u=i.width,p=i.trailingCut||0;if(i.noNewLine)for(var g=0,y=o.length;g<y;g++){var w=o[g];if(u+=w.width,p+=w.trailingCut||0,!w.noNewLine)break}return this.inlineWidths+u-this.leadingCut-p<=this.maxWidth},Line.prototype.addInline=function(i){0===this.inlines.length&&(this.leadingCut=i.leadingCut||0),this.trailingCut=i.trailingCut||0,i.x=this.inlineWidths-this.leadingCut,this.inlines.push(i),this.inlineWidths+=i.width,i.lineEnd&&(this.newLineForced=!0)},Line.prototype.getWidth=function(){return this.inlineWidths-this.leadingCut-this.trailingCut},Line.prototype.getAvailableWidth=function(){return this.maxWidth-this.getWidth()},Line.prototype.getHeight=function(){var i=0;return this.inlines.forEach((function(o){i=Math.max(i,o.height||0)})),i},i.exports=Line},function(i,o,u){"use strict";var p=u(1),g=u(58),y=u(46),w=u(45),_=u(18),x=u(41),k=u(138),P=function AggregateError(i,o){var u=this;if(!(u instanceof P))return new P(i,o);y&&(u=y(new Error(void 0),g(u))),void 0!==o&&_(u,"message",String(o));var p=[];return k(i,p.push,{that:p}),_(u,"errors",p),u};P.prototype=w(Error.prototype,{constructor:x(5,P),message:x(5,""),name:x(5,"AggregateError")}),p({global:!0},{AggregateError:P})},function(i,o,u){var p=u(4);i.exports=p.Promise},function(i,o,u){var p,g,y,w=u(4),_=u(3),x=u(93),k=u(209),P=u(140),E=u(298),O=u(114),I=w.location,B=w.setImmediate,D=w.clearImmediate,R=w.process,N=w.MessageChannel,U=w.Dispatch,W=0,G={},run=function(i){if(G.hasOwnProperty(i)){var o=G[i];delete G[i],o()}},runner=function(i){return function(){run(i)}},listener=function(i){run(i.data)},post=function(i){w.postMessage(i+"",I.protocol+"//"+I.host)};B&&D||(B=function setImmediate(i){for(var o=[],u=1;arguments.length>u;)o.push(arguments[u++]);return G[++W]=function(){("function"==typeof i?i:Function(i)).apply(void 0,o)},p(W),W},D=function clearImmediate(i){delete G[i]},O?p=function(i){R.nextTick(runner(i))}:U&&U.now?p=function(i){U.now(runner(i))}:N&&!E?(y=(g=new N).port2,g.port1.onmessage=listener,p=x(y.postMessage,y,1)):w.addEventListener&&"function"==typeof postMessage&&!w.importScripts&&I&&"file:"!==I.protocol&&!_(post)?(p=post,w.addEventListener("message",listener,!1)):p="onreadystatechange"in P("script")?function(i){k.appendChild(P("script")).onreadystatechange=function(){k.removeChild(this),run(i)}}:function(i){setTimeout(runner(i),0)}),i.exports={set:B,clear:D}},function(i,o,u){var p=u(148);i.exports=/(iphone|ipod|ipad).*applewebkit/i.test(p)},function(i,o,u){var p=u(12),g=u(11),y=u(106);i.exports=function(i,o){if(p(i),g(o)&&o.constructor===i)return o;var u=y.f(i);return(0,u.resolve)(o),u.promise}},function(i,o,u){"use strict";var p=u(1),g=u(31),y=u(106),w=u(139),_=u(138);p({target:"Promise",stat:!0},{allSettled:function allSettled(i){var o=this,u=y.f(o),p=u.resolve,x=u.reject,k=w((function(){var u=g(o.resolve),y=[],w=0,x=1;_(i,(function(i){var g=w++,_=!1;y.push(void 0),x++,u.call(o,i).then((function(i){_||(_=!0,y[g]={status:"fulfilled",value:i},--x||p(y))}),(function(i){_||(_=!0,y[g]={status:"rejected",reason:i},--x||p(y))}))})),--x||p(y)}));return k.error&&x(k.value),u.promise}})},function(i,o,u){"use strict";var p=u(1),g=u(31),y=u(30),w=u(106),_=u(139),x=u(138);p({target:"Promise",stat:!0},{any:function any(i){var o=this,u=w.f(o),p=u.resolve,k=u.reject,P=_((function(){var u=g(o.resolve),w=[],_=0,P=1,E=!1;x(i,(function(i){var g=_++,x=!1;w.push(void 0),P++,u.call(o,i).then((function(i){x||E||(E=!0,p(i))}),(function(i){x||E||(x=!0,w[g]=i,--P||k(new(y("AggregateError"))(w,"No one promise resolved")))}))})),--P||k(new(y("AggregateError"))(w,"No one promise resolved"))}));return P.error&&k(P.value),u.promise}})},function(i,o,u){var p=u(303),g=u(572);void 0===g.pdfMake&&(g.pdfMake=p),i.exports=p},function(i,o,u){"use strict";(function(o,p){var g=u(0).isFunction,y=u(0).isUndefined,w=(u(0).isNull,u(306).saveAs),_={Roboto:{normal:"Roboto-Regular.ttf",bold:"Roboto-Medium.ttf",italics:"Roboto-Italic.ttf",bolditalics:"Roboto-MediumItalic.ttf"}};function Document(i,o,u,p){this.docDefinition=i,this.tableLayouts=o||null,this.fonts=u||_,this.vfs=p}Document.prototype._createDoc=function(i,o){i=i||{},this.tableLayouts&&(i.tableLayouts=this.tableLayouts);var p=new(u(307))(this.fonts);if(u(54).bindFS(this.vfs),!g(o))return p.createPdfKitDocument(this.docDefinition,i);var y=new(u(557))(u(54));for(var w in this.fonts)this.fonts.hasOwnProperty(w)&&(this.fonts[w].normal&&y.resolve(this.fonts[w].normal),this.fonts[w].bold&&y.resolve(this.fonts[w].bold),this.fonts[w].italics&&y.resolve(this.fonts[w].italics),this.fonts[w].bolditalics&&y.resolve(this.fonts[w].bolditalics));if(this.docDefinition.images)for(var _ in this.docDefinition.images)this.docDefinition.images.hasOwnProperty(_)&&y.resolve(this.docDefinition.images[_]);var x=this;y.resolved().then((function(){var u=p.createPdfKitDocument(x.docDefinition,i);o(u)}),(function(i){throw i}))},Document.prototype._flushDoc=function(i,u){var p,g=[];i.on("readable",(function(){for(var o;null!==(o=i.read(9007199254740991));)g.push(o)})),i.on("end",(function(){p=o.concat(g),u(p,i._pdfMakePages)})),i.end()},Document.prototype._getPages=function(i,o){if(!o)throw"_getPages is an async method and needs a callback argument";var u=this;this._createDoc(i,(function(i){u._flushDoc(i,(function(i,u){o(u)}))}))},Document.prototype._bufferToBlob=function(i){var o;try{o=new Blob([i],{type:"application/pdf"})}catch(p){if("InvalidStateError"===p.name){var u=new Uint8Array(i);o=new Blob([u.buffer],{type:"application/pdf"})}}if(!o)throw"Could not generate blob";return o},Document.prototype._openWindow=function(){var i=window.open("","_blank");if(null===i)throw"Open PDF in new window blocked by browser";return i},Document.prototype._openPdf=function(i,o){o||(o=this._openWindow());try{this.getBlob((function(i){var u=(window.URL||window.webkitURL).createObjectURL(i);o.location.href=u}),i)}catch(i){throw o.close(),i}},Document.prototype.open=function(i,o){(i=i||{}).autoPrint=!1,o=o||null,this._openPdf(i,o)},Document.prototype.print=function(i,o){(i=i||{}).autoPrint=!0,o=o||null,this._openPdf(i,o)},Document.prototype.download=function(i,o,u){g(i)&&(y(o)||(u=o),o=i,i=null),i=i||"file.pdf",this.getBlob((function(u){w(u,i),g(o)&&o()}),u)},Document.prototype.getBase64=function(i,o){if(!i)throw"getBase64 is an async method and needs a callback argument";this.getBuffer((function(o){i(o.toString("base64"))}),o)},Document.prototype.getDataUrl=function(i,o){if(!i)throw"getDataUrl is an async method and needs a callback argument";this.getBuffer((function(o){i("data:application/pdf;base64,"+o.toString("base64"))}),o)},Document.prototype.getBlob=function(i,o){if(!i)throw"getBlob is an async method and needs a callback argument";var u=this;this.getBuffer((function(o){var p=u._bufferToBlob(o);i(p)}),o)},Document.prototype.getBuffer=function(i,o){if(!i)throw"getBuffer is an async method and needs a callback argument";var u=this;this._createDoc(o,(function(o){u._flushDoc(o,(function(o){i(o)}))}))},Document.prototype.getStream=function(i,o){if(!g(o))return this._createDoc(i);this._createDoc(i,(function(i){o(i)}))},i.exports={createPdf:function(i,o,u,g){if(!function canCreatePdf(){return!(!Object.keys||"undefined"==typeof Uint16Array)}())throw"Your browser does not provide the level of support needed";return new Document(i,o||p.pdfMake.tableLayouts,u||p.pdfMake.fonts,g||p.pdfMake.vfs)}}}).call(this,u(10).Buffer,u(27))},function(i,o,u){"use strict";o.byteLength=function byteLength(i){var o=getLens(i),u=o[0],p=o[1];return 3*(u+p)/4-p},o.toByteArray=function toByteArray(i){var o,u,p=getLens(i),w=p[0],_=p[1],x=new y(function _byteLength(i,o,u){return 3*(o+u)/4-u}(0,w,_)),k=0,P=_>0?w-4:w;for(u=0;u<P;u+=4)o=g[i.charCodeAt(u)]<<18|g[i.charCodeAt(u+1)]<<12|g[i.charCodeAt(u+2)]<<6|g[i.charCodeAt(u+3)],x[k++]=o>>16&255,x[k++]=o>>8&255,x[k++]=255&o;2===_&&(o=g[i.charCodeAt(u)]<<2|g[i.charCodeAt(u+1)]>>4,x[k++]=255&o);1===_&&(o=g[i.charCodeAt(u)]<<10|g[i.charCodeAt(u+1)]<<4|g[i.charCodeAt(u+2)]>>2,x[k++]=o>>8&255,x[k++]=255&o);return x},o.fromByteArray=function fromByteArray(i){for(var o,u=i.length,g=u%3,y=[],w=0,_=u-g;w<_;w+=16383)y.push(encodeChunk(i,w,w+16383>_?_:w+16383));1===g?(o=i[u-1],y.push(p[o>>2]+p[o<<4&63]+"==")):2===g&&(o=(i[u-2]<<8)+i[u-1],y.push(p[o>>10]+p[o>>4&63]+p[o<<2&63]+"="));return y.join("")};for(var p=[],g=[],y="undefined"!=typeof Uint8Array?Uint8Array:Array,w="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/",_=0,x=w.length;_<x;++_)p[_]=w[_],g[w.charCodeAt(_)]=_;function getLens(i){var o=i.length;if(o%4>0)throw new Error("Invalid string. Length must be a multiple of 4");var u=i.indexOf("=");return-1===u&&(u=o),[u,u===o?0:4-u%4]}function encodeChunk(i,o,u){for(var g,y,w=[],_=o;_<u;_+=3)g=(i[_]<<16&16711680)+(i[_+1]<<8&65280)+(255&i[_+2]),w.push(p[(y=g)>>18&63]+p[y>>12&63]+p[y>>6&63]+p[63&y]);return w.join("")}g["-".charCodeAt(0)]=62,g["_".charCodeAt(0)]=63},function(i,o){o.read=function(i,o,u,p,g){var y,w,_=8*g-p-1,x=(1<<_)-1,k=x>>1,P=-7,E=u?g-1:0,O=u?-1:1,I=i[o+E];for(E+=O,y=I&(1<<-P)-1,I>>=-P,P+=_;P>0;y=256*y+i[o+E],E+=O,P-=8);for(w=y&(1<<-P)-1,y>>=-P,P+=p;P>0;w=256*w+i[o+E],E+=O,P-=8);if(0===y)y=1-k;else{if(y===x)return w?NaN:1/0*(I?-1:1);w+=Math.pow(2,p),y-=k}return(I?-1:1)*w*Math.pow(2,y-p)},o.write=function(i,o,u,p,g,y){var w,_,x,k=8*y-g-1,P=(1<<k)-1,E=P>>1,O=23===g?Math.pow(2,-24)-Math.pow(2,-77):0,I=p?0:y-1,B=p?1:-1,D=o<0||0===o&&1/o<0?1:0;for(o=Math.abs(o),isNaN(o)||o===1/0?(_=isNaN(o)?1:0,w=P):(w=Math.floor(Math.log(o)/Math.LN2),o*(x=Math.pow(2,-w))<1&&(w--,x*=2),(o+=w+E>=1?O/x:O*Math.pow(2,1-E))*x>=2&&(w++,x/=2),w+E>=P?(_=0,w=P):w+E>=1?(_=(o*x-1)*Math.pow(2,g),w+=E):(_=o*Math.pow(2,E-1)*Math.pow(2,g),w=0));g>=8;i[u+I]=255&_,I+=B,_/=256,g-=8);for(w=w<<g|_,k+=g;k>0;i[u+I]=255&w,I+=B,w/=256,k-=8);i[u+I-B]|=128*D}},function(i,o,u){(function(u){var p,g,y;g=[],void 0===(y="function"==typeof(p=function(){"use strict";function c(i,o,u){var p=new XMLHttpRequest;p.open("GET",i),p.responseType="blob",p.onload=function(){g(p.response,o,u)},p.onerror=function(){},p.send()}function d(i){var o=new XMLHttpRequest;o.open("HEAD",i,!1);try{o.send()}catch(i){}return 200<=o.status&&299>=o.status}function e(i){try{i.dispatchEvent(new MouseEvent("click"))}catch(u){var o=document.createEvent("MouseEvents");o.initMouseEvent("click",!0,!0,window,0,0,0,80,20,!1,!1,!1,!1,0,null),i.dispatchEvent(o)}}var o="object"==typeof window&&window.window===window?window:"object"==typeof self&&self.self===self?self:"object"==typeof u&&u.global===u?u:void 0,p=o.navigator&&/Macintosh/.test(navigator.userAgent)&&/AppleWebKit/.test(navigator.userAgent)&&!/Safari/.test(navigator.userAgent),g=o.saveAs||("object"!=typeof window||window!==o?function(){}:"undefined"!=typeof HTMLAnchorElement&&"download"in HTMLAnchorElement.prototype&&!p?function(i,u,p){var g=o.URL||o.webkitURL,y=document.createElement("a");u=u||i.name||"download",y.download=u,y.rel="noopener","string"==typeof i?(y.href=i,y.origin===location.origin?e(y):d(y.href)?c(i,u,p):e(y,y.target="_blank")):(y.href=g.createObjectURL(i),setTimeout((function(){g.revokeObjectURL(y.href)}),4e4),setTimeout((function(){e(y)}),0))}:"msSaveOrOpenBlob"in navigator?function(i,o,u){if(o=o||i.name||"download","string"!=typeof i)navigator.msSaveOrOpenBlob(function b(i,o){return void 0===o?o={autoBom:!1}:"object"!=typeof o&&(o={autoBom:!o}),o.autoBom&&/^\s*(?:text\/\S*|application\/xml|\S*\/\S*\+xml)\s*;.*charset\s*=\s*utf-8/i.test(i.type)?new Blob(["\ufeff",i],{type:i.type}):i}(i,u),o);else if(d(i))c(i,o,u);else{var p=document.createElement("a");p.href=i,p.target="_blank",setTimeout((function(){e(p)}))}}:function(i,u,g,y){if((y=y||open("","_blank"))&&(y.document.title=y.document.body.innerText="downloading..."),"string"==typeof i)return c(i,u,g);var w="application/octet-stream"===i.type,_=/constructor/i.test(o.HTMLElement)||o.safari,x=/CriOS\/[\d]+/.test(navigator.userAgent);if((x||w&&_||p)&&"undefined"!=typeof FileReader){var k=new FileReader;k.onloadend=function(){var i=k.result;i=x?i:i.replace(/^data:[^;]*;/,"data:attachment/file;"),y?y.location.href=i:location=i,y=null},k.readAsDataURL(i)}else{var P=o.URL||o.webkitURL,E=P.createObjectURL(i);y?y.location=E:location.href=E,y=null,setTimeout((function(){P.revokeObjectURL(E)}),4e4)}});o.saveAs=g.saveAs=g,i.exports=g})?p.apply(o,g):p)||(i.exports=y)}).call(this,u(27))},function(i,o,u){"use strict";var p=u(308),g=u(540),y=u(541),w=u(548),_=u(549),x=u(550),k=u(554),P=u(199),E=u(0).isFunction,O=u(0).isString,I=u(0).isNumber,B=u(0).isBoolean,D=u(0).isArray,R=u(0).isUndefined;function PdfPrinter(i){this.fontDescriptors=i}function fixPageMargins(i){if(I(i))i={left:i,right:i,top:i,bottom:i};else if(D(i))if(2===i.length)i={left:i[0],top:i[1],right:i[0],bottom:i[1]};else{if(4!==i.length)throw"Invalid pageMargins definition";i={left:i[0],top:i[1],right:i[2],bottom:i[3]}}return i}function updatePageOrientationInOptions(i,o){var u=o.options.size[0]>o.options.size[1]?"landscape":"portrait";if(i.pageSize.orientation!==u){var p=o.options.size[0],g=o.options.size[1];o.options.size=[g,p]}}function offsetText(i,o){var u=i;return o.sup&&(u-=.75*o.fontSize),o.sub&&(u+=.35*o.fontSize),u}function renderLine(i,o,u,p){function preparePageNodeRefLine(i,o){var u,p,g=new P(null);if(R(i.positions))throw"Page reference id not found";var y=i.positions[0].pageNumber.toString();switch(o.text=y,u=g.widthOfString(o.text,o.font,o.fontSize,o.characterSpacing,o.fontFeatures),p=o.width-u,o.width=u,o.alignment){case"right":o.x+=p;break;case"center":o.x+=p/2}}i._pageNodeRef&&preparePageNodeRefLine(i._pageNodeRef,i.inlines[0]),o=o||0,u=u||0;var g=i.getHeight(),y=g-i.getAscenderHeight();k.drawBackground(i,o,u,p);for(var w=0,_=i.inlines.length;w<_;w++){var x=i.inlines[w],E=g-x.font.ascender/1e3*x.fontSize-y;x._pageNodeRef&&preparePageNodeRefLine(x._pageNodeRef,x);var O={lineBreak:!1,textWidth:x.width,characterSpacing:x.characterSpacing,wordCount:1,link:x.link};x.linkToDestination&&(O.goTo=x.linkToDestination),i.id&&0===w&&(O.destination=i.id),x.fontFeatures&&(O.features=x.fontFeatures);var B=I(x.opacity)?x.opacity:1;p.opacity(B),p.fill(x.color||"black"),p._font=x.font,p.fontSize(x.fontSize);var D=offsetText(u+E,x);if(p.text(x.text,o+x.x,D,O),x.linkToPage){p.ref({Type:"Action",S:"GoTo",D:[x.linkToPage,0,0]}).end();p.annotate(o+x.x,D,x.width,x.height,{Subtype:"Link",Dest:[x.linkToPage-1,"XYZ",null,null,null]})}}k.drawDecorations(i,o,u,p)}function renderWatermark(i,o){var u=i.watermark;o.fill(u.color),o.opacity(u.opacity),o.save(),o.rotate(u.angle,{origin:[o.page.width/2,o.page.height/2]});var p=o.page.width/2-u._size.size.width/2,g=o.page.height/2-u._size.size.height/2;o._font=u.font,o.fontSize(u.fontSize),o.text(u.text,p,g,{lineBreak:!1}),o.restore()}function renderVector(i,o){o.lineWidth(i.lineWidth||1),i.dash?o.dash(i.dash.length,{space:i.dash.space||i.dash.length,phase:i.dash.phase||0}):o.undash(),o.lineJoin(i.lineJoin||"miter"),o.lineCap(i.lineCap||"butt");var u=null;switch(i.type){case"ellipse":o.ellipse(i.x,i.y,i.r1,i.r2),i.linearGradient&&(u=o.linearGradient(i.x-i.r1,i.y,i.x+i.r1,i.y));break;case"rect":i.r?o.roundedRect(i.x,i.y,i.w,i.h,i.r):o.rect(i.x,i.y,i.w,i.h),i.linearGradient&&(u=o.linearGradient(i.x,i.y,i.x+i.w,i.y));break;case"line":o.moveTo(i.x1,i.y1),o.lineTo(i.x2,i.y2);break;case"polyline":if(0===i.points.length)break;o.moveTo(i.points[0].x,i.points[0].y);for(var p=1,g=i.points.length;p<g;p++)o.lineTo(i.points[p].x,i.points[p].y);if(i.points.length>1){var y=i.points[0],w=i.points[i.points.length-1];(i.closePath||y.x===w.x&&y.y===w.y)&&o.closePath()}break;case"path":o.path(i.d)}if(i.linearGradient&&u){var _=1/(i.linearGradient.length-1);for(p=0;p<i.linearGradient.length;p++)u.stop(p*_,i.linearGradient[p]);i.color=u}var x=I(i.fillOpacity)?i.fillOpacity:1,k=I(i.strokeOpacity)?i.strokeOpacity:1;i.color&&i.lineColor?(o.fillColor(i.color,x),o.strokeColor(i.lineColor,k),o.fillAndStroke()):i.color?(o.fillColor(i.color,x),o.fill()):(o.strokeColor(i.lineColor||"black",k),o.stroke())}function renderImage(i,o,u,p){var g=I(i.opacity)?i.opacity:1;if(p.opacity(g),i.cover){var y=i.cover.align||"center",w=i.cover.valign||"center",_=i.cover.width?i.cover.width:i.width,x=i.cover.height?i.cover.height:i.height;p.save(),p.rect(i.x,i.y,_,x).clip(),p.image(i.image,i.x,i.y,{cover:[_,x],align:y,valign:w}),p.restore()}else p.image(i.image,i.x,i.y,{width:i._width,height:i._height});i.link&&p.link(i.x,i.y,i._width,i._height,i.link),i.linkToPage&&(p.ref({Type:"Action",S:"GoTo",D:[i.linkToPage,0,0]}).end(),p.annotate(i.x,i.y,i._width,i._height,{Subtype:"Link",Dest:[i.linkToPage-1,"XYZ",null,null,null]})),i.linkToDestination&&p.goTo(i.x,i.y,i._width,i._height,i.linkToDestination)}function renderSVG(i,o,p,g,y){var w=Object.assign({width:i._width,height:i._height,assumePt:!0},i.options);w.fontCallback=function(o,u,p){var g=o.split(",").map((function(i){return i.trim().replace(/('|")/g,"")})),w=function(i,o,u){for(var p=0;p<o.length;p++){var g=o[p].toLowerCase();for(var y in i)if(y.toLowerCase()===g)return y}return u}(y.fonts,g,i.font||"Roboto"),_=y.getFontFile(w,u,p);if(null===_){var x=y.getFontType(u,p);throw new Error("Font '"+w+"' in style '"+x+"' is not defined in the font section of the document definition.")}return _},function(){try{return u(555)}catch(i){throw new Error("Please install svg-to-pdfkit to enable svg nodes")}}()(g,i.svg,i.x,i.y,w)}function beginClip(i,o){o.save(),o.addContent(i.x+" "+i.y+" "+i.width+" "+i.height+" re"),o.clip()}function endClip(i){i.restore()}PdfPrinter.prototype.createPdfKitDocument=function(i,o){o=o||{},i.version=i.version||"1.3",i.compress=!B(i.compress)||i.compress,i.images=i.images||{},i.pageMargins=void 0!==i.pageMargins&&null!==i.pageMargins?i.pageMargins:40;var u=function fixPageSize(i,o){i&&"auto"===i.height&&(i.height=1/0);var u=function pageSize2widthAndHeight(i){if(O(i)){var o=w[i.toUpperCase()];if(!o)throw"Page size "+i+" not recognized";return{width:o[0],height:o[1]}}return i}(i||"A4");(function isNeedSwapPageSizes(i){return!!O(i)&&("portrait"===(i=i.toLowerCase())&&u.width>u.height||"landscape"===i&&u.width<u.height)})(o)&&(u={width:u.height,height:u.width});return u.orientation=u.width>u.height?"landscape":"portrait",u}(i.pageSize,i.pageOrientation),k={size:[u.width,u.height],pdfVersion:i.version,compress:i.compress,userPassword:i.userPassword,ownerPassword:i.ownerPassword,permissions:i.permissions,fontLayoutCache:!B(o.fontLayoutCache)||o.fontLayoutCache,bufferPages:o.bufferPages||!1,autoFirstPage:!1,font:null};this.pdfKitDoc=p.createPdfDocument(k),function setMetadata(i,o){function standardizePropertyKey(i){var o=i.charAt(0).toUpperCase()+i.slice(1);return-1!==["Title","Author","Subject","Keywords","Creator","Producer","CreationDate","ModDate","Trapped"].indexOf(o)?o:i.replace(/\s+/g,"")}if(o.info.Producer="pdfmake",o.info.Creator="pdfmake",i.info)for(var u in i.info){var p=i.info[u];p&&(u=standardizePropertyKey(u),o.info[u]=p)}}(i,this.pdfKitDoc),this.fontProvider=new g(this.fontDescriptors,this.pdfKitDoc);var P=new y(u,fixPageMargins(i.pageMargins),new _(this.pdfKitDoc,i.images),new x);!function registerDefaultTableLayouts(i){i.registerTableLayouts({noBorders:{hLineWidth:function(i){return 0},vLineWidth:function(i){return 0},paddingLeft:function(i){return i?4:0},paddingRight:function(i,o){return i<o.table.widths.length-1?4:0}},headerLineOnly:{hLineWidth:function(i,o){return 0===i||i===o.table.body.length?0:i===o.table.headerRows?2:0},vLineWidth:function(i){return 0},paddingLeft:function(i){return 0===i?0:8},paddingRight:function(i,o){return i===o.table.widths.length-1?0:8}},lightHorizontalLines:{hLineWidth:function(i,o){return 0===i||i===o.table.body.length?0:i===o.table.headerRows?2:1},vLineWidth:function(i){return 0},hLineColor:function(i){return 1===i?"black":"#aaa"},paddingLeft:function(i){return 0===i?0:8},paddingRight:function(i,o){return i===o.table.widths.length-1?0:8}}})}(P),o.tableLayouts&&P.registerTableLayouts(o.tableLayouts);var D=P.layoutDocument(i.content,this.fontProvider,i.styles||{},i.defaultStyle||{fontSize:12,font:"Roboto"},i.background,i.header,i.footer,i.images,i.watermark,i.pageBreakBefore),R=i.maxPagesNumber||-1;if(I(R)&&R>-1&&(D=D.slice(0,R)),u.height===1/0){var N=function calculatePageHeight(i,o){var u=fixPageMargins(o||40),p=u.top;return i.forEach((function(i){i.items.forEach((function(i){var o=function getBottomPosition(i){var o=i.item.y||0,u=function getItemHeight(i){return E(i.item.getHeight)?i.item.getHeight():i.item._height?i.item._height:"vector"===i.type?i.item.y1>i.item.y2?i.item.y1:i.item.y2:0}(i);return o+u}(i);o>p&&(p=o)}))})),p+=u.bottom}(D,i.pageMargins);this.pdfKitDoc.options.size=[u.width,N]}if(function renderPages(i,o,u,p){u._pdfMakePages=i,u.addPage();var g=0;p&&i.forEach((function(i){g+=i.items.length}));var y=0;p=p||function(){};for(var w=0;w<i.length;w++){w>0&&(updatePageOrientationInOptions(i[w],u),u.addPage(u.options));for(var _=i[w],x=0,k=_.items.length;x<k;x++){var P=_.items[x];switch(P.type){case"vector":renderVector(P.item,u);break;case"line":renderLine(P.item,P.item.x,P.item.y,u);break;case"image":renderImage(P.item,P.item.x,P.item.y,u);break;case"svg":renderSVG(P.item,P.item.x,P.item.y,u,o);break;case"beginClip":beginClip(P.item,u);break;case"endClip":endClip(u)}y++,p(y/g)}_.watermark&&renderWatermark(_,u)}}(D,this.fontProvider,this.pdfKitDoc,o.progressCallback),o.autoPrint){var U=this.pdfKitDoc.ref({Type:"Action",S:"Named",N:"Print"});this.pdfKitDoc._root.data.OpenAction=U,U.end()}return this.pdfKitDoc},i.exports=PdfPrinter},function(i,o,u){"use strict";var p=function _interopDefault(i){return i&&"object"==typeof i&&"default"in i?i.default:i}(u(309));i.exports={getEngineInstance:function getEngineInstance(){return p},createPdfDocument:function createPdfDocument(i){return new p(i=i||{})}}},function(i,o,u){"use strict";(function(i,p){Object.defineProperty(o,"__esModule",{value:!0}),o.default=void 0,u(202),u(207),u(313),u(314),u(315),u(317),u(318),u(320),u(321),u(32),u(215),u(216),u(151),u(28),u(120),u(219),u(47),u(157),u(221),u(222),u(327),u(224),u(121),u(330),u(331),u(332),u(225),u(333),u(226),u(230),u(161),u(336),u(59),u(60),u(61),u(62),u(63),u(64),u(65),u(66),u(67),u(68),u(69),u(70),u(71),u(72),u(73),u(74),u(75),u(76),u(77),u(78),u(79),u(80),u(81),u(82),u(83),u(344),u(237),u(345),u(238),u(239),u(348);var g=_interopRequireDefault(u(123)),y=_interopRequireDefault(u(244)),w=_interopRequireDefault(u(370)),_=_interopRequireDefault(u(394)),x=u(124),k=_interopRequireDefault(u(290)),P=_interopRequireDefault(u(539));function _interopRequireDefault(i){return i&&i.__esModule?i:{default:i}}var E=u(54);function _classCallCheck(i,o){if(!(i instanceof o))throw new TypeError("Cannot call a class as a function")}function _defineProperties(i,o){for(var u=0;u<o.length;u++){var p=o[u];p.enumerable=p.enumerable||!1,p.configurable=!0,"value"in p&&(p.writable=!0),Object.defineProperty(i,p.key,p)}}function _createClass(i,o,u){return o&&_defineProperties(i.prototype,o),u&&_defineProperties(i,u),i}function _defineProperty(i,o,u){return o in i?Object.defineProperty(i,o,{value:u,enumerable:!0,configurable:!0,writable:!0}):i[o]=u,i}function ownKeys(i,o){var u=Object.keys(i);if(Object.getOwnPropertySymbols){var p=Object.getOwnPropertySymbols(i);o&&(p=p.filter((function(o){return Object.getOwnPropertyDescriptor(i,o).enumerable}))),u.push.apply(u,p)}return u}function _objectSpread2(i){for(var o=1;o<arguments.length;o++){var u=null!=arguments[o]?arguments[o]:{};o%2?ownKeys(Object(u),!0).forEach((function(o){_defineProperty(i,o,u[o])})):Object.getOwnPropertyDescriptors?Object.defineProperties(i,Object.getOwnPropertyDescriptors(u)):ownKeys(Object(u)).forEach((function(o){Object.defineProperty(i,o,Object.getOwnPropertyDescriptor(u,o))}))}return i}function _inherits(i,o){if("function"!=typeof o&&null!==o)throw new TypeError("Super expression must either be null or a function");i.prototype=Object.create(o&&o.prototype,{constructor:{value:i,writable:!0,configurable:!0}}),o&&_setPrototypeOf(i,o)}function _getPrototypeOf(i){return(_getPrototypeOf=Object.setPrototypeOf?Object.getPrototypeOf:function _getPrototypeOf(i){return i.__proto__||Object.getPrototypeOf(i)})(i)}function _setPrototypeOf(i,o){return(_setPrototypeOf=Object.setPrototypeOf||function _setPrototypeOf(i,o){return i.__proto__=o,i})(i,o)}function _assertThisInitialized(i){if(void 0===i)throw new ReferenceError("this hasn't been initialised - super() hasn't been called");return i}function _possibleConstructorReturn(i,o){return!o||"object"!=typeof o&&"function"!=typeof o?_assertThisInitialized(i):o}function _createSuper(i){var o=function _isNativeReflectConstruct(){if("undefined"==typeof Reflect||!Reflect.construct)return!1;if(Reflect.construct.sham)return!1;if("function"==typeof Proxy)return!0;try{return Date.prototype.toString.call(Reflect.construct(Date,[],(function(){}))),!0}catch(i){return!1}}();return function _createSuperInternal(){var u,p=_getPrototypeOf(i);if(o){var g=_getPrototypeOf(this).constructor;u=Reflect.construct(p,arguments,g)}else u=p.apply(this,arguments);return _possibleConstructorReturn(this,u)}}function _slicedToArray(i,o){return function _arrayWithHoles(i){if(Array.isArray(i))return i}(i)||function _iterableToArrayLimit(i,o){if("undefined"==typeof Symbol||!(Symbol.iterator in Object(i)))return;var u=[],p=!0,g=!1,y=void 0;try{for(var w,_=i[Symbol.iterator]();!(p=(w=_.next()).done)&&(u.push(w.value),!o||u.length!==o);p=!0);}catch(i){g=!0,y=i}finally{try{p||null==_.return||_.return()}finally{if(g)throw y}}return u}(i,o)||_unsupportedIterableToArray(i,o)||function _nonIterableRest(){throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.")}()}function _toConsumableArray(i){return function _arrayWithoutHoles(i){if(Array.isArray(i))return _arrayLikeToArray(i)}(i)||function _iterableToArray(i){if("undefined"!=typeof Symbol&&Symbol.iterator in Object(i))return Array.from(i)}(i)||_unsupportedIterableToArray(i)||function _nonIterableSpread(){throw new TypeError("Invalid attempt to spread non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.")}()}function _unsupportedIterableToArray(i,o){if(i){if("string"==typeof i)return _arrayLikeToArray(i,o);var u=Object.prototype.toString.call(i).slice(8,-1);return"Object"===u&&i.constructor&&(u=i.constructor.name),"Map"===u||"Set"===u?Array.from(i):"Arguments"===u||/^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(u)?_arrayLikeToArray(i,o):void 0}}function _arrayLikeToArray(i,o){(null==o||o>i.length)&&(o=i.length);for(var u=0,p=new Array(o);u<o;u++)p[u]=i[u];return p}function _createForOfIteratorHelper(i,o){var u;if("undefined"==typeof Symbol||null==i[Symbol.iterator]){if(Array.isArray(i)||(u=_unsupportedIterableToArray(i))||o&&i&&"number"==typeof i.length){u&&(i=u);var p=0,g=function F(){};return{s:g,n:function n(){return p>=i.length?{done:!0}:{done:!1,value:i[p++]}},e:function e(i){throw i},f:g}}throw new TypeError("Invalid attempt to iterate non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method.")}var y,w=!0,_=!1;return{s:function s(){u=i[Symbol.iterator]()},n:function n(){var i=u.next();return w=i.done,i},e:function e(i){_=!0,y=i},f:function f(){try{w||null==u.return||u.return()}finally{if(_)throw y}}}}var O=function(){function PDFAbstractReference(){_classCallCheck(this,PDFAbstractReference)}return _createClass(PDFAbstractReference,[{key:"toString",value:function toString(){throw new Error("Must be implemented by subclasses")}}]),PDFAbstractReference}(),I=function(){function PDFTree(){var i=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{};_classCallCheck(this,PDFTree),this._items={},this.limits="boolean"!=typeof i.limits||i.limits}return _createClass(PDFTree,[{key:"add",value:function add(i,o){return this._items[i]=o}},{key:"get",value:function get(i){return this._items[i]}},{key:"toString",value:function toString(){var i=this,o=Object.keys(this._items).sort((function(o,u){return i._compareKeys(o,u)})),u=["<<"];if(this.limits&&o.length>1){var p=o[0],g=o[o.length-1];u.push("  /Limits ".concat(U.convert([this._dataForKey(p),this._dataForKey(g)])))}u.push("  /".concat(this._keysName()," ["));var y,w=_createForOfIteratorHelper(o);try{for(w.s();!(y=w.n()).done;){var _=y.value;u.push("    ".concat(U.convert(this._dataForKey(_))," ").concat(U.convert(this._items[_])))}}catch(i){w.e(i)}finally{w.f()}return u.push("]"),u.push(">>"),u.join("\n")}},{key:"_compareKeys",value:function _compareKeys(){throw new Error("Must be implemented by subclasses")}},{key:"_keysName",value:function _keysName(){throw new Error("Must be implemented by subclasses")}},{key:"_dataForKey",value:function _dataForKey(){throw new Error("Must be implemented by subclasses")}}]),PDFTree}(),B=function pad(i,o){return(Array(o+1).join("0")+i).slice(-o)},D=/[\n\r\t\b\f()\\]/g,R={"\n":"\\n","\r":"\\r","\t":"\\t","\b":"\\b","\f":"\\f","\\":"\\\\","(":"\\(",")":"\\)"},N=function swapBytes(i){var o=i.length;if(1&o)throw new Error("Buffer length must be even");for(var u=0,p=o-1;u<p;u+=2){var g=i[u];i[u]=i[u+1],i[u+1]=g}return i},U=function(){function PDFObject(){_classCallCheck(this,PDFObject)}return _createClass(PDFObject,null,[{key:"convert",value:function convert(o){var u=arguments.length>1&&void 0!==arguments[1]?arguments[1]:null;if("string"==typeof o)return"/".concat(o);if(o instanceof String){for(var p,g=o,y=!1,w=0,_=g.length;w<_;w++)if(g.charCodeAt(w)>127){y=!0;break}return p=y?N(i.from("\ufeff".concat(g),"utf16le")):i.from(g.valueOf(),"ascii"),g=(g=u?u(p).toString("binary"):p.toString("binary")).replace(D,(function(i){return R[i]})),"(".concat(g,")")}if(i.isBuffer(o))return"<".concat(o.toString("hex"),">");if(o instanceof O||o instanceof I)return o.toString();if(o instanceof Date){var x="D:".concat(B(o.getUTCFullYear(),4))+B(o.getUTCMonth()+1,2)+B(o.getUTCDate(),2)+B(o.getUTCHours(),2)+B(o.getUTCMinutes(),2)+B(o.getUTCSeconds(),2)+"Z";return u&&(x=(x=u(i.from(x,"ascii")).toString("binary")).replace(D,(function(i){return R[i]}))),"(".concat(x,")")}if(Array.isArray(o)){var k=o.map((function(i){return PDFObject.convert(i,u)})).join(" ");return"[".concat(k,"]")}if("[object Object]"==={}.toString.call(o)){var P=["<<"];for(var E in o){var U=o[E];P.push("/".concat(E," ").concat(PDFObject.convert(U,u)))}return P.push(">>"),P.join("\n")}return"number"==typeof o?PDFObject.number(o):"".concat(o)}},{key:"number",value:function number(i){if(i>-1e21&&i<1e21)return Math.round(1e6*i)/1e6;throw new Error("unsupported number: ".concat(i))}}]),PDFObject}(),W=function(o){_inherits(PDFReference,o);var u=_createSuper(PDFReference);function PDFReference(i,o){var p,g=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{};return _classCallCheck(this,PDFReference),(p=u.call(this)).document=i,p.id=o,p.data=g,p.gen=0,p.compress=p.document.compress&&!p.data.Filter,p.uncompressedLength=0,p.buffer=[],p}return _createClass(PDFReference,[{key:"write",value:function write(o){if(i.isBuffer(o)||(o=i.from(o+"\n","binary")),this.uncompressedLength+=o.length,null==this.data.Length&&(this.data.Length=0),this.buffer.push(o),this.data.Length+=o.length,this.compress)return this.data.Filter="FlateDecode"}},{key:"end",value:function end(i){return i&&this.write(i),this.finalize()}},{key:"finalize",value:function finalize(){this.offset=this.document._offset;var o=this.document._security?this.document._security.getEncryptFn(this.id,this.gen):null;this.buffer.length&&(this.buffer=i.concat(this.buffer),this.compress&&(this.buffer=y.default.deflateSync(this.buffer)),o&&(this.buffer=o(this.buffer)),this.data.Length=this.buffer.length),this.document._write("".concat(this.id," ").concat(this.gen," obj")),this.document._write(U.convert(this.data,o)),this.buffer.length&&(this.document._write("stream"),this.document._write(this.buffer),this.buffer=[],this.document._write("\nendstream")),this.document._write("endobj"),this.document._refEnd(this)}},{key:"toString",value:function toString(){return"".concat(this.id," ").concat(this.gen," R")}}]),PDFReference}(O),G={top:72,left:72,bottom:72,right:72},j={"4A0":[4767.87,6740.79],"2A0":[3370.39,4767.87],A0:[2383.94,3370.39],A1:[1683.78,2383.94],A2:[1190.55,1683.78],A3:[841.89,1190.55],A4:[595.28,841.89],A5:[419.53,595.28],A6:[297.64,419.53],A7:[209.76,297.64],A8:[147.4,209.76],A9:[104.88,147.4],A10:[73.7,104.88],B0:[2834.65,4008.19],B1:[2004.09,2834.65],B2:[1417.32,2004.09],B3:[1000.63,1417.32],B4:[708.66,1000.63],B5:[498.9,708.66],B6:[354.33,498.9],B7:[249.45,354.33],B8:[175.75,249.45],B9:[124.72,175.75],B10:[87.87,124.72],C0:[2599.37,3676.54],C1:[1836.85,2599.37],C2:[1298.27,1836.85],C3:[918.43,1298.27],C4:[649.13,918.43],C5:[459.21,649.13],C6:[323.15,459.21],C7:[229.61,323.15],C8:[161.57,229.61],C9:[113.39,161.57],C10:[79.37,113.39],RA0:[2437.8,3458.27],RA1:[1729.13,2437.8],RA2:[1218.9,1729.13],RA3:[864.57,1218.9],RA4:[609.45,864.57],SRA0:[2551.18,3628.35],SRA1:[1814.17,2551.18],SRA2:[1275.59,1814.17],SRA3:[907.09,1275.59],SRA4:[637.8,907.09],EXECUTIVE:[521.86,756],FOLIO:[612,936],LEGAL:[612,1008],LETTER:[612,792],TABLOID:[792,1224]},X=function(){function PDFPage(i){var o=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};_classCallCheck(this,PDFPage),this.document=i,this.size=o.size||"letter",this.layout=o.layout||"portrait","number"==typeof o.margin?this.margins={top:o.margin,left:o.margin,bottom:o.margin,right:o.margin}:this.margins=o.margins||G;var u=Array.isArray(this.size)?this.size:j[this.size.toUpperCase()];this.width=u["portrait"===this.layout?0:1],this.height=u["portrait"===this.layout?1:0],this.content=this.document.ref(),this.resources=this.document.ref({ProcSet:["PDF","Text","ImageB","ImageC","ImageI"]}),this.dictionary=this.document.ref({Type:"Page",Parent:this.document._root.data.Pages,MediaBox:[0,0,this.width,this.height],Contents:this.content,Resources:this.resources}),this.markings=[]}return _createClass(PDFPage,[{key:"maxY",value:function maxY(){return this.height-this.margins.bottom}},{key:"write",value:function write(i){return this.content.write(i)}},{key:"end",value:function end(){return this.dictionary.end(),this.resources.end(),this.content.end()}},{key:"fonts",get:function get(){var i=this.resources.data;return null!=i.Font?i.Font:i.Font={}}},{key:"xobjects",get:function get(){var i=this.resources.data;return null!=i.XObject?i.XObject:i.XObject={}}},{key:"ext_gstates",get:function get(){var i=this.resources.data;return null!=i.ExtGState?i.ExtGState:i.ExtGState={}}},{key:"patterns",get:function get(){var i=this.resources.data;return null!=i.Pattern?i.Pattern:i.Pattern={}}},{key:"annotations",get:function get(){var i=this.dictionary.data;return null!=i.Annots?i.Annots:i.Annots=[]}},{key:"structParentTreeKey",get:function get(){var i=this.dictionary.data;return null!=i.StructParents?i.StructParents:i.StructParents=this.document.createStructParentTreeNextKey()}}]),PDFPage}(),K=function(i){_inherits(PDFNameTree,i);var o=_createSuper(PDFNameTree);function PDFNameTree(){return _classCallCheck(this,PDFNameTree),o.apply(this,arguments)}return _createClass(PDFNameTree,[{key:"_compareKeys",value:function _compareKeys(i,o){return i.localeCompare(o)}},{key:"_keysName",value:function _keysName(){return"Names"}},{key:"_dataForKey",value:function _dataForKey(i){return new String(i)}}]),PDFNameTree}(I);function inRange(i,o){if(i<o[0])return!1;for(var u=0,p=o.length/2;u<=p;){var g=Math.floor((u+p)/2),y=2*g;if(i>=o[y]&&i<=o[y+1])return!0;i>o[y+1]?u=g+1:p=g-1}return!1}var Y=[545,545,564,591,686,687,751,767,848,863,880,883,886,889,891,893,895,899,907,907,909,909,930,930,975,975,1015,1023,1159,1159,1231,1231,1270,1271,1274,1279,1296,1328,1367,1368,1376,1376,1416,1416,1419,1424,1442,1442,1466,1466,1477,1487,1515,1519,1525,1547,1549,1562,1564,1566,1568,1568,1595,1599,1622,1631,1774,1775,1791,1791,1806,1806,1837,1839,1867,1919,1970,2304,2308,2308,2362,2363,2382,2383,2389,2391,2417,2432,2436,2436,2445,2446,2449,2450,2473,2473,2481,2481,2483,2485,2490,2491,2493,2493,2501,2502,2505,2506,2510,2518,2520,2523,2526,2526,2532,2533,2555,2561,2563,2564,2571,2574,2577,2578,2601,2601,2609,2609,2612,2612,2615,2615,2618,2619,2621,2621,2627,2630,2633,2634,2638,2648,2653,2653,2655,2661,2677,2688,2692,2692,2700,2700,2702,2702,2706,2706,2729,2729,2737,2737,2740,2740,2746,2747,2758,2758,2762,2762,2766,2767,2769,2783,2785,2789,2800,2816,2820,2820,2829,2830,2833,2834,2857,2857,2865,2865,2868,2869,2874,2875,2884,2886,2889,2890,2894,2901,2904,2907,2910,2910,2914,2917,2929,2945,2948,2948,2955,2957,2961,2961,2966,2968,2971,2971,2973,2973,2976,2978,2981,2983,2987,2989,2998,2998,3002,3005,3011,3013,3017,3017,3022,3030,3032,3046,3059,3072,3076,3076,3085,3085,3089,3089,3113,3113,3124,3124,3130,3133,3141,3141,3145,3145,3150,3156,3159,3167,3170,3173,3184,3201,3204,3204,3213,3213,3217,3217,3241,3241,3252,3252,3258,3261,3269,3269,3273,3273,3278,3284,3287,3293,3295,3295,3298,3301,3312,3329,3332,3332,3341,3341,3345,3345,3369,3369,3386,3389,3396,3397,3401,3401,3406,3414,3416,3423,3426,3429,3440,3457,3460,3460,3479,3481,3506,3506,3516,3516,3518,3519,3527,3529,3531,3534,3541,3541,3543,3543,3552,3569,3573,3584,3643,3646,3676,3712,3715,3715,3717,3718,3721,3721,3723,3724,3726,3731,3736,3736,3744,3744,3748,3748,3750,3750,3752,3753,3756,3756,3770,3770,3774,3775,3781,3781,3783,3783,3790,3791,3802,3803,3806,3839,3912,3912,3947,3952,3980,3983,3992,3992,4029,4029,4045,4046,4048,4095,4130,4130,4136,4136,4139,4139,4147,4149,4154,4159,4186,4255,4294,4303,4345,4346,4348,4351,4442,4446,4515,4519,4602,4607,4615,4615,4679,4679,4681,4681,4686,4687,4695,4695,4697,4697,4702,4703,4743,4743,4745,4745,4750,4751,4783,4783,4785,4785,4790,4791,4799,4799,4801,4801,4806,4807,4815,4815,4823,4823,4847,4847,4879,4879,4881,4881,4886,4887,4895,4895,4935,4935,4955,4960,4989,5023,5109,5120,5751,5759,5789,5791,5873,5887,5901,5901,5909,5919,5943,5951,5972,5983,5997,5997,6001,6001,6004,6015,6109,6111,6122,6143,6159,6159,6170,6175,6264,6271,6314,7679,7836,7839,7930,7935,7958,7959,7966,7967,8006,8007,8014,8015,8024,8024,8026,8026,8028,8028,8030,8030,8062,8063,8117,8117,8133,8133,8148,8149,8156,8156,8176,8177,8181,8181,8191,8191,8275,8278,8280,8286,8292,8297,8306,8307,8335,8351,8370,8399,8427,8447,8507,8508,8524,8530,8580,8591,9167,9215,9255,9279,9291,9311,9471,9471,9748,9749,9752,9752,9854,9855,9866,9984,9989,9989,9994,9995,10024,10024,10060,10060,10062,10062,10067,10069,10071,10071,10079,10080,10133,10135,10160,10160,10175,10191,10220,10223,11008,11903,11930,11930,12020,12031,12246,12271,12284,12287,12352,12352,12439,12440,12544,12548,12589,12592,12687,12687,12728,12783,12829,12831,12868,12880,12924,12926,13004,13007,13055,13055,13175,13178,13278,13279,13311,13311,19894,19967,40870,40959,42125,42127,42183,44031,55204,55295,64046,64047,64107,64255,64263,64274,64280,64284,64311,64311,64317,64317,64319,64319,64322,64322,64325,64325,64434,64466,64832,64847,64912,64913,64968,64975,65021,65023,65040,65055,65060,65071,65095,65096,65107,65107,65127,65127,65132,65135,65141,65141,65277,65278,65280,65280,65471,65473,65480,65481,65488,65489,65496,65497,65501,65503,65511,65511,65519,65528,65536,66303,66335,66335,66340,66351,66379,66559,66598,66599,66638,118783,119030,119039,119079,119081,119262,119807,119893,119893,119965,119965,119968,119969,119971,119972,119975,119976,119981,119981,119994,119994,119996,119996,120001,120001,120004,120004,120070,120070,120075,120076,120085,120085,120093,120093,120122,120122,120127,120127,120133,120133,120135,120137,120145,120145,120484,120487,120778,120781,120832,131069,173783,194559,195102,196605,196608,262141,262144,327677,327680,393213,393216,458749,458752,524285,524288,589821,589824,655357,655360,720893,720896,786429,786432,851965,851968,917501,917504,917504,917506,917535,917632,983037],J=function isUnassignedCodePoint(i){return inRange(i,Y)},$=[173,173,847,847,6150,6150,6155,6155,6156,6156,6157,6157,8203,8203,8204,8204,8205,8205,8288,8288,65024,65024,65025,65025,65026,65026,65027,65027,65028,65028,65029,65029,65030,65030,65031,65031,65032,65032,65033,65033,65034,65034,65035,65035,65036,65036,65037,65037,65038,65038,65039,65039,65279,65279],tt=[160,160,5760,5760,8192,8192,8193,8193,8194,8194,8195,8195,8196,8196,8197,8197,8198,8198,8199,8199,8200,8200,8201,8201,8202,8202,8203,8203,8239,8239,8287,8287,12288,12288],et=[128,159,1757,1757,1807,1807,6158,6158,8204,8204,8205,8205,8232,8232,8233,8233,8288,8288,8289,8289,8290,8290,8291,8291,8298,8303,65279,65279,65529,65532,119155,119162],rt=[64976,65007,65534,65535,131070,131071,196606,196607,262142,262143,327678,327679,393214,393215,458750,458751,524286,524287,589822,589823,655358,655359,720894,720895,786430,786431,851966,851967,917502,917503,983038,983039,1114110,1114111],nt=[0,31,127,127,832,832,833,833,8206,8206,8207,8207,8234,8234,8235,8235,8236,8236,8237,8237,8238,8238,8298,8298,8299,8299,8300,8300,8301,8301,8302,8302,8303,8303,12272,12283,55296,57343,57344,63743,65529,65529,65530,65530,65531,65531,65532,65532,65533,65533,917505,917505,917536,917631,983040,1048573,1048576,1114109],it=function isProhibitedCharacter(i){return inRange(i,tt)||inRange(i,nt)||inRange(i,et)||inRange(i,rt)},ot=[1470,1470,1472,1472,1475,1475,1488,1514,1520,1524,1563,1563,1567,1567,1569,1594,1600,1610,1645,1647,1649,1749,1757,1757,1765,1766,1786,1790,1792,1805,1808,1808,1810,1836,1920,1957,1969,1969,8207,8207,64285,64285,64287,64296,64298,64310,64312,64316,64318,64318,64320,64321,64323,64324,64326,64433,64467,64829,64848,64911,64914,64967,65008,65020,65136,65140,65142,65276],at=function isBidirectionalRAL(i){return inRange(i,ot)},st=[65,90,97,122,170,170,181,181,186,186,192,214,216,246,248,544,546,563,592,685,688,696,699,705,720,721,736,740,750,750,890,890,902,902,904,906,908,908,910,929,931,974,976,1013,1024,1154,1162,1230,1232,1269,1272,1273,1280,1295,1329,1366,1369,1375,1377,1415,1417,1417,2307,2307,2309,2361,2365,2368,2377,2380,2384,2384,2392,2401,2404,2416,2434,2435,2437,2444,2447,2448,2451,2472,2474,2480,2482,2482,2486,2489,2494,2496,2503,2504,2507,2508,2519,2519,2524,2525,2527,2529,2534,2545,2548,2554,2565,2570,2575,2576,2579,2600,2602,2608,2610,2611,2613,2614,2616,2617,2622,2624,2649,2652,2654,2654,2662,2671,2674,2676,2691,2691,2693,2699,2701,2701,2703,2705,2707,2728,2730,2736,2738,2739,2741,2745,2749,2752,2761,2761,2763,2764,2768,2768,2784,2784,2790,2799,2818,2819,2821,2828,2831,2832,2835,2856,2858,2864,2866,2867,2870,2873,2877,2878,2880,2880,2887,2888,2891,2892,2903,2903,2908,2909,2911,2913,2918,2928,2947,2947,2949,2954,2958,2960,2962,2965,2969,2970,2972,2972,2974,2975,2979,2980,2984,2986,2990,2997,2999,3001,3006,3007,3009,3010,3014,3016,3018,3020,3031,3031,3047,3058,3073,3075,3077,3084,3086,3088,3090,3112,3114,3123,3125,3129,3137,3140,3168,3169,3174,3183,3202,3203,3205,3212,3214,3216,3218,3240,3242,3251,3253,3257,3262,3262,3264,3268,3271,3272,3274,3275,3285,3286,3294,3294,3296,3297,3302,3311,3330,3331,3333,3340,3342,3344,3346,3368,3370,3385,3390,3392,3398,3400,3402,3404,3415,3415,3424,3425,3430,3439,3458,3459,3461,3478,3482,3505,3507,3515,3517,3517,3520,3526,3535,3537,3544,3551,3570,3572,3585,3632,3634,3635,3648,3654,3663,3675,3713,3714,3716,3716,3719,3720,3722,3722,3725,3725,3732,3735,3737,3743,3745,3747,3749,3749,3751,3751,3754,3755,3757,3760,3762,3763,3773,3773,3776,3780,3782,3782,3792,3801,3804,3805,3840,3863,3866,3892,3894,3894,3896,3896,3902,3911,3913,3946,3967,3967,3973,3973,3976,3979,4030,4037,4039,4044,4047,4047,4096,4129,4131,4135,4137,4138,4140,4140,4145,4145,4152,4152,4160,4183,4256,4293,4304,4344,4347,4347,4352,4441,4447,4514,4520,4601,4608,4614,4616,4678,4680,4680,4682,4685,4688,4694,4696,4696,4698,4701,4704,4742,4744,4744,4746,4749,4752,4782,4784,4784,4786,4789,4792,4798,4800,4800,4802,4805,4808,4814,4816,4822,4824,4846,4848,4878,4880,4880,4882,4885,4888,4894,4896,4934,4936,4954,4961,4988,5024,5108,5121,5750,5761,5786,5792,5872,5888,5900,5902,5905,5920,5937,5941,5942,5952,5969,5984,5996,5998,6e3,6016,6070,6078,6085,6087,6088,6100,6106,6108,6108,6112,6121,6160,6169,6176,6263,6272,6312,7680,7835,7840,7929,7936,7957,7960,7965,7968,8005,8008,8013,8016,8023,8025,8025,8027,8027,8029,8029,8031,8061,8064,8116,8118,8124,8126,8126,8130,8132,8134,8140,8144,8147,8150,8155,8160,8172,8178,8180,8182,8188,8206,8206,8305,8305,8319,8319,8450,8450,8455,8455,8458,8467,8469,8469,8473,8477,8484,8484,8486,8486,8488,8488,8490,8493,8495,8497,8499,8505,8509,8511,8517,8521,8544,8579,9014,9082,9109,9109,9372,9449,12293,12295,12321,12329,12337,12341,12344,12348,12353,12438,12445,12447,12449,12538,12540,12543,12549,12588,12593,12686,12688,12727,12784,12828,12832,12867,12896,12923,12927,12976,12992,13003,13008,13054,13056,13174,13179,13277,13280,13310,13312,19893,19968,40869,40960,42124,44032,55203,55296,64045,64048,64106,64256,64262,64275,64279,65313,65338,65345,65370,65382,65470,65474,65479,65482,65487,65490,65495,65498,65500,66304,66334,66336,66339,66352,66378,66560,66597,66600,66637,118784,119029,119040,119078,119082,119142,119146,119154,119171,119172,119180,119209,119214,119261,119808,119892,119894,119964,119966,119967,119970,119970,119973,119974,119977,119980,119982,119993,119995,119995,119997,12e4,120002,120003,120005,120069,120071,120074,120077,120084,120086,120092,120094,120121,120123,120126,120128,120132,120134,120134,120138,120144,120146,120483,120488,120777,131072,173782,194560,195101,983040,1048573,1048576,1114109],lt=function isBidirectionalL(i){return inRange(i,st)},ct=function isNonASCIISpaceCharacter(i){return inRange(i,tt)},ut=function isCommonlyMappedToNothing(i){return inRange(i,$)},ft=function getCodePoint(i){return i.codePointAt(0)},ht=function first(i){return i[0]},dt=function last(i){return i[i.length-1]};function toCodePoints(i){for(var o=[],u=i.length,p=0;p<u;p+=1){var g=i.charCodeAt(p);if(g>=55296&&g<=56319&&u>p+1){var y=i.charCodeAt(p+1);if(y>=56320&&y<=57343){o.push(1024*(g-55296)+y-56320+65536),p+=1;continue}}o.push(g)}return o}function saslprep(i){var o=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};if("string"!=typeof i)throw new TypeError("Expected string.");if(0===i.length)return"";var u=toCodePoints(i).map((function(i){return ct(i)?32:i})).filter((function(i){return!ut(i)})),p=String.fromCodePoint.apply(null,u).normalize("NFKC"),g=toCodePoints(p),y=g.some(it);if(y)throw new Error("Prohibited character, see https://tools.ietf.org/html/rfc4013#section-2.3");if(!0!==o.allowUnassigned){var w=g.some(J);if(w)throw new Error("Unassigned code point, see https://tools.ietf.org/html/rfc4013#section-2.5")}var _=g.some(at),x=g.some(lt);if(_&&x)throw new Error("String must not contain RandALCat and LCat at the same time, see https://tools.ietf.org/html/rfc3454#section-6");var k=at(ft(ht(p))),P=at(ft(dt(p)));if(_&&(!k||!P))throw new Error("Bidirectional RandALCat character must be the first and the last character of the string, see https://tools.ietf.org/html/rfc3454#section-6");return p}var pt=function(){function PDFSecurity(i){var o=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};if(_classCallCheck(this,PDFSecurity),!o.ownerPassword&&!o.userPassword)throw new Error("None of owner password and user password is defined.");this.document=i,this._setupEncryption(o)}return _createClass(PDFSecurity,null,[{key:"generateFileID",value:function generateFileID(){var i=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{},o="".concat(i.CreationDate.getTime(),"\n");for(var u in i)i.hasOwnProperty(u)&&(o+="".concat(u,": ").concat(i[u].valueOf(),"\n"));return wordArrayToBuffer(w.default.MD5(o))}},{key:"generateRandomWordArray",value:function generateRandomWordArray(i){return w.default.lib.WordArray.random(i)}},{key:"create",value:function create(i){var o=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};return o.ownerPassword||o.userPassword?new PDFSecurity(i,o):null}}]),_createClass(PDFSecurity,[{key:"_setupEncryption",value:function _setupEncryption(i){switch(i.pdfVersion){case"1.4":case"1.5":this.version=2;break;case"1.6":case"1.7":this.version=4;break;case"1.7ext3":this.version=5;break;default:this.version=1}var o={Filter:"Standard"};switch(this.version){case 1:case 2:case 4:this._setupEncryptionV1V2V4(this.version,o,i);break;case 5:this._setupEncryptionV5(o,i)}this.dictionary=this.document.ref(o)}},{key:"_setupEncryptionV1V2V4",value:function _setupEncryptionV1V2V4(i,o,u){var p,g;switch(i){case 1:p=2,this.keyBits=40,g=function getPermissionsR2(){var i=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{},o=-64;i.printing&&(o|=4);i.modifying&&(o|=8);i.copying&&(o|=16);i.annotating&&(o|=32);return o}(u.permissions);break;case 2:p=3,this.keyBits=128,g=getPermissionsR3(u.permissions);break;case 4:p=4,this.keyBits=128,g=getPermissionsR3(u.permissions)}var y,_=processPasswordR2R3R4(u.userPassword),x=u.ownerPassword?processPasswordR2R3R4(u.ownerPassword):_,k=function getOwnerPasswordR2R3R4(i,o,u,p){for(var g=p,y=i>=3?51:1,_=0;_<y;_++)g=w.default.MD5(g);var x=g.clone();x.sigBytes=o/8;var k=u;y=i>=3?20:1;for(var P=0;P<y;P++){for(var E=Math.ceil(x.sigBytes/4),O=0;O<E;O++)x.words[O]=g.words[O]^(P|P<<8|P<<16|P<<24);k=w.default.RC4.encrypt(k,x).ciphertext}return k}(p,this.keyBits,_,x);this.encryptionKey=function getEncryptionKeyR2R3R4(i,o,u,p,g,y){for(var _=p.clone().concat(g).concat(w.default.lib.WordArray.create([lsbFirstWord(y)],4)).concat(w.default.lib.WordArray.create(u)),x=i>=3?51:1,k=0;k<x;k++)(_=w.default.MD5(_)).sigBytes=o/8;return _}(p,this.keyBits,this.document._id,_,k,g),y=2===p?function getUserPasswordR2(i){return w.default.RC4.encrypt(processPasswordR2R3R4(),i).ciphertext}(this.encryptionKey):function getUserPasswordR3R4(i,o){for(var u=o.clone(),p=w.default.MD5(processPasswordR2R3R4().concat(w.default.lib.WordArray.create(i))),g=0;g<20;g++){for(var y=Math.ceil(u.sigBytes/4),_=0;_<y;_++)u.words[_]=o.words[_]^(g|g<<8|g<<16|g<<24);p=w.default.RC4.encrypt(p,u).ciphertext}return p.concat(w.default.lib.WordArray.create(null,16))}(this.document._id,this.encryptionKey),o.V=i,i>=2&&(o.Length=this.keyBits),4===i&&(o.CF={StdCF:{AuthEvent:"DocOpen",CFM:"AESV2",Length:this.keyBits/8}},o.StmF="StdCF",o.StrF="StdCF"),o.R=p,o.O=wordArrayToBuffer(k),o.U=wordArrayToBuffer(y),o.P=g}},{key:"_setupEncryptionV5",value:function _setupEncryptionV5(i,o){this.keyBits=256;var u=getPermissionsR3(o.permissions),p=processPasswordR5(o.userPassword),g=o.ownerPassword?processPasswordR5(o.ownerPassword):p;this.encryptionKey=function getEncryptionKeyR5(i){return i(32)}(PDFSecurity.generateRandomWordArray);var y=function getUserPasswordR5(i,o){var u=o(8),p=o(8);return w.default.SHA256(i.clone().concat(u)).concat(u).concat(p)}(p,PDFSecurity.generateRandomWordArray),_=function getUserEncryptionKeyR5(i,o,u){var p=w.default.SHA256(i.clone().concat(o)),g={mode:w.default.mode.CBC,padding:w.default.pad.NoPadding,iv:w.default.lib.WordArray.create(null,16)};return w.default.AES.encrypt(u,p,g).ciphertext}(p,w.default.lib.WordArray.create(y.words.slice(10,12),8),this.encryptionKey),x=function getOwnerPasswordR5(i,o,u){var p=u(8),g=u(8);return w.default.SHA256(i.clone().concat(p).concat(o)).concat(p).concat(g)}(g,y,PDFSecurity.generateRandomWordArray),k=function getOwnerEncryptionKeyR5(i,o,u,p){var g=w.default.SHA256(i.clone().concat(o).concat(u)),y={mode:w.default.mode.CBC,padding:w.default.pad.NoPadding,iv:w.default.lib.WordArray.create(null,16)};return w.default.AES.encrypt(p,g,y).ciphertext}(g,w.default.lib.WordArray.create(x.words.slice(10,12),8),y,this.encryptionKey),P=function getEncryptedPermissionsR5(i,o,u){var p=w.default.lib.WordArray.create([lsbFirstWord(i),4294967295,1415668834],12).concat(u(4)),g={mode:w.default.mode.ECB,padding:w.default.pad.NoPadding};return w.default.AES.encrypt(p,o,g).ciphertext}(u,this.encryptionKey,PDFSecurity.generateRandomWordArray);i.V=5,i.Length=this.keyBits,i.CF={StdCF:{AuthEvent:"DocOpen",CFM:"AESV3",Length:this.keyBits/8}},i.StmF="StdCF",i.StrF="StdCF",i.R=5,i.O=wordArrayToBuffer(x),i.OE=wordArrayToBuffer(k),i.U=wordArrayToBuffer(y),i.UE=wordArrayToBuffer(_),i.P=u,i.Perms=wordArrayToBuffer(P)}},{key:"getEncryptFn",value:function getEncryptFn(i,o){var u,p;if(this.version<5&&(u=this.encryptionKey.clone().concat(w.default.lib.WordArray.create([(255&i)<<24|(65280&i)<<8|i>>8&65280|255&o,(65280&o)<<16],5))),1===this.version||2===this.version){var g=w.default.MD5(u);return g.sigBytes=Math.min(16,this.keyBits/8+5),function(i){return wordArrayToBuffer(w.default.RC4.encrypt(w.default.lib.WordArray.create(i),g).ciphertext)}}p=4===this.version?w.default.MD5(u.concat(w.default.lib.WordArray.create([1933667412],4))):this.encryptionKey;var y=PDFSecurity.generateRandomWordArray(16),_={mode:w.default.mode.CBC,padding:w.default.pad.Pkcs7,iv:y};return function(i){return wordArrayToBuffer(y.clone().concat(w.default.AES.encrypt(w.default.lib.WordArray.create(i),p,_).ciphertext))}}},{key:"end",value:function end(){this.dictionary.end()}}]),PDFSecurity}();function getPermissionsR3(){var i=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{},o=-3904;return"lowResolution"===i.printing&&(o|=4),"highResolution"===i.printing&&(o|=2052),i.modifying&&(o|=8),i.copying&&(o|=16),i.annotating&&(o|=32),i.fillingForms&&(o|=256),i.contentAccessibility&&(o|=512),i.documentAssembly&&(o|=1024),o}function processPasswordR2R3R4(){for(var o=arguments.length>0&&void 0!==arguments[0]?arguments[0]:"",u=i.alloc(32),p=o.length,g=0;g<p&&g<32;){var y=o.charCodeAt(g);if(y>255)throw new Error("Password contains one or more invalid characters.");u[g]=y,g++}for(;g<32;)u[g]=_t[g-p],g++;return w.default.lib.WordArray.create(u)}function processPasswordR5(){var o=arguments.length>0&&void 0!==arguments[0]?arguments[0]:"";o=unescape(encodeURIComponent(saslprep(o)));for(var u=Math.min(127,o.length),p=i.alloc(u),g=0;g<u;g++)p[g]=o.charCodeAt(g);return w.default.lib.WordArray.create(p)}function lsbFirstWord(i){return(255&i)<<24|(65280&i)<<8|i>>8&65280|i>>24&255}function wordArrayToBuffer(o){for(var u=[],p=0;p<o.sigBytes;p++)u.push(o.words[Math.floor(p/4)]>>8*(3-p%4)&255);return i.from(u)}var gt,yt,vt,mt,bt,wt,_t=[40,191,78,94,78,117,138,65,100,0,78,86,255,250,1,8,46,46,0,182,208,104,62,128,47,12,169,254,100,83,105,122],xt=U.number,St=function(){function PDFGradient(i){_classCallCheck(this,PDFGradient),this.doc=i,this.stops=[],this.embedded=!1,this.transform=[1,0,0,1,0,0]}return _createClass(PDFGradient,[{key:"stop",value:function stop(i,o,u){if(null==u&&(u=1),o=this.doc._normalizeColor(o),0===this.stops.length)if(3===o.length)this._colorSpace="DeviceRGB";else if(4===o.length)this._colorSpace="DeviceCMYK";else{if(1!==o.length)throw new Error("Unknown color space");this._colorSpace="DeviceGray"}else if("DeviceRGB"===this._colorSpace&&3!==o.length||"DeviceCMYK"===this._colorSpace&&4!==o.length||"DeviceGray"===this._colorSpace&&1!==o.length)throw new Error("All gradient stops must use the same color space");return u=Math.max(0,Math.min(1,u)),this.stops.push([i,o,u]),this}},{key:"setTransform",value:function setTransform(i,o,u,p,g,y){return this.transform=[i,o,u,p,g,y],this}},{key:"embed",value:function embed(i){var o,u=this.stops.length;if(0!==u){this.embedded=!0,this.matrix=i;var p=this.stops[u-1];p[0]<1&&this.stops.push([1,p[1],p[2]]);for(var g=[],y=[],w=[],_=0;_<u-1;_++)y.push(0,1),_+2!==u&&g.push(this.stops[_+1][0]),o=this.doc.ref({FunctionType:2,Domain:[0,1],C0:this.stops[_+0][1],C1:this.stops[_+1][1],N:1}),w.push(o),o.end();1===u?o=w[0]:(o=this.doc.ref({FunctionType:3,Domain:[0,1],Functions:w,Bounds:g,Encode:y})).end(),this.id="Sh".concat(++this.doc._gradCount);var x=this.shader(o);x.end();var k=this.doc.ref({Type:"Pattern",PatternType:2,Shading:x,Matrix:this.matrix.map(xt)});if(k.end(),this.stops.some((function(i){return i[2]<1}))){var P=this.opacityGradient();P._colorSpace="DeviceGray";var E,O=_createForOfIteratorHelper(this.stops);try{for(O.s();!(E=O.n()).done;){var I=E.value;P.stop(I[0],[I[2]])}}catch(i){O.e(i)}finally{O.f()}P=P.embed(this.matrix);var B=[0,0,this.doc.page.width,this.doc.page.height],D=this.doc.ref({Type:"XObject",Subtype:"Form",FormType:1,BBox:B,Group:{Type:"Group",S:"Transparency",CS:"DeviceGray"},Resources:{ProcSet:["PDF","Text","ImageB","ImageC","ImageI"],Pattern:{Sh1:P}}});D.write("/Pattern cs /Sh1 scn"),D.end("".concat(B.join(" ")," re f"));var R=this.doc.ref({Type:"ExtGState",SMask:{Type:"Mask",S:"Luminosity",G:D}});R.end();var N=this.doc.ref({Type:"Pattern",PatternType:1,PaintType:1,TilingType:2,BBox:B,XStep:B[2],YStep:B[3],Resources:{ProcSet:["PDF","Text","ImageB","ImageC","ImageI"],Pattern:{Sh1:k},ExtGState:{Gs1:R}}});N.write("/Gs1 gs /Pattern cs /Sh1 scn"),N.end("".concat(B.join(" ")," re f")),this.doc.page.patterns[this.id]=N}else this.doc.page.patterns[this.id]=k;return k}}},{key:"apply",value:function apply(i){var o=_slicedToArray(this.doc._ctm,6),u=o[0],p=o[1],g=o[2],y=o[3],w=o[4],_=o[5],x=_slicedToArray(this.transform,6),k=x[0],P=x[1],E=x[2],O=x[3],I=x[4],B=x[5],D=[u*k+g*P,p*k+y*P,u*E+g*O,p*E+y*O,u*I+g*B+w,p*I+y*B+_];return this.embedded&&D.join(" ")===this.matrix.join(" ")||this.embed(D),this.doc.addContent("/".concat(this.id," ").concat(i))}}]),PDFGradient}(),Ct={PDFGradient:St,PDFLinearGradient:function(i){_inherits(PDFLinearGradient,i);var o=_createSuper(PDFLinearGradient);function PDFLinearGradient(i,u,p,g,y){var w;return _classCallCheck(this,PDFLinearGradient),(w=o.call(this,i)).x1=u,w.y1=p,w.x2=g,w.y2=y,w}return _createClass(PDFLinearGradient,[{key:"shader",value:function shader(i){return this.doc.ref({ShadingType:2,ColorSpace:this._colorSpace,Coords:[this.x1,this.y1,this.x2,this.y2],Function:i,Extend:[!0,!0]})}},{key:"opacityGradient",value:function opacityGradient(){return new PDFLinearGradient(this.doc,this.x1,this.y1,this.x2,this.y2)}}]),PDFLinearGradient}(St),PDFRadialGradient:function(i){_inherits(PDFRadialGradient,i);var o=_createSuper(PDFRadialGradient);function PDFRadialGradient(i,u,p,g,y,w,_){var x;return _classCallCheck(this,PDFRadialGradient),(x=o.call(this,i)).doc=i,x.x1=u,x.y1=p,x.r1=g,x.x2=y,x.y2=w,x.r2=_,x}return _createClass(PDFRadialGradient,[{key:"shader",value:function shader(i){return this.doc.ref({ShadingType:3,ColorSpace:this._colorSpace,Coords:[this.x1,this.y1,this.r1,this.x2,this.y2,this.r2],Function:i,Extend:[!0,!0]})}},{key:"opacityGradient",value:function opacityGradient(){return new PDFRadialGradient(this.doc,this.x1,this.y1,this.r1,this.x2,this.y2,this.r2)}}]),PDFRadialGradient}(St)},kt=Ct.PDFGradient,At=Ct.PDFLinearGradient,Pt=Ct.PDFRadialGradient,Tt={initColor:function initColor(){return this._opacityRegistry={},this._opacityCount=0,this._gradCount=0},_normalizeColor:function _normalizeColor(i){if(i instanceof kt)return i;if("string"==typeof i)if("#"===i.charAt(0)){4===i.length&&(i=i.replace(/#([0-9A-F])([0-9A-F])([0-9A-F])/i,"#$1$1$2$2$3$3"));var o=parseInt(i.slice(1),16);i=[o>>16,o>>8&255,255&o]}else Et[i]&&(i=Et[i]);return Array.isArray(i)?(3===i.length?i=i.map((function(i){return i/255})):4===i.length&&(i=i.map((function(i){return i/100}))),i):null},_setColor:function _setColor(i,o){if(!(i=this._normalizeColor(i)))return!1;var u=o?"SCN":"scn";if(i instanceof kt)this._setColorSpace("Pattern",o),i.apply(u);else{var p=4===i.length?"DeviceCMYK":"DeviceRGB";this._setColorSpace(p,o),i=i.join(" "),this.addContent("".concat(i," ").concat(u))}return!0},_setColorSpace:function _setColorSpace(i,o){var u=o?"CS":"cs";return this.addContent("/".concat(i," ").concat(u))},fillColor:function fillColor(i,o){return this._setColor(i,!1)&&this.fillOpacity(o),this._fillColor=[i,o],this},strokeColor:function strokeColor(i,o){return this._setColor(i,!0)&&this.strokeOpacity(o),this},opacity:function opacity(i){return this._doOpacity(i,i),this},fillOpacity:function fillOpacity(i){return this._doOpacity(i,null),this},strokeOpacity:function strokeOpacity(i){return this._doOpacity(null,i),this},_doOpacity:function _doOpacity(i,o){var u,p;if(null!=i||null!=o){null!=i&&(i=Math.max(0,Math.min(1,i))),null!=o&&(o=Math.max(0,Math.min(1,o)));var g="".concat(i,"_").concat(o);if(this._opacityRegistry[g]){var y=_slicedToArray(this._opacityRegistry[g],2);u=y[0],p=y[1]}else{u={Type:"ExtGState"},null!=i&&(u.ca=i),null!=o&&(u.CA=o),(u=this.ref(u)).end();var w=++this._opacityCount;p="Gs".concat(w),this._opacityRegistry[g]=[u,p]}return this.page.ext_gstates[p]=u,this.addContent("/".concat(p," gs"))}},linearGradient:function linearGradient(i,o,u,p){return new At(this,i,o,u,p)},radialGradient:function radialGradient(i,o,u,p,g,y){return new Pt(this,i,o,u,p,g,y)}},Et={aliceblue:[240,248,255],antiquewhite:[250,235,215],aqua:[0,255,255],aquamarine:[127,255,212],azure:[240,255,255],beige:[245,245,220],bisque:[255,228,196],black:[0,0,0],blanchedalmond:[255,235,205],blue:[0,0,255],blueviolet:[138,43,226],brown:[165,42,42],burlywood:[222,184,135],cadetblue:[95,158,160],chartreuse:[127,255,0],chocolate:[210,105,30],coral:[255,127,80],cornflowerblue:[100,149,237],cornsilk:[255,248,220],crimson:[220,20,60],cyan:[0,255,255],darkblue:[0,0,139],darkcyan:[0,139,139],darkgoldenrod:[184,134,11],darkgray:[169,169,169],darkgreen:[0,100,0],darkgrey:[169,169,169],darkkhaki:[189,183,107],darkmagenta:[139,0,139],darkolivegreen:[85,107,47],darkorange:[255,140,0],darkorchid:[153,50,204],darkred:[139,0,0],darksalmon:[233,150,122],darkseagreen:[143,188,143],darkslateblue:[72,61,139],darkslategray:[47,79,79],darkslategrey:[47,79,79],darkturquoise:[0,206,209],darkviolet:[148,0,211],deeppink:[255,20,147],deepskyblue:[0,191,255],dimgray:[105,105,105],dimgrey:[105,105,105],dodgerblue:[30,144,255],firebrick:[178,34,34],floralwhite:[255,250,240],forestgreen:[34,139,34],fuchsia:[255,0,255],gainsboro:[220,220,220],ghostwhite:[248,248,255],gold:[255,215,0],goldenrod:[218,165,32],gray:[128,128,128],grey:[128,128,128],green:[0,128,0],greenyellow:[173,255,47],honeydew:[240,255,240],hotpink:[255,105,180],indianred:[205,92,92],indigo:[75,0,130],ivory:[255,255,240],khaki:[240,230,140],lavender:[230,230,250],lavenderblush:[255,240,245],lawngreen:[124,252,0],lemonchiffon:[255,250,205],lightblue:[173,216,230],lightcoral:[240,128,128],lightcyan:[224,255,255],lightgoldenrodyellow:[250,250,210],lightgray:[211,211,211],lightgreen:[144,238,144],lightgrey:[211,211,211],lightpink:[255,182,193],lightsalmon:[255,160,122],lightseagreen:[32,178,170],lightskyblue:[135,206,250],lightslategray:[119,136,153],lightslategrey:[119,136,153],lightsteelblue:[176,196,222],lightyellow:[255,255,224],lime:[0,255,0],limegreen:[50,205,50],linen:[250,240,230],magenta:[255,0,255],maroon:[128,0,0],mediumaquamarine:[102,205,170],mediumblue:[0,0,205],mediumorchid:[186,85,211],mediumpurple:[147,112,219],mediumseagreen:[60,179,113],mediumslateblue:[123,104,238],mediumspringgreen:[0,250,154],mediumturquoise:[72,209,204],mediumvioletred:[199,21,133],midnightblue:[25,25,112],mintcream:[245,255,250],mistyrose:[255,228,225],moccasin:[255,228,181],navajowhite:[255,222,173],navy:[0,0,128],oldlace:[253,245,230],olive:[128,128,0],olivedrab:[107,142,35],orange:[255,165,0],orangered:[255,69,0],orchid:[218,112,214],palegoldenrod:[238,232,170],palegreen:[152,251,152],paleturquoise:[175,238,238],palevioletred:[219,112,147],papayawhip:[255,239,213],peachpuff:[255,218,185],peru:[205,133,63],pink:[255,192,203],plum:[221,160,221],powderblue:[176,224,230],purple:[128,0,128],red:[255,0,0],rosybrown:[188,143,143],royalblue:[65,105,225],saddlebrown:[139,69,19],salmon:[250,128,114],sandybrown:[244,164,96],seagreen:[46,139,87],seashell:[255,245,238],sienna:[160,82,45],silver:[192,192,192],skyblue:[135,206,235],slateblue:[106,90,205],slategray:[112,128,144],slategrey:[112,128,144],snow:[255,250,250],springgreen:[0,255,127],steelblue:[70,130,180],tan:[210,180,140],teal:[0,128,128],thistle:[216,191,216],tomato:[255,99,71],turquoise:[64,224,208],violet:[238,130,238],wheat:[245,222,179],white:[255,255,255],whitesmoke:[245,245,245],yellow:[255,255,0],yellowgreen:[154,205,50]};gt=yt=vt=mt=bt=wt=0;var Ot={A:7,a:7,C:6,c:6,H:1,h:1,L:2,l:2,M:2,m:2,Q:4,q:4,S:4,s:4,T:2,t:2,V:1,v:1,Z:0,z:0},It={M:function M(i,o){return gt=o[0],yt=o[1],vt=mt=null,bt=gt,wt=yt,i.moveTo(gt,yt)},m:function m(i,o){return gt+=o[0],yt+=o[1],vt=mt=null,bt=gt,wt=yt,i.moveTo(gt,yt)},C:function C(i,o){return gt=o[4],yt=o[5],vt=o[2],mt=o[3],i.bezierCurveTo.apply(i,_toConsumableArray(o))},c:function c(i,o){return i.bezierCurveTo(o[0]+gt,o[1]+yt,o[2]+gt,o[3]+yt,o[4]+gt,o[5]+yt),vt=gt+o[2],mt=yt+o[3],gt+=o[4],yt+=o[5]},S:function S(i,o){return null===vt&&(vt=gt,mt=yt),i.bezierCurveTo(gt-(vt-gt),yt-(mt-yt),o[0],o[1],o[2],o[3]),vt=o[0],mt=o[1],gt=o[2],yt=o[3]},s:function s(i,o){return null===vt&&(vt=gt,mt=yt),i.bezierCurveTo(gt-(vt-gt),yt-(mt-yt),gt+o[0],yt+o[1],gt+o[2],yt+o[3]),vt=gt+o[0],mt=yt+o[1],gt+=o[2],yt+=o[3]},Q:function Q(i,o){return vt=o[0],mt=o[1],gt=o[2],yt=o[3],i.quadraticCurveTo(o[0],o[1],gt,yt)},q:function q(i,o){return i.quadraticCurveTo(o[0]+gt,o[1]+yt,o[2]+gt,o[3]+yt),vt=gt+o[0],mt=yt+o[1],gt+=o[2],yt+=o[3]},T:function T(i,o){return null===vt?(vt=gt,mt=yt):(vt=gt-(vt-gt),mt=yt-(mt-yt)),i.quadraticCurveTo(vt,mt,o[0],o[1]),vt=gt-(vt-gt),mt=yt-(mt-yt),gt=o[0],yt=o[1]},t:function t(i,o){return null===vt?(vt=gt,mt=yt):(vt=gt-(vt-gt),mt=yt-(mt-yt)),i.quadraticCurveTo(vt,mt,gt+o[0],yt+o[1]),gt+=o[0],yt+=o[1]},A:function A(i,o){return Bt(i,gt,yt,o),gt=o[5],yt=o[6]},a:function a(i,o){return o[5]+=gt,o[6]+=yt,Bt(i,gt,yt,o),gt=o[5],yt=o[6]},L:function L(i,o){return gt=o[0],yt=o[1],vt=mt=null,i.lineTo(gt,yt)},l:function l(i,o){return gt+=o[0],yt+=o[1],vt=mt=null,i.lineTo(gt,yt)},H:function H(i,o){return gt=o[0],vt=mt=null,i.lineTo(gt,yt)},h:function h(i,o){return gt+=o[0],vt=mt=null,i.lineTo(gt,yt)},V:function V(i,o){return yt=o[0],vt=mt=null,i.lineTo(gt,yt)},v:function v(i,o){return yt+=o[0],vt=mt=null,i.lineTo(gt,yt)},Z:function Z(i){return i.closePath(),gt=bt,yt=wt},z:function z(i){return i.closePath(),gt=bt,yt=wt}},Bt=function solveArc(i,o,u,p){var g,y=_slicedToArray(p,7),w=y[0],_=y[1],x=y[2],k=y[3],P=y[4],E=y[5],O=y[6],I=_createForOfIteratorHelper(Ft(E,O,w,_,k,P,x,o,u));try{for(I.s();!(g=I.n()).done;){var B=g.value,D=Dt.apply(void 0,_toConsumableArray(B));i.bezierCurveTo.apply(i,_toConsumableArray(D))}}catch(i){I.e(i)}finally{I.f()}},Ft=function arcToSegments(i,o,u,p,g,y,w,_,x){var k=w*(Math.PI/180),P=Math.sin(k),E=Math.cos(k);u=Math.abs(u),p=Math.abs(p);var O=(vt=E*(_-i)*.5+P*(x-o)*.5)*vt/(u*u)+(mt=E*(x-o)*.5-P*(_-i)*.5)*mt/(p*p);O>1&&(u*=O=Math.sqrt(O),p*=O);var I=E/u,B=P/u,D=-P/p,R=E/p,N=I*_+B*x,U=D*_+R*x,W=I*i+B*o,G=D*i+R*o,j=1/((W-N)*(W-N)+(G-U)*(G-U))-.25;j<0&&(j=0);var X=Math.sqrt(j);y===g&&(X=-X);var K=.5*(N+W)-X*(G-U),Y=.5*(U+G)+X*(W-N),J=Math.atan2(U-Y,N-K),$=Math.atan2(G-Y,W-K)-J;$<0&&1===y?$+=2*Math.PI:$>0&&0===y&&($-=2*Math.PI);for(var tt=Math.ceil(Math.abs($/(.5*Math.PI+.001))),et=[],rt=0;rt<tt;rt++){var nt=J+rt*$/tt,it=J+(rt+1)*$/tt;et[rt]=[K,Y,nt,it,u,p,P,E]}return et},Dt=function segmentToBezier(i,o,u,p,g,y,w,_){var x=_*g,k=-w*y,P=w*g,E=_*y,O=.5*(p-u),I=8/3*Math.sin(.5*O)*Math.sin(.5*O)/Math.sin(O),B=i+Math.cos(u)-I*Math.sin(u),D=o+Math.sin(u)+I*Math.cos(u),R=i+Math.cos(p),N=o+Math.sin(p),U=R+I*Math.sin(p),W=N-I*Math.cos(p);return[x*B+k*D,P*B+E*D,x*U+k*W,P*U+E*W,x*R+k*N,P*R+E*N]},Lt=function(){function SVGPath(){_classCallCheck(this,SVGPath)}return _createClass(SVGPath,null,[{key:"apply",value:function apply(i,o){!function apply(i,o){gt=yt=vt=mt=bt=wt=0;for(var u=0;u<i.length;u++){var p=i[u];"function"==typeof It[p.cmd]&&It[p.cmd](o,p.args)}}(function parse(i){var o,u,p=[],g=[],y="",w=!1,_=0,x=_createForOfIteratorHelper(i);try{for(x.s();!(u=x.n()).done;){var k=u.value;if(null!=Ot[k])_=Ot[k],o&&(y.length>0&&(g[g.length]=+y),p[p.length]={cmd:o,args:g},g=[],y="",w=!1),o=k;else if([" ",","].includes(k)||"-"===k&&y.length>0&&"e"!==y[y.length-1]||"."===k&&w){if(0===y.length)continue;g.length===_?(p[p.length]={cmd:o,args:g},g=[+y],"M"===o&&(o="L"),"m"===o&&(o="l")):g[g.length]=+y,w="."===k,y=["-","."].includes(k)?k:""}else y+=k,"."===k&&(w=!0)}}catch(i){x.e(i)}finally{x.f()}return y.length>0&&(g.length===_?(p[p.length]={cmd:o,args:g},g=[+y],"M"===o&&(o="L"),"m"===o&&(o="l")):g[g.length]=+y),p[p.length]={cmd:o,args:g},p}(o),i)}}]),SVGPath}(),Mt=U.number,Rt=(Math.sqrt(2)-1)/3*4,Nt={initVector:function initVector(){return this._ctm=[1,0,0,1,0,0],this._ctmStack=[]},save:function save(){return this._ctmStack.push(this._ctm.slice()),this.addContent("q")},restore:function restore(){return this._ctm=this._ctmStack.pop()||[1,0,0,1,0,0],this.addContent("Q")},closePath:function closePath(){return this.addContent("h")},lineWidth:function lineWidth(i){return this.addContent("".concat(Mt(i)," w"))},_CAP_STYLES:{BUTT:0,ROUND:1,SQUARE:2},lineCap:function lineCap(i){return"string"==typeof i&&(i=this._CAP_STYLES[i.toUpperCase()]),this.addContent("".concat(i," J"))},_JOIN_STYLES:{MITER:0,ROUND:1,BEVEL:2},lineJoin:function lineJoin(i){return"string"==typeof i&&(i=this._JOIN_STYLES[i.toUpperCase()]),this.addContent("".concat(i," j"))},miterLimit:function miterLimit(i){return this.addContent("".concat(Mt(i)," M"))},dash:function dash(i){var o=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},u=i;Array.isArray(i)||(i=[i,o.space||i]);var p=i.every((function(i){return Number.isFinite(i)&&i>0}));if(!p)throw new Error("dash(".concat(JSON.stringify(u),", ").concat(JSON.stringify(o),") invalid, lengths must be numeric and greater than zero"));return i=i.map(Mt).join(" "),this.addContent("[".concat(i,"] ").concat(Mt(o.phase||0)," d"))},undash:function undash(){return this.addContent("[] 0 d")},moveTo:function moveTo(i,o){return this.addContent("".concat(Mt(i)," ").concat(Mt(o)," m"))},lineTo:function lineTo(i,o){return this.addContent("".concat(Mt(i)," ").concat(Mt(o)," l"))},bezierCurveTo:function bezierCurveTo(i,o,u,p,g,y){return this.addContent("".concat(Mt(i)," ").concat(Mt(o)," ").concat(Mt(u)," ").concat(Mt(p)," ").concat(Mt(g)," ").concat(Mt(y)," c"))},quadraticCurveTo:function quadraticCurveTo(i,o,u,p){return this.addContent("".concat(Mt(i)," ").concat(Mt(o)," ").concat(Mt(u)," ").concat(Mt(p)," v"))},rect:function rect(i,o,u,p){return this.addContent("".concat(Mt(i)," ").concat(Mt(o)," ").concat(Mt(u)," ").concat(Mt(p)," re"))},roundedRect:function roundedRect(i,o,u,p,g){null==g&&(g=0);var y=(g=Math.min(g,.5*u,.5*p))*(1-Rt);return this.moveTo(i+g,o),this.lineTo(i+u-g,o),this.bezierCurveTo(i+u-y,o,i+u,o+y,i+u,o+g),this.lineTo(i+u,o+p-g),this.bezierCurveTo(i+u,o+p-y,i+u-y,o+p,i+u-g,o+p),this.lineTo(i+g,o+p),this.bezierCurveTo(i+y,o+p,i,o+p-y,i,o+p-g),this.lineTo(i,o+g),this.bezierCurveTo(i,o+y,i+y,o,i+g,o),this.closePath()},ellipse:function ellipse(i,o,u,p){null==p&&(p=u);var g=u*Rt,y=p*Rt,w=(i-=u)+2*u,_=(o-=p)+2*p,x=i+u,k=o+p;return this.moveTo(i,k),this.bezierCurveTo(i,k-y,x-g,o,x,o),this.bezierCurveTo(x+g,o,w,k-y,w,k),this.bezierCurveTo(w,k+y,x+g,_,x,_),this.bezierCurveTo(x-g,_,i,k+y,i,k),this.closePath()},circle:function circle(i,o,u){return this.ellipse(i,o,u)},arc:function arc(i,o,u,p,g,y){null==y&&(y=!1);var w=2*Math.PI,_=.5*Math.PI,x=g-p;if(Math.abs(x)>w)x=w;else if(0!==x&&y!==x<0){x=(y?-1:1)*w+x}var k=Math.ceil(Math.abs(x)/_),P=x/k,E=P/_*Rt*u,O=p,I=-Math.sin(O)*E,B=Math.cos(O)*E,D=i+Math.cos(O)*u,R=o+Math.sin(O)*u;this.moveTo(D,R);for(var N=0;N<k;N++){var U=D+I,W=R+B;O+=P,D=i+Math.cos(O)*u,R=o+Math.sin(O)*u;var G=D-(I=-Math.sin(O)*E),j=R-(B=Math.cos(O)*E);this.bezierCurveTo(U,W,G,j,D,R)}return this},polygon:function polygon(){for(var i=arguments.length,o=new Array(i),u=0;u<i;u++)o[u]=arguments[u];this.moveTo.apply(this,_toConsumableArray(o.shift()||[]));for(var p=0,g=o;p<g.length;p++){var y=g[p];this.lineTo.apply(this,_toConsumableArray(y||[]))}return this.closePath()},path:function path(i){return Lt.apply(this,i),this},_windingRule:function _windingRule(i){return/even-?odd/.test(i)?"*":""},fill:function fill(i,o){return/(even-?odd)|(non-?zero)/.test(i)&&(o=i,i=null),i&&this.fillColor(i),this.addContent("f".concat(this._windingRule(o)))},stroke:function stroke(i){return i&&this.strokeColor(i),this.addContent("S")},fillAndStroke:function fillAndStroke(i,o,u){null==o&&(o=i);var p=/(even-?odd)|(non-?zero)/;return p.test(i)&&(u=i,i=null),p.test(o)&&(u=o,o=i),i&&(this.fillColor(i),this.strokeColor(o)),this.addContent("B".concat(this._windingRule(u)))},clip:function clip(i){return this.addContent("W".concat(this._windingRule(i)," n"))},transform:function transform(i,o,u,p,g,y){var w=this._ctm,_=_slicedToArray(w,6),x=_[0],k=_[1],P=_[2],E=_[3],O=_[4],I=_[5];w[0]=x*i+P*o,w[1]=k*i+E*o,w[2]=x*u+P*p,w[3]=k*u+E*p,w[4]=x*g+P*y+O,w[5]=k*g+E*y+I;var B=[i,o,u,p,g,y].map((function(i){return Mt(i)})).join(" ");return this.addContent("".concat(B," cm"))},translate:function translate(i,o){return this.transform(1,0,0,1,i,o)},rotate:function rotate(i){var o,u=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},p=i*Math.PI/180,g=Math.cos(p),y=Math.sin(p),w=o=0;if(null!=u.origin){var _=_slicedToArray(u.origin,2),x=(w=_[0])*g-(o=_[1])*y,k=w*y+o*g;w-=x,o-=k}return this.transform(g,y,-y,g,w,o)},scale:function scale(i,o){var u,p=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{};null==o&&(o=i),"object"==typeof o&&(p=o,o=i);var g=u=0;if(null!=p.origin){var y=_slicedToArray(p.origin,2);g=y[0],u=y[1],g-=i*g,u-=o*u}return this.transform(i,0,0,o,g,u)}},Ut={402:131,8211:150,8212:151,8216:145,8217:146,8218:130,8220:147,8221:148,8222:132,8224:134,8225:135,8226:149,8230:133,8364:128,8240:137,8249:139,8250:155,710:136,8482:153,338:140,339:156,732:152,352:138,353:154,376:159,381:142,382:158},Wt=".notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n.notdef       .notdef        .notdef        .notdef\n  \nspace         exclam         quotedbl       numbersign\ndollar        percent        ampersand      quotesingle\nparenleft     parenright     asterisk       plus\ncomma         hyphen         period         slash\nzero          one            two            three\nfour          five           six            seven\neight         nine           colon          semicolon\nless          equal          greater        question\n  \nat            A              B              C\nD             E              F              G\nH             I              J              K\nL             M              N              O\nP             Q              R              S\nT             U              V              W\nX             Y              Z              bracketleft\nbackslash     bracketright   asciicircum    underscore\n  \ngrave         a              b              c\nd             e              f              g\nh             i              j              k\nl             m              n              o\np             q              r              s\nt             u              v              w\nx             y              z              braceleft\nbar           braceright     asciitilde     .notdef\n  \nEuro          .notdef        quotesinglbase florin\nquotedblbase  ellipsis       dagger         daggerdbl\ncircumflex    perthousand    Scaron         guilsinglleft\nOE            .notdef        Zcaron         .notdef\n.notdef       quoteleft      quoteright     quotedblleft\nquotedblright bullet         endash         emdash\ntilde         trademark      scaron         guilsinglright\noe            .notdef        zcaron         ydieresis\n  \nspace         exclamdown     cent           sterling\ncurrency      yen            brokenbar      section\ndieresis      copyright      ordfeminine    guillemotleft\nlogicalnot    hyphen         registered     macron\ndegree        plusminus      twosuperior    threesuperior\nacute         mu             paragraph      periodcentered\ncedilla       onesuperior    ordmasculine   guillemotright\nonequarter    onehalf        threequarters  questiondown\n  \nAgrave        Aacute         Acircumflex    Atilde\nAdieresis     Aring          AE             Ccedilla\nEgrave        Eacute         Ecircumflex    Edieresis\nIgrave        Iacute         Icircumflex    Idieresis\nEth           Ntilde         Ograve         Oacute\nOcircumflex   Otilde         Odieresis      multiply\nOslash        Ugrave         Uacute         Ucircumflex\nUdieresis     Yacute         Thorn          germandbls\n  \nagrave        aacute         acircumflex    atilde\nadieresis     aring          ae             ccedilla\negrave        eacute         ecircumflex    edieresis\nigrave        iacute         icircumflex    idieresis\neth           ntilde         ograve         oacute\nocircumflex   otilde         odieresis      divide\noslash        ugrave         uacute         ucircumflex\nudieresis     yacute         thorn          ydieresis".split(/\s+/),zt=function(){function AFMFont(i){_classCallCheck(this,AFMFont),this.contents=i,this.attributes={},this.glyphWidths={},this.boundingBoxes={},this.kernPairs={},this.parse(),this.charWidths=new Array(256);for(var o=0;o<=255;o++)this.charWidths[o]=this.glyphWidths[Wt[o]];this.bbox=this.attributes.FontBBox.split(/\s+/).map((function(i){return+i})),this.ascender=+(this.attributes.Ascender||0),this.descender=+(this.attributes.Descender||0),this.xHeight=+(this.attributes.XHeight||0),this.capHeight=+(this.attributes.CapHeight||0),this.lineGap=this.bbox[3]-this.bbox[1]-(this.ascender-this.descender)}return _createClass(AFMFont,null,[{key:"open",value:function open(i){return new AFMFont(E.readFileSync(i,"utf8"))}}]),_createClass(AFMFont,[{key:"parse",value:function parse(){var i,o="",u=_createForOfIteratorHelper(this.contents.split("\n"));try{for(u.s();!(i=u.n()).done;){var p,g,y=i.value;if(p=y.match(/^Start(\w+)/))o=p[1];else if(p=y.match(/^End(\w+)/))o="";else switch(o){case"FontMetrics":var w=(p=y.match(/(^\w+)\s+(.*)/))[1],_=p[2];(g=this.attributes[w])?(Array.isArray(g)||(g=this.attributes[w]=[g]),g.push(_)):this.attributes[w]=_;break;case"CharMetrics":if(!/^CH?\s/.test(y))continue;var x=y.match(/\bN\s+(\.?\w+)\s*;/)[1];this.glyphWidths[x]=+y.match(/\bWX\s+(\d+)\s*;/)[1];break;case"KernPairs":(p=y.match(/^KPX\s+(\.?\w+)\s+(\.?\w+)\s+(-?\d+)/))&&(this.kernPairs[p[1]+"\0"+p[2]]=parseInt(p[3]))}}}catch(i){u.e(i)}finally{u.f()}}},{key:"encodeText",value:function encodeText(i){for(var o=[],u=0,p=i.length;u<p;u++){var g=i.charCodeAt(u);g=Ut[g]||g,o.push(g.toString(16))}return o}},{key:"glyphsForString",value:function glyphsForString(i){for(var o=[],u=0,p=i.length;u<p;u++){var g=i.charCodeAt(u);o.push(this.characterToGlyph(g))}return o}},{key:"characterToGlyph",value:function characterToGlyph(i){return Wt[Ut[i]||i]||".notdef"}},{key:"widthOfGlyph",value:function widthOfGlyph(i){return this.glyphWidths[i]||0}},{key:"getKernPair",value:function getKernPair(i,o){return this.kernPairs[i+"\0"+o]||0}},{key:"advancesForGlyphs",value:function advancesForGlyphs(i){for(var o=[],u=0;u<i.length;u++){var p=i[u],g=i[u+1];o.push(this.widthOfGlyph(p)+this.getKernPair(p,g))}return o}}]),AFMFont}(),Gt=function(){function PDFFont(){_classCallCheck(this,PDFFont)}return _createClass(PDFFont,[{key:"encode",value:function encode(){throw new Error("Must be implemented by subclasses")}},{key:"widthOfString",value:function widthOfString(){throw new Error("Must be implemented by subclasses")}},{key:"ref",value:function ref(){return null!=this.dictionary?this.dictionary:this.dictionary=this.document.ref()}},{key:"finalize",value:function finalize(){if(!this.embedded&&null!=this.dictionary)return this.embed(),this.embedded=!0}},{key:"embed",value:function embed(){throw new Error("Must be implemented by subclasses")}},{key:"lineHeight",value:function lineHeight(i,o){null==o&&(o=!1);var u=o?this.lineGap:0;return(this.ascender+u-this.descender)/1e3*i}}]),PDFFont}(),jt={Courier:function Courier(){return E.readFileSync(p+"/data/Courier.afm","utf8")},"Courier-Bold":function CourierBold(){return E.readFileSync(p+"/data/Courier-Bold.afm","utf8")},"Courier-Oblique":function CourierOblique(){return E.readFileSync(p+"/data/Courier-Oblique.afm","utf8")},"Courier-BoldOblique":function CourierBoldOblique(){return E.readFileSync(p+"/data/Courier-BoldOblique.afm","utf8")},Helvetica:function Helvetica(){return E.readFileSync(p+"/data/Helvetica.afm","utf8")},"Helvetica-Bold":function HelveticaBold(){return E.readFileSync(p+"/data/Helvetica-Bold.afm","utf8")},"Helvetica-Oblique":function HelveticaOblique(){return E.readFileSync(p+"/data/Helvetica-Oblique.afm","utf8")},"Helvetica-BoldOblique":function HelveticaBoldOblique(){return E.readFileSync(p+"/data/Helvetica-BoldOblique.afm","utf8")},"Times-Roman":function TimesRoman(){return E.readFileSync(p+"/data/Times-Roman.afm","utf8")},"Times-Bold":function TimesBold(){return E.readFileSync(p+"/data/Times-Bold.afm","utf8")},"Times-Italic":function TimesItalic(){return E.readFileSync(p+"/data/Times-Italic.afm","utf8")},"Times-BoldItalic":function TimesBoldItalic(){return E.readFileSync(p+"/data/Times-BoldItalic.afm","utf8")},Symbol:function Symbol(){return E.readFileSync(p+"/data/Symbol.afm","utf8")},ZapfDingbats:function ZapfDingbats(){return E.readFileSync(p+"/data/ZapfDingbats.afm","utf8")}},Vt=function(i){_inherits(StandardFont,i);var o=_createSuper(StandardFont);function StandardFont(i,u,p){var g;_classCallCheck(this,StandardFont),(g=o.call(this)).document=i,g.name=u,g.id=p,g.font=new zt(jt[g.name]());var y=g.font;return g.ascender=y.ascender,g.descender=y.descender,g.bbox=y.bbox,g.lineGap=y.lineGap,g.xHeight=y.xHeight,g.capHeight=y.capHeight,g}return _createClass(StandardFont,[{key:"embed",value:function embed(){return this.dictionary.data={Type:"Font",BaseFont:this.name,Subtype:"Type1",Encoding:"WinAnsiEncoding"},this.dictionary.end()}},{key:"encode",value:function encode(i){for(var o=this.font.encodeText(i),u=this.font.glyphsForString("".concat(i)),p=this.font.advancesForGlyphs(u),g=[],y=0;y<u.length;y++){var w=u[y];g.push({xAdvance:p[y],yAdvance:0,xOffset:0,yOffset:0,advanceWidth:this.font.widthOfGlyph(w)})}return[o,g]}},{key:"widthOfString",value:function widthOfString(i,o){var u,p=this.font.glyphsForString("".concat(i)),g=0,y=_createForOfIteratorHelper(this.font.advancesForGlyphs(p));try{for(y.s();!(u=y.n()).done;){g+=u.value}}catch(i){y.e(i)}finally{y.f()}return g*(o/1e3)}}],[{key:"isStandardFont",value:function isStandardFont(i){return i in jt}}]),StandardFont}(Gt),Ht=function toHex(i){return"0000".concat(i.toString(16)).slice(-4)},qt=function(i){_inherits(EmbeddedFont,i);var o=_createSuper(EmbeddedFont);function EmbeddedFont(i,u,p){var g;return _classCallCheck(this,EmbeddedFont),(g=o.call(this)).document=i,g.font=u,g.id=p,g.subset=g.font.createSubset(),g.unicode=[[0]],g.widths=[g.font.getGlyph(0).advanceWidth],g.name=g.font.postscriptName,g.scale=1e3/g.font.unitsPerEm,g.ascender=g.font.ascent*g.scale,g.descender=g.font.descent*g.scale,g.xHeight=g.font.xHeight*g.scale,g.capHeight=g.font.capHeight*g.scale,g.lineGap=g.font.lineGap*g.scale,g.bbox=g.font.bbox,!1!==i.options.fontLayoutCache&&(g.layoutCache=Object.create(null)),g}return _createClass(EmbeddedFont,[{key:"layoutRun",value:function layoutRun(i,o){for(var u=this.font.layout(i,o),p=0;p<u.positions.length;p++){var g=u.positions[p];for(var y in g)g[y]*=this.scale;g.advanceWidth=u.glyphs[p].advanceWidth*this.scale}return u}},{key:"layoutCached",value:function layoutCached(i){if(!this.layoutCache)return this.layoutRun(i);var o;if(o=this.layoutCache[i])return o;var u=this.layoutRun(i);return this.layoutCache[i]=u,u}},{key:"layout",value:function layout(i,o,u){if(o)return this.layoutRun(i,o);for(var p=u?null:[],g=u?null:[],y=0,w=0,_=0;_<=i.length;){var x;if(_===i.length&&w<_||(x=i.charAt(_),[" ","\t"].includes(x))){var k=this.layoutCached(i.slice(w,++_));u||(p=p.concat(k.glyphs),g=g.concat(k.positions)),y+=k.advanceWidth,w=_}else _++}return{glyphs:p,positions:g,advanceWidth:y}}},{key:"encode",value:function encode(i,o){for(var u=this.layout(i,o),p=u.glyphs,g=u.positions,y=[],w=0;w<p.length;w++){var _=p[w],x=this.subset.includeGlyph(_.id);y.push("0000".concat(x.toString(16)).slice(-4)),null==this.widths[x]&&(this.widths[x]=_.advanceWidth*this.scale),null==this.unicode[x]&&(this.unicode[x]=_.codePoints)}return[y,g]}},{key:"widthOfString",value:function widthOfString(i,o,u){return this.layout(i,u,!0).advanceWidth*(o/1e3)}},{key:"embed",value:function embed(){var i=this,o=null!=this.subset.cff,u=this.document.ref();o&&(u.data.Subtype="CIDFontType0C"),this.subset.encodeStream().on("data",(function(i){return u.write(i)})).on("end",(function(){return u.end()}));var p=((null!=this.font["OS/2"]?this.font["OS/2"].sFamilyClass:void 0)||0)>>8,g=0;this.font.post.isFixedPitch&&(g|=1),1<=p&&p<=7&&(g|=2),g|=4,10===p&&(g|=8),this.font.head.macStyle.italic&&(g|=64);var y=[1,2,3,4,5,6].map((function(o){return String.fromCharCode((i.id.charCodeAt(o)||73)+17)})).join("")+"+"+this.font.postscriptName,w=this.font.bbox,_=this.document.ref({Type:"FontDescriptor",FontName:y,Flags:g,FontBBox:[w.minX*this.scale,w.minY*this.scale,w.maxX*this.scale,w.maxY*this.scale],ItalicAngle:this.font.italicAngle,Ascent:this.ascender,Descent:this.descender,CapHeight:(this.font.capHeight||this.font.ascent)*this.scale,XHeight:(this.font.xHeight||0)*this.scale,StemV:0});o?_.data.FontFile3=u:_.data.FontFile2=u,_.end();var x={Type:"Font",Subtype:"CIDFontType0",BaseFont:y,CIDSystemInfo:{Registry:new String("Adobe"),Ordering:new String("Identity"),Supplement:0},FontDescriptor:_,W:[0,this.widths]};o||(x.Subtype="CIDFontType2",x.CIDToGIDMap="Identity");var k=this.document.ref(x);return k.end(),this.dictionary.data={Type:"Font",Subtype:"Type0",BaseFont:y,Encoding:"Identity-H",DescendantFonts:[k],ToUnicode:this.toUnicodeCmap()},this.dictionary.end()}},{key:"toUnicodeCmap",value:function toUnicodeCmap(){var i,o=this.document.ref(),u=[],p=_createForOfIteratorHelper(this.unicode);try{for(p.s();!(i=p.n()).done;){var g,y=i.value,w=[],_=_createForOfIteratorHelper(y);try{for(_.s();!(g=_.n()).done;){var x=g.value;x>65535&&(x-=65536,w.push(Ht(x>>>10&1023|55296)),x=56320|1023&x),w.push(Ht(x))}}catch(i){_.e(i)}finally{_.f()}u.push("<".concat(w.join(" "),">"))}}catch(i){p.e(i)}finally{p.f()}return o.end("/CIDInit /ProcSet findresource begin\n12 dict begin\nbegincmap\n/CIDSystemInfo <<\n  /Registry (Adobe)\n  /Ordering (UCS)\n  /Supplement 0\n>> def\n/CMapName /Adobe-Identity-UCS def\n/CMapType 2 def\n1 begincodespacerange\n<0000><ffff>\nendcodespacerange\n1 beginbfrange\n<0000> <".concat(Ht(u.length-1),"> [").concat(u.join(" "),"]\nendbfrange\nendcmap\nCMapName currentdict /CMap defineresource pop\nend\nend")),o}}]),EmbeddedFont}(Gt),Xt=function(){function PDFFontFactory(){_classCallCheck(this,PDFFontFactory)}return _createClass(PDFFontFactory,null,[{key:"open",value:function open(o,u,p,g){var y;if("string"==typeof u){if(Vt.isStandardFont(u))return new Vt(o,u,g);u=E.readFileSync(u)}if(i.isBuffer(u)?y=_.default.create(u,p):u instanceof Uint8Array?y=_.default.create(i.from(u),p):u instanceof ArrayBuffer&&(y=_.default.create(i.from(new Uint8Array(u)),p)),null==y)throw new Error("Not a supported font format or standard PDF font.");return new qt(o,y,g)}}]),PDFFontFactory}(),Zt={initFonts:function initFonts(){var i=arguments.length>0&&void 0!==arguments[0]?arguments[0]:"Helvetica";this._fontFamilies={},this._fontCount=0,this._fontSize=12,this._font=null,this._registeredFonts={},i&&this.font(i)},font:function font(i,o,u){var p,font;if("number"==typeof o&&(u=o,o=null),"string"==typeof i&&this._registeredFonts[i]){p=i;var g=this._registeredFonts[i];i=g.src,o=g.family}else"string"!=typeof(p=o||i)&&(p=null);if(null!=u&&this.fontSize(u),font=this._fontFamilies[p])return this._font=font,this;var y="F".concat(++this._fontCount);return this._font=Xt.open(this,i,o,y),(font=this._fontFamilies[this._font.name])?(this._font=font,this):(p&&(this._fontFamilies[p]=this._font),this._font.name&&(this._fontFamilies[this._font.name]=this._font),this)},fontSize:function fontSize(i){return this._fontSize=i,this},currentLineHeight:function currentLineHeight(i){return null==i&&(i=!1),this._font.lineHeight(this._fontSize,i)},registerFont:function registerFont(i,o,u){return this._registeredFonts[i]={src:o,family:u},this}},Kt=function(i){_inherits(LineWrapper,i);var o=_createSuper(LineWrapper);function LineWrapper(i,u){var p;return _classCallCheck(this,LineWrapper),(p=o.call(this)).document=i,p.indent=u.indent||0,p.characterSpacing=u.characterSpacing||0,p.wordSpacing=0===u.wordSpacing,p.columns=u.columns||1,p.columnGap=null!=u.columnGap?u.columnGap:18,p.lineWidth=(u.width-p.columnGap*(p.columns-1))/p.columns,p.spaceLeft=p.lineWidth,p.startX=p.document.x,p.startY=p.document.y,p.column=1,p.ellipsis=u.ellipsis,p.continuedX=0,p.features=u.features,null!=u.height?(p.height=u.height,p.maxY=p.startY+u.height):p.maxY=p.document.page.maxY(),p.on("firstLine",(function(i){var o=p.continuedX||p.indent;return p.document.x+=o,p.lineWidth-=o,p.once("line",(function(){if(p.document.x-=o,p.lineWidth+=o,i.continued&&!p.continuedX&&(p.continuedX=p.indent),!i.continued)return p.continuedX=0}))})),p.on("lastLine",(function(i){var o=i.align;return"justify"===o&&(i.align="left"),p.lastLine=!0,p.once("line",(function(){return p.document.y+=i.paragraphGap||0,i.align=o,p.lastLine=!1}))})),p}return _createClass(LineWrapper,[{key:"wordWidth",value:function wordWidth(i){return this.document.widthOfString(i,this)+this.characterSpacing+this.wordSpacing}},{key:"eachWord",value:function eachWord(i,o){for(var u,p=new k.default(i),g=null,y=Object.create(null);u=p.nextBreak();){var w,_=i.slice((null!=g?g.position:void 0)||0,u.position),x=null!=y[_]?y[_]:y[_]=this.wordWidth(_);if(x>this.lineWidth+this.continuedX)for(var P=g,E={};_.length;){var O,I;x>this.spaceLeft?(O=Math.ceil(this.spaceLeft/(x/_.length)),I=(x=this.wordWidth(_.slice(0,O)))<=this.spaceLeft&&O<_.length):O=_.length;for(var B=x>this.spaceLeft&&O>0;B||I;)B?B=(x=this.wordWidth(_.slice(0,--O)))>this.spaceLeft&&O>0:(B=(x=this.wordWidth(_.slice(0,++O)))>this.spaceLeft&&O>0,I=x<=this.spaceLeft&&O<_.length);if(0===O&&this.spaceLeft===this.lineWidth&&(O=1),E.required=u.required||O<_.length,w=o(_.slice(0,O),x,E,P),P={required:!1},_=_.slice(O),x=this.wordWidth(_),!1===w)break}else w=o(_,x,u,g);if(!1===w)break;g=u}}},{key:"wrap",value:function wrap(i,o){var u=this;null!=o.indent&&(this.indent=o.indent),null!=o.characterSpacing&&(this.characterSpacing=o.characterSpacing),null!=o.wordSpacing&&(this.wordSpacing=o.wordSpacing),null!=o.ellipsis&&(this.ellipsis=o.ellipsis);var p=this.document.y+this.document.currentLineHeight(!0);(this.document.y>this.maxY||p>this.maxY)&&this.nextSection();var g="",y=0,w=0,_=0,x=this.document.y,k=function emitLine(){return o.textWidth=y+u.wordSpacing*(w-1),o.wordCount=w,o.lineWidth=u.lineWidth,x=u.document.y,u.emit("line",g,o,u),_++};return this.emit("sectionStart",o,this),this.eachWord(i,(function(i,p,_,x){if((null==x||x.required)&&(u.emit("firstLine",o,u),u.spaceLeft=u.lineWidth),p<=u.spaceLeft&&(g+=i,y+=p,w++),_.required||p>u.spaceLeft){var P=u.document.currentLineHeight(!0);if(null!=u.height&&u.ellipsis&&u.document.y+2*P>u.maxY&&u.column>=u.columns){for(!0===u.ellipsis&&(u.ellipsis="…"),g=g.replace(/\s+$/,""),y=u.wordWidth(g+u.ellipsis);g&&y>u.lineWidth;)g=g.slice(0,-1).replace(/\s+$/,""),y=u.wordWidth(g+u.ellipsis);y<=u.lineWidth&&(g+=u.ellipsis),y=u.wordWidth(g)}if(_.required&&(p>u.spaceLeft&&(k(),g=i,y=p,w=1),u.emit("lastLine",o,u)),k(),u.document.y+P>u.maxY)if(!u.nextSection())return w=0,g="",!1;return _.required?(u.spaceLeft=u.lineWidth,g="",y=0,w=0):(u.spaceLeft=u.lineWidth-p,g=i,y=p,w=1)}return u.spaceLeft-=p})),w>0&&(this.emit("lastLine",o,this),k()),this.emit("sectionEnd",o,this),!0===o.continued?(_>1&&(this.continuedX=0),this.continuedX+=o.textWidth||0,this.document.y=x):this.document.x=this.startX}},{key:"nextSection",value:function nextSection(i){if(this.emit("sectionEnd",i,this),++this.column>this.columns){if(null!=this.height)return!1;var o;if(this.document.continueOnNewPage(),this.column=1,this.startY=this.document.page.margins.top,this.maxY=this.document.page.maxY(),this.document.x=this.startX,this.document._fillColor)(o=this.document).fillColor.apply(o,_toConsumableArray(this.document._fillColor));this.emit("pageBreak",i,this)}else this.document.x+=this.lineWidth+this.columnGap,this.document.y=this.startY,this.emit("columnBreak",i,this);return this.emit("sectionStart",i,this),!0}}]),LineWrapper}(x.EventEmitter),Yt=U.number,Jt={initText:function initText(){return this._line=this._line.bind(this),this.x=0,this.y=0,this._lineGap=0},lineGap:function lineGap(i){return this._lineGap=i,this},moveDown:function moveDown(i){return null==i&&(i=1),this.y+=this.currentLineHeight(!0)*i+this._lineGap,this},moveUp:function moveUp(i){return null==i&&(i=1),this.y-=this.currentLineHeight(!0)*i+this._lineGap,this},_text:function _text(i,o,u,p,g){var y=this;p=this._initOptions(o,u,p),i=null==i?"":"".concat(i),p.wordSpacing&&(i=i.replace(/\s{2,}/g," "));var w=function addStructure(){p.structParent&&p.structParent.add(y.struct(p.structType||"P",[y.markStructureContent(p.structType||"P")]))};if(p.width){var _=this._wrapper;_||((_=new Kt(this,p)).on("line",g),_.on("firstLine",w)),this._wrapper=p.continued?_:null,this._textOptions=p.continued?p:null,_.wrap(i,p)}else{var x,k=_createForOfIteratorHelper(i.split("\n"));try{for(k.s();!(x=k.n()).done;){var P=x.value;w(),g(P,p)}}catch(i){k.e(i)}finally{k.f()}}return this},text:function text(i,o,u,p){return this._text(i,o,u,p,this._line)},widthOfString:function widthOfString(i){var o=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};return this._font.widthOfString(i,this._fontSize,o.features)+(o.characterSpacing||0)*(i.length-1)},heightOfString:function heightOfString(i,o){var u=this,p=this.x,g=this.y;(o=this._initOptions(o)).height=1/0;var y=o.lineGap||this._lineGap||0;this._text(i,this.x,this.y,o,(function(){return u.y+=u.currentLineHeight(!0)+y}));var w=this.y-g;return this.x=p,this.y=g,w},list:function list(i,o,u,p,g){var y=this,w=(p=this._initOptions(o,u,p)).listType||"bullet",_=Math.round(this._font.ascender/1e3*this._fontSize),x=_/2,k=p.bulletRadius||_/3,P=p.textIndent||("bullet"===w?5*k:2*_),E=p.bulletIndent||("bullet"===w?8*k:2*_),O=1,I=[],B=[],D=[];!function flatten(i){for(var o=1,u=0;u<i.length;u++){var p=i[u];Array.isArray(p)?(O++,flatten(p),O--):(I.push(p),B.push(O),"bullet"!==w&&D.push(o++))}}(i);(g=new Kt(this,p)).on("line",this._line),O=1;var R=0;return g.on("firstLine",(function(){var i,o,u,_,I;if(p.structParent)if(p.structTypes){var N=_slicedToArray(p.structTypes,3);o=N[0],u=N[1],_=N[2]}else o="LI",u="Lbl",_="LBody";if(o?(i=y.struct(o),p.structParent.add(i)):p.structParent&&(i=p.structParent),(I=B[R++])!==O){var U=E*(I-O);y.x+=U,g.lineWidth-=U,O=I}switch(i&&(u||_)&&i.add(y.struct(u||_,[y.markStructureContent(u||_)])),w){case"bullet":y.circle(y.x-P+k,y.y+x,k),y.fill();break;case"numbered":case"lettered":var W=function label(i){switch(w){case"numbered":return"".concat(i,".");case"lettered":var o=String.fromCharCode((i-1)%26+65),u=Math.floor((i-1)/26+1),p=Array(u+1).join(o);return"".concat(p,".")}}(D[R-1]);y._fragment(W,y.x-P,y.y,p)}i&&u&&_&&i.add(y.struct(_,[y.markStructureContent(_)])),i&&i!==p.structParent&&i.end()})),g.on("sectionStart",(function(){var i=P+E*(O-1);return y.x+=i,g.lineWidth-=i})),g.on("sectionEnd",(function(){var i=P+E*(O-1);return y.x-=i,g.lineWidth+=i})),g.wrap(I.join("\n"),p),this},_initOptions:function _initOptions(){var i=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{},o=arguments.length>1?arguments[1]:void 0,u=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{};"object"==typeof i&&(u=i,i=null);var p=Object.assign({},u);if(this._textOptions)for(var g in this._textOptions){var y=this._textOptions[g];"continued"!==g&&void 0===p[g]&&(p[g]=y)}return null!=i&&(this.x=i),null!=o&&(this.y=o),!1!==p.lineBreak&&(null==p.width&&(p.width=this.page.width-this.x-this.page.margins.right),p.width=Math.max(p.width,0)),p.columns||(p.columns=0),null==p.columnGap&&(p.columnGap=18),p},_line:function _line(i){var o=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},u=arguments.length>2?arguments[2]:void 0;this._fragment(i,this.x,this.y,o);var p=o.lineGap||this._lineGap||0;return u?this.y+=this.currentLineHeight(!0)+p:this.x+=this.widthOfString(i)},_fragment:function _fragment(i,o,u,p){var g,y,w,_,x,k,P=this;if(0!==(i="".concat(i).replace(/\n/g,"")).length){var E=p.align||"left",O=p.wordSpacing||0,I=p.characterSpacing||0;if(p.width)switch(E){case"right":x=this.widthOfString(i.replace(/\s+$/,""),p),o+=p.lineWidth-x;break;case"center":o+=p.lineWidth/2-p.textWidth/2;break;case"justify":k=i.trim().split(/\s+/),x=this.widthOfString(i.replace(/\s+/g,""),p);var B=this.widthOfString(" ")+I;O=Math.max(0,(p.lineWidth-x)/Math.max(1,k.length-1)-B)}if("number"==typeof p.baseline)g=-p.baseline;else{switch(p.baseline){case"svg-middle":g=.5*this._font.xHeight;break;case"middle":case"svg-central":g=.5*(this._font.descender+this._font.ascender);break;case"bottom":case"ideographic":g=this._font.descender;break;case"alphabetic":g=0;break;case"mathematical":g=.5*this._font.ascender;break;case"hanging":g=.8*this._font.ascender;break;case"top":g=this._font.ascender;break;default:g=this._font.ascender}g=g/1e3*this._fontSize}var D,R=p.textWidth+O*(p.wordCount-1)+I*(i.length-1);if(null!=p.link&&this.link(o,u,R,this.currentLineHeight(),p.link),null!=p.goTo&&this.goTo(o,u,R,this.currentLineHeight(),p.goTo),null!=p.destination&&this.addNamedDestination(p.destination,"XYZ",o,u,null),p.underline||p.strike){this.save(),p.stroke||this.strokeColor.apply(this,_toConsumableArray(this._fillColor||[]));var N=this._fontSize<10?.5:Math.floor(this._fontSize/10);this.lineWidth(N);var U=p.underline?1:2,W=u+this.currentLineHeight()/U;p.underline&&(W-=N),this.moveTo(o,W),this.lineTo(o+R,W),this.stroke(),this.restore()}if(this.save(),p.oblique)D="number"==typeof p.oblique?-Math.tan(p.oblique*Math.PI/180):-.25,this.transform(1,0,0,1,o,u),this.transform(1,0,D,1,-D*g,0),this.transform(1,0,0,1,-o,-u);this.transform(1,0,0,-1,0,this.page.height),u=this.page.height-u-g,null==this.page.fonts[this._font.id]&&(this.page.fonts[this._font.id]=this._font.ref()),this.addContent("BT"),this.addContent("1 0 0 1 ".concat(Yt(o)," ").concat(Yt(u)," Tm")),this.addContent("/".concat(this._font.id," ").concat(Yt(this._fontSize)," Tf"));var G=p.fill&&p.stroke?2:p.stroke?1:0;if(G&&this.addContent("".concat(G," Tr")),I&&this.addContent("".concat(Yt(I)," Tc")),O){k=i.trim().split(/\s+/),O+=this.widthOfString(" ")+I,O*=1e3/this._fontSize,y=[],_=[];var j,X=_createForOfIteratorHelper(k);try{for(X.s();!(j=X.n()).done;){var K=j.value,Y=_slicedToArray(this._font.encode(K,p.features),2),J=Y[0],$=Y[1];y=y.concat(J),_=_.concat($);var tt={},et=_[_.length-1];for(var rt in et){var nt=et[rt];tt[rt]=nt}tt.xAdvance+=O,_[_.length-1]=tt}}catch(i){X.e(i)}finally{X.f()}}else{var it=_slicedToArray(this._font.encode(i,p.features),2);y=it[0],_=it[1]}var ot=this._fontSize/1e3,at=[],st=0,lt=!1,ct=function addSegment(i){if(st<i){var o=y.slice(st,i).join(""),u=_[i-1].xAdvance-_[i-1].advanceWidth;at.push("<".concat(o,"> ").concat(Yt(-u)))}return st=i},ut=function flush(i){if(ct(i),at.length>0)return P.addContent("[".concat(at.join(" "),"] TJ")),at.length=0};for(w=0;w<_.length;w++){var ft=_[w];ft.xOffset||ft.yOffset?(ut(w),this.addContent("1 0 0 1 ".concat(Yt(o+ft.xOffset*ot)," ").concat(Yt(u+ft.yOffset*ot)," Tm")),ut(w+1),lt=!0):(lt&&(this.addContent("1 0 0 1 ".concat(Yt(o)," ").concat(Yt(u)," Tm")),lt=!1),ft.xAdvance-ft.advanceWidth!=0&&ct(w+1)),o+=ft.xAdvance*ot}return ut(w),this.addContent("ET"),this.restore()}}},Qt=[65472,65473,65474,65475,65477,65478,65479,65480,65481,65482,65483,65484,65485,65486,65487],$t={1:"DeviceGray",3:"DeviceRGB",4:"DeviceCMYK"},te=function(){function JPEG(i,o){var u;if(_classCallCheck(this,JPEG),this.data=i,this.label=o,65496!==this.data.readUInt16BE(0))throw"SOI not found in JPEG";for(var p=2;p<this.data.length&&(u=this.data.readUInt16BE(p),p+=2,!Qt.includes(u));)p+=this.data.readUInt16BE(p);if(!Qt.includes(u))throw"Invalid JPEG.";p+=2,this.bits=this.data[p++],this.height=this.data.readUInt16BE(p),p+=2,this.width=this.data.readUInt16BE(p),p+=2;var g=this.data[p++];this.colorSpace=$t[g],this.obj=null}return _createClass(JPEG,[{key:"embed",value:function embed(i){if(!this.obj)return this.obj=i.ref({Type:"XObject",Subtype:"Image",BitsPerComponent:this.bits,Width:this.width,Height:this.height,ColorSpace:this.colorSpace,Filter:"DCTDecode"}),"DeviceCMYK"===this.colorSpace&&(this.obj.data.Decode=[1,0,1,0,1,0,1,0]),this.obj.end(this.data),this.data=null}}]),JPEG}(),ee=function(){function PNGImage(i,o){_classCallCheck(this,PNGImage),this.label=o,this.image=new P.default(i),this.width=this.image.width,this.height=this.image.height,this.imgData=this.image.imgData,this.obj=null}return _createClass(PNGImage,[{key:"embed",value:function embed(o){var u=!1;if(this.document=o,!this.obj){var p=this.image.hasAlphaChannel,g=1===this.image.interlaceMethod;if(this.obj=this.document.ref({Type:"XObject",Subtype:"Image",BitsPerComponent:p?8:this.image.bits,Width:this.width,Height:this.height,Filter:"FlateDecode"}),!p){var y=this.document.ref({Predictor:g?1:15,Colors:this.image.colors,BitsPerComponent:this.image.bits,Columns:this.width});this.obj.data.DecodeParms=y,y.end()}if(0===this.image.palette.length)this.obj.data.ColorSpace=this.image.colorSpace;else{var w=this.document.ref();w.end(i.from(this.image.palette)),this.obj.data.ColorSpace=["Indexed","DeviceRGB",this.image.palette.length/3-1,w]}if(null!=this.image.transparency.grayscale){var _=this.image.transparency.grayscale;this.obj.data.Mask=[_,_]}else if(this.image.transparency.rgb){var x,k=[],P=_createForOfIteratorHelper(this.image.transparency.rgb);try{for(P.s();!(x=P.n()).done;){var E=x.value;k.push(E,E)}}catch(i){P.e(i)}finally{P.f()}this.obj.data.Mask=k}else{if(this.image.transparency.indexed)return u=!0,this.loadIndexedAlphaChannel();if(p)return u=!0,this.splitAlphaChannel()}if(g&&!u)return this.decodeData();this.finalize()}}},{key:"finalize",value:function finalize(){if(this.alphaChannel){var i=this.document.ref({Type:"XObject",Subtype:"Image",Height:this.height,Width:this.width,BitsPerComponent:8,Filter:"FlateDecode",ColorSpace:"DeviceGray",Decode:[0,1]});i.end(this.alphaChannel),this.obj.data.SMask=i}return this.obj.end(this.imgData),this.image=null,this.imgData=null}},{key:"splitAlphaChannel",value:function splitAlphaChannel(){var o=this;return this.image.decodePixels((function(u){for(var p,g,w=o.image.colors,_=o.width*o.height,x=i.alloc(_*w),k=i.alloc(_),P=g=p=0,E=u.length,O=16===o.image.bits?1:0;P<E;){for(var I=0;I<w;I++)x[g++]=u[P++],P+=O;k[p++]=u[P++],P+=O}return o.imgData=y.default.deflateSync(x),o.alphaChannel=y.default.deflateSync(k),o.finalize()}))}},{key:"loadIndexedAlphaChannel",value:function loadIndexedAlphaChannel(){var o=this,u=this.image.transparency.indexed;return this.image.decodePixels((function(p){for(var g=i.alloc(o.width*o.height),w=0,_=0,x=p.length;_<x;_++)g[w++]=u[p[_]];return o.alphaChannel=y.default.deflateSync(g),o.finalize()}))}},{key:"decodeData",value:function decodeData(){var i=this;this.image.decodePixels((function(o){i.imgData=y.default.deflateSync(o),i.finalize()}))}}]),PNGImage}(),re=function(){function PDFImage(){_classCallCheck(this,PDFImage)}return _createClass(PDFImage,null,[{key:"open",value:function open(o,u){var p;if(i.isBuffer(o))p=o;else if(o instanceof ArrayBuffer)p=i.from(new Uint8Array(o));else{var g;if(g=/^data:.+;base64,(.*)$/.exec(o))p=i.from(g[1],"base64");else if(!(p=E.readFileSync(o)))return}if(255===p[0]&&216===p[1])return new te(p,u);if(137===p[0]&&"PNG"===p.toString("ascii",1,4))return new ee(p,u);throw new Error("Unknown image format.")}}]),PDFImage}(),ne={initImages:function initImages(){return this._imageRegistry={},this._imageCount=0},image:function image(i,o,u){var p,g,y,image,w,_,x,k=arguments.length>3&&void 0!==arguments[3]?arguments[3]:{};"object"==typeof o&&(k=o,o=null),o=null!=(_=null!=o?o:k.x)?_:this.x,u=null!=(x=null!=u?u:k.y)?x:this.y,"string"==typeof i&&(image=this._imageRegistry[i]),image||(image=i.width&&i.height?i:this.openImage(i)),image.obj||image.embed(this),null==this.page.xobjects[image.label]&&(this.page.xobjects[image.label]=image.obj);var P=k.width||image.width,E=k.height||image.height;if(k.width&&!k.height){var O=P/image.width;P=image.width*O,E=image.height*O}else if(k.height&&!k.width){var I=E/image.height;P=image.width*I,E=image.height*I}else if(k.scale)P=image.width*k.scale,E=image.height*k.scale;else if(k.fit){var B=_slicedToArray(k.fit,2);g=(y=B[0])/(p=B[1]),(w=image.width/image.height)>g?(P=y,E=y/w):(E=p,P=p*w)}else if(k.cover){var D=_slicedToArray(k.cover,2);g=(y=D[0])/(p=D[1]),(w=image.width/image.height)>g?(E=p,P=p*w):(P=y,E=y/w)}return(k.fit||k.cover)&&("center"===k.align?o=o+y/2-P/2:"right"===k.align&&(o=o+y-P),"center"===k.valign?u=u+p/2-E/2:"bottom"===k.valign&&(u=u+p-E)),null!=k.link&&this.link(o,u,P,E,k.link),null!=k.goTo&&this.goTo(o,u,P,E,k.goTo),null!=k.destination&&this.addNamedDestination(k.destination,"XYZ",o,u,null),this.y===u&&(this.y+=E),this.save(),this.transform(P,0,0,-E,o,u+E),this.addContent("/".concat(image.label," Do")),this.restore(),this},openImage:function openImage(i){var o;return"string"==typeof i&&(o=this._imageRegistry[i]),o||(o=re.open(i,"I".concat(++this._imageCount)),"string"==typeof i&&(this._imageRegistry[i]=o)),o}},ie={annotate:function annotate(i,o,u,p,g){for(var y in g.Type="Annot",g.Rect=this._convertRect(i,o,u,p),g.Border=[0,0,0],"Link"===g.Subtype&&void 0===g.F&&(g.F=4),"Link"!==g.Subtype&&null==g.C&&(g.C=this._normalizeColor(g.color||[0,0,0])),delete g.color,"string"==typeof g.Dest&&(g.Dest=new String(g.Dest)),g){var w=g[y];g[y[0].toUpperCase()+y.slice(1)]=w}var _=this.ref(g);return this.page.annotations.push(_),_.end(),this},note:function note(i,o,u,p,g){var y=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return y.Subtype="Text",y.Contents=new String(g),y.Name="Comment",null==y.color&&(y.color=[243,223,92]),this.annotate(i,o,u,p,y)},goTo:function goTo(i,o,u,p,g){var y=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return y.Subtype="Link",y.A=this.ref({S:"GoTo",D:new String(g)}),y.A.end(),this.annotate(i,o,u,p,y)},link:function link(i,o,u,p,g){var y=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};if(y.Subtype="Link","number"==typeof g){var w=this._root.data.Pages.data;if(!(g>=0&&g<w.Kids.length))throw new Error("The document has no page ".concat(g));y.A=this.ref({S:"GoTo",D:[w.Kids[g],"XYZ",null,null,null]}),y.A.end()}else y.A=this.ref({S:"URI",URI:new String(g)}),y.A.end();return this.annotate(i,o,u,p,y)},_markup:function _markup(i,o,u,p){var g=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{},y=this._convertRect(i,o,u,p),w=_slicedToArray(y,4),_=w[0],x=w[1],k=w[2],P=w[3];return g.QuadPoints=[_,P,k,P,_,x,k,x],g.Contents=new String,this.annotate(i,o,u,p,g)},highlight:function highlight(i,o,u,p){var g=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return g.Subtype="Highlight",null==g.color&&(g.color=[241,238,148]),this._markup(i,o,u,p,g)},underline:function underline(i,o,u,p){var g=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return g.Subtype="Underline",this._markup(i,o,u,p,g)},strike:function strike(i,o,u,p){var g=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return g.Subtype="StrikeOut",this._markup(i,o,u,p,g)},lineAnnotation:function lineAnnotation(i,o,u,p){var g=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return g.Subtype="Line",g.Contents=new String,g.L=[i,this.page.height-o,u,this.page.height-p],this.annotate(i,o,u,p,g)},rectAnnotation:function rectAnnotation(i,o,u,p){var g=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return g.Subtype="Square",g.Contents=new String,this.annotate(i,o,u,p,g)},ellipseAnnotation:function ellipseAnnotation(i,o,u,p){var g=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{};return g.Subtype="Circle",g.Contents=new String,this.annotate(i,o,u,p,g)},textAnnotation:function textAnnotation(i,o,u,p,g){var y=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return y.Subtype="FreeText",y.Contents=new String(g),y.DA=new String,this.annotate(i,o,u,p,y)},fileAnnotation:function fileAnnotation(i,o,u,p){var g=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{},y=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{},w=this.file(g.src,Object.assign({hidden:!0},g));return y.Subtype="FileAttachment",y.FS=w,y.Contents?y.Contents=new String(y.Contents):w.data.Desc&&(y.Contents=w.data.Desc),this.annotate(i,o,u,p,y)},_convertRect:function _convertRect(i,o,u,p){var g=o;o+=p;var y=i+u,w=_slicedToArray(this._ctm,6),_=w[0],x=w[1],k=w[2],P=w[3],E=w[4],O=w[5];return[i=_*i+k*o+E,o=x*i+P*o+O,y=_*y+k*g+E,g=x*y+P*g+O]}},oe=function(){function PDFOutline(i,o,u,p){var g=arguments.length>4&&void 0!==arguments[4]?arguments[4]:{expanded:!1};_classCallCheck(this,PDFOutline),this.document=i,this.options=g,this.outlineData={},null!==p&&(this.outlineData.Dest=[p.dictionary,"Fit"]),null!==o&&(this.outlineData.Parent=o),null!==u&&(this.outlineData.Title=new String(u)),this.dictionary=this.document.ref(this.outlineData),this.children=[]}return _createClass(PDFOutline,[{key:"addItem",value:function addItem(i){var o=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{expanded:!1},u=new PDFOutline(this.document,this.dictionary,i,this.document.page,o);return this.children.push(u),u}},{key:"endOutline",value:function endOutline(){if(this.children.length>0){this.options.expanded&&(this.outlineData.Count=this.children.length);var i=this.children[0],o=this.children[this.children.length-1];this.outlineData.First=i.dictionary,this.outlineData.Last=o.dictionary;for(var u=0,p=this.children.length;u<p;u++){var g=this.children[u];u>0&&(g.outlineData.Prev=this.children[u-1].dictionary),u<this.children.length-1&&(g.outlineData.Next=this.children[u+1].dictionary),g.endOutline()}}return this.dictionary.end()}}]),PDFOutline}(),ae={initOutline:function initOutline(){return this.outline=new oe(this,null,null,null)},endOutline:function endOutline(){if(this.outline.endOutline(),this.outline.children.length>0)return this._root.data.Outlines=this.outline.dictionary,this._root.data.PageMode="UseOutlines"}},se=function(){function PDFStructureContent(i,o){_classCallCheck(this,PDFStructureContent),this.refs=[{pageRef:i,mcid:o}]}return _createClass(PDFStructureContent,[{key:"push",value:function push(i){var o=this;i.refs.forEach((function(i){return o.refs.push(i)}))}}]),PDFStructureContent}(),le=function(){function PDFStructureElement(i,o){var u=this,p=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{},g=arguments.length>3&&void 0!==arguments[3]?arguments[3]:null;_classCallCheck(this,PDFStructureElement),this.document=i,this._attached=!1,this._ended=!1,this._flushed=!1,this.dictionary=i.ref({S:o});var y=this.dictionary.data;(Array.isArray(p)||this._isValidChild(p))&&(g=p,p={}),void 0!==p.title&&(y.T=new String(p.title)),void 0!==p.lang&&(y.Lang=new String(p.lang)),void 0!==p.alt&&(y.Alt=new String(p.alt)),void 0!==p.expanded&&(y.E=new String(p.expanded)),void 0!==p.actual&&(y.ActualText=new String(p.actual)),this._children=[],g&&(Array.isArray(g)||(g=[g]),g.forEach((function(i){return u.add(i)})),this.end())}return _createClass(PDFStructureElement,[{key:"add",value:function add(i){if(this._ended)throw new Error("Cannot add child to already-ended structure element");if(!this._isValidChild(i))throw new Error("Invalid structure element child");return i instanceof PDFStructureElement&&(i.setParent(this.dictionary),this._attached&&i.setAttached()),i instanceof se&&this._addContentToParentTree(i),"function"==typeof i&&this._attached&&(i=this._contentForClosure(i)),this._children.push(i),this}},{key:"_addContentToParentTree",value:function _addContentToParentTree(i){var o=this;i.refs.forEach((function(i){var u=i.pageRef,p=i.mcid;o.document.getStructParentTree().get(u.data.StructParents)[p]=o.dictionary}))}},{key:"setParent",value:function setParent(i){if(this.dictionary.data.P)throw new Error("Structure element added to more than one parent");this.dictionary.data.P=i,this._flush()}},{key:"setAttached",value:function setAttached(){var i=this;this._attached||(this._children.forEach((function(o,u){o instanceof PDFStructureElement&&o.setAttached(),"function"==typeof o&&(i._children[u]=i._contentForClosure(o))})),this._attached=!0,this._flush())}},{key:"end",value:function end(){this._ended||(this._children.filter((function(i){return i instanceof PDFStructureElement})).forEach((function(i){return i.end()})),this._ended=!0,this._flush())}},{key:"_isValidChild",value:function _isValidChild(i){return i instanceof PDFStructureElement||i instanceof se||"function"==typeof i}},{key:"_contentForClosure",value:function _contentForClosure(i){var o=this.document.markStructureContent(this.dictionary.data.S);return i(),this.document.endMarkedContent(),this._addContentToParentTree(o),o}},{key:"_isFlushable",value:function _isFlushable(){return!(!this.dictionary.data.P||!this._ended)&&this._children.every((function(i){return"function"!=typeof i&&(!(i instanceof PDFStructureElement)||i._isFlushable())}))}},{key:"_flush",value:function _flush(){var i=this;!this._flushed&&this._isFlushable()&&(this.dictionary.data.K=[],this._children.forEach((function(o){return i._flushChild(o)})),this.dictionary.end(),this._children=[],this.dictionary.data.K=null,this._flushed=!0)}},{key:"_flushChild",value:function _flushChild(i){var o=this;i instanceof PDFStructureElement&&this.dictionary.data.K.push(i.dictionary),i instanceof se&&i.refs.forEach((function(i){var u=i.pageRef,p=i.mcid;o.dictionary.data.Pg||(o.dictionary.data.Pg=u),o.dictionary.data.Pg===u?o.dictionary.data.K.push(p):o.dictionary.data.K.push({Type:"MCR",Pg:u,MCID:p})}))}}]),PDFStructureElement}(),ce=function(i){_inherits(PDFNumberTree,i);var o=_createSuper(PDFNumberTree);function PDFNumberTree(){return _classCallCheck(this,PDFNumberTree),o.apply(this,arguments)}return _createClass(PDFNumberTree,[{key:"_compareKeys",value:function _compareKeys(i,o){return parseInt(i)-parseInt(o)}},{key:"_keysName",value:function _keysName(){return"Nums"}},{key:"_dataForKey",value:function _dataForKey(i){return parseInt(i)}}]),PDFNumberTree}(I),ue={initMarkings:function initMarkings(i){this.structChildren=[],i.tagged&&(this.getMarkingsDictionary().data.Marked=!0)},markContent:function markContent(i){var o=arguments.length>1&&void 0!==arguments[1]?arguments[1]:null;if("Artifact"===i||o&&o.mcid){var u=0;for(this.page.markings.forEach((function(i){(u||i.structContent||"Artifact"===i.tag)&&u++}));u--;)this.endMarkedContent()}if(!o)return this.page.markings.push({tag:i}),this.addContent("/".concat(i," BMC")),this;this.page.markings.push({tag:i,options:o});var p={};return void 0!==o.mcid&&(p.MCID=o.mcid),"Artifact"===i&&("string"==typeof o.type&&(p.Type=o.type),Array.isArray(o.bbox)&&(p.BBox=[o.bbox[0],this.page.height-o.bbox[3],o.bbox[2],this.page.height-o.bbox[1]]),Array.isArray(o.attached)&&o.attached.every((function(i){return"string"==typeof i}))&&(p.Attached=o.attached)),"Span"===i&&(o.lang&&(p.Lang=new String(o.lang)),o.alt&&(p.Alt=new String(o.alt)),o.expanded&&(p.E=new String(o.expanded)),o.actual&&(p.ActualText=new String(o.actual))),this.addContent("/".concat(i," ").concat(U.convert(p)," BDC")),this},markStructureContent:function markStructureContent(i){var o=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},u=this.getStructParentTree().get(this.page.structParentTreeKey),p=u.length;u.push(null),this.markContent(i,_objectSpread2(_objectSpread2({},o),{},{mcid:p}));var g=new se(this.page.dictionary,p);return this.page.markings.slice(-1)[0].structContent=g,g},endMarkedContent:function endMarkedContent(){return this.page.markings.pop(),this.addContent("EMC"),this},struct:function struct(i){var o=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},u=arguments.length>2&&void 0!==arguments[2]?arguments[2]:null;return new le(this,i,o,u)},addStructure:function addStructure(i){var o=this.getStructTreeRoot();return i.setParent(o),i.setAttached(),this.structChildren.push(i),o.data.K||(o.data.K=[]),o.data.K.push(i.dictionary),this},initPageMarkings:function initPageMarkings(i){var o=this;i.forEach((function(i){if(i.structContent){var u=i.structContent,p=o.markStructureContent(i.tag,i.options);u.push(p),o.page.markings.slice(-1)[0].structContent=u}else o.markContent(i.tag,i.options)}))},endPageMarkings:function endPageMarkings(i){var o=i.markings;return o.forEach((function(){return i.write("EMC")})),i.markings=[],o},getMarkingsDictionary:function getMarkingsDictionary(){return this._root.data.Markings||(this._root.data.Markings=this.ref({})),this._root.data.Markings},getStructTreeRoot:function getStructTreeRoot(){return this._root.data.StructTreeRoot||(this._root.data.StructTreeRoot=this.ref({Type:"StructTreeRoot",ParentTree:new ce,ParentTreeNextKey:0})),this._root.data.StructTreeRoot},getStructParentTree:function getStructParentTree(){return this.getStructTreeRoot().data.ParentTree},createStructParentTreeNextKey:function createStructParentTreeNextKey(){this.getMarkingsDictionary();var i=this.getStructTreeRoot(),o=i.data.ParentTreeNextKey++;return i.data.ParentTree.add(o,[]),o},endMarkings:function endMarkings(){var i=this._root.data.StructTreeRoot;i&&(i.end(),this.structChildren.forEach((function(i){return i.end()}))),this._root.data.Markings&&this._root.data.Markings.end()}},fe={readOnly:1,required:2,noExport:4,multiline:4096,password:8192,toggleToOffButton:16384,radioButton:32768,pushButton:65536,combo:131072,edit:262144,sort:524288,multiSelect:2097152,noSpell:4194304},he={left:0,center:1,right:2},de={value:"V",defaultValue:"DV"},pe={zip:"0",zipPlus4:"1",zip4:"1",phone:"2",ssn:"3"},ge={nDec:0,sepComma:!1,negStyle:"MinusBlack",currency:"",currencyPrepend:!0},ye={nDec:0,sepComma:!1},ve={initForm:function initForm(){if(!this._font)throw new Error("Must set a font before calling initForm method");this._acroform={fonts:{},defaultFont:this._font.name},this._acroform.fonts[this._font.id]=this._font.ref();var i={Fields:[],NeedAppearances:!0,DA:new String("/".concat(this._font.id," 0 Tf 0 g")),DR:{Font:{}}};i.DR.Font[this._font.id]=this._font.ref();var o=this.ref(i);return this._root.data.AcroForm=o,this},endAcroForm:function endAcroForm(){var i=this;if(this._root.data.AcroForm){if(!Object.keys(this._acroform.fonts).length&&!this._acroform.defaultFont)throw new Error("No fonts specified for PDF form");var o=this._root.data.AcroForm.data.DR.Font;Object.keys(this._acroform.fonts).forEach((function(u){o[u]=i._acroform.fonts[u]})),this._root.data.AcroForm.data.Fields.forEach((function(o){i._endChild(o)})),this._root.data.AcroForm.end()}return this},_endChild:function _endChild(i){var o=this;return Array.isArray(i.data.Kids)&&(i.data.Kids.forEach((function(i){o._endChild(i)})),i.end()),this},formField:function formField(i){var o=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{},u=this._fieldDict(i,null,o),p=this.ref(u);return this._addToParent(p),p},formAnnotation:function formAnnotation(i,o,u,p,g,y){var w=arguments.length>6&&void 0!==arguments[6]?arguments[6]:{},_=this._fieldDict(i,o,w);_.Subtype="Widget",void 0===_.F&&(_.F=4),this.annotate(u,p,g,y,_);var x=this.page.annotations[this.page.annotations.length-1];return this._addToParent(x)},formText:function formText(i,o,u,p,g){var y=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(i,"text",o,u,p,g,y)},formPushButton:function formPushButton(i,o,u,p,g){var y=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(i,"pushButton",o,u,p,g,y)},formCombo:function formCombo(i,o,u,p,g){var y=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(i,"combo",o,u,p,g,y)},formList:function formList(i,o,u,p,g){var y=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(i,"list",o,u,p,g,y)},formRadioButton:function formRadioButton(i,o,u,p,g){var y=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(i,"radioButton",o,u,p,g,y)},formCheckbox:function formCheckbox(i,o,u,p,g){var y=arguments.length>5&&void 0!==arguments[5]?arguments[5]:{};return this.formAnnotation(i,"checkbox",o,u,p,g,y)},_addToParent:function _addToParent(i){var o=i.data.Parent;return o?(o.data.Kids||(o.data.Kids=[]),o.data.Kids.push(i)):this._root.data.AcroForm.data.Fields.push(i),this},_fieldDict:function _fieldDict(i,o){var u=arguments.length>2&&void 0!==arguments[2]?arguments[2]:{};if(!this._acroform)throw new Error("Call document.initForms() method before adding form elements to document");var p=Object.assign({},u);return null!==o&&(p=this._resolveType(o,u)),p=this._resolveFlags(p),p=this._resolveJustify(p),p=this._resolveFont(p),p=this._resolveStrings(p),p=this._resolveColors(p),(p=this._resolveFormat(p)).T=new String(i),p.parent&&(p.Parent=p.parent,delete p.parent),p},_resolveType:function _resolveType(i,o){if("text"===i)o.FT="Tx";else if("pushButton"===i)o.FT="Btn",o.pushButton=!0;else if("radioButton"===i)o.FT="Btn",o.radioButton=!0;else if("checkbox"===i)o.FT="Btn";else if("combo"===i)o.FT="Ch",o.combo=!0;else{if("list"!==i)throw new Error("Invalid form annotation type '".concat(i,"'"));o.FT="Ch"}return o},_resolveFormat:function _resolveFormat(i){var o=i.format;if(o&&o.type){var u,p,g="";if(void 0!==pe[o.type])u="AFSpecial_Keystroke",p="AFSpecial_Format",g=pe[o.type];else{var y=o.type.charAt(0).toUpperCase()+o.type.slice(1);if(u="AF".concat(y,"_Keystroke"),p="AF".concat(y,"_Format"),"date"===o.type)u+="Ex",g=String(o.param);else if("time"===o.type)g=String(o.param);else if("number"===o.type){var w=Object.assign({},ge,o);g=String([String(w.nDec),w.sepComma?"0":"1",'"'+w.negStyle+'"',"null",'"'+w.currency+'"',String(w.currencyPrepend)].join(","))}else if("percent"===o.type){var _=Object.assign({},ye,o);g=String([String(_.nDec),_.sepComma?"0":"1"].join(","))}}i.AA=i.AA?i.AA:{},i.AA.K={S:"JavaScript",JS:new String("".concat(u,"(").concat(g,");"))},i.AA.F={S:"JavaScript",JS:new String("".concat(p,"(").concat(g,");"))}}return delete i.format,i},_resolveColors:function _resolveColors(i){var o=this._normalizeColor(i.backgroundColor);return o&&(i.MK||(i.MK={}),i.MK.BG=o),(o=this._normalizeColor(i.borderColor))&&(i.MK||(i.MK={}),i.MK.BC=o),delete i.backgroundColor,delete i.borderColor,i},_resolveFlags:function _resolveFlags(i){var o=0;return Object.keys(i).forEach((function(u){fe[u]&&(o|=fe[u],delete i[u])})),0!==o&&(i.Ff=i.Ff?i.Ff:0,i.Ff|=o),i},_resolveJustify:function _resolveJustify(i){var o=0;return void 0!==i.align&&("number"==typeof he[i.align]&&(o=he[i.align]),delete i.align),0!==o&&(i.Q=o),i},_resolveFont:function _resolveFont(i){return null===this._acroform.fonts[this._font.id]&&(this._acroform.fonts[this._font.id]=this._font.ref()),this._acroform.defaultFont!==this._font.name&&(i.DR={Font:{}},i.DR.Font[this._font.id]=this._font.ref(),i.DA=new String("/".concat(this._font.id," 0 Tf 0 g"))),i},_resolveStrings:function _resolveStrings(i){var o=[];function appendChoices(i){if(Array.isArray(i))for(var u=0;u<i.length;u++)"string"==typeof i[u]?o.push(new String(i[u])):o.push(i[u])}return appendChoices(i.Opt),i.select&&(appendChoices(i.select),delete i.select),o.length&&(i.Opt=o),Object.keys(de).forEach((function(o){void 0!==i[o]&&(i[de[o]]=i[o],delete i[o])})),["V","DV"].forEach((function(o){"string"==typeof i[o]&&(i[o]=new String(i[o]))})),i.MK&&i.MK.CA&&(i.MK.CA=new String(i.MK.CA)),i.label&&(i.MK=i.MK?i.MK:{},i.MK.CA=new String(i.label),delete i.label),i}},me={file:function file(o){var u=arguments.length>1&&void 0!==arguments[1]?arguments[1]:{};u.name=u.name||o;var p,g={Type:"EmbeddedFile",Params:{}};if(!o)throw new Error("No src specified");if(i.isBuffer(o))p=o;else if(o instanceof ArrayBuffer)p=i.from(new Uint8Array(o));else{var y;if(y=/^data:(.*);base64,(.*)$/.exec(o))y[1]&&(g.Subtype=y[1].replace("/","#2F")),p=i.from(y[2],"base64");else{if(!(p=E.readFileSync(o)))throw new Error("Could not read contents of file at filepath ".concat(o));var _=E.statSync(o),x=_.birthtime,k=_.ctime;g.Params.CreationDate=x,g.Params.ModDate=k}}u.creationDate instanceof Date&&(g.Params.CreationDate=u.creationDate),u.modifiedDate instanceof Date&&(g.Params.ModDate=u.modifiedDate),u.type&&(g.Subtype=u.type.replace("/","#2F"));var P,O=w.default.MD5(w.default.lib.WordArray.create(new Uint8Array(p)));g.Params.CheckSum=new String(O),g.Params.Size=p.byteLength,this._fileRegistry||(this._fileRegistry={});var file=this._fileRegistry[u.name];file&&isEqual(g,file)?P=file.ref:((P=this.ref(g)).end(p),this._fileRegistry[u.name]=_objectSpread2(_objectSpread2({},g),{},{ref:P}));var I={Type:"Filespec",F:new String(u.name),EF:{F:P},UF:new String(u.name)};u.description&&(I.Desc=new String(u.description));var B=this.ref(I);return B.end(),u.hidden||this.addNamedEmbeddedFile(u.name,B),B}};function isEqual(i,o){return i.Subtype===o.Subtype&&i.Params.CheckSum.toString()===o.Params.CheckSum.toString()&&i.Params.Size===o.Params.Size&&i.Params.CreationDate===o.Params.CreationDate&&i.Params.ModDate===o.Params.ModDate}var be=function(o){_inherits(PDFDocument,o);var u=_createSuper(PDFDocument);function PDFDocument(){var i,o=arguments.length>0&&void 0!==arguments[0]?arguments[0]:{};switch(_classCallCheck(this,PDFDocument),(i=u.call(this,o)).options=o,o.pdfVersion){case"1.4":i.version=1.4;break;case"1.5":i.version=1.5;break;case"1.6":i.version=1.6;break;case"1.7":case"1.7ext3":i.version=1.7;break;default:i.version=1.3}i.compress=null==i.options.compress||i.options.compress,i._pageBuffer=[],i._pageBufferStart=0,i._offsets=[],i._waiting=0,i._ended=!1,i._offset=0;var p=i.ref({Type:"Pages",Count:0,Kids:[]}),g=i.ref({Dests:new K});if(i._root=i.ref({Type:"Catalog",Pages:p,Names:g}),i.options.lang&&(i._root.data.Lang=new String(i.options.lang)),i.page=null,i.initColor(),i.initVector(),i.initFonts(o.font),i.initText(),i.initImages(),i.initOutline(),i.initMarkings(o),i.info={Producer:"PDFKit",Creator:"PDFKit",CreationDate:new Date},i.options.info)for(var y in i.options.info){var w=i.options.info[y];i.info[y]=w}return i.options.displayTitle&&(i._root.data.ViewerPreferences=i.ref({DisplayDocTitle:!0})),i._id=pt.generateFileID(i.info),i._security=pt.create(_assertThisInitialized(i),o),i._write("%PDF-".concat(i.version)),i._write("%ÿÿÿÿ"),!1!==i.options.autoFirstPage&&i.addPage(),i}return _createClass(PDFDocument,[{key:"addPage",value:function addPage(i){null==i&&(i=this.options),this.options.bufferPages||this.flushPages(),this.page=new X(this,i),this._pageBuffer.push(this.page);var o=this._root.data.Pages.data;return o.Kids.push(this.page.dictionary),o.Count++,this.x=this.page.margins.left,this.y=this.page.margins.top,this._ctm=[1,0,0,1,0,0],this.transform(1,0,0,-1,0,this.page.height),this.emit("pageAdded"),this}},{key:"continueOnNewPage",value:function continueOnNewPage(i){var o=this.endPageMarkings(this.page);return this.addPage(i),this.initPageMarkings(o),this}},{key:"bufferedPageRange",value:function bufferedPageRange(){return{start:this._pageBufferStart,count:this._pageBuffer.length}}},{key:"switchToPage",value:function switchToPage(i){var o;if(!(o=this._pageBuffer[i-this._pageBufferStart]))throw new Error("switchToPage(".concat(i,") out of bounds, current buffer covers pages ").concat(this._pageBufferStart," to ").concat(this._pageBufferStart+this._pageBuffer.length-1));return this.page=o}},{key:"flushPages",value:function flushPages(){var i=this._pageBuffer;this._pageBuffer=[],this._pageBufferStart+=i.length;var o,u=_createForOfIteratorHelper(i);try{for(u.s();!(o=u.n()).done;){var p=o.value;this.endPageMarkings(p),p.end()}}catch(i){u.e(i)}finally{u.f()}}},{key:"addNamedDestination",value:function addNamedDestination(i){for(var o=arguments.length,u=new Array(o>1?o-1:0),p=1;p<o;p++)u[p-1]=arguments[p];0===u.length&&(u=["XYZ",null,null,null]),"XYZ"===u[0]&&null!==u[2]&&(u[2]=this.page.height-u[2]),u.unshift(this.page.dictionary),this._root.data.Names.data.Dests.add(i,u)}},{key:"addNamedEmbeddedFile",value:function addNamedEmbeddedFile(i,o){this._root.data.Names.data.EmbeddedFiles||(this._root.data.Names.data.EmbeddedFiles=new K({limits:!1})),this._root.data.Names.data.EmbeddedFiles.add(i,o)}},{key:"addNamedJavaScript",value:function addNamedJavaScript(i,o){this._root.data.Names.data.JavaScript||(this._root.data.Names.data.JavaScript=new K);var u={JS:new String(o),S:"JavaScript"};this._root.data.Names.data.JavaScript.add(i,u)}},{key:"ref",value:function ref(i){var ref=new W(this,this._offsets.length+1,i);return this._offsets.push(null),this._waiting++,ref}},{key:"_read",value:function _read(){}},{key:"_write",value:function _write(o){return i.isBuffer(o)||(o=i.from(o+"\n","binary")),this.push(o),this._offset+=o.length}},{key:"addContent",value:function addContent(i){return this.page.write(i),this}},{key:"_refEnd",value:function _refEnd(i){if(this._offsets[i.id-1]=i.offset,0==--this._waiting&&this._ended)return this._finalize(),this._ended=!1}},{key:"write",value:function write(i,o){new Error("PDFDocument#write is deprecated, and will be removed in a future version of PDFKit. Please pipe the document into a Node stream.");return this.pipe(E.createWriteStream(i)),this.end(),this.once("end",o)}},{key:"end",value:function end(){for(var i in this.flushPages(),this._info=this.ref(),this.info){var o=this.info[i];"string"==typeof o&&(o=new String(o));var u=this.ref(o);u.end(),this._info.data[i]=u}for(var p in this._info.end(),this._fontFamilies){this._fontFamilies[p].finalize()}return this.endOutline(),this.endMarkings(),this._root.end(),this._root.data.Pages.end(),this._root.data.Names.end(),this.endAcroForm(),this._root.data.ViewerPreferences&&this._root.data.ViewerPreferences.end(),this._security&&this._security.end(),0===this._waiting?this._finalize():this._ended=!0}},{key:"_finalize",value:function _finalize(){var i=this._offset;this._write("xref"),this._write("0 ".concat(this._offsets.length+1)),this._write("0000000000 65535 f ");var o,u=_createForOfIteratorHelper(this._offsets);try{for(u.s();!(o=u.n()).done;){var p=o.value;p="0000000000".concat(p).slice(-10),this._write(p+" 00000 n ")}}catch(i){u.e(i)}finally{u.f()}var g={Size:this._offsets.length+1,Root:this._root,Info:this._info,ID:[this._id,this._id]};return this._security&&(g.Encrypt=this._security.dictionary),this._write("trailer"),this._write(U.convert(g)),this._write("startxref"),this._write("".concat(i)),this._write("%%EOF"),this.push(null)}},{key:"toString",value:function toString(){return"[object PDFDocument]"}}]),PDFDocument}(g.default.Readable),we=function mixin(i){Object.assign(be.prototype,i)};we(Tt),we(Nt),we(Zt),we(Jt),we(ne),we(ie),we(ae),we(ue),we(ve),we(me),be.LineWrapper=Kt;var _e=be;o.default=_e}).call(this,u(10).Buffer,"/")},function(i,o,u){var p=u(4),g=u(142),y=p.WeakMap;i.exports="function"==typeof y&&/native code/.test(g(y))},function(i,o,u){var p=u(13),g=u(16),y=u(12),w=u(91);i.exports=p?Object.defineProperties:function defineProperties(i,o){y(i);for(var u,p=w(o),_=p.length,x=0;_>x;)g.f(i,u=p[x++],o[u]);return i}},function(i,o,u){var p=u(21),g=u(57).f,y={}.toString,w="object"==typeof window&&window&&Object.getOwnPropertyNames?Object.getOwnPropertyNames(window):[];i.exports.f=function getOwnPropertyNames(i){return w&&"[object Window]"==y.call(i)?function(i){try{return g(i)}catch(i){return w.slice()}}(i):g(p(i))}},function(i,o,u){"use strict";var p=u(1),g=u(22).filter;p({target:"Array",proto:!0,forced:!u(117)("filter")},{filter:function filter(i){return g(this,i,arguments.length>1?arguments[1]:void 0)}})},function(i,o,u){var p=u(1),g=u(3),y=u(21),w=u(34).f,_=u(13),x=g((function(){w(1)}));p({target:"Object",stat:!0,forced:!_||x,sham:!_},{getOwnPropertyDescriptor:function getOwnPropertyDescriptor(i,o){return w(y(i),o)}})},function(i,o,u){var p=u(4),g=u(213),y=u(316),w=u(18);for(var _ in g){var x=p[_],k=x&&x.prototype;if(k&&k.forEach!==y)try{w(k,"forEach",y)}catch(i){k.forEach=y}}},function(i,o,u){"use strict";var p=u(22).forEach,g=u(149)("forEach");i.exports=g?[].forEach:function forEach(i){return p(this,i,arguments.length>1?arguments[1]:void 0)}},function(i,o,u){var p=u(1),g=u(13),y=u(205),w=u(21),_=u(34),x=u(118);p({target:"Object",stat:!0,sham:!g},{getOwnPropertyDescriptors:function getOwnPropertyDescriptors(i){for(var o,u,p=w(i),g=_.f,k=y(p),P={},E=0;k.length>E;)void 0!==(u=g(p,o=k[E++]))&&x(P,o,u);return P}})},function(i,o,u){u(1)({target:"Object",stat:!0},{setPrototypeOf:u(46)})},function(i,o,u){var p=u(11);i.exports=function(i){if(!p(i)&&null!==i)throw TypeError("Can't set "+String(i)+" as a prototype");return i}},function(i,o,u){var p=u(1),g=u(3),y=u(19),w=u(58),_=u(214);p({target:"Object",stat:!0,forced:g((function(){w(1)})),sham:!_},{getPrototypeOf:function getPrototypeOf(i){return w(y(i))}})},function(i,o,u){var p=u(1),g=u(30),y=u(31),w=u(12),_=u(11),x=u(45),k=u(322),P=u(3),E=g("Reflect","construct"),O=P((function(){function F(){}return!(E((function(){}),[],F)instanceof F)})),I=!P((function(){E((function(){}))})),B=O||I;p({target:"Reflect",stat:!0,forced:B,sham:B},{construct:function construct(i,o){y(i),w(o);var u=arguments.length<3?i:y(arguments[2]);if(I&&!O)return E(i,o,u);if(i==u){switch(o.length){case 0:return new i;case 1:return new i(o[0]);case 2:return new i(o[0],o[1]);case 3:return new i(o[0],o[1],o[2]);case 4:return new i(o[0],o[1],o[2],o[3])}var p=[null];return p.push.apply(p,o),new(k.apply(i,p))}var g=u.prototype,P=x(_(g)?g:Object.prototype),B=Function.apply.call(i,P,o);return _(B)?B:P}})},function(i,o,u){"use strict";var p=u(31),g=u(11),y=[].slice,w={},construct=function(i,o,u){if(!(o in w)){for(var p=[],g=0;g<o;g++)p[g]="a["+g+"]";w[o]=Function("C,a","return new C("+p.join(",")+")")}return w[o](i,u)};i.exports=Function.bind||function bind(i){var o=p(this),u=y.call(arguments,1),w=function bound(){var p=u.concat(y.call(arguments));return this instanceof w?construct(o,p.length,p):o.apply(i,p)};return g(o.prototype)&&(w.prototype=o.prototype),w}},function(i,o,u){"use strict";var p=u(150),g=u(119);i.exports=p?{}.toString:function toString(){return"[object "+g(this)+"]"}},function(i,o,u){"use strict";var p=u(218).IteratorPrototype,g=u(45),y=u(41),w=u(92),_=u(94),returnThis=function(){return this};i.exports=function(i,o,u){var x=o+" Iterator";return i.prototype=g(p,{next:y(1,u)}),w(i,x,!1,!0),_[x]=returnThis,i}},function(i,o,u){"use strict";var p=u(93),g=u(19),y=u(326),w=u(154),_=u(8),x=u(118),k=u(155);i.exports=function from(i){var o,u,P,E,O,I,B=g(i),D="function"==typeof this?this:Array,R=arguments.length,N=R>1?arguments[1]:void 0,U=void 0!==N,W=k(B),G=0;if(U&&(N=p(N,R>2?arguments[2]:void 0,2)),null==W||D==Array&&w(W))for(u=new D(o=_(B.length));o>G;G++)I=U?N(B[G],G):B[G],x(u,G,I);else for(O=(E=W.call(B)).next,u=new D;!(P=O.call(E)).done;G++)I=U?y(E,N,[P.value,G],!0):P.value,x(u,G,I);return u.length=G,u}},function(i,o,u){var p=u(12),g=u(220);i.exports=function(i,o,u,y){try{return y?o(p(u)[0],u[1]):o(u)}catch(o){throw g(i),o}}},function(i,o,u){"use strict";var p=u(25),g=u(12),y=u(3),w=u(223),_=RegExp.prototype,x=_.toString,k=y((function(){return"/a/b"!=x.call({source:"a",flags:"b"})})),P="toString"!=x.name;(k||P)&&p(RegExp.prototype,"toString",(function toString(){var i=g(this),o=String(i.source),u=i.flags;return"/"+o+"/"+String(void 0===u&&i instanceof RegExp&&!("flags"in _)?w.call(i):u)}),{unsafe:!0})},function(i,o,u){"use strict";var p=u(3);function RE(i,o){return RegExp(i,o)}o.UNSUPPORTED_Y=p((function(){var i=RE("a","y");return i.lastIndex=2,null!=i.exec("abcd")})),o.BROKEN_CARET=p((function(){var i=RE("^r","gy");return i.lastIndex=2,null!=i.exec("str")}))},function(i,o,u){var p=u(19),g=Math.floor,y="".replace,w=/\$([$&'`]|\d{1,2}|<[^>]*>)/g,_=/\$([$&'`]|\d{1,2})/g;i.exports=function(i,o,u,x,k,P){var E=u+i.length,O=x.length,I=_;return void 0!==k&&(k=p(k),I=w),y.call(P,I,(function(p,y){var w;switch(y.charAt(0)){case"$":return"$";case"&":return i;case"`":return o.slice(0,u);case"'":return o.slice(E);case"<":w=k[y.slice(1,-1)];break;default:var _=+y;if(0===_)return p;if(_>O){var P=g(_/10);return 0===P?p:P<=O?void 0===x[P-1]?y.charAt(1):x[P-1]+y.charAt(1):p}w=x[_-1]}return void 0===w?"":w}))}},function(i,o,u){"use strict";var p=u(1),g=u(22).map;p({target:"Array",proto:!0,forced:!u(117)("map")},{map:function map(i){return g(this,i,arguments.length>1?arguments[1]:void 0)}})},function(i,o,u){"use strict";var p=u(1),g=u(152).codeAt;p({target:"String",proto:!0},{codePointAt:function codePointAt(i){return g(this,i)}})},function(i,o,u){var p=u(1),g=u(44),y=String.fromCharCode,w=String.fromCodePoint;p({target:"String",stat:!0,forced:!!w&&1!=w.length},{fromCodePoint:function fromCodePoint(i){for(var o,u=[],p=arguments.length,w=0;p>w;){if(o=+arguments[w++],g(o,1114111)!==o)throw RangeError(o+" is not a valid code point");u.push(o<65536?y(o):y(55296+((o-=65536)>>10),o%1024+56320))}return u.join("")}})},function(i,o,u){u(1)({target:"Number",stat:!0},{isFinite:u(334)})},function(i,o,u){var p=u(4).isFinite;i.exports=Number.isFinite||function isFinite(i){return"number"==typeof i&&p(i)}},function(i,o,u){var p=u(11),g=u(42),y=u(6)("match");i.exports=function(i){var o;return p(i)&&(void 0!==(o=i[y])?!!o:"RegExp"==g(i))}},function(i,o,u){"use strict";var p=u(1),g=u(162);p({target:"String",proto:!0,forced:u(163)("italics")},{italics:function italics(){return g(this,"i","","")}})},function(i,o,u){var p=u(4),g=u(3),y=u(156),w=u(5).NATIVE_ARRAY_BUFFER_VIEWS,_=p.ArrayBuffer,x=p.Int8Array;i.exports=!w||!g((function(){x(1)}))||!g((function(){new x(-1)}))||!y((function(i){new x,new x(null),new x(1.5),new x(i)}),!0)||g((function(){return 1!==new x(new _(2),1,void 0).length}))},function(i,o){var u=Math.abs,p=Math.pow,g=Math.floor,y=Math.log,w=Math.LN2;i.exports={pack:function(i,o,_){var x,k,P,E=new Array(_),O=8*_-o-1,I=(1<<O)-1,B=I>>1,D=23===o?p(2,-24)-p(2,-77):0,R=i<0||0===i&&1/i<0?1:0,N=0;for((i=u(i))!=i||i===1/0?(k=i!=i?1:0,x=I):(x=g(y(i)/w),i*(P=p(2,-x))<1&&(x--,P*=2),(i+=x+B>=1?D/P:D*p(2,1-B))*P>=2&&(x++,P/=2),x+B>=I?(k=0,x=I):x+B>=1?(k=(i*P-1)*p(2,o),x+=B):(k=i*p(2,B-1)*p(2,o),x=0));o>=8;E[N++]=255&k,k/=256,o-=8);for(x=x<<o|k,O+=o;O>0;E[N++]=255&x,x/=256,O-=8);return E[--N]|=128*R,E},unpack:function(i,o){var u,g=i.length,y=8*g-o-1,w=(1<<y)-1,_=w>>1,x=y-7,k=g-1,P=i[k--],E=127&P;for(P>>=7;x>0;E=256*E+i[k],k--,x-=8);for(u=E&(1<<-x)-1,E>>=-x,x+=o;x>0;u=256*u+i[k],k--,x-=8);if(0===E)E=1-_;else{if(E===w)return u?NaN:P?-1/0:1/0;u+=p(2,o),E-=_}return(P?-1:1)*u*p(2,E-o)}}},function(i,o,u){var p=u(36);i.exports=function(i){var o=p(i);if(o<0)throw RangeError("The argument can't be less than 0");return o}},function(i,o,u){var p=u(19),g=u(8),y=u(155),w=u(154),_=u(93),x=u(5).aTypedArrayConstructor;i.exports=function from(i){var o,u,k,P,E,O,I=p(i),B=arguments.length,D=B>1?arguments[1]:void 0,R=void 0!==D,N=y(I);if(null!=N&&!w(N))for(O=(E=N.call(I)).next,I=[];!(P=O.call(E)).done;)I.push(P.value);for(R&&B>2&&(D=_(D,arguments[2],2)),u=g(I.length),k=new(x(this))(u),o=0;u>o;o++)k[o]=R?D(I[o],o):I[o];return k}},function(i,o,u){"use strict";var p=u(19),g=u(44),y=u(8),w=Math.min;i.exports=[].copyWithin||function copyWithin(i,o){var u=p(this),_=y(u.length),x=g(i,_),k=g(o,_),P=arguments.length>2?arguments[2]:void 0,E=w((void 0===P?_:g(P,_))-k,_-x),O=1;for(k<x&&x<k+E&&(O=-1,k+=E-1,x+=E-1);E-- >0;)k in u?u[x]=u[k]:delete u[x],x+=O,k+=O;return u}},function(i,o,u){var p=u(5).aTypedArrayConstructor,g=u(37);i.exports=function(i,o){for(var u=g(i,i.constructor),y=0,w=o.length,_=new(p(u))(w);w>y;)_[y]=o[y++];return _}},function(i,o,u){"use strict";var p=u(21),g=u(36),y=u(8),w=u(149),_=Math.min,x=[].lastIndexOf,k=!!x&&1/[1].lastIndexOf(1,-0)<0,P=w("lastIndexOf"),E=k||!P;i.exports=E?function lastIndexOf(i){if(k)return x.apply(this,arguments)||0;var o=p(this),u=y(o.length),w=u-1;for(arguments.length>1&&(w=_(w,g(arguments[1]))),w<0&&(w=u+w);w>=0;w--)if(w in o&&o[w]===i)return w||0;return-1}:x},function(i,o,u){"use strict";var p=u(1),g=u(4),y=u(164),w=u(167),_=y.ArrayBuffer;p({global:!0,forced:g.ArrayBuffer!==_},{ArrayBuffer:_}),w("ArrayBuffer")},function(i,o,u){var p=u(1),g=u(346);p({target:"Object",stat:!0,forced:Object.assign!==g},{assign:g})},function(i,o,u){"use strict";var p=u(13),g=u(3),y=u(91),w=u(145),_=u(107),x=u(19),k=u(90),P=Object.assign,E=Object.defineProperty;i.exports=!P||g((function(){if(p&&1!==P({b:1},P(E({},"a",{enumerable:!0,get:function(){E(this,"b",{value:3,enumerable:!1})}}),{b:2})).b)return!0;var i={},o={},u=Symbol();return i[u]=7,"abcdefghijklmnopqrst".split("").forEach((function(i){o[i]=i})),7!=P({},i)[u]||"abcdefghijklmnopqrst"!=y(P({},o)).join("")}))?function assign(i,o){for(var u=x(i),g=arguments.length,P=1,E=w.f,O=_.f;g>P;)for(var I,B=k(arguments[P++]),D=E?y(B).concat(E(B)):y(B),R=D.length,N=0;R>N;)I=D[N++],p&&!O.call(B,I)||(u[I]=B[I]);return u}:P},function(i,o,u){var p=u(3),g=u(229);i.exports=function(i){return p((function(){return!!g[i]()||"​᠎"!="​᠎"[i]()||g[i].name!==i}))}},function(i,o,u){"use strict";var p=u(1),g=u(162);p({target:"String",proto:!0,forced:u(163)("strike")},{strike:function strike(){return g(this,"strike","","")}})},function(i,o){},function(i,o,u){"use strict";var p=u(126).Buffer,g=u(351);i.exports=function(){function BufferList(){!function _classCallCheck(i,o){if(!(i instanceof o))throw new TypeError("Cannot call a class as a function")}(this,BufferList),this.head=null,this.tail=null,this.length=0}return BufferList.prototype.push=function push(i){var o={data:i,next:null};this.length>0?this.tail.next=o:this.head=o,this.tail=o,++this.length},BufferList.prototype.unshift=function unshift(i){var o={data:i,next:this.head};0===this.length&&(this.tail=o),this.head=o,++this.length},BufferList.prototype.shift=function shift(){if(0!==this.length){var i=this.head.data;return 1===this.length?this.head=this.tail=null:this.head=this.head.next,--this.length,i}},BufferList.prototype.clear=function clear(){this.head=this.tail=null,this.length=0},BufferList.prototype.join=function join(i){if(0===this.length)return"";for(var o=this.head,u=""+o.data;o=o.next;)u+=i+o.data;return u},BufferList.prototype.concat=function concat(i){if(0===this.length)return p.alloc(0);if(1===this.length)return this.head.data;for(var o,u,g,y=p.allocUnsafe(i>>>0),w=this.head,_=0;w;)o=w.data,u=y,g=_,o.copy(u,g),_+=w.data.length,w=w.next;return y},BufferList}(),g&&g.inspect&&g.inspect.custom&&(i.exports.prototype[g.inspect.custom]=function(){var i=g.inspect({length:this.length});return this.constructor.name+" "+i})},function(i,o){},function(i,o,u){(function(o){function config(i){try{if(!o.localStorage)return!1}catch(i){return!1}var u=o.localStorage[i];return null!=u&&"true"===String(u).toLowerCase()}i.exports=function deprecate(i,o){if(config("noDeprecation"))return i;var u=!1;return function deprecated(){if(!u){if(config("throwDeprecation"))throw new Error(o);config("traceDeprecation"),u=!0}return i.apply(this,arguments)}}}).call(this,u(27))},function(i,o,u){"use strict";i.exports=PassThrough;var p=u(243),g=Object.create(u(95));function PassThrough(i){if(!(this instanceof PassThrough))return new PassThrough(i);p.call(this,i)}g.inherits=u(84),g.inherits(PassThrough,p),PassThrough.prototype._transform=function(i,o,u){u(null,i)}},function(i,o,u){i.exports=u(169)},function(i,o,u){i.exports=u(49)},function(i,o,u){i.exports=u(168).Transform},function(i,o,u){i.exports=u(168).PassThrough},function(i,o,u){"use strict";(function(i,p){var g=u(245),y=u(362),w=u(363),_=u(366),x=u(369);for(var k in x)o[k]=x[k];o.NONE=0,o.DEFLATE=1,o.INFLATE=2,o.GZIP=3,o.GUNZIP=4,o.DEFLATERAW=5,o.INFLATERAW=6,o.UNZIP=7;function Zlib(i){if("number"!=typeof i||i<o.DEFLATE||i>o.UNZIP)throw new TypeError("Bad argument");this.dictionary=null,this.err=0,this.flush=0,this.init_done=!1,this.level=0,this.memLevel=0,this.mode=i,this.strategy=0,this.windowBits=0,this.write_in_progress=!1,this.pending_close=!1,this.gzip_id_bytes_read=0}Zlib.prototype.close=function(){this.write_in_progress?this.pending_close=!0:(this.pending_close=!1,g(this.init_done,"close before init"),g(this.mode<=o.UNZIP),this.mode===o.DEFLATE||this.mode===o.GZIP||this.mode===o.DEFLATERAW?w.deflateEnd(this.strm):this.mode!==o.INFLATE&&this.mode!==o.GUNZIP&&this.mode!==o.INFLATERAW&&this.mode!==o.UNZIP||_.inflateEnd(this.strm),this.mode=o.NONE,this.dictionary=null)},Zlib.prototype.write=function(i,o,u,p,g,y,w){return this._write(!0,i,o,u,p,g,y,w)},Zlib.prototype.writeSync=function(i,o,u,p,g,y,w){return this._write(!1,i,o,u,p,g,y,w)},Zlib.prototype._write=function(u,y,w,_,x,k,P,E){if(g.equal(arguments.length,8),g(this.init_done,"write before init"),g(this.mode!==o.NONE,"already finalized"),g.equal(!1,this.write_in_progress,"write already in progress"),g.equal(!1,this.pending_close,"close is pending"),this.write_in_progress=!0,g.equal(!1,void 0===y,"must provide flush value"),this.write_in_progress=!0,y!==o.Z_NO_FLUSH&&y!==o.Z_PARTIAL_FLUSH&&y!==o.Z_SYNC_FLUSH&&y!==o.Z_FULL_FLUSH&&y!==o.Z_FINISH&&y!==o.Z_BLOCK)throw new Error("Invalid flush value");if(null==w&&(w=i.alloc(0),x=0,_=0),this.strm.avail_in=x,this.strm.input=w,this.strm.next_in=_,this.strm.avail_out=E,this.strm.output=k,this.strm.next_out=P,this.flush=y,!u)return this._process(),this._checkError()?this._afterSync():void 0;var O=this;return p.nextTick((function(){O._process(),O._after()})),this},Zlib.prototype._afterSync=function(){var i=this.strm.avail_out,o=this.strm.avail_in;return this.write_in_progress=!1,[o,i]},Zlib.prototype._process=function(){var i=null;switch(this.mode){case o.DEFLATE:case o.GZIP:case o.DEFLATERAW:this.err=w.deflate(this.strm,this.flush);break;case o.UNZIP:switch(this.strm.avail_in>0&&(i=this.strm.next_in),this.gzip_id_bytes_read){case 0:if(null===i)break;if(31!==this.strm.input[i]){this.mode=o.INFLATE;break}if(this.gzip_id_bytes_read=1,i++,1===this.strm.avail_in)break;case 1:if(null===i)break;139===this.strm.input[i]?(this.gzip_id_bytes_read=2,this.mode=o.GUNZIP):this.mode=o.INFLATE;break;default:throw new Error("invalid number of gzip magic number bytes read")}case o.INFLATE:case o.GUNZIP:case o.INFLATERAW:for(this.err=_.inflate(this.strm,this.flush),this.err===o.Z_NEED_DICT&&this.dictionary&&(this.err=_.inflateSetDictionary(this.strm,this.dictionary),this.err===o.Z_OK?this.err=_.inflate(this.strm,this.flush):this.err===o.Z_DATA_ERROR&&(this.err=o.Z_NEED_DICT));this.strm.avail_in>0&&this.mode===o.GUNZIP&&this.err===o.Z_STREAM_END&&0!==this.strm.next_in[0];)this.reset(),this.err=_.inflate(this.strm,this.flush);break;default:throw new Error("Unknown mode "+this.mode)}},Zlib.prototype._checkError=function(){switch(this.err){case o.Z_OK:case o.Z_BUF_ERROR:if(0!==this.strm.avail_out&&this.flush===o.Z_FINISH)return this._error("unexpected end of file"),!1;break;case o.Z_STREAM_END:break;case o.Z_NEED_DICT:return null==this.dictionary?this._error("Missing dictionary"):this._error("Bad dictionary"),!1;default:return this._error("Zlib error"),!1}return!0},Zlib.prototype._after=function(){if(this._checkError()){var i=this.strm.avail_out,o=this.strm.avail_in;this.write_in_progress=!1,this.callback(o,i),this.pending_close&&this.close()}},Zlib.prototype._error=function(i){this.strm.msg&&(i=this.strm.msg),this.onerror(i,this.err),this.write_in_progress=!1,this.pending_close&&this.close()},Zlib.prototype.init=function(i,u,p,y,w){g(4===arguments.length||5===arguments.length,"init(windowBits, level, memLevel, strategy, [dictionary])"),g(i>=8&&i<=15,"invalid windowBits"),g(u>=-1&&u<=9,"invalid compression level"),g(p>=1&&p<=9,"invalid memlevel"),g(y===o.Z_FILTERED||y===o.Z_HUFFMAN_ONLY||y===o.Z_RLE||y===o.Z_FIXED||y===o.Z_DEFAULT_STRATEGY,"invalid strategy"),this._init(u,i,p,y,w),this._setDictionary()},Zlib.prototype.params=function(){throw new Error("deflateParams Not supported")},Zlib.prototype.reset=function(){this._reset(),this._setDictionary()},Zlib.prototype._init=function(i,u,p,g,x){switch(this.level=i,this.windowBits=u,this.memLevel=p,this.strategy=g,this.flush=o.Z_NO_FLUSH,this.err=o.Z_OK,this.mode!==o.GZIP&&this.mode!==o.GUNZIP||(this.windowBits+=16),this.mode===o.UNZIP&&(this.windowBits+=32),this.mode!==o.DEFLATERAW&&this.mode!==o.INFLATERAW||(this.windowBits=-1*this.windowBits),this.strm=new y,this.mode){case o.DEFLATE:case o.GZIP:case o.DEFLATERAW:this.err=w.deflateInit2(this.strm,this.level,o.Z_DEFLATED,this.windowBits,this.memLevel,this.strategy);break;case o.INFLATE:case o.GUNZIP:case o.INFLATERAW:case o.UNZIP:this.err=_.inflateInit2(this.strm,this.windowBits);break;default:throw new Error("Unknown mode "+this.mode)}this.err!==o.Z_OK&&this._error("Init error"),this.dictionary=x,this.write_in_progress=!1,this.init_done=!0},Zlib.prototype._setDictionary=function(){if(null!=this.dictionary){switch(this.err=o.Z_OK,this.mode){case o.DEFLATE:case o.DEFLATERAW:this.err=w.deflateSetDictionary(this.strm,this.dictionary)}this.err!==o.Z_OK&&this._error("Failed to set dictionary")}},Zlib.prototype._reset=function(){switch(this.err=o.Z_OK,this.mode){case o.DEFLATE:case o.DEFLATERAW:case o.GZIP:this.err=w.deflateReset(this.strm);break;case o.INFLATE:case o.INFLATERAW:case o.GUNZIP:this.err=_.inflateReset(this.strm)}this.err!==o.Z_OK&&this._error("Failed to reset stream")},o.Zlib=Zlib}).call(this,u(10).Buffer,u(48))},function(i,o,u){"use strict";var p=Object.getOwnPropertySymbols,g=Object.prototype.hasOwnProperty,y=Object.prototype.propertyIsEnumerable;function toObject(i){if(null==i)throw new TypeError("Object.assign cannot be called with null or undefined");return Object(i)}i.exports=function shouldUseNative(){try{if(!Object.assign)return!1;var i=new String("abc");if(i[5]="de","5"===Object.getOwnPropertyNames(i)[0])return!1;for(var o={},u=0;u<10;u++)o["_"+String.fromCharCode(u)]=u;if("0123456789"!==Object.getOwnPropertyNames(o).map((function(i){return o[i]})).join(""))return!1;var p={};return"abcdefghijklmnopqrst".split("").forEach((function(i){p[i]=i})),"abcdefghijklmnopqrst"===Object.keys(Object.assign({},p)).join("")}catch(i){return!1}}()?Object.assign:function(i,o){for(var u,w,_=toObject(i),x=1;x<arguments.length;x++){for(var k in u=Object(arguments[x]))g.call(u,k)&&(_[k]=u[k]);if(p){w=p(u);for(var P=0;P<w.length;P++)y.call(u,w[P])&&(_[w[P]]=u[w[P]])}}return _}},function(i,o){i.exports=function isBuffer(i){return i&&"object"==typeof i&&"function"==typeof i.copy&&"function"==typeof i.fill&&"function"==typeof i.readUInt8}},function(i,o){"function"==typeof Object.create?i.exports=function inherits(i,o){i.super_=o,i.prototype=Object.create(o.prototype,{constructor:{value:i,enumerable:!1,writable:!0,configurable:!0}})}:i.exports=function inherits(i,o){i.super_=o;var TempCtor=function(){};TempCtor.prototype=o.prototype,i.prototype=new TempCtor,i.prototype.constructor=i}},function(i,o,u){"use strict";i.exports=function ZStream(){this.input=null,this.next_in=0,this.avail_in=0,this.total_in=0,this.output=null,this.next_out=0,this.avail_out=0,this.total_out=0,this.msg="",this.state=null,this.data_type=2,this.adler=0}},function(i,o,u){"use strict";var p,g=u(128),y=u(364),w=u(246),_=u(247),x=u(365);function err(i,o){return i.msg=x[o],o}function rank(i){return(i<<1)-(i>4?9:0)}function zero(i){for(var o=i.length;--o>=0;)i[o]=0}function flush_pending(i){var o=i.state,u=o.pending;u>i.avail_out&&(u=i.avail_out),0!==u&&(g.arraySet(i.output,o.pending_buf,o.pending_out,u,i.next_out),i.next_out+=u,o.pending_out+=u,i.total_out+=u,i.avail_out-=u,o.pending-=u,0===o.pending&&(o.pending_out=0))}function flush_block_only(i,o){y._tr_flush_block(i,i.block_start>=0?i.block_start:-1,i.strstart-i.block_start,o),i.block_start=i.strstart,flush_pending(i.strm)}function put_byte(i,o){i.pending_buf[i.pending++]=o}function putShortMSB(i,o){i.pending_buf[i.pending++]=o>>>8&255,i.pending_buf[i.pending++]=255&o}function longest_match(i,o){var u,p,g=i.max_chain_length,y=i.strstart,w=i.prev_length,_=i.nice_match,x=i.strstart>i.w_size-262?i.strstart-(i.w_size-262):0,k=i.window,P=i.w_mask,E=i.prev,O=i.strstart+258,I=k[y+w-1],B=k[y+w];i.prev_length>=i.good_match&&(g>>=2),_>i.lookahead&&(_=i.lookahead);do{if(k[(u=o)+w]===B&&k[u+w-1]===I&&k[u]===k[y]&&k[++u]===k[y+1]){y+=2,u++;do{}while(k[++y]===k[++u]&&k[++y]===k[++u]&&k[++y]===k[++u]&&k[++y]===k[++u]&&k[++y]===k[++u]&&k[++y]===k[++u]&&k[++y]===k[++u]&&k[++y]===k[++u]&&y<O);if(p=258-(O-y),y=O-258,p>w){if(i.match_start=o,w=p,p>=_)break;I=k[y+w-1],B=k[y+w]}}}while((o=E[o&P])>x&&0!=--g);return w<=i.lookahead?w:i.lookahead}function fill_window(i){var o,u,p,y,x,k,P,E,O,I,B=i.w_size;do{if(y=i.window_size-i.lookahead-i.strstart,i.strstart>=B+(B-262)){g.arraySet(i.window,i.window,B,B,0),i.match_start-=B,i.strstart-=B,i.block_start-=B,o=u=i.hash_size;do{p=i.head[--o],i.head[o]=p>=B?p-B:0}while(--u);o=u=B;do{p=i.prev[--o],i.prev[o]=p>=B?p-B:0}while(--u);y+=B}if(0===i.strm.avail_in)break;if(k=i.strm,P=i.window,E=i.strstart+i.lookahead,O=y,I=void 0,(I=k.avail_in)>O&&(I=O),u=0===I?0:(k.avail_in-=I,g.arraySet(P,k.input,k.next_in,I,E),1===k.state.wrap?k.adler=w(k.adler,P,I,E):2===k.state.wrap&&(k.adler=_(k.adler,P,I,E)),k.next_in+=I,k.total_in+=I,I),i.lookahead+=u,i.lookahead+i.insert>=3)for(x=i.strstart-i.insert,i.ins_h=i.window[x],i.ins_h=(i.ins_h<<i.hash_shift^i.window[x+1])&i.hash_mask;i.insert&&(i.ins_h=(i.ins_h<<i.hash_shift^i.window[x+3-1])&i.hash_mask,i.prev[x&i.w_mask]=i.head[i.ins_h],i.head[i.ins_h]=x,x++,i.insert--,!(i.lookahead+i.insert<3)););}while(i.lookahead<262&&0!==i.strm.avail_in)}function deflate_fast(i,o){for(var u,p;;){if(i.lookahead<262){if(fill_window(i),i.lookahead<262&&0===o)return 1;if(0===i.lookahead)break}if(u=0,i.lookahead>=3&&(i.ins_h=(i.ins_h<<i.hash_shift^i.window[i.strstart+3-1])&i.hash_mask,u=i.prev[i.strstart&i.w_mask]=i.head[i.ins_h],i.head[i.ins_h]=i.strstart),0!==u&&i.strstart-u<=i.w_size-262&&(i.match_length=longest_match(i,u)),i.match_length>=3)if(p=y._tr_tally(i,i.strstart-i.match_start,i.match_length-3),i.lookahead-=i.match_length,i.match_length<=i.max_lazy_match&&i.lookahead>=3){i.match_length--;do{i.strstart++,i.ins_h=(i.ins_h<<i.hash_shift^i.window[i.strstart+3-1])&i.hash_mask,u=i.prev[i.strstart&i.w_mask]=i.head[i.ins_h],i.head[i.ins_h]=i.strstart}while(0!=--i.match_length);i.strstart++}else i.strstart+=i.match_length,i.match_length=0,i.ins_h=i.window[i.strstart],i.ins_h=(i.ins_h<<i.hash_shift^i.window[i.strstart+1])&i.hash_mask;else p=y._tr_tally(i,0,i.window[i.strstart]),i.lookahead--,i.strstart++;if(p&&(flush_block_only(i,!1),0===i.strm.avail_out))return 1}return i.insert=i.strstart<2?i.strstart:2,4===o?(flush_block_only(i,!0),0===i.strm.avail_out?3:4):i.last_lit&&(flush_block_only(i,!1),0===i.strm.avail_out)?1:2}function deflate_slow(i,o){for(var u,p,g;;){if(i.lookahead<262){if(fill_window(i),i.lookahead<262&&0===o)return 1;if(0===i.lookahead)break}if(u=0,i.lookahead>=3&&(i.ins_h=(i.ins_h<<i.hash_shift^i.window[i.strstart+3-1])&i.hash_mask,u=i.prev[i.strstart&i.w_mask]=i.head[i.ins_h],i.head[i.ins_h]=i.strstart),i.prev_length=i.match_length,i.prev_match=i.match_start,i.match_length=2,0!==u&&i.prev_length<i.max_lazy_match&&i.strstart-u<=i.w_size-262&&(i.match_length=longest_match(i,u),i.match_length<=5&&(1===i.strategy||3===i.match_length&&i.strstart-i.match_start>4096)&&(i.match_length=2)),i.prev_length>=3&&i.match_length<=i.prev_length){g=i.strstart+i.lookahead-3,p=y._tr_tally(i,i.strstart-1-i.prev_match,i.prev_length-3),i.lookahead-=i.prev_length-1,i.prev_length-=2;do{++i.strstart<=g&&(i.ins_h=(i.ins_h<<i.hash_shift^i.window[i.strstart+3-1])&i.hash_mask,u=i.prev[i.strstart&i.w_mask]=i.head[i.ins_h],i.head[i.ins_h]=i.strstart)}while(0!=--i.prev_length);if(i.match_available=0,i.match_length=2,i.strstart++,p&&(flush_block_only(i,!1),0===i.strm.avail_out))return 1}else if(i.match_available){if((p=y._tr_tally(i,0,i.window[i.strstart-1]))&&flush_block_only(i,!1),i.strstart++,i.lookahead--,0===i.strm.avail_out)return 1}else i.match_available=1,i.strstart++,i.lookahead--}return i.match_available&&(p=y._tr_tally(i,0,i.window[i.strstart-1]),i.match_available=0),i.insert=i.strstart<2?i.strstart:2,4===o?(flush_block_only(i,!0),0===i.strm.avail_out?3:4):i.last_lit&&(flush_block_only(i,!1),0===i.strm.avail_out)?1:2}function Config(i,o,u,p,g){this.good_length=i,this.max_lazy=o,this.nice_length=u,this.max_chain=p,this.func=g}function DeflateState(){this.strm=null,this.status=0,this.pending_buf=null,this.pending_buf_size=0,this.pending_out=0,this.pending=0,this.wrap=0,this.gzhead=null,this.gzindex=0,this.method=8,this.last_flush=-1,this.w_size=0,this.w_bits=0,this.w_mask=0,this.window=null,this.window_size=0,this.prev=null,this.head=null,this.ins_h=0,this.hash_size=0,this.hash_bits=0,this.hash_mask=0,this.hash_shift=0,this.block_start=0,this.match_length=0,this.prev_match=0,this.match_available=0,this.strstart=0,this.match_start=0,this.lookahead=0,this.prev_length=0,this.max_chain_length=0,this.max_lazy_match=0,this.level=0,this.strategy=0,this.good_match=0,this.nice_match=0,this.dyn_ltree=new g.Buf16(1146),this.dyn_dtree=new g.Buf16(122),this.bl_tree=new g.Buf16(78),zero(this.dyn_ltree),zero(this.dyn_dtree),zero(this.bl_tree),this.l_desc=null,this.d_desc=null,this.bl_desc=null,this.bl_count=new g.Buf16(16),this.heap=new g.Buf16(573),zero(this.heap),this.heap_len=0,this.heap_max=0,this.depth=new g.Buf16(573),zero(this.depth),this.l_buf=0,this.lit_bufsize=0,this.last_lit=0,this.d_buf=0,this.opt_len=0,this.static_len=0,this.matches=0,this.insert=0,this.bi_buf=0,this.bi_valid=0}function deflateResetKeep(i){var o;return i&&i.state?(i.total_in=i.total_out=0,i.data_type=2,(o=i.state).pending=0,o.pending_out=0,o.wrap<0&&(o.wrap=-o.wrap),o.status=o.wrap?42:113,i.adler=2===o.wrap?0:1,o.last_flush=0,y._tr_init(o),0):err(i,-2)}function deflateReset(i){var o=deflateResetKeep(i);return 0===o&&function lm_init(i){i.window_size=2*i.w_size,zero(i.head),i.max_lazy_match=p[i.level].max_lazy,i.good_match=p[i.level].good_length,i.nice_match=p[i.level].nice_length,i.max_chain_length=p[i.level].max_chain,i.strstart=0,i.block_start=0,i.lookahead=0,i.insert=0,i.match_length=i.prev_length=2,i.match_available=0,i.ins_h=0}(i.state),o}function deflateInit2(i,o,u,p,y,w){if(!i)return-2;var _=1;if(-1===o&&(o=6),p<0?(_=0,p=-p):p>15&&(_=2,p-=16),y<1||y>9||8!==u||p<8||p>15||o<0||o>9||w<0||w>4)return err(i,-2);8===p&&(p=9);var x=new DeflateState;return i.state=x,x.strm=i,x.wrap=_,x.gzhead=null,x.w_bits=p,x.w_size=1<<x.w_bits,x.w_mask=x.